package com.verivital.hyst.generators;

import org.kohsuke.args4j.Option;

import com.verivital.hyst.geometry.Interval;
import com.verivital.hyst.ir.AutomatonExportException;
import com.verivital.hyst.ir.Configuration;

/**
 * Creates the NAV benchmark, from "Benchmarks for Hybrid Systems Verification", Fehnker et. al,
 * HSCC 2004
 * 
 * This one creates benchmark from names (the standard instances). Valid names are "nav01", "nav02",
 * ... as well as "fig1"and "fig2"(corresponding to the figures in the paper)
 * 
 * @author Stanley Bak (Sept 2016)
 *
 */
public class NamedNavigationGenerator extends ModelGenerator
{
	@Option(name = "-name", required = true, usage = "named instance, like 'nav01'", metaVar = "NAME")
	String name;

	// set with loadBenchmarkParams()
	private String[] map;
	private double[] matA;
	private Interval[] x0;
	private Interval[] v0;

	@Override
	public String getCommandLineFlag()
	{
		return "named_nav";
	}

	@Override
	public String getName()
	{
		return "Standard-Instance Navigation [Fehnker06]";
	}

	@Override
	protected Configuration generateModel()
	{
		loadBenchmarkParams();

		NavigationGenerator navGen = new NavigationGenerator();

		String param = NavigationGenerator.makeParamString(map, matA, x0, v0);

		return navGen.generate(param);
	}

	private void loadBenchmarkParams()
	{
		String map = null;
		String matA = null;
		String x0 = null;
		String v0 = null;

		// these were created from the nav_data_new.txt file provided with the nav benchmarks using
		// some light search and replace to make it java-parsable

		if (name.toUpperCase().equals("NAV01"))
		{
			map = "B 2 4 4 3 4 2 2 A";
			matA = "-1.2 0.1 0.1 -1.2";
			x0 = "2 3 1 2";
			v0 = "-0.3 0.3 -0.3 0";
		}
		else if (name.toUpperCase().equals("NAV02"))
		{
			map = "B 2 4 4 3 4 2 2 A";
			matA = "-1.2 0.1 0.1 -1.2";
			x0 = "2 3 1 2";
			v0 = "-0.3 0.3 -0.3 0.3";
		}
		else if (name.toUpperCase().equals("NAV03"))
		{
			map = "B 2 4 4 3 4 2 2 A";
			matA = "-1.2 0.1 0.1 -1.2";
			x0 = "2 3 1 2";
			v0 = "-0.4 0.4 -0.4 0.4";
		}
		else if (name.toUpperCase().equals("NAV04"))
		{
			map = "B 2 4 2 2 4 1 1 A";
			matA = "-1.2 0.1 0.1 -1.2";
			x0 = "0 1 0 1";
			v0 = "0.1 0.5 0.05 0.25";
		}
		else if (name.toUpperCase().equals("NAV05"))
		{
			map = "B 2 4 1 2 4 1 1 A";
			matA = "-1.2 0.1 0.1 -1.2";
			x0 = "0 1 0 1";
			v0 = "0.3 0.5 0.05 0.25";
		}
		else if (name.toUpperCase().equals("NAV06"))
		{
			map = "B 2 4 3 0 4 2 1 A";
			matA = "-1.2 0.1 0.1 -1.2";
			x0 = "0 1 0 1";
			v0 = "0.3 0.5 0.05 0.25";

		}
		else if (name.toUpperCase().equals("NAV07"))
		{
			map = "0 2 3 4 B 2 3 4 4 3 4 4 2 A 6 6";
			matA = "-0.8 -0.2 0.2 -0.8";
			x0 = "2 3 3 4";
			v0 = "0.1 0.5 -0.5 -0.1";

		}
		else if (name.toUpperCase().equals("NAV08"))
		{
			map = "4 6 3 4 B 6 3 4 0 7 4 4 2 A 6 6";
			matA = "-0.8 -0.2 0.2 -0.8";
			x0 = "2 3 3 4";
			v0 = "0.1 0.5 -0.5 -0.1";

		}
		else if (name.toUpperCase().equals("NAV09"))
		{
			map = "4 6 4 4 B 6 4 4 0 7 5 5 2 A 6 6";
			matA = "-1.8 -0.2 -0.2 -1.8";
			x0 = "2.5 3.5 3 4";
			v0 = "-0.1 0.1 -0.5 -0.1";
		}
		else if (name.toUpperCase().equals("NAV10"))
		{
			map = "2 4 6 6 6 2 4 7 7 4 2 4 B 3 4 2 4 6 6 6 2 A 0 0 0";
			matA = "-0.8 -0.1 -0.2 -0.8";
			x0 = "3.5 4 3.5 4";
			v0 = "-0.5 0.5 -0.5 -0";

		}
		else if (name.toUpperCase().equals("NAV11"))
		{
			map = "2 4 6 6 6 2 4 7 7 4 2 4 B 3 4 2 4 6 6 6 2 A 0 0 0";
			matA = "-0.8 -0.1 -0.2 -0.8";
			x0 = "3.5 4 3.5 4";
			v0 = "-1 0.5 -1 0.5";

		}
		else if (name.toUpperCase().equals("NAV12"))
		{
			map = "2 4 6 6 6 2 4 7 7 4 2 4 B 3 4 2 4 6 6 6 2 A 0 0 0";
			matA = "-0.8 -0.1 -0.2 -0.8";
			x0 = "3 3.5 3 3.5";
			v0 = "0.5 0.5 -0.5 0.5";

		}
		else if (name.toUpperCase().equals("NAV13"))
		{
			map = "4 4 4 4 4 4 4 2 4 4 6 6 6 6 2 4 A 6 6 6 6 2 4 B 1 7 7 0 2 4 2 1 7 7 0 2 2 2 1 1 0 0";
			matA = "-1.2 0.2 0.1 -1.2";
			x0 = "4.5 5.5 1.5 2.5";
			v0 = "0.5 0.5 -0.5 0.5";
		}
		else if (name.toUpperCase().equals("NAV14"))
		{
			map = "4 4 4 4 4 4 4 2 4 4 6 6 6 6 2 4 A 7 7 7 6 2 4 B 1 7 7 0 2 4 2 1 7 7 0 2 2 2 1 1 0 0";
			matA = "-1.2 0.2 0.1 -1.2";
			x0 = "4.5 5.5 1.5 2.5";
			v0 = "0.5 0.5 -0.5 0.5";

		}
		else if (name.toUpperCase().equals("NAV15"))
		{
			map = "4 4 4 4 4 4 4 2 4 5 6 6 6 6 2 4 A 7 7 7 6 2 4 B 1 7 7 0 2 4 2 1 7 7 0 2 2 2 1 1 0 0";
			matA = "-1.2 0.2 0.1 -1.2";
			x0 = "4.5 5.5 1.5 2.5";
			v0 = "0.5 0.5 -0.5 0.5";
		}
		else if (name.toUpperCase().equals("NAV16"))
		{
			map = "2 2 2 2 3 4 4 4 4 1 1 3 2 3 4 4 B 4 1 1 3 2 3 2 3 4 5 1 1 3 2 3 2 3 4 5 1 1 3 2 3 2 3 4 5 1 1 3 2 3 2 3 4 6 1 1 1 2 2 2 2 A 6 1 1 1 1 1 1 1 0 0 1 1 1 1 1 1 1 0 0";
			matA = "-0.2 -0.2 0 -0.7";
			x0 = "0.5 1.5 0.5 1.5";
			v0 = "0 0.5 0 0.5";

		}
		else if (name.toUpperCase().equals("NAV17"))
		{
			// there was a '-2' in this one which I converted to a 'B'
			map = "0 1 2 3 4 5 0 0 0 0 1 2 3 4 5 6 B 2 0 1 2 3 4 5 4 4 4 3 3 3 3 3 3 3 4 5 2 2 2 2 2 2 3 4 5 0 0 3 3 3 2 3 4 6 0 0 3 3 3 3 3 A 6 1 1 2 2 3 3 3 0 0 1 1 2 2 2 2 2 0 0";
			matA = "-1.5 -0.2 -0.5 -1.2";
			x0 = "0.5 1.5 0.5 1.5";
			v0 = "0 0.5 0 0.5";

		}
		else if (name.toUpperCase().equals("NAV18"))
		{
			map = "2 2 2 2 3 4 4 1 1 0 0 1 2 3 4 4 B 1 0 0 1 2 3 2 4 4 4 0 0 1 2 3 2 3 4 5 0 0 1 2 3 2 3 4 5 0 0 1 2 3 2 3 4 6 0 0 1 1 1 1 1 A 6 1 1 2 2 2 2 2 0 0 1 1 2 2 2 2 2 0 0";
			matA = "-1.2 -0.2 0.5 -0.7";
			x0 = "0.5 1.5 0.5 1.5";
			v0 = "0 0.5 0 0.5";

		}
		else if (name.toUpperCase().equals("NAV19"))
		{
			map = "3 3 3 3 4 4 4 4 5 5 5 5 3 3 2 0 1 7 3 4 5 3 4 5 3 7 6 1 7 1 2 A 6 6 7 5 6 6 7 6 0 1 3 0 0 7 7 5 6 7 6 5 B 2 3 1 0 7 7 6 6 3 4 5 4 3 4 2 1 0 0 6 4 5 4 4 5 3 2 1 1 1 0 6 2 3 3 4 4 5 3 3 4 1 0 6 1 2 4 3 4 6 7 1 2 1 0 7 1 2 2 3 3 3 3 3 3 1 0 7 1 1 2 2 2 2 2 2 2 0 0 7 1 1 1 1 0 0 0 0 0 0 7 7";
			matA = "-1 -0.7 1 -1.8";
			x0 = "0 1 0 1";
			v0 = "0 0.5 0 0.5";

		}
		else if (name.toUpperCase().equals("NAV20"))
		{
			map = "3 3 3 3 4 4 4 4 5 5 5 5 3 3 2 0 1 7 6 4 5 3 4 5 3 7 6 4 5 3 2 A 6 6 7 5 3 6 7 6 0 2 3 0 0 7 7 5 2 2 3 5 B 6 7 1 0 0 7 6 2 3 2 0 1 7 6 4 5 0 0 6 2 7 6 4 5 3 2 0 1 1 0 6 2 6 7 1 4 5 3 5 4 1 0 6 1 2 4 3 4 6 7 1 0 1 0 7 1 3 2 3 4 0 2 4 3 1 0 7 1 1 2 2 2 2 2 2 2 0 0 7 1 1 1 1 0 0 0 0 7 7 7 7";
			matA = "-1.5 -0.6 0.3 -0.7";
			x0 = "0 1 0 1";
			v0 = "0 0.5 0 0.5";

		}
		else if (name.toUpperCase().equals("NAV21"))
		{
			map = "3 3 3 3 4 4 4 4 5 5 5 5 3 3 2 0 1 7 6 4 5 3 4 5 3 7 6 4 5 3 2 A 6 6 7 5 3 6 7 6 0 2 3 0 0 7 7 5 2 2 3 5 B 6 7 1 0 0 7 6 2 3 2 0 1 7 6 4 5 0 0 6 2 7 6 4 5 3 2 0 1 1 0 6 2 6 7 1 4 5 3 5 4 1 0 6 1 2 4 3 4 6 7 1 0 1 0 7 1 3 2 3 1 0 2 4 3 1 0 7 1 1 2 2 2 2 2 2 2 0 0 7 1 1 1 1 0 0 0 0 7 7 7 7";
			matA = "-1.5 -0.6 0.3 -0.7";
			x0 = "0 1 0 1";
			v0 = "0 0.5 0 0.5";

		}
		else if (name.toUpperCase().equals("NAV22"))
		{
			map = "1 2 3 2 3 3 3 3 3 3 3 3 3 4 5 0 7 5 3 1 2 2 2 2 2 2 2 3 4 5 1 0 6 1 1 0 1 1 1 1 1 1 3 4 5 0 7 5 2 1 0 7 0 1 4 0 7 3 4 5 7 6 7 1 1 0 7 7 1 5 7 6 3 4 5 1 2 3 1 1 0 7 7 0 3 1 2 3 4 5 0 7 5 0 1 0 0 6 3 4 0 7 3 4 5 1 0 6 0 1 0 0 6 4 5 1 0 3 4 5 0 7 5 0 1 0 7 5 3 4 0 7 3 4 5 7 6 7 1 7 0 6 7 6 5 7 6 3 4 5 1 2 3 2 0 1 1 1 1 2 1 2 3 4 5 0 7 5 3 0 0 0 0 0 1 0 2 3 3 5 1 0 6 4 5 1 0 6 4 5 1 2 3 3 4 0 7 5 3 4 0 7 5 3 4 0 2 2 3 4 B 6 7 6 5 7 6 7 6 5 7 2 2 2 A";
			matA = "-0.9 -0.5 -0.1 -0.5";
			x0 = "7 8 7 8";
			v0 = "-0.4 -0.1 -0.4 -0.1";

		}
		else if (name.toUpperCase().equals("NAV23"))
		{
			map = "1 2 3 2 3 3 3 2 2 3 2 3 3 2 5 0 7 5 3 1 2 3 2 2 3 2 2 2 4 5 1 0 6 1 1 1 1 1 2 2 1 2 3 5 5 0 7 5 2 1 0 0 0 1 4 0 7 4 3 4 7 6 7 1 1 0 7 7 1 5 7 6 3 5 4 1 2 3 1 0 1 0 7 0 3 1 2 3 5 5 0 7 5 0 1 0 0 6 3 4 0 7 3 4 4 1 0 6 0 0 0 0 6 4 5 1 0 4 5 4 0 7 5 0 1 1 7 5 3 4 0 7 4 5 4 7 6 7 1 7 0 6 7 6 5 7 6 3 4 5 1 2 3 2 0 1 2 3 2 3 1 2 4 5 4 0 7 5 3 4 0 7 5 3 4 0 7 4 3 4 1 0 6 4 5 1 0 6 4 5 1 0 3 5 5 0 7 5 3 4 0 7 5 3 4 0 7 1 2 4 B 6 7 6 5 7 6 7 6 5 7 6 2 1 A";
			matA = "-0.9 -0.5 -0.1 -0.5";
			x0 = "7 8 7 8";
			v0 = "-0.4 -0.1 -0.4 -0.1";

		}
		else if (name.toUpperCase().equals("NAV24"))
		{
			map = "1 2 3 2 3 1 2 3 2 3 1 2 3 2 3 0 7 5 3 4 0 7 5 3 4 0 7 5 3 4 1 0 6 4 5 1 0 6 4 5 1 0 6 4 5 0 7 5 3 4 0 7 5 3 4 0 7 5 3 4 7 6 7 6 5 7 6 7 6 5 7 6 7 6 5 1 2 3 2 3 1 2 3 2 3 1 2 3 2 3 0 7 5 3 4 0 7 5 3 4 0 7 5 3 4 1 0 6 4 5 1 0 6 4 5 1 0 6 4 5 0 7 5 3 4 0 7 5 3 4 0 7 5 3 4 7 6 7 6 5 7 6 7 6 5 7 6 7 6 5 1 2 3 2 3 1 2 3 2 3 1 2 3 2 3 0 7 5 3 4 0 7 5 3 4 0 7 5 3 4 1 0 6 4 5 1 0 6 4 5 1 0 6 4 5 0 7 5 3 4 0 7 5 3 4 0 7 5 3 4 B 6 7 6 5 7 6 7 6 5 7 6 7 6 A";
			matA = "-0.9 -0.5 -0.1 -0.5";
			x0 = "7 8 7 8";
			v0 = "-0.4 -0.1 -0.4 -0.1";
		}
		else if (name.toUpperCase().equals("NAV25"))
		{
			map = "2 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 4 5 1 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 A 6 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 7 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 7 1 0 7 7 4 2 6 0 2 0 3 2 7 6 2 0 1 1 0 7 1 0 7 7 0 6 0 2 1 7 1 6 1 7 6 6 1 1 0 7 1 0 7 7 1 4 2 5 0 1 1 1 6 1 1 2 1 1 0 7 1 0 7 7 7 4 6 1 0 0 4 0 1 3 3 4 1 1 0 7 1 0 7 7 0 1 2 4 6 4 5 5 5 2 0 1 1 1 0 7 1 0 7 7 2 6 7 7 2 B 3 4 5 1 2 2 1 1 0 7 1 0 7 7 7 2 5 6 3 0 2 2 6 7 7 6 1 1 0 7 1 0 7 7 1 6 0 7 7 6 2 2 3 2 6 5 1 1 0 7 1 0 7 7 3 5 0 0 1 0 6 0 3 5 2 0 1 1 0 7 1 0 7 7 2 1 0 6 6 4 5 7 3 3 7 4 1 1 0 7 1 0 7 7 5 4 1 7 5 1 1 6 4 3 3 3 1 1 0 7 1 0 7 7 7 4 4 1 7 5 7 2 1 4 4 7 1 1 0 7 1 0 7 3 3 3 3 3 3 3 3 3 3 3 3 3 1 1 0 7 1 0 7 3 3 3 3 3 3 3 3 3 3 3 3 3 1 1 0 7 1 1 2 2 2 2 2 2 2 2 2 2 2 2 2 2 1 1 0 7 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0";
			matA = "-0.5 -0.7 0.2 -0.7";
			x0 = "0 1 0 1";
			v0 = "0.1 0.5 0.1 0.5";

		}
		else if (name.toUpperCase().equals("NAV26"))
		{
			map = "2 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 4 5 1 2 2 3 2 3 2 3 2 2 2 3 3 2 2 2 2 A 6 6 1 1 1 2 1 2 1 2 1 1 1 2 2 1 1 2 4 6 7 7 1 1 1 1 1 1 1 1 1 1 1 1 1 1 2 4 5 7 7 7 1 0 7 7 4 2 6 0 2 0 3 2 7 6 3 5 5 1 0 7 1 0 7 7 0 6 0 2 7 7 1 6 1 3 5 5 1 1 0 7 1 0 7 7 1 4 2 5 7 1 1 0 3 5 5 5 1 1 0 0 1 0 7 7 7 4 6 1 0 0 3 3 5 5 5 5 1 1 0 7 1 0 7 7 0 1 2 4 6 4 4 5 5 5 5 7 1 1 0 7 1 1 0 7 2 6 7 7 2 B 6 6 6 6 6 7 1 1 0 7 1 1 0 7 7 2 5 6 7 0 0 0 0 0 0 0 1 0 0 7 1 1 0 7 1 6 0 7 7 6 2 2 3 2 6 5 1 1 0 7 1 0 7 7 3 5 0 0 1 0 6 0 3 5 2 0 1 1 0 7 1 0 7 7 2 1 0 6 7 4 5 7 3 3 7 4 1 0 7 7 1 0 7 7 5 4 1 7 5 1 1 6 4 3 3 3 1 0 7 7 1 0 7 7 7 4 4 1 7 5 7 2 1 4 4 7 1 1 0 7 1 0 7 3 3 3 3 3 3 3 3 3 3 3 3 3 1 1 0 7 1 0 7 3 3 2 2 3 3 3 2 3 3 2 3 3 1 1 0 7 1 1 2 2 2 1 1 2 2 2 1 2 2 1 2 2 1 1 0 7 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0";
			matA = "-0.5 -0.7 0.2 -0.7";
			x0 = "0 1 0 1";
			v0 = "0.1 0.5 0.1 0.5";

		}
		else if (name.toUpperCase().equals("NAV27"))
		{
			map = "2 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 4 5 1 2 2 3 2 3 2 2 3 2 2 1 3 3 1 2 2 A 6 6 1 1 2 2 1 2 2 3 1 1 1 2 2 1 1 2 4 6 7 7 1 1 1 1 1 1 1 1 1 1 1 1 1 1 2 4 5 7 7 7 1 0 7 7 4 2 6 0 2 0 3 2 7 6 3 5 5 1 0 7 1 0 7 7 0 6 0 2 7 7 1 6 1 3 5 4 1 1 0 7 1 0 7 7 1 4 2 5 7 1 1 0 3 5 4 5 1 1 0 0 1 0 7 7 7 4 6 1 0 0 3 3 5 4 5 2 1 1 0 0 1 0 7 7 0 1 2 4 6 4 4 5 4 5 2 7 1 1 0 0 1 1 0 7 2 6 7 7 2 B 6 6 6 6 6 7 1 1 0 7 1 0 1 7 7 2 5 6 7 0 0 0 0 0 0 0 1 0 0 7 1 1 0 7 1 6 0 7 7 6 2 2 3 2 6 5 1 1 0 7 1 0 7 7 3 5 0 0 1 0 6 0 3 5 2 0 1 1 0 7 1 0 7 7 2 1 0 6 7 4 5 7 3 3 7 4 1 0 0 7 1 0 7 7 5 4 1 7 5 1 1 6 4 3 3 3 1 0 7 7 1 0 7 7 7 4 4 1 7 5 7 2 1 4 4 7 1 1 0 7 1 0 7 3 3 3 3 3 3 3 3 3 3 3 3 3 1 1 0 7 0 0 7 3 3 2 2 3 3 3 2 3 1 2 3 3 1 1 0 7 1 1 2 2 2 1 3 2 2 2 1 2 2 1 2 2 1 1 0 7 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0";
			matA = "-0.5 -0.7 0.2 -0.7";
			x0 = "0 1 0 1";
			v0 = "0.1 0.5 0.1 0.5";

		}
		else if (name.toUpperCase().equals("NAV28"))
		{
			map = "6 2 6 3 4 7 7 5 5 6 5 6 5 2 3 5 7 5 7 6 6 5 7 6 6  3 3 2 6 5 7 5 6 6 5 5 5 4 3 3 6 5 5 5 6 5 6 7 5 6  1 2 2 3 3 3 4 5 5 5 6 6 5 7 1 7 6 5 6 5 5 5 5 6 6  3 3 2 3 3 3 4 5 5 7 5 5 7 7 0 6 7 7 5 5 7 7 7 5 6  1 5 5 3 3 3 4 5 5 7 7 7 7 5 3 5 7 7 5 7 7 6 6 6 7  0 4 3 2 2 2 3 3 3 4 5 5 6 6 5 7 6 6 7 5 B 7 5 5 5  3 3 1 2 2 2 3 3 3 4 5 6 6 7 7 6 7 6 7 7 6 7 7 5 6  1 3 2 1 1 2 3 3 3 4 5 5 7 6 5 7 5 5 6 6 6 5 6 5 7  3 2 2 1 1 3 2 2 2 3 3 3 4 5 6 7 6 6 5 5 6 7 7 5 5  3 2 3 2 2 3 2 3 1 3 3 3 4 5 5 6 7 6 7 6 7 6 6 7 5  2 2 3 2 2 2 1 2 2 3 3 3 4 5 7 6 5 6 5 7 5 7 5 6 5  3 1 2 2 2 3 2 3 3 2 2 2 3 3 3 4 5 5 7 6 5 7 5 6 6  3 7 1 2 0 5 2 2 1 3 1 1 3 3 3 4 5 5 6 5 6 6 7 6 7  6 6 6 5 4 5 1 2 3 2 1 1 3 3 3 4 5 5 5 5 5 7 5 6 7  2 1 6 6 7 3 7 0 2 2 3 1 2 2 2 3 3 3 4 5 5 7 6 7 5  3 2 3 3 1 1 2 3 3 2 1 2 3 1 3 3 3 3 4 5 6 7 7 7 6  2 3 3 3 1 3 1 1 2 1 2 1 1 1 2 3 3 3 4 5 7 5 7 5 7  1 1 3 3 3 2 2 2 1 3 7 3 3 1 2 2 2 2 3 3 3 4 5 5 5  2 3 1 1 2 3 2 3 3 3 2 4 2 2 2 3 1 2 3 3 3 4 5 5 5  3 2 1 1 2 3 1 2 1 2 2 1 1 2 2 2 3 2 3 3 3 4 5 5 5  2 1 3 3 3 3 2 1 3 3 2 2 3 2 3 3 1 3 2 2 2 A 6 6 7  2 2 2 3 1 1 2 1 3 1 2 2 1 1 1 2 2 3 1 3 1 0 7 7 1  2 1 1 1 1 2 1 2 1 2 3 3 3 3 1 2 3 3 2 3 3 0 7 6 6  2 2 1 2 2 2 2 2 3 2 2 2 2 3 3 1 3 3 1 3 1 4 2 3 1  1 3 1 1 2 2 1 3 2 3 1 1 1 3 1 1 3 3 1 3 1 1 5 7 5";
			matA = "-0.5 0.6 0.2 -0.7";
			x0 = "4 5 21 22";
			v0 = "0.2 0.5 -0.5 0.5";

		}
		else if (name.toUpperCase().equals("NAV29"))
		{
			map = "4 5 5 4 5 5 5 4 6 6 5 6 6 6 6 5 5 6 4 6 4 5 5 4 6  4 5 4 5 3 4 4 4 4 5 6 6 4 6 5 6 6 6 4 5 6 6 4 5 4  3 3 4 3 3 3 3 6 4 6 5 4 4 4 5 6 4 4 6 4 4 4 5 5 4  4 5 4 3 3 3 4 6 6 5 4 6 5 6 5 6 4 6 6 5 5 4 6 4 5  3 5 3 3 3 3 3 4 6 4 5 5 5 5 4 5 5 6 4 6 5 6 5 5 6  4 3 4 5 3 3 3 4 6 6 5 6 6 6 5 4 6 5 6 5 B 6 5 5 5  3 3 5 4 3 3 3 4 4 6 6 6 6 4 6 4 4 6 6 5 6 4 6 5 6  5 3 4 4 3 3 3 5 4 6 4 5 6 4 5 5 6 6 5 5 4 6 4 5 4  5 3 5 5 3 3 3 3 6 4 5 6 4 5 6 4 5 5 6 4 5 4 4 5 5  5 4 5 4 4 3 3 3 5 6 5 6 4 6 4 6 5 4 4 6 5 5 6 6 4  4 5 4 5 5 3 3 3 4 4 5 6 6 6 5 6 4 6 4 4 6 6 6 6 6  4 4 3 5 4 4 3 3 3 5 5 4 6 5 4 4 6 5 4 4 5 4 4 4 6  4 5 4 5 4 5 3 3 3 4 4 6 4 4 5 6 4 5 5 6 6 6 4 6 5  5 4 4 4 3 4 3 3 3 5 5 5 5 4 4 6 5 5 6 6 4 6 4 5 4  5 4 3 3 3 4 3 3 3 3 5 4 5 4 5 6 6 6 3 3 3 6 6 4 5  5 4 4 5 3 4 4 3 3 3 4 5 6 4 4 5 6 4 4 5 6 6 5 5 5  3 5 3 3 5 4 4 3 3 3 6 5 6 6 6 6 4 4 5 5 6 6 4 4 5  5 3 3 5 5 3 4 6 3 3 3 3 3 5 6 4 4 4 4 3 3 4 5 5 5  2 3 2 3 1 3 1 2 3 3 3 3 3 3 3 3 3 3 3 3 3 4 5 5 3  3 2 3 2 1 3 2 1 3 3 3 3 3 3 3 3 3 3 3 3 3 4 5 5 5  2 1 3 2 3 1 1 3 2 3 2 3 3 3 3 3 3 3 3 2 2 A 6 6 6  2 2 1 1 2 2 1 3 2 1 2 1 3 3 3 2 2 1 2 1 1 0 7 7 7  2 1 2 2 2 1 2 3 1 3 1 3 3 3 3 1 2 2 2 1 1 0 7 7 6  2 2 1 2 1 1 1 1 1 2 2 2 1 1 2 1 1 2 2 1 1 0 7 6 6  1 3 3 3 3 2 3 2 1 3 2 2 1 1 1 1 1 1 1 1 1 0 7 6 6";
			matA = "-0.5 0.6 0.2 -0.7";
			x0 = "4 5 21 22";
			v0 = "0 0.5 -0.5 0";

		}
		else if (name.toUpperCase().equals("NAV30"))
		{
			map = "4 5 5 4 5 5 5 4 6 6 5 6 6 6 6 5 5 6 4 6 4 5 5 4 6  4 5 4 5 3 4 4 4 4 5 6 6 4 6 5 6 6 6 4 5 6 6 4 5 4  3 3 4 3 3 3 3 6 4 6 5 4 4 4 5 6 4 4 6 4 4 4 5 5 4  4 5 4 3 3 3 3 3 3 5 4 6 5 6 5 6 4 6 6 5 5 4 6 4 5  3 5 3 3 3 3 3 3 3 3 3 3 5 5 4 5 5 6 4 6 5 6 5 5 6  4 3 4 5 3 3 3 3 3 3 3 3 3 3 3 4 6 5 6 5 B 6 5 5 5  3 3 5 4 3 3 3 4 4 3 3 3 3 3 3 3 3 3 6 5 6 4 6 5 6  5 3 4 4 3 3 3 5 4 2 2 1 3 3 3 3 3 3 3 3 3 6 4 5 4  5 3 5 5 3 3 3 3 6 4 5 1 2 3 1 1 3 3 3 3 3 4 4 5 5  5 4 5 4 4 3 3 3 5 6 5 6 2 2 2 1 2 1 3 3 3 5 6 6 4  4 5 4 5 5 3 3 3 4 4 5 6 6 6 1 1 3 3 3 4 5 5 4 5 5  4 4 3 5 4 4 3 3 3 5 5 4 6 5 4 1 1 3 3 3 5 4 4 4 6  4 5 4 5 4 5 3 3 3 4 4 6 4 4 5 1 1 3 3 3 4 6 4 6 5  5 4 4 4 3 4 3 3 3 5 5 5 5 4 4 2 2 3 3 3 4 5 4 5 4  5 4 3 3 3 4 3 3 3 3 5 4 5 4 5 6 2 3 3 3 3 5 6 4 5  5 4 4 5 3 4 4 3 3 3 4 5 6 4 4 5 1 2 3 3 3 5 5 5 5  3 5 3 3 5 4 4 3 3 3 6 5 6 6 6 6 2 3 3 3 3 5 4 4 5  5 3 3 5 5 3 4 6 3 3 3 3 3 5 6 4 4 4 4 3 3 5 5 5 5  2 3 2 3 1 3 1 2 3 3 3 3 3 3 3 3 3 3 3 3 3 4 5 5 3  3 2 3 2 1 3 2 1 3 3 3 3 3 3 3 3 3 3 3 3 3 4 5 5 5  2 1 3 2 3 1 1 3 2 3 2 3 3 3 3 3 3 3 3 2 2 A 6 6 6  2 2 1 1 2 2 1 3 2 1 2 1 3 3 3 2 2 1 2 1 1 0 7 7 7  2 1 2 2 2 1 2 3 1 3 1 3 3 3 3 1 2 2 2 1 1 0 7 7 6  2 2 1 2 1 1 1 1 1 2 2 2 1 1 2 1 1 2 2 1 1 0 7 6 6  1 3 3 3 3 2 3 2 1 3 2 2 1 1 1 1 1 1 1 1 1 0 7 6 6";
			matA = "-0.5 0.6 0.2 -0.7";
			x0 = "4 5 21 22";
			v0 = "0.2 0.5 -0.5 0.5";
		}
		else
			throw new AutomatonExportException(
					"Unrecognized standard-instance benchmark name: '" + name + "'");

		// convert strings into expected formats
		this.map = map.split(" ");

		String[] parts = matA.split(" ");
		double[] vals = new double[4];

		for (int i = 0; i < 4; ++i)
			vals[i] = Double.parseDouble(parts[i]);

		this.matA = vals;

		parts = x0.split(" ");
		vals = new double[4];

		for (int i = 0; i < 4; ++i)
			vals[i] = Double.parseDouble(parts[i]);

		this.x0 = new Interval[] { new Interval(vals[0], vals[1]), new Interval(vals[2], vals[3]) };

		parts = v0.split(" ");
		vals = new double[4];

		for (int i = 0; i < 4; ++i)
			vals[i] = Double.parseDouble(parts[i]);

		this.v0 = new Interval[] { new Interval(vals[0], vals[1]), new Interval(vals[2], vals[3]) };
	}
}
