﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using GME.MGA;
using System.IO;
using System.Diagnostics;
using System.Xml;
using System.Windows.Forms;

namespace HiLiTeVerifierPlugin
{

    
    public class HiLiTeGateway
    {

        private const string modelNamePlaceholder = @"@@@MODEL_NAME@@@"; //w/o ".mdl" extension
        private const string intermediateRangesPlaceholder = @"@@@INTERMEDIATE_FILENAME@@@";         // Default: intermediateRanges.csv
        private const string externalRangesPlaceholder = @"@@@EXTERNAL_FILENAME@@@";  // Default: ExternalInputRanges.csv
       
        public const string summaryRelPath = @"Output\Reports";
        public const string externalFileName = @"ConfigTemplate.hilite";
 


        public List<DefectInfo> defects;
        private string configTemplate;
        private StringBuilder configFile = new StringBuilder();
        private MgaFCO formalRequirement;

        private List<SignalInfo> signals;

        public List<SignalInfo> Signals
        {
            get { return signals; }
        }

        public HiLiTeGateway(MgaFCO formalRequirement)
        {
            this.formalRequirement = formalRequirement;
            signals = GetSignalInfo(formalRequirement);


            string fileName = Path.Combine(Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location), externalFileName);
            if (File.Exists(fileName))
            {               
                try
                {
                    configTemplate = File.ReadAllText(fileName);
                }
                catch (Exception ex) // If exception occures, we default to the default path
                {
                    MessageBox.Show(String.Format("Cannot load external config file {0}, due to the following error: {1}. Defaulting to the internal config file.", fileName, ex.Message));
                    configTemplate = Resources.ConfigTemplate;
                }
            }
            else
            {
                configTemplate = Resources.ConfigTemplate;
            }
        }


        public static bool IsSimulinkContainer(MgaModel container)
        {
            return (container.GetChildrenOfKind("SimulinkLink").Count == 0) ? false : true;
        }

        private static List<SignalInfo> GetSignalInfo(MgaFCO formalRequirement)
        {
            List<SignalInfo> signals = new List<SignalInfo>();
            MgaModel component = formalRequirement.ParentModel;

            if (!IsSimulinkContainer(component))
            {
                throw new ApplicationException(String.Format("Component {0} does not contain a SimulinkLink element. Only containers referencing a Simulink model can be verified by HiLiTe", component.Name));
            }

            MgaFCOs actuatorControlPorts = component.GetChildrenOfKind("ActuatorControlPort");
            foreach(IMgaFCO actuatorControlPort in actuatorControlPorts)
            {
                if(!IsSimulinkContainer(actuatorControlPort as MgaModel))
                {
                    throw new ApplicationException(String.Format("Port {0} does not contain a SimulinkLink element. Only ports referencing a Simulink model can be verified by HiLiTe", actuatorControlPort.Name));
                }

                MgaFCOs values = (actuatorControlPort as MgaModel).GetChildrenOfKind("Value");
                if (values.Count > 1)
                {
                    throw new ApplicationException(String.Format("Multiple values was found in port {0}. Please provide only one input range specification", actuatorControlPort.Name));
                }

                MgaFCO range = values[1];


                MgaAttribute attr = range.get_Attribute(range.Meta.get_AttributeByName("Min"));
                string min = attr.Status != -1 ? attr.FloatValue.ToString() : String.Empty;

                attr = range.get_Attribute(range.Meta.get_AttributeByName("Max"));
                string max = attr.Status != -1 ? attr.FloatValue.ToString() : String.Empty;

                attr = range.get_Attribute(range.Meta.get_AttributeByName("HardConstraintMin"));
                string hardConstraintMin = attr.Status != -1 ? attr.FloatValue.ToString() : String.Empty;

                attr = range.get_Attribute(range.Meta.get_AttributeByName("HardConstraintMax"));
                string hardConstraintMax = attr.Status != -1 ? attr.FloatValue.ToString() : String.Empty;


                SignalInfo signal = new SignalInfo(actuatorControlPort.Name,
                    min,
                    max,
                    range.get_StrAttrByName("DataType"),
                    hardConstraintMin,
                    hardConstraintMax,
                    range.get_StrAttrByName("Resolution"),
                    range.get_StrAttrByName("Units"),
                    range.get_StrAttrByName("Description"));
                
                signals.Add(signal);
            }

            return signals;
        }

        public string ExportSignals()
        {
            StringBuilder exported = new StringBuilder(signals.Count*32 + 256); // Heuristic values based on an example

            exported.AppendLine(@"# This file contains ranges and other properties of external signals inputs into a partition,,,,,,,");
            // Header
            exported.AppendLine(@"# signal name, operational range min,operational range max,data type,hard constraint min,hard constraint max, resolution, units,description");

            foreach (SignalInfo signal in signals)
            {
                exported.AppendLine(String.Format("{0},{1},{2},{3},{4},{5},{6},{7},{8}", signal.Name, signal.MinimumValue, signal.MaximumValue,
                    signal.DataType, signal.HardMinimumValue, signal.HardMaximumValue, signal.Resolution, signal.Units, signal.Description));
            }

            
            return exported.ToString();
        }

        public void GenerateFiles(Settings settings)
        {
            if (!Directory.Exists(settings.workingDir))
            {
                Directory.CreateDirectory(settings.workingDir);
            }

            if (File.Exists(settings.hiliteProjectFilePath))
            {
                File.Delete(settings.hiliteProjectFilePath);
            }

            string path = GetModelPath();
            string modelName = Path.GetFileNameWithoutExtension(path); 

            using (StreamWriter writer = new StreamWriter(settings.hiliteProjectFilePath))
            {
                GenerateHiliteFile(writer, settings, modelName);
            }

            using (StreamWriter writer = new StreamWriter(settings.externalRangesPath, true))
            {
                writer.Write(ExportSignals());
                
            }

            string directoryPath = Path.GetDirectoryName(path);
            if(settings.workingDir != directoryPath)
            {
                string targetPath = Path.Combine(settings.workingDir, Path.GetFileName(path));
                if (!File.Exists(targetPath))
                {
                    // Copying simulink file if not in the working dir
                    File.Copy(path, targetPath);
                }
            }

        }

        public void GenerateHiliteFile(StreamWriter writer, Settings settings, string modelName)
        {
            StringBuilder content = new StringBuilder(configTemplate);
            content.Replace(intermediateRangesPlaceholder, Path.GetFileName(settings.intermediatePath));
            content.Replace(modelNamePlaceholder, modelName );
            content.Replace(externalRangesPlaceholder, Path.GetFileName(settings.externalRangesPath));

            writer.Write(content.ToString());

        }

        public void Execute(Settings settings, DataReceivedEventHandler dataHandler, EventHandler exitHandler)
        {
            string command = settings.hilitePath;
            string args = String.Format("{0} Output", settings.hiliteProjectFilePath);
         
             Process hilite = new Process();
            
            
            ProcessStartInfo psInfo = hilite.StartInfo;
            psInfo.FileName = command;
            psInfo.Arguments = args;
            psInfo.WorkingDirectory = settings.workingDir;
                      
            psInfo.UseShellExecute = false;
            psInfo.CreateNoWindow = true;

            psInfo.RedirectStandardOutput = true;
            hilite.OutputDataReceived+= dataHandler;

            hilite.EnableRaisingEvents = true;
            hilite.Exited += exitHandler;

            hilite.Start();

            hilite.BeginOutputReadLine();
            
          
            //hilite.WaitForExit();
         
        }

       

        public void GetResults(Settings settings, out string resultXML, out List<DefectInfo> defects)
        {
            string path = GetModelPath();
            string modelName = Path.GetFileNameWithoutExtension(path);

            string summaryPath = Path.Combine(settings.workingDir, summaryRelPath, String.Format("{0}_Summary.xml", modelName));

            using (StreamReader reader = new StreamReader(summaryPath))
            {
                resultXML = reader.ReadToEnd();
            }

            defects = ParseResult(resultXML, path);

        }

        private List<DefectInfo> ParseResult(string resultXML, string modelPath)
        {
            XmlDocument doc = new XmlDocument();
            doc.LoadXml(resultXML);

            List<DefectInfo> defectInfos = new List<DefectInfo>();

            XmlNodeList defects = doc.SelectNodes("HiLiTESummary/Defects/Defect");
            uint counter = 0;

            foreach (XmlNode defect in defects)
            {                
                Debug.Assert(defect.ChildNodes.Count == 1);
                foreach(XmlNode msgText in defect.ChildNodes)
                {
                    defectInfos.Add(new DefectInfo(true, counter++, defect.Attributes["Type"].Value,
                        msgText.InnerText, msgText.Attributes["Block"].Value,
                        msgText.Attributes["Port"].Value,
                        Int32.Parse(defect.Attributes["Count"].Value),
                        modelPath));
                }                                
            }
            return defectInfos;
        }

        string GetModelPath()
        {
            MgaModel component = formalRequirement.ParentModel;
            MgaFCOs simulinkLinks = component.GetChildrenOfKind("SimulinkLink");


            if (simulinkLinks.Count != 1)
            {
                throw new ApplicationException(String.Format("A component must have exactly one SimulinkLink element, which is not satisfied in case of the component {1}.", component.Name));
            }


            MgaFCO simulinkLink = simulinkLinks[1];
            string uri = simulinkLink.get_StrAttrByName("URI");

            string projectPath = formalRequirement.Project.ProjectConnStr;
            projectPath = projectPath.Substring(4);

            uri = SimulinkGateway.ResolveURI(uri, Path.GetDirectoryName(projectPath));

            string path;
            string block;
            SimulinkGateway.GetPathFormUri(uri,out path, out block);
            return path;
      
        }
    }

   
}
