﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using GME.MGA;
using GME.MGA.Meta;
using System.IO;
using System.Diagnostics;

namespace HiLiTeVerifierPlugin
{
    public partial class MainForm : Form
    {
        private const string additionalToolParameterName = @"AdditionalToolParameters";
        private const string verificationResultAttrName = @"VerificationResult";
        private const string toolSpecAttrName = @"ToolSpecificMessage";
        private const string rtfHeader = @"{\rtf1\ansi\ansicpg1252\deff0\deflang1033{\fonttbl{\f0\fnil\fcharset0 Consolas;}}" +
                             @"{\colortbl ;\red0\green0\blue0;\red0\green128\blue0;\red0\green0\blue255;\red43\green145\blue177;\red139\green0\blue0;}\viewkind4\uc1\pard\cf1\f0\fs20";
        private Icon errorIcon = new Icon(Resources.error, new Size(16, 16));
        private Icon warningIcon = new Icon(Resources.warning, new Size(16, 16));

        private HiLiTeGateway hilite = null;
        private MgaFCO mgaFormalRequirement;
        SortedDictionary<string, GMEAttribute> propertyMap = new SortedDictionary<string,GMEAttribute>();
        private Settings settings;
        private StringBuilder outputText;
        private string result;

 

        public MainForm(MgaFCO mgaFormalRequirement)
        {
            this.mgaFormalRequirement = mgaFormalRequirement;
            hilite = new HiLiTeGateway(mgaFormalRequirement);
            InitializeComponent();

            settings = new Settings(mgaFormalRequirement.get_StrAttrByName(additionalToolParameterName));
            FillSettings();

            FillRanges();
            
            FillPropertyMap();
            FillPropertyDataGrid();
        }

        private void FillSettings()
        {
            textBoxHiliteFile.Text = settings.hiliteProjectFilePath;
            textBoxIntermediatePath.Text = settings.intermediatePath;
            textBoxWorkDir.Text = settings.workingDir;
            textBoxExternalRanges.Text = settings.externalRangesPath;
            textBoxHilitePath.Text = settings.hilitePath;
        }

        private void FillRanges()
        {
            dataGridViewRanges.Rows.Clear();
            foreach(SignalInfo signal in hilite.Signals)
            {
                int row = dataGridViewRanges.Rows.Add();

                dataGridViewRanges.Rows[row].Cells[0].Value = signal.Name;
                dataGridViewRanges.Rows[row].Cells[1].Value = signal.MinimumValue;
                dataGridViewRanges.Rows[row].Cells[2].Value = signal.MaximumValue;
                dataGridViewRanges.Rows[row].Cells[3].Value = signal.DataType;
                dataGridViewRanges.Rows[row].Cells[4].Value = signal.HardMinimumValue;
                dataGridViewRanges.Rows[row].Cells[5].Value = signal.HardMaximumValue;
                dataGridViewRanges.Rows[row].Cells[6].Value = signal.Units;
                dataGridViewRanges.Rows[row].Cells[7].Value = signal.Description;
            }
        }

        private void FillPropertyMap()
        {
            propertyMap.Clear();

            IMgaMetaFCO meta = mgaFormalRequirement.Meta;
            foreach (MgaMetaAttribute metaAttr in meta.Attributes)
            {
                GMEAttribute attribute = new GMEAttribute();
                MgaAttribute attr = mgaFormalRequirement.get_Attribute(metaAttr);
                attribute.name = metaAttr.Name;
                attribute.value = attr.StringValue;
                attribute.displayedName = metaAttr.DisplayedName;

                propertyMap.Add(attribute.displayedName, attribute);
            }
        }

        private bool fillingPropertyDataGrid = false;
        private void FillPropertyDataGrid()
        {
            fillingPropertyDataGrid = true;
            dataGridViewProperties.Rows.Clear();
            foreach (var property in propertyMap)
            {
                int row = dataGridViewProperties.Rows.Add();
                dataGridViewProperties.Rows[row].Cells[0].Value = property.Key;
                dataGridViewProperties.Rows[row].Cells[1].Value = property.Value.value;
   
            }
            fillingPropertyDataGrid = false;
        }

        private void FillIntermediateRangesFromFile(string path)
        {
            if (!File.Exists(path))
            {
                dataGridViewIntermediateRanges.Columns.Clear();
                dataGridViewIntermediateRanges.Rows.Clear();
                return;
            }
            string content = null;
            using (StreamReader reader = new StreamReader(path))
            {
                content = reader.ReadToEnd();
            }

            if (content!=null)
            {
                FillIntermediateRanges(content);
            }

        }
        private void FillIntermediateRanges(string rangesFileContent)
        {
            dataGridViewIntermediateRanges.Columns.Clear();
            dataGridViewIntermediateRanges.Rows.Clear();

            string[] lines = rangesFileContent.Split(new string [1] {Environment.NewLine}, StringSplitOptions.RemoveEmptyEntries);

            if (lines.Length < 2)
            {
                return;
            }

            string[] headers = lines[1].Split(',');

            
            foreach(string header in headers)
            {
                dataGridViewIntermediateRanges.Columns.Add(header, header);
            }
            dataGridViewIntermediateRanges.ColumnHeadersVisible = true;

            if (lines.Length <= 2)
            {
                return;
            }

            for (int i = 3; i < lines.Length; i++ )
            {
                string[] items = lines[i].Split(',');
                if (items.Length != headers.Length)
                {
                    dataGridViewIntermediateRanges.Rows.Clear();
                    throw new ApplicationException("Inconsistent row and header length in the given intermediate ranges file.");                  
                }

                int row = dataGridViewIntermediateRanges.Rows.Add();
                for (int j = 0; j < items.Length; j++)
                {
                    dataGridViewIntermediateRanges.Rows[row].Cells[j].Value = items[j];
                }
            }

        }

        private void buttonSaveToCyPhy_Click(object sender, EventArgs e)
        {
            foreach (DataGridViewRow row in dataGridViewProperties.Rows)
            {
                GMEAttribute attr = propertyMap[(string)row.Cells[0].Value];
                mgaFormalRequirement.set_StrAttrByName(attr.name, (string)row.Cells[1].Value);
            }
        }

        private void dataGridViewProperties_CellValueChanged(object sender, DataGridViewCellEventArgs e)
        {
            if (fillingPropertyDataGrid)
            {
                return;
            }

            if (e.ColumnIndex != 1)
            {
                return;
            }

            int index = e.RowIndex;

            if (index < 0)
            {
                return;
            }
            

            GMEAttribute attr = propertyMap[(string)dataGridViewProperties.Rows[index].Cells[0].Value];
            attr.value = (string)dataGridViewProperties.Rows[index].Cells[1].Value;
        }

        private void dataGridViewProperties_CurrentCellDirtyStateChanged(object sender, EventArgs e)
        {
            if (fillingPropertyDataGrid)
            {
                return;
            }

            if (dataGridViewProperties.IsCurrentCellDirty)
            {
                dataGridViewProperties.CommitEdit(DataGridViewDataErrorContexts.Commit);
            }
        }

        private void DisplayMessages()
        {
            dataGridHiLiTeMessages.Rows.Clear();

          

            foreach (DefectInfo defect in hilite.defects)
            {
                int row = dataGridHiLiTeMessages.Rows.Add();
                if (defect.isError)
                {

                    dataGridHiLiTeMessages.Rows[row].Cells[0].Value = errorIcon;
                }
                else
                {

                    dataGridHiLiTeMessages.Rows[row].Cells[0].Value = warningIcon;

                }

                dataGridHiLiTeMessages.Rows[row].Cells[1].Value = defect.defaultOrder;
                dataGridHiLiTeMessages.Rows[row].Cells[2].Value = defect.description;
                dataGridHiLiTeMessages.Rows[row].Cells[3].Value = defect.block;
                dataGridHiLiTeMessages.Rows[row].Cells[4].Value = defect.port;
                dataGridHiLiTeMessages.Rows[row].Cells[5].Value = defect.count;
                dataGridHiLiTeMessages.Rows[row].Cells[6].Value = defect.model;
            }

        }

        private void buttonBrowseFolder_Click(object sender, EventArgs e)
        {
            if (folderBrowserDialog.ShowDialog() == DialogResult.OK)
            {
                textBoxWorkDir.Text = folderBrowserDialog.SelectedPath;
            }
        }

        private void buttonBrowseHiliteFile_Click(object sender, EventArgs e)
        {
            if (Directory.Exists(textBoxWorkDir.Text))
            {
                saveFileDialogProjectFile.InitialDirectory = textBoxWorkDir.Text;
            }
            if (saveFileDialogProjectFile.ShowDialog() == DialogResult.OK)
            {
                textBoxHiliteFile.Text = saveFileDialogProjectFile.FileName;
            }
        }

        private void buttonBrowseIntermediate_Click(object sender, EventArgs e)
        {
            if (Directory.Exists(textBoxWorkDir.Text))
            {
                saveFileDialogIntermediateFile.InitialDirectory = textBoxWorkDir.Text;
            }
        
            if (saveFileDialogIntermediateFile.ShowDialog() == DialogResult.OK)
            {
                textBoxIntermediatePath.Text = saveFileDialogIntermediateFile.FileName;
            }
        }

        private void buttonDelIntermediateRanges_Click(object sender, EventArgs e)
        {
            try
            {
                File.Delete(textBoxIntermediatePath.Text);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error");
            }
        }

        private void RetrieveSettings()
        {
            settings.hiliteProjectFilePath = textBoxHiliteFile.Text;
            settings.workingDir = textBoxWorkDir.Text;
            settings.intermediatePath = textBoxIntermediatePath.Text;
            settings.externalRangesPath = textBoxExternalRanges.Text;
            settings.hilitePath = textBoxHilitePath.Text;

        }

        private void buttonSaveSettings_Click(object sender, EventArgs e)
        {

            RetrieveSettings();
            mgaFormalRequirement.set_StrAttrByName(additionalToolParameterName, settings.ToString());

            FillPropertyMap();
            FillPropertyDataGrid();
        }        

        public void WriteOutput(string line)
        {
            StringBuilder output = new StringBuilder();
            string rtfLine = line.Replace("\\", "\\\\").Replace("{", "\\{").Replace("}", "\\}").Replace(" ", @"\~").Replace("\r\n", @"\par ").Replace("\n", @"\par ");
            if (line.StartsWith("[ERROR]"))
            {
                output.AppendLine(@"\cf5 " + rtfLine + @"\par ");
            }
            else if (line.StartsWith(@"****"))
            {
                output.AppendLine(@"\cf2 " + rtfLine + @"\par ");                
                result = line.Substring(4);
            }
            else
            {
                output.AppendLine(@"\cf1 " + rtfLine + @"\par "); 
            }
            outputText.Append(output);
            
            UpdateOutput();
           
        }

        delegate void UpdateOutputDelegate();
        void UpdateOutput()
        {     
            
            if (richTextBoxOutput.InvokeRequired)
            {
                richTextBoxOutput.Invoke(new UpdateOutputDelegate(this.UpdateOutput));
            }
            else
            {
                Win32.LockWindowUpdate(richTextBoxOutput.Handle);
                richTextBoxOutput.Rtf = outputText.ToString();
                richTextBoxOutput.SelectionStart = richTextBoxOutput.TextLength;
                richTextBoxOutput.ScrollToCaret();
                Win32.LockWindowUpdate((IntPtr)0);
             
            }
        }

        private void buttonGenerateInput_Click(object sender, EventArgs e)
        {
            try
            {
                RetrieveSettings();
                hilite.GenerateFiles(settings);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error");
            }
        }

        private void OutputHandler(object sendingProcess, DataReceivedEventArgs outLine)
        {
            if (!String.IsNullOrEmpty(outLine.Data))
            {
                WriteOutput(outLine.Data);
            }
        }

        private void buttonRunHilite_Click(object sender, EventArgs e)
        {
            try
            {
                RetrieveSettings();
                outputText = new StringBuilder(rtfHeader);
             
                result = String.Empty;

                richTextBoxOutput.Clear();

                buttonRunHilite.Enabled = false;
                this.Cursor = Cursors.WaitCursor;
                hilite.Execute(settings, new DataReceivedEventHandler(OutputHandler), new EventHandler(hiliteExited));


            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error");
                buttonRunHilite.Enabled = true;
                this.Cursor = Cursors.Default;
            }
     
        }

        private void hiliteExited(object sender, System.EventArgs e)
        {
            if (buttonRunHilite.InvokeRequired)
            {
                Invoke(new EventHandler(hiliteExited), new object[] { sender, e });
            }
            else
            {
                try
                {
                    
                    mgaFormalRequirement.set_StrAttrByName(verificationResultAttrName, result);

                    string resultXML = null;
                    List<DefectInfo> defects = null;

                    hilite.GetResults(settings, out resultXML, out defects);

                    mgaFormalRequirement.set_StrAttrByName(toolSpecAttrName, resultXML);

                    FillPropertyMap();
                    FillPropertyDataGrid();

                    FillMessagesDataGrid(defects);
                    
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message, "Error");
                }
                finally
                {
                    buttonRunHilite.Enabled = true;
                    this.Cursor = Cursors.Default;
                }
            }           
        }

        private void FillMessagesDataGrid(List<DefectInfo> defects)
        {
            dataGridHiLiTeMessages.Rows.Clear();

            foreach(DefectInfo defect in defects)
            {
                int row = dataGridHiLiTeMessages.Rows.Add();

                if (defect.isError)
                {
                    dataGridHiLiTeMessages.Rows[row].Cells[0].Value = errorIcon;
                }
                else
                {
                    dataGridHiLiTeMessages.Rows[row].Cells[0].Value = warningIcon;
                }

                dataGridHiLiTeMessages.Rows[row].Cells[1].Value = defect.defaultOrder;
                dataGridHiLiTeMessages.Rows[row].Cells[2].Value = defect.description;
                dataGridHiLiTeMessages.Rows[row].Cells[3].Value = defect.block;
                dataGridHiLiTeMessages.Rows[row].Cells[4].Value = defect.port;
                dataGridHiLiTeMessages.Rows[row].Cells[5].Value = defect.count;
                dataGridHiLiTeMessages.Rows[row].Cells[6].Value = defect.model;
            }

        }

        private void buttonDeleteExternalRanges_Click(object sender, EventArgs e)
        {
            try
            {
                File.Delete(textBoxExternalRanges.Text);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error");
            }
        }

        private void buttonBrowseExternalRanges_Click(object sender, EventArgs e)
        {
            if (Directory.Exists(textBoxWorkDir.Text))
            {
                saveFileDialogExternalRanges.InitialDirectory = textBoxWorkDir.Text;
            }
   

            if (saveFileDialogExternalRanges.ShowDialog() == DialogResult.OK)
            {
                textBoxExternalRanges.Text = saveFileDialogExternalRanges.FileName;
            }
        }

        private void buttonBrowseHilite_Click(object sender, EventArgs e)
        {
            const string hiliteDefaultPath = @"C:\HiLiTE\Build\bin"; // default installation directory
            if (Directory.Exists(hiliteDefaultPath)) 
            {
                openFileDialogHilitePath.InitialDirectory = hiliteDefaultPath;
            }

            if (openFileDialogHilitePath.ShowDialog() == DialogResult.OK)
            {
                textBoxHilitePath.Text = openFileDialogHilitePath.FileName;
            }
        }

        private void buttonLoadReport_Click(object sender, EventArgs e)
        {
            if (Directory.Exists(textBoxWorkDir.Text))
            {
                string path = Path.Combine(textBoxWorkDir.Text, HiLiTeGateway.summaryRelPath);
                if (Directory.Exists(path))
                {
                    openFileDialogReport.InitialDirectory = path;
                }
                else
                {
                    openFileDialogReport.InitialDirectory = textBoxWorkDir.Text;
                }
            }

            if (openFileDialogReport.ShowDialog() == DialogResult.OK)
            {
                webBrowserReport.Navigate(openFileDialogReport.FileName);
            }
        }

        private void buttonDeleteReports_Click(object sender, EventArgs e)
        {
            if (Directory.Exists(textBoxWorkDir.Text))
            {
                string path = Path.Combine(textBoxWorkDir.Text, HiLiTeGateway.summaryRelPath);
                if (Directory.Exists(path))
                {
                    if (MessageBox.Show(String.Format("Are you sure that you want to delete the directory {0} ?", path), "Confirmation", MessageBoxButtons.YesNo) == DialogResult.Yes)
                    {
                        Directory.Delete(path, true);
                    }
                }
            }
        }

        private void tabControlMain_Selecting(object sender, TabControlCancelEventArgs e)
        {
            if (e.TabPageIndex == 3)
            {
                FillIntermediateRangesFromFile(textBoxIntermediatePath.Text);
            }

        }

        private void dataGridHiLiTeMessages_CellMouseDoubleClick(object sender, DataGridViewCellMouseEventArgs e)
        {
            
            int index = e.RowIndex;
            if (index < 0)
            {
                return;
            }
            string simulinkFilePath = dataGridHiLiTeMessages.Rows[index].Cells[6].Value.ToString();
            string blockPath = dataGridHiLiTeMessages.Rows[index].Cells[3].Value.ToString();

            if(String.IsNullOrWhiteSpace(simulinkFilePath))
            {
                return;
            }

            if(String.IsNullOrWhiteSpace(blockPath))
            {
                SimulinkGateway.ExecuteMatlab(simulinkFilePath, null);
            }
            else
            {
                SimulinkGateway.ExecuteMatlab(simulinkFilePath, String.Format("{0}/{1}", 
                                            Path.GetFileNameWithoutExtension(simulinkFilePath), blockPath));
            }
        }    
    }

    public class GMEAttribute
    {
        public string name; // Internal GME-specific short name
        public string displayedName;
        public string value;
    }

}
