﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Reflection;
using System.Windows.Forms;
using System.Text.RegularExpressions;

namespace HiLiTeVerifierPlugin
{
    class SimulinkGateway
    {
        static object matlabObject = null;
        static Type matlab = null;
        private const string uriPrefix = @"simulink:///";

        public static void GetPathFormUri(string uri, out string path, out string pathToElement)
        {
            string[] parts = uri.Split('#');

            StringBuilder sbPath = new StringBuilder(parts[0]);
            sbPath.Remove(0, uriPrefix.Length);
            sbPath.Replace('/', Path.DirectorySeparatorChar);
            path = sbPath.ToString();
            pathToElement = parts.Length>1? parts[1]:"";
        }


        public static string ResolveURI(string fullPath, string currentDir)
        {
            if (currentDir.EndsWith(Path.DirectorySeparatorChar.ToString()))
            {
                currentDir = currentDir.Substring(0, currentDir.Length -1);
            }

            
            // Handling ../
            Regex regexp = new Regex(@"(\.\./)+");            
            Match match = regexp.Match(fullPath);
            if (match.Success)
            {
                string upDir = currentDir;
                for (int i = 0; i < match.Groups.Count; i++)
                {
                    upDir = Directory.GetParent(upDir).FullName;
                }

                StringBuilder newPath = new StringBuilder(fullPath.Substring(0, match.Index));
                newPath.Append(upDir.Replace(Path.DirectorySeparatorChar, '/'));
                newPath.Append('/');
                newPath.Append(fullPath.Substring(match.Index + match.Length));

                fullPath = newPath.ToString();

            }
            else
            {
                // Handling ./            
                fullPath = fullPath.Replace("./", currentDir.Replace(Path.DirectorySeparatorChar, '/') + '/');
            }
            return fullPath;
        }



        public static void ExecuteMatlab(string localPath, string pathToElement)
        {
            string commandLine = null;
            if (!String.IsNullOrEmpty(pathToElement))
            {

                commandLine = String.Format("cd {0};load_system('{1}');open_system('{1}');set_param({1},'HiliteAncestors','off');open_and_hilite_system('{2}')", Path.GetDirectoryName(localPath), Path.GetFileNameWithoutExtension(localPath), pathToElement);
            }
            else
            {
                commandLine = String.Format("cd {0};load_system('{1}');open_system('{1}');set_param({1},'HiliteAncestors','off')", Path.GetDirectoryName(localPath), Path.GetFileNameWithoutExtension(localPath));
            }



            if (matlabObject == null)
            {
                matlab = Type.GetTypeFromProgID("Matlab.Application.single");
                matlabObject = Activator.CreateInstance(matlab);
            }

            object[] parameter = new object[1];
            parameter[0] = commandLine;
            string result = null;
            try
            {
                matlab.InvokeMember("Visible", BindingFlags.SetProperty, null, matlabObject, new object[] { 0 });
                result = (string)matlab.InvokeMember("Execute", BindingFlags.InvokeMethod, null, matlabObject, parameter);
            }
            catch (System.Runtime.InteropServices.COMException ex)
            {
                // HRESULT 800706BA (The COM server is not valid, e.g. someone quits MATLAB. We try to reuse the previous matlab automation object as much as possible.
                if (ex.ErrorCode == -2147023174)
                {
                    matlabObject = Activator.CreateInstance(matlab);
                    result = (string)matlab.InvokeMember("Execute", BindingFlags.InvokeMethod, null, matlabObject, parameter);
                }
            }

            if (result.Contains("??? Error using ==> load_system"))
            {
                MessageBox.Show(String.Format("Matlab cannot load the specified Simulink model file. Path: {0}.", localPath), "Error");

            }
            else if (result.Contains("??? Error using ==> hilite_system") || result.Contains("??? Error using ==> open_and_hilite_system"))
            {
                MessageBox.Show(String.Format("Matlab cannot find the specified model element. Element: {0} Path: {1}.", pathToElement, localPath),"Error");
            }
            else if(result.Contains("??? Error"))
            {
                MessageBox.Show(String.Format("Matlab cannot load the specified Simulink model.{0}{1}",Environment.NewLine, result.Replace("\n", Environment.NewLine)), "Error");
            }

        }

        public static void Quit()
        {
            try
            {
                if (matlabObject != null && matlab != null)
                {
                    matlab.InvokeMember("Quit", BindingFlags.InvokeMethod, null, matlabObject, null);
                    matlabObject = null;
                    matlab = null;
                }
            }
            catch { } // We are exiting now, tried our best to quit Matlab, nothing to do here
        }

    }
}
