﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using GME.MGA;


namespace HybridSALVerifierPlugin
{
    class CyPhyTraverser
    {
        private MgaModel rootSubsystem;
        private StateSpace stateSpace;
        

        public StateSpace StateSpace
        {
            get { return stateSpace; }
        }


        public CyPhyTraverser(MgaModel rootSubsystem)
        {
            if (rootSubsystem.Meta.Name != "Subsystem")
            {
                throw new ApplicationException("CyPhyTraverser needs Subsystem as rootSubsystem");
            }

                       
            this.rootSubsystem = rootSubsystem;
        }

        public string Translate(string fileNameNoExtension,  string toolSpecificDescription)
        {
            // Building State Space
            stateSpace = new StateSpace();
            BuildStateSpace(rootSubsystem); // Columns of the matrix
            FillComponentCharacteristics();
            FillIntegrators();
            FillOutput();
            ResolveSaturations();

            stateSpace.Dump();

            stateSpace.ode.Eliminate();
            stateSpace.ode.Dump();

            HybridSALGenerator generator = new HybridSALGenerator(stateSpace, fileNameNoExtension, toolSpecificDescription);
            return generator.Generate();
        }


        private void BuildStateSpace(MgaModel subsystem)
        {
            
            List<MgaModel> intermediateVariablePorts = new List<MgaModel>();         
            List<MgaModel> outputVariablePorts = new List<MgaModel>();
            List<MgaModel> derivativeVariablePorts = new List<MgaModel>();   
            List<MgaModel> stateVariablePorts = new List<MgaModel>();   
            List<MgaModel> inputVariablePorts = new List<MgaModel>();
           

            #region COLLECT VARIABLES
            MgaFCOs childFCOs = subsystem.ChildFCOs;
            foreach (MgaFCO fco in childFCOs)
            {
                string kind = fco.Meta.Name;
                if(kind == "Primitive")
                {
                    string blockType = fco.get_StrAttrByName("BlockType");
                    if (blockType == "BusCreator" || blockType == "Scope" || blockType == "Sin")
                    {
                        continue;
                    }
                    if (blockType == "Integrator")
                    {
                       stateVariablePorts.Add((fco as MgaModel).GetChildrenOfKind("OutputPort")[1] as MgaModel);                    
                       stateSpace.integrators.Add(new Integrator(fco.Name, fco as MgaModel));   
                 
                        MgaModel integratorBlock = fco as MgaModel;
                        MgaFCOs inputPorts = integratorBlock.GetChildrenOfKind("InputPort");

                        foreach (MgaFCO inputPort in inputPorts)
                        {
                            if (inputPort.GetIntAttrByNameDisp("Number") == 1)
                            {
                                derivativeVariablePorts.Add(inputPort as MgaModel);
                            }
                        }
                    }
                    else if (blockType == "Sum")
                    {
                        MgaModel sum = fco as MgaModel;
                        MgaFCOs outputPorts = sum.GetChildrenOfKind("OutputPort");
                        MgaFCO outputPort = outputPorts[1];
                        intermediateVariablePorts.Add(outputPort as MgaModel); 
                    }
                    else if (blockType == "Constant")
                    {
                        intermediateVariablePorts.Add((fco as MgaModel).GetChildrenOfKind("OutputPort")[1] as MgaModel);
                    }
                    else if (blockType == "Gain")
                    {
                        intermediateVariablePorts.Add((fco as MgaModel).GetChildrenOfKind("OutputPort")[1] as MgaModel); 
                    }
                   else if (blockType == "Saturate")
                    {
                        intermediateVariablePorts.Add((fco as MgaModel).GetChildrenOfKind("OutputPort")[1] as MgaModel);
                        stateSpace.saturations.Add(new Saturation(fco.Name,fco as MgaModel));

                    }
                    else
                    {
                        // Warning on Console
                        System.Diagnostics.Debug.Assert(false);
                    }
                }
               
                else if (kind == "InputPort")
                {
                    inputVariablePorts.Add(fco as MgaModel); 
                }
                else if (kind == "OutputPort")
                {
                    outputVariablePorts.Add(fco as MgaModel); 
                }
                else if (kind == "SubSystem")
                {
                    // Hierarchical models are not supported
                    // Warning on Console
                    System.Diagnostics.Debug.Assert(false);

                }
                else if (kind == "Parameter" || kind == "Line")
                {
                    continue;
                }
                else
                {
                    System.Diagnostics.Debug.Assert(false);
                    // Warning on Console
                }

            }
            #endregion

            // Setting up the matrix
            int n = intermediateVariablePorts.Count + outputVariablePorts.Count + stateVariablePorts.Count + derivativeVariablePorts.Count
                                                    + inputVariablePorts.Count + 1; // One column constant at the end
            int m = intermediateVariablePorts.Count + stateVariablePorts.Count + outputVariablePorts.Count;
            stateSpace.ode = new Matrix(m, n);


            // Setting up the columns
            stateSpace.columnOutputPorts.AddRange(intermediateVariablePorts);
            stateSpace.outputIndex = stateSpace.columnOutputPorts.Count;

            stateSpace.columnOutputPorts.AddRange(outputVariablePorts);
            stateSpace.derivativeIndex = stateSpace.columnOutputPorts.Count;

            stateSpace.columnOutputPorts.AddRange(derivativeVariablePorts);
            stateSpace.variableIndex = stateSpace.columnOutputPorts.Count;

            stateSpace.columnOutputPorts.AddRange(stateVariablePorts);
            stateSpace.inputIndex = stateSpace.columnOutputPorts.Count;

            stateSpace.columnOutputPorts.AddRange(inputVariablePorts);

            stateSpace.BuildPort2ColumnMapAndVariableNames();

        }

        private void FillComponentCharacteristics()
        {
            // Component characteristics
            for (int i = 0; i < stateSpace.outputIndex; i++)
            {
                MgaModel block = stateSpace.columnOutputPorts[i].ParentModel;

                string blockType = block.get_StrAttrByName("BlockType");
              
                if (blockType == "Sum")
                {
                    List<MgaModel> inputPorts = GetInputPorts(block);
                    System.Diagnostics.Debug.Assert(inputPorts.Count > 1);

                    stateSpace.ode[i, i] = -1;

                    string inputs = GetParameter(block, "Inputs").GetStrAttrByNameDisp("Value");

                    foreach(MgaModel inputPort in inputPorts)
                    {
                        MgaModel connOutputPort = GetConnectedOutputPort(inputPort);
                        int idx = stateSpace.outputPort2Column[connOutputPort];
                        int portNum = inputPort.GetIntAttrByNameDisp("Number");

                        if (inputs[portNum-1] == '+')
                        {
                            stateSpace.ode[i, idx] = 1;
                        }
                        else
                        {
                            stateSpace.ode[i, idx] = -1;
                        }
                    }
                }
                else if (blockType == "Constant")
                {
                    string value = GetParameter(block, "Value").GetStrAttrByNameDisp("Value");
                    double val = Double.Parse(value);

                    stateSpace.ode[i, i] = -1; 
                    stateSpace.ode[i, stateSpace.ode.ColumnCount-1] = val;

                }
                else if (blockType == "Gain")
                {
                    List<MgaModel> inputPorts = GetInputPorts(block);
                    System.Diagnostics.Debug.Assert(inputPorts.Count == 1);

                    stateSpace.ode[i, i] = -1;

                    string strGain = GetParameter(block, "Gain").GetStrAttrByNameDisp("Value");
                    double gain = Double.Parse(strGain);

                    MgaModel connOutputPort = GetConnectedOutputPort(inputPorts[0]);

                    int idx = stateSpace.outputPort2Column[connOutputPort];
                    stateSpace.ode[i, idx] = gain;                                
                }
                else if (blockType == "Saturate")
                {
                    List<MgaModel> inputPorts = GetInputPorts(block);
                    System.Diagnostics.Debug.Assert(inputPorts.Count == 1);

                    stateSpace.ode[i, i] = -1;
                   

                    MgaModel connOutputPort = GetConnectedOutputPort(inputPorts[0]);

                    int idx = stateSpace.outputPort2Column[connOutputPort];
                    stateSpace.ode[i, idx] = 1;         
                }
                else
                {
                    // Warning on Console
                    System.Diagnostics.Debug.Assert(false);
                }
            }
        }

        private void FillIntegrators()
        {
            for (int i = stateSpace.derivativeIndex; i < stateSpace.variableIndex; i++)
            {
                Integrator integrator = stateSpace.integrators[i - stateSpace.derivativeIndex];


                List<MgaModel> inputPorts = GetInputPorts(integrator.block);

                foreach (MgaModel inputPort in inputPorts)
                {
                    if (inputPort.GetIntAttrByNameDisp("Number") == 1)
                    {
                        integrator.derivativePos = stateSpace.outputPort2Column[inputPort];
                        stateSpace.ode[i, i] = -1;                      

                        MgaModel connOutputPort = GetConnectedOutputPort(inputPort);
                        int idx = stateSpace.outputPort2Column[connOutputPort];
                        stateSpace.ode[i, idx] = 1;                         
                    }

                    if (inputPort.GetIntAttrByNameDisp("Number") == 2)
                    {
                        integrator.resetVariablePos = stateSpace.outputPort2Column[GetConnectedOutputPort(inputPort)];

                    }

                    if (inputPort.GetIntAttrByNameDisp("Number") == 3)
                    {
                        integrator.initialValuePos = stateSpace.outputPort2Column[GetConnectedOutputPort(inputPort)];
                    }

                    integrator.stateVariablePos =
                        stateSpace.outputPort2Column[integrator.block.GetChildrenOfKind("OutputPort")[1] as MgaModel];
                }

                // Adding saturation
                MgaAtom parameter = GetParameter(integrator.block, "UpperSaturationLimit");
                if (parameter != null)
                {
                    string value = parameter.GetStrAttrByNameDisp("Value");
                    double val = Double.Parse(value);
                    integrator.upperSaturation = val;
                }

                parameter = GetParameter(integrator.block, "LowerSaturationLimit");
                if (parameter != null)
                {
                    string value = parameter.GetStrAttrByNameDisp("Value");
                    double val = Double.Parse(value);
                    integrator.lowerSaturation = val;
                }

            }

        }

        private void FillOutput()
        {
            for (int i = stateSpace.outputIndex; i < stateSpace.derivativeIndex; i++)
            {
                MgaModel inputPorts = stateSpace.columnOutputPorts[i];             

                stateSpace.ode[i, i] = -1;

                MgaModel connOutputPort = GetPeerConnectedOutputPort(inputPorts);
                int idx = stateSpace.outputPort2Column[connOutputPort];
                stateSpace.ode[i, idx] = 1;                               
            }
        }

        private void ResolveSaturations()
        {
            foreach (Saturation saturation in stateSpace.saturations)
            {
                MgaModel outputPort = saturation.block.GetChildrenOfKind("OutputPort")[1] as MgaModel;
                saturation.variablePos = stateSpace.outputPort2Column[outputPort];
            }
        }

    /*    private void ResolveConstantInputs()
        {
            for (int i = stateSpace.inputIndex; i < stateSpace.ode.ColumnCount - 2; i++)
            {
                MgaModel outputPort = GetConnectedOutputPort(stateSpace.columnOutputPorts[i]);
                MgaModel block = outputPort.ParentModel;

                if (block.Meta.Name == "Primitive" || block.get_StrAttrByName("BlockType") == "Constant")
                {
                    string value = GetParameter(block, "Value").GetStrAttrByNameDisp("Value");
                    double val = Double.Parse(value);

                    // Add a row to ode

                }                
            }
         
        }
     * */


        private static List<MgaModel> GetInputPorts(MgaModel block)
        {
            List<MgaModel> inputPorts = new List<MgaModel>();
            MgaFCOs inPorts = block.GetChildrenOfKind("InputPort");
            foreach (MgaFCO inPort in inPorts)
            {
                inputPorts.Add(inPort as MgaModel);
            }

            return inputPorts;
        }

        private static MgaAtom GetParameter(MgaModel block, string parameterName)
        {

            MgaFCOs parameters = block.GetChildrenOfKind("Parameter");
            foreach (MgaFCO parameter in parameters)
            {
                if (parameter.Name == parameterName)
                {
                    return parameter as MgaAtom;
                }
            }

            return null;
        }


        private static MgaModel GetConnectedOutputPort(MgaModel inputPort)
        {
            List<MgaModel> outputPorts = new List<MgaModel>();

            MgaConnPoints connPoints = inputPort.PartOfConns;

            foreach (MgaConnPoint connPoint in connPoints)
            {
                if (connPoint.Owner.Meta.Name == "Line")
                {
                    MgaSimpleConnection connection = (MgaSimpleConnection)connPoint.Owner;
                    MgaModel outputPort = connection.Src as MgaModel;

                    outputPorts.Add(outputPort);
                }
            }

            System.Diagnostics.Debug.Assert(outputPorts.Count == 1);
            return outputPorts[0];
        }


        private static MgaModel GetPeerConnectedOutputPort(MgaModel outputPort)
        {
            List<MgaModel> outputPorts = new List<MgaModel>();

            MgaConnPoints connPoints = outputPort.PartOfConns;

            foreach (MgaConnPoint connPoint in connPoints)
            {
                if (connPoint.Owner.Meta.Name == "Line")
                {
                    MgaSimpleConnection connection = (MgaSimpleConnection)connPoint.Owner;
                    MgaModel outputPort2 = connection.Src as MgaModel;
                    if (outputPort != outputPort2)
                    {
                        outputPorts.Add(outputPort2);
                    }
                }
            }

            System.Diagnostics.Debug.Assert(outputPorts.Count == 1);
            return outputPorts[0];
        }

        


    }


}