﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/




using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace HybridSALVerifierPlugin
{
    public class HybridSALGenerator
    {
        private StateSpace stateSpace;
        private string contextName;
        private string theorem;

        private StringBuilder hsCode;

        public HybridSALGenerator(StateSpace stateSpace, string contextName, string theorem)
        {
            this.stateSpace = stateSpace;
            this.contextName = contextName;
            this.theorem = theorem;
        }

        public string Generate()
        {
            hsCode = new StringBuilder(3000);
            AppendHeader();

            AppendVariableDeclarations();
            AppendInvariants();
            AppendInitFormulas();

            AppendTransitions();

            AppendModuleClosing();
            AppendTheorem();
            AppendFooter();
            return hsCode.ToString();
        }


        private void AppendVariableDeclarations()
        {
            for (int i = stateSpace.outputIndex; i < stateSpace.ode.ColumnCount - 1; i++)
            {
                if (i >= stateSpace.outputIndex && i < stateSpace.derivativeIndex) // Outputs
                {
                    hsCode.AppendFormat("  OUTPUT {0}, {0}dot : REAL", stateSpace.columnVariableNames[i]);
                    hsCode.AppendLine();
                }

                if (i >= stateSpace.derivativeIndex && i < stateSpace.inputIndex) // State variables and their derivatives
                {
                    hsCode.AppendFormat("  LOCAL {0} : REAL", stateSpace.columnVariableNames[i]);
                    hsCode.AppendLine();
                }

                if (i >= stateSpace.inputIndex) // Inputs
                {
                    hsCode.AppendFormat("  INPUT {0}, {0}dot : REAL", stateSpace.columnVariableNames[i]);
                    hsCode.AppendLine();
                }
            }

            foreach (Integrator integrator in stateSpace.integrators)
            {
                hsCode.AppendFormat("  LOCAL {0}_stateNormal : BOOLEAN", integrator.name);
                hsCode.AppendLine();

                hsCode.AppendFormat("  LOCAL {0}_initialValue: REAL", integrator.name);
                hsCode.AppendLine();

                hsCode.AppendFormat("  LOCAL {0}_reset : REAL", integrator.name);
                hsCode.AppendLine();


                if (integrator.lowerSaturation != Double.NaN)
                {
                    hsCode.AppendFormat("  LOCAL {0}_lowerSaturation : REAL", integrator.name);
                    hsCode.AppendLine();
                }
                if (integrator.upperSaturation != Double.NaN)
                {
                    hsCode.AppendFormat("  LOCAL {0}_upperSaturation : REAL", integrator.name);
                    hsCode.AppendLine();
                }
            }
        }

        private void AppendTransitions()
        {
            hsCode.AppendLine();
            hsCode.AppendLine(@"  TRANSITION");
            hsCode.AppendLine(@"    [");

            bool first = true;
            foreach (Integrator integrator in stateSpace.integrators)
            {

                if (first)
                {
                    first = false;
                }
                else
                {
                    hsCode.AppendLine(@"    []");
                }

                hsCode.AppendLine();
                hsCode.AppendFormat("    %% Differential equations in normal mode for integrator {0}", integrator.name);
                hsCode.AppendLine();
                hsCode.AppendFormat("    {0}_stateNormal = TRUE ", integrator.name);

                if (integrator.upperSaturation != Double.NaN && integrator.lowerSaturation != Double.NaN)
                {
                    hsCode.AppendFormat(" AND {1}<= {0}_upperSaturation AND {1} >= {0}_lowerSaturation", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);
                }
                else if (integrator.upperSaturation != Double.NaN)
                {
                    hsCode.AppendFormat(" AND {1} <= {0}_upperSaturation", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);
                }
                else if (integrator.lowerSaturation != Double.NaN)
                {
                    hsCode.AppendFormat(" AND {1} >= {0}_lowerSaturation", integrator.name);
                }

                hsCode.AppendLine(@" -->");
                hsCode.AppendFormat("       {0}' = {1}", stateSpace.columnVariableNames[integrator.derivativePos], GetEquation(integrator.derivativePos));
                hsCode.AppendLine();
                hsCode.AppendLine(@"    []");

                if (integrator.upperSaturation != Double.NaN || integrator.lowerSaturation != Double.NaN)
                {
                    hsCode.AppendLine();
                    hsCode.AppendFormat("    %% Differential equations for saturated mode for integrator {0}", integrator.name);
                    hsCode.AppendLine();
                    hsCode.AppendFormat("    {0}_stateNormal = FALSE ", integrator.name);

                    if (integrator.upperSaturation != Double.NaN && integrator.lowerSaturation != Double.NaN)
                    {
                        hsCode.AppendFormat(" AND ({1}>= {0}_upperSaturation OR {1} <= {0}_lowerSaturation)", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);
                    }
                    else if (integrator.upperSaturation != Double.NaN)
                    {
                        hsCode.AppendFormat(" AND {1} >= {0}_upperSaturation", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);
                    }
                    else if (integrator.lowerSaturation != Double.NaN)
                    {
                        hsCode.AppendFormat(" AND {1} <= {0}_lowerSaturation", integrator.name);
                    }

                    hsCode.AppendLine(@" -->");
                    hsCode.AppendFormat("       {0}' = {1}", stateSpace.columnVariableNames[integrator.derivativePos], "0");
                    hsCode.AppendLine();
                    hsCode.AppendLine(@"    []");

                    hsCode.AppendLine();
                    hsCode.AppendFormat("    %% Discrete transition: entering saturated mode for integrator {0}", integrator.name);
                    hsCode.AppendLine();
                    hsCode.AppendFormat("    {0}_stateNormal = TRUE ", integrator.name);

                    if (integrator.upperSaturation != Double.NaN && integrator.lowerSaturation != Double.NaN)
                    {
                        hsCode.AppendFormat(" AND ({1}>= {0}_upperSaturation OR {1} <= {0}_lowerSaturation)", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);
                    }
                    else if (integrator.upperSaturation != Double.NaN)
                    {
                        hsCode.AppendFormat(" AND {1} >= {0}_upperSaturation", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);
                    }
                    else if (integrator.lowerSaturation != Double.NaN)
                    {
                        hsCode.AppendFormat(" AND {1} <= {0}_lowerSaturation", integrator.name);
                    }

                    hsCode.AppendLine(@" -->");
                    hsCode.AppendFormat("       {0}_stateNormal' = FALSE", integrator.name);
                    hsCode.AppendLine();
                    hsCode.AppendLine(@"    []");

          

                }

                // Reset
                hsCode.AppendLine();
                hsCode.AppendFormat("    %% Discrete transition: exiting saturated mode by reset for integrator {0}", integrator.name);
                hsCode.AppendLine();
                hsCode.AppendFormat("    {0}_stateNormal = FALSE ", integrator.name);


                hsCode.AppendFormat(" AND({0}_reset > 0) ", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);                           

                hsCode.AppendLine(@" -->");
                hsCode.AppendFormat("       {0}_stateNormal' = TRUE;", integrator.name);
                hsCode.AppendLine();
                hsCode.AppendFormat("       {1}' = {0}_initialValue ", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);
                hsCode.AppendLine();
                hsCode.AppendLine(@"    []");


 

                hsCode.AppendLine();
                hsCode.AppendFormat("    %% Discrete transition:  reset for integrator {0}", integrator.name);
                hsCode.AppendLine();
                hsCode.AppendFormat("    {0}_stateNormal = TRUE ", integrator.name);


                hsCode.AppendFormat(" AND({0}_reset > 0) ", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);


                hsCode.AppendLine(@" -->");
                hsCode.AppendFormat("       {0}_stateNormal' = TRUE;", integrator.name);
                hsCode.AppendLine();
                hsCode.AppendFormat("       {1}' = {0}_initialValue ", integrator.name, stateSpace.columnVariableNames[integrator.stateVariablePos]);

                
                hsCode.AppendLine();


            }


            hsCode.AppendLine(@"    ]");

        }

        private void AppendInvariants()
        {           
            hsCode.AppendLine();
            hsCode.AppendLine("INVARIANT");
            foreach (Integrator integrator in stateSpace.integrators)
            {
                if (integrator.upperSaturation != Double.NaN && integrator.lowerSaturation != Double.NaN)
                {
                    hsCode.AppendFormat("   {0}_initialValue <= {0}_upperSaturation AND {0}_initialValue >= {0}_lowerSaturation AND", integrator.name);
                    hsCode.AppendLine();
                }
                else if (integrator.upperSaturation != Double.NaN)
                {
                    hsCode.AppendFormat("   {0}_initialValue <= {0}_upperSaturation AND", integrator.name);
                    hsCode.AppendLine();
                }
                else if (integrator.lowerSaturation != Double.NaN)
                {
                    hsCode.AppendFormat("   {0}_initialValue >= {0}_lowerSaturation AND", integrator.name);
                    hsCode.AppendLine();

                }
            }

            for (int i = stateSpace.outputIndex; i < stateSpace.derivativeIndex; i++)
            {
                hsCode.AppendFormat("   {0} = ", stateSpace.columnVariableNames[i]);
                hsCode.AppendLine(GetEquation(i));
            }

            hsCode.AppendLine();

        }

        private void AppendInitFormulas()
        {
            hsCode.AppendLine();
            hsCode.AppendLine("INITFORMULA");
            bool first = true;

            foreach (Integrator integrator in stateSpace.integrators)
            {

                if(first == true)
                {
                    first = false;
                }
                else
                {
                    hsCode.AppendLine(" AND");
                }

                hsCode.AppendFormat("  {0}_stateNormal = TRUE ", integrator.name);
                

                if (integrator.upperSaturation != Double.NaN)
                {
                    hsCode.AppendLine(" AND");
                    hsCode.AppendFormat("  {0}_upperSaturation = {1}", integrator.name, integrator.upperSaturation);

                }
                if (integrator.lowerSaturation != Double.NaN)
                {
                    hsCode.AppendLine(" AND");
                    hsCode.AppendFormat("  {0}_lowerSaturation = {1}", integrator.name, integrator.lowerSaturation);
                }

                // Initial value
                hsCode.AppendLine(" AND");

                int index = integrator.initialValuePos;

                hsCode.AppendFormat("  {0}_initialValue =", integrator.name);
                hsCode.Append(GetEquation(index));
                hsCode.AppendLine(" AND");

                hsCode.AppendFormat("  {0} = {1}_initialValue", stateSpace.columnVariableNames[integrator.stateVariablePos], integrator.name);

                hsCode.AppendLine();
            }

        }

        private void AppendModuleClosing()
        {
            hsCode.AppendLine(@" END;");
        }

        private void AppendHeader()
        {
            hsCode.AppendFormat(
            @"%% --------------------------------------------------------------------
%% Generated by CyPhy2HybridSAL {0}.
%% ISIS, Vanderbilt University
%% --------------------------------------------------------------------

%% This is HYBRID SAL, differs from standard SAL.

{1}:CONTEXT = 
BEGIN", 
            DateTime.Now, contextName);
            hsCode.AppendLine();

            hsCode.AppendLine(@" CyPhyComponent : MODULE =");
            hsCode.AppendLine(@" BEGIN");

        }

        private void AppendTheorem()
        {
            hsCode.AppendFormat(@" correct: THEOREM
    {0};", theorem);
            hsCode.AppendLine();

        }

        private void AppendFooter()
        {
            hsCode.AppendLine(@"END");
        }

    
    
       private string GetEquation(int row)
       {
            StringBuilder equation = new StringBuilder(stateSpace.ode.ColumnCount * 4);
            bool first2 = true;
            for (int i = stateSpace.variableIndex; i < stateSpace.ode.ColumnCount; i++)
            {
                double coefficient = stateSpace.ode[row, i];
                if (coefficient != 0.0)
                {
                    if (first2)
                    {
                        first2 = false;
                    }
                    else
                    {
                        if (coefficient > 0)
                        {
                            equation.Append(" + ");
                        }
                    }

                    equation.Append(coefficient);
                    

                    if (i != stateSpace.ode.ColumnCount - 1) // Constant column
                    {
                        equation.Append("*");
                        equation.Append(stateSpace.columnVariableNames[i]);
                    }
                }

                if (i == stateSpace.ode.ColumnCount - 1 && String.IsNullOrEmpty(equation.ToString())) // Constant column
                {
                    equation.Append(coefficient); // If only a constant
                }
            }
            return equation.ToString();
       }

    }


 


   
}
