﻿
/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using GME.MGA;
using GME.MGA.Meta;
using System.IO;
using System.Threading;
using GME.CSharp;
using System.Runtime.InteropServices;
using HWND = System.IntPtr;
using System.Diagnostics;
using System.Collections;

namespace HybridSALVerifierPlugin
{
   
    public partial class MainForm : Form
    {

        private const string rtfHeader = @"{\rtf1\ansi\ansicpg1252\deff0\deflang1033{\fonttbl{\f0\fnil\fcharset0 Consolas;}}" +
                     @"{\colortbl ;\red0\green0\blue0;\red0\green128\blue0;\red0\green0\blue255;\red43\green145\blue177;\red139\green0\blue0;\red205\green205\blue0;\red128\green128\blue128;}\viewkind4\uc1\pard\cf1\f0\fs20";

        private const string additionalToolParameterName = @"AdditionalToolParameters";
        private const string verificationResultAttrName = @"VerificationResult";

      

        private string result;

        private Settings settings;

        private StringBuilder outputText;
        private StringBuilder errorText;
        
        public MgaFCO mgaFormalRequirement;
        private string salFileName;
        private string projectDir;

        public MainForm(MgaFCO mgaFormalRequirement)
        {
            this.mgaFormalRequirement = mgaFormalRequirement;
            propertyMap = new SortedDictionary<string, GMEAttribute>();
            settings = new Settings(mgaFormalRequirement.get_StrAttrByName(additionalToolParameterName));


            InitializeComponent();

            FillSettings();
        }

        private void buttonOK_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            // Setting the output to the folder of the current model
            string projectPath = mgaFormalRequirement.Project.ProjectConnStr;
            projectPath = projectPath.Substring(4);
            projectDir = Path.GetDirectoryName(projectPath);
            

            salFileName = Path.GetFileNameWithoutExtension(projectPath) + ".hsal";
            //buttonRunTool.Enabled = false;
            FillPropertyMap();
            FillDataGrid();

        }

        SortedDictionary<string, GMEAttribute> propertyMap;

        private void FillPropertyMap()
        {
            propertyMap.Clear();

            MgaMetaFCO meta = mgaFormalRequirement.Meta;
            foreach (MgaMetaAttribute metaAttr in meta.Attributes)
            {
                GMEAttribute attribute = new GMEAttribute();
                MgaAttribute attr = mgaFormalRequirement.get_Attribute(metaAttr);
                attribute.name = metaAttr.Name;
                attribute.value = attr.StringValue;
                attribute.displayedName = metaAttr.DisplayedName;

                propertyMap.Add(attribute.displayedName, attribute);
            }
        }
        private bool fillingDataGrid = false;
        private void FillDataGrid()
        {
            fillingDataGrid = true;
            dataGridViewProperties.Rows.Clear();
            foreach (var property in propertyMap)
            {
                int row = dataGridViewProperties.Rows.Add();
                dataGridViewProperties.Rows[row].Cells[0].Value = property.Key;
                dataGridViewProperties.Rows[row].Cells[1].Value = property.Value.value;
            }
            fillingDataGrid = false;
        }


        private void GenerateHSFile(MgaModel rootSubsystem)
        {
            

            if (rootSubsystem == null || rootSubsystem.Meta.Name != "Subsystem")
            {
                MessageBox.Show("The META Component to which the selected Formal Requirement is attached to does have a ESMoL subsystem reference, but it does not refer to an Subsystem.");
                return;
            }


            if(rootSubsystem.GetChildrenOfKind("InputPort").Count == 0)
            {
                MgaFCOs subsystems = rootSubsystem.GetChildrenOfKind("Subsystem");

                if (subsystems.Count == 0)
                {
                    MessageBox.Show("Cannot find a component with an input port");
                    return;
                }
                rootSubsystem = null;

                foreach (MgaFCO subsystem in subsystems)
                {
                    if ((subsystem as MgaModel).GetChildrenOfKind("InputPort").Count != 0)
                    {
                        rootSubsystem = subsystem as MgaModel;
                    }
                }

                if (rootSubsystem == null)
                {
                    MessageBox.Show("Cannot find a component with an input port");
                    return;
                }
            }

            CyPhyTraverser traverser = new CyPhyTraverser(rootSubsystem);
            string result = traverser.Translate(Path.GetFileNameWithoutExtension(salFileName), propertyMap["ToolSpecificDescription"].value);

            System.IO.StreamWriter file = new System.IO.StreamWriter(Path.Combine(projectDir, salFileName));
            try
            {
                file.WriteLine(result);
            }
            finally
            {
                file.Close();
            }

        }

 
        private void buttonRunTool_Click(object sender, EventArgs e)
        {

            outputText = new StringBuilder(rtfHeader);
            errorText = new StringBuilder(rtfHeader);
            result = @"Errors";
            richTextBoxOutput.Clear();

            try
            {
                MgaModel component = mgaFormalRequirement.ParentModel;

                MgaFCOs esmolLinks = component.GetChildrenOfKind("ESMoLLink");

                if (esmolLinks.Count != 1)
                {
                    throw new ApplicationException("The component containing the selected formal requirement object must contain exactly one ESMoL Link.");
                }

                MgaFCO esmolLink = esmolLinks[1];
                string uri = esmolLink.get_StrAttrByName("URI");

                string projectPath = mgaFormalRequirement.Project.ProjectConnStr;
                projectPath = projectPath.Substring(4);
                uri = MgaURL.Resolve(uri, Path.GetDirectoryName(projectPath));

                string mgaPath = MgaURL.GetFilePath(uri);

                if (!File.Exists(mgaPath))
                {
                    throw new ApplicationException(String.Format("The referenced file {0} does not exist.", mgaPath));
                }

                MgaProject project = new MgaProject();

                bool romode;
                project.Open("MGA=" + mgaPath, out romode);

                IMgaTerritory territory = project.BeginTransactionInNewTerr(transactiontype_enum.TRANSACTION_READ_ONLY);
                try
                {

                    MgaModel rootSubsystem = MgaGateway.GetObjectByPath(project, MgaURL.GetInternalPath(uri)) as MgaModel;

                    if (rootSubsystem == null)
                    {
                        throw new ApplicationException(String.Format("The referenced path {0} within the file {1} does not exist.", MgaURL.GetInternalPath(uri), mgaPath));
                    }

                    GenerateHSFile(rootSubsystem);
                }
                finally
                {
                    project.AbortTransaction();
                    territory.Destroy();
                }


                /////////////// File has been generated, let's prepare the UI and run the tools
                dataGridViewProperties.Enabled = false;
                buttonRunTool.Enabled = false;
                buttonSaveToCyPhy.Enabled = false;
                buttonBrowseHybridSALFolder.Enabled = false;
                label1.Enabled = false;
                textBoxHSFolder.Enabled = false;

                this.Cursor = Cursors.WaitCursor;

                ExecuteCommands();

            }
            catch (Exception ex)
            {
                WriteOutput(ex.Message, true);

                dataGridViewProperties.Enabled = true;
                buttonRunTool.Enabled = true;
                buttonSaveToCyPhy.Enabled = true;
                buttonBrowseHybridSALFolder.Enabled = true;
                label1.Enabled = true;
                textBoxHSFolder.Enabled = true;
                this.Cursor = Cursors.Default;
            }

            

            
        }


        private void OutputHandler(object sendingProcess, DataReceivedEventArgs outLine)
        {
            
            if (!String.IsNullOrEmpty(outLine.Data))
            {             
                WriteOutput(outLine.Data);
            }
        }

        private void ErrorHandler(object sendingProcess, DataReceivedEventArgs outLine)
        {
            if (!String.IsNullOrEmpty(outLine.Data))
            {

                WriteOutput(outLine.Data, true);
            }
        }

        private void ExecuteCommands()
        {
            string command = "cmd";
            

            Process shell = new Process();


            ProcessStartInfo psInfo = shell.StartInfo;
            psInfo.FileName = command;
            //psInfo.Arguments = args;
            psInfo.WorkingDirectory = projectDir;

          
            
            foreach (DictionaryEntry variable in Environment.GetEnvironmentVariables(EnvironmentVariableTarget.Machine))
            {
                if (!psInfo.EnvironmentVariables.ContainsKey(variable.Key as string))
                {
                    psInfo.EnvironmentVariables.Add(variable.Key as string, variable.Value as string);
                }
                else
                {
                    psInfo.EnvironmentVariables[variable.Key as string] = variable.Value as string; 
                }
            }
            psInfo.RedirectStandardInput = true;

            psInfo.UseShellExecute = false;
            psInfo.CreateNoWindow = true;

            psInfo.RedirectStandardOutput = true;
            shell.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);

            psInfo.RedirectStandardError = true;
            shell.ErrorDataReceived += new DataReceivedEventHandler(ErrorHandler);

            shell.EnableRaisingEvents = true;
            shell.Exited += new EventHandler(ShellExited);

            

            shell.Start();

            shell.BeginOutputReadLine();
            shell.BeginErrorReadLine();

            

            StreamWriter input = shell.StandardInput;
            input.AutoFlush = true;

            input.WriteLine("java -classpath {0}\\bin;{0}\\hybridsal2xml;{0}\\hybridsal2xml\\antlr-2.7.1 HybridSal2Xml -o .\\{1}.hxml .\\{1}.hsal 2>&1", textBoxHSFolder.Text, Path.GetFileNameWithoutExtension(salFileName));
            input.WriteLine("python {0}\\src\\HSalRelAbsCons.py -c ./{1}.hxml 2>&1", textBoxHSFolder.Text, Path.GetFileNameWithoutExtension(salFileName));

            
            
            input.WriteLine(@"cygwin");
            input.WriteLine("cd {0}", projectDir.Replace(Path.DirectorySeparatorChar, '/'));
            input.WriteLine("{0}/sal-inf-bmc -i -d 4 ./{1}.sal correct",textBoxSalFolder.Text.Replace(Path.DirectorySeparatorChar,'/') ,Path.GetFileNameWithoutExtension(salFileName));

            input.Close();

        }



        private void ShellExited(object sender, System.EventArgs e)
        {
            if (buttonRunTool.InvokeRequired)
            {
                Invoke(new EventHandler(ShellExited), new object[] { sender, e });
            }
            else
            {
                try
                {

                    mgaFormalRequirement.set_StrAttrByName(verificationResultAttrName, result);
                    
                    FillPropertyMap();
                    FillDataGrid();

                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message, "Error");
                }
                finally
                {
                    dataGridViewProperties.Enabled = true;
                    buttonRunTool.Enabled = true;
                    buttonSaveToCyPhy.Enabled = true;
                    buttonBrowseHybridSALFolder.Enabled = true;
                    label1.Enabled = true;
                    textBoxHSFolder.Enabled = true;
                    this.Cursor = Cursors.Default;
                }
            }
        }




        private void buttonSaveToCyPhy_Click(object sender, EventArgs e)
        {
            RetrieveSettings();
            mgaFormalRequirement.set_StrAttrByName(additionalToolParameterName, settings.ToString());

            FillPropertyMap();
            FillDataGrid();
        }

        private void buttonBrowseHybridSALFolder_Click(object sender, EventArgs e)
        {
            if (folderBrowserDialogHSOutput.ShowDialog() == DialogResult.OK)
            {
                textBoxHSFolder.Text = folderBrowserDialogHSOutput.SelectedPath;                
                buttonRunTool.Enabled = true;                
            }
        }


        private void dataGridViewProperties_CellValueChanged(object sender, DataGridViewCellEventArgs e)
        {
            if (fillingDataGrid)
            {
                return;
            }
            int index = e.RowIndex;
            if (index < 0)
            {
                return;
            }
            GMEAttribute attr = propertyMap[(string)dataGridViewProperties.Rows[index].Cells[0].Value];
            attr.value = (string)dataGridViewProperties.Rows[index].Cells[1].Value;
        }


        public void WriteOutput(string line, bool isError = false)
        {
            StringBuilder output = new StringBuilder();

            string rtfLine = line.Replace("\\", "\\\\").Replace("{", "\\{").Replace("}", "\\}").Replace(" ", @"\~").Replace("\r\n", @"\par ").Replace("\n", @"\par ");
            Escape2Rtf(ref rtfLine);
            

            if (isError)
            {
                output.AppendLine(@"\cf5 " + rtfLine + @"\par ");
            }
            else if (line.StartsWith(@"WARNING"))
            {
                output.AppendLine(@"\cf3 " + rtfLine + @"\par ");
                //result = line.Substring(4);
            }
            else if (line.StartsWith("proved."))
            {
                result = "proved.";
                output.AppendLine(@"\cf2 " + rtfLine + @"\par ");
            }
            else if (line.Length >= 2 && line[1] == ':')
            {
                output.AppendLine(@"\cf1 " + rtfLine + @"\par ");
            }            
            else
            {
                output.AppendLine(@"\cf7 " + rtfLine + @"\par ");
            }

            if (isError)
            {
                errorText.Append(output);
                UpdateError();
            }
            else
            {
                outputText.Append(output);
                UpdateOutput();
            }

            

        }

        delegate void UpdateOutputDelegate();
        void UpdateOutput()
        {

            if (richTextBoxOutput.InvokeRequired)
            {
                richTextBoxOutput.Invoke(new UpdateOutputDelegate(this.UpdateOutput));
            }
            else
            {                
                Win32.LockWindowUpdate(richTextBoxOutput.Handle);
                richTextBoxOutput.Rtf = outputText.ToString();
                richTextBoxOutput.SelectionStart = richTextBoxOutput.TextLength;
                richTextBoxOutput.ScrollToCaret();
                Win32.LockWindowUpdate((IntPtr)0);

            }
        }

        delegate void UpdateErrorDelegate();
        void UpdateError()
        {

            if (richTextBoxOutput.InvokeRequired)
            {
                richTextBoxOutput.Invoke(new UpdateErrorDelegate(this.UpdateError));
            }
            else
            {
                Win32.LockWindowUpdate(richTextBoxError.Handle);
                richTextBoxError.Rtf = errorText.ToString();
                richTextBoxError.SelectionStart = richTextBoxError.TextLength;
                richTextBoxError.ScrollToCaret();
                Win32.LockWindowUpdate((IntPtr)0);

            }
        }

        private static bool Escape2Rtf(ref string str)
        {
            if(!str.Contains("\x1b"))
            {
                return false;
            }
          
            str = str.Replace("\x1b"+"[32m",@"\cf2 ");
            str = str.Replace("\x1b"+"[33m", @"\cf6 ");
            str = str.Replace("\x1b" + "[0m", @"\cf1 ");
            str = str.Replace("\x1b" + "]0;~\a", @"\cf1 ");
            str = str.Replace("\x1b" + "]0;", @"\cf1 ");
            str = str.Replace("\a", @"");
            

            return true;
        }

        private void buttonBrowseSALFolder_Click(object sender, EventArgs e)
        {
            if (folderBrowserDialogSALOutput.ShowDialog() == DialogResult.OK)
            {
                textBoxSalFolder.Text = folderBrowserDialogSALOutput.SelectedPath;
                buttonRunTool.Enabled = true;
            }
        }


        private void  FillSettings()
        {
            textBoxSalFolder.Text = settings.salPath;
            textBoxHSFolder.Text = settings.hyridSALProjectFilePath;
        }

        private void RetrieveSettings()
        {
            settings.salPath = textBoxSalFolder.Text;
            settings.hyridSALProjectFilePath = textBoxHSFolder.Text;
        }
    }


    public class GMEAttribute
    {
        public string name; // Internal GME-specific short name
        public string displayedName;
        public string value;
    }

    class Win32
    {
        [DllImport("user32")]
        public static extern int LockWindowUpdate(HWND hwnd);

    }

}


