﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;

namespace HybridSALVerifierPlugin
{
    public class Matrix
    {
        private int m, n;

        public int ColumnCount
        {
            get { return n; }
            set { n = value; }
        }

        public int RowsCount
        {
            get { return m; }
            set { m = value; }
        }

        private double[][] data; // Jagged because of pivoting

        public Matrix(int m, int n)
        {
            this.m = m;
            this.n = n;
            data = new double[m][];
            for (int i = 0; i < m; i++)
            {
                data[i] = new double[n];
                data[i].Initialize();
            }
        }

        public Matrix(int m, int n, params double[] elements)
        {
            this.m = m;
            this.n = n;
            data = new double[m][];
            for (int i = 0; i < m; i++)
            {
                data[i] = new double[n];
                for (uint j = 0; j < n; j++)
                {
                    data[i][j] = elements[i * n + j];
                }
            }
        }

        public void Eliminate()
        {
            for (int i = 0; i < m; i++)
            {
                double pivot = Double.NaN;

                for (int i1 = i; i1 < m; i1++)
                {

                    if (Math.Abs(data[i1][i]) > Double.Epsilon)
                    {
                        if (i1 != i) // Pivoting
                        {
                            var temp = data[i];
                            data[i] = data[i1];
                            data[i1] = temp;
                        }
                        pivot = data[i][i];
                        break;
                    }
                }

                if (pivot == Double.NaN)
                {
                    throw new ApplicationException("Error in eliminating temporary variable. An equation for a component is missing.");
                }

                // Elimination

                for (int i1 = 0; i1 < m; i1++) // Remaining rows
                {
                    if (i1 == i)
                    {
                        continue;
                    }

                    if (data[i1][i] == 0.0) // maybe with abs
                    {
                        continue;
                    }
                    double factor = data[i1][i] / data[i][i];
                    for (uint j = 0; j < n; j++)
                    {
                        data[i1][j] -= data[i][j] * factor;
                    }

                   
                   
                }
            }

        }

        public double this[int i, int j]
        {
            get { return data[i][j]; }
            set { data[i][j] = value; }
        }


        public void Dump()
        {
            Debug.WriteLine(Environment.NewLine);
            for (uint ik = 0; ik < m; ik++)
            {                
                for (uint jk = 0; jk < n; jk++)
                {
                    Debug.Write(String.Format("{0,7:0.0}", data[ik][jk]));

                }
                Debug.WriteLine("");
            }
        }


    }
}
