﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using GME.MGA;
using GME.MGA.Meta;
using System.IO;
using System.Text.RegularExpressions;



namespace GME.CSharp
{
    class MgaGateway
    {
        public MgaGateway(MgaProject project)
        {
            this.project = project;
        }

        public MgaProject project = null;
        public MgaTerritory territory = null;

        #region TRANSACTION HANDLING
        public void BeginTransaction(transactiontype_enum mode = transactiontype_enum.TRANSACTION_GENERAL)
        {
            project.BeginTransaction(territory, mode);
        }

        public void CommitTransaction()
        {
            if ((project.ProjectStatus & 8) != 0)
            {
                project.CommitTransaction();
            }
        }

        public void AbortTransaction()
        {
            if ((project.ProjectStatus & 8) != 0)
            {
                project.AbortTransaction();
            }
        }

        public delegate void voidDelegate();
        public void PerformInTransaction(voidDelegate d, transactiontype_enum mode = transactiontype_enum.TRANSACTION_GENERAL)
        {
            BeginTransaction(mode);
            try
            {
                d();
                CommitTransaction();
            }
            catch (Exception e)
            {
                try
                {
                    AbortTransaction();
                }
                catch { }
                throw e;
            }
        }
        #endregion
        #region UTILITIES
        public MgaMetaBase GetMetaByName(string name)
        {
            try
            {
                return project.RootMeta.RootFolder.get_DefinedFCOByName(name, false) as MgaMetaBase;
            }
#pragma warning disable 0168
            catch (System.Runtime.InteropServices.COMException e)
            {
                return project.RootMeta.RootFolder.get_DefinedFolderByName(name, false) as MgaMetaBase;
            }
#pragma warning restore 0168
        }

        #endregion


        public static MgaObject GetObjectByPath(MgaProject project, string pathWithinFile)
        {
            string[] pathElements = pathWithinFile.Split('/');
            MgaObject iterator = project.RootFolder as MgaObject;                   
            
            for(int i=0;  i< pathElements.Length; i++)
            {
                if (iterator == null || pathElements[i] != iterator.Name)
                {
                    throw new ApplicationException(String.Format("Object path {0} cannot be found. Unable to match element {1}.", pathWithinFile, pathElements[i]));
                }

                if (i == pathElements.Length - 1)
                {
                    break;
                }

                MgaFolder folder = iterator as MgaFolder;
                if (folder != null)
                {
                    iterator = null;
                    MgaObjects children = folder.ChildObjects;
                    foreach (MgaObject childObject in children)
                    {
                        if (childObject.Name == pathElements[i + 1])
                        {
                            iterator = childObject;
                            continue;
                        }
                    }
                }
                else
                {
                    MgaModel model = iterator as MgaModel;

                    if (model != null)
                    {
                        iterator = null;
                        MgaObjects childObjects = model.ChildObjects;
                        
                        foreach (MgaObject childObject in childObjects)
                        {
                            if (childObject.Name == pathElements[i + 1])
                            {
                                iterator = childObject;
                                continue;
                            }
                        }
                        
                    }

                }

            }

            return iterator;
            
        }

    }

    
    public class MgaURL
    {
        public static string Resolve(string fullPath, string currentDir)
        {
            if (currentDir.EndsWith(Path.DirectorySeparatorChar.ToString()))
            {
                currentDir = currentDir.Substring(0, currentDir.Length -1);
            }

            
            // Handling ../
            Regex regexp = new Regex(@"(\.\./)+");            
            Match match = regexp.Match(fullPath);
            if (match.Success)
            {
                string upDir = currentDir;
                for (int i = 0; i < match.Groups.Count; i++)
                {
                    upDir = Directory.GetParent(upDir).FullName;
                }

                StringBuilder newPath = new StringBuilder(fullPath.Substring(0, match.Index));
                newPath.Append(upDir.Replace(Path.DirectorySeparatorChar, '/'));
                newPath.Append('/');
                newPath.Append(fullPath.Substring(match.Index + match.Length));

                fullPath = newPath.ToString();

            }
            else
            {
                // Handling ./            
                fullPath = fullPath.Replace("./", currentDir.Replace(Path.DirectorySeparatorChar, '/') + '/');
            }
            return fullPath;
        }




        public static string GetFilePath(string fullPath)
        {
            string[] parts = fullPath.Split('#')[0].Split(new string[]{"///"}, StringSplitOptions.RemoveEmptyEntries);

            if(parts.Length < 1)
            {
                throw new ApplicationException("MGA URIs must start with mga:///.");
            }

            return parts[1].Replace('/', Path.DirectorySeparatorChar);
        }

        public static string GetInternalPath(string fullPath)
        {
            string[] parts= fullPath.Split('#');

            if (parts.Length > 1)
            {
                return parts[1];
            }
            else
            {
                return String.Empty;
            }
        }
    }
}
