﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using MGALib;


namespace CyPhy2HybridSAL
{
    class CyPhyTraverser
    {
        private IMgaModel rootSubsystem;
        private StateSpace stateSpace;
        

        public StateSpace StateSpace
        {
            get { return stateSpace; }
        }


        public CyPhyTraverser(IMgaModel rootSubsystem)
        {
            if (rootSubsystem.Meta.Name != "ESMoL::Subsystem")
            {
                throw new ApplicationException("CyPhyTraverser needs ESMoL::Subsystem as rootSubsystem");
            }

                       
            this.rootSubsystem = rootSubsystem;
        }

        public string Translate(string fileNameNoExtension,  string toolSpecificDescription)
        {
            // Building State Space
            stateSpace = new StateSpace();
            BuildStateSpace(rootSubsystem); // Columns of the matrix
            FillComponentCharacteristics();
            FillIntegrators();
            FillOutput();
            ResolveSaturations();

            stateSpace.Dump();

            stateSpace.ode.Eliminate();
            stateSpace.ode.Dump();

            HybridSALGenerator generator = new HybridSALGenerator(stateSpace, fileNameNoExtension, toolSpecificDescription);
            return generator.Generate();
        }


        private void BuildStateSpace(IMgaModel subsystem)
        {
            
            List<IMgaModel> intermediateVariablePorts = new List<IMgaModel>();         
            List<IMgaModel> outputVariablePorts = new List<IMgaModel>();
            List<IMgaModel> derivativeVariablePorts = new List<IMgaModel>();   
            List<IMgaModel> stateVariablePorts = new List<IMgaModel>();   
            List<IMgaModel> inputVariablePorts = new List<IMgaModel>();
           

            #region COLLECT VARIABLES
            IMgaFCOs childFCOs = subsystem.ChildFCOs;
            foreach (IMgaFCO fco in childFCOs)
            {
                string kind = fco.Meta.Name;
                if(kind == "ESMoL::Primitive")
                {
                    string blockType = fco.get_StrAttrByName("BlockType");
                    if (blockType == "BusCreator" || blockType == "Scope" || blockType == "Sin")
                    {
                        continue;
                    }
                    if (blockType == "Integrator")
                    {
                       stateVariablePorts.Add((fco as IMgaModel).GetChildrenOfKind("ESMoL::OutputPort")[1] as IMgaModel);                    
                       stateSpace.integrators.Add(new Integrator(fco.Name, fco as IMgaModel));   
                 
                        IMgaModel integratorBlock = fco as IMgaModel;
                        IMgaFCOs inputPorts = integratorBlock.GetChildrenOfKind("ESMoL::InputPort");

                        foreach (IMgaFCO inputPort in inputPorts)
                        {
                            if (inputPort.GetIntAttrByNameDisp("Number") == 1)
                            {
                                derivativeVariablePorts.Add(inputPort as IMgaModel);
                            }
                        }
                    }
                    else if (blockType == "Sum")
                    {
                        IMgaModel sum = fco as IMgaModel;
                        IMgaFCOs outputPorts = sum.GetChildrenOfKind("ESMoL::OutputPort");
                        IMgaFCO outputPort = outputPorts[1];
                        intermediateVariablePorts.Add(outputPort as IMgaModel); 
                    }
                    else if (blockType == "Constant")
                    {
                        intermediateVariablePorts.Add((fco as IMgaModel).GetChildrenOfKind("ESMoL::OutputPort")[1] as IMgaModel);
                    }
                    else if (blockType == "Gain")
                    {
                        intermediateVariablePorts.Add((fco as IMgaModel).GetChildrenOfKind("ESMoL::OutputPort")[1] as IMgaModel); 
                    }
                   else if (blockType == "Saturate")
                    {
                        intermediateVariablePorts.Add((fco as IMgaModel).GetChildrenOfKind("ESMoL::OutputPort")[1] as IMgaModel);
                        stateSpace.saturations.Add(new Saturation(fco.Name,fco as IMgaModel));

                    }
                    else
                    {
                        // Warning on Console
                        System.Diagnostics.Debug.Assert(false);
                    }
                }
               
                else if (kind == "ESMoL::InputPort")
                {
                    inputVariablePorts.Add(fco as IMgaModel); 
                }
                else if (kind == "ESMoL::OutputPort")
                {
                    outputVariablePorts.Add(fco as IMgaModel); 
                }
                else if (kind == "ESMoL::SubSystem")
                {
                    // Hierarchical models are not supported
                    // Warning on Console
                    System.Diagnostics.Debug.Assert(false);

                }
                else if (kind == "ESMoL::Parameter" || kind == "ESMoL::Line")
                {
                    continue;
                }
                else
                {
                    System.Diagnostics.Debug.Assert(false);
                    // Warning on Console
                }

            }
            #endregion

            // Setting up the matrix
            int n = intermediateVariablePorts.Count + outputVariablePorts.Count + stateVariablePorts.Count + derivativeVariablePorts.Count
                                                    + inputVariablePorts.Count + 1; // One column constant at the end
            int m = intermediateVariablePorts.Count + stateVariablePorts.Count + outputVariablePorts.Count;
            stateSpace.ode = new Matrix(m, n);


            // Setting up the columns
            stateSpace.columnOutputPorts.AddRange(intermediateVariablePorts);
            stateSpace.outputIndex = stateSpace.columnOutputPorts.Count;

            stateSpace.columnOutputPorts.AddRange(outputVariablePorts);
            stateSpace.derivativeIndex = stateSpace.columnOutputPorts.Count;

            stateSpace.columnOutputPorts.AddRange(derivativeVariablePorts);
            stateSpace.variableIndex = stateSpace.columnOutputPorts.Count;

            stateSpace.columnOutputPorts.AddRange(stateVariablePorts);
            stateSpace.inputIndex = stateSpace.columnOutputPorts.Count;

            stateSpace.columnOutputPorts.AddRange(inputVariablePorts);

            stateSpace.BuildPort2ColumnMapAndVariableNames();

        }

        private void FillComponentCharacteristics()
        {
            // Component characteristics
            for (int i = 0; i < stateSpace.outputIndex; i++)
            {
                IMgaModel block = stateSpace.columnOutputPorts[i].ParentModel;

                string blockType = block.get_StrAttrByName("BlockType");
              
                if (blockType == "Sum")
                {
                    List<IMgaModel> inputPorts = GetInputPorts(block);
                    System.Diagnostics.Debug.Assert(inputPorts.Count > 1);

                    stateSpace.ode[i, i] = -1;

                    string inputs = GetParameter(block, "Inputs").GetStrAttrByNameDisp("Value");

                    foreach(IMgaModel inputPort in inputPorts)
                    {
                        IMgaModel connOutputPort = GetConnectedOutputPort(inputPort);
                        int idx = stateSpace.outputPort2Column[connOutputPort];
                        int portNum = inputPort.GetIntAttrByNameDisp("Number");

                        if (inputs[portNum-1] == '+')
                        {
                            stateSpace.ode[i, idx] = 1;
                        }
                        else
                        {
                            stateSpace.ode[i, idx] = -1;
                        }
                    }
                }
                else if (blockType == "Constant")
                {
                    string value = GetParameter(block, "Value").GetStrAttrByNameDisp("Value");
                    double val = Double.Parse(value);

                    stateSpace.ode[i, i] = -1; 
                    stateSpace.ode[i, stateSpace.ode.ColumnCount-1] = val;

                }
                else if (blockType == "Gain")
                {
                    List<IMgaModel> inputPorts = GetInputPorts(block);
                    System.Diagnostics.Debug.Assert(inputPorts.Count == 1);

                    stateSpace.ode[i, i] = -1;

                    string strGain = GetParameter(block, "Gain").GetStrAttrByNameDisp("Value");
                    double gain = Double.Parse(strGain);

                    IMgaModel connOutputPort = GetConnectedOutputPort(inputPorts[0]);

                    int idx = stateSpace.outputPort2Column[connOutputPort];
                    stateSpace.ode[i, idx] = gain;                                
                }
                else if (blockType == "Saturate")
                {
                    List<IMgaModel> inputPorts = GetInputPorts(block);
                    System.Diagnostics.Debug.Assert(inputPorts.Count == 1);

                    stateSpace.ode[i, i] = -1;
                   

                    IMgaModel connOutputPort = GetConnectedOutputPort(inputPorts[0]);

                    int idx = stateSpace.outputPort2Column[connOutputPort];
                    stateSpace.ode[i, idx] = 1;         
                }
                else
                {
                    // Warning on Console
                    System.Diagnostics.Debug.Assert(false);
                }
            }
        }

        private void FillIntegrators()
        {
            for (int i = stateSpace.derivativeIndex; i < stateSpace.variableIndex; i++)
            {
                Integrator integrator = stateSpace.integrators[i - stateSpace.derivativeIndex];


                List<IMgaModel> inputPorts = GetInputPorts(integrator.block);

                foreach (IMgaModel inputPort in inputPorts)
                {
                    if (inputPort.GetIntAttrByNameDisp("Number") == 1)
                    {
                        integrator.derivativePos = stateSpace.outputPort2Column[inputPort];
                        stateSpace.ode[i, i] = -1;                      

                        IMgaModel connOutputPort = GetConnectedOutputPort(inputPorts[0]);
                        int idx = stateSpace.outputPort2Column[connOutputPort];
                        stateSpace.ode[i, idx] = 1;                         
                    }

                    if (inputPort.GetIntAttrByNameDisp("Number") == 2)
                    {
                        integrator.resetVariablePos = stateSpace.outputPort2Column[GetConnectedOutputPort(inputPort)];

                    }

                    if (inputPort.GetIntAttrByNameDisp("Number") == 3)
                    {
                        integrator.initialValuePos = stateSpace.outputPort2Column[GetConnectedOutputPort(inputPort)];
                    }

                    integrator.stateVariablePos =
                        stateSpace.outputPort2Column[integrator.block.GetChildrenOfKind("ESMoL::OutputPort")[1] as IMgaModel];
                }

                // Adding saturation
                IMgaAtom parameter = GetParameter(integrator.block, "UpperSaturationLimit");
                if (parameter != null)
                {
                    string value = parameter.GetStrAttrByNameDisp("Value");
                    double val = Double.Parse(value);
                    integrator.upperSaturation = val;
                }

                parameter = GetParameter(integrator.block, "LowerSaturationLimit");
                if (parameter != null)
                {
                    string value = parameter.GetStrAttrByNameDisp("Value");
                    double val = Double.Parse(value);
                    integrator.lowerSaturation = val;
                }

            }

        }

        private void FillOutput()
        {
            for (int i = stateSpace.outputIndex; i < stateSpace.derivativeIndex; i++)
            {
                IMgaModel inputPorts = stateSpace.columnOutputPorts[i];             

                stateSpace.ode[i, i] = -1;

                IMgaModel connOutputPort = GetPeerConnectedOutputPort(inputPorts);
                int idx = stateSpace.outputPort2Column[connOutputPort];
                stateSpace.ode[i, idx] = 1;                               
            }
        }

        private void ResolveSaturations()
        {
            foreach (Saturation saturation in stateSpace.saturations)
            {
                IMgaModel outputPort = saturation.block.GetChildrenOfKind("ESMoL::OutputPort")[1] as IMgaModel;
                saturation.variablePos = stateSpace.outputPort2Column[outputPort];
            }
        }

    /*    private void ResolveConstantInputs()
        {
            for (int i = stateSpace.inputIndex; i < stateSpace.ode.ColumnCount - 2; i++)
            {
                IMgaModel outputPort = GetConnectedOutputPort(stateSpace.columnOutputPorts[i]);
                IMgaModel block = outputPort.ParentModel;

                if (block.Meta.Name == "ESMoL::Primitive" || block.get_StrAttrByName("BlockType") == "Constant")
                {
                    string value = GetParameter(block, "Value").GetStrAttrByNameDisp("Value");
                    double val = Double.Parse(value);

                    // Add a row to ode

                }                
            }
         
        }
     * */


        private static List<IMgaModel> GetInputPorts(IMgaModel block)
        {
            List<IMgaModel> inputPorts = new List<IMgaModel>();
            IMgaFCOs inPorts = block.GetChildrenOfKind("ESMoL::InputPort");
            foreach (IMgaFCO inPort in inPorts)
            {
                inputPorts.Add(inPort as IMgaModel);
            }

            return inputPorts;
        }

        private static IMgaAtom GetParameter(IMgaModel block, string parameterName)
        {

            IMgaFCOs parameters = block.GetChildrenOfKind("ESMoL::Parameter");
            foreach (IMgaFCO parameter in parameters)
            {
                if (parameter.Name == parameterName)
                {
                    return parameter as IMgaAtom;
                }
            }

            return null;
        }


        private static IMgaModel GetConnectedOutputPort(IMgaModel inputPort)
        {
            List<IMgaModel> outputPorts = new List<IMgaModel>();

            IMgaConnPoints connPoints = inputPort.PartOfConns;

            foreach (IMgaConnPoint connPoint in connPoints)
            {
                if (connPoint.Owner.Meta.Name == "ESMoL::Line")
                {
                    IMgaSimpleConnection connection = (IMgaSimpleConnection)connPoint.Owner;
                    IMgaModel outputPort = connection.src as IMgaModel;

                    outputPorts.Add(outputPort);
                }
            }

            System.Diagnostics.Debug.Assert(outputPorts.Count == 1);
            return outputPorts[0];
        }


        private static IMgaModel GetPeerConnectedOutputPort(IMgaModel outputPort)
        {
            List<IMgaModel> outputPorts = new List<IMgaModel>();

            IMgaConnPoints connPoints = outputPort.PartOfConns;

            foreach (IMgaConnPoint connPoint in connPoints)
            {
                if (connPoint.Owner.Meta.Name == "ESMoL::Line")
                {
                    IMgaSimpleConnection connection = (IMgaSimpleConnection)connPoint.Owner;
                    IMgaModel outputPort2 = connection.src as IMgaModel;
                    if (outputPort != outputPort2)
                    {
                        outputPorts.Add(outputPort2);
                    }
                }
            }

            System.Diagnostics.Debug.Assert(outputPorts.Count == 1);
            return outputPorts[0];
        }

        


    }


}