﻿
/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using MGALib;
using MGAMetaLib;
using System.IO;
using System.Threading;

namespace CyPhy2HybridSAL
{
   
    public partial class MainForm : Form
    {
        public IMgaFCO mgaFormalRequirement;

        public MainForm(IMgaFCO mgaFormalRequirement)
        {
            this.mgaFormalRequirement = mgaFormalRequirement;
            propertyMap = new SortedDictionary<string, GMEAttribute>();

            InitializeComponent();

        }

        private void buttonOK_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            // Testing                                         
            textBoxTargetFolder.Text = @"D:\META\trunk\PROMISE\src\mdl2ha\Output";
            //buttonRunTool.Enabled = false;
            FillPropertyMap();
            FillDataGrid();

        }

        SortedDictionary<string, GMEAttribute> propertyMap;

        private void FillPropertyMap()
        {
            propertyMap.Clear();

            IMgaMetaFCO meta = mgaFormalRequirement.Meta;
            foreach (MgaMetaAttribute metaAttr in meta.Attributes)
            {
                GMEAttribute attribute = new GMEAttribute();
                MgaAttribute attr = mgaFormalRequirement.get_Attribute(metaAttr);
                attribute.name = metaAttr.Name;
                attribute.value = attr.StringValue;
                attribute.displayedName = metaAttr.DisplayedName;

                propertyMap.Add(attribute.displayedName, attribute);
            }
        }
        private bool fillingDataGrid = false;
        private void FillDataGrid()
        {
            fillingDataGrid = true;
            dataGridViewProperties.Rows.Clear();
            foreach (var property in propertyMap)
            {
                int row = dataGridViewProperties.Rows.Add();
                dataGridViewProperties.Rows[row].Cells[0].Value = property.Key;
                dataGridViewProperties.Rows[row].Cells[1].Value = property.Value.value;
            }
            fillingDataGrid = false;
        }


        private void GenerateHSFile()
        {
            IMgaModel component = mgaFormalRequirement.ParentModel;
            IMgaReference block = component.GetChildrenOfKind("SLSF_Block")[1] as IMgaReference;

            if (block == null)
            {
                MessageBox.Show("The META Component to which the selected Formal Requirement is attached to does not have any ESMoL subsystem reference.");
                return;
            }


            IMgaModel rootSubsystem = block.Referred as IMgaModel;

            if (block == null || rootSubsystem.Meta.Name != "ESMoL::Subsystem")
            {
                MessageBox.Show("The META Component to which the selected Formal Requirement is attached to does have a ESMoL subsystem reference, but it does not refer to an ESMoL::Subsystem.");
                return;
            }


            if(rootSubsystem.GetChildrenOfKind("ESMoL::InputPort").Count == 0)
            {
                IMgaFCOs subsystems = rootSubsystem.GetChildrenOfKind("ESMoL::Subsystem");

                if (subsystems.Count == 0)
                {
                    MessageBox.Show("Cannot find a component with an input port");
                    return;
                }
                rootSubsystem = null;

                foreach (IMgaFCO subsystem in subsystems)
                {
                    if ((subsystem as IMgaModel).GetChildrenOfKind("ESMoL::InputPort").Count != 0)
                    {
                        rootSubsystem = subsystem as IMgaModel;
                    }
                }

                if (rootSubsystem == null)
                {
                    MessageBox.Show("Cannot find a component with an input port");
                    return;
                }
            }

            CyPhyTraverser traverser = new CyPhyTraverser(rootSubsystem);
            string result = traverser.Translate("input", propertyMap["ToolSpecificDescription"].value);

            System.IO.StreamWriter file = new System.IO.StreamWriter(Path.Combine(textBoxTargetFolder.Text, @"input.sal"));
            try
            {
                file.WriteLine(result);
            }
            finally
            {
                file.Close();
            }

        }

        private void buttonRunTool_Click(object sender, EventArgs e)
        {

            GenerateHSFile();


            fileSystemWatcher.Path = textBoxTargetFolder.Text;
            fileSystemWatcher.EnableRaisingEvents = true;

            System.IO.StreamWriter file = new System.IO.StreamWriter(Path.Combine ( textBoxTargetFolder.Text, @"ready"));
            try
            {
                file.WriteLine("aaaa");
            }
            finally
            {
                file.Close();
            }

            ///////////////
            dataGridViewProperties.Enabled = false;
            buttonRunTool.Enabled = false;
            buttonSaveToCyPhy.Enabled = false;
            buttonBrowseHybridSALFolder.Enabled = false;
            label1.Enabled = false;
            textBoxTargetFolder.Enabled = false;

            progressBar.Visible = true;
            progressBar.MarqueeAnimationSpeed = 100;
            
        }

        private void buttonSaveToCyPhy_Click(object sender, EventArgs e)
        {
            foreach (DataGridViewRow row in dataGridViewProperties.Rows)
            {
                GMEAttribute attr = propertyMap[(string)row.Cells[0].Value];
                mgaFormalRequirement.set_StrAttrByName(attr.name, (string)row.Cells[1].Value);                
            }
        }

        private void buttonBrowseHybridSALFolder_Click(object sender, EventArgs e)
        {
            if (folderBrowserDialogHSOutput.ShowDialog() == DialogResult.OK)
            {
                textBoxTargetFolder.Text = folderBrowserDialogHSOutput.SelectedPath;                
                buttonRunTool.Enabled = true;                
            }
        }

        private void fileSystemWatcher_Changed(object sender, System.IO.FileSystemEventArgs e)
        {
            fileSystemWatcher.EnableRaisingEvents = false;

            dataGridViewProperties.Enabled = true;
            buttonRunTool.Enabled = true;
            buttonSaveToCyPhy.Enabled = true;
            buttonBrowseHybridSALFolder.Enabled = true;
            label1.Enabled = true;
            textBoxTargetFolder.Enabled = true;

            progressBar.Visible = false;
            progressBar.MarqueeAnimationSpeed = 0;

            
            string processingPath = Path.Combine(textBoxTargetFolder.Text, @"processing");
            
            if(File.Exists(processingPath))
            {
                File.Delete(processingPath);
            }

            string readyPath = Path.Combine(textBoxTargetFolder.Text, @"ready");
            if(File.Exists(readyPath))
            {
                File.Delete(readyPath);
            }

            Thread.Sleep(2000);

            StreamReader streamReader = new StreamReader(Path.Combine(textBoxTargetFolder.Text, @"result.txt"));
            try
            {
                // Might be changing with the model
                dataGridViewProperties.Rows[5].Cells[1].Value = streamReader.ReadToEnd();
            }
            finally
            {
                streamReader.Close();
            }

        }

        private void dataGridViewProperties_CellValueChanged(object sender, DataGridViewCellEventArgs e)
        {
            if (fillingDataGrid)
            {
                return;
            }
            int index = e.RowIndex;
            if (index < 0)
            {
                return;
            }
            GMEAttribute attr = propertyMap[(string)dataGridViewProperties.Rows[index].Cells[0].Value];
            attr.value = (string)dataGridViewProperties.Rows[index].Cells[1].Value;
        }



    }




    public class GMEAttribute
    {
        public string name; // Internal GME-specific short name
        public string displayedName;
        public string value;
    }

}
