﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

/*
	MetaUri Class.

	Author: Zsolt Lattmann (lattmann@isis.vanderbilt.edu)

	Date: 2011/05/18 
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;

namespace Meta
{
	/// <summary>
	/// Use this class to parse/split a META URI. Direct access to
	/// specific fields.
	/// </summary>
	public class MetaUri
	{
		/// <summary>
		/// Use the default regular expression pattern.
		/// </summary>
		/// <param name="Uri">
		/// <para>The actual Uri link. Use one of the following formats:</para>
		/// <para>model type://path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// <para>model type://serverhost/path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// <para>model type:///drive-letter:/path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// </param>
		public MetaUri(string Uri)
		{
			ClearAll();
			LoadExtensions();
			this.Link = Uri;
			// default pattern
			SplitUri(Pattern);
		}

		/// <summary>
		/// Use user defined regular expression pattern. ONLY for testing.
		/// </summary>
		/// <param name="Uri">
		/// <para>The actual Uri link. Use one of the following formats:</para>
		/// <para>model type://path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// <para>model type://serverhost/path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// <para>model type:///drive-letter:/path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// </param>
		/// <param name="Pattern">User defined regular expression.</param>
		public MetaUri(string Uri, string Pattern)
		{
			ClearAll();
			LoadExtensions();
			this.Link = Uri;
			SplitUri(Pattern);
		}

        public static string Resolve(string fullPath, string currentDir)
        {
            if (currentDir.EndsWith(Path.DirectorySeparatorChar.ToString()))
            {
                currentDir = currentDir.Substring(0, currentDir.Length -1);
            }

            
            // Handling ../
            Regex regexp = new Regex(@"(\.\./)+");            
            Match match = regexp.Match(fullPath);
            if (match.Success)
            {
                string upDir = currentDir;
                for (int i = 0; i < match.Groups.Count; i++)
                {
                    upDir = Directory.GetParent(upDir).FullName;
                }

                StringBuilder newPath = new StringBuilder(fullPath.Substring(0, match.Index));
                newPath.Append(upDir.Replace(Path.DirectorySeparatorChar, '/'));
                newPath.Append('/');
                newPath.Append(fullPath.Substring(match.Index + match.Length));

                fullPath = newPath.ToString();

            }
            else
            {
                // Handling ./            
                fullPath = fullPath.Replace("./", currentDir.Replace(Path.DirectorySeparatorChar, '/') + '/');
            }
            return fullPath;
        }


		/// <summary>
		/// Apply the regex pattern and store the result.
		/// </summary>
		/// <param name="pattern">Specified pattern</param>
		private void SplitUri(string pattern)
		{
			//string pat = MetaUri.Pattern;
			Regex r = new Regex(pattern);
			// get the list of group numbers
			int[] gnums = r.GetGroupNumbers();
			// get first match
			Match m = r.Match(this.Link);
			while (m.Success)
			{
				this.IsValid = true;
				// start at group 1
				for (int i = 1; i < gnums.Length; i++)
				{
					Group g = m.Groups[gnums[i]];
					System.Diagnostics.Debug.WriteLine(String.Format("{0}: {1}", i, g.Value));
					switch (i)
					{
						case 1:
							//if (Enum.IsDefined(typeof(ModelType_enum), g.Value))
							//{
							//  this.ModelType = (ModelType_enum)Enum.Parse(typeof(ModelType_enum), g.Value, true);
							//}
							//else
							//{
							//  this.ModelType = ModelType_enum.undefined;
							//}
							this.ModelType = g.Value.ToLower();
							break;
						case 2:

							break;
						case 3:
							this.DriveLetter = g.Value;
							break;
						case 4:

							break;
						case 5:
							this.ServerHost = g.Value;
							break;
						case 7:
							string FullPath = g.Value;
							this.PathToFile = Path.GetDirectoryName(FullPath).Replace('\\', '/');
							this.FileName = Path.GetFileName(FullPath);
							this.Extension = Path.GetExtension(FullPath);
							break;
						case 8:

							break;
						case 9:
							string FullElementPath = g.Value;
							if (FullElementPath != "")
							{
								this.PathToElement = Path.GetDirectoryName(FullElementPath).Replace('\\', '/');
								this.ElementName = Path.GetFileName(FullElementPath);
							}
							break;
						case 10:
							break;
						case 11:
							if (g.Value != "")
							{
								this.AttributeName = g.Value;
							}
							break;
					}
				}
				// get next match
				m = m.NextMatch();
			}
			if (this.IsValid)
			{
				if (this.DriveLetter.Equals("") &&
					this.ServerHost.Equals(""))
				{
					this.IsPathAbs = false;
				}
				else
				{
					this.IsPathAbs = true;
				}
				CollectErrorMessages();
			}
		}

		/// <summary>
		/// Collects all error messages and throws an exception if
		/// there is at least one.
		/// </summary>
		private void CollectErrorMessages()
		{
			// collect error messages and information
			StringBuilder sb = new StringBuilder();
			if (!this.SupportedFormats.ContainsKey(this.ModelType))
			{
				sb.Append("The following model TYPE is not supported: ");
				sb.Append(this.ModelType);
				sb.Append(".");
				sb.AppendLine();
			}
			else
			{
				// safe condition, the ModelType is inside the Formats.
				if (!this.SupportedFormats[this.ModelType].Equals(this.Extension))
				{
					sb.Append("Model TYPE - EXTENSION pair mismatch is detected: ");
					sb.Append(this.ModelType);
					sb.Append(".");
					sb.AppendLine();
					sb.Append("Please use the: ");
					sb.Append(this.SupportedFormats[this.ModelType]);
					sb.Append(" extension for this format.");
					sb.AppendLine();
				}
			}
			if (!this.SupportedFormats.ContainsValue(this.Extension))
			{
				sb.Append("The following EXTENSION is not supported: ");
				sb.Append(this.Extension);
				sb.Append(".");
				sb.AppendLine();
			}
			if (sb.Length > 0)
			{
				throw new MetaUriException(sb.ToString());
			}
		}

		/// <summary>
		/// <para>Type of the uri. E.g. simulink, esmol, gme...</para>
		/// </summary>
		public string ModelType { get; set; }
		/// <summary>
		/// <para>Server host name</para>
		/// </summary>
		public string ServerHost { get; set; }
		/// <summary>
		/// <para>Drive letter in c:/ format.</para>
		/// </summary>
		public string DriveLetter { get; set; }
		/// <summary>
		/// <para>Directory path to the file without the file name</para>
		/// </summary>
		public string PathToFile { get; set; }
		/// <summary>
		/// File name with extension
		/// </summary>
		public string FileName { get; set; }
		/// <summary>
		/// Extension with dot. E.g. .mdl or .mga
		/// </summary>
		public string Extension { get; set; }
		/// <summary>
		/// Path to the referred element without the name of the element
		/// </summary>
		public string PathToElement { get; set; }
		/// <summary>
		/// Name of the referred element
		/// </summary>
		public string ElementName { get; set; }
		/// <summary>
		/// Name of the requested attribute.
		/// </summary>
		public string AttributeName { get; set; }

		/// <summary>
		/// Path delimiter: '/'. Format: model-name/path-to-element/element-name
		/// </summary>
		public string FullElementPath
		{
			get
			{
				return Path.Combine(PathToElement, ElementName).Replace('\\', '/');
			}
		}

		/// <summary>
		/// Retrives the full OS path (e.g. c:\path-to-file\filename) if the file is on the local machine
		/// </summary>
		public string FullOSPath
		{
			get
			{
				if (String.IsNullOrEmpty(this.ServerHost))
				{
					return Path.Combine(DriveLetter, PathToFile, FileName);
				}
				else
				{
					return "";
				}
			}
		}

		/// <summary>
		/// True if the Meta Uri Link is syntactically valid.
		/// </summary>
		public bool IsValid { get; set; }
		/// <summary>
		/// True when the path is absolute (e.g. c:/ or server.com), otherwise it is false (relative path).
		/// </summary>
		public bool IsPathAbs { get; set; }
		/// <summary>
		/// The file is local.
		/// </summary>
		public bool IsLocal
		{
			get
			{
				if (String.IsNullOrEmpty(this.ServerHost))
				{
					return true;
				}
				else
				{
					return false;
				}
			}
		}
		/// <summary>
		/// Regular expression default pattern.
		/// </summary>
		public string Pattern
		{
			get
			{
				return @"^([^:/?#]+)://(/([a-z|A-Z]:/))?((([a-z|A-Z|0-9]+[.])+[a-z|A-Z]{2,4})/)?([^?#:]*)(#([^?#:]*))?(\?([^#?:]*))?$";
			}
		}
		/// <summary>
		/// <para>The actual Uri link. Use one of the following formats:</para>
		/// <para>model type://path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// <para>model type://serverhost/path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// <para>model type:///drive-letter:/path-to-file/filename[#path-to-element/element-name[?attribute-name]]</para>
		/// </summary>
		public string Link { get; set; }

		public enum ModelType_enum
		{
			undefined,
			simulink,
			esmol,
			gme
		};

		/// <summary>
		/// Valid ModelType - Extension pairs. (Extension definition with dot: .mga)
		/// See: <see cref="LoadExtensions"/>
		/// </summary>
		private Dictionary<string, string> SupportedFormats = new Dictionary<string, string>();

		private void LoadExtensions()
		{
			// TODO: what happens if 'one' model type supports muliple extenstions?
			SupportedFormats.Add(ModelType_enum.simulink.ToString(), ".mdl");
			SupportedFormats.Add(ModelType_enum.esmol.ToString(), ".mga");
			SupportedFormats.Add(ModelType_enum.gme.ToString(), ".mga");

			//foreach ( s in ModelType_enum)
			//{

			//}
		}

		/// <summary>
		/// Clear all member values. Default values:
		/// <para>String: Empty ""</para>
		/// <para>Boolean: false</para>
		/// </summary>
		private void ClearAll()
		{
			this.AttributeName = "";
			this.DriveLetter = "";
			this.ElementName = "";
			this.Extension = "";
			this.FileName = "";
			this.IsValid = false;
			this.Link = "";
			this.PathToElement = "";
			this.PathToFile = "";
			this.ServerHost = "";
			this.ModelType = "";
			this.IsPathAbs = false;
		}
	}
}
