﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using GME.MGA.Meta;
using GME.MGA;
using System.Reflection;
using VerificationIntegratorLib;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace VerificationManager
{
    public partial class MainForm : Form
    {
        const string progIDPropertyName = "VerifierID";
        const string progIDPropertyDisplayedName = "Verifier ID:";


        public MainForm(MgaProject project, MgaFCO mgaFormalRequirement)
        {
            this.mgaFormalRequirement = mgaFormalRequirement;
            this.project = project;
            this.rootFolder = project.RootFolder;

            registrationInfo.Load(rootFolder);

            propertyMap = new SortedDictionary<string, GMEAttribute>();

            InitializeComponent();

            FillPropertyMap();

            FillPropertyDataGrid();
            FillVerifiersDataGrid();
        }

        PluginList registrationInfo = new PluginList();

        public MgaFCO mgaFormalRequirement;
        public MgaFolder rootFolder;
        public MgaProject project;
        SortedDictionary<string, GMEAttribute> propertyMap;

        private void FillPropertyMap()
        {
            propertyMap.Clear();

            IMgaMetaFCO meta = mgaFormalRequirement.Meta;
            foreach (MgaMetaAttribute metaAttr in meta.Attributes)
            {
                GMEAttribute attribute = new GMEAttribute();
                MgaAttribute attr = mgaFormalRequirement.get_Attribute(metaAttr);
                attribute.name = metaAttr.Name;
                attribute.value = attr.StringValue;
                attribute.displayedName = metaAttr.DisplayedName;

                propertyMap.Add(attribute.displayedName, attribute);
            }
        }

        private bool fillingVerifiersDataGrid = false;

        private void FillVerifiersDataGrid()
        {
            fillingVerifiersDataGrid = true;
            dataGridViewRegisteredVerifiers.Rows.Clear();

            foreach (PluginInfo pluginInfo in registrationInfo.PluginInfos)
            {
                int row = dataGridViewRegisteredVerifiers.Rows.Add();
                dataGridViewRegisteredVerifiers.Rows[row].Cells[0].Value = pluginInfo.name;
                dataGridViewRegisteredVerifiers.Rows[row].Cells[1].Value = pluginInfo.progID;
                dataGridViewRegisteredVerifiers.Rows[row].Cells[2].Value = pluginInfo.path;
            }

            fillingVerifiersDataGrid = false;
        }

        private bool fillingPropertyDataGrid = false;
        private void FillPropertyDataGrid()
        {
            fillingPropertyDataGrid = true;
            dataGridViewProperties.Rows.Clear();
            foreach (var property in propertyMap)
            {
                int row = dataGridViewProperties.Rows.Add();
                dataGridViewProperties.Rows[row].Cells[0].Value = property.Key;
                

                if (property.Value.name == progIDPropertyName )
                {
                    string verifierProgID = property.Value.value;
                    DataGridViewComboBoxCell cell = new DataGridViewComboBoxCell();
                    cell.Items.Add("");
                    cell.Value = "";
                    foreach (PluginInfo pluginInfo in registrationInfo.PluginInfos)
                    {
                        string element = String.Format("{0}({1})", pluginInfo.name, pluginInfo.progID);
                        cell.Items.Add(element);
                        if (verifierProgID.Contains(pluginInfo.progID))
                        {
                            cell.Value = element;
                        }
                    }

                    dataGridViewProperties.Rows[row].Cells[1] = cell;
                }
                else
                {
                    dataGridViewProperties.Rows[row].Cells[1].Value = property.Value.value;
                }
            }
            fillingPropertyDataGrid = false;
        }

        private void buttonSaveToCyPhy_Click(object sender, EventArgs e)
        {
            SaveToCyPhy();
        }

        private void SaveToCyPhy()
        {
            foreach (DataGridViewRow row in dataGridViewProperties.Rows)
            {
                GMEAttribute attr = propertyMap[(string)row.Cells[0].Value];
                mgaFormalRequirement.set_StrAttrByName(attr.name, (string)row.Cells[1].Value);
            }

        }

        private void buttonRegisterNew_Click(object sender, EventArgs e)
        {
            if (openFileDialogDLL.ShowDialog() != System.Windows.Forms.DialogResult.OK)
            {
                return;
            }

            Assembly pluginAssembly = Assembly.LoadFrom(openFileDialogDLL.FileName);
            bool componentFound = false;
            foreach (Type t in pluginAssembly.GetExportedTypes())
            {

                object[] attrs = t.GetCustomAttributes(typeof(ProgIdAttribute), false);
                Debug.Assert(attrs.Length <= 1); // No multiple interface is supported (yet)

                if (attrs.Length == 0)
                {
                    continue;
                }
                else
                {
                    componentFound = true;
                }
                ProgIdAttribute attr = attrs[0] as ProgIdAttribute;
                string progID = attr.Value;

                Type pluginType = Type.GetTypeFromProgID(progID);

                if (pluginType == null)
                {
                    if (MessageBox.Show("The .NET plugin you selected is not registered. Do you want Verification Manager to register it?", "Unregistered .NET Component", MessageBoxButtons.YesNo) != System.Windows.Forms.DialogResult.Yes)
                    {
                        return;
                    }
                    else
                    {

                        // Registering .NET component
                        RegistrationServices regAsm = new RegistrationServices();
                        bool bResult = regAsm.RegisterAssembly(pluginAssembly, AssemblyRegistrationFlags.SetCodeBase);
                        if (bResult == false)
                        {
                            MessageBox.Show("Failed to register plugin", "Error");
                            return;
                        }

                        pluginType = Type.GetTypeFromProgID(progID);
                    }
                }

                IVerificationIntegrator plugin = Activator.CreateInstance(pluginType) as IVerificationIntegrator;

                string name = plugin.get_Parameter("Full Name");
                string path = plugin.get_Parameter("Path");

                int row = dataGridViewRegisteredVerifiers.Rows.Add();
                dataGridViewRegisteredVerifiers.Rows[row].Cells[0].Value = name;
                dataGridViewRegisteredVerifiers.Rows[row].Cells[1].Value = progID;
                dataGridViewRegisteredVerifiers.Rows[row].Cells[2].Value = path;

                pluginAssembly = null; // CLR won't unload it anyway, but feels good to me   
                registrationInfo.PluginInfos.Add(new PluginInfo(progID, name, path));

                FillPropertyDataGrid();

                break;
            }

            if (componentFound == false)
            {
                MessageBox.Show("The DLL supplied is not a valid plugin component.", "Plugin error");
            }
            else
            {
                registrationInfo.Save(rootFolder);
            }
        }

        private void buttonRemove_Click(object sender, EventArgs e)
        {

            Int32 selectedCellCount = dataGridViewRegisteredVerifiers.GetCellCount(DataGridViewElementStates.Selected);
            if (selectedCellCount > 0)
            {
                int selRow = dataGridViewRegisteredVerifiers.SelectedCells[0].RowIndex;
                dataGridViewRegisteredVerifiers.Rows.RemoveAt(selRow);
                FillPropertyDataGrid();
            }
        }

        private void dataGridViewRegisteredVerifiers_RowsRemoved(object sender, DataGridViewRowsRemovedEventArgs e)
        {
            registrationInfo.PluginInfos.RemoveAt(e.RowIndex);
            registrationInfo.Save(rootFolder);
        }

        private void buttonRunTool_Click(object sender, EventArgs e)
        {
            try
            {
                SaveToCyPhy();
                GMEAttribute attr = propertyMap[progIDPropertyDisplayedName];
                if (!String.IsNullOrWhiteSpace(attr.value))
                {
                    string[] parts = attr.value.Split('(');
                    if (parts.Length < 2)
                    {
                        MessageBox.Show(String.Format("Invalid Verifier ID: {0}.", attr.value));
                        return;
                    }

                    string progID = parts[1];
                    progID = progID.Substring(0, progID.Length - 1); // removing last ')'

                    Type pluginType = Type.GetTypeFromProgID(progID);


                    if (pluginType == null)
                    {
                        MessageBox.Show(String.Format("Cannot find plugin. ProgID: {0}.", attr.value));
                    }
                    else
                    {
                        IVerificationIntegrator plugin = Activator.CreateInstance(pluginType) as IVerificationIntegrator;

                        plugin.Invoke(project, mgaFormalRequirement);

                        FillPropertyMap();
                        FillPropertyDataGrid();
                    }
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error");
            }

        }

        private void dataGridViewProperties_CellValueChanged(object sender, DataGridViewCellEventArgs e)
        {
            if (fillingPropertyDataGrid)
            {
                return;
            }

            if (e.ColumnIndex != 1)
            {
                return;
            }

            int index = e.RowIndex;

            if (index < 0)
            {
                return;
            }
            

            GMEAttribute attr = propertyMap[(string)dataGridViewProperties.Rows[index].Cells[0].Value];
            attr.value = (string)dataGridViewProperties.Rows[index].Cells[1].Value;
        }

        private void dataGridViewProperties_CurrentCellDirtyStateChanged(object sender, EventArgs e)
        {
            if (fillingPropertyDataGrid)
            {
                return;
            }

            if (dataGridViewProperties.IsCurrentCellDirty)
            {
                dataGridViewProperties.CommitEdit(DataGridViewDataErrorContexts.Commit);
            }
        }

    }

    public class GMEAttribute
    {
        public string name; // Internal GME-specific short name
        public string displayedName;
        public string value;
    }

    
}
