﻿/*
 * Copyright (c) 2011, Vanderbilt University.
 * Developed with the sponsorship of the Defense Advanced Research Projects Agency (DARPA). Permission is hereby granted, free of charge, 
 * to any person obtaining a copy of this data, including any software or models in source or binary form, as well as any drawings, 
 * specifications, and documentation (collectively "the Data"), to deal in the Data without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Data, and to permit persons 
 * to whom the Data is furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Data.

 * THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using GME.MGA;
using System.Windows.Forms;
using VerificationIntegratorLib;

namespace VerificationManager
{
    public class PluginInfo
    {
        public PluginInfo(string progID, string name, string path)
        {
            this.progID = progID;
            this.name = name;
            this.path = path;
        }
        public readonly String progID;
        public readonly String name;
        public readonly String path;      
    }

    public class PluginList
    {
        private const string registryProgIDPath = "VerificationManager/RegisteredVerifiers";

        private List<PluginInfo> pluginInfos;

        public List<PluginInfo> PluginInfos
        {
            get { return pluginInfos; }
            set { pluginInfos = value; }
        }
 
        public void Load(MgaFolder root)
        {
            String registeredPlugins =  root.GetRegistryValueDisp(registryProgIDPath);
            pluginInfos = new List<PluginInfo>();              
            if (registeredPlugins != null)
            {
                string[] progIDs = registeredPlugins.Split(';');
                foreach (string progID in progIDs)
                {

                    Type pluginType = Type.GetTypeFromProgID(progID);

                    string name = null;
                    string path = null;
                    if (pluginType == null)
                    {
                        MessageBox.Show(String.Format("Cannot find previously registered plugin. ProgID: {0}.", progID));
                        name = "<not found>";
                        path = "<not found>";
                    }
                    else
                    {
                        IVerificationIntegrator plugin = Activator.CreateInstance(pluginType) as IVerificationIntegrator;

                        name = plugin.get_Parameter("Full Name");
                        path = plugin.get_Parameter("Path");
                    }

                    pluginInfos.Add(new PluginInfo(progID,name,path));
                }
            }
        }

        public void Save(MgaFolder root)
        {
            if (pluginInfos == null)
            {
                return;
            }

            StringBuilder registeredPlugins = new StringBuilder();
            foreach (PluginInfo pluginInfo in pluginInfos)
            {
                registeredPlugins.Append(pluginInfo.progID);
                registeredPlugins.Append(";");
            }

            root.SetRegistryValueDisp(registryProgIDPath, registeredPlugins.ToString(0, registeredPlugins.Length-1));
        }

    }
}
