"""Surrogate Model based on second order response surface equations."""

from numpy import matrix, linalg, power, multiply, concatenate, ones

from enthought.traits.api import HasTraits

from openmdao.main.interfaces import implements,ISurrogate
from openmdao.lib.datatypes.api import Float, Bool

class ResponseSurface(HasTraits): 
    implements(ISurrogate) 
    
    def __init__(self,X=None,Y=None): 
        # must call HasTraits init to set up Traits stuff 
        super(ResponseSurface, self).__init__() 
        
        self.m = None #number of training points 
        self.n = None #number of independents
        self.betas = None #vector of response surface equation coefficients
        
        if X is not None and Y is not None: 
            self.train(X,Y)
            
    def get_uncertain_value(self,value): 
        """Returns the value iself. Response surface equations don't have uncertainty.""" 
        return value

    def train(self,X,Y): 
        """ Calculate response surface equation coefficients using least squares regression. """ 
        
        X = matrix(X)
        Y = matrix(Y).T
        
        self.m = X.shape[0]
        self.n = X.shape[1]
        
        # Modify X to include constant, squared terms and cross terms
        X = concatenate((matrix(ones((self.m,1))),X),1) 
        for i in range(1,self.n+1):
            X = concatenate((X,power(X[:,i],2)),1)
        for i in range(1,self.n):
            for j in range(i+1,self.n+1):
                X = concatenate((X,multiply(X[:,i],X[:,j])),1)
        
        # Determine response surface equation coefficients (betas) using least squares
        self.betas, rs, r, s = linalg.lstsq(X,Y)
        
    def predict(self,new_x): 
        """Calculates a predicted value of the response based on the current response surface model for the supplied list of inputs. """ 
        
        new_x = matrix(new_x)
        
        # Modify new_x to include constant, squared terms and cross terms
        new_x = concatenate((matrix(ones((1,1))),new_x),1) 
        for i in range(1,self.n+1):
            new_x = concatenate((new_x,power(new_x[:,i],2)),1)
        for i in range(1,self.n):
            for j in range(i+1,self.n+1):
                new_x = concatenate((new_x,multiply(new_x[:,i],new_x[:,j])),1)
        
        # Predict new_y using new_x and betas
        new_y = new_x*self.betas
        return new_y[0,0]


if __name__ == "__main__":
    
    import time
    tt = time.time()

    # X = [[2.26,0.65],[6.11,9.53],[7.92,2.88],[2.03,9.97],[9.16,7.18],[2.98,2.5],[8.2,5.58],[6.37,4.92],[1.12,5.7],[4.76,0.9],[8.16,6.98],[6.06,1.54],\
        # [4.54,5.82],[5.24,8.66],[7.8,2.41],[2.94,1.77],[8.84,6.64],[5.16,7.62],[9.14,6.51],[1.5,7.75],[7.62,5.45],[9.98,1.39],[3.15,5.25],[1.75,9.9],\
        # [1.27,1.96],[3.63,1.64],[8.73,9.15],[8.34,9.45],[9.46,4.09],[9.61,4.18],[9.01,6.4],[3.69,4],[8.2,8.43],[6.81,6.1],[6.67,6.11]]

    # Y = [11.031825,278.8868,115.772,185.305,279.3373,36.6028,197.9325,137.621475,64.4913,33.1837,243.8833,58.042,124.4148,223.8245,102.401125,26.901425,\
        # 250.4164,188.2781,254.037625,113.640625,180.029425,117.424825,86.495,176.559375,15.700075,31.741075,344.5268,344.435825,181.506625,187.824175,\
        # 246.523075,69.922075,298.858125,180.836575,177.8742]
        
    X = [[3000,2,1,1,0],\
        [3000,2,1,1,5],\
        [3000,2,1,1,10],\
        [3000,2,1,1,15],\
        [3000,2,1,1,20],\
        [3000,2,1,2,0],\
        [3000,2,1,2,5],\
        [3000,2,1,2,10],\
        [3000,2,1,2,15],\
        [3000,2,1,2,20],\
        [3000,2,1,3,0],\
        [3000,2,1,3,5],\
        [3000,2,1,3,10],\
        [3000,2,1,3,15],\
        [3000,2,1,3,20],\
        [3000,2,1,4,0],\
        [3000,2,1,4,5],\
        [3000,2,1,4,10],\
        [3000,2,1,4,15],\
        [3000,2,1,4,20],\
        [3000,2,1,5,0],\
        [3000,2,1,5,5],\
        [3000,2,1,5,10],\
        [3000,2,1,5,15],\
        [3000,2,1,5,20],\
        [3000,2,1.375,1,0],\
        [3000,2,1.375,1,5],\
        [3000,2,1.375,1,10],\
        [3000,2,1.375,1,15],\
        [3000,2,1.375,1,20],\
        [3000,2,1.375,2,0],\
        [3000,2,1.375,2,5],\
        [3000,2,1.375,2,10],\
        [3000,2,1.375,2,15],\
        [3000,2,1.375,2,20],\
        [3000,2,1.375,3,0],\
        [3000,2,1.375,3,5],\
        [3000,2,1.375,3,10],\
        [3000,2,1.375,3,15],\
        [3000,2,1.375,3,20],\
        [3000,2,1.375,4,0],\
        [3000,2,1.375,4,5],\
        [3000,2,1.375,4,10],\
        [3000,2,1.375,4,15],\
        [3000,2,1.375,4,20],\
        [3000,2,1.375,5,0],\
        [3000,2,1.375,5,5],\
        [3000,2,1.375,5,10],\
        [3000,2,1.375,5,15],\
        [3000,2,1.375,5,20],\
        [3000,2,1.75,1,0],\
        [3000,2,1.75,1,5],\
        [3000,2,1.75,1,10],\
        [3000,2,1.75,1,15],\
        [3000,2,1.75,1,20],\
        [3000,2,1.75,2,0],\
        [3000,2,1.75,2,5],\
        [3000,2,1.75,2,10],\
        [3000,2,1.75,2,15],\
        [3000,2,1.75,2,20],\
        [3000,2,1.75,3,0],\
        [3000,2,1.75,3,5],\
        [3000,2,1.75,3,10],\
        [3000,2,1.75,3,15],\
        [3000,2,1.75,3,20],\
        [3000,2,1.75,4,0],\
        [3000,2,1.75,4,5],\
        [3000,2,1.75,4,10],\
        [3000,2,1.75,4,15],\
        [3000,2,1.75,4,20],\
        [3000,2,1.75,5,0],\
        [3000,2,1.75,5,5],\
        [3000,2,1.75,5,10],\
        [3000,2,1.75,5,15],\
        [3000,2,1.75,5,20],\
        [3000,2,2.125,1,0],\
        [3000,2,2.125,1,5],\
        [3000,2,2.125,1,10],\
        [3000,2,2.125,1,15],\
        [3000,2,2.125,1,20],\
        [3000,2,2.125,2,0],\
        [3000,2,2.125,2,5],\
        [3000,2,2.125,2,10],\
        [3000,2,2.125,2,15],\
        [3000,2,2.125,2,20],\
        [3000,2,2.125,3,0],\
        [3000,2,2.125,3,5],\
        [3000,2,2.125,3,10],\
        [3000,2,2.125,3,15],\
        [3000,2,2.125,3,20],\
        [3000,2,2.125,4,0],\
        [3000,2,2.125,4,5],\
        [3000,2,2.125,4,10],\
        [3000,2,2.125,4,15],\
        [3000,2,2.125,4,20],\
        [3000,2,2.125,5,0],\
        [3000,2,2.125,5,5],\
        [3000,2,2.125,5,10],\
        [3000,2,2.125,5,15],\
        [3000,2,2.125,5,20],\
        [3000,2,2.5,1,0],\
        [3000,2,2.5,1,5],\
        [3000,2,2.5,1,10],\
        [3000,2,2.5,1,15],\
        [3000,2,2.5,1,20],\
        [3000,2,2.5,2,0],\
        [3000,2,2.5,2,5],\
        [3000,2,2.5,2,10],\
        [3000,2,2.5,2,15],\
        [3000,2,2.5,2,20],\
        [3000,2,2.5,3,0],\
        [3000,2,2.5,3,5],\
        [3000,2,2.5,3,10],\
        [3000,2,2.5,3,15],\
        [3000,2,2.5,3,20],\
        [3000,2,2.5,4,0],\
        [3000,2,2.5,4,5],\
        [3000,2,2.5,4,10],\
        [3000,2,2.5,4,15],\
        [3000,2,2.5,4,20],\
        [3000,2,2.5,5,0],\
        [3000,2,2.5,5,5],\
        [3000,2,2.5,5,10],\
        [3000,2,2.5,5,15],\
        [3000,2,2.5,5,20],\
        [3000,5,1,1,0],\
        [3000,5,1,1,5],\
        [3000,5,1,1,10],\
        [3000,5,1,1,15],\
        [3000,5,1,1,20],\
        [3000,5,1,2,0],\
        [3000,5,1,2,5],\
        [3000,5,1,2,10],\
        [3000,5,1,2,15],\
        [3000,5,1,2,20],\
        [3000,5,1,3,0],\
        [3000,5,1,3,5],\
        [3000,5,1,3,10],\
        [3000,5,1,3,15],\
        [3000,5,1,3,20],\
        [3000,5,1,4,0],\
        [3000,5,1,4,5],\
        [3000,5,1,4,10],\
        [3000,5,1,4,15],\
        [3000,5,1,4,20],\
        [3000,5,1,5,0],\
        [3000,5,1,5,5],\
        [3000,5,1,5,10],\
        [3000,5,1,5,15],\
        [3000,5,1,5,20],\
        [3000,5,1.375,1,0],\
        [3000,5,1.375,1,5],\
        [3000,5,1.375,1,10],\
        [3000,5,1.375,1,15],\
        [3000,5,1.375,1,20],\
        [3000,5,1.375,2,0],\
        [3000,5,1.375,2,5],\
        [3000,5,1.375,2,10],\
        [3000,5,1.375,2,15],\
        [3000,5,1.375,2,20],\
        [3000,5,1.375,3,0],\
        [3000,5,1.375,3,5],\
        [3000,5,1.375,3,10],\
        [3000,5,1.375,3,15],\
        [3000,5,1.375,3,20],\
        [3000,5,1.375,4,0],\
        [3000,5,1.375,4,5],\
        [3000,5,1.375,4,10],\
        [3000,5,1.375,4,15],\
        [3000,5,1.375,4,20],\
        [3000,5,1.375,5,0],\
        [3000,5,1.375,5,5],\
        [3000,5,1.375,5,10],\
        [3000,5,1.375,5,15],\
        [3000,5,1.375,5,20],\
        [3000,5,1.75,1,0],\
        [3000,5,1.75,1,5],\
        [3000,5,1.75,1,10],\
        [3000,5,1.75,1,15],\
        [3000,5,1.75,1,20],\
        [3000,5,1.75,2,0],\
        [3000,5,1.75,2,5],\
        [3000,5,1.75,2,10],\
        [3000,5,1.75,2,15],\
        [3000,5,1.75,2,20],\
        [3000,5,1.75,3,0],\
        [3000,5,1.75,3,5],\
        [3000,5,1.75,3,10],\
        [3000,5,1.75,3,15],\
        [3000,5,1.75,3,20],\
        [3000,5,1.75,4,0],\
        [3000,5,1.75,4,5],\
        [3000,5,1.75,4,10],\
        [3000,5,1.75,4,15],\
        [3000,5,1.75,4,20],\
        [3000,5,1.75,5,0],\
        [3000,5,1.75,5,5],\
        [3000,5,1.75,5,10],\
        [3000,5,1.75,5,15],\
        [3000,5,1.75,5,20],\
        [3000,5,2.125,1,0],\
        [3000,5,2.125,1,5],\
        [3000,5,2.125,1,10],\
        [3000,5,2.125,1,15],\
        [3000,5,2.125,1,20],\
        [3000,5,2.125,2,0],\
        [3000,5,2.125,2,5],\
        [3000,5,2.125,2,10],\
        [3000,5,2.125,2,15],\
        [3000,5,2.125,2,20],\
        [3000,5,2.125,3,0],\
        [3000,5,2.125,3,5],\
        [3000,5,2.125,3,10],\
        [3000,5,2.125,3,15],\
        [3000,5,2.125,3,20],\
        [3000,5,2.125,4,0],\
        [3000,5,2.125,4,5],\
        [3000,5,2.125,4,10],\
        [3000,5,2.125,4,15],\
        [3000,5,2.125,4,20],\
        [3000,5,2.125,5,0],\
        [3000,5,2.125,5,5],\
        [3000,5,2.125,5,10],\
        [3000,5,2.125,5,15],\
        [3000,5,2.125,5,20],\
        [3000,5,2.5,1,0],\
        [3000,5,2.5,1,5],\
        [3000,5,2.5,1,10],\
        [3000,5,2.5,1,15],\
        [3000,5,2.5,1,20],\
        [3000,5,2.5,2,0],\
        [3000,5,2.5,2,5],\
        [3000,5,2.5,2,10],\
        [3000,5,2.5,2,15],\
        [3000,5,2.5,2,20],\
        [3000,5,2.5,3,0],\
        [3000,5,2.5,3,5],\
        [3000,5,2.5,3,10],\
        [3000,5,2.5,3,15],\
        [3000,5,2.5,3,20],\
        [3000,5,2.5,4,0],\
        [3000,5,2.5,4,5],\
        [3000,5,2.5,4,10],\
        [3000,5,2.5,4,15],\
        [3000,5,2.5,4,20],\
        [3000,5,2.5,5,0],\
        [3000,5,2.5,5,5],\
        [3000,5,2.5,5,10],\
        [3000,5,2.5,5,15],\
        [3000,5,2.5,5,20],\
        [3000,8,1,1,0],\
        [3000,8,1,1,5],\
        [3000,8,1,1,10],\
        [3000,8,1,1,15],\
        [3000,8,1,1,20],\
        [3000,8,1,2,0],\
        [3000,8,1,2,5],\
        [3000,8,1,2,10],\
        [3000,8,1,2,15],\
        [3000,8,1,2,20],\
        [3000,8,1,3,0],\
        [3000,8,1,3,5],\
        [3000,8,1,3,10],\
        [3000,8,1,3,15],\
        [3000,8,1,3,20],\
        [3000,8,1,4,0],\
        [3000,8,1,4,5],\
        [3000,8,1,4,10],\
        [3000,8,1,4,15],\
        [3000,8,1,4,20],\
        [3000,8,1,5,0],\
        [3000,8,1,5,5],\
        [3000,8,1,5,10],\
        [3000,8,1,5,15],\
        [3000,8,1,5,20],\
        [3000,8,1.375,1,0],\
        [3000,8,1.375,1,5],\
        [3000,8,1.375,1,10],\
        [3000,8,1.375,1,15],\
        [3000,8,1.375,1,20],\
        [3000,8,1.375,2,0],\
        [3000,8,1.375,2,5],\
        [3000,8,1.375,2,10],\
        [3000,8,1.375,2,15],\
        [3000,8,1.375,2,20],\
        [3000,8,1.375,3,0],\
        [3000,8,1.375,3,5],\
        [3000,8,1.375,3,10],\
        [3000,8,1.375,3,15],\
        [3000,8,1.375,3,20],\
        [3000,8,1.375,4,0],\
        [3000,8,1.375,4,5],\
        [3000,8,1.375,4,10],\
        [3000,8,1.375,4,15],\
        [3000,8,1.375,4,20],\
        [3000,8,1.375,5,0],\
        [3000,8,1.375,5,5],\
        [3000,8,1.375,5,10],\
        [3000,8,1.375,5,15],\
        [3000,8,1.375,5,20],\
        [3000,8,1.75,1,0],\
        [3000,8,1.75,1,5],\
        [3000,8,1.75,1,10],\
        [3000,8,1.75,1,15],\
        [3000,8,1.75,1,20],\
        [3000,8,1.75,2,0],\
        [3000,8,1.75,2,5],\
        [3000,8,1.75,2,10],\
        [3000,8,1.75,2,15],\
        [3000,8,1.75,2,20],\
        [3000,8,1.75,3,0],\
        [3000,8,1.75,3,5],\
        [3000,8,1.75,3,10],\
        [3000,8,1.75,3,15],\
        [3000,8,1.75,3,20],\
        [3000,8,1.75,4,0],\
        [3000,8,1.75,4,5],\
        [3000,8,1.75,4,10],\
        [3000,8,1.75,4,15],\
        [3000,8,1.75,4,20],\
        [3000,8,1.75,5,0],\
        [3000,8,1.75,5,5],\
        [3000,8,1.75,5,10],\
        [3000,8,1.75,5,15],\
        [3000,8,1.75,5,20],\
        [3000,8,2.125,1,0],\
        [3000,8,2.125,1,5],\
        [3000,8,2.125,1,10],\
        [3000,8,2.125,1,15],\
        [3000,8,2.125,1,20],\
        [3000,8,2.125,2,0],\
        [3000,8,2.125,2,5],\
        [3000,8,2.125,2,10],\
        [3000,8,2.125,2,15],\
        [3000,8,2.125,2,20],\
        [3000,8,2.125,3,0],\
        [3000,8,2.125,3,5],\
        [3000,8,2.125,3,10],\
        [3000,8,2.125,3,15],\
        [3000,8,2.125,3,20],\
        [3000,8,2.125,4,0],\
        [3000,8,2.125,4,5],\
        [3000,8,2.125,4,10],\
        [3000,8,2.125,4,15],\
        [3000,8,2.125,4,20],\
        [3000,8,2.125,5,0],\
        [3000,8,2.125,5,5],\
        [3000,8,2.125,5,10],\
        [3000,8,2.125,5,15],\
        [3000,8,2.125,5,20],\
        [3000,8,2.5,1,0],\
        [3000,8,2.5,1,5],\
        [3000,8,2.5,1,10],\
        [3000,8,2.5,1,15],\
        [3000,8,2.5,1,20],\
        [3000,8,2.5,2,0],\
        [3000,8,2.5,2,5],\
        [3000,8,2.5,2,10],\
        [3000,8,2.5,2,15],\
        [3000,8,2.5,2,20],\
        [3000,8,2.5,3,0],\
        [3000,8,2.5,3,5],\
        [3000,8,2.5,3,10],\
        [3000,8,2.5,3,15],\
        [3000,8,2.5,3,20],\
        [3000,8,2.5,4,0],\
        [3000,8,2.5,4,5],\
        [3000,8,2.5,4,10],\
        [3000,8,2.5,4,15],\
        [3000,8,2.5,4,20],\
        [3000,8,2.5,5,0],\
        [3000,8,2.5,5,5],\
        [3000,8,2.5,5,10],\
        [3000,8,2.5,5,15],\
        [3000,8,2.5,5,20],\
        [3000,11,1,1,0],\
        [3000,11,1,1,5],\
        [3000,11,1,1,10],\
        [3000,11,1,1,15],\
        [3000,11,1,1,20],\
        [3000,11,1,2,0],\
        [3000,11,1,2,5],\
        [3000,11,1,2,10],\
        [3000,11,1,2,15],\
        [3000,11,1,2,20],\
        [3000,11,1,3,0],\
        [3000,11,1,3,5],\
        [3000,11,1,3,10],\
        [3000,11,1,3,15],\
        [3000,11,1,3,20],\
        [3000,11,1,4,0],\
        [3000,11,1,4,5],\
        [3000,11,1,4,10],\
        [3000,11,1,4,15],\
        [3000,11,1,4,20],\
        [3000,11,1,5,0],\
        [3000,11,1,5,5],\
        [3000,11,1,5,10],\
        [3000,11,1,5,15],\
        [3000,11,1,5,20],\
        [3000,11,1.375,1,0],\
        [3000,11,1.375,1,5],\
        [3000,11,1.375,1,10],\
        [3000,11,1.375,1,15],\
        [3000,11,1.375,1,20],\
        [3000,11,1.375,2,0],\
        [3000,11,1.375,2,5],\
        [3000,11,1.375,2,10],\
        [3000,11,1.375,2,15],\
        [3000,11,1.375,2,20],\
        [3000,11,1.375,3,0],\
        [3000,11,1.375,3,5],\
        [3000,11,1.375,3,10],\
        [3000,11,1.375,3,15],\
        [3000,11,1.375,3,20],\
        [3000,11,1.375,4,0],\
        [3000,11,1.375,4,5],\
        [3000,11,1.375,4,10],\
        [3000,11,1.375,4,15],\
        [3000,11,1.375,4,20],\
        [3000,11,1.375,5,0],\
        [3000,11,1.375,5,5],\
        [3000,11,1.375,5,10],\
        [3000,11,1.375,5,15],\
        [3000,11,1.375,5,20],\
        [3000,11,1.75,1,0],\
        [3000,11,1.75,1,5],\
        [3000,11,1.75,1,10],\
        [3000,11,1.75,1,15],\
        [3000,11,1.75,1,20],\
        [3000,11,1.75,2,0],\
        [3000,11,1.75,2,5],\
        [3000,11,1.75,2,10],\
        [3000,11,1.75,2,15],\
        [3000,11,1.75,2,20],\
        [3000,11,1.75,3,0],\
        [3000,11,1.75,3,5],\
        [3000,11,1.75,3,10],\
        [3000,11,1.75,3,15],\
        [3000,11,1.75,3,20],\
        [3000,11,1.75,4,0],\
        [3000,11,1.75,4,5],\
        [3000,11,1.75,4,10],\
        [3000,11,1.75,4,15],\
        [3000,11,1.75,4,20],\
        [3000,11,1.75,5,0],\
        [3000,11,1.75,5,5],\
        [3000,11,1.75,5,10],\
        [3000,11,1.75,5,15],\
        [3000,11,1.75,5,20],\
        [3000,11,2.125,1,0],\
        [3000,11,2.125,1,5],\
        [3000,11,2.125,1,10],\
        [3000,11,2.125,1,15],\
        [3000,11,2.125,1,20],\
        [3000,11,2.125,2,0],\
        [3000,11,2.125,2,5],\
        [3000,11,2.125,2,10],\
        [3000,11,2.125,2,15],\
        [3000,11,2.125,2,20],\
        [3000,11,2.125,3,0],\
        [3000,11,2.125,3,5],\
        [3000,11,2.125,3,10],\
        [3000,11,2.125,3,15],\
        [3000,11,2.125,3,20],\
        [3000,11,2.125,4,0],\
        [3000,11,2.125,4,5],\
        [3000,11,2.125,4,10],\
        [3000,11,2.125,4,15],\
        [3000,11,2.125,4,20],\
        [3000,11,2.125,5,0],\
        [3000,11,2.125,5,5],\
        [3000,11,2.125,5,10],\
        [3000,11,2.125,5,15],\
        [3000,11,2.125,5,20],\
        [3000,11,2.5,1,0],\
        [3000,11,2.5,1,5],\
        [3000,11,2.5,1,10],\
        [3000,11,2.5,1,15],\
        [3000,11,2.5,1,20],\
        [3000,11,2.5,2,0],\
        [3000,11,2.5,2,5],\
        [3000,11,2.5,2,10],\
        [3000,11,2.5,2,15],\
        [3000,11,2.5,2,20],\
        [3000,11,2.5,3,0],\
        [3000,11,2.5,3,5],\
        [3000,11,2.5,3,10],\
        [3000,11,2.5,3,15],\
        [3000,11,2.5,3,20],\
        [3000,11,2.5,4,0],\
        [3000,11,2.5,4,5],\
        [3000,11,2.5,4,10],\
        [3000,11,2.5,4,15],\
        [3000,11,2.5,4,20],\
        [3000,11,2.5,5,0],\
        [3000,11,2.5,5,5],\
        [3000,11,2.5,5,10],\
        [3000,11,2.5,5,15],\
        [3000,11,2.5,5,20],\
        [3000,14,1,1,0],\
        [3000,14,1,1,5],\
        [3000,14,1,1,10],\
        [3000,14,1,1,15],\
        [3000,14,1,1,20],\
        [3000,14,1,2,0],\
        [3000,14,1,2,5],\
        [3000,14,1,2,10],\
        [3000,14,1,2,15],\
        [3000,14,1,2,20],\
        [3000,14,1,3,0],\
        [3000,14,1,3,5],\
        [3000,14,1,3,10],\
        [3000,14,1,3,15],\
        [3000,14,1,3,20],\
        [3000,14,1,4,0],\
        [3000,14,1,4,5],\
        [3000,14,1,4,10],\
        [3000,14,1,4,15],\
        [3000,14,1,4,20],\
        [3000,14,1,5,0],\
        [3000,14,1,5,5],\
        [3000,14,1,5,10],\
        [3000,14,1,5,15],\
        [3000,14,1,5,20],\
        [3000,14,1.375,1,0],\
        [3000,14,1.375,1,5],\
        [3000,14,1.375,1,10],\
        [3000,14,1.375,1,15],\
        [3000,14,1.375,1,20],\
        [3000,14,1.375,2,0],\
        [3000,14,1.375,2,5],\
        [3000,14,1.375,2,10],\
        [3000,14,1.375,2,15],\
        [3000,14,1.375,2,20],\
        [3000,14,1.375,3,0],\
        [3000,14,1.375,3,5],\
        [3000,14,1.375,3,10],\
        [3000,14,1.375,3,15],\
        [3000,14,1.375,3,20],\
        [3000,14,1.375,4,0],\
        [3000,14,1.375,4,5],\
        [3000,14,1.375,4,10],\
        [3000,14,1.375,4,15],\
        [3000,14,1.375,4,20],\
        [3000,14,1.375,5,0],\
        [3000,14,1.375,5,5],\
        [3000,14,1.375,5,10],\
        [3000,14,1.375,5,15],\
        [3000,14,1.375,5,20],\
        [3000,14,1.75,1,0],\
        [3000,14,1.75,1,5],\
        [3000,14,1.75,1,10],\
        [3000,14,1.75,1,15],\
        [3000,14,1.75,1,20],\
        [3000,14,1.75,2,0],\
        [3000,14,1.75,2,5],\
        [3000,14,1.75,2,10],\
        [3000,14,1.75,2,15],\
        [3000,14,1.75,2,20],\
        [3000,14,1.75,3,0],\
        [3000,14,1.75,3,5],\
        [3000,14,1.75,3,10],\
        [3000,14,1.75,3,15],\
        [3000,14,1.75,3,20],\
        [3000,14,1.75,4,0],\
        [3000,14,1.75,4,5],\
        [3000,14,1.75,4,10],\
        [3000,14,1.75,4,15],\
        [3000,14,1.75,4,20],\
        [3000,14,1.75,5,0],\
        [3000,14,1.75,5,5],\
        [3000,14,1.75,5,10],\
        [3000,14,1.75,5,15],\
        [3000,14,1.75,5,20],\
        [3000,14,2.125,1,0],\
        [3000,14,2.125,1,5],\
        [3000,14,2.125,1,10],\
        [3000,14,2.125,1,15],\
        [3000,14,2.125,1,20],\
        [3000,14,2.125,2,0],\
        [3000,14,2.125,2,5],\
        [3000,14,2.125,2,10],\
        [3000,14,2.125,2,15],\
        [3000,14,2.125,2,20],\
        [3000,14,2.125,3,0],\
        [3000,14,2.125,3,5],\
        [3000,14,2.125,3,10],\
        [3000,14,2.125,3,15],\
        [3000,14,2.125,3,20],\
        [3000,14,2.125,4,0],\
        [3000,14,2.125,4,5],\
        [3000,14,2.125,4,10],\
        [3000,14,2.125,4,15],\
        [3000,14,2.125,4,20],\
        [3000,14,2.125,5,0],\
        [3000,14,2.125,5,5],\
        [3000,14,2.125,5,10],\
        [3000,14,2.125,5,15],\
        [3000,14,2.125,5,20],\
        [3000,14,2.5,1,0],\
        [3000,14,2.5,1,5],\
        [3000,14,2.5,1,10],\
        [3000,14,2.5,1,15],\
        [3000,14,2.5,1,20],\
        [3000,14,2.5,2,0],\
        [3000,14,2.5,2,5],\
        [3000,14,2.5,2,10],\
        [3000,14,2.5,2,15],\
        [3000,14,2.5,2,20],\
        [3000,14,2.5,3,0],\
        [3000,14,2.5,3,5],\
        [3000,14,2.5,3,10],\
        [3000,14,2.5,3,15],\
        [3000,14,2.5,3,20],\
        [3000,14,2.5,4,0],\
        [3000,14,2.5,4,5],\
        [3000,14,2.5,4,10],\
        [3000,14,2.5,4,15],\
        [3000,14,2.5,4,20],\
        [3000,14,2.5,5,0],\
        [3000,14,2.5,5,5],\
        [3000,14,2.5,5,10],\
        [3000,14,2.5,5,15],\
        [3000,14,2.5,5,20],\
        [4000,2,1,1,0],\
        [4000,2,1,1,5],\
        [4000,2,1,1,10],\
        [4000,2,1,1,15],\
        [4000,2,1,1,20],\
        [4000,2,1,2,0],\
        [4000,2,1,2,5],\
        [4000,2,1,2,10],\
        [4000,2,1,2,15],\
        [4000,2,1,2,20],\
        [4000,2,1,3,0],\
        [4000,2,1,3,5],\
        [4000,2,1,3,10],\
        [4000,2,1,3,15],\
        [4000,2,1,3,20],\
        [4000,2,1,4,0],\
        [4000,2,1,4,5],\
        [4000,2,1,4,10],\
        [4000,2,1,4,15],\
        [4000,2,1,4,20],\
        [4000,2,1,5,0],\
        [4000,2,1,5,5],\
        [4000,2,1,5,10],\
        [4000,2,1,5,15],\
        [4000,2,1,5,20],\
        [4000,2,1.375,1,0],\
        [4000,2,1.375,1,5],\
        [4000,2,1.375,1,10],\
        [4000,2,1.375,1,15],\
        [4000,2,1.375,1,20],\
        [4000,2,1.375,2,0],\
        [4000,2,1.375,2,5],\
        [4000,2,1.375,2,10],\
        [4000,2,1.375,2,15],\
        [4000,2,1.375,2,20],\
        [4000,2,1.375,3,0],\
        [4000,2,1.375,3,5],\
        [4000,2,1.375,3,10],\
        [4000,2,1.375,3,15],\
        [4000,2,1.375,3,20],\
        [4000,2,1.375,4,0],\
        [4000,2,1.375,4,5],\
        [4000,2,1.375,4,10],\
        [4000,2,1.375,4,15],\
        [4000,2,1.375,4,20],\
        [4000,2,1.375,5,0],\
        [4000,2,1.375,5,5],\
        [4000,2,1.375,5,10],\
        [4000,2,1.375,5,15],\
        [4000,2,1.375,5,20],\
        [4000,2,1.75,1,0],\
        [4000,2,1.75,1,5],\
        [4000,2,1.75,1,10],\
        [4000,2,1.75,1,15],\
        [4000,2,1.75,1,20],\
        [4000,2,1.75,2,0],\
        [4000,2,1.75,2,5],\
        [4000,2,1.75,2,10],\
        [4000,2,1.75,2,15],\
        [4000,2,1.75,2,20],\
        [4000,2,1.75,3,0],\
        [4000,2,1.75,3,5],\
        [4000,2,1.75,3,10],\
        [4000,2,1.75,3,15],\
        [4000,2,1.75,3,20],\
        [4000,2,1.75,4,0],\
        [4000,2,1.75,4,5],\
        [4000,2,1.75,4,10],\
        [4000,2,1.75,4,15],\
        [4000,2,1.75,4,20],\
        [4000,2,1.75,5,0],\
        [4000,2,1.75,5,5],\
        [4000,2,1.75,5,10],\
        [4000,2,1.75,5,15],\
        [4000,2,1.75,5,20],\
        [4000,2,2.125,1,0],\
        [4000,2,2.125,1,5],\
        [4000,2,2.125,1,10],\
        [4000,2,2.125,1,15],\
        [4000,2,2.125,1,20],\
        [4000,2,2.125,2,0],\
        [4000,2,2.125,2,5],\
        [4000,2,2.125,2,10],\
        [4000,2,2.125,2,15],\
        [4000,2,2.125,2,20],\
        [4000,2,2.125,3,0],\
        [4000,2,2.125,3,5],\
        [4000,2,2.125,3,10],\
        [4000,2,2.125,3,15],\
        [4000,2,2.125,3,20],\
        [4000,2,2.125,4,0],\
        [4000,2,2.125,4,5],\
        [4000,2,2.125,4,10],\
        [4000,2,2.125,4,15],\
        [4000,2,2.125,4,20],\
        [4000,2,2.125,5,0],\
        [4000,2,2.125,5,5],\
        [4000,2,2.125,5,10],\
        [4000,2,2.125,5,15],\
        [4000,2,2.125,5,20],\
        [4000,2,2.5,1,0],\
        [4000,2,2.5,1,5],\
        [4000,2,2.5,1,10],\
        [4000,2,2.5,1,15],\
        [4000,2,2.5,1,20],\
        [4000,2,2.5,2,0],\
        [4000,2,2.5,2,5],\
        [4000,2,2.5,2,10],\
        [4000,2,2.5,2,15],\
        [4000,2,2.5,2,20],\
        [4000,2,2.5,3,0],\
        [4000,2,2.5,3,5],\
        [4000,2,2.5,3,10],\
        [4000,2,2.5,3,15],\
        [4000,2,2.5,3,20],\
        [4000,2,2.5,4,0],\
        [4000,2,2.5,4,5],\
        [4000,2,2.5,4,10],\
        [4000,2,2.5,4,15],\
        [4000,2,2.5,4,20],\
        [4000,2,2.5,5,0],\
        [4000,2,2.5,5,5],\
        [4000,2,2.5,5,10],\
        [4000,2,2.5,5,15],\
        [4000,2,2.5,5,20],\
        [4000,5,1,1,0],\
        [4000,5,1,1,5],\
        [4000,5,1,1,10],\
        [4000,5,1,1,15],\
        [4000,5,1,1,20],\
        [4000,5,1,2,0],\
        [4000,5,1,2,5],\
        [4000,5,1,2,10],\
        [4000,5,1,2,15],\
        [4000,5,1,2,20],\
        [4000,5,1,3,0],\
        [4000,5,1,3,5],\
        [4000,5,1,3,10],\
        [4000,5,1,3,15],\
        [4000,5,1,3,20],\
        [4000,5,1,4,0],\
        [4000,5,1,4,5],\
        [4000,5,1,4,10],\
        [4000,5,1,4,15],\
        [4000,5,1,4,20],\
        [4000,5,1,5,0],\
        [4000,5,1,5,5],\
        [4000,5,1,5,10],\
        [4000,5,1,5,15],\
        [4000,5,1,5,20],\
        [4000,5,1.375,1,0],\
        [4000,5,1.375,1,5],\
        [4000,5,1.375,1,10],\
        [4000,5,1.375,1,15],\
        [4000,5,1.375,1,20],\
        [4000,5,1.375,2,0],\
        [4000,5,1.375,2,5],\
        [4000,5,1.375,2,10],\
        [4000,5,1.375,2,15],\
        [4000,5,1.375,2,20],\
        [4000,5,1.375,3,0],\
        [4000,5,1.375,3,5],\
        [4000,5,1.375,3,10],\
        [4000,5,1.375,3,15],\
        [4000,5,1.375,3,20],\
        [4000,5,1.375,4,0],\
        [4000,5,1.375,4,5],\
        [4000,5,1.375,4,10],\
        [4000,5,1.375,4,15],\
        [4000,5,1.375,4,20],\
        [4000,5,1.375,5,0],\
        [4000,5,1.375,5,5],\
        [4000,5,1.375,5,10],\
        [4000,5,1.375,5,15],\
        [4000,5,1.375,5,20],\
        [4000,5,1.75,1,0],\
        [4000,5,1.75,1,5],\
        [4000,5,1.75,1,10],\
        [4000,5,1.75,1,15],\
        [4000,5,1.75,1,20],\
        [4000,5,1.75,2,0],\
        [4000,5,1.75,2,5],\
        [4000,5,1.75,2,10],\
        [4000,5,1.75,2,15],\
        [4000,5,1.75,2,20],\
        [4000,5,1.75,3,0],\
        [4000,5,1.75,3,5],\
        [4000,5,1.75,3,10],\
        [4000,5,1.75,3,15],\
        [4000,5,1.75,3,20],\
        [4000,5,1.75,4,0],\
        [4000,5,1.75,4,5],\
        [4000,5,1.75,4,10],\
        [4000,5,1.75,4,15],\
        [4000,5,1.75,4,20],\
        [4000,5,1.75,5,0],\
        [4000,5,1.75,5,5],\
        [4000,5,1.75,5,10],\
        [4000,5,1.75,5,15],\
        [4000,5,1.75,5,20],\
        [4000,5,2.125,1,0],\
        [4000,5,2.125,1,5],\
        [4000,5,2.125,1,10],\
        [4000,5,2.125,1,15],\
        [4000,5,2.125,1,20],\
        [4000,5,2.125,2,0],\
        [4000,5,2.125,2,5],\
        [4000,5,2.125,2,10],\
        [4000,5,2.125,2,15],\
        [4000,5,2.125,2,20],\
        [4000,5,2.125,3,0],\
        [4000,5,2.125,3,5],\
        [4000,5,2.125,3,10],\
        [4000,5,2.125,3,15],\
        [4000,5,2.125,3,20],\
        [4000,5,2.125,4,0],\
        [4000,5,2.125,4,5],\
        [4000,5,2.125,4,10],\
        [4000,5,2.125,4,15],\
        [4000,5,2.125,4,20],\
        [4000,5,2.125,5,0],\
        [4000,5,2.125,5,5],\
        [4000,5,2.125,5,10],\
        [4000,5,2.125,5,15],\
        [4000,5,2.125,5,20],\
        [4000,5,2.5,1,0],\
        [4000,5,2.5,1,5],\
        [4000,5,2.5,1,10],\
        [4000,5,2.5,1,15],\
        [4000,5,2.5,1,20],\
        [4000,5,2.5,2,0],\
        [4000,5,2.5,2,5],\
        [4000,5,2.5,2,10],\
        [4000,5,2.5,2,15],\
        [4000,5,2.5,2,20],\
        [4000,5,2.5,3,0],\
        [4000,5,2.5,3,5],\
        [4000,5,2.5,3,10],\
        [4000,5,2.5,3,15],\
        [4000,5,2.5,3,20],\
        [4000,5,2.5,4,0],\
        [4000,5,2.5,4,5],\
        [4000,5,2.5,4,10],\
        [4000,5,2.5,4,15],\
        [4000,5,2.5,4,20],\
        [4000,5,2.5,5,0],\
        [4000,5,2.5,5,5],\
        [4000,5,2.5,5,10],\
        [4000,5,2.5,5,15],\
        [4000,5,2.5,5,20],\
        [4000,8,1,1,0],\
        [4000,8,1,1,5],\
        [4000,8,1,1,10],\
        [4000,8,1,1,15],\
        [4000,8,1,1,20],\
        [4000,8,1,2,0],\
        [4000,8,1,2,5],\
        [4000,8,1,2,10],\
        [4000,8,1,2,15],\
        [4000,8,1,2,20],\
        [4000,8,1,3,0],\
        [4000,8,1,3,5],\
        [4000,8,1,3,10],\
        [4000,8,1,3,15],\
        [4000,8,1,3,20],\
        [4000,8,1,4,0],\
        [4000,8,1,4,5],\
        [4000,8,1,4,10],\
        [4000,8,1,4,15],\
        [4000,8,1,4,20],\
        [4000,8,1,5,0],\
        [4000,8,1,5,5],\
        [4000,8,1,5,10],\
        [4000,8,1,5,15],\
        [4000,8,1,5,20],\
        [4000,8,1.375,1,0],\
        [4000,8,1.375,1,5],\
        [4000,8,1.375,1,10],\
        [4000,8,1.375,1,15],\
        [4000,8,1.375,1,20],\
        [4000,8,1.375,2,0],\
        [4000,8,1.375,2,5],\
        [4000,8,1.375,2,10],\
        [4000,8,1.375,2,15],\
        [4000,8,1.375,2,20],\
        [4000,8,1.375,3,0],\
        [4000,8,1.375,3,5],\
        [4000,8,1.375,3,10],\
        [4000,8,1.375,3,15],\
        [4000,8,1.375,3,20],\
        [4000,8,1.375,4,0],\
        [4000,8,1.375,4,5],\
        [4000,8,1.375,4,10],\
        [4000,8,1.375,4,15],\
        [4000,8,1.375,4,20],\
        [4000,8,1.375,5,0],\
        [4000,8,1.375,5,5],\
        [4000,8,1.375,5,10],\
        [4000,8,1.375,5,15],\
        [4000,8,1.375,5,20],\
        [4000,8,1.75,1,0],\
        [4000,8,1.75,1,5],\
        [4000,8,1.75,1,10],\
        [4000,8,1.75,1,15],\
        [4000,8,1.75,1,20],\
        [4000,8,1.75,2,0],\
        [4000,8,1.75,2,5],\
        [4000,8,1.75,2,10],\
        [4000,8,1.75,2,15],\
        [4000,8,1.75,2,20],\
        [4000,8,1.75,3,0],\
        [4000,8,1.75,3,5],\
        [4000,8,1.75,3,10],\
        [4000,8,1.75,3,15],\
        [4000,8,1.75,3,20],\
        [4000,8,1.75,4,0],\
        [4000,8,1.75,4,5],\
        [4000,8,1.75,4,10],\
        [4000,8,1.75,4,15],\
        [4000,8,1.75,4,20],\
        [4000,8,1.75,5,0],\
        [4000,8,1.75,5,5],\
        [4000,8,1.75,5,10],\
        [4000,8,1.75,5,15],\
        [4000,8,1.75,5,20],\
        [4000,8,2.125,1,0],\
        [4000,8,2.125,1,5],\
        [4000,8,2.125,1,10],\
        [4000,8,2.125,1,15],\
        [4000,8,2.125,1,20],\
        [4000,8,2.125,2,0],\
        [4000,8,2.125,2,5],\
        [4000,8,2.125,2,10],\
        [4000,8,2.125,2,15],\
        [4000,8,2.125,2,20],\
        [4000,8,2.125,3,0],\
        [4000,8,2.125,3,5],\
        [4000,8,2.125,3,10],\
        [4000,8,2.125,3,15],\
        [4000,8,2.125,3,20],\
        [4000,8,2.125,4,0],\
        [4000,8,2.125,4,5],\
        [4000,8,2.125,4,10],\
        [4000,8,2.125,4,15],\
        [4000,8,2.125,4,20],\
        [4000,8,2.125,5,0],\
        [4000,8,2.125,5,5],\
        [4000,8,2.125,5,10],\
        [4000,8,2.125,5,15],\
        [4000,8,2.125,5,20],\
        [4000,8,2.5,1,0],\
        [4000,8,2.5,1,5],\
        [4000,8,2.5,1,10],\
        [4000,8,2.5,1,15],\
        [4000,8,2.5,1,20],\
        [4000,8,2.5,2,0],\
        [4000,8,2.5,2,5],\
        [4000,8,2.5,2,10],\
        [4000,8,2.5,2,15],\
        [4000,8,2.5,2,20],\
        [4000,8,2.5,3,0],\
        [4000,8,2.5,3,5],\
        [4000,8,2.5,3,10],\
        [4000,8,2.5,3,15],\
        [4000,8,2.5,3,20],\
        [4000,8,2.5,4,0],\
        [4000,8,2.5,4,5],\
        [4000,8,2.5,4,10],\
        [4000,8,2.5,4,15],\
        [4000,8,2.5,4,20],\
        [4000,8,2.5,5,0],\
        [4000,8,2.5,5,5],\
        [4000,8,2.5,5,10],\
        [4000,8,2.5,5,15],\
        [4000,8,2.5,5,20],\
        [4000,11,1,1,0],\
        [4000,11,1,1,5],\
        [4000,11,1,1,10],\
        [4000,11,1,1,15],\
        [4000,11,1,1,20],\
        [4000,11,1,2,0],\
        [4000,11,1,2,5],\
        [4000,11,1,2,10],\
        [4000,11,1,2,15],\
        [4000,11,1,2,20],\
        [4000,11,1,3,0],\
        [4000,11,1,3,5],\
        [4000,11,1,3,10],\
        [4000,11,1,3,15],\
        [4000,11,1,3,20],\
        [4000,11,1,4,0],\
        [4000,11,1,4,5],\
        [4000,11,1,4,10],\
        [4000,11,1,4,15],\
        [4000,11,1,4,20],\
        [4000,11,1,5,0],\
        [4000,11,1,5,5],\
        [4000,11,1,5,10],\
        [4000,11,1,5,15],\
        [4000,11,1,5,20],\
        [4000,11,1.375,1,0],\
        [4000,11,1.375,1,5],\
        [4000,11,1.375,1,10],\
        [4000,11,1.375,1,15],\
        [4000,11,1.375,1,20],\
        [4000,11,1.375,2,0],\
        [4000,11,1.375,2,5],\
        [4000,11,1.375,2,10],\
        [4000,11,1.375,2,15],\
        [4000,11,1.375,2,20],\
        [4000,11,1.375,3,0],\
        [4000,11,1.375,3,5],\
        [4000,11,1.375,3,10],\
        [4000,11,1.375,3,15],\
        [4000,11,1.375,3,20],\
        [4000,11,1.375,4,0],\
        [4000,11,1.375,4,5],\
        [4000,11,1.375,4,10],\
        [4000,11,1.375,4,15],\
        [4000,11,1.375,4,20],\
        [4000,11,1.375,5,0],\
        [4000,11,1.375,5,5],\
        [4000,11,1.375,5,10],\
        [4000,11,1.375,5,15],\
        [4000,11,1.375,5,20],\
        [4000,11,1.75,1,0],\
        [4000,11,1.75,1,5],\
        [4000,11,1.75,1,10],\
        [4000,11,1.75,1,15],\
        [4000,11,1.75,1,20],\
        [4000,11,1.75,2,0],\
        [4000,11,1.75,2,5],\
        [4000,11,1.75,2,10],\
        [4000,11,1.75,2,15],\
        [4000,11,1.75,2,20],\
        [4000,11,1.75,3,0],\
        [4000,11,1.75,3,5],\
        [4000,11,1.75,3,10],\
        [4000,11,1.75,3,15],\
        [4000,11,1.75,3,20],\
        [4000,11,1.75,4,0],\
        [4000,11,1.75,4,5],\
        [4000,11,1.75,4,10],\
        [4000,11,1.75,4,15],\
        [4000,11,1.75,4,20],\
        [4000,11,1.75,5,0],\
        [4000,11,1.75,5,5],\
        [4000,11,1.75,5,10],\
        [4000,11,1.75,5,15],\
        [4000,11,1.75,5,20],\
        [4000,11,2.125,1,0],\
        [4000,11,2.125,1,5],\
        [4000,11,2.125,1,10],\
        [4000,11,2.125,1,15],\
        [4000,11,2.125,1,20],\
        [4000,11,2.125,2,0],\
        [4000,11,2.125,2,5],\
        [4000,11,2.125,2,10],\
        [4000,11,2.125,2,15],\
        [4000,11,2.125,2,20],\
        [4000,11,2.125,3,0],\
        [4000,11,2.125,3,5],\
        [4000,11,2.125,3,10],\
        [4000,11,2.125,3,15],\
        [4000,11,2.125,3,20],\
        [4000,11,2.125,4,0],\
        [4000,11,2.125,4,5],\
        [4000,11,2.125,4,10],\
        [4000,11,2.125,4,15],\
        [4000,11,2.125,4,20],\
        [4000,11,2.125,5,0],\
        [4000,11,2.125,5,5],\
        [4000,11,2.125,5,10],\
        [4000,11,2.125,5,15],\
        [4000,11,2.125,5,20],\
        [4000,11,2.5,1,0],\
        [4000,11,2.5,1,5],\
        [4000,11,2.5,1,10],\
        [4000,11,2.5,1,15],\
        [4000,11,2.5,1,20],\
        [4000,11,2.5,2,0],\
        [4000,11,2.5,2,5],\
        [4000,11,2.5,2,10],\
        [4000,11,2.5,2,15],\
        [4000,11,2.5,2,20],\
        [4000,11,2.5,3,0],\
        [4000,11,2.5,3,5],\
        [4000,11,2.5,3,10],\
        [4000,11,2.5,3,15],\
        [4000,11,2.5,3,20],\
        [4000,11,2.5,4,0],\
        [4000,11,2.5,4,5],\
        [4000,11,2.5,4,10],\
        [4000,11,2.5,4,15],\
        [4000,11,2.5,4,20],\
        [4000,11,2.5,5,0],\
        [4000,11,2.5,5,5],\
        [4000,11,2.5,5,10],\
        [4000,11,2.5,5,15],\
        [4000,11,2.5,5,20],\
        [4000,14,1,1,0],\
        [4000,14,1,1,5],\
        [4000,14,1,1,10],\
        [4000,14,1,1,15],\
        [4000,14,1,1,20],\
        [4000,14,1,2,0],\
        [4000,14,1,2,5],\
        [4000,14,1,2,10],\
        [4000,14,1,2,15],\
        [4000,14,1,2,20],\
        [4000,14,1,3,0],\
        [4000,14,1,3,5],\
        [4000,14,1,3,10],\
        [4000,14,1,3,15],\
        [4000,14,1,3,20],\
        [4000,14,1,4,0],\
        [4000,14,1,4,5],\
        [4000,14,1,4,10],\
        [4000,14,1,4,15],\
        [4000,14,1,4,20],\
        [4000,14,1,5,0],\
        [4000,14,1,5,5],\
        [4000,14,1,5,10],\
        [4000,14,1,5,15],\
        [4000,14,1,5,20],\
        [4000,14,1.375,1,0],\
        [4000,14,1.375,1,5],\
        [4000,14,1.375,1,10],\
        [4000,14,1.375,1,15],\
        [4000,14,1.375,1,20],\
        [4000,14,1.375,2,0],\
        [4000,14,1.375,2,5],\
        [4000,14,1.375,2,10],\
        [4000,14,1.375,2,15],\
        [4000,14,1.375,2,20],\
        [4000,14,1.375,3,0],\
        [4000,14,1.375,3,5],\
        [4000,14,1.375,3,10],\
        [4000,14,1.375,3,15],\
        [4000,14,1.375,3,20],\
        [4000,14,1.375,4,0],\
        [4000,14,1.375,4,5],\
        [4000,14,1.375,4,10],\
        [4000,14,1.375,4,15],\
        [4000,14,1.375,4,20],\
        [4000,14,1.375,5,0],\
        [4000,14,1.375,5,5],\
        [4000,14,1.375,5,10],\
        [4000,14,1.375,5,15],\
        [4000,14,1.375,5,20],\
        [4000,14,1.75,1,0],\
        [4000,14,1.75,1,5],\
        [4000,14,1.75,1,10],\
        [4000,14,1.75,1,15],\
        [4000,14,1.75,1,20],\
        [4000,14,1.75,2,0],\
        [4000,14,1.75,2,5],\
        [4000,14,1.75,2,10],\
        [4000,14,1.75,2,15],\
        [4000,14,1.75,2,20],\
        [4000,14,1.75,3,0],\
        [4000,14,1.75,3,5],\
        [4000,14,1.75,3,10],\
        [4000,14,1.75,3,15],\
        [4000,14,1.75,3,20],\
        [4000,14,1.75,4,0],\
        [4000,14,1.75,4,5],\
        [4000,14,1.75,4,10],\
        [4000,14,1.75,4,15],\
        [4000,14,1.75,4,20],\
        [4000,14,1.75,5,0],\
        [4000,14,1.75,5,5],\
        [4000,14,1.75,5,10],\
        [4000,14,1.75,5,15],\
        [4000,14,1.75,5,20],\
        [4000,14,2.125,1,0],\
        [4000,14,2.125,1,5],\
        [4000,14,2.125,1,10],\
        [4000,14,2.125,1,15],\
        [4000,14,2.125,1,20],\
        [4000,14,2.125,2,0],\
        [4000,14,2.125,2,5],\
        [4000,14,2.125,2,10],\
        [4000,14,2.125,2,15],\
        [4000,14,2.125,2,20],\
        [4000,14,2.125,3,0],\
        [4000,14,2.125,3,5],\
        [4000,14,2.125,3,10],\
        [4000,14,2.125,3,15],\
        [4000,14,2.125,3,20],\
        [4000,14,2.125,4,0],\
        [4000,14,2.125,4,5],\
        [4000,14,2.125,4,10],\
        [4000,14,2.125,4,15],\
        [4000,14,2.125,4,20],\
        [4000,14,2.125,5,0],\
        [4000,14,2.125,5,5],\
        [4000,14,2.125,5,10],\
        [4000,14,2.125,5,15],\
        [4000,14,2.125,5,20],\
        [4000,14,2.5,1,0],\
        [4000,14,2.5,1,5],\
        [4000,14,2.5,1,10],\
        [4000,14,2.5,1,15],\
        [4000,14,2.5,1,20],\
        [4000,14,2.5,2,0],\
        [4000,14,2.5,2,5],\
        [4000,14,2.5,2,10],\
        [4000,14,2.5,2,15],\
        [4000,14,2.5,2,20],\
        [4000,14,2.5,3,0],\
        [4000,14,2.5,3,5],\
        [4000,14,2.5,3,10],\
        [4000,14,2.5,3,15],\
        [4000,14,2.5,3,20],\
        [4000,14,2.5,4,0],\
        [4000,14,2.5,4,5],\
        [4000,14,2.5,4,10],\
        [4000,14,2.5,4,15],\
        [4000,14,2.5,4,20],\
        [4000,14,2.5,5,0],\
        [4000,14,2.5,5,5],\
        [4000,14,2.5,5,10],\
        [4000,14,2.5,5,15],\
        [4000,14,2.5,5,20],\
        [5000,2,1,1,0],\
        [5000,2,1,1,5],\
        [5000,2,1,1,10],\
        [5000,2,1,1,15],\
        [5000,2,1,1,20],\
        [5000,2,1,2,0],\
        [5000,2,1,2,5],\
        [5000,2,1,2,10],\
        [5000,2,1,2,15],\
        [5000,2,1,2,20],\
        [5000,2,1,3,0],\
        [5000,2,1,3,5],\
        [5000,2,1,3,10],\
        [5000,2,1,3,15],\
        [5000,2,1,3,20],\
        [5000,2,1,4,0],\
        [5000,2,1,4,5],\
        [5000,2,1,4,10],\
        [5000,2,1,4,15],\
        [5000,2,1,4,20],\
        [5000,2,1,5,0],\
        [5000,2,1,5,5],\
        [5000,2,1,5,10],\
        [5000,2,1,5,15],\
        [5000,2,1,5,20],\
        [5000,2,1.375,1,0],\
        [5000,2,1.375,1,5],\
        [5000,2,1.375,1,10],\
        [5000,2,1.375,1,15],\
        [5000,2,1.375,1,20],\
        [5000,2,1.375,2,0],\
        [5000,2,1.375,2,5],\
        [5000,2,1.375,2,10],\
        [5000,2,1.375,2,15],\
        [5000,2,1.375,2,20],\
        [5000,2,1.375,3,0],\
        [5000,2,1.375,3,5],\
        [5000,2,1.375,3,10],\
        [5000,2,1.375,3,15],\
        [5000,2,1.375,3,20],\
        [5000,2,1.375,4,0],\
        [5000,2,1.375,4,5],\
        [5000,2,1.375,4,10],\
        [5000,2,1.375,4,15],\
        [5000,2,1.375,4,20],\
        [5000,2,1.375,5,0],\
        [5000,2,1.375,5,5],\
        [5000,2,1.375,5,10],\
        [5000,2,1.375,5,15],\
        [5000,2,1.375,5,20],\
        [5000,2,1.75,1,0],\
        [5000,2,1.75,1,5],\
        [5000,2,1.75,1,10],\
        [5000,2,1.75,1,15],\
        [5000,2,1.75,1,20],\
        [5000,2,1.75,2,0],\
        [5000,2,1.75,2,5],\
        [5000,2,1.75,2,10],\
        [5000,2,1.75,2,15],\
        [5000,2,1.75,2,20],\
        [5000,2,1.75,3,0],\
        [5000,2,1.75,3,5],\
        [5000,2,1.75,3,10],\
        [5000,2,1.75,3,15],\
        [5000,2,1.75,3,20],\
        [5000,2,1.75,4,0],\
        [5000,2,1.75,4,5],\
        [5000,2,1.75,4,10],\
        [5000,2,1.75,4,15],\
        [5000,2,1.75,4,20],\
        [5000,2,1.75,5,0],\
        [5000,2,1.75,5,5],\
        [5000,2,1.75,5,10],\
        [5000,2,1.75,5,15],\
        [5000,2,1.75,5,20],\
        [5000,2,2.125,1,0],\
        [5000,2,2.125,1,5],\
        [5000,2,2.125,1,10],\
        [5000,2,2.125,1,15],\
        [5000,2,2.125,1,20],\
        [5000,2,2.125,2,0],\
        [5000,2,2.125,2,5],\
        [5000,2,2.125,2,10],\
        [5000,2,2.125,2,15],\
        [5000,2,2.125,2,20],\
        [5000,2,2.125,3,0],\
        [5000,2,2.125,3,5],\
        [5000,2,2.125,3,10],\
        [5000,2,2.125,3,15],\
        [5000,2,2.125,3,20],\
        [5000,2,2.125,4,0],\
        [5000,2,2.125,4,5],\
        [5000,2,2.125,4,10],\
        [5000,2,2.125,4,15],\
        [5000,2,2.125,4,20],\
        [5000,2,2.125,5,0],\
        [5000,2,2.125,5,5],\
        [5000,2,2.125,5,10],\
        [5000,2,2.125,5,15],\
        [5000,2,2.125,5,20],\
        [5000,2,2.5,1,0],\
        [5000,2,2.5,1,5],\
        [5000,2,2.5,1,10],\
        [5000,2,2.5,1,15],\
        [5000,2,2.5,1,20],\
        [5000,2,2.5,2,0],\
        [5000,2,2.5,2,5],\
        [5000,2,2.5,2,10],\
        [5000,2,2.5,2,15],\
        [5000,2,2.5,2,20],\
        [5000,2,2.5,3,0],\
        [5000,2,2.5,3,5],\
        [5000,2,2.5,3,10],\
        [5000,2,2.5,3,15],\
        [5000,2,2.5,3,20],\
        [5000,2,2.5,4,0],\
        [5000,2,2.5,4,5],\
        [5000,2,2.5,4,10],\
        [5000,2,2.5,4,15],\
        [5000,2,2.5,4,20],\
        [5000,2,2.5,5,0],\
        [5000,2,2.5,5,5],\
        [5000,2,2.5,5,10],\
        [5000,2,2.5,5,15],\
        [5000,2,2.5,5,20],\
        [5000,5,1,1,0],\
        [5000,5,1,1,5],\
        [5000,5,1,1,10],\
        [5000,5,1,1,15],\
        [5000,5,1,1,20],\
        [5000,5,1,2,0],\
        [5000,5,1,2,5],\
        [5000,5,1,2,10],\
        [5000,5,1,2,15],\
        [5000,5,1,2,20],\
        [5000,5,1,3,0],\
        [5000,5,1,3,5],\
        [5000,5,1,3,10],\
        [5000,5,1,3,15],\
        [5000,5,1,3,20],\
        [5000,5,1,4,0],\
        [5000,5,1,4,5],\
        [5000,5,1,4,10],\
        [5000,5,1,4,15],\
        [5000,5,1,4,20],\
        [5000,5,1,5,0],\
        [5000,5,1,5,5],\
        [5000,5,1,5,10],\
        [5000,5,1,5,15],\
        [5000,5,1,5,20],\
        [5000,5,1.375,1,0],\
        [5000,5,1.375,1,5],\
        [5000,5,1.375,1,10],\
        [5000,5,1.375,1,15],\
        [5000,5,1.375,1,20],\
        [5000,5,1.375,2,0],\
        [5000,5,1.375,2,5],\
        [5000,5,1.375,2,10],\
        [5000,5,1.375,2,15],\
        [5000,5,1.375,2,20],\
        [5000,5,1.375,3,0],\
        [5000,5,1.375,3,5],\
        [5000,5,1.375,3,10],\
        [5000,5,1.375,3,15],\
        [5000,5,1.375,3,20],\
        [5000,5,1.375,4,0],\
        [5000,5,1.375,4,5],\
        [5000,5,1.375,4,10],\
        [5000,5,1.375,4,15],\
        [5000,5,1.375,4,20],\
        [5000,5,1.375,5,0],\
        [5000,5,1.375,5,5],\
        [5000,5,1.375,5,10],\
        [5000,5,1.375,5,15],\
        [5000,5,1.375,5,20],\
        [5000,5,1.75,1,0],\
        [5000,5,1.75,1,5],\
        [5000,5,1.75,1,10],\
        [5000,5,1.75,1,15],\
        [5000,5,1.75,1,20],\
        [5000,5,1.75,2,0],\
        [5000,5,1.75,2,5],\
        [5000,5,1.75,2,10],\
        [5000,5,1.75,2,15],\
        [5000,5,1.75,2,20],\
        [5000,5,1.75,3,0],\
        [5000,5,1.75,3,5],\
        [5000,5,1.75,3,10],\
        [5000,5,1.75,3,15],\
        [5000,5,1.75,3,20],\
        [5000,5,1.75,4,0],\
        [5000,5,1.75,4,5],\
        [5000,5,1.75,4,10],\
        [5000,5,1.75,4,15],\
        [5000,5,1.75,4,20],\
        [5000,5,1.75,5,0],\
        [5000,5,1.75,5,5],\
        [5000,5,1.75,5,10],\
        [5000,5,1.75,5,15],\
        [5000,5,1.75,5,20],\
        [5000,5,2.125,1,0],\
        [5000,5,2.125,1,5],\
        [5000,5,2.125,1,10],\
        [5000,5,2.125,1,15],\
        [5000,5,2.125,1,20],\
        [5000,5,2.125,2,0],\
        [5000,5,2.125,2,5],\
        [5000,5,2.125,2,10],\
        [5000,5,2.125,2,15],\
        [5000,5,2.125,2,20],\
        [5000,5,2.125,3,0],\
        [5000,5,2.125,3,5],\
        [5000,5,2.125,3,10],\
        [5000,5,2.125,3,15],\
        [5000,5,2.125,3,20],\
        [5000,5,2.125,4,0],\
        [5000,5,2.125,4,5],\
        [5000,5,2.125,4,10],\
        [5000,5,2.125,4,15],\
        [5000,5,2.125,4,20],\
        [5000,5,2.125,5,0],\
        [5000,5,2.125,5,5],\
        [5000,5,2.125,5,10],\
        [5000,5,2.125,5,15],\
        [5000,5,2.125,5,20],\
        [5000,5,2.5,1,0],\
        [5000,5,2.5,1,5],\
        [5000,5,2.5,1,10],\
        [5000,5,2.5,1,15],\
        [5000,5,2.5,1,20],\
        [5000,5,2.5,2,0],\
        [5000,5,2.5,2,5],\
        [5000,5,2.5,2,10],\
        [5000,5,2.5,2,15],\
        [5000,5,2.5,2,20],\
        [5000,5,2.5,3,0],\
        [5000,5,2.5,3,5],\
        [5000,5,2.5,3,10],\
        [5000,5,2.5,3,15],\
        [5000,5,2.5,3,20],\
        [5000,5,2.5,4,0],\
        [5000,5,2.5,4,5],\
        [5000,5,2.5,4,10],\
        [5000,5,2.5,4,15],\
        [5000,5,2.5,4,20],\
        [5000,5,2.5,5,0],\
        [5000,5,2.5,5,5],\
        [5000,5,2.5,5,10],\
        [5000,5,2.5,5,15],\
        [5000,5,2.5,5,20],\
        [5000,8,1,1,0],\
        [5000,8,1,1,5],\
        [5000,8,1,1,10],\
        [5000,8,1,1,15],\
        [5000,8,1,1,20],\
        [5000,8,1,2,0],\
        [5000,8,1,2,5],\
        [5000,8,1,2,10],\
        [5000,8,1,2,15],\
        [5000,8,1,2,20],\
        [5000,8,1,3,0],\
        [5000,8,1,3,5],\
        [5000,8,1,3,10],\
        [5000,8,1,3,15],\
        [5000,8,1,3,20],\
        [5000,8,1,4,0],\
        [5000,8,1,4,5],\
        [5000,8,1,4,10],\
        [5000,8,1,4,15],\
        [5000,8,1,4,20],\
        [5000,8,1,5,0],\
        [5000,8,1,5,5],\
        [5000,8,1,5,10],\
        [5000,8,1,5,15],\
        [5000,8,1,5,20],\
        [5000,8,1.375,1,0],\
        [5000,8,1.375,1,5],\
        [5000,8,1.375,1,10],\
        [5000,8,1.375,1,15],\
        [5000,8,1.375,1,20],\
        [5000,8,1.375,2,0],\
        [5000,8,1.375,2,5],\
        [5000,8,1.375,2,10],\
        [5000,8,1.375,2,15],\
        [5000,8,1.375,2,20],\
        [5000,8,1.375,3,0],\
        [5000,8,1.375,3,5],\
        [5000,8,1.375,3,10],\
        [5000,8,1.375,3,15],\
        [5000,8,1.375,3,20],\
        [5000,8,1.375,4,0],\
        [5000,8,1.375,4,5],\
        [5000,8,1.375,4,10],\
        [5000,8,1.375,4,15],\
        [5000,8,1.375,4,20],\
        [5000,8,1.375,5,0],\
        [5000,8,1.375,5,5],\
        [5000,8,1.375,5,10],\
        [5000,8,1.375,5,15],\
        [5000,8,1.375,5,20],\
        [5000,8,1.75,1,0],\
        [5000,8,1.75,1,5],\
        [5000,8,1.75,1,10],\
        [5000,8,1.75,1,15],\
        [5000,8,1.75,1,20],\
        [5000,8,1.75,2,0],\
        [5000,8,1.75,2,5],\
        [5000,8,1.75,2,10],\
        [5000,8,1.75,2,15],\
        [5000,8,1.75,2,20],\
        [5000,8,1.75,3,0],\
        [5000,8,1.75,3,5],\
        [5000,8,1.75,3,10],\
        [5000,8,1.75,3,15],\
        [5000,8,1.75,3,20],\
        [5000,8,1.75,4,0],\
        [5000,8,1.75,4,5],\
        [5000,8,1.75,4,10],\
        [5000,8,1.75,4,15],\
        [5000,8,1.75,4,20],\
        [5000,8,1.75,5,0],\
        [5000,8,1.75,5,5],\
        [5000,8,1.75,5,10],\
        [5000,8,1.75,5,15],\
        [5000,8,1.75,5,20],\
        [5000,8,2.125,1,0],\
        [5000,8,2.125,1,5],\
        [5000,8,2.125,1,10],\
        [5000,8,2.125,1,15],\
        [5000,8,2.125,1,20],\
        [5000,8,2.125,2,0],\
        [5000,8,2.125,2,5],\
        [5000,8,2.125,2,10],\
        [5000,8,2.125,2,15],\
        [5000,8,2.125,2,20],\
        [5000,8,2.125,3,0],\
        [5000,8,2.125,3,5],\
        [5000,8,2.125,3,10],\
        [5000,8,2.125,3,15],\
        [5000,8,2.125,3,20],\
        [5000,8,2.125,4,0],\
        [5000,8,2.125,4,5],\
        [5000,8,2.125,4,10],\
        [5000,8,2.125,4,15],\
        [5000,8,2.125,4,20],\
        [5000,8,2.125,5,0],\
        [5000,8,2.125,5,5],\
        [5000,8,2.125,5,10],\
        [5000,8,2.125,5,15],\
        [5000,8,2.125,5,20],\
        [5000,8,2.5,1,0],\
        [5000,8,2.5,1,5],\
        [5000,8,2.5,1,10],\
        [5000,8,2.5,1,15],\
        [5000,8,2.5,1,20],\
        [5000,8,2.5,2,0],\
        [5000,8,2.5,2,5],\
        [5000,8,2.5,2,10],\
        [5000,8,2.5,2,15],\
        [5000,8,2.5,2,20],\
        [5000,8,2.5,3,0],\
        [5000,8,2.5,3,5],\
        [5000,8,2.5,3,10],\
        [5000,8,2.5,3,15],\
        [5000,8,2.5,3,20],\
        [5000,8,2.5,4,0],\
        [5000,8,2.5,4,5],\
        [5000,8,2.5,4,10],\
        [5000,8,2.5,4,15],\
        [5000,8,2.5,4,20],\
        [5000,8,2.5,5,0],\
        [5000,8,2.5,5,5],\
        [5000,8,2.5,5,10],\
        [5000,8,2.5,5,15],\
        [5000,8,2.5,5,20],\
        [5000,11,1,1,0],\
        [5000,11,1,1,5],\
        [5000,11,1,1,10],\
        [5000,11,1,1,15],\
        [5000,11,1,1,20],\
        [5000,11,1,2,0],\
        [5000,11,1,2,5],\
        [5000,11,1,2,10],\
        [5000,11,1,2,15],\
        [5000,11,1,2,20],\
        [5000,11,1,3,0],\
        [5000,11,1,3,5],\
        [5000,11,1,3,10],\
        [5000,11,1,3,15],\
        [5000,11,1,3,20],\
        [5000,11,1,4,0],\
        [5000,11,1,4,5],\
        [5000,11,1,4,10],\
        [5000,11,1,4,15],\
        [5000,11,1,4,20],\
        [5000,11,1,5,0],\
        [5000,11,1,5,5],\
        [5000,11,1,5,10],\
        [5000,11,1,5,15],\
        [5000,11,1,5,20],\
        [5000,11,1.375,1,0],\
        [5000,11,1.375,1,5],\
        [5000,11,1.375,1,10],\
        [5000,11,1.375,1,15],\
        [5000,11,1.375,1,20],\
        [5000,11,1.375,2,0],\
        [5000,11,1.375,2,5],\
        [5000,11,1.375,2,10],\
        [5000,11,1.375,2,15],\
        [5000,11,1.375,2,20],\
        [5000,11,1.375,3,0],\
        [5000,11,1.375,3,5],\
        [5000,11,1.375,3,10],\
        [5000,11,1.375,3,15],\
        [5000,11,1.375,3,20],\
        [5000,11,1.375,4,0],\
        [5000,11,1.375,4,5],\
        [5000,11,1.375,4,10],\
        [5000,11,1.375,4,15],\
        [5000,11,1.375,4,20],\
        [5000,11,1.375,5,0],\
        [5000,11,1.375,5,5],\
        [5000,11,1.375,5,10],\
        [5000,11,1.375,5,15],\
        [5000,11,1.375,5,20],\
        [5000,11,1.75,1,0],\
        [5000,11,1.75,1,5],\
        [5000,11,1.75,1,10],\
        [5000,11,1.75,1,15],\
        [5000,11,1.75,1,20],\
        [5000,11,1.75,2,0],\
        [5000,11,1.75,2,5],\
        [5000,11,1.75,2,10],\
        [5000,11,1.75,2,15],\
        [5000,11,1.75,2,20],\
        [5000,11,1.75,3,0],\
        [5000,11,1.75,3,5],\
        [5000,11,1.75,3,10],\
        [5000,11,1.75,3,15],\
        [5000,11,1.75,3,20],\
        [5000,11,1.75,4,0],\
        [5000,11,1.75,4,5],\
        [5000,11,1.75,4,10],\
        [5000,11,1.75,4,15],\
        [5000,11,1.75,4,20],\
        [5000,11,1.75,5,0],\
        [5000,11,1.75,5,5],\
        [5000,11,1.75,5,10],\
        [5000,11,1.75,5,15],\
        [5000,11,1.75,5,20],\
        [5000,11,2.125,1,0],\
        [5000,11,2.125,1,5],\
        [5000,11,2.125,1,10],\
        [5000,11,2.125,1,15],\
        [5000,11,2.125,1,20],\
        [5000,11,2.125,2,0],\
        [5000,11,2.125,2,5],\
        [5000,11,2.125,2,10],\
        [5000,11,2.125,2,15],\
        [5000,11,2.125,2,20],\
        [5000,11,2.125,3,0],\
        [5000,11,2.125,3,5],\
        [5000,11,2.125,3,10],\
        [5000,11,2.125,3,15],\
        [5000,11,2.125,3,20],\
        [5000,11,2.125,4,0],\
        [5000,11,2.125,4,5],\
        [5000,11,2.125,4,10],\
        [5000,11,2.125,4,15],\
        [5000,11,2.125,4,20],\
        [5000,11,2.125,5,0],\
        [5000,11,2.125,5,5],\
        [5000,11,2.125,5,10],\
        [5000,11,2.125,5,15],\
        [5000,11,2.125,5,20],\
        [5000,11,2.5,1,0],\
        [5000,11,2.5,1,5],\
        [5000,11,2.5,1,10],\
        [5000,11,2.5,1,15],\
        [5000,11,2.5,1,20],\
        [5000,11,2.5,2,0],\
        [5000,11,2.5,2,5],\
        [5000,11,2.5,2,10],\
        [5000,11,2.5,2,15],\
        [5000,11,2.5,2,20],\
        [5000,11,2.5,3,0],\
        [5000,11,2.5,3,5],\
        [5000,11,2.5,3,10],\
        [5000,11,2.5,3,15],\
        [5000,11,2.5,3,20],\
        [5000,11,2.5,4,0],\
        [5000,11,2.5,4,5],\
        [5000,11,2.5,4,10],\
        [5000,11,2.5,4,15],\
        [5000,11,2.5,4,20],\
        [5000,11,2.5,5,0],\
        [5000,11,2.5,5,5],\
        [5000,11,2.5,5,10],\
        [5000,11,2.5,5,15],\
        [5000,11,2.5,5,20],\
        [5000,14,1,1,0],\
        [5000,14,1,1,5],\
        [5000,14,1,1,10],\
        [5000,14,1,1,15],\
        [5000,14,1,1,20],\
        [5000,14,1,2,0],\
        [5000,14,1,2,5],\
        [5000,14,1,2,10],\
        [5000,14,1,2,15],\
        [5000,14,1,2,20],\
        [5000,14,1,3,0],\
        [5000,14,1,3,5],\
        [5000,14,1,3,10],\
        [5000,14,1,3,15],\
        [5000,14,1,3,20],\
        [5000,14,1,4,0],\
        [5000,14,1,4,5],\
        [5000,14,1,4,10],\
        [5000,14,1,4,15],\
        [5000,14,1,4,20],\
        [5000,14,1,5,0],\
        [5000,14,1,5,5],\
        [5000,14,1,5,10],\
        [5000,14,1,5,15],\
        [5000,14,1,5,20],\
        [5000,14,1.375,1,0],\
        [5000,14,1.375,1,5],\
        [5000,14,1.375,1,10],\
        [5000,14,1.375,1,15],\
        [5000,14,1.375,1,20],\
        [5000,14,1.375,2,0],\
        [5000,14,1.375,2,5],\
        [5000,14,1.375,2,10],\
        [5000,14,1.375,2,15],\
        [5000,14,1.375,2,20],\
        [5000,14,1.375,3,0],\
        [5000,14,1.375,3,5],\
        [5000,14,1.375,3,10],\
        [5000,14,1.375,3,15],\
        [5000,14,1.375,3,20],\
        [5000,14,1.375,4,0],\
        [5000,14,1.375,4,5],\
        [5000,14,1.375,4,10],\
        [5000,14,1.375,4,15],\
        [5000,14,1.375,4,20],\
        [5000,14,1.375,5,0],\
        [5000,14,1.375,5,5],\
        [5000,14,1.375,5,10],\
        [5000,14,1.375,5,15],\
        [5000,14,1.375,5,20],\
        [5000,14,1.75,1,0],\
        [5000,14,1.75,1,5],\
        [5000,14,1.75,1,10],\
        [5000,14,1.75,1,15],\
        [5000,14,1.75,1,20],\
        [5000,14,1.75,2,0],\
        [5000,14,1.75,2,5],\
        [5000,14,1.75,2,10],\
        [5000,14,1.75,2,15],\
        [5000,14,1.75,2,20],\
        [5000,14,1.75,3,0],\
        [5000,14,1.75,3,5],\
        [5000,14,1.75,3,10],\
        [5000,14,1.75,3,15],\
        [5000,14,1.75,3,20],\
        [5000,14,1.75,4,0],\
        [5000,14,1.75,4,5],\
        [5000,14,1.75,4,10],\
        [5000,14,1.75,4,15],\
        [5000,14,1.75,4,20],\
        [5000,14,1.75,5,0],\
        [5000,14,1.75,5,5],\
        [5000,14,1.75,5,10],\
        [5000,14,1.75,5,15],\
        [5000,14,1.75,5,20],\
        [5000,14,2.125,1,0],\
        [5000,14,2.125,1,5],\
        [5000,14,2.125,1,10],\
        [5000,14,2.125,1,15],\
        [5000,14,2.125,1,20],\
        [5000,14,2.125,2,0],\
        [5000,14,2.125,2,5],\
        [5000,14,2.125,2,10],\
        [5000,14,2.125,2,15],\
        [5000,14,2.125,2,20],\
        [5000,14,2.125,3,0],\
        [5000,14,2.125,3,5],\
        [5000,14,2.125,3,10],\
        [5000,14,2.125,3,15],\
        [5000,14,2.125,3,20],\
        [5000,14,2.125,4,0],\
        [5000,14,2.125,4,5],\
        [5000,14,2.125,4,10],\
        [5000,14,2.125,4,15],\
        [5000,14,2.125,4,20],\
        [5000,14,2.125,5,0],\
        [5000,14,2.125,5,5],\
        [5000,14,2.125,5,10],\
        [5000,14,2.125,5,15],\
        [5000,14,2.125,5,20],\
        [5000,14,2.5,1,0],\
        [5000,14,2.5,1,5],\
        [5000,14,2.5,1,10],\
        [5000,14,2.5,1,15],\
        [5000,14,2.5,1,20],\
        [5000,14,2.5,2,0],\
        [5000,14,2.5,2,5],\
        [5000,14,2.5,2,10],\
        [5000,14,2.5,2,15],\
        [5000,14,2.5,2,20],\
        [5000,14,2.5,3,0],\
        [5000,14,2.5,3,5],\
        [5000,14,2.5,3,10],\
        [5000,14,2.5,3,15],\
        [5000,14,2.5,3,20],\
        [5000,14,2.5,4,0],\
        [5000,14,2.5,4,5],\
        [5000,14,2.5,4,10],\
        [5000,14,2.5,4,15],\
        [5000,14,2.5,4,20],\
        [5000,14,2.5,5,0],\
        [5000,14,2.5,5,5],\
        [5000,14,2.5,5,10],\
        [5000,14,2.5,5,15],\
        [5000,14,2.5,5,20],\
        [6000,2,1,1,0],\
        [6000,2,1,1,5],\
        [6000,2,1,1,10],\
        [6000,2,1,1,15],\
        [6000,2,1,1,20],\
        [6000,2,1,2,0],\
        [6000,2,1,2,5],\
        [6000,2,1,2,10],\
        [6000,2,1,2,15],\
        [6000,2,1,2,20],\
        [6000,2,1,3,0],\
        [6000,2,1,3,5],\
        [6000,2,1,3,10],\
        [6000,2,1,3,15],\
        [6000,2,1,3,20],\
        [6000,2,1,4,0],\
        [6000,2,1,4,5],\
        [6000,2,1,4,10],\
        [6000,2,1,4,15],\
        [6000,2,1,4,20],\
        [6000,2,1,5,0],\
        [6000,2,1,5,5],\
        [6000,2,1,5,10],\
        [6000,2,1,5,15],\
        [6000,2,1,5,20],\
        [6000,2,1.375,1,0],\
        [6000,2,1.375,1,5],\
        [6000,2,1.375,1,10],\
        [6000,2,1.375,1,15],\
        [6000,2,1.375,1,20],\
        [6000,2,1.375,2,0],\
        [6000,2,1.375,2,5],\
        [6000,2,1.375,2,10],\
        [6000,2,1.375,2,15],\
        [6000,2,1.375,2,20],\
        [6000,2,1.375,3,0],\
        [6000,2,1.375,3,5],\
        [6000,2,1.375,3,10],\
        [6000,2,1.375,3,15],\
        [6000,2,1.375,3,20],\
        [6000,2,1.375,4,0],\
        [6000,2,1.375,4,5],\
        [6000,2,1.375,4,10],\
        [6000,2,1.375,4,15],\
        [6000,2,1.375,4,20],\
        [6000,2,1.375,5,0],\
        [6000,2,1.375,5,5],\
        [6000,2,1.375,5,10],\
        [6000,2,1.375,5,15],\
        [6000,2,1.375,5,20],\
        [6000,2,1.75,1,0],\
        [6000,2,1.75,1,5],\
        [6000,2,1.75,1,10],\
        [6000,2,1.75,1,15],\
        [6000,2,1.75,1,20],\
        [6000,2,1.75,2,0],\
        [6000,2,1.75,2,5],\
        [6000,2,1.75,2,10],\
        [6000,2,1.75,2,15],\
        [6000,2,1.75,2,20],\
        [6000,2,1.75,3,0],\
        [6000,2,1.75,3,5],\
        [6000,2,1.75,3,10],\
        [6000,2,1.75,3,15],\
        [6000,2,1.75,3,20],\
        [6000,2,1.75,4,0],\
        [6000,2,1.75,4,5],\
        [6000,2,1.75,4,10],\
        [6000,2,1.75,4,15],\
        [6000,2,1.75,4,20],\
        [6000,2,1.75,5,0],\
        [6000,2,1.75,5,5],\
        [6000,2,1.75,5,10],\
        [6000,2,1.75,5,15],\
        [6000,2,1.75,5,20],\
        [6000,2,2.125,1,0],\
        [6000,2,2.125,1,5],\
        [6000,2,2.125,1,10],\
        [6000,2,2.125,1,15],\
        [6000,2,2.125,1,20],\
        [6000,2,2.125,2,0],\
        [6000,2,2.125,2,5],\
        [6000,2,2.125,2,10],\
        [6000,2,2.125,2,15],\
        [6000,2,2.125,2,20],\
        [6000,2,2.125,3,0],\
        [6000,2,2.125,3,5],\
        [6000,2,2.125,3,10],\
        [6000,2,2.125,3,15],\
        [6000,2,2.125,3,20],\
        [6000,2,2.125,4,0],\
        [6000,2,2.125,4,5],\
        [6000,2,2.125,4,10],\
        [6000,2,2.125,4,15],\
        [6000,2,2.125,4,20],\
        [6000,2,2.125,5,0],\
        [6000,2,2.125,5,5],\
        [6000,2,2.125,5,10],\
        [6000,2,2.125,5,15],\
        [6000,2,2.125,5,20],\
        [6000,2,2.5,1,0],\
        [6000,2,2.5,1,5],\
        [6000,2,2.5,1,10],\
        [6000,2,2.5,1,15],\
        [6000,2,2.5,1,20],\
        [6000,2,2.5,2,0],\
        [6000,2,2.5,2,5],\
        [6000,2,2.5,2,10],\
        [6000,2,2.5,2,15],\
        [6000,2,2.5,2,20],\
        [6000,2,2.5,3,0],\
        [6000,2,2.5,3,5],\
        [6000,2,2.5,3,10],\
        [6000,2,2.5,3,15],\
        [6000,2,2.5,3,20],\
        [6000,2,2.5,4,0],\
        [6000,2,2.5,4,5],\
        [6000,2,2.5,4,10],\
        [6000,2,2.5,4,15],\
        [6000,2,2.5,4,20],\
        [6000,2,2.5,5,0],\
        [6000,2,2.5,5,5],\
        [6000,2,2.5,5,10],\
        [6000,2,2.5,5,15],\
        [6000,2,2.5,5,20],\
        [6000,5,1,1,0],\
        [6000,5,1,1,5],\
        [6000,5,1,1,10],\
        [6000,5,1,1,15],\
        [6000,5,1,1,20],\
        [6000,5,1,2,0],\
        [6000,5,1,2,5],\
        [6000,5,1,2,10],\
        [6000,5,1,2,15],\
        [6000,5,1,2,20],\
        [6000,5,1,3,0],\
        [6000,5,1,3,5],\
        [6000,5,1,3,10],\
        [6000,5,1,4,0],\
        [6000,5,1,4,5],\
        [6000,5,1,5,0],\
        [6000,5,1,5,5],\
        [6000,5,1.375,1,0],\
        [6000,5,1.375,1,5],\
        [6000,5,1.375,1,10],\
        [6000,5,1.375,1,15],\
        [6000,5,1.375,1,20],\
        [6000,5,1.375,2,0],\
        [6000,5,1.375,2,5],\
        [6000,5,1.375,2,10],\
        [6000,5,1.375,2,15],\
        [6000,5,1.375,2,20],\
        [6000,5,1.375,3,0],\
        [6000,5,1.375,3,5],\
        [6000,5,1.375,3,10],\
        [6000,5,1.375,3,15],\
        [6000,5,1.375,3,20],\
        [6000,5,1.375,4,0],\
        [6000,5,1.375,4,5],\
        [6000,5,1.375,4,10],\
        [6000,5,1.375,4,15],\
        [6000,5,1.375,4,20],\
        [6000,5,1.375,5,0],\
        [6000,5,1.375,5,5],\
        [6000,5,1.375,5,10],\
        [6000,5,1.375,5,15],\
        [6000,5,1.375,5,20],\
        [6000,5,1.75,1,0],\
        [6000,5,1.75,1,5],\
        [6000,5,1.75,1,10],\
        [6000,5,1.75,1,15],\
        [6000,5,1.75,1,20],\
        [6000,5,1.75,2,0],\
        [6000,5,1.75,2,5],\
        [6000,5,1.75,2,10],\
        [6000,5,1.75,2,15],\
        [6000,5,1.75,2,20],\
        [6000,5,1.75,3,0],\
        [6000,5,1.75,3,5],\
        [6000,5,1.75,3,10],\
        [6000,5,1.75,3,15],\
        [6000,5,1.75,3,20],\
        [6000,5,1.75,4,0],\
        [6000,5,1.75,4,5],\
        [6000,5,1.75,4,10],\
        [6000,5,1.75,4,15],\
        [6000,5,1.75,4,20],\
        [6000,5,1.75,5,0],\
        [6000,5,1.75,5,5],\
        [6000,5,1.75,5,10],\
        [6000,5,1.75,5,15],\
        [6000,5,1.75,5,20],\
        [6000,5,2.125,1,0],\
        [6000,5,2.125,1,5],\
        [6000,5,2.125,1,10],\
        [6000,5,2.125,1,15],\
        [6000,5,2.125,1,20],\
        [6000,5,2.125,2,0],\
        [6000,5,2.125,2,5],\
        [6000,5,2.125,2,10],\
        [6000,5,2.125,2,15],\
        [6000,5,2.125,2,20],\
        [6000,5,2.125,3,0],\
        [6000,5,2.125,3,5],\
        [6000,5,2.125,3,10],\
        [6000,5,2.125,3,15],\
        [6000,5,2.125,3,20],\
        [6000,5,2.125,4,0],\
        [6000,5,2.125,4,5],\
        [6000,5,2.125,4,10],\
        [6000,5,2.125,4,15],\
        [6000,5,2.125,4,20],\
        [6000,5,2.125,5,0],\
        [6000,5,2.125,5,5],\
        [6000,5,2.125,5,10],\
        [6000,5,2.125,5,15],\
        [6000,5,2.125,5,20],\
        [6000,5,2.5,1,0],\
        [6000,5,2.5,1,5],\
        [6000,5,2.5,1,10],\
        [6000,5,2.5,1,15],\
        [6000,5,2.5,1,20],\
        [6000,5,2.5,2,0],\
        [6000,5,2.5,2,5],\
        [6000,5,2.5,2,10],\
        [6000,5,2.5,2,15],\
        [6000,5,2.5,2,20],\
        [6000,5,2.5,3,0],\
        [6000,5,2.5,3,5],\
        [6000,5,2.5,3,10],\
        [6000,5,2.5,3,15],\
        [6000,5,2.5,3,20],\
        [6000,5,2.5,4,0],\
        [6000,5,2.5,4,5],\
        [6000,5,2.5,4,10],\
        [6000,5,2.5,4,15],\
        [6000,5,2.5,4,20],\
        [6000,5,2.5,5,0],\
        [6000,5,2.5,5,5],\
        [6000,5,2.5,5,10],\
        [6000,5,2.5,5,15],\
        [6000,5,2.5,5,20],\
        [6000,8,1,1,0],\
        [6000,8,1,1,5],\
        [6000,8,1,1,10],\
        [6000,8,1,2,0],\
        [6000,8,1,2,5],\
        [6000,8,1,3,0],\
        [6000,8,1,4,0],\
        [6000,8,1,5,0],\
        [6000,8,1.375,1,0],\
        [6000,8,1.375,1,5],\
        [6000,8,1.375,1,10],\
        [6000,8,1.375,1,15],\
        [6000,8,1.375,1,20],\
        [6000,8,1.375,2,0],\
        [6000,8,1.375,2,5],\
        [6000,8,1.375,2,10],\
        [6000,8,1.375,2,15],\
        [6000,8,1.375,2,20],\
        [6000,8,1.375,3,0],\
        [6000,8,1.375,3,5],\
        [6000,8,1.375,3,10],\
        [6000,8,1.375,3,15],\
        [6000,8,1.375,3,20],\
        [6000,8,1.375,4,0],\
        [6000,8,1.375,4,5],\
        [6000,8,1.375,4,10],\
        [6000,8,1.375,4,15],\
        [6000,8,1.375,4,20],\
        [6000,8,1.375,5,0],\
        [6000,8,1.375,5,5],\
        [6000,8,1.375,5,10],\
        [6000,8,1.375,5,15],\
        [6000,8,1.375,5,20],\
        [6000,8,1.75,1,0],\
        [6000,8,1.75,1,5],\
        [6000,8,1.75,1,10],\
        [6000,8,1.75,1,15],\
        [6000,8,1.75,1,20],\
        [6000,8,1.75,2,0],\
        [6000,8,1.75,2,5],\
        [6000,8,1.75,2,10],\
        [6000,8,1.75,2,15],\
        [6000,8,1.75,2,20],\
        [6000,8,1.75,3,0],\
        [6000,8,1.75,3,5],\
        [6000,8,1.75,3,10],\
        [6000,8,1.75,3,15],\
        [6000,8,1.75,3,20],\
        [6000,8,1.75,4,0],\
        [6000,8,1.75,4,5],\
        [6000,8,1.75,4,10],\
        [6000,8,1.75,4,15],\
        [6000,8,1.75,4,20],\
        [6000,8,1.75,5,0],\
        [6000,8,1.75,5,5],\
        [6000,8,1.75,5,10],\
        [6000,8,1.75,5,15],\
        [6000,8,1.75,5,20],\
        [6000,8,2.125,1,0],\
        [6000,8,2.125,1,5],\
        [6000,8,2.125,1,10],\
        [6000,8,2.125,1,15],\
        [6000,8,2.125,1,20],\
        [6000,8,2.125,2,0],\
        [6000,8,2.125,2,5],\
        [6000,8,2.125,2,10],\
        [6000,8,2.125,2,15],\
        [6000,8,2.125,2,20],\
        [6000,8,2.125,3,0],\
        [6000,8,2.125,3,5],\
        [6000,8,2.125,3,10],\
        [6000,8,2.125,3,15],\
        [6000,8,2.125,3,20],\
        [6000,8,2.125,4,0],\
        [6000,8,2.125,4,5],\
        [6000,8,2.125,4,10],\
        [6000,8,2.125,4,15],\
        [6000,8,2.125,4,20],\
        [6000,8,2.125,5,0],\
        [6000,8,2.125,5,5],\
        [6000,8,2.125,5,10],\
        [6000,8,2.125,5,15],\
        [6000,8,2.125,5,20],\
        [6000,8,2.5,1,0],\
        [6000,8,2.5,1,5],\
        [6000,8,2.5,1,10],\
        [6000,8,2.5,1,15],\
        [6000,8,2.5,1,20],\
        [6000,8,2.5,2,0],\
        [6000,8,2.5,2,5],\
        [6000,8,2.5,2,10],\
        [6000,8,2.5,2,15],\
        [6000,8,2.5,2,20],\
        [6000,8,2.5,3,0],\
        [6000,8,2.5,3,5],\
        [6000,8,2.5,3,10],\
        [6000,8,2.5,3,15],\
        [6000,8,2.5,3,20],\
        [6000,8,2.5,4,0],\
        [6000,8,2.5,4,5],\
        [6000,8,2.5,4,10],\
        [6000,8,2.5,4,15],\
        [6000,8,2.5,4,20],\
        [6000,8,2.5,5,0],\
        [6000,8,2.5,5,5],\
        [6000,8,2.5,5,10],\
        [6000,8,2.5,5,15],\
        [6000,8,2.5,5,20],\
        [6000,11,1,1,0],\
        [6000,11,1,1,5],\
        [6000,11,1,2,0],\
        [6000,11,1,3,0],\
        [6000,11,1,4,0],\
        [6000,11,1,5,0],\
        [6000,11,1.375,1,0],\
        [6000,11,1.375,1,5],\
        [6000,11,1.375,1,10],\
        [6000,11,1.375,1,15],\
        [6000,11,1.375,1,20],\
        [6000,11,1.375,2,0],\
        [6000,11,1.375,2,5],\
        [6000,11,1.375,2,10],\
        [6000,11,1.375,2,15],\
        [6000,11,1.375,2,20],\
        [6000,11,1.375,3,0],\
        [6000,11,1.375,3,5],\
        [6000,11,1.375,3,10],\
        [6000,11,1.375,3,15],\
        [6000,11,1.375,3,20],\
        [6000,11,1.375,4,0],\
        [6000,11,1.375,4,5],\
        [6000,11,1.375,4,10],\
        [6000,11,1.375,4,15],\
        [6000,11,1.375,4,20],\
        [6000,11,1.375,5,0],\
        [6000,11,1.375,5,5],\
        [6000,11,1.375,5,10],\
        [6000,11,1.375,5,15],\
        [6000,11,1.375,5,20],\
        [6000,11,1.75,1,0],\
        [6000,11,1.75,1,5],\
        [6000,11,1.75,1,10],\
        [6000,11,1.75,1,15],\
        [6000,11,1.75,1,20],\
        [6000,11,1.75,2,0],\
        [6000,11,1.75,2,5],\
        [6000,11,1.75,2,10],\
        [6000,11,1.75,2,15],\
        [6000,11,1.75,2,20],\
        [6000,11,1.75,3,0],\
        [6000,11,1.75,3,5],\
        [6000,11,1.75,3,10],\
        [6000,11,1.75,3,15],\
        [6000,11,1.75,3,20],\
        [6000,11,1.75,4,0],\
        [6000,11,1.75,4,5],\
        [6000,11,1.75,4,10],\
        [6000,11,1.75,4,15],\
        [6000,11,1.75,4,20],\
        [6000,11,1.75,5,0],\
        [6000,11,1.75,5,5],\
        [6000,11,1.75,5,10],\
        [6000,11,1.75,5,15],\
        [6000,11,1.75,5,20],\
        [6000,11,2.125,1,0],\
        [6000,11,2.125,1,5],\
        [6000,11,2.125,1,10],\
        [6000,11,2.125,1,15],\
        [6000,11,2.125,1,20],\
        [6000,11,2.125,2,0],\
        [6000,11,2.125,2,5],\
        [6000,11,2.125,2,10],\
        [6000,11,2.125,2,15],\
        [6000,11,2.125,2,20],\
        [6000,11,2.125,3,0],\
        [6000,11,2.125,3,5],\
        [6000,11,2.125,3,10],\
        [6000,11,2.125,3,15],\
        [6000,11,2.125,3,20],\
        [6000,11,2.125,4,0],\
        [6000,11,2.125,4,5],\
        [6000,11,2.125,4,10],\
        [6000,11,2.125,4,15],\
        [6000,11,2.125,4,20],\
        [6000,11,2.125,5,0],\
        [6000,11,2.125,5,5],\
        [6000,11,2.125,5,10],\
        [6000,11,2.125,5,15],\
        [6000,11,2.125,5,20],\
        [6000,11,2.5,1,0],\
        [6000,11,2.5,1,5],\
        [6000,11,2.5,1,10],\
        [6000,11,2.5,1,15],\
        [6000,11,2.5,1,20],\
        [6000,11,2.5,2,0],\
        [6000,11,2.5,2,5],\
        [6000,11,2.5,2,10],\
        [6000,11,2.5,2,15],\
        [6000,11,2.5,2,20],\
        [6000,11,2.5,3,0],\
        [6000,11,2.5,3,5],\
        [6000,11,2.5,3,10],\
        [6000,11,2.5,3,15],\
        [6000,11,2.5,3,20],\
        [6000,11,2.5,4,0],\
        [6000,11,2.5,4,5],\
        [6000,11,2.5,4,10],\
        [6000,11,2.5,4,15],\
        [6000,11,2.5,4,20],\
        [6000,11,2.5,5,0],\
        [6000,11,2.5,5,5],\
        [6000,11,2.5,5,10],\
        [6000,11,2.5,5,15],\
        [6000,11,2.5,5,20],\
        [6000,14,1,1,0],\
        [6000,14,1,2,0],\
        [6000,14,1,3,0],\
        [6000,14,1,4,0],\
        [6000,14,1,5,0],\
        [6000,14,1.375,1,0],\
        [6000,14,1.375,1,5],\
        [6000,14,1.375,1,10],\
        [6000,14,1.375,1,15],\
        [6000,14,1.375,1,20],\
        [6000,14,1.375,2,0],\
        [6000,14,1.375,2,5],\
        [6000,14,1.375,2,10],\
        [6000,14,1.375,2,15],\
        [6000,14,1.375,2,20],\
        [6000,14,1.375,3,0],\
        [6000,14,1.375,3,5],\
        [6000,14,1.375,3,10],\
        [6000,14,1.375,3,15],\
        [6000,14,1.375,3,20],\
        [6000,14,1.375,4,0],\
        [6000,14,1.375,4,5],\
        [6000,14,1.375,4,10],\
        [6000,14,1.375,4,15],\
        [6000,14,1.375,4,20],\
        [6000,14,1.375,5,0],\
        [6000,14,1.375,5,5],\
        [6000,14,1.375,5,10],\
        [6000,14,1.375,5,15],\
        [6000,14,1.375,5,20],\
        [6000,14,1.75,1,0],\
        [6000,14,1.75,1,5],\
        [6000,14,1.75,1,10],\
        [6000,14,1.75,1,15],\
        [6000,14,1.75,1,20],\
        [6000,14,1.75,2,0],\
        [6000,14,1.75,2,5],\
        [6000,14,1.75,2,10],\
        [6000,14,1.75,2,15],\
        [6000,14,1.75,2,20],\
        [6000,14,1.75,3,0],\
        [6000,14,1.75,3,5],\
        [6000,14,1.75,3,10],\
        [6000,14,1.75,3,15],\
        [6000,14,1.75,3,20],\
        [6000,14,1.75,4,0],\
        [6000,14,1.75,4,5],\
        [6000,14,1.75,4,10],\
        [6000,14,1.75,4,15],\
        [6000,14,1.75,4,20],\
        [6000,14,1.75,5,0],\
        [6000,14,1.75,5,5],\
        [6000,14,1.75,5,10],\
        [6000,14,1.75,5,15],\
        [6000,14,1.75,5,20],\
        [6000,14,2.125,1,0],\
        [6000,14,2.125,1,5],\
        [6000,14,2.125,1,10],\
        [6000,14,2.125,1,15],\
        [6000,14,2.125,1,20],\
        [6000,14,2.125,2,0],\
        [6000,14,2.125,2,5],\
        [6000,14,2.125,2,10],\
        [6000,14,2.125,2,15],\
        [6000,14,2.125,2,20],\
        [6000,14,2.125,3,0],\
        [6000,14,2.125,3,5],\
        [6000,14,2.125,3,10],\
        [6000,14,2.125,3,15],\
        [6000,14,2.125,3,20],\
        [6000,14,2.125,4,0],\
        [6000,14,2.125,4,5],\
        [6000,14,2.125,4,10],\
        [6000,14,2.125,4,15],\
        [6000,14,2.125,4,20],\
        [6000,14,2.125,5,0],\
        [6000,14,2.125,5,5],\
        [6000,14,2.125,5,10],\
        [6000,14,2.125,5,15],\
        [6000,14,2.125,5,20],\
        [6000,14,2.5,1,0],\
        [6000,14,2.5,1,5],\
        [6000,14,2.5,1,10],\
        [6000,14,2.5,1,15],\
        [6000,14,2.5,1,20],\
        [6000,14,2.5,2,0],\
        [6000,14,2.5,2,5],\
        [6000,14,2.5,2,10],\
        [6000,14,2.5,2,15],\
        [6000,14,2.5,2,20],\
        [6000,14,2.5,3,0],\
        [6000,14,2.5,3,5],\
        [6000,14,2.5,3,10],\
        [6000,14,2.5,3,15],\
        [6000,14,2.5,3,20],\
        [6000,14,2.5,4,0],\
        [6000,14,2.5,4,5],\
        [6000,14,2.5,4,10],\
        [6000,14,2.5,4,15],\
        [6000,14,2.5,4,20],\
        [6000,14,2.5,5,0],\
        [6000,14,2.5,5,5],\
        [6000,14,2.5,5,10],\
        [6000,14,2.5,5,15],\
        [6000,14,2.5,5,20],\
        [7000,2,1,1,0],\
        [7000,2,1,1,5],\
        [7000,2,1,1,10],\
        [7000,2,1,1,15],\
        [7000,2,1,1,20],\
        [7000,2,1,2,0],\
        [7000,2,1,2,5],\
        [7000,2,1,2,10],\
        [7000,2,1,2,15],\
        [7000,2,1,2,20],\
        [7000,2,1,3,0],\
        [7000,2,1,3,5],\
        [7000,2,1,3,10],\
        [7000,2,1,3,15],\
        [7000,2,1,3,20],\
        [7000,2,1,4,0],\
        [7000,2,1,4,5],\
        [7000,2,1,4,10],\
        [7000,2,1,4,15],\
        [7000,2,1,4,20],\
        [7000,2,1,5,0],\
        [7000,2,1,5,5],\
        [7000,2,1,5,10],\
        [7000,2,1,5,15],\
        [7000,2,1,5,20],\
        [7000,2,1.375,1,0],\
        [7000,2,1.375,1,5],\
        [7000,2,1.375,1,10],\
        [7000,2,1.375,1,15],\
        [7000,2,1.375,1,20],\
        [7000,2,1.375,2,0],\
        [7000,2,1.375,2,5],\
        [7000,2,1.375,2,10],\
        [7000,2,1.375,2,15],\
        [7000,2,1.375,2,20],\
        [7000,2,1.375,3,0],\
        [7000,2,1.375,3,5],\
        [7000,2,1.375,3,10],\
        [7000,2,1.375,3,15],\
        [7000,2,1.375,3,20],\
        [7000,2,1.375,4,0],\
        [7000,2,1.375,4,5],\
        [7000,2,1.375,4,10],\
        [7000,2,1.375,4,15],\
        [7000,2,1.375,4,20],\
        [7000,2,1.375,5,0],\
        [7000,2,1.375,5,5],\
        [7000,2,1.375,5,10],\
        [7000,2,1.375,5,15],\
        [7000,2,1.375,5,20],\
        [7000,2,1.75,1,0],\
        [7000,2,1.75,1,5],\
        [7000,2,1.75,1,10],\
        [7000,2,1.75,1,15],\
        [7000,2,1.75,1,20],\
        [7000,2,1.75,2,0],\
        [7000,2,1.75,2,5],\
        [7000,2,1.75,2,10],\
        [7000,2,1.75,2,15],\
        [7000,2,1.75,2,20],\
        [7000,2,1.75,3,0],\
        [7000,2,1.75,3,5],\
        [7000,2,1.75,3,10],\
        [7000,2,1.75,3,15],\
        [7000,2,1.75,3,20],\
        [7000,2,1.75,4,0],\
        [7000,2,1.75,4,5],\
        [7000,2,1.75,4,10],\
        [7000,2,1.75,4,15],\
        [7000,2,1.75,4,20],\
        [7000,2,1.75,5,0],\
        [7000,2,1.75,5,5],\
        [7000,2,1.75,5,10],\
        [7000,2,1.75,5,15],\
        [7000,2,1.75,5,20],\
        [7000,2,2.125,1,0],\
        [7000,2,2.125,1,5],\
        [7000,2,2.125,1,10],\
        [7000,2,2.125,1,15],\
        [7000,2,2.125,1,20],\
        [7000,2,2.125,2,0],\
        [7000,2,2.125,2,5],\
        [7000,2,2.125,2,10],\
        [7000,2,2.125,2,15],\
        [7000,2,2.125,2,20],\
        [7000,2,2.125,3,0],\
        [7000,2,2.125,3,5],\
        [7000,2,2.125,3,10],\
        [7000,2,2.125,3,15],\
        [7000,2,2.125,3,20],\
        [7000,2,2.125,4,0],\
        [7000,2,2.125,4,5],\
        [7000,2,2.125,4,10],\
        [7000,2,2.125,4,15],\
        [7000,2,2.125,4,20],\
        [7000,2,2.125,5,0],\
        [7000,2,2.125,5,5],\
        [7000,2,2.125,5,10],\
        [7000,2,2.125,5,15],\
        [7000,2,2.125,5,20],\
        [7000,2,2.5,1,0],\
        [7000,2,2.5,1,5],\
        [7000,2,2.5,1,10],\
        [7000,2,2.5,1,15],\
        [7000,2,2.5,1,20],\
        [7000,2,2.5,2,0],\
        [7000,2,2.5,2,5],\
        [7000,2,2.5,2,10],\
        [7000,2,2.5,2,15],\
        [7000,2,2.5,2,20],\
        [7000,2,2.5,3,0],\
        [7000,2,2.5,3,5],\
        [7000,2,2.5,3,10],\
        [7000,2,2.5,3,15],\
        [7000,2,2.5,3,20],\
        [7000,2,2.5,4,0],\
        [7000,2,2.5,4,5],\
        [7000,2,2.5,4,10],\
        [7000,2,2.5,4,15],\
        [7000,2,2.5,4,20],\
        [7000,2,2.5,5,0],\
        [7000,2,2.5,5,5],\
        [7000,2,2.5,5,10],\
        [7000,2,2.5,5,15],\
        [7000,2,2.5,5,20],\
        [7000,5,1,1,0],\
        [7000,5,1,1,5],\
        [7000,5,1,1,10],\
        [7000,5,1,2,0],\
        [7000,5,1,2,5],\
        [7000,5,1,3,0],\
        [7000,5,1,4,0],\
        [7000,5,1,5,0],\
        [7000,5,1.375,1,0],\
        [7000,5,1.375,1,5],\
        [7000,5,1.375,1,10],\
        [7000,5,1.375,1,15],\
        [7000,5,1.375,1,20],\
        [7000,5,1.375,2,0],\
        [7000,5,1.375,2,5],\
        [7000,5,1.375,2,10],\
        [7000,5,1.375,2,15],\
        [7000,5,1.375,3,0],\
        [7000,5,1.375,3,5],\
        [7000,5,1.375,3,10],\
        [7000,5,1.375,4,0],\
        [7000,5,1.375,4,5],\
        [7000,5,1.375,5,0],\
        [7000,5,1.375,5,5],\
        [7000,5,1.75,1,0],\
        [7000,5,1.75,1,5],\
        [7000,5,1.75,1,10],\
        [7000,5,1.75,1,15],\
        [7000,5,1.75,1,20],\
        [7000,5,1.75,2,0],\
        [7000,5,1.75,2,5],\
        [7000,5,1.75,2,10],\
        [7000,5,1.75,2,15],\
        [7000,5,1.75,2,20],\
        [7000,5,1.75,3,0],\
        [7000,5,1.75,3,5],\
        [7000,5,1.75,3,10],\
        [7000,5,1.75,3,15],\
        [7000,5,1.75,3,20],\
        [7000,5,1.75,4,0],\
        [7000,5,1.75,4,5],\
        [7000,5,1.75,4,10],\
        [7000,5,1.75,4,15],\
        [7000,5,1.75,4,20],\
        [7000,5,1.75,5,0],\
        [7000,5,1.75,5,5],\
        [7000,5,1.75,5,10],\
        [7000,5,2.125,1,0],\
        [7000,5,2.125,1,5],\
        [7000,5,2.125,1,10],\
        [7000,5,2.125,1,15],\
        [7000,5,2.125,1,20],\
        [7000,5,2.125,2,0],\
        [7000,5,2.125,2,5],\
        [7000,5,2.125,2,10],\
        [7000,5,2.125,2,15],\
        [7000,5,2.125,2,20],\
        [7000,5,2.125,3,0],\
        [7000,5,2.125,3,5],\
        [7000,5,2.125,3,10],\
        [7000,5,2.125,3,15],\
        [7000,5,2.125,3,20],\
        [7000,5,2.125,4,0],\
        [7000,5,2.125,4,5],\
        [7000,5,2.125,4,10],\
        [7000,5,2.125,4,15],\
        [7000,5,2.125,4,20],\
        [7000,5,2.125,5,0],\
        [7000,5,2.125,5,5],\
        [7000,5,2.125,5,10],\
        [7000,5,2.125,5,15],\
        [7000,5,2.125,5,20],\
        [7000,5,2.5,1,0],\
        [7000,5,2.5,1,5],\
        [7000,5,2.5,1,10],\
        [7000,5,2.5,1,15],\
        [7000,5,2.5,1,20],\
        [7000,5,2.5,2,0],\
        [7000,5,2.5,2,5],\
        [7000,5,2.5,2,10],\
        [7000,5,2.5,2,15],\
        [7000,5,2.5,2,20],\
        [7000,5,2.5,3,0],\
        [7000,5,2.5,3,5],\
        [7000,5,2.5,3,10],\
        [7000,5,2.5,3,15],\
        [7000,5,2.5,3,20],\
        [7000,5,2.5,4,0],\
        [7000,5,2.5,4,5],\
        [7000,5,2.5,4,10],\
        [7000,5,2.5,4,15],\
        [7000,5,2.5,4,20],\
        [7000,5,2.5,5,0],\
        [7000,5,2.5,5,5],\
        [7000,5,2.5,5,10],\
        [7000,5,2.5,5,15],\
        [7000,5,2.5,5,20],\
        [7000,8,1,1,0],\
        [7000,8,1,1,5],\
        [7000,8,1,2,0],\
        [7000,8,1,3,0],\
        [7000,8,1,4,0],\
        [7000,8,1,5,0],\
        [7000,8,1.375,1,0],\
        [7000,8,1.375,1,5],\
        [7000,8,1.375,1,10],\
        [7000,8,1.375,2,0],\
        [7000,8,1.375,2,5],\
        [7000,8,1.375,3,0],\
        [7000,8,1.375,4,0],\
        [7000,8,1.375,5,0],\
        [7000,8,1.75,1,0],\
        [7000,8,1.75,1,5],\
        [7000,8,1.75,1,10],\
        [7000,8,1.75,1,15],\
        [7000,8,1.75,1,20],\
        [7000,8,1.75,2,0],\
        [7000,8,1.75,2,5],\
        [7000,8,1.75,2,10],\
        [7000,8,1.75,3,0],\
        [7000,8,1.75,3,5],\
        [7000,8,1.75,4,0],\
        [7000,8,1.75,4,5],\
        [7000,8,1.75,5,0],\
        [7000,8,2.125,1,0],\
        [7000,8,2.125,1,5],\
        [7000,8,2.125,1,10],\
        [7000,8,2.125,1,15],\
        [7000,8,2.125,1,20],\
        [7000,8,2.125,2,0],\
        [7000,8,2.125,2,5],\
        [7000,8,2.125,2,10],\
        [7000,8,2.125,2,15],\
        [7000,8,2.125,2,20],\
        [7000,8,2.125,3,0],\
        [7000,8,2.125,3,5],\
        [7000,8,2.125,3,10],\
        [7000,8,2.125,3,15],\
        [7000,8,2.125,3,20],\
        [7000,8,2.125,4,0],\
        [7000,8,2.125,4,5],\
        [7000,8,2.125,4,10],\
        [7000,8,2.125,4,15],\
        [7000,8,2.125,4,20],\
        [7000,8,2.125,5,0],\
        [7000,8,2.125,5,5],\
        [7000,8,2.125,5,10],\
        [7000,8,2.125,5,15],\
        [7000,8,2.125,5,20],\
        [7000,8,2.5,1,0],\
        [7000,8,2.5,1,5],\
        [7000,8,2.5,1,10],\
        [7000,8,2.5,1,15],\
        [7000,8,2.5,1,20],\
        [7000,8,2.5,2,0],\
        [7000,8,2.5,2,5],\
        [7000,8,2.5,2,10],\
        [7000,8,2.5,2,15],\
        [7000,8,2.5,2,20],\
        [7000,8,2.5,3,0],\
        [7000,8,2.5,3,5],\
        [7000,8,2.5,3,10],\
        [7000,8,2.5,3,15],\
        [7000,8,2.5,3,20],\
        [7000,8,2.5,4,0],\
        [7000,8,2.5,4,5],\
        [7000,8,2.5,4,10],\
        [7000,8,2.5,4,15],\
        [7000,8,2.5,4,20],\
        [7000,8,2.5,5,0],\
        [7000,8,2.5,5,5],\
        [7000,8,2.5,5,10],\
        [7000,8,2.5,5,15],\
        [7000,8,2.5,5,20],\
        [7000,11,1,1,0],\
        [7000,11,1,2,0],\
        [7000,11,1,3,0],\
        [7000,11,1,4,0],\
        [7000,11,1,5,0],\
        [7000,11,1.375,1,0],\
        [7000,11,1.375,1,5],\
        [7000,11,1.375,2,0],\
        [7000,11,1.375,3,0],\
        [7000,11,1.375,4,0],\
        [7000,11,1.375,5,0],\
        [7000,11,1.75,1,0],\
        [7000,11,1.75,1,5],\
        [7000,11,1.75,1,10],\
        [7000,11,1.75,2,0],\
        [7000,11,1.75,2,5],\
        [7000,11,1.75,3,0],\
        [7000,11,1.75,4,0],\
        [7000,11,1.75,5,0],\
        [7000,11,2.125,1,0],\
        [7000,11,2.125,1,5],\
        [7000,11,2.125,1,10],\
        [7000,11,2.125,1,15],\
        [7000,11,2.125,1,20],\
        [7000,11,2.125,2,0],\
        [7000,11,2.125,2,5],\
        [7000,11,2.125,2,10],\
        [7000,11,2.125,2,15],\
        [7000,11,2.125,2,20],\
        [7000,11,2.125,3,0],\
        [7000,11,2.125,3,5],\
        [7000,11,2.125,3,10],\
        [7000,11,2.125,3,15],\
        [7000,11,2.125,3,20],\
        [7000,11,2.125,4,0],\
        [7000,11,2.125,4,5],\
        [7000,11,2.125,4,10],\
        [7000,11,2.125,4,15],\
        [7000,11,2.125,4,20],\
        [7000,11,2.125,5,0],\
        [7000,11,2.125,5,5],\
        [7000,11,2.125,5,10],\
        [7000,11,2.125,5,15],\
        [7000,11,2.125,5,20],\
        [7000,11,2.5,1,0],\
        [7000,11,2.5,1,5],\
        [7000,11,2.5,1,10],\
        [7000,11,2.5,1,15],\
        [7000,11,2.5,1,20],\
        [7000,11,2.5,2,0],\
        [7000,11,2.5,2,5],\
        [7000,11,2.5,2,10],\
        [7000,11,2.5,2,15],\
        [7000,11,2.5,2,20],\
        [7000,11,2.5,3,0],\
        [7000,11,2.5,3,5],\
        [7000,11,2.5,3,10],\
        [7000,11,2.5,3,15],\
        [7000,11,2.5,3,20],\
        [7000,11,2.5,4,0],\
        [7000,11,2.5,4,5],\
        [7000,11,2.5,4,10],\
        [7000,11,2.5,4,15],\
        [7000,11,2.5,4,20],\
        [7000,11,2.5,5,0],\
        [7000,11,2.5,5,5],\
        [7000,11,2.5,5,10],\
        [7000,11,2.5,5,15],\
        [7000,11,2.5,5,20],\
        [7000,14,1,1,0],\
        [7000,14,1,2,0],\
        [7000,14,1,3,0],\
        [7000,14,1,4,0],\
        [7000,14,1,5,0],\
        [7000,14,1.375,1,0],\
        [7000,14,1.375,2,0],\
        [7000,14,1.375,3,0],\
        [7000,14,1.375,4,0],\
        [7000,14,1.375,5,0],\
        [7000,14,1.75,1,0],\
        [7000,14,1.75,1,5],\
        [7000,14,1.75,2,0],\
        [7000,14,1.75,3,0],\
        [7000,14,1.75,4,0],\
        [7000,14,1.75,5,0],\
        [7000,14,2.125,1,0],\
        [7000,14,2.125,1,5],\
        [7000,14,2.125,1,10],\
        [7000,14,2.125,1,15],\
        [7000,14,2.125,1,20],\
        [7000,14,2.125,2,0],\
        [7000,14,2.125,2,5],\
        [7000,14,2.125,2,10],\
        [7000,14,2.125,2,15],\
        [7000,14,2.125,2,20],\
        [7000,14,2.125,3,0],\
        [7000,14,2.125,3,5],\
        [7000,14,2.125,3,10],\
        [7000,14,2.125,3,15],\
        [7000,14,2.125,3,20],\
        [7000,14,2.125,4,0],\
        [7000,14,2.125,4,5],\
        [7000,14,2.125,4,10],\
        [7000,14,2.125,4,15],\
        [7000,14,2.125,4,20],\
        [7000,14,2.125,5,0],\
        [7000,14,2.125,5,5],\
        [7000,14,2.125,5,10],\
        [7000,14,2.125,5,15],\
        [7000,14,2.125,5,20],\
        [7000,14,2.5,1,0],\
        [7000,14,2.5,1,5],\
        [7000,14,2.5,1,10],\
        [7000,14,2.5,1,15],\
        [7000,14,2.5,1,20],\
        [7000,14,2.5,2,0],\
        [7000,14,2.5,2,5],\
        [7000,14,2.5,2,10],\
        [7000,14,2.5,2,15],\
        [7000,14,2.5,2,20],\
        [7000,14,2.5,3,0],\
        [7000,14,2.5,3,5],\
        [7000,14,2.5,3,10],\
        [7000,14,2.5,3,15],\
        [7000,14,2.5,3,20],\
        [7000,14,2.5,4,0],\
        [7000,14,2.5,4,5],\
        [7000,14,2.5,4,10],\
        [7000,14,2.5,4,15],\
        [7000,14,2.5,4,20],\
        [7000,14,2.5,5,0],\
        [7000,14,2.5,5,5],\
        [7000,14,2.5,5,10],\
        [7000,14,2.5,5,15],\
        [7000,14,2.5,5,20]]

    
    
    Y = [8225.94247,\
        8228.30594,\
        8231.64789,\
        8235.65152,\
        8239.61682,\
        8227.26001,\
        8235.28511,\
        8248.51754,\
        8264.45868999999,\
        8278.49559,\
        8226.9702,\
        8241.57039,\
        8269.16051,\
        8301.49852,\
        8326.55344,\
        8226.09047,\
        8247.21469,\
        8292.3285,\
        8342.80734,\
        8377.14905,\
        8225.24482,\
        8252.68896,\
        8317.61033,\
        8386.10648,\
        8427.02868,\
        8251.02635,\
        8251.63645,\
        8252.88919,\
        8254.43695,\
        8256.0483,\
        8251.51299,\
        8254.78327,\
        8260.39098,\
        8267.74363,\
        8274.95561,\
        8251.57005,\
        8257.81552,\
        8270.48185,\
        8287.54787,\
        8303.0152,\
        8251.26359,\
        8260.55812,\
        8282.61254,\
        8312.49205,\
        8337.23746,\
        8251.02635,\
        8263.14936,\
        8296.71493,\
        8341.6523,\
        8375.41946,\
        8277.55612,\
        8277.55612,\
        8277.55612,\
        8277.98723,\
        8278.67584,\
        8277.55612,\
        8278.36416,\
        8280.86738,\
        8284.2362,\
        8287.72423,\
        8277.55612,\
        8279.94493,\
        8285.85236,\
        8294.25525,\
        8302.57918,\
        8277.55612,\
        8281.39783,\
        8292.02029,\
        8307.70042,\
        8322.3792,\
        8277.55612,\
        8282.74549,\
        8299.35288,\
        8324.39651,\
        8346.33854,\
        8304.08589,\
        8304.08589,\
        8304.08589,\
        8304.08589,\
        8304.08589,\
        8304.08589,\
        8304.08589,\
        8304.61102,\
        8306.22662,\
        8307.94444,\
        8304.08589,\
        8304.29214,\
        8307.19799,\
        8311.39248,\
        8315.75232,\
        8304.08589,\
        8305.11864,\
        8310.44496,\
        8318.55227,\
        8326.70312,\
        8304.08589,\
        8305.882,\
        8314.32853,\
        8327.71544,\
        8340.61779,\
        8330.61565,\
        8330.61565,\
        8330.61565,\
        8330.61565,\
        8330.61565,\
        8330.61565,\
        8330.61565,\
        8330.61565,\
        8330.49334,\
        8331.3771,\
        8330.61565,\
        8330.61565,\
        8331.09081,\
        8333.2721,\
        8335.59821,\
        8330.61565,\
        8330.61565,\
        8332.88684,\
        8337.18856,\
        8341.69171,\
        8330.61565,\
        8330.5342,\
        8335.03693,\
        8342.27222,\
        8349.65214,\
        8224.52276,\
        8259.94562,\
        8351.72964,\
        8440.73535,\
        8485.9753,\
        8224.49659,\
        8316.35029,\
        8539.73727,\
        8674.76952,\
        8702.12634,\
        8224.49659,\
        8406.75461,\
        8703.13223,\
        8810.71227,\
        8810.66652,\
        8224.49659,\
        8511.34395,\
        8810.1827,\
        8866.91308,\
        8857.80777,\
        8224.49659,\
        8612.44189,\
        8862.93133,\
        8882.51695,\
        8875.86238,\
        8251.02635,\
        8266.46415,\
        8317.19945,\
        8382.82115,\
        8426.21866,\
        8251.02635,\
        8293.3467,\
        8463.89844,\
        8621.28148,\
        8672.63049,\
        8251.02635,\
        8348.48413,\
        8648.05205,\
        8833.50629,\
        8853.46328,\
        8251.02635,\
        8430.93975,\
        8819.38574,\
        8978.30501,\
        8967.89172,\
        8251.02635,\
        8530.14328,\
        8951.54106,\
        9058.54723,\
        9034.63118,\
        8277.55612,\
        8284.39564,\
        8310.29864,\
        8349.63176,\
        8381.02366,\
        8277.55612,\
        8296.74042,\
        8401.18011,\
        8532.65942,\
        8592.23784,\
        8277.55612,\
        8324.54081,\
        8549.12394,\
        8752.63955,\
        8796.43681,\
        8277.55612,\
        8373.88954,\
        8724.01182,\
        8949.39962,\
        8957.3505,\
        8277.55612,\
        8444.20324,\
        8893.83898,\
        9098.32017,\
        9073.61506,\
        8304.08589,\
        8306.78809,\
        8320.15401,\
        8342.07131,\
        8361.88933,\
        8304.08589,\
        8312.79044,\
        8371.51336,\
        8463.12132,\
        8516.31466,\
        8304.08589,\
        8326.19709,\
        8470.40248,\
        8646.25524,\
        8703.25291,\
        8304.08589,\
        8352.31411,\
        8611.35986,\
        8846.88636,\
        8878.20033,\
        8304.08589,\
        8394.19493,\
        8773.88142,\
        9030.53726,\
        9024.28646,\
        8330.61565,\
        8331.07321,\
        8338.25071,\
        8350.36738,\
        8362.2166,\
        8330.61565,\
        8334.24052,\
        8366.77519,\
        8424.79031,\
        8465.36882,\
        8330.61565,\
        8340.86189,\
        8426.80661,\
        8557.78723,\
        8614.77494,\
        8330.61565,\
        8354.22038,\
        8524.98624,\
        8729.8935,\
        8777.03412,\
        8330.61565,\
        8377.20631,\
        8655.18562,\
        8912.64257,\
        8929.75786,\
        8224.49659,\
        8363.90787,\
        8637.35096,\
        8762.75374,\
        8772.9849,\
        8224.49659,\
        8623.79882,\
        8866.45563,\
        8883.21465,\
        8877.03149,\
        8224.7895,\
        8805.64678,\
        8885.87324,\
        8886.0712,\
        8885.24332,\
        8225.34152,\
        8873.41351,\
        8886.0891,\
        8886.08974,\
        8885.95443,\
        8225.87848,\
        8885.05561999999,\
        8886.08217,\
        8886.08178,\
        8886.05102,\
        8251.02635,\
        8320.44349,\
        8566.11699,\
        8747.51056,\
        8782.98012,\
        8251.02635,\
        8542.61478,\
        8964.26984,\
        9064.69407,\
        9040.30774,\
        8251.02635,\
        8800.20603,\
        9098.12618,\
        9109.15596,\
        9097.40907,\
        8251.02635,\
        8985.65585,\
        9110.73542,\
        9111.07673,\
        9107.84909,\
        8251.02635,\
        9076.81712,\
        9111.13059,\
        9111.13354,\
        9110.05049,\
        8277.55612,\
        8309.85206,\
        8478.55084999999,\
        8656.05346,\
        8711.04313,\
        8277.55612,\
        8453.83882,\
        8912.77887,\
        9112.69857,\
        9084.93134,\
        8277.55612,\
        8693.85774,\
        9201.65958,\
        9274.07721,\
        9230.64524,\
        8277.55612,\
        8935.98236,\
        9288.5972,\
        9297.97094,\
        9276.80073,\
        8277.55612,\
        9121.22747,\
        9299.34649,\
        9299.92217,\
        9290.96818,\
        8304.08589,\
        8319.00749,\
        8420.84127,\
        8560.83521,\
        8620.60315,\
        8304.08589,\
        8400.31795,\
        8793.78521,\
        9050.464,\
        9039.68889,\
        8304.08589,\
        8578.93793,\
        9172.34907,\
        9345.37047,\
        9272.52881,\
        8304.08589,\
        8809.7782,\
        9383.48193,\
        9440.57979,\
        9377.81335,\
        8304.08589,\
        9035.89578,\
        9448.71591,\
        9458.19287,\
        9421.92158,\
        8330.61565,\
        8337.3089,\
        8395.7747,\
        8492.81648,\
        8545.60419,\
        8330.61565,\
        8380.71501,\
        8672.35316,\
        8934.1408,\
        8946.8699,\
        8330.61565,\
        8496.68059,\
        9057.90776,\
        9315.64171,\
        9237.87994,\
        8330.61565,\
        8679.58277,\
        9365.94695,\
        9512.98611,\
        9404.64963,\
        8330.61565,\
        8894.89906,\
        9529.618,\
        9579.65209,\
        9492.2881,\
        8224.49659,\
        8596.98891,\
        8857.44651,\
        8881.33736,\
        8874.05819,\
        8225.22125,\
        8865.87898,\
        8886.08707,\
        8886.08952,\
        8885.89938,\
        8226.22247,\
        8885.94672,\
        8886.08045,\
        8886.08055,\
        8886.06843,\
        8227.10034,\
        8886.08958,\
        8886.08031,\
        8886.08031,\
        8886.08128,\
        8227.84163,\
        8886.08056,\
        8886.08031,\
        8886.08031,\
        8886.08277,\
        8251.02635,\
        8513.63455,\
        8933.48853,\
        9049.33429,\
        9026.38896,\
        8251.02635,\
        8953.40024,\
        9110.22464,\
        9111.00967,\
        9106.76153,\
        8251.02635,\
        9099.75434,\
        9111.13373,\
        9111.1337,\
        9110.52473,\
        8251.26203,\
        9110.88962,\
        9111.1333,\
        9111.13332,\
        9110.91919,\
        8251.61472,\
        9111.13226,\
        9111.13328,\
        9111.13328,\
        9110.99596,\
        8277.55612,\
        8431.71825,\
        8867.98803,\
        9077.99518,\
        9057.71399,\
        8277.55612,\
        8886.52734,\
        9280.49453,\
        9296.34846,\
        9270.82697,\
        8277.55612,\
        9203.18767,\
        9299.97843,\
        9300.04491,\
        9294.6707,\
        8277.55612,\
        9290.87157,\
        9300.07853,\
        9300.07922,\
        9298.17127,\
        8277.55612,\
        9299.6334,\
        9300.07525,\
        9300.07581,\
        9298.98514,\
        8304.08589,\
        8386.39352,\
        8747.34974,\
        9003.17207,\
        9003.04326,\
        8304.08589,\
        8757.94197,\
        9352.74039,\
        9430.19894,\
        9361.81693,\
        8304.08589,\
        9165.34429,\
        9458.13244,\
        9460.24543,\
        9436.03821,\
        8304.08589,\
        9389.40695,\
        9460.91162,\
        9460.92152,\
        9451.55334,\
        8304.08589,\
        9451.5265,\
        9460.93776,\
        9460.93735,\
        9455.80246,\
        8330.61565,\
        8372.78828,\
        8632.71798,\
        8883.70546,\
        8906.48457,\
        8330.61565,\
        8635.43391,\
        9310.48497,\
        9484.10874,\
        9376.51401,\
        8330.61565,\
        9039.15032,\
        9574.33866,\
        9592.7085,\
        9525.56063,\
        8330.61565,\
        9369.01688,\
        9598.6971,\
        9599.14263,\
        9568.36729,\
        8330.61565,\
        9537.44516,\
        9599.44275,\
        9599.43911,\
        9582.25735,\
        8224.82392,\
        8812.88334,\
        8885.93351,\
        8886.07636,\
        8885.34634,\
        8226.44537,\
        8886.05478,\
        8886.08034,\
        8886.08039,\
        8886.07415,\
        8227.74536,\
        8886.0808,\
        8886.08031,\
        8886.08031,\
        8886.08279,\
        8228.71661,\
        8886.08031,\
        8886.08031,\
        8886.08031,\
        8886.07924,\
        8229.43465,\
        8886.08031,\
        8886.08031,\
        8886.08031,\
        8886.06484,\
        8251.02635,\
        8814.49138,\
        9100.40175,\
        9109.54265,\
        9098.68867,\
        8250.9612,\
        9106.19809,\
        9111.13354,\
        9111.13354,\
        9110.69197,\
        8251.56819,\
        9111.13058,\
        9111.13328,\
        9111.13329,\
        9110.99096,\
        8252.04777,\
        9111.13331,\
        9111.13328,\
        9111.13328,\
        9111.01345,\
        8252.41741,\
        9111.13328,\
        9111.13328,\
        9111.13328,\
        9110.94211,\
        8277.55612,\
        8709.77413,\
        9211.98589,\
        9277.61149,\
        9235.40766,\
        8277.55612,\
        9240.18546,\
        9300.0491,\
        9300.06249,\
        9296.08817,\
        8277.55612,\
        9299.37997,\
        9300.07537,\
        9300.07601,\
        9298.92294,\
        8277.55612,\
        9300.08031,\
        9300.07501,\
        9300.07511,\
        9299.23516,\
        8277.55612,\
        9300.07542,\
        9300.07501,\
        9300.07503,\
        9299.0969,\
        8304.08589,\
        8592.48248,\
        9191.03268,\
        9355.92956,\
        9282.01032,\
        8304.08589,\
        9238.41671,\
        9459.96533,\
        9460.6624,\
        9441.97623,\
        8304.08589,\
        9448.02594,\
        9460.93759,\
        9460.93704,\
        9455.45652,\
        8304.08589,\
        9460.71486,\
        9460.93791,\
        9460.9379,\
        9457.49697,\
        8304.08589,\
        9460.93586,\
        9460.93791,\
        9460.93791,\
        9457.59114,\
        8330.61565,\
        8506.43132,\
        9080.76857,\
        9333.33757,\
        9251.50753,\
        8330.61565,\
        9130.7979,\
        9587.81767,\
        9596.23368,\
        9540.91565,\
        8330.61565,\
        9522.83226,\
        9599.43152,\
        9599.43065,\
        9581.04616,\
        8330.61565,\
        9595.18447,\
        9599.45711,\
        9599.45669,\
        9588.74577,\
        8330.61565,\
        9599.33015,\
        9599.45712,\
        9599.45711,\
        9590.22435,\
        14911.8452,\
        14928.9372,\
        14950.4966,\
        14973.7913,\
        14994.95,\
        14923.5429,\
        14976.6637,\
        15046.1089,\
        15115.6835,\
        15169.2842,\
        14922.9801,\
        15017.8646,\
        15144.8801,\
        15261.7781,\
        15339.2995,\
        14916.4988,\
        15053.8125,\
        15241.7076,\
        15400.5264,\
        15492.2293,\
        14908.952,\
        15086.8426,\
        15335.496,\
        15529.0614,\
        15627.0741,\
        14924.4064,\
        14929.9016,\
        14937.3585,\
        14946.1229,\
        14954.8421,\
        14929.4559,\
        14948.9896,\
        14978.9,\
        15014.1123,\
        15045.663,\
        14930.1177,\
        14967.0377,\
        15029.7802,\
        15101.7649,\
        15159.3929,\
        14928.292,\
        14983.2896,\
        15086.1969,\
        15199.6076,\
        15280.6616,\
        14925.7258,\
        14998.4195,\
        15146.8055,\
        15302.4705,\
        15401.9411,\
        14946.7685,\
        14948.8473,\
        14951.7064,\
        14955.1638,\
        14958.7522,\
        14949.311,\
        14957.3516,\
        14970.2262,\
        14986.5486,\
        15002.5461,\
        14950.0621,\
        14965.9334,\
        14995.0462,\
        15032.6555,\
        15066.8945999999,\
        14949.5737,\
        14973.8174,\
        15024.3784,\
        15089.7741,\
        15144.6161,\
        14948.5552,\
        14981.1118,\
        15057.6632,\
        15155.4818,\
        15230.6735,\
        14972.2274,\
        14972.4586,\
        14973.677,\
        14975.1597,\
        14976.7285,\
        14972.9537,\
        14976.612,\
        14982.4814,\
        14990.1414,\
        14998.03,\
        14973.5496,\
        14981.0364,\
        14994.9192,\
        15013.8686,\
        15032.5452,\
        14973.4884,\
        14985.1958,\
        15010.1251,\
        15045.1529,\
        15077.8558,\
        14973.0731,\
        14989.0383,\
        15027.8363,\
        15083.2367,\
        15131.8594,\
        14998.753,\
        14998.753,\
        14998.753,\
        14998.8534,\
        14999.5911,\
        14998.753,\
        14999.7805,\
        15002.6428,\
        15006.4061,\
        15010.3853,\
        14998.753,\
        15002.2203,\
        15009.1913,\
        15018.9054,\
        15028.9336,\
        14998.753,\
        15004.5724,\
        15017.3585,\
        15035.992,\
        15054.5964,\
        14998.753,\
        15006.7582,\
        15026.9871,\
        15057.486,\
        15086.7102,\
        14901.6509,\
        15126.7366,\
        15447.9503,\
        15674.8315,\
        15772.7816,\
        14893.6189,\
        15346.9687,\
        15923.601,\
        16196.3014,\
        16241.7694,\
        14894.2981,\
        15597.3597,\
        16260.4645,\
        16484.7367,\
        16475.8415,\
        14895.642,\
        15838.7026,\
        16485.2053,\
        16639.6764,\
        16601.6885,\
        14897.1378,\
        16051.3257,\
        16625.8753,\
        16716.951,\
        16671.838,\
        14922.8951,\
        15017.0399,\
        15227.3673,\
        15433.2698,\
        15547.9431,\
        14919.1762,\
        15139.0958,\
        15669.9086,\
        16029.596,\
        16128.1076,\
        14919.1762,\
        15328.8194,\
        16091.6356,\
        16463.3547,\
        16492.5874,\
        14919.1762,\
        15560.4444,\
        16436.1782,\
        16751.2972,\
        16720.981,\
        14919.4421,\
        15802.0688,\
        16695.859,\
        16931.9322,\
        16866.2549,\
        14947.226,\
        14989.9175,\
        15104.6281,\
        15246.8779,\
        15344.5848,\
        14945.7018,\
        15048.3926,\
        15419.9309,\
        15769.5646,\
        15902.5849,\
        14945.7018,\
        15157.0601,\
        15811.8081,\
        16261.2235,\
        16341.2783,\
        14945.7018,\
        15319.3739,\
        16200.1164,\
        16652.038,\
        16658.119,\
        14945.7018,\
        15519.4551,\
        16541.4562,\
        16940.4181,\
        16883.3248,\
        14972.4093,\
        14993.5774,\
        15053.5364,\
        15139.5368,\
        15208.8417,\
        14972.2274,\
        15021.7577,\
        15248.8555,\
        15527.4319,\
        15664.5177,\
        14972.2274,\
        15077.3474,\
        15548.1287,\
        15986.4573,\
        16106.3701,\
        14972.2274,\
        15172.7487,\
        15902.5349,\
        16416.6672,\
        16471.5311,\
        14972.2274,\
        15308.0775,\
        16261.1685,\
        16778.8059,\
        16758.1302,\
        14998.753,\
        15009.3058,\
        15041.1064,\
        15090.444,\
        15134.9464,\
        14998.753,\
        15023.7449,\
        15154.9928,\
        15350.4219,\
        15468.7945,\
        14998.753,\
        15051.6242,\
        15356.4643,\
        15723.5137,\
        15859.6351,\
        14998.753,\
        15103.2328,\
        15634.6562,\
        16130.9845,\
        16227.416,\
        14998.753,\
        15184.2418,\
        15955.5567,\
        16517.3933,\
        16544.4775,\
        14893.8392,\
        15486.2977,\
        16127.8455,\
        16378.7623,\
        16391.036,\
        14897.3221,\
        16074.5867,\
        16638.2622,\
        16722.8122,\
        16677.8754,\
        14901.3174,\
        16457.9289,\
        16759.5817,\
        16769.0651,\
        16744.5575,\
        14905.1998,\
        16660.4117,\
        16772.0083,\
        16772.3798,\
        16762.1085,\
        14908.8058,\
        16743.2991,\
        16772.5602,\
        16772.5633,\
        16767.582,\
        14919.1762,\
        15238.0383,\
        15913.8884,\
        16292.6925,\
        16353.1834,\
        14919.4968,\
        15830.6591,\
        16721.6693,\
        16948.0337,\
        16879.7353,\
        14920.728,\
        16373.7187,\
        17055.0505,\
        17117.4715,\
        17048.5997,\
        14921.9901,\
        16752.8089,\
        17136.426,\
        17144.2905,\
        17105.8271,\
        14923.2075,\
        16978.0624,\
        17146.8011,\
        17147.2913,\
        17126.8613,\
        14945.7018,\
        15102.1498,\
        15635.7962,\
        16055.6849,\
        16164.799,\
        14945.7018,\
        15545.076,\
        16578.3699,\
        16968.8568,\
        16905.5203,\
        14945.7018,\
        16108.654,\
        17152.5432,\
        17339.4853,\
        17215.0448,\
        14945.7407,\
        16604.3781,\
        17393.6092,\
        17445.5358,\
        17346.1806,\
        14946.2588,\
        16974.9395,\
        17458.9145,\
        17467.2099,\
        17403.7626,\
        14972.2274,\
        15048.402,\
        15406.1951,\
        15783.7854,\
        15919.9761,\
        14972.2274,\
        15326.6777,\
        16302.7778,\
        16817.1823,\
        16787.8206,\
        14972.2274,\
        15800.1262,\
        17048.046,\
        17398.5514,\
        17239.9667,\
        14972.2274,\
        16314.2125,\
        17485.1225,\
        17645.2742,\
        17467.9088,\
        14972.2274,\
        16771.5693,\
        17677.233,\
        17725.8831,\
        17584.2734,\
        14998.753,\
        15036.9446,\
        15256.8549,\
        15550.6476,\
        15687.3294,\
        14998.753,\
        15196.0177,\
        15995.2722,\
        16560.8712999999,\
        16578.8371,\
        14998.753,\
        15538.7681,\
        16800.7363,\
        17301.6008,\
        17144.2995,\
        14998.753,\
        15990.644,\
        17398.4329,\
        17711.5863,\
        17472.1283,\
        14998.753,\
        16459.2155,\
        17752.5062,\
        17897.9424,\
        17660.7972,\
        14896.8934,\
        16019.5459,\
        16607.998,\
        16708.1934,\
        16663.124,\
        14904.371,\
        16628.5463,\
        16771.3883,\
        16772.2068,\
        16759.8685,\
        14911.0475,\
        16762.7291,\
        16772.5736,\
        16772.5727,\
        16769.145,\
        14916.5776,\
        16772.3372,\
        16772.5745,\
        16772.5745,\
        16770.7882,\
        14921.0755,\
        16772.5718,\
        16772.5745,\
        16772.5745,\
        16771.172,\
        14919.37,\
        15763.7066,\
        16659.7474,\
        16908.7994,\
        16847.1329,\
        14921.7161,\
        16684.0616,\
        17129.0068,\
        17142.1489,\
        17097.8208,\
        14923.9829,\
        17062.0834,\
        17147.5145,\
        17147.5585,\
        17133.4104,\
        14925.9465,\
        17140.1915,\
        17147.6352,\
        17147.634,\
        17140.7716,\
        14927.5889,\
        17147.2823,\
        17147.6355,\
        17147.6355,\
        17142.8607,\
        14945.7018,\
        15485.7159,\
        16490.8326,\
        16900.56,\
        16852.2747,\
        14945.6261,\
        16505.9096,\
        17361.5583,\
        17433.7151,\
        17326.0867,\
        14946.5959,\
        17152.2921,\
        17468.4978,\
        17470.3163,\
        17423.8192,\
        14947.472,\
        17406.2661,\
        17471.5684,\
        17471.5954,\
        17448.6204,\
        14948.2278,\
        17463.5413,\
        17471.6086,\
        17471.6057,\
        17456.5833,\
        14972.2274,\
        15284.0051,\
        16205.0512,\
        16725.9374,\
        16717.0666,\
        14972.2274,\
        16204.3601,\
        17414.0291,\
        17610.6655,\
        17430.3752,\
        14972.2274,\
        17026.423,\
        17725.7735,\
        17743.1475,\
        17629.4018,\
        14972.2274,\
        17501.5696,\
        17752.8685,\
        17753.1605,\
        17691.1784,\
        14972.3371,\
        17692.8266,\
        17753.8751,\
        17753.8375,\
        17713.6182,\
        14998.753,\
        15169.2232,\
        15902.8187,\
        16458.3365,\
        16497.4423,\
        14998.753,\
        15887.586,\
        17289.3064,\
        17645.1972,\
        17415.0836,\
        14998.753,\
        16752.3941,\
        17879.9193,\
        17952.8721,\
        17740.8892,\
        14998.753,\
        17404.1142,\
        17991.7802,\
        17996.5986,\
        17861.5909,\
        14998.753,\
        17777.0947,\
        18001.556,\
        18001.4569,\
        17910.9889,\
        14901.566,\
        16475.2774,\
        16761.668,\
        16769.6366,\
        16746.4209,\
        14912.4757,\
        16768.2033,\
        16772.5744,\
        16772.574,\
        16769.7826,\
        14920.4989,\
        16772.5692,\
        16772.5745,\
        16772.5745,\
        16771.1491,\
        14926.2251,\
        16772.5745,\
        16772.5745,\
        16772.5745,\
        16771.0827,\
        14930.3415,\
        16772.5745,\
        16772.5745,\
        16772.5745,\
        16770.4579,\
        14920.8071,\
        16402.3872,\
        17065.0820999999,\
        17121.1802,\
        17054.2057,\
        14924.4834,\
        17096.6718,\
        17147.6046,\
        17147.6067,\
        17136.1577,\
        14927.3764,\
        17147.0778,\
        17147.6355,\
        17147.6355,\
        17142.6954,\
        14929.5085,\
        17147.6337,\
        17147.6355,\
        17147.6355,\
        17143.4136,\
        14931.0673,\
        17147.6355,\
        17147.6355,\
        17147.6355,\
        17142.6079,\
        14945.7018,\
        16142.6945,\
        17175.8932,\
        17351.3262,\
        17226.8879,\
        14946.8163,\
        17242.57,\
        17470.4907,\
        17471.0506,\
        17432.8059,\
        14948.1289,\
        17460.494,\
        17471.6082,\
        17471.6043,\
        17455.8957,\
        14949.1371,\
        17471.4552,\
        17471.6089,\
        17471.6088,\
        17459.9198,\
        14949.8965,\
        17471.607,\
        17471.6089,\
        17471.6089,\
        17459.525,\
        14972.2274,\
        15832.464,\
        17084.2757,\
        17421.8593,\
        17259.2187,\
        14972.2274,\
        17172.2382,\
        17740.4379,\
        17748.2914,\
        17650.83,\
        14972.2848,\
        17677.6942,\
        17753.8572,\
        17753.8129,\
        17711.5692,\
        14972.8259,\
        17749.7519,\
        17753.9,\
        17753.8974,\
        17724.8692,\
        14973.2465,\
        17753.7722,\
        17753.9001,\
        17753.8999,\
        17726.8449,\
        14998.753,\
        15564.8821,\
        16845.3304,\
        17336.0365,\
        17170.5257,\
        14998.753,\
        16933.8041,\
        17930.5942,\
        17972.8695,\
        17780.9679,\
        14998.753,\
        17739.8441,\
        18001.2612,\
        18001.2395,\
        17906.2566,\
        14998.753,\
        17969.2138,\
        18002.1058,\
        18002.0699,\
        17938.9415,\
        14998.753,\
        17999.6778,\
        18002.1107,\
        18002.1074,\
        17947.2729,\
        21613.1298,\
        21672.1219,\
        21737.1647,\
        21800.2028,\
        21852.8802,\
        21663.6858,\
        21822.0038,\
        21985.5739,\
        22126.0086,\
        22223.8686,\
        21671.1408999999,\
        21939.1046,\
        22197.8536,\
        22397.2732,\
        22516.5403,\
        21653.3522,\
        22034.8311,\
        22380.4316,\
        22622.0205,\
        22747.7388,\
        21623.9138,\
        22117.215,\
        22540.1134,\
        22810.3524,\
        22933.6209,\
        21574.2675,\
        21593.3334,\
        21617.5641,\
        21644.3656,\
        21669.6754,\
        21593.2818,\
        21656.2783,\
        21740.4827,\
        21829.0003,\
        21901.8009,\
        21597.0915,\
        21713.2306,\
        21874.0305,\
        22032.1806,\
        22146.2327,\
        21591.5541,\
        21763.3625,\
        22008.7091,\
        22233.3446,\
        22376.4876,\
        21582.2945,\
        21808.8414,\
        22141.8114,\
        22425.2794,\
        22585.8957,\
        21584.8093,\
        21591.7454,\
        21600.9738,\
        21611.7733,\
        21622.6479,\
        21593.5633,\
        21619.2609,\
        21657.4179,\
        21702.4455,\
        21743.9714,\
        21596.3936,\
        21646.3375,\
        21727.6826,\
        21821.9946,\
        21900.8894,\
        21595.011,\
        21670.9853,\
        21805.7803,\
        21957.2361,\
        22071.9267,\
        21591.645,\
        21693.5959,\
        21889.4653,\
        22100.8613,\
        22245.8275,\
        21605.8349,\
        21608.6861,\
        21612.5168,\
        21617.0955,\
        21621.8583,\
        21610.3255,\
        21621.7191,\
        21639.2394,\
        21661.1336,\
        21682.8051,\
        21612.3093,\
        21635.377,\
        21675.3019,\
        21726.1075,\
        21773.2268,\
        21612.1946,\
        21648.1498,\
        21717.7189,\
        21806.4872,\
        21883.0303,\
        21610.9044,\
        21659.9173,\
        21765.3396,\
        21898.6636,\
        22005.0808,\
        21630.2212,\
        21631.2863,\
        21633.0241,\
        21635.1125,\
        21637.3195,\
        21632.4656,\
        21637.9423,\
        21646.3984,\
        21657.2303,\
        21668.4021,\
        21633.8172,\
        21645.2814,\
        21665.3878,\
        21692.199,\
        21718.7648,\
        21634.0618,\
        21652.3296,\
        21688.4993,\
        21737.9976,\
        21784.6849,\
        21633.5897,\
        21658.8717,\
        21715.1222,\
        21793.2524,\
        21862.9372,\
        21586.9004,\
        22208.8334,\
        22714.5302,\
        23006.0986,\
        23119.1866,\
        21534.0751,\
        22598.4641,\
        23317.9925,\
        23590.7739,\
        23629.1889,\
        21536.7554,\
        22937.1847,\
        23657.1641,\
        23847.8484,\
        23837.7143,\
        21543.0697,\
        23220.527,\
        23849.9815,\
        23965.188,\
        23936.1695,\
        21549.9117,\
        23446.505,\
        23955.9411,\
        24017.1829,\
        23986.3147,\
        21571.004,\
        21862.424,\
        22304.3798,\
        22648.3252,\
        22817.6432,\
        21552.5407,\
        22149.1173,\
        23024.6951,\
        23483.1364,\
        23592.6627,\
        21552.3001,\
        22495.5223,\
        23565.3884,\
        23968.7443,\
        23993.2319,\
        21553.608,\
        22854.6072,\
        23943.593,\
        24245.6975,\
        24214.8341,\
        21555.2229,\
        23187.716,\
        24196.1302,\
        24399.7191,\
        24343.3426,\
        21586.9933,\
        21720.3952,\
        22000.3397,\
        22285.3041,\
        22458.2589,\
        21577.3017,\
        21877.06,\
        22613.3864,\
        23149.4962,\
        23327.8196,\
        21577.1783,\
        22116.3748,\
        23217.6139,\
        23802.1508,\
        23891.4027,\
        21577.1783,\
        22422.5973,\
        23729.6577,\
        24251.5201,\
        24252.5939,\
        21577.555,\
        22756.6738,\
        24130.5745,\
        24548.6641999999,\
        24488.0756,\
        21608.7561,\
        21673.7209,\
        21831.5714,\
        22026.492,\
        22167.2527,\
        21603.6997,\
        21753.3734,\
        22262.6659,\
        22758.0085,\
        22971.06,\
        21603.6997,\
        21890.2503,\
        22797.7959,\
        23456.9314,\
        23614.093,\
        21603.6997,\
        22097.0486,\
        23339.2083,\
        24024.4075,\
        24085.3084,\
        21603.6997,\
        22358.9617,\
        23826.5915,\
        24454.4344,\
        24425.2615,\
        21632.5523,\
        21666.4777,\
        21753.118,\
        21874.0848,\
        21974.0124,\
        21630.2212,\
        21707.8647,\
        22026.9388,\
        22418.3005,\
        22626.7045,\
        21630.2212,\
        21780.7219,\
        22432.8861,\
        23057.2057,\
        23259.8761,\
        21630.2212,\
        21903.3981,\
        22914.6032,\
        23660.5092,\
        23786.4159,\
        21630.2212,\
        22078.6583,\
        23407.8452,\
        24174.9906,\
        24202.8448,\
        21534.6353,\
        22794.3558,\
        23530.881,\
        23758.9159,\
        23765.8037,\
        21550.7394,\
        23470.0206,\
        23964.655,\
        24020.92,\
        23990.4566,\
        21568.0604,\
        23822.8103,\
        24043.6653,\
        24049.328,\
        24034.182,\
        21584.0792,\
        23978.3676,\
        24051.1157,\
        24051.3452,\
        24045.0561,\
        21598.5061,\
        24033.5263,\
        24051.4607,\
        24051.4627,\
        24048.4119,\
        21552.0212,\
        22338.3126,\
        23349.4349,\
        23787.6414,\
        23846.8821,\
        21555.4249,\
        23225.01,\
        24219.6783,\
        24412.6062,\
        24354.7379,\
        21559.87,\
        23865.587,\
        24494.1779,\
        24537.832,\
        24488.671,\
        21564.2618,\
        24242.2346,\
        24550.4033,\
        24555.467,\
        24529.6648,\
        21568.386,\
        24434.7689,\
        24557.0739,\
        24557.3856,\
        24543.9985,\
        21577.1783,\
        22000.6744,\
        22960.1203,\
        23542.687,\
        23674.3002,\
        21577.6246,\
        22797.0534,\
        24171.4827,\
        24576.36,\
        24510.3249,\
        21579.2147,\
        23593.7432,\
        24747.5621,\
        24906.89,\
        24801.0738,\
        21580.856,\
        24188.2835,\
        24949.4075,\
        24988.278,\
        24911.5103,\
        21582.445,\
        24577.2077,\
        24997.928,\
        25003.8142,\
        24956.8664,\
        21603.6997,\
        21821.253,\
        22556.322,\
        23163.0572,\
        23354.1901,\
        21603.6997,\
        22393.0082,\
        23880.0727,\
        24497.7098,\
        24459.0902,\
        21603.6997,\
        23165.3718,\
        24748.1411,\
        25099.1657,\
        24942.5473,\
        21604.0704,\
        23877.2111,\
        25178.949,\
        25318.812,\
        25161.8879,\
        21604.7964,\
        24435.7731,\
        25345.1204,\
        25384.0114,\
        25266.2782,\
        21630.2212,\
        21743.2027,\
        22240.7827,\
        22774.8313,\
        22992.39,\
        21630.2212,\
        22102.9165,\
        23465.531,\
        24230.0367,\
        24246.2089,\
        21630.2212,\
        22732.9244,\
        24522.4433,\
        25090.9455,\
        24915.5574,\
        21630.2212,\
        23438.7218,\
        25192.2718,\
        25504.365,\
        25266.257,\
        21630.2212,\
        24084.7869,\
        25542.5945,\
        25673.8472,\
        25454.4882,\
        21548.8086,\
        23413.9988,\
        23943.1527,\
        24011.5414,\
        23980.2862,\
        21580.7089,\
        23955.666,\
        24050.7369,\
        24051.2368,\
        24043.6804,\
        21607.3234,\
        24045.5061,\
        24051.4695,\
        24051.469,\
        24049.3717,\
        21628.7496,\
        24051.3229,\
        24051.4702,\
        24051.4702,\
        24050.3872,\
        21646.0087,\
        24051.4685,\
        24051.4702,\
        24051.4702,\
        24050.6325,\
        21554.9555,\
        23137.0291,\
        24162.6584,\
        24380.9464,\
        24327.0171,\
        21563.3196,\
        24178.4468,\
        24545.5236,\
        24554.0925,\
        24524.096,\
        21570.967,\
        24498.7404,\
        24557.5283,\
        24557.5563,\
        24548.365,\
        21577.3773,\
        24552.8529,\
        24557.6052,\
        24557.6045,\
        24553.2234,\
        21582.6263,\
        24557.3844,\
        24557.6054,\
        24557.6054,\
        24554.5877,\
        21577.4635,\
        22702.7611,\
        24073.6893,\
        24509.3476,\
        24456.6573,\
        21580.4991,\
        24076.9462,\
        24924.4309,\
        24979.6223,\
        24895.1897,\
        21583.4593,\
        24745.0415,\
        25004.7256,\
        25006.0039,\
        24972.1123,\
        21586.0344,\
        24958.9682,\
        25006.9137,\
        25006.904,\
        24990.5023,\
        21588.1935,\
        25001.2433,\
        25006.9404,\
        25006.9386,\
        24996.2891999999,\
        21603.6997,\
        22314.2898,\
        23753.5224,\
        24394.1022,\
        24378.0391999999,\
        21603.9097,\
        23733.2665,\
        25113.4038,\
        25289.6735,\
        25127.0982,\
        21605.2682,\
        24719.7836,\
        25383.929,\
        25397.4366,\
        25305.0535,\
        21606.4931,\
        25193.1579,\
        25404.9297,\
        25405.1551,\
        25356.3121,\
        21607.5478,\
        25357.7402,\
        25405.7108,\
        25405.6812,\
        25374.3481,\
        21630.2212,\
        22047.3282,\
        23330.1993,\
        24099.3546,\
        24142.9491,\
        21630.2212,\
        23286.2024,\
        25076.9072,\
        25440.8879,\
        25207.2924,\
        21630.2212,\
        24455.402,\
        25658.0478,\
        25720.9411,\
        25530.9318,\
        21630.3847,\
        25196.3843,\
        25753.4182,\
        25757.4051,\
        25641.5796,\
        21630.9579,\
        25565.207,\
        25761.4344,\
        25761.3494,\
        25685.0541,\
        21569.1059,\
        23837.0997,\
        24044.9155,\
        24049.672,\
        24035.3483,\
        21612.8934,\
        24048.8184,\
        24051.4701,\
        24051.4698,\
        24049.7643,\
        21643.7992,\
        24051.4668,\
        24051.4702,\
        24051.4702,\
        24050.6169,\
        21665.7839,\
        24051.4702,\
        24051.4702,\
        24051.4702,\
        24050.6011,\
        21681.7576,\
        24051.4702,\
        24051.4702,\
        24051.4702,\
        24050.268,\
        21560.1494,\
        23896.1101,\
        24501.4129,\
        24540.316,\
        24492.8017,\
        21572.6172,\
        24523.4596,\
        24557.5858,\
        24557.5871,\
        24550.2017,\
        21581.9523,\
        24557.2556,\
        24557.6054,\
        24557.6054,\
        24554.4793,\
        21588.6578,\
        24557.6043,\
        24557.6054,\
        24557.6054,\
        24554.9676,\
        21593.4872,\
        24557.6054,\
        24557.6054,\
        24557.6054,\
        24554.4926,\
        21579.3173,\
        23637.3693,\
        24768.3632,\
        24916.352,\
        24811.4042,\
        21584.1149,\
        24825.1288,\
        25006.1276,\
        25006.5206,\
        24978.8384,\
        21587.9139,\
        24999.0775,\
        25006.9402,\
        25006.9376,\
        24995.7914,\
        21590.7215,\
        25006.8072,\
        25006.9406,\
        25006.9405,\
        24998.7136,\
        21592.7773,\
        25006.9394,\
        25006.9406,\
        25006.9406,\
        24998.4783,\
        21603.6997,\
        23213.2947,\
        24786.1848,\
        25120.9863,\
        24961.7616,\
        21605.5765,\
        24873.3087999999,\
        25395.346,\
        25401.4027,\
        25323.0855,\
        21607.4098,\
        25345.4598,\
        25405.6967,\
        25405.662,\
        25372.7148,\
        21608.8139,\
        25402.5414,\
        25405.7305,\
        25405.7284,\
        25383.2781,\
        21609.8686,\
        25405.631,\
        25405.7306,\
        25405.7305,\
        25384.8791,\
        21630.2212,\
        22776.7131,\
        24575.5647,\
        25127.5109,\
        24944.6396,\
        21630.2212,\
        24673.0357,\
        25702.0069,\
        25737.7166,\
        25568.3159,\
        21630.8821,\
        25530.5246,\
        25761.1823,\
        25761.1636,\
        25680.9395,\
        21631.6636,\
        25734.6633,\
        25761.9133,\
        25761.8808,\
        25709.1369,\
        21632.2669,\
        25759.9632,\
        25761.9181,\
        25761.9148,\
        25716.2675,\
        28412.8864,\
        28543.9182,\
        28669.3786,\
        28779.0991,\
        28864.2693,\
        28550.95,\
        28838.6929,\
        29082.4259,\
        29268.5529,\
        29388.8786,\
        28593.0381,\
        29039.2417,\
        29375.2255,\
        29603.388,\
        29729.4396,\
        28579.8283999999,\
        29189.8954,\
        29598.332,\
        29846.9305,\
        29965.4319,\
        28530.6844,\
        29310.8264,\
        29775.6054,\
        30030.2395,\
        30135.2501,\
        28216.9262,\
        28265.6932,\
        28323.0459,\
        28382.3543,\
        28435.3631,\
        28270.1927,\
        28415.9164,\
        28584.1611,\
        28742.6807,\
        28863.3531,\
        28284.9007,\
        28542.5532,\
        28831.377,\
        29078.9064,\
        29242.4216,\
        28274.6339,\
        28649.3127,\
        29056.5515,\
        29374.8456,\
        29559.575,\
        28251.537,\
        28742.4397,\
        29261.0965,\
        29632.1419,\
        29822.7331,\
        28195.0246,\
        28212.7672,\
        28235.4206,\
        28260.9139,\
        28285.7137,\
        28218.2584,\
        28280.2844,\
        28364.6121,\
        28456.7305,\
        28536.6485,\
        28226.511,\
        28344.1963,\
        28511.6785,\
        28685.7912,\
        28820.4376,\
        28223.8933,\
        28401.2458,\
        28663.9075,\
        28921.6037,\
        29099.968,\
        28215.5945,\
        28452.7285,\
        28817.0769,\
        29153.0071,\
        29362.0065,\
        28206.5063,\
        28213.6474,\
        28223.0285,\
        28233.993,\
        28245.1635,\
        28217.9232,\
        28245.4406,\
        28285.5994,\
        28333.3243,\
        28378.5487,\
        28223.1231,\
        28277.954,\
        28365.2837,\
        28468.1512,\
        28557.7925,\
        28223.1052,\
        28308.0327,\
        28454.6389,\
        28623.5215,\
        28758.3643,\
        28220.0325,\
        28335.5541,\
        28550.6605,\
        28790.8026,\
        28966.3317,\
        28227.1996,\
        28230.3731,\
        28234.5663,\
        28239.5401,\
        28244.7318,\
        28233.2907,\
        28246.3605,\
        28265.9196,\
        28290.1878,\
        28314.496,\
        28236.6473,\
        28263.7007,\
        28308.7814,\
        28365.8552,\
        28419.8902,\
        28237.3289,\
        28280.2343,\
        28359.319,\
        28460.0331,\
        28549.4855,\
        28236.2448,\
        28295.5299,\
        28415.8504,\
        28568.3834,\
        28694.8978,\
        28439.2992,\
        29435.8808,\
        29952.1153,\
        30200.8309,\
        30286.3294,\
        28176.6334,\
        29870.5012,\
        30430.5072,\
        30573.573,\
        30589.8428,\
        28179.9394,\
        30164.739,\
        30600.7166,\
        28202.995,\
        30369.4748,\
        28227.4158,\
        30503.7878,\
        28218.7079,\
        28846.85,\
        29491.4082,\
        29906.1974,\
        30090.3121,\
        28154.5097,\
        29310.7076,\
        30328.583,\
        30739.5891,\
        30825.0929,\
        28153.0965,\
        29755.426,\
        30809.3696,\
        31089.8948,\
        31104.3671,\
        28157.0182,\
        30153.0286,\
        31074.6975,\
        31237.8077,\
        31223.8828,\
        28161.8501,\
        30480.095,\
        31214.3263,\
        31299.9955,\
        31279.6557,\
        28203.1591,\
        28512.298,\
        29007.2455,\
        29428.3892,\
        29658.881,\
        28174.8354,\
        28818.1535,\
        29887.1788,\
        30502.6481,\
        30684.3956,\
        28172.8876,\
        29203.3969,\
        30582.6042,\
        31142.7675,\
        31217.2822,\
        28173.7957,\
        29632.0262,\
        31080.2565,\
        31504.6811,\
        31504.6259,\
        28175.2358,\
        30051.1105,\
        31414.3198,\
        31703.4399,\
        31665.8888,\
        28214.6183,\
        28367.5041,\
        28678.0179,\
        29008.3765,\
        29224.7143,\
        28199.2634,\
        28537.7744,\
        29390.798,\
        30059.7429,\
        30316.3602,\
        28197.9871,\
        28790.8521,\
        30118.2914,\
        30878.8415,\
        31039.4585,\
        28197.9871,\
        29129.2382,\
        30753.5443,\
        31448.7201,\
        31502.0773,\
        28198.1726,\
        29515.8414,\
        31259.8892,\
        31825.8106,\
        31800.3028,\
        28233.7347,\
        28313.2348,\
        28493.923,\
        28719.0354,\
        28889.7949,\
        28224.6508,\
        28405.2633,\
        28993.69,\
        29591.0371,\
        29875.4992,\
        28224.5044,\
        28551.5651,\
        29618.7327,\
        30441.3331,\
        30679.4181,\
        28224.5044,\
        28775.0282,\
        30265.4745,\
        31141.0842,\
        31271.9019,\
        28224.5044,\
        29066.5808,\
        30858.6766,\
        31674.4239,\
        31698.357,\
        28172.9433,\
        30047.6036,\
        30546.5998,\
        28230.3003,\
        30516.0469,\
        28287.7128,\
        28337.1166,\
        28379.6125,\
        28152.3537,\
        29563.1834,\
        30631.1286,\
        30971.8104,\
        31011.1537,\
        28162.4495,\
        30514.3664,\
        31225.5532,\
        31304.4051,\
        31284.0526,\
        28175.3683,\
        31023.2264,\
        31328.797,\
        31338.556,\
        31327.6245,\
        28187.757,\
        31236.4815,\
        31340.5917,\
        31341.227,\
        31337.0421,\
        28199.1665,\
        31311.8073,\
        31341.3749,\
        31341.3972,\
        31339.6165,\
        28173.0187,\
        29024.9225,\
        30301.9401,\
        30904.0993,\
        31023.7329,\
        28175.422,\
        30099.0562,\
        31445.4841,\
        31720.0422,\
        31679.9715,\
        28179.6137,\
        30945.7855,\
        31813.9586,\
        31885.2473,\
        31840.4409,\
        28183.8362,\
        31455.4301,\
        31900.3638,\
        31911.9983,\
        31887.1341,\
        28187.8495,\
        31720.1522,\
        31914.3042,\
        31915.5208,\
        31902.8349,\
        28197.9871,\
        28667.5,\
        29805.0306,\
        30551.425,\
        30760.4371,\
        28198.244,\
        29563.6109,\
        31311.9316,\
        31860.9474,\
        31828.2483,\
        28199.9317,\
        30540.6257,\
        32052.8298,\
        32284.6662,\
        32187.6813,\
        28201.704,\
        31299.968,\
        32329.3208,\
        32397.8283,\
        32320.4017,\
        28203.4333,\
        31807.1276,\
        32408.6294,\
        32423.0013,\
        32373.9149,\
        28224.5044,\
        28478.0502,\
        29335.0051,\
        30081.9378,\
        30353.4298,\
        28224.5044,\
        29105.1232,\
        30924.2998,\
        31728.1841,\
        31740.6588,\
        28224.5044,\
        30009.5727,\
        32002.0128,\
        32478.3487,\
        32339.7457,\
        28224.9223,\
        30878.8672,\
        32553.333,\
        32762.1842,\
        32606.437,\
        28225.7709,\
        31576.5739,\
        32784.4179,\
        32854.6639,\
        32731.5154,\
        28223.5484,\
        30486.2578,\
        28326.9444,\
        28404.8948,\
        28464.8324,\
        28512.1212999999,\
        28161.055,\
        30432.7338,\
        31197.7806,\
        31293.3292,\
        31273.1912,\
        28185.1224,\
        31205.9704,\
        31339.8571,\
        31341.0747,\
        31335.9215,\
        28206.2288,\
        31329.9461,\
        31341.4053,\
        31341.4067,\
        31340.2883,\
        28223.5947,\
        31340.896,\
        31341.4085,\
        31341.4085,\
        31340.9451,\
        28237.7116,\
        31341.3956,\
        31341.4085,\
        31341.4085,\
        31341.1031,\
        28174.9906,\
        29986.2167,\
        31370.0186,\
        31679.2674,\
        31645.7098,\
        28182.9254,\
        31368.7792,\
        31891.6865,\
        31909.6817,\
        31880.9114,\
        28190.3806,\
        31812.5752,\
        31915.6792,\
        31915.8882,\
        31907.4667,\
        28196.7205,\
        31903.2112,\
        31916.0056,\
        31916.0047,\
        31912.4491,\
        28201.9574,\
        31914.9581,\
        31916.0075,\
        31916.0074,\
        31913.8233,\
        28198.0797,\
        29452.389,\
        31187.6626,\
        31775.9303,\
        31760.7648,\
        28201.3173,\
        31156.3781,\
        32292.7089,\
        32384.8987,\
        32300.9207,\
        28204.5431,\
        32030.6971,\
        32422.9346,\
        32427.1343,\
        32391.6733,\
        28207.3789,\
        32336.1143,\
        32429.0338,\
        32429.0794,\
        32412.7896,\
        28209.7751,\
        32413.103,\
        32429.1769,\
        32429.1728,\
        32419.3613,\
        28224.5044,\
        29016.2218,\
        30769.1775,\
        31599.491,\
        31639.2609,\
        28224.735,\
        30701.0931,\
        32467.2541,\
        32723.1503,\
        32564.3492,\
        28226.3242,\
        31936.4209,\
        32846.7826,\
        32876.0111,\
        32777.6172,\
        28227.7665,\
        32554.8621,\
        32888.5948,\
        32889.9192,\
        32838.1083,\
        28229.0149,\
        32796.0289,\
        32891.1106,\
        32891.0893,\
        32859.2628,\
        28291.0253,\
        28420.6559,\
        28506.0973,\
        28565.9292,\
        28609.6765,\
        28176.1658,\
        31043.233,\
        31330.6157,\
        31338.9984,\
        31328.6822,\
        28210.7198,\
        31335.6412,\
        31341.4079,\
        31341.408,\
        31340.5498,\
        28235.9012,\
        31341.3862,\
        31341.4085,\
        31341.4085,\
        31341.0914,\
        28253.9166,\
        31341.4085,\
        31341.4085,\
        31341.4085,\
        31341.1347,\
        28266.949,\
        31341.4085,\
        31341.4085,\
        31341.4085,\
        31341.056,\
        28179.8805,\
        30986.8722,\
        31824.2735,\
        31888.8045,\
        31845.2212,\
        28192.0057,\
        31850.7807,\
        31915.9054,\
        31915.9605,\
        31909.377,\
        28201.2827,\
        31914.4827,\
        31916.0075,\
        31916.0074,\
        31913.7108,\
        28208.0181,\
        31915.9934,\
        31916.0075,\
        31916.0075,\
        31914.3346,\
        28212.897,\
        31916.0075,\
        31916.0075,\
        31916.0075,\
        31914.2146,\
        28200.0421,\
        30595.6458,\
        32080.2409,\
        32297.2535,\
        32200.2056,\
        28205.2628,\
        32140.136,\
        32426.5162,\
        32428.2044,\
        32399.446,\
        28209.4639,\
        32408.2451,\
        32429.175,\
        32429.1699,\
        32418.7915,\
        28212.6012,\
        32428.4352,\
        32429.1791,\
        32429.1789,\
        32422.3361,\
        28214.915,\
        32429.1635,\
        32429.1791,\
        32429.1791,\
        32422.7253,\
        28224.5044,\
        30067.2307,\
        32049.881,\
        32505.9218,\
        32363.2108,\
        28226.6865,\
        32133.0975,\
        32867.6465,\
        32882.834,\
        32798.9676,\
        28228.8513,\
        32776.0359,\
        32891.0468,\
        32891.042,\
        32857.342,\
        28230.521,\
        32881.1727,\
        32891.2182,\
        32891.2123,\
        32870.0506,\
        28231.7822,\
        32890.6503,\
        32891.2188,\
        32891.2184,\
        32872.9128,\
        35445.743,\
        35631.4695,\
        35790.4668,\
        35919.0926,\
        36013.5131,\
        35675.1101,\
        36008.8747,\
        36255.1667,\
        36425.5346,\
        36527.3543,\
        35762.7132,\
        36231.6093,\
        36526.5281,\
        36698.4626,\
        36781.1906,\
        35781.1662,\
        36383.2836,\
        36700.6019,\
        36851.6519,\
        36908.139,\
        35755.8877,\
        36494.8393,\
        36815.1972,\
        36934.5654,\
        36967.6172,\
        34885.2452,\
        34986.6913,\
        35095.8103,\
        35200.7005,\
        35289.2391,\
        35006.9081,\
        35272.1626,\
        35535.4669,\
        35758.8835,\
        35917.1215,\
        35048.9838,\
        35489.804,\
        35891.6555,\
        36195.1983,\
        36379.8853,\
        35040.0659,\
        35661.8614,\
        36181.9151,\
        36532.9874,\
        36718.4554,\
        34999.788,\
        35804.0916,\
        36422.2649,\
        36796.0352,\
        36968.1296,\
        34785.8035,\
        34824.1929,\
        34870.8626,\
        34921.0849,\
        34968.1116,\
        34838.1763999999,\
        34962.7008,\
        35116.156,\
        35270.8632,\
        35397.1844,\
        34858.5219,\
        35086.8553,\
        35367.7125,\
        35629.9428,\
        35818.3198,\
        34855.5399,\
        35194.1657,\
        35608.0234,\
        35964.7,\
        36191.9158,\
        34839.0711,\
        35288.4927,\
        35833.792,\
        36267.2392,\
        36513.6757,\
        34776.0694,\
        34791.428,\
        34811.0587,\
        34833.399,\
        34855.6177,\
        34801.0476,\
        34857.5425,\
        34935.1324,\
        35022.3555,\
        35101.3129,\
        34812.787,\
        34922.9379,\
        35082.5465,\
        35255.8748,\
        35397.8759,\
        34813.4691,\
        34982.3784,\
        35238.8764,\
        35505.1611,\
        35702.7695,\
        34807.3701,\
        35036.095,\
        35398.7526,\
        35756.4791,\
        35997.3972,\
        34789.4777,\
        34796.1936,\
        34804.9272,\
        34815.1242,\
        34825.6106,\
        34802.4546,\
        34829.2853,\
        34867.9603,\
        34914.2127,\
        34959.0648,\
        34809.681,\
        34864.3939,\
        34950.219,\
        35052.7942,\
        35145.3693,\
        34811.3375,\
        34897.5007,\
        35043.5598,\
        35215.6943,\
        35359.5777,\
        34809.2393,\
        34927.9251,\
        35144.4065,\
        35393.6991,\
        35586.3016,\
        35680.173,\
        36599.7512,\
        36904.5308,\
        35071.3231,\
        36872.8461,\
        34979.7749,\
        35032.5668,\
        35093.7316,\
        34928.5983,\
        35954.6655,\
        36669.1427,\
        37045.6596,\
        37191.8359,\
        34735.6099,\
        36514.2457,\
        37366.4996,\
        37585.6948,\
        34728.4653,\
        36936.0066,\
        37612.7187,\
        34738.811,\
        37248.2032,\
        34751.6711,\
        37457.0294,\
        34811.4771,\
        35394.2632,\
        36095.4981,\
        36599.5872,\
        36849.5745,\
        34739.46,\
        35870.0482,\
        37105.9784,\
        37654.5123,\
        37794.4,\
        34734.0032,\
        36364.2833,\
        37720.3733,\
        38099.2397,\
        38141.4924,\
        34736.1104,\
        36843.3741,\
        38062.2558,\
        38278.3399,\
        38279.2695,\
        34739.5337,\
        37259.4253,\
        38238.6651,\
        34795.7742,\
        35097.5245,\
        35600.1316,\
        36062.6845,\
        36339.5622,\
        34760.1658,\
        35397.9258,\
        36570.2504,\
        37313.6043,\
        37568.128,\
        34755.8788,\
        35782.2293,\
        37380.8169,\
        38085.4892,\
        38215.7327,\
        34756.1964,\
        36238.4141,\
        37979.81,\
        38520.373,\
        38556.6939,\
        34757.3418,\
        36710.3082,\
        38386.2325,\
        38754.6403,\
        38741.2376,\
        34804.0251,\
        34962.8933,\
        35278.5606,\
        35628.4006,\
        35873.1459,\
        34784.0979,\
        35135.5447,\
        36038.926,\
        36801.1589,\
        37128.5179,\
        34781.0755,\
        35381.4962,\
        36842.5812,\
        37747.8745,\
        37981.8933,\
        34781.0755,\
        35721.9071,\
        37567.9373,\
        38414.9109,\
        38526.9938,\
        34781.0755,\
        36128.7743,\
        38158.1121,\
        38855.022,\
        38873.0567,\
        34978.9321,\
        36942.1826,\
        35100.7734,\
        35230.8515,\
        35330.7285,\
        35410.6658,\
        34726.9872,\
        36763.8302,\
        37536.1796,\
        34753.253,\
        37476.0201,\
        34786.5817,\
        34817.4936,\
        34845.3751,\
        34734.455,\
        36144.9585,\
        37490.7126,\
        37951.2095,\
        38028.3654,\
        34739.9751,\
        37303.9672,\
        38252.6629,\
        34749.7808,\
        37977.8836,\
        34759.4562,\
        38258.2735,\
        34768.5213,\
        34756.5906,\
        35601.2998,\
        37049.3763,\
        37796.8581,\
        37981.5447,\
        34757.5001,\
        36765.7213,\
        38424.1185,\
        38773.9856,\
        38756.9374,\
        34761.2051,\
        37781.1047,\
        38871.0205,\
        38966.182,\
        38930.5367,\
        34765.0293,\
        38414.1445,\
        38980.6324,\
        38999.4053,\
        38978.1309,\
        34768.7098,\
        38744.3159,\
        39001.7061,\
        39004.3799,\
        38993.4985,\
        34781.3904,\
        35261.1277,\
        36492.7701,\
        37366.9179,\
        37651.8214,\
        34780.9941,\
        36180.2053,\
        38219.1566,\
        38895.7607,\
        38905.1047,\
        34782.6224,\
        37276.2156,\
        39092.115,\
        39383.4361,\
        39307.1361,\
        34784.3831,\
        38169.3313,\
        39421.8193,\
        39515.4723,\
        39448.3771,\
        34786.1179,\
        38776.6215,\
        39523.6494,\
        39547.1272,\
        39503.34,\
        35084.2043,\
        35310.8575,\
        35456.2828,\
        35560.0346,\
        35638.6406,\
        34749.5677,\
        37429.829,\
        34810.9862,\
        34862.434,\
        34903.9896,\
        34937.5744,\
        34738.952,\
        37198.1688,\
        38218.0665,\
        34757.3828,\
        38218.7232,\
        34774.1906,\
        34788.2715,\
        34799.8223,\
        34757.1358,\
        36635.7794,\
        38332.378,\
        38726.4109,\
        38718.5349,\
        34764.1998,\
        38306.102,\
        38968.7985,\
        38996.3315,\
        38971.9182,\
        34771.0502,\
        38860.6072,\
        39004.3582,\
        39004.9644,\
        38997.8911,\
        34776.9684,\
        38982.169,\
        39005.1648,\
        39005.1681,\
        39002.4527,\
        34781.912,\
        39002.4254,\
        39005.1737,\
        39005.1736,\
        39003.66,\
        34781.0755,\
        36060.7945,\
        38073.4481,\
        38797.0422,\
        38827.6617,\
        34783.9973,\
        37998.5269,\
        39377.2684,\
        39499.9164,\
        39428.0819,\
        34787.2385,\
        39045.9159,\
        39544.7157,\
        39552.8092,\
        39521.1511,\
        34790.1256,\
        39421.2905,\
        39555.5017,\
        39555.7259,\
        39541.8098,\
        34792.5898,\
        39526.8878,\
        39555.8898,\
        39555.8875,\
        39548.041,\
        35237.8445,\
        35484.1155,\
        35628.7503,\
        35725.7726,\
        35795.5427,\
        34788.5976,\
        34873.226,\
        34933.2682,\
        34976.2245,\
        35007.6279,\
        34750.3973,\
        38004.4893,\
        34777.8143,\
        34798.337,\
        34813.1478,\
        34823.8785,\
        34761.445,\
        37831.8038,\
        38883.6685,\
        38970.4465,\
        38935.4953,\
        34772.56,\
        38909.783,\
        39004.8818,\
        39005.0876,\
        38999.6674,\
        34781.2728,\
        39001.4053,\
        39005.1737,\
        39005.1735,\
        39003.5614,\
        34787.6867,\
        39005.1006,\
        39005.1738,\
        39005.1738,\
        39004.1401,\
        34792.3758,\
        39005.173,\
        39005.1738,\
        39005.1738,\
        39004.1563,\
        34782.7313,\
        37339.9566,\
        39124.4763,\
        39397.9092,\
        39320.6821,\
        34787.9681,\
        39178.4793,\
        39550.6476,\
        39554.368,\
        39528.8404,\
        34792.2686,\
        39519.4568,\
        39555.8828,\
        39555.8823,\
        39547.5047,\
        34795.5242,\
        39553.8111,\
        39555.8996,\
        39555.8992,\
        39550.8922,\
        34797.9502,\
        39555.8215,\
        39555.8996,\
        39555.8996,\
        39551.5096]



    bob = ResponseSurface()
    bob.train(X,Y)
    print "Coefficients: ", bob.betas
    # new_y = bob.predict([1., 5.])
    new_y = bob.predict([3125.,7.0,2.1, 2.5, 7.0])
    

    print "Predicted: ", new_y
    
    print "Elapsed time: ", time.time()-tt, "seconds"
