/*
 *	Copyright (c) Vanderbilt University, 2006
 *	ALL RIGHTS RESERVED
 *
 *	Vanderbilt University disclaims all warranties with regard to this
 *	software, including all implied warranties of merchantability
 *	and fitness.  In no event shall Vanderbilt University be liable for
 *	any special, indirect or consequential damages or any damages
 *	whatsoever resulting from loss of use, data or profits, whether
 *	in an action of contract, negligence or other tortious action,
 *	arising out of or in connection with the use or performance of
 *	this software.
 *
 *  FILE NAME:	mdl2mga.cpp 
 *	SYNOPSIS:	Main program for translating SL/SF mdl files into GME ESMoL files.					
 *  AUTHOR:		Harmon Nine - hnine@isis.vanderbilt.edu
 *  CREATED:	07/20/06
 *  LAST UPDATE: 
 */
#include <fstream>
#include <string>
#include <set>
#include <vector>
#include <algorithm>
#include <ctype.h>
#include <matrix.h>
#include <boost/regex.hpp>
#include <boost/filesystem.hpp>
#include <boost/algorithm/string.hpp>
#include "UdmEngine.hpp"
#include "MatLabEngine.hpp"
#include "MatLabUdmBlock.hpp"
#include "MatLabUdmMiscSimulink.hpp"
#include "MatLabUdmState.hpp"
#include "MatLabUdmTransition.hpp"
#include "MatLabUdmData.hpp"
#include "SLSF.h"
#include "MatLabUdm.hpp"
#include "ConfigKeeper.hpp"

#ifdef _WIN32
#include <windows.h>
#include <tchar.h>
#endif
#include <stdio.h>

using namespace SLSF;


typedef std::vector< std::string > DirectoryVector;
typedef std::vector< std::string > MfileVector;
typedef std::vector< std::string > StringVector;

typedef std::set< SLSF::Subsystem > SubsystemSet;
typedef std::set< SLSF::InPort > InPortSet;
typedef std::set< SLSF::OutPort > OutPortSet;


void usage( const std::string &exeName );


#if PARADIGM == CyPhyML_PARADIGM

typedef std::set< CyPhyML::Components > ComponentsSet;
typedef std::set< CyPhyML::ComponentAssemblies > ComponentAssembliesSet;
typedef std::set< CyPhyML::ComponentAssembly > ComponentAssemblySet;
typedef std::set< CyPhyML::ComponentRef > ComponentRefSet;
typedef std::set< CyPhyML::Component > ComponentSet;


CyPhyML::Component getComponent( CyPhyML::RootFolder cyPhyMLParentObject, const StringVector &stringVector, StringVector::iterator stvItr );
CyPhyML::Component getComponent( CyPhyML::Components cyPhyMLParentObject, const StringVector &stringVector, StringVector::iterator stvItr );
CyPhyML::Component getComponent( CyPhyML::ComponentAssemblies cyPhyMLParentObject, const StringVector &stringVector, StringVector::iterator stvItr );
CyPhyML::Component getComponent( CyPhyML::ComponentAssembly cyPhyMLParentObject, const StringVector &stringVector, StringVector::iterator stvItr );
CyPhyML::Component getComponent( CyPhyML::ComponentRef cyPhyMLComponentRef, const StringVector &stringVector, StringVector::iterator stvItr );
CyPhyML::Component getComponent( CyPhyML::Component cyPhyMLComponent, const StringVector &stringVector, StringVector::iterator stvItr );

#define FINDDESCENDANT( CLASS )\
	{\
		typedef CLASS##Set CyPhyMLObjectSet;\
		CyPhyMLObjectSet cyPhyMLObjectSet = cyPhyMLParentObject.CLASS##_kind_children();\
		CyPhyMLObjectSet::iterator cosItr;\
		for( cosItr = cyPhyMLObjectSet.begin() ; cosItr != cyPhyMLObjectSet.end() ; ++cosItr ) {\
			if (  static_cast< std::string >( cosItr->name() ) == objectName  ) {\
				return getComponent( *cosItr, stringVector, stvItr );\
			}\
		}\
	}

CyPhyML::Component getComponent( CyPhyML::RootFolder cyPhyMLParentObject, const StringVector &stringVector, StringVector::iterator stvItr ) {

	if ( stvItr == stringVector.end() ) {
		std::cerr << "ERROR: Could not find a CyPhyML Component in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
		return CyPhyML::Component();
	}
	std::string objectName = *stvItr;

	FINDDESCENDANT( Components )
	FINDDESCENDANT( ComponentAssemblies )

	std::cerr << "ERROR: Could not find \"" << objectName << "\" in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
	return CyPhyML::Component();
}

CyPhyML::Component getComponent( CyPhyML::Components cyPhyMLParentObject, const StringVector &stringVector, StringVector::iterator stvItr ) {

	if ( ++stvItr == stringVector.end() ) {
		std::cerr << "ERROR: Could not find a CyPhyML Component in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
		return CyPhyML::Component();
	}
	std::string objectName = *stvItr;

	FINDDESCENDANT( Components )
	FINDDESCENDANT( ComponentAssemblies )
	FINDDESCENDANT( Component )

	std::cerr << "ERROR: Could not find \"" << objectName << "\" in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
	return CyPhyML::Component();
}

CyPhyML::Component getComponent( CyPhyML::ComponentAssemblies cyPhyMLParentObject, const StringVector &stringVector, StringVector::iterator stvItr ) {

	if ( ++stvItr == stringVector.end() ) {
		std::cerr << "ERROR: Could not find a CyPhyML Component in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
		return CyPhyML::Component();
	}
	std::string objectName = *stvItr;

	FINDDESCENDANT( ComponentAssemblies )
	FINDDESCENDANT( ComponentAssembly )

	std::cerr << "ERROR: Could not find \"" << objectName << "\" in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
	return CyPhyML::Component();
}

CyPhyML::Component getComponent( CyPhyML::ComponentAssembly cyPhyMLParentObject, const StringVector &stringVector, StringVector::iterator stvItr ) {

	if ( ++stvItr == stringVector.end() ) {
		std::cerr << "ERROR: Could not find a CyPhyML Component in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
		return CyPhyML::Component();
	}
	std::string objectName = *stvItr;

	FINDDESCENDANT( ComponentAssembly )
	FINDDESCENDANT( ComponentRef )
	FINDDESCENDANT( Component )

	std::cerr << "ERROR: Could not find \"" << objectName << "\" in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
	return CyPhyML::Component();
}

CyPhyML::Component getComponent( CyPhyML::ComponentRef cyPhyMLComponentRef, const StringVector &stringVector, StringVector::iterator stvItr ) {

	std::string objectName = *stvItr;

	CyPhyML::Component cyPhyMLComponent = CyPhyML::Component::Cast( cyPhyMLComponentRef.ref() );
	if ( cyPhyMLComponent == Udm::null ) {
		std::cerr << "ERROR: Null Component Reference \"" << *stvItr << "\" in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
		return CyPhyML::Component();
	}

	return getComponent( cyPhyMLComponent, stringVector, stvItr );
}

CyPhyML::Component getComponent( CyPhyML::Component cyPhyMLComponent, const StringVector &stringVector, StringVector::iterator stvItr ) {
	if ( ( stvItr + 1 ) != stringVector.end() ) {
		std::cerr << "WARNING:  CyPhyML Component found at \"" << *stvItr << "\" in path \"" << boost::join( stringVector, "/" ) << "\"" << std::endl;
	}
	return cyPhyMLComponent;
}

#endif


void constructModels( MatLabUdm::SmartPointer &matLabUdmParentSP );

#ifdef _WIN32
namespace {
std::string GetLastErrorText() {
	LPTSTR errorText = NULL;
	FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_IGNORE_INSERTS,
		NULL, GetLastError(), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		(LPTSTR)&errorText, 0, NULL);
	std::string ret;
	if (NULL != errorText) {
		ret = errorText;
		LocalFree(errorText);
	}
	return ret;
}
}
#endif


/***************************************************************************************/


int main( int argc, char *argv[] ) {
#ifdef _WIN32
	HMODULE libeng = LoadLibraryA("libeng.dll");
	HMODULE libmx = LoadLibraryA("libmx.dll");
	if (libeng == NULL || libmx == NULL)
	{
		std::string missing_dll;
		if (libeng == NULL)
			missing_dll = "libeng.dll";
		if (libmx == NULL)
			missing_dll = "libmx.dll";
		HMODULE loaded_datafile = LoadLibraryExA(missing_dll.c_str(), NULL, LOAD_LIBRARY_AS_DATAFILE);
#if defined(_M_AMD64)
		const char* arch = "Win64";
#elif defined(_M_IX86)
		const char* arch = "Win32";
#else
#error Unrecognized processor
#endif
		std::cerr << "Error: could not load " << missing_dll << ".\n";
		if (loaded_datafile == NULL) {
#pragma warning(push)
#pragma warning(disable: 4996)
			std::cerr << "Please confirm you have Matlab installed and " << missing_dll << " is on the %PATH%\n"
				<< "Current %PATH% is " << (getenv("PATH") ? getenv("PATH") : "empty") << std::endl;
#pragma warning(pop)
		} else {
			std::cerr << "Please confirm " << missing_dll << " is of the correct architecture. "
				"(This is " << arch << " MDL2MGA)" << std::endl;
		}
		return ERROR_MOD_NOT_FOUND;
	}
#endif /* WIN32 */

	int exit_status = 0;
	// Try to process the command line arguments
	if ( !ConfigKeeper::getSingleton().processCommandLineArguments( argc, argv ) ) {
		return 0;
	}
	// Get the input file name
	if (  !ConfigKeeper::getSingleton().getCount( "inputFile" )  ) {
		usage( argv[0] );
		return 1;
	}
	std::string matLabFilename = ConfigKeeper::getSingleton().getStringValue( "inputFile" );
	boost::filesystem::path matLabFile( matLabFilename );

#if BOOST_VERSION / 100 % 1000 < 46
	matLabFile = boost::filesystem::complete( matLabFile );
#else
	matLabFile = boost::filesystem::absolute( matLabFile );
#endif
	// CHECK IF INPUT .mdl FILE EXISTS
	if (  !boost::filesystem::exists( matLabFile )  ) {
		std::cerr << "ERROR: Could not access file \"" << matLabFile << "\"." << std::endl << std::endl;
		return 2;
	}

	boost::filesystem::path matLabFile_mdl2mga;
	try {
		matLabFile_mdl2mga = MatLabUdm::getTopLevelModelManager().registerModel( matLabFile );
	} catch (const boost::filesystem::filesystem_error& e) {
		std::cerr << "ERROR: " << e.what() << std::endl;
		return 2;
	}

#if BOOST_VERSION / 100 % 1000 < 46
	std::string udmFilename = ConfigKeeper::getSingleton().getCount( "outputFile" ) ?
	 ConfigKeeper::getSingleton().getStringValue( "outputFile" ) : boost::filesystem::change_extension( matLabFile, ".xml" ).string();
#else
	std::string udmFilename = ConfigKeeper::getSingleton().getCount( "outputFile" ) ?
	 ConfigKeeper::getSingleton().getStringValue( "outputFile" ) : boost::filesystem::change_extension( matLabFile, boost::filesystem::path( ".xml" ) ).string();
#endif

	if (  ConfigKeeper::getSingleton().getCount( "help" )  ) {
		usage( argv[0] );
		return 0;
	}

#if PARADIGM != CyPhyML_PARADIGM
	bool newUdmFile = false;
	if (  ConfigKeeper::getSingleton().getCount( "newfile" )  ) newUdmFile = true;
#endif

	std::string host = "";
	if (  ConfigKeeper::getSingleton().getCount( "host" )  ) {
		host = ConfigKeeper::getSingleton().getStringValue( "host" );
	}

	if (  ConfigKeeper::getSingleton().getCount( "toplevel" )  ) {
		MatLabUdm::onlyTopLevel = true;
	}

	DirectoryVector directoryVector;
	if (  ConfigKeeper::getSingleton().getCount( "libdir" )  ) {
		directoryVector = ConfigKeeper::getSingleton().getStringVector( "libdir" );
	}

	MfileVector mfileVector;
	if (  ConfigKeeper::getSingleton().getCount( "mfile" )  ) {
		mfileVector = ConfigKeeper::getSingleton().getStringVector( "mfile" );
		for( MfileVector::iterator mfvItr = mfileVector.begin() ; mfvItr != mfileVector.end() ; ++mfvItr ) {
			std::string &mfileName = *mfvItr;
			if (  mfileName.substr( mfileName.length() - 2 ) == ".m"  ) mfileName = mfileName.substr(  0, mfileName.length() - 2  );
		}
	}

#if PARADIGM == CyPhyML_PARADIGM
	if (  !ConfigKeeper::getSingleton().getCount( "path" )  ) {
		std::cerr << "For CyPhyML model, \"path <path-to-component>\" option MUST be specified." << std::endl;
		return 1;
	}

	std::string componentPath = ConfigKeeper::getSingleton().getStringValue( "path" );
	boost::trim( componentPath );
	if ( componentPath.empty() ) {
		std::cerr << "Path to component MUST NOT BE EMPTY." << std::endl;
		return 1;
	}
#endif

	// INITIALIZE THE DATA NETWORK
	try {
#if PARADIGM == CyPhyML_PARADIGM
		UdmEngine::get_singleton().open( udmFilename );
#else
		if ( newUdmFile ) UdmEngine::get_singleton().createNew( udmFilename );
		else              UdmEngine::get_singleton().open( udmFilename );
#endif

		// INITIALIZE MatLab ENGINE
		MatLabEngine::initGlobalEngine( host );

		if (  ConfigKeeper::getSingleton().getCount( "matlabLog" )  ) {
			std::string matLabLogFileName = ConfigKeeper::getSingleton().getStringValue( "matlabLog" );
			MatLabEngine::globalEngine().setLoggingEnabled( matLabLogFileName );
		}

		MatLabEngine::globalEngine().executeCommand( "clear all" );
		MatLabEngine::globalEngine().executeCommand( "sfpref( 'ignoreUnsafeTransitionActions', 1 )" );

		// ADD DIRECTORY OF INPUT FILE TO LIBRARY SEARCH PATH IN MATLAB
		MatLabEngine::globalEngine().executeCommand(  std::string( "addpath( '" ) + matLabFile.parent_path().string() + "' )"  );

		for( DirectoryVector::iterator drvItr = directoryVector.begin() ; drvItr != directoryVector.end() ; ++drvItr ) {
			boost::filesystem::path directory( *drvItr );
#if BOOST_VERSION / 100 % 1000 < 46
			MatLabEngine::globalEngine().executeCommand(  std::string( "addpath( '" ) + boost::filesystem::complete( directory ).string() + "' )"  );
#else
			MatLabEngine::globalEngine().executeCommand(  std::string( "addpath( '" ) + boost::filesystem::absolute( directory ).string() + "' )"  );
#endif
		}

		for( MfileVector::iterator mfvItr = mfileVector.begin() ; mfvItr != mfileVector.end() ; ++mfvItr ) {
			MatLabEngine::globalEngine().executeCommand( *mfvItr );			
		}

#ifdef _WIN32
		DWORD tmpSize = GetTempPath(0, 0);
		TCHAR* tmpPath = new TCHAR[tmpSize+1];
		if (GetTempPath(tmpSize+1, tmpPath) == 0) {
			delete[] tmpPath;
			std::string error = "Error calling GetTempPath: ";
			error += GetLastErrorText();
			throw std::runtime_error(error);
		}
		std::string cmd = "cd ";
		cmd += std::string("'") + tmpPath + "'";
		delete[] tmpPath;
		MatLabEngine::globalEngine().executeCommand( cmd );
#else
		MatLabEngine::globalEngine().executeCommand( "cd /tmp" );
#endif
		MatLabEngine::globalEngine().loadFile( matLabFile_mdl2mga.string() );
		MatLabEngine::globalEngine().executeCommand( "r = sfroot" );

		// TO ALLOW COMPILATION
		MatLabEngine::globalEngine().executeCommand( "r.SaveFormat = 'Structure'" );

#if BOOST_VERSION / 100 % 1000 < 46
		UdmEngine::get_singleton().setModelName( matLabFile.stem() );
#else
		UdmEngine::get_singleton().setModelName( matLabFile.stem().string() );
#endif

#if PARADIGM == CyPhyML_PARADIGM
		StringVector stringVector;
		boost::split(  stringVector, componentPath, boost::is_any_of( "/\\" )  );

		CyPhyML::RootFolder rootFolder = UdmEngine::get_singleton().getRootFolder();
		CyPhyML::Component rootComponent = getComponent( rootFolder, stringVector, stringVector.begin() );

		if ( rootComponent == Udm::null ) return 1;

		CyPhyML::SignalFlowModel signalFlowModel = CyPhyML::SignalFlowModel::Create( rootComponent );
		signalFlowModel.name() = "SignalFlowModel";
		UdmEngine::get_singleton().setDataflowParent( signalFlowModel );

		MatLabUdm::SmartPointer factoryDataflow = MatLabUdmRoot::Factory().create( UdmEngine::get_singleton().getRootComponent() );
#else
		MatLabUdm::SmartPointer factoryDataflow = MatLabUdmRoot::Factory().create( UdmEngine::get_singleton().getDataflow() );
#endif
//		MatLabEngine::globalEngine().executeCommand( "root_diagram = r.find('-depth', 1, 'name', '" + MatLabEngine::globalEngine().getModelName() + "')" );
//		MatLabEngine::globalEngine().executeCommand("root_diagram.ModelWorkspace.save('mdl2mga_modelworkspace.mat')");
//		MatLabEngine::globalEngine().executeCommand("load('mdl2mga_modelworkspace.mat')");

		// It seems masks' boolean parameters are initialized to 'off' or 'on'. create aliases
//		MatLabEngine::globalEngine().executeCommand("off = false");
//		MatLabEngine::globalEngine().executeCommand("on = true");

		constructModels( factoryDataflow );

		MatLabUdmTransition::getTransitionParents().buildTransitions();

		// TYPE DETERMINATION
		try {
			MatLabUdm::getTopLevelModelManager().saveModels();
			MatLabUdm::getTopLevelModelManager().compileModels();
			MatLabUdmData::getAllDataTypes();
			MatLabUdmPort::getAllPortTypes();
			MatLabUdmBlock::getAllPortTypes();
			MatLabUdmBlock::getAllSampleTimes();
			MatLabUdm::getTopLevelModelManager().stopCompileModels();
			MatLabUdm::getTopLevelModelManager().saveModels();
		} catch( ... ) {
			std::cerr << "CAUGHT EXCEPTION ACQUIRING TYPES!!" << std::endl;
			MatLabUdm::getTopLevelModelManager().stopCompileModels();
			MatLabUdm::getTopLevelModelManager().saveModels();
			throw;
		}

		MatLabUdmBlock::kludgeMFileSFuncs();

		MatLabUdmBlock::processGotoFromBlocks();

		MatLabUdmBlock::propagateTypes();
		MatLabUdmBlock::propagateArchetypeOutPorts();

		MatLabUdmBlock::processCollectors();

		MatLabUdmBlock::checkTypes();

#if PARADIGM == CyPhyML_PARADIGM
		SubsystemSet subsystemSet = signalFlowModel.SignalFlow_Subsystem_kind_children();
		SLSF::Subsystem subsystem;
		for( SubsystemSet::iterator sssItr = subsystemSet.begin() ; sssItr != subsystemSet.end() ; ++sssItr ) {
			if ( !sssItr->HasObjectInstances() ) {
				subsystem = *sssItr;
				break;
			}
		}

		if ( subsystem == Udm::null ) {
			std::cerr << "ERROR: Could not find Subsystem to link to SignalFlowModel!" << std::endl;
		} else {
			InPortSet inPortSet = subsystem.InPort_kind_children();
			for( InPortSet::iterator ipsItr = inPortSet.begin() ; ipsItr != inPortSet.end() ; ++ipsItr ) {
				SLSF_ROOT::InSignal inSignal = SLSF_ROOT::InSignal::Create( signalFlowModel );
				SLSF_ROOT::IOSignal2InPort ioSignal2InPort = SLSF_ROOT::IOSignal2InPort::Create( signalFlowModel );
				ioSignal2InPort.srcIOSignal2InPort_end() = inSignal;
				ioSignal2InPort.dstIOSignal2InPort_end() = *ipsItr;
			}
			OutPortSet OutPortSet = subsystem.OutPort_kind_children();
			for( OutPortSet::iterator opsItr = OutPortSet.begin() ; opsItr != OutPortSet.end() ; ++opsItr ) {
				SLSF_ROOT::OutSignal outSignal = SLSF_ROOT::OutSignal::Create( signalFlowModel );
				SLSF_ROOT::OutPort2IOSignal outPort2IOSignal = SLSF_ROOT::OutPort2IOSignal::Create( signalFlowModel );
				outPort2IOSignal.srcOutPort2IOSignal_end() = *opsItr;
				outPort2IOSignal.dstOutPort2IOSignal_end() = outSignal;
			}
		}
#endif
	} catch ( udm_exception &err ) {

		std::cerr << "The following UDM exception occurred:" << std::endl;
		std::cerr << err.what() << std::endl << std::endl;

		int length = udmFilename.length();
		if (  udmFilename.substr( length - 3, length ) == "mga"  ) {
			std::cerr << "As you are using the GME backend of UDM, please make sure that the" << std::endl;
			std::cerr << "ESMoL paradigm is properly registered in GME." << std::endl << std::endl;
		}
		exit_status = 1;

	} catch ( MatLabUdm::Exception &err ) {

		std::cerr << "The following MatLab exception occurred:" << std::endl;
		std::cerr << err.what() << std::endl << std::endl;

		exit_status = 2;

	} catch ( std::exception &err ) {

		std::cerr << "The following std::exception occurred:" << std::endl;
		std::cerr << err.what() << std::endl << std::endl;
		exit_status = 3;

	} catch ( ... ) {

		std::cerr << "Unknown exception caught" << std::endl;
		exit_status = 4;
	}

	try {

#if defined(LINUX) && defined(DEVEL)
//		MatLabEngine::globalEngine().executeInteractiveCommands();
#endif
		UdmEngine::close();
		MatLabUdm::getTopLevelModelManager().deleteModels();

	} catch ( udm_exception &err ) {

		std::cerr << "The following UDM exception occurred on closing the UDM file \"" << udmFilename << "\":" << std::endl;
		std::cerr << err.what() << std::endl << std::endl;

		exit_status = 5;

	} catch( ... ) {

		std::cerr << "Unknown exception caught on close of UDM file \"" << udmFilename << "\"." << std::endl;
		exit_status = 6;
	}

	return exit_status;
}


void usage( const std::string &exeName ) {

	std::string::size_type lastForwardSlash = exeName.rfind("/");
//	std::cerr << "lastForwardSlash = " << lastForwardSlash << std::endl;
	std::string::size_type lastBackSlash = exeName.rfind("\\");
//	std::cerr << "lastBackSlash = " << lastBackSlash << std::endl;

	std::string::size_type lastSlash = lastForwardSlash == std::string::npos ? lastBackSlash : lastForwardSlash;
	std::string baseName = exeName.substr( lastSlash + 1 );

	std::cerr << "Usage:" << std::endl;
	std::cerr << ConfigKeeper::getSingleton().getDescription();
	std::cerr << baseName << " [options] <mdl-file> [ <mga-file> | <xml-file> ]" << std::endl << std::endl;
	std::cerr << "Required arguments:" << std::endl;
	std::cerr << "  <mdl-file> is the name of a MatLab .mdl file." << std::endl << std::endl;
	std::cerr << "  If the second argument is not specified, it defaults to a file that" << std::endl;
	std::cerr << "   has the same name as the .mdl file, but ending in .xml -- see below" << std::endl;
	std::cerr << "   for a description of this argument." << std::endl << std::endl;
	std::cerr << "Optional arguments:" << std::endl;
	std::cerr << "  <mga-file> is the name of an GME-based .mga model file that will be" << std::endl;
	std::cerr << "   created or overwritten, and will conform to the ESMoL paradigm in GME." << std::endl << std::endl;
	std::cerr << "  <xml-file> is the name of an XML-based (.xml) model file that will be" << std::endl;
	std::cerr << "   created or overwritten, and will conform to the ESMoL paradigm as" << std::endl;
	std::cerr << "   specified in the ESMoL.xsd XML schema file.  This schema file is embedded" << std::endl;
	std::cerr << "   in this program and so does not have to be explicitly specified." << std::endl << std::endl;
	std::cerr << "  <directory> is the name of a directory that is given to MatLab to search for" << std::endl;
	std::cerr << "   SL/SF library files.  Such library files may be used by the <mdl-file> in" << std::endl;
	std::cerr << "   specification of its SL/SF model.  As many of these directories may be" << std::endl;
	std::cerr << "   specified as needed, though each with its own '-L' option specifier." << std::endl;
	std::cerr << "   NOTE:  the directory in which the <mdl-file> resides is always implicitly" << std::endl;
	std::cerr << "   specified by this program as a library search path to MatLab." << std::endl << std::endl;
	std::cerr << "  <host> is the hostname or ip-address of a machine with a MatLab installation." << std::endl;
	std::cerr << "   MDL2MGA will direct this machine to process the <mdl-file>, and will then query" << std::endl;
	std::cerr << "   this same machine about the model therein.  By default, MDL2MGA uses the local machine." << std::endl << std::endl;
}

void constructModels( MatLabUdm::SmartPointer &matLabUdmParentSP ) {

	MatLabUdm::SPVector recurseVector;

	// GET MODELS
	recurseVector = MatLabUdm::buildUdm( matLabUdmParentSP, MatLabUdmModel::Factory() );
	for( MatLabUdm::SPVector::iterator spvItr = recurseVector.begin() ; spvItr != recurseVector.end() ; ++spvItr ) {
		MatLabUdmBlock::constructBlocks( *spvItr );
		(*spvItr)->cleanup();
	}

}


/***************************************************************************************/

