function [results,reference,T,X,Y] = esmol_test(model, basedir)
%
%[results,reference, T,X,Y] = test_esmol(model, basedir);
%[results,reference, T,X,Y] = test_esmol(model);
%
%   Given a Simulink model containing some number of top-level output ports,
%   compares the results computed between a native Simulink simulation and
%   an execution instance of the actual source code generated by the ESMOL
%   tool chain.
%
%   $model   - name of the simulink model (not the name of the .mdl file) 
%   $basedir - directory in which the above model resides; optional if 
%              named the same as the model and under the current 
%              subdirectory (e.g. ./sl_basic/, containing the sl_basic.mdl 
%              file)
%

%TODO: replace all '\' & '\\' strings with '/'
    global directoryStack;
    global vcp_path;
    
    directoryStack = {};
    vcp_path = getenv( 'VCP_PATH' );

%    setenv(  'PATH', [ '/usr/local/MATLAB/R2011a/bin', ':', getenv( 'PATH' ) ]  );
%    setenv( 'LD_LIBRARY_PATH', [ getenv( 'LD_LIBRARY_PATH' ), ':', vcp_path, '/lib' ] )

    addpath( pwd );
    testintermediate = true;
    makegoldfiles = true;

    script     = which('esmol_test');
    global scriptdir;
    scriptdir  = script(1:length(script)-12);    
    BIN_DIR    = [scriptdir, '../../bin'];        % where are the ESMOL binaries?
    %clc;
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    
    % Create a temporary directory to hold all of our artifacts
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
    if ( length(model)>4 && strcmp('.mdl', model(length(model)-3:length(model))))
        model = model(1:length(model)-4);  % not supposed to contain '.mdl' 
    end
    if (~exist('basedir')) 
        basedir = model;    % second param is optional, given naming convention
    end
    pushd      ( toString(basedir) );    
    OUTPUT_DIR = [pwd,'/esmol_test'];
    outputdir = toString(OUTPUT_DIR);
    goldendir = sprintf('%sGOLDEN',outputdir);
    clean();
    
    system(['mkdir -p "', OUTPUT_DIR, '"' ]);
    cd ( outputdir );
    if ( makegoldfiles )
        system(['mkdir -p "', goldendir, '"']);
    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    
    %% Toolchain stages:  mdl2mga | addtypes | codegen
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    disp 'Translating to VCP ... '
    cmdline = sprintf('"%s/MDL2MGA" -n "../%s.mdl" "%s.xml"', BIN_DIR,model,model);
    assert_command(cmdline);
    %cmdline = sprintf('%s/MDL2MGA ../%s.mdl %s.mga', BIN_DIR,model,model);
    %assert_command(cmdline);
    
    if (makegoldfiles)
        cmdline = sprintf('cp "%s.xml" "%s/%sMDL2MGA.xml"',model,goldendir,model);
        system(cmdline);
    end
%    if (testintermediate)
%        disp 'Checking MDL2MGA output ... '
%        cd(scriptdir);
%        cmdline = sprintf('UdmModelDiff "%s/%s.xml" "%sGOLDEN/%sMDL2MGA.xml" ESMoL_udm.xml', outputdir,model,outputdir,model);
%        dos(cmdline); %UdmModelDiff sometimes fails on identical files, so don't assert for now
%        cd(outputdir);
%    end
        
    disp 'Translating to SFC ...'
    cmdline = sprintf('"%s/SL_CodeGen" "-L" ".." "%s.xml"', BIN_DIR,model);
    assert_command(cmdline);   
    cmdline = sprintf('"%s/SF_CodeGen" "%s.xml"', BIN_DIR,model);
    assert_command(cmdline);
    
    newDirectory = [ 'New', model, '00000000' ];
    
    if (makegoldfiles)
        cmdline = sprintf('cp "%s/%s_SFC.xml" "%s/"',newDirectory,newDirectory,goldendir);
        system( cmdline );
    end
%    if (testintermediate)
%        disp 'Checking SL_CodeGen.exe output ... '
%        cd(scriptdir);
%        cmdline = sprintf('UdmModelDiff "%s/%s/%s_SFC.xml" "%sGOLDEN/%s_SFC.xml" SFC_udm.xml', outputdir,newDirectory,model,outputdir,model);
%        dos(cmdline);
%        cd(outputdir);
%    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    
    %% Auto-generate main() routine
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%      
    disp 'Generating source code ...'
    %../../bin/ctestgen 'FileType="sl_basic_SFC.xml";"output.xml"
    %!"SFC.xsd"'
    pushd( newDirectory );
    
    cmdline = sprintf('"%s/ctestgen" "FileType=New%s00000000_SFC.xml;%s_main.xml"', BIN_DIR,model,model);
    assert_command(cmdline);  
    cmdline = sprintf('"%s/SFCPrinter" "%s_main.xml"', BIN_DIR,model);
    assert_command(cmdline);
    cmdline = sprintf('"%s/SFCPrinter" -j "%s_main.xml"', BIN_DIR,model);
    assert_command(cmdline);
    file = dir(  sprintf( 'New%s00000000_*_SFC.xml', model )  );
    nofiles = size( file, 1 );
    for ix = 1:nofiles
        cmdline = sprintf( '"%s/SFCPrinter" -j "%s"', BIN_DIR, file(ix).name );
        assert_command( cmdline );
    end
        
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    
    %% Compile & Execute
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    
    disp 'Compiling C ...'
    compile();
    

    disp 'Compiling Java ...'
    jcompile(model);
    
    disp 'Simulations ...'
    % MATLAB Simulation
    addpath ../..
    myopts = simset( 'SaveFormat', 'StructureWithTime' );
    [T,X,S] = sim(  toString( model ), 10, myopts  );
    Y = extractData( S );

    reference = Y;  %WAS: [T,Y];

    save simulation_matlab Y
    rmpath  ../..
    
    % ESMOL "C" Simulation
    !./esmol > c_simulation_esmol.csv
    c_results = load( 'c_simulation_esmol.csv' );

    % ESMOL "Java" Simulation
    cmdline = sprintf( '"java" "%s" > java_simulation_esmol.csv', model );
    system( cmdline );
    cmdline = sprintf( '"%s/TestOutputTranslator" java_simulation_esmol.csv', BIN_DIR );
    system( cmdline );
    java_results = load( 'java_simulation_esmol.csv' );
    
    popd();
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %% Compare Results
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%    

    tolerances = esmol_tolerances();
    tolerance = 0.0;
    NUM_TOL = size(tolerances,1);
    for i=1:NUM_TOL
        tol_model_file = char(  tolerances( i, 1 )  );
        [path,tol_model] = fileparts( tol_model_file );    % the .mdl extension is optional
        if strcmp( tol_model, model )
            tolerance = tolerances{ i, 2 }(1);
            break;
        end
    end

    disp 'Verifying "C" results ...';
    FAILED = verifyResults( c_results, reference, tolerance );
    if ~FAILED
        disp 'Verifying "Java" results ...';
        FAILED = verifyResults( java_results, reference, tolerance );
    end

    popd();
    assert(~FAILED);    
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Utility Functions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function dos(cmd)  
  eval(sprintf('!%s',cmd));
end
function assert_command(cmd)
    disp(['    executing: ', cmd ]);
    [err,stdout] = system(cmd);
    if(err ~= 0 && err ~= 139)
        disp(['ERROR(',num2str(err),'): `', cmd, '` failed']);
        stdout
        disp ' '
        disp 'FAILURE !!!'
        popalld()
        error(['ERROR(',num2str(err),'): `', cmd, '` failed'])
    end
end
function del(file)
  cmd = sprintf('rm -rf "%s"', file);
  dos(cmd)
end
function result = toString(var)
  result = sprintf('%s',var);
end
function pushd(dir)
  global directoryStack;
  directoryStack{  length( directoryStack ) + 1  } = toString( pwd );
  cd(  toString( dir )  );
end
function popd() 
  global directoryStack;
  directory = directoryStack{  length( directoryStack )  };
  cd(  toString( directory )  );
  directoryStack = {   directoryStack{  1 : length( directoryStack ) - 1  }   };
end
function popalld() 
  global directoryStack;
  if (  length( directoryStack ) >= 1  )
    cd( directoryStack{1} );
    directoryStack = {};
  end
end
function result = sameDimensions(X,Y)
    result = sum(size(X)~=size(Y)) == 0;
end


function clean()
    disp ('Cleaning up [previous] temporary files ...');
    % Convoluted way, which avoids some noisy err msgs for the user
    files = dir;
    for i= 1:length(files)
        if (strcmp(files(i).name,'esmol_test'))
            !rm -rf esmol_test
        end
    end
end

function compile()
    % TODO: the SF-code-generator automatically creates unused SFunc
    % wrapper code
    global vcp_path;
    !rm -f sfunc_wrapper_*.c 
    CC  = 'gcc -m32 -lm -o esmol';
    SRC = ' *.c';
    INCLUDES = sprintf( ' -I"%s/tests/SLSF_CodeGen"', vcp_path );
    assert_command([CC, INCLUDES, SRC]);
end

function jcompile(model)
    
    JC  = 'javac *.java'; %sprintf('"%%JDK_PATH%%\\bin\\javac" %s_class.java',model);
    disp(JC)
    assert_command(JC);
end

function Y = extractData( S )
    Y = [];

    % "S" MUST BE A STRUCT WITH 2 MEMBERS:
    %  time    : AN ARRAY OF TIME VALUES OVER WHICH THE SIMULATION TOOK PLACE
    %  signals : OUTPUT SIGNALS OF SIMULATION

    if isstruct( S )

        TimeDim = prod(  size( S.time )  );  % GET NO. OF SIMULATION STEPS
        Sig = S.signals();                   % GET OUTPUT SIGNALS
        SigDim = prod(  size( Sig )  );      % GET NO. OF OUTPUT SIGNALS

        for ix = 1:TimeDim
            Y1 = [];
            for jx = 1:SigDim
                Dims = Sig( jx ).dimensions;
                Values = Sig( jx ).values;
                Dims_size = prod(  size( Dims )  );
                if Dims_size > 1
                    Values_ndims = prod(   size(  size( Values )  )   );
                    permMatrix = [ Values_ndims , 1:Values_ndims-1 ];
                    Values = permute( Values, permMatrix );
                end
                Y1 = [ Y1 , double(  Values( ix, : )  ) ];
            end
            Y = [ Y ; Y1 ];
        end
    end

    Y = double( Y );
end

function FAILED = verifyResults( results, reference, tolerance )
    if ( ~sameDimensions(results,reference) ) 
        FAILED = true;
    else    
        [rows,cols] = size(reference);
        FAILED = ~( ...
         sum(sum( ...
          abs(results - reference) <= tolerance | ...
          (  isnan( results ) & isnan( reference )  ) | ...
          (  results < 0 & isinf( results ) & reference < 0 & isinf( reference )  ) | ...
          (  results > 0 & isinf( results ) & reference > 0 & isinf( reference )  ) ...
         )) == ...
         rows*cols ...
        );
    end
    
    if (FAILED)
        disp ' '
        disp 'MATLAB:'
        disp '============================================================'
        reference'
        disp 'ESMoL:'
        disp '============================================================'        
        results'
        disp ' '        
        disp 'MATLAB-ESMoL:'
        disp '============================================================'   
        if (size(reference) == size(results))
            reference' - results'
        else
            disp 'simulations returned results of dissimilar dimensions'
        end        
        disp ' '        
        disp 'FAILURE !!!'        
    else
        cd ..
        % clean();
        disp 'SUCCESS !!!'    
        disp ' '
    end
end