﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using AVM;
using Newtonsoft.Json;
using System.Xml.Serialization;

namespace AVM.ComponentUtility
{
    class Program
    {
        static void Main(string[] args)
        {
            // Check for READ mode
            if (args.Length == 2 && args[0] == "-r")
            {
                #region Read a component model
                Console.Out.WriteLine("READ MODE: Print contents of component model " + args[1]);

                // Deserialize into an instance of an AVM.Component class
                AVM.Component comp = AVM.Component.DeserializeFromFile(args[1]);

                // Iterate over the FILES that are included in the component package
                Console.Out.WriteLine("\n=== FILES ===");
                foreach (AVM.File f in comp.Files)
                {
                    String s_output = String.Format("{0}: {1}", f.Location, f.Description);
                    Console.Out.WriteLine(s_output);
                }

                // Iterate over the NAMED VALUES that are not parametric
                Console.Out.WriteLine("\n=== NAMED VALUES: Properties ===");
                foreach (AVM.META.NamedValue nv in comp.Features.Where(nv =>
                    (nv is AVM.META.NamedValue &&
                    (nv as AVM.META.NamedValue).IsParameter == false)))
                {
                    String s_output = String.Format("{0}: {1} {2}", nv.Name, nv.Value, nv.Unit);
                    Console.Out.WriteLine(s_output);
                }

                // Iterate over the NAMED VALUES that are parametric
                Console.Out.WriteLine("\n=== NAMED VALUES: Parameters ===");
                foreach (AVM.META.NamedValue nv in comp.Features.Where(nv =>
                    (nv is AVM.META.NamedValue &&
                    (nv as AVM.META.NamedValue).IsParameter == true)))
                {
                    String s_output = String.Format("{0}: {1} {2}", nv.Name, nv.ValidRange, nv.Unit);
                    Console.Out.WriteLine(s_output);
                }

                // Iterate over the Component-external POWER PORTS
                Console.Out.WriteLine("\n=== POWER PORTS ===");
                foreach (AVM.META.PowerPort pp in comp.Features.Where(pp => (pp is AVM.META.PowerPort)))
                {
                    String s_output = String.Format("{0}: {1}", pp.Name, pp.GetType().ToString());
                    Console.Out.WriteLine(s_output);
                }

                // Iterate over the ASSOCIATION objects
                Console.Out.WriteLine("\n=== ASSOCIATIONS ===");
                foreach (AVM.Association assoc in comp.Associations)
                {
                    String s_srcName = "";
                    String s_dstName = "";

                    Boolean b_FoundBothEnds = true;
                    // The NAME attribute is not defined at the ASSOCIABLE level, but at the class level, so use reflection to get this property.
                    try
                    {
                        Type t_src = assoc.SrcAssociable.GetType();
                        PropertyInfo pi_src = t_src.GetProperty("Name");
                        object o_srcVal = pi_src.GetValue(assoc.SrcAssociable, null);
                        s_srcName = o_srcVal.ToString();
                    }
                    catch
                    {
                        b_FoundBothEnds = false;
                    }
                    try
                    {
                        Type t_dst = assoc.DstAssociable.GetType();
                        PropertyInfo pi_dst = t_dst.GetProperty("Name");
                        object o_dstVal = pi_dst.GetValue(assoc.DstAssociable, null);
                        s_dstName = o_dstVal.ToString();
                    }
                    catch
                    {
                        b_FoundBothEnds = false;
                    }

                    if (b_FoundBothEnds)
                    {
                        String s_output = String.Format("{0} ({1}) to {2} ({3})", s_srcName, assoc.SrcAssociable.GetType(), s_dstName, assoc.DstAssociable.GetType());
                        Console.Out.WriteLine(s_output);
                    }
                }
                #endregion
            }
            else if (args.Length == 2 && args[0] == "-w")
            {
                #region Write a component example
                Console.Out.WriteLine("WRITE MODE: Write example component model to " + args[1]);

                // Create a new instance of a COMPONENT, and initialize the primary arrays.
                AVM.Component ac_new = new AVM.Component();
                ac_new.Name = "Example Component";
                ac_new.AVMID = "{2B4346AC-AF8C-4EB4-BB34-18723DC38F16}";

                // Create a NAMED VALUE for "Weight"
                AVM.META.NamedValue nv_weight = new AVM.META.NamedValue();
                nv_weight.Name = "Weight";
                nv_weight.Value = "2.23";
                nv_weight.Unit = "kg";
                nv_weight.Type = "AVM.META.NamedValues.Weight";
                nv_weight.DistributionType = null;
                nv_weight.id = "nv0";
                ac_new.Features.AddItem(nv_weight);

                // Create a NAMED VALUE for "Volume". If there is some uncertainty, we can use a distribution.
                AVM.META.NamedValue nv_volume = new AVM.META.NamedValue();
                nv_volume.Name = "Volume";
                nv_volume.Value = null;
                nv_volume.Unit = "m^3";
                nv_volume.Type = "AVM.META.NamedValues.Volume";
                nv_volume.DistributionType = DistributionTypeEnum.Uniform;
                nv_volume.DistributionParameters = "6 7.2";
                nv_volume.id = "nv1";
                ac_new.Features.AddItem(nv_volume);

                // Create a FILE class for the Behavior Model
                AVM.File f_modelicaModel = new AVM.File();
                f_modelicaModel.Location = "modelica/behavior/main.mo";
                f_modelicaModel.Description = "The Modelica behavior model for this component.";
                ac_new.Files.AddItem(f_modelicaModel);

                // Create a USESLIBRARY class for a library dependency
                AVM.UsesLibrary ul_modelicaLibrary = new AVM.UsesLibrary();
                ul_modelicaLibrary.id = "234";
                ul_modelicaLibrary.Path = "http://location.com/archive.zip";
                ul_modelicaLibrary.Revision = 3;
                ul_modelicaLibrary.Version = "2.3a_preview";
                ac_new.Features.AddItem(ul_modelicaLibrary);

                // Create a BEHAVIOR MODEL
                AVM.META.BehaviorModel bm_modelicaModel = new AVM.META.BehaviorModel();
                bm_modelicaModel.Name = "Modelica Behavior Model";
                bm_modelicaModel.Type = "AVM.META.BehaviorModel";
                bm_modelicaModel.Location = f_modelicaModel.Location;
                ac_new.Features.AddItem(bm_modelicaModel);

                // Create a BEHAVIOR PARAMETER for the BEHAVIOR MODEL
                AVM.META.BehaviorParameter bp_weight = new AVM.META.BehaviorParameter();
                bp_weight.Name = "weight";
                bp_weight.Unit = "kg";
                bm_modelicaModel.Interfaces.AddItem(bp_weight);

                // Create an ASSOCIATION between the Component-level Named Value and the Behavior Model's Parameter
                AVM.Association a_weightMap = new AVM.Association();
                a_weightMap.SrcAssociable = nv_weight;
                a_weightMap.DstAssociable = bp_weight;
                ac_new.Associations.AddItem(a_weightMap);
                a_weightMap.Note = "Map component weight into Behavior Model";
                if (nv_weight.Value != null)
                    bp_weight.Value = nv_weight.Value.ToString();

                // Create an ELECTRICAL power port at the Component level
                AVM.META.Electrical e_powerOut = new AVM.META.Electrical();
                e_powerOut.Name = "Electrical Power Out";
                e_powerOut.id = "epp0";
                ac_new.Features.AddItem(e_powerOut);

                // Create an POWER PORT INTERFACE in the Behavior Model
                AVM.META.PowerPortInterface ppi_powerOut = new AVM.META.PowerPortInterface();
                ppi_powerOut.Name = "Electrical Power Out";
                ppi_powerOut.PowerPortType = AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaElectrical;
                bm_modelicaModel.Interfaces.AddItem(ppi_powerOut);

                // Create an ASSOCIATION mapping the Behavior Model's Power Port to the Component-level Power Port
                AVM.Association a_powerMap = new AVM.Association();
                a_powerMap.SrcAssociable = ppi_powerOut;
                a_powerMap.DstAssociable = e_powerOut;
                ac_new.Associations.AddItem(a_powerMap);
                a_powerMap.Note = "Map electrical power output";
                
                // Add a hierarhical port (unconnected to behavior model for now)
                AVM.META.AggregatePort ap_NEMA115 = new AVM.META.AggregatePort();
                ap_NEMA115.Description = "Two-prong electrical power supply";
                ap_NEMA115.Name = "Power Supply";
                ap_NEMA115.Type = "typesdatabase://hierarchicalports/nema115";
                ac_new.Features.AddItem(ap_NEMA115);

                AVM.META.Electrical e_NEMA_neutral = new AVM.META.Electrical();
                e_NEMA_neutral.Name = "Neutral";
                ap_NEMA115.AggregatedPorts.AddItem(e_NEMA_neutral);

                AVM.META.Electrical e_NEMA_live = new AVM.META.Electrical();
                e_NEMA_live.Name = "Live";
                ap_NEMA115.AggregatedPorts.AddItem(e_NEMA_live);

                AVM.StructuralInterface si_NEMA = new AVM.StructuralInterface();
                si_NEMA.Name = "Socket";
                si_NEMA.Type = "typesdatabase://structuralinterfaces/nema115";
                si_NEMA.Role = "Socket";
                ap_NEMA115.AggregatedPorts.AddItem(si_NEMA);

                AVM.StructuralInterfaceDatum sid_NEMA_axis1 = new AVM.StructuralInterfaceDatum();
                sid_NEMA_axis1.DatumType = AVM.DatumTypeEnum.Axis;
                sid_NEMA_axis1.Name = "Axis1";
                si_NEMA.Datums.AddItem(sid_NEMA_axis1);

                AVM.StructuralInterfaceDatum sid_NEMA_axis2 = new AVM.StructuralInterfaceDatum();
                sid_NEMA_axis2.DatumType = AVM.DatumTypeEnum.Axis;
                sid_NEMA_axis2.Name = "Axis2";
                si_NEMA.Datums.AddItem(sid_NEMA_axis2);

                AVM.StructuralInterfaceDatum sid_NEMA_surface = new AVM.StructuralInterfaceDatum();
                sid_NEMA_surface.DatumType = AVM.DatumTypeEnum.Surface;
                sid_NEMA_surface.Name = "Surface";
                si_NEMA.Datums.AddItem(sid_NEMA_surface);

                // Serialize the object to file, and write to console as well
                string jsonString = ac_new.SerializeToFile(args[1]);
                Console.Out.WriteLine(jsonString);
                #endregion
            }
            else if ((args.Length >= 2) && args[0] == "-tryparse")
            {
                // For each file, try to open it, and report if it was successful or had an exception
                foreach (string arg in args)
                {
                    if (arg == "-tryparse")
                        continue;

                    try
                    {
                        if (Path.GetExtension(arg) == ".component.json")
                        {
                            AVM.Component.DeserializeFromFile(arg);
                            Console.Out.WriteLine("{0} OK", arg);
                        }
                        else if (Path.GetExtension(arg) == ".json")
                        {
                            AVM.Component.DeserializeFromFile(arg);
                            Console.Out.WriteLine("{0} OK", arg);
                        }
                        else
                            Console.Out.WriteLine("{0} FAILED: {1}", arg, "Unsupported file type");
                    }
                    catch (Exception ex)
                    {
                        Console.Out.WriteLine("{0} FAILED: {1}", arg, ex.Message);
                    }
                }
            }
            else if ((args.Length >= 2) && args[0] == "-toxml")
            {
                // For each JSON file, open it, and then write an XML version to the same folder.
                foreach (string p_JSONFile in args.Skip<string>(1))
                {
                    AVM.Component ac = AVM.Component.DeserializeFromFile(p_JSONFile);

                    string p_XMLFile = Path.GetDirectoryName(p_JSONFile) + "\\" + Path.GetFileNameWithoutExtension(p_JSONFile) + ".xml";

                    FileStream fs = new FileStream(p_XMLFile,System.IO.FileMode.Create);
                    XmlSerializer xs = new XmlSerializer(typeof(AVM.Component));
                    xs.Serialize(fs, ac);
                    fs.Close();
                }
            }
            else
            {
                // Display USAGE info
                Console.Out.WriteLine("Component Model API demonstration");
                Console.Out.WriteLine("USAGE: " + System.AppDomain.CurrentDomain.FriendlyName + " -r (path of JSON component model to read)");
                Console.Out.WriteLine("USAGE: " + System.AppDomain.CurrentDomain.FriendlyName + " -w (path to write JSON component example)");
            }
        }
    }
}
