﻿using System;
using System.Globalization;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace BuildAVMComponentJavaLib {

    public enum Accessibility {
        PUBLIC, PROTECTED, PRIVATE
    };

    public class AVMJavaTypeMap {
        private Dictionary< String, String > _map = new Dictionary<string,string>();


        public static AVMJavaTypeMap _singleton = new AVMJavaTypeMap();

    }

    public class JavaEntity {
        static protected Dictionary<Accessibility, string> _accessibilityStringMap = new Dictionary<Accessibility, string>();
        static JavaEntity() {
            _accessibilityStringMap[ Accessibility.PRIVATE ]   = "private";
            _accessibilityStringMap[ Accessibility.PROTECTED ] = "protected";
            _accessibilityStringMap[ Accessibility.PUBLIC ]    = "public";
        }
        public string _name { get; set; }
        public string _namespace { get; set; }

        public string _fullName {
            get { return _namespace == "" ? _name : _namespace + "." + _name; }
        }
    }

    public class JavaEntityComparer : IComparer< JavaEntity > {
        public int Compare( JavaEntity javaEntity1, JavaEntity javaEntity2 ) {
            return javaEntity1._name.CompareTo( javaEntity2._name );
        }
    }

    public class JavaType : JavaEntity {
        private static Dictionary<string, JavaType> _avmJavaTypeMap = new Dictionary<string, JavaType>();
        private static HashSet<string> _javaObjectBaseTypeSet = new HashSet<string>();
        private static Dictionary<string, string> _javaBaseToObjectMap = new Dictionary<string, string>();
        private static Dictionary<string, string> _javaBaseDefaultValueMap = new Dictionary<string,string>();
        private static Dictionary<string, string> _pythonBaseDefaultValueMap = new Dictionary<string, string>();

        static JavaType() {
            _avmJavaTypeMap.Add( "AVMID", new JavaType("String") );
            _avmJavaTypeMap.Add( "Boolean", new JavaType("boolean") );
            _avmJavaTypeMap.Add( "Date", new JavaType("String") );
            _avmJavaTypeMap.Add( "DimType", new JavaType( "String" ) );
            _avmJavaTypeMap.Add( "Double", new JavaType( "double" ) );
            _avmJavaTypeMap.Add( "DistributionParameterType", new JavaType("String") );
            _avmJavaTypeMap.Add( "HashType", new JavaType("String") );
            _avmJavaTypeMap.Add( "ID", new JavaType("String") );
            _avmJavaTypeMap.Add( "Integer", new JavaType("int") );
            _avmJavaTypeMap.Add( "Path", new JavaType("String") );
            _avmJavaTypeMap.Add( "RangeType", new JavaType("String") );
            _avmJavaTypeMap.Add( "Real", new JavaType("double") );
            _avmJavaTypeMap.Add( "String", new JavaType("String") );
            _avmJavaTypeMap.Add( "String[*]", new JavaType("String", true) );
            _avmJavaTypeMap.Add( "UnitType", new JavaType("String") );
            _avmJavaTypeMap.Add( "URI", new JavaType("String") );
            _avmJavaTypeMap.Add( "XML", new JavaType("String") );

            _javaObjectBaseTypeSet.Add( "Boolean" );
            _javaObjectBaseTypeSet.Add( "Byte" );
            _javaObjectBaseTypeSet.Add( "Character" );
            _javaObjectBaseTypeSet.Add( "Short" );
            _javaObjectBaseTypeSet.Add( "Integer" );
            _javaObjectBaseTypeSet.Add( "Long" );
            _javaObjectBaseTypeSet.Add( "Float" );
            _javaObjectBaseTypeSet.Add( "Double" );

            _javaBaseToObjectMap.Add( "boolean", "Boolean" );
            _javaBaseToObjectMap.Add( "byte", "Byte" );
            _javaBaseToObjectMap.Add( "char", "Character" );
            _javaBaseToObjectMap.Add( "short", "Short" );
            _javaBaseToObjectMap.Add( "int", "Integer" );
            _javaBaseToObjectMap.Add( "long", "Long" );
            _javaBaseToObjectMap.Add( "float", "Float" );
            _javaBaseToObjectMap.Add( "double", "Double" );

            _javaBaseDefaultValueMap.Add( "boolean", "false" );
            _javaBaseDefaultValueMap.Add( "byte", "0" );
            _javaBaseDefaultValueMap.Add( "char", "\u0000" );
            _javaBaseDefaultValueMap.Add( "short", "0" );
            _javaBaseDefaultValueMap.Add( "int", "0" );
            _javaBaseDefaultValueMap.Add( "long", "0L" );
            _javaBaseDefaultValueMap.Add( "float", "0.0f" );
            _javaBaseDefaultValueMap.Add( "double", "0.0d" );

            _pythonBaseDefaultValueMap.Add( "boolean", "False" );
            _pythonBaseDefaultValueMap.Add( "byte", "0" );
            _pythonBaseDefaultValueMap.Add( "char", "0" );
            _pythonBaseDefaultValueMap.Add( "short", "0" );
            _pythonBaseDefaultValueMap.Add( "int", "0" );
            _pythonBaseDefaultValueMap.Add( "long", "0" );
            _pythonBaseDefaultValueMap.Add( "float", "0.0" );
            _pythonBaseDefaultValueMap.Add( "double", "0.0" );
        }

        private bool _isTristate = false;
        private bool _isAggregatePredicate = false;
        private bool isAggregateAux( ICollection<string> rawNameList ) {

            if ( _isAggregatePredicate ) return true;
            bool retval = false;

            string rawName = _rawName;
            if ( rawNameList.Contains( rawName ) ) return false;
            rawNameList.Add( rawName );

            if ( _avmJavaTypeMap.ContainsKey( rawName ) ) {
                JavaType javaType = _avmJavaTypeMap[ rawName ];
                retval = javaType.isAggregateAux( rawNameList );
            }
            return retval;
        }

        public bool _isAggregate {
            get {
                return isAggregateAux( new List<string>() );
            }
            set {
                _isAggregatePredicate = value;
            }
        }

        public bool _isJavaClass {
            get { return JavaClass._javaClassNameMap.ContainsKey( _fullName ); }
        }

        public bool _isJavaEnum {
            get { return JavaEnum._nameJavaEnumMap.ContainsKey( _fullName ); }
        }

        public string _rawName {
            get { return base._name; }
            set { base._name = value; }
        }
        public string _rawNamespace {
            get { return base._namespace; }
            set { base._namespace = value; }
        }
        public string _fullRawName {
            get { return _rawNamespace == "" ? _rawName : _rawNamespace + "." + _rawName; }
        }
        public new string _name {
            get{
                string type = _rawName;
                if ( _avmJavaTypeMap.ContainsKey( _rawName ) ) {
                    JavaType javaType = _avmJavaTypeMap[ _rawName ];
                    type = javaType._rawName;
                }
                if ( _isTristate && _javaBaseToObjectMap.ContainsKey( type ) ) type = _javaBaseToObjectMap[ type ];
                return type;
            }
        }
        public new string _namespace {
            get {
                string namespace_var = _rawNamespace;
                if ( _avmJavaTypeMap.ContainsKey( _rawName ) ) {
                    JavaType javaType = _avmJavaTypeMap[ _rawName ];
                    namespace_var = javaType._rawNamespace;
                }
                return namespace_var;
            }
        }

        public new string _fullName {
            get {
                string namespace_var = _namespace;
                return namespace_var == "" ? _name : namespace_var + "." + _name;
            }
        }

        public string _interfaceName {
            get { return _isJavaClass ? "I" + _name : _name; }
        }
        
        public string _fullInterfaceName {
            get {
                string namespace_var = _namespace;
                return namespace_var == "" ? _interfaceName : namespace_var + "." + _interfaceName;
            }
        }

        public string _completeType {
            get { return _isAggregate ? "java.util.ArrayList< " + _fullInterfaceName + " >" : _fullInterfaceName; }
        }

        public string _defaultValue {
            get {
                if ( _isAggregate ) return "new " + _completeType + "()";
                if ( _javaBaseDefaultValueMap.ContainsKey( _name ) ) return _javaBaseDefaultValueMap[ _name ];
                return "null";
            }
        }

        public string _pythonDefaultValue {
            get {
                if ( _isAggregate ) return "[]";
                if ( _name == "String" ) return "\"\"";
                if ( _pythonBaseDefaultValueMap.ContainsKey( _name ) ) return _pythonBaseDefaultValueMap[ _name ];
                return "None";
            }
        }

        private void init( string name ) {
            _rawName = name;
            if ( _name.EndsWith( "?" ) ) {
                _rawName = _name.Substring( 0, name.Length - 1 );
                _isTristate = true;
            }
        }

        public JavaType( string name, bool isAggregate = false ) {
            _rawNamespace = "";
            _isAggregate = isAggregate;
            init( name );
        }

        public JavaType( string namespace_var, string name, bool isAggregate = false ) {
            _isAggregate = isAggregate;
            init( name );
            if ( _isTristate && JavaEnum._nameJavaEnumMap.ContainsKey( _rawName ) ) {
                _rawNamespace = JavaEnum._nameJavaEnumMap[ _rawName ]._namespace;
            } else {
                _rawNamespace = namespace_var;
            }
        }

        public JavaType( JavaClass javaClass ) : this( javaClass._namespace, javaClass._name ) { }


        public bool isJavaBaseType() {
            return _javaBaseToObjectMap.ContainsKey( _name );
        }

        public bool isStringType() {
            string javaType = _name;
            return !_javaBaseToObjectMap.ContainsKey( javaType ) && !_javaObjectBaseTypeSet.Contains( javaType );
        }

    }

    public class JavaTypedEntity : JavaEntity {
        public JavaType _javaType = null;
    }

    public class JavaEncapsulator : JavaEntity {
        public Accessibility _accessibility { get; set; }
        public string getAccessibilityString() {
            return _accessibilityStringMap[ _accessibility ];
        }
    }

    public class JavaMember : JavaTypedEntity {
        public Accessibility _accessibility { get; set; }
        public string getAccessibilityString() {
            return _accessibilityStringMap[ _accessibility ];
        }
    }

    public class JavaMemberComparer : IComparer<JavaMember> {
        public int Compare( JavaMember javaMember1, JavaMember javaMember2 ) {
            bool javaMember1Test = javaMember1._javaType._isJavaClass;
            bool javaMember2Test = javaMember2._javaType._isJavaClass;

            if ( !javaMember1Test && javaMember2Test ) return -1;
            if ( javaMember1Test && !javaMember2Test ) return 1;
            return javaMember1._fullName.CompareTo( javaMember2._fullName );
        }
    }

    public class JavaMethod : JavaMember {
        public List<JavaTypedEntity> _parameterList = new List<JavaTypedEntity>();
        public string _body { get; set; }
        public string _pythonBody { get; set; }
    }

    public class JavaInterface : JavaEncapsulator {
        public string _interfaceName { get { return "I" + _name; } }
        public string _fullInterfaceName { get { return _namespace == "" ? _interfaceName : _namespace + "." + _interfaceName; } }

        public List<JavaClass> _baseInterfaceList = new List<JavaClass>();

        public SortedSet<JavaMethod> _methodSet = new SortedSet<JavaMethod>( new JavaMemberComparer() );

        public SortedSet<JavaMethod> getCompleteMethodSet() {
            SortedSet<JavaMethod> completeMethodSet = new SortedSet<JavaMethod>( _methodSet, new JavaMemberComparer() );
            foreach ( JavaClass javaClass in _baseInterfaceList ) {
                completeMethodSet.UnionWith( javaClass.getCompleteMethodSet() );
            }
            return completeMethodSet;
        }
    }

    public class JavaClass : JavaInterface {
        public static Dictionary<string, JavaClass> _javaClassNameMap = new Dictionary<string,JavaClass>();

        public List<JavaClass> _containmentJavaClassList = new List<JavaClass>();
        public SortedSet<JavaMember> _dataMemberSet = new SortedSet<JavaMember>( new JavaEntityComparer() );

        public bool _isAbstract { get; set; }

        public static bool isIdDataMemberName( string name ) {
            return name == "id" || name == "_id";
        }

        private string _hasIdDataMember = null;
        private string _idDataMemberName;

        public JavaClass() {
            _isAbstract = false;
        }
        
        public bool getHasIdDataMember() {
            if ( _hasIdDataMember != null ) return _hasIdDataMember == "true";
            foreach ( JavaMember javaMember in _dataMemberSet ) {
                string javaMemberName = javaMember._name.ToLower(  new CultureInfo( "en-US", false )  );
                if (  isIdDataMemberName( javaMemberName )  ) {
                    _idDataMemberName = javaMember._name;
                    _hasIdDataMember = "true";
                    return true;
                }
            }
            foreach ( JavaClass javaClass in _baseInterfaceList ) {
                if ( javaClass.getHasIdDataMember() ) {
                    _idDataMemberName = javaClass.getIdDataMemberName();
                    _hasIdDataMember = "true";
                    return true;
                }
            }
            _hasIdDataMember = "false";
            return false;
        }
        public string getIdDataMemberName() {
            return _idDataMemberName;
        }

    }

    public class JavaEnum : JavaEncapsulator {
        public static Dictionary<string, JavaEnum> _nameJavaEnumMap = new Dictionary<string, JavaEnum>();

        public JavaEnum( string packageName, string name ) {
            _namespace = packageName;
            _name = name;
            _nameJavaEnumMap.Add( _name, this );
            _nameJavaEnumMap.Add( _fullName, this );
        }

        public List<string> _memberNames = new List<string>();
    }

    public partial class JavaEnumTemplate {
        public JavaEnum _javaEnum { get; set; }
    }

    public partial class JavaInterfaceTemplate {
        public JavaInterface _javaInterface { get; set; }
    }

    public partial class JavaClassTemplate {
        private JavaClass _javaClass_property;

        public JavaClassTemplate() {
            _javaClass_property = new JavaClass();
        }
        public JavaClassTemplate( JavaClass javaClass ) {
            _javaClass_property = javaClass;
        }

        public JavaClass _javaClass {
            get { return _javaClass_property; }
        }
    }

    public partial class PythonEnumTemplate {
        public JavaEnum _javaEnum { get; set; }
    }

    public partial class PythonClassTemplate {
        private JavaClass _javaClass_property;

        public PythonClassTemplate() {
            _javaClass_property = new JavaClass();
        }
        public PythonClassTemplate( JavaClass javaClass ) {
            _javaClass_property = javaClass;
        }

        public JavaClass _javaClass {
            get { return _javaClass_property; }
        }
    }

}
