﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.VisualStudio.Uml.Classes;
using Microsoft.VisualStudio.ArchitectureTools.Extensibility;
using Microsoft.VisualStudio.ArchitectureTools.Extensibility.Uml;
using System.IO;


namespace BuildAVMComponentJavaLib {

    public class ClassDepth {
        private Dictionary<IClass, int> _classDepthMap = new Dictionary<IClass, int>();

        public int getDepth( IClass iClass ) {

            if ( _classDepthMap.ContainsKey( iClass ) ) {
                return _classDepthMap[ iClass ];
            }

            int depth = 0;
            foreach ( IGeneralization iGeneralization in iClass.Generalizations ) {
                IClass iClassTarget = iGeneralization.TargetElement as IClass;
                depth = System.Math.Max( depth, getDepth( iClassTarget ) + 1 );
            }

            _classDepthMap[ iClass ] = depth;

            return depth;
        }

        public static ClassDepth _singleton = new ClassDepth();
    }

    public class IClassComparer : IComparer<IClass> {

        private ClassClosure _classClosure = new ClassClosure();

        public int Compare( IClass iClass1, IClass iClass2 ) {
            int depth1 = ClassDepth._singleton.getDepth( iClass1 );
            int depth2 = ClassDepth._singleton.getDepth( iClass2 );

            if ( depth1 < depth2 ) return -1;
            if ( depth1 > depth2 ) return 1;
            return ( Program.getFullName( iClass1.Package ) + "." + iClass1.Name ).CompareTo( Program.getFullName( iClass2.Package ) + "." + iClass2.Name );
        }
    }

    public class ClassClosure {
        private Dictionary<  IClass, SortedSet< IClass >  > _classClosureMap = new Dictionary<  IClass, SortedSet< IClass >  >();

        public SortedSet< IClass > getClosure( IClass iClass ) {
            if (  _classClosureMap.ContainsKey( iClass )  ) {
                return _classClosureMap[ iClass ];
            }

            SortedSet< IClass > iClassSet = new SortedSet< IClass >( new IClassComparer() );
            foreach ( IGeneralization iGeneralization in iClass.Generalizations ) {
                IClass iClassTarget = iGeneralization.TargetElement as IClass;
                iClassSet.Add( iClassTarget );
                iClassSet.UnionWith(  getClosure( iClassTarget )  );
            }

            _classClosureMap[ iClass ] = iClassSet;

            return iClassSet;
        }

        public static ClassClosure _singleton = new ClassClosure();
    }

    public class Program {

        public static string getFullName( INamedElement iNamedElement ) {

            if ( iNamedElement == null ) return "";

            string fullName = iNamedElement.Name;

            IElement iElement = iNamedElement.Owner;
            while ( iElement != null && iElement is INamedElement && ( iElement as INamedElement ).Name != "DesignDataPackageAPI" ) {
                fullName = ( iElement as INamedElement).Name + "." + fullName;
                iElement = iElement.Owner;
            }

            return fullName;
        }

        static void Main(string[] args) {

            Dictionary< IClass, JavaClass > iClassJavaClassMap = new Dictionary<IClass,JavaClass>();
            Dictionary<string, HashSet<JavaClass>> packageJavaClassSetMap = new Dictionary<string, HashSet<JavaClass>>();
            Dictionary<string, HashSet<string>> packageImportSetMap = new Dictionary<string, HashSet<string>>();

            // Step 1, load the UML model
            String umlProjectFile = @"..\..\..\Model\Model.modelproj";

            using ( IModelingProjectReader iModelingProjectReader = ModelingProject.LoadReadOnly( umlProjectFile ) ) {

                IModelStore iModelStore = iModelingProjectReader.Store;

                SortedSet<IClass> sortedIClassSet = new SortedSet<IClass>( new IClassComparer() );
                HashSet<IEnumeration> iEnumerationSet = new HashSet<IEnumeration>();

                foreach ( IPackage iPackage in iModelStore.AllInstances<IPackage>().Where( iPackage => iPackage.Name != "DesignDataPackageAPI" ) ) {
                    iEnumerationSet.UnionWith( iPackage.OwnedMembers.OfType<IEnumeration>() );
                    sortedIClassSet.UnionWith( iPackage.OwnedMembers.OfType<IClass>() );
                }

                string generatedCodeDir = "..\\..\\..\\AVMComponentJavaLib\\generated";
                Directory.Delete( generatedCodeDir, true );
                Directory.CreateDirectory( generatedCodeDir );

                string pythonGeneratedCodeDir = "..\\..\\..\\AVMComponentPythonLib\\generated";
                Directory.Delete( pythonGeneratedCodeDir, true );
                Directory.CreateDirectory( pythonGeneratedCodeDir );

                foreach ( IClass iClass in sortedIClassSet ) {
                    JavaClass javaClass = new JavaClass();
                    string namespace_var = getFullName( iClass.Package );
                    javaClass._namespace = namespace_var;

                    javaClass._name = iClass.Name;
                    javaClass._accessibility = Accessibility.PUBLIC;
                    javaClass._isAbstract = iClass.IsAbstract;
                    JavaClass._javaClassNameMap.Add( javaClass._fullName, javaClass );

                    if ( !packageJavaClassSetMap.ContainsKey( namespace_var ) ) {
                        packageJavaClassSetMap.Add( namespace_var, new HashSet<JavaClass>() );
                    }
                    packageJavaClassSetMap[ namespace_var ].Add( javaClass );

                    if ( !packageImportSetMap.ContainsKey( namespace_var ) ) {
                        packageImportSetMap.Add( namespace_var, new HashSet<string>() );
                    }
                    packageImportSetMap[ namespace_var ].Add( "ISIS" );
                }

                foreach ( IEnumeration iEnumeration in iEnumerationSet ) {

                    IPackage iPackage = iEnumeration.Owner as IPackage;

                    string packageName = getFullName( iPackage );

                    JavaEnumTemplate javaEnumTemplate = new JavaEnumTemplate();
                    javaEnumTemplate._javaEnum = new JavaEnum( packageName, iEnumeration.Name );
                    javaEnumTemplate._javaEnum._accessibility = Accessibility.PUBLIC;

                    foreach ( INamedElement iNamedElement in iEnumeration.Members ) {
                        javaEnumTemplate._javaEnum._memberNames.Add( iNamedElement.Name );
                    }

                    String packageDir = generatedCodeDir + "\\" + packageName.Replace( ".", "\\" );
                    Directory.CreateDirectory( packageDir );

                    String javaEnumOutput = javaEnumTemplate.TransformText();
                    String javaOutFile = packageDir + "\\" + javaEnumTemplate._javaEnum._name + ".java";
                    StreamWriter javaOutFileStream = new StreamWriter( javaOutFile );
                    javaOutFileStream.Write( javaEnumOutput );
                    javaOutFileStream.Close();

                    PythonEnumTemplate pythonEnumTemplate = new PythonEnumTemplate();
                    pythonEnumTemplate._javaEnum = javaEnumTemplate._javaEnum;

                    string pythonModuleDir = pythonGeneratedCodeDir + "\\" + javaEnumTemplate._javaEnum._namespace.Replace( ".", "\\" );

                    Directory.CreateDirectory( pythonModuleDir );
                    StreamWriter pythonOutFileStream = new StreamWriter( pythonModuleDir + "\\__init__.py", true );
                    String pythonEnumOutput = pythonEnumTemplate.TransformText();
                    pythonOutFileStream.Write( pythonEnumOutput );
                    pythonOutFileStream.Close();
                }

                foreach ( IClass iClass in sortedIClassSet ) {

                    JavaClass javaClass = JavaClass._javaClassNameMap[ getFullName( iClass ) ];

                    iClassJavaClassMap.Add( iClass, javaClass );

                    string packageName = javaClass._namespace;

                    foreach ( IGeneralization iGeneralization in iClass.Generalizations ) {
                        IClass iClassTarget = iGeneralization.TargetElement as IClass;
                        JavaClass javaClassTarget = iClassJavaClassMap[ iClassTarget ];
                        javaClass._baseInterfaceList.Add( javaClassTarget );

                        string namespace_var = javaClassTarget._namespace;
                        if ( namespace_var != packageName ) {
                            if ( !packageImportSetMap.ContainsKey( packageName ) ) {
                                packageImportSetMap.Add( packageName, new HashSet<string>() );
                            }
                            packageImportSetMap[ packageName ].Add( namespace_var );
                        }
                    }

                    foreach ( IProperty iProperty in iClass.OwnedAttributes ) {

                        JavaMember javaMember = new JavaMember();
                        javaMember._accessibility = Accessibility.PRIVATE;
                        javaMember._name = "_" + iProperty.Name;

                        string namespace_var;
                        string typeName;
                        if ( iProperty.Type == null ) {
                            namespace_var = "";
                            typeName = "String";
                        } else {
                            namespace_var = getFullName( iProperty.Type.Package );
                            typeName = iProperty.Type.Name;
                        }

                        javaMember._javaType = new JavaType( namespace_var, typeName );

                        if ( javaMember._javaType._namespace != "" && javaMember._javaType._namespace != packageName && javaMember._javaType._isJavaEnum ) {
                            if ( !packageImportSetMap.ContainsKey( packageName ) ) {
                                packageImportSetMap.Add( packageName, new HashSet<string>() );
                            }
                            packageImportSetMap[ packageName ].Add( javaMember._javaType._namespace );
                        }

                        if ( iProperty.LowerValue != null && iProperty.LowerValue.ToString() == "*" || iProperty.UpperValue != null && iProperty.UpperValue.ToString() == "*" ) {
                            javaMember._javaType._isAggregate = true;
                        }

                        javaClass._dataMemberSet.Add( javaMember );

                        if ( !javaMember._javaType._isAggregate ) {
                            JavaTypedEntity parameter = new JavaTypedEntity();
                            parameter._javaType = javaMember._javaType;
                            parameter._name = iProperty.Name;

                            JavaMethod javaMethod = new JavaMethod();
                            javaMethod._accessibility = Accessibility.PUBLIC;
                            javaMethod._javaType = new JavaType( "void" );
                            javaMethod._name = "set_" + iProperty.Name;
                            javaMethod._parameterList.Add( parameter );
                            javaMethod._body = javaMember._name + " = " + parameter._name + ";";
                            javaMethod._pythonBody = "self." + javaMember._name + " = " + parameter._name;

                            javaClass._methodSet.Add( javaMethod );
                        }

                        {
                            JavaMethod javaMethod = new JavaMethod();
                            javaMethod._accessibility = Accessibility.PUBLIC;
                            javaMethod._javaType = javaMember._javaType;
                            javaMethod._name = "get_" + iProperty.Name;
                            javaMethod._body = "return " + javaMember._name + ";";
                            javaMethod._pythonBody = "return self." + javaMember._name;

                            javaClass._methodSet.Add( javaMethod );
                        }
                    }

                    foreach ( IProperty iPropertyAux in iClass.GetOutgoingAssociationEnds().Where( iProperty => iProperty.Name != "" ) ) {

                        IProperty iProperty = iPropertyAux; // MAY NEED TO ASSIGN TO iProperty, AND CAN'T ASSIGN IF IT'S AN ITERATION VARIABLE

                        IProperty oppositeIProperty = iProperty.Opposite;
                        if ( oppositeIProperty.IsComposite ) {
                            if ( iProperty.Type != iClass && oppositeIProperty.Type == iClass ) continue; // CHILDREN DO NOT HAVE A LINK BACK TO THEIR PARENT
                            iProperty = iProperty.Opposite;
                            oppositeIProperty = iProperty.Opposite;
                        }

                        // IF ASSOCIATION IS NOT PARENT-CHILD (I.E. NOT "COMPOSITE") AND IS MANY-TO-ONE, CREATE LINK ONLY IN CLASS THAT HAS THE "ONE" INSTANCE OF THE ASSOCIATION
                        if (
                         !iProperty.IsComposite &&
                         iProperty.UpperValue != null && iProperty.UpperValue.ToString() != "1" &&
                         ( oppositeIProperty.UpperValue == null || oppositeIProperty.UpperValue.ToString() == "1" )
                        ) continue;

                        IClass childIClass = iProperty.Type as IClass;

                        JavaMember javaMember = new JavaMember();
                        javaMember._accessibility = Accessibility.PRIVATE;
                        javaMember._name = "_" + iProperty.Name;

                        string namespace_var = getFullName( childIClass.Package );
                        //if ( namespace_var != "" && namespace_var != packageName ) {
                        //    if ( !packageImportSetMap.ContainsKey( packageName ) ) {
                        //        packageImportSetMap.Add( packageName, new HashSet<string>() );
                        //    }
                        //    packageImportSetMap[ packageName ].Add( namespace_var );
                        //}

                        if ( iProperty.UpperValue == null || iProperty.UpperValue.ToString() == "1" ) {
                            javaMember._javaType = new JavaType( namespace_var, childIClass.Name );

                            JavaTypedEntity parameter = new JavaTypedEntity();
                            parameter._javaType = javaMember._javaType;
                            parameter._name = iProperty.Name;

                            JavaMethod javaMethod = new JavaMethod();
                            javaMethod._accessibility = Accessibility.PUBLIC;
                            javaMethod._javaType = new JavaType( "void" );
                            javaMethod._name = "set_" + iProperty.Name;
                            javaMethod._parameterList.Add( parameter );
                            javaMethod._body = javaMember._name + " = " + parameter._name + ";";
                            javaMethod._pythonBody = "self." + javaMember._name + " = " + parameter._name;

                            javaClass._methodSet.Add( javaMethod );
                        } else {
                            javaMember._javaType = new JavaType( namespace_var, childIClass.Name );
                            javaMember._javaType._isAggregate = true;
                        }

                        javaClass._dataMemberSet.Add( javaMember );

                        {
                            JavaMethod javaMethod = new JavaMethod();
                            javaMethod._accessibility = Accessibility.PUBLIC;
                            javaMethod._javaType = javaMember._javaType;
                            javaMethod._name = "get_" + iProperty.Name;
                            javaMethod._body = "return " + javaMember._name + ";";
                            javaMethod._pythonBody = "return self." + javaMember._name;

                            javaClass._methodSet.Add( javaMethod );
                        }


                    }

                    String packageDir = generatedCodeDir + "\\" + packageName.Replace( ".", "\\" );
                    Directory.CreateDirectory( packageDir );

                    SortedSet<IClass> closure = new SortedSet<IClass>( ClassClosure._singleton.getClosure( iClass ), new IClassComparer() );
                    bool first = true;
                    foreach ( IGeneralization iGeneralization in iClass.Generalizations ) {

                        IClass iClassTarget = iGeneralization.TargetElement as IClass;
                        if ( !closure.Contains( iClassTarget ) ) continue;

                        SortedSet<IClass> targetClosure = ClassClosure._singleton.getClosure( iClassTarget );

                        if ( first ) {
                            first = false;
                            closure.ExceptWith( targetClosure );
                            continue;
                        }

                        JavaClass javaClassTarget = iClassJavaClassMap[ iClassTarget ];
                        javaClass._baseInterfaceList.Add( javaClassTarget );

                        string namespace_var = javaClassTarget._namespace;
                        if ( !packageImportSetMap.ContainsKey( packageName ) ) {
                            packageImportSetMap.Add( packageName, new HashSet<string>() );
                        }
                        packageImportSetMap[ packageName ].Add( namespace_var );

                        JavaMember javaMember = new JavaMember();
                        javaMember._accessibility = Accessibility.PRIVATE;
                        javaMember._javaType = new JavaType( javaClassTarget );
                        javaMember._name = "_" + javaClassTarget._name;

                        javaClass._dataMemberSet.Add( javaMember );

                        SortedSet<IClass> interfaceIClasses = new SortedSet<IClass>( closure, new IClassComparer() );
                        interfaceIClasses.IntersectWith( targetClosure );
                        closure.ExceptWith( targetClosure );

                        foreach ( IClass interfaceIClass in interfaceIClasses ) {

                            JavaClass interfaceJavaClass = iClassJavaClassMap[ interfaceIClass ];

                            foreach ( JavaMethod interfaceJavaMethod in interfaceJavaClass._methodSet ) {

                                if ( !interfaceJavaMethod._name.StartsWith( "get" ) && !interfaceJavaMethod._name.StartsWith( "set" ) ) continue;

                                JavaMethod javaMethod = new JavaMethod();
                                javaMethod._accessibility = Accessibility.PUBLIC;
                                javaMethod._javaType = interfaceJavaMethod._javaType;
                                javaMethod._name = interfaceJavaMethod._name;
                                javaMethod._parameterList = new List< JavaTypedEntity >( interfaceJavaMethod._parameterList );

                                if ( javaMethod._javaType._fullName != "void" ) {
                                    javaMethod._body = "return ";
                                }
                                javaMethod._body += javaMember._name + "." + javaMethod._name + "(";
                                if ( javaMethod._parameterList.Count != 0 ) {
                                    bool firstParameter = true;
                                    foreach ( JavaTypedEntity javaTypedEntity in javaMethod._parameterList ) {
                                        if ( firstParameter ) firstParameter = false;
                                        else                  javaMethod._body += ",";
                                        javaMethod._body += " " + javaTypedEntity._name;
                                    }
                                    javaMethod._body += " ";
                                }
                                javaMethod._body += ");";
                            }
                        }
                    }

                }

                foreach ( string packageName in packageJavaClassSetMap.Keys ) {

                    String packageDir = generatedCodeDir + "\\" + packageName.Replace( ".", "\\" );
                    Directory.CreateDirectory( packageDir );

                    string pythonModuleDir = pythonGeneratedCodeDir + "\\" + packageName.Replace( ".", "\\" );

                    Directory.CreateDirectory( pythonModuleDir );
                    StreamWriter pythonOutFileStream = new StreamWriter( pythonModuleDir + "\\__init__.py", true );

                    foreach ( string importPackageName in packageImportSetMap[ packageName ] ) {
                        pythonOutFileStream.WriteLine( "import " + importPackageName );
                    }

                    foreach ( JavaClass javaClass in packageJavaClassSetMap[ packageName ] ) {

                        JavaClassTemplate javaClassTemplate = new JavaClassTemplate( javaClass );
                        JavaInterfaceTemplate javaInterfaceTemplate = new JavaInterfaceTemplate();
                        PythonClassTemplate pythonClassTemplate = new PythonClassTemplate( javaClass );


                        javaInterfaceTemplate._javaInterface = new JavaInterface();
                        javaInterfaceTemplate._javaInterface._accessibility = javaClassTemplate._javaClass._accessibility;
                        javaInterfaceTemplate._javaInterface._baseInterfaceList = javaClassTemplate._javaClass._baseInterfaceList;

                        javaInterfaceTemplate._javaInterface._methodSet = javaClassTemplate._javaClass._methodSet;
                        javaInterfaceTemplate._javaInterface._name = javaClassTemplate._javaClass._name;
                        javaInterfaceTemplate._javaInterface._namespace = javaClassTemplate._javaClass._namespace;

                        String javaInterfaceOutput = javaInterfaceTemplate.TransformText();
                        String javaInterfaceOutFile = packageDir + "\\" + javaInterfaceTemplate._javaInterface._interfaceName + ".java";
                        StreamWriter javaInterfaceOutFileStream = new StreamWriter( javaInterfaceOutFile );
                        javaInterfaceOutFileStream.Write( javaInterfaceOutput );
                        javaInterfaceOutFileStream.Close();


                        String javaClassOutput = javaClassTemplate.TransformText();
                        String javaClassOutFile = packageDir + "\\" + javaClassTemplate._javaClass._name + ".java";
                        StreamWriter javaClassOutFileStream = new StreamWriter( javaClassOutFile );
                        javaClassOutFileStream.Write( javaClassOutput );
                        javaClassOutFileStream.Close();

                        String pythonClassOutput = pythonClassTemplate.TransformText();
                        pythonOutFileStream.Write( pythonClassOutput );
                    }

                    pythonOutFileStream.Close();

                }
            }
        }
    }
}
