from GMEComComponent import GMEComComponent

import os.path

class ICCAExport(GMEComComponent):

	# Component specific parts for COM registration
	_comname_ = "ICCAExport"
	_comp_version_ = "1.0"
	_reg_clsid_ = "{38C890E7-3F59-4228-93E9-72CD80BB4520}"
	_reg_iconfile_ = "raven.ico"
	
	_reg_desc_ = _comname_
	_reg_progid_ = "MGA.PythonInterpreter.ICCAExport"
	_reg_verprogid_ = "MGA.PythonInterpreter.ICCAExport.1.0"
	_reg_class_spec_ = "%s.%s" % (__module__, _comname_)

	# Component specific parts for GME registration
	_tooltip_ = "ICCA Export"
	_paradigm_ = "Hydraulics"

	def InvokeEx(self, project, currentobj, selectedobjs, param):
		super(self.__class__, self).InvokeEx(project)
		aborted = False
		try:
			self.project.BeginTransaction(self.project.CreateTerritory(None, None, None))
			projectStr = self.project.ProjectConnStr
			projectPath = projectStr.split("MGA=")[1]
			projectDir = os.path.dirname(projectPath)
			
			currentobj = currentobj and self.mga.IMgaModel(currentobj)
			
			self.iccaString = ""
			if currentobj.Meta.Name == 'HydraulicModel':
				exportFileName = currentobj.Name+".txt"
				self.Logger("Exporting ICCA information to file: %s" % exportFileName)
				
				# printing models and connections two levels deep:
				for fco in currentobj.ChildFCOs:
					if fco.ObjType == self.meta.constants.OBJTYPE_MODEL:
						model = self.mga.IMgaModel(fco)
						self.Logger(model.Meta.Name)
						self.writeICCAln(self.basicAttributes(model))
						self.writeICCAln(self.performanceAttributes(model))
					elif fco.ObjType == self.meta.constants.OBJTYPE_ATOM:
						atom = self.mga.IMgaAtom(fco)
						if atom.Meta.Name in ['BoundaryCondition']:
							self.Logger(atom.Meta.Name)
							self.writeICCAln(self.basicAttributes(atom))
					#	if model.Meta.Name is 'Tank':
					#		model = self.mga.IMgaModel(child)
					elif fco.ObjType == self.meta.constants.OBJTYPE_CONNECTION:
						conn = self.mga.IMgaConnection(fco)
						self.Logger("PIPE")
						self.writeICCAln(self.basicAttributes(conn))
						self.writeICCAln(self.performanceAttributes(conn))
						self.writeICCAln(self.ports(conn))
					
					self.writeICCAln("#")
					
				f = open(os.path.join(projectDir,exportFileName), 'w')
				f.write(self.iccaString)
				f.close()
		finally:
			if not aborted:
				self.project.CommitTransaction()

	def writeICCAln(self, line):
		self.iccaString += line+"\n"

	def ports(self, fco):
		portString = ""
		if fco.ObjType == self.meta.constants.OBJTYPE_MODEL:
			pass
			
		elif fco.ObjType == self.meta.constants.OBJTYPE_CONNECTION:
			conn = self.mga.IMgaSimpleConnection(fco)
			srcX, srcY = self.getPosition(conn.src)
			dstX, dstY = self.getPosition(conn.dst)
			portString = "PORT||||%s|%s|L1|" % (srcX,srcY)
			portString += "\nPORT||||%s|%s|L2|" % (dstX,dstY)
		elif fco.ObjType == self.meta.constants.OBJTYPE_ATOM:
			pass
		
		return portString

	def basicAttributes(self, fco):
		uid = fco.AttributeByName("UID")
		iccaType = fco.AttributeByName("ICCAType")
		compName = fco.AttributeByName("ComponentName")
		posX,posY = self.getPosition(fco)
		if fco.ObjType == self.meta.constants.OBJTYPE_MODEL:
			type = self.determineModelType(fco)
		elif fco.ObjType == self.meta.constants.OBJTYPE_ATOM:
			type = atom.Meta.Name
		elif fco.ObjType == self.meta.constants.OBJTYPE_CONNECTION:
			type = "Pipe Line"
		iccaBasicAttributes = "PART|%s|%s|%s||||%s|%s||%s" % (uid,iccaType,compName,posX,posY,type)
		return iccaBasicAttributes
	
	def performanceAttributes(self, fco):
		perfString = "PERFORMANCE|TYPE:SINGLE|"
		
		if fco.ObjType == self.meta.constants.OBJTYPE_MODEL:
			model = self.mga.IMgaModel(fco)
			arrays = model.GetChildrenOfKind("Array")
			for array in arrays:
				if array.Name == "PerformanceData":
					perfData = self.mga.IMgaModel(array)
					if model.Meta.Name in ['Valve']:
						counter = 2
						ports = perfData.GetChildrenOfKind("Struct")
						for port in ports:
							portData = self.mga.IMgaModel(port)
							propNames = ["FlowRate","ElementType","InsideDiameter","LineLength","KFactor","AdditionalProperties"]
							for propName in propNames:
								prop = portData.ChildFCO(propName)
								if prop.Meta.Name != "Array":
									propAtom = self.mga.IMgaAtom(prop)
									if propAtom.Meta.Name == "Float":
										attribute = propAtom.AttributeByName("RealValue")
									elif propAtom.Meta.Name == "Integer":
										attribute = propAtom.AttributeByName("IntegerValue")
									perfString += str(counter)+":"+str(attribute)+"|"
									counter += 1
								elif prop.Meta.Name == "Array":
									propModel = self.mga.IMgaModel(prop)
									childParams = propModel.GetChildrenOfKind("Float")
									for childParam in childParams:
										childParam = self.mga.IMgaAtom(childParam)
										perfString += str(counter)+":"+str(childParam.AttributeByName("RealValue"))+"|"
										counter += 1
					else:
						perfString += 'TBD'
		elif fco.ObjType == self.meta.constants.OBJTYPE_CONNECTION:
			conn = self.mga.IMgaSimpleConnection(fco)
			flowRate = conn.AttributeByName("FlowRate")
			insideD = conn.AttributeByName("InsideDiameter")
			lineLength = conn.AttributeByName("LineLength")
			kFactor = conn.AttributeByName("KFactor")
			perfString = "PERFORMANCE|TYPE:SINGLE|2:%s|3:1|4:%s|5:%s|6:%s|" % (flowRate,insideD,lineLength,kFactor)
		return perfString

	def determineModelType(self, model):
		type = "TBD"
		if model.DerivedFrom:
			superType = self.mga.IMgaModel(model.DerivedFrom)
			type = superType.Name
		return type

	def getPosition(self, fco):
		if fco.ObjType != self.meta.constants.OBJTYPE_CONNECTION:
			regValue = fco.RegistryValue("PartRegs/Schematics/Position")
			position = regValue.split(",")
		else:
			conn = self.mga.IMgaSimpleConnection(fco)
			srcX, srcY = self.getPosition(conn.src)
			dstX, dstY = self.getPosition(conn.dst)
			position = [(int(srcX)+int(dstX))/2,(int(srcY)+int(dstY))/2]
		return position
		