%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Import function code from a source model to a target model. It works for
% code regions. See an example below.
% 
% Author: Zsolt Lattmann (lattmann@isis.vanderbilt.edu)
%
% Date: 2011/05/07 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [] = ImportFunctionCode(SourceModelName, ...
                                 SubsystemName, ...
                                 TargetModelName, ...
                                 PatternBeginKey, ...
                                 PatternEndKey, ...
                                 PatternPostFix, ...
                                 FcnType)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Input example:
%   SourceModelName = 'Engines';
%   SubsystemName = 'liquid_cooled_v10_2';
%   TargetModelName = 'HEV_Flat3';
%   PatternBeginKey = '%%% BEGIN ';
%   PatternEndKey = '%%% END ';
%   PatternPostFix = ' CODE %%%';
%   FcnType = 'PostLoadFcn';
%
% Engines\%ANYOTHERPATH%\liquid_cooled_v10_2 requires PostLoadFcn, which
% starts with %%% BEGIN liquid_cooled_v10_2 CODE %%% and the end line is
% %%% END liquid_cooled_v10_2 CODE %%% everything between these two lines
% (including begin and end lines) are appended to HEV_Flat3 system's
% PostLoadFcn.
% REMARK: %ANYOTHERPATH% is not supported at this point.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % all possible function type
  FcnAllType = {'PreLoadFcn'; ...
                'PostLoadFcn'; ...
                'InitFcn'; ...
                'StartFcn'; ...
                'PauseFcn'; ...
                'ContinueFcn'; ...
                'StopFcn'; ...
                'PreSaveFcn'; ...
                'PostSaveFcn'; ...
                'CloseFcn'; ...
                };

  if max(strcmp(FcnAllType, FcnType)) == 1

    % load source model
    load_system(SourceModelName);
    PostLoadFunction = get_param(SourceModelName, FcnType);

    % find the code region
    idx_start = strfind(PostLoadFunction, ...
      [PatternBeginKey SubsystemName PatternPostFix]);    
    idx_end = strfind(PostLoadFunction, ...
      [PatternEndKey SubsystemName PatternPostFix]);    
    idx_end = idx_end + length([PatternEndKey SubsystemName PatternPostFix]);
    
    if ~isempty(idx_end) && ~isempty(idx_start) && (idx_end > idx_start)
      % select the code region
      PostLoadThisSubSystem = PostLoadFunction(idx_start:idx_end);

      % load target model
      load_system(TargetModelName);
      % get actual code text
      PostLoadFunctionOld = get_param(TargetModelName, FcnType);
      % append new code text
      PostLoadFunctionNew = [PostLoadFunctionOld PostLoadThisSubSystem];
      % set the model parameter with the appended code region
      set_param(TargetModelName, FcnType, PostLoadFunctionNew);
      fprintf(['Model %s code imported from %s. ' ...
        ' (Name id: %s)\n'], FcnType, SourceModelName, SubsystemName);
    end
  else
    fprintf('Invalid FcnType: %s.\n', FcnType);
  end    
end