%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Summary: Runs a generated model.
% 
% Required files:
%   - MDAO_Init.m in the start directory (and all generated script files)
%   - %sysName%.mdl already built model
%   - installed MDAO-matlab-library
%   - installed HBG-matlab-library
%
%   - input file if specified
% 
% Usage:
%  (0) Start it from MatLab
%  Go to the model's directory
%  >> run_model('input.txt','out.txt')
%
%  (1) Start it from the same directory
%  cmd> start matlab -wait -nodesktop -nosplash -r
%  run_model('input.txt','out.txt');exit;
%
%  (2) Start it from a different directory. Model and files are in the
%  generated subfolder.
%  cmd> start matlab -sd ./generated -wait -nodesktop -nosplash -r
%  run_model('input.txt','out.txt');exit;
%
%% Author: Zsolt Lattmann (lattmann@isis.vanderbilt.edu)
%
%% Date: 2011/08/16 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function run_model( ...
  InputFileName, ...  % name of the input file
  OutputFileName, ... % name of the output file
  ShowMessages)       % turns the global warnings 'on' or 'off'

try
    %% CLEAN UP
    clc;
    tic; % start timer

    if exist('_FAILED.txt', 'file')        
        delete('_FAILED.txt');
    end

    %% VARIABLES
    if (nargin < 3) || isempty(ShowMessages)
      ShowMessages = 'off';
    end
    if (nargin < 2) || isempty(OutputFileName)
      OutputFileName = 'output.txt';
    end
    if (nargin < 1) || isempty(InputFileName)
      InputFileName = 'input.txt';
    end

    if strcmp(ShowMessages, 'off')
      warning off;
    end

    %% INIT ALL VARIABLES WITH THEIR DEFAULT VALUES
    disp('=== Call MDAO_Init. ==============================================');
    MDAO_Init;

    %% LOAD MODEL
    disp('=== Load model. ==================================================');

    close_system(sysName);
    load_system(sysName);

    %% READ THE CURRENT INPUT VALUES
    InputFileExist = exist(InputFileName, 'file');

    n = 0; % number of input sets

    if InputFileExist == 0
      % file does not exist
      disp('=== Use default input variables from MDAO_Init. ==================');
      n = 1;
    else
      % file does exist
      disp('=== Read input variables from file. ==============================');
      modelParameters = readInfile(InputFileName);
      n = length(modelParameters(:,1));
      responseValues = repmat(NaN, n, length(responseNames));
      fprintf('===  - Found %d sets of inputs in %s.\n', n, InputFileName);
    end

    % iterate through all sets
    for j=1:n

      if InputFileExist ~= 0
        % overwrite values
        for k=1:length(modelParameters(1,:))
            eval([Parameters{k} ' = ' ...
              sprintf('modelParameters(%f,k);',j)]);
        end

        % run the model with the new parameters
        fprintf('==== > Evaluating set %d: =========================================\n', j);
        disp(' ');
        for k=1:length(modelParameters(1,:))
            fprintf('%33s = %f', Parameters{k}, modelParameters(j,k));
            if k < length(modelParameters(1,:))
                fprintf('\n');
            else
                fprintf('\n');
            end
        end   
      end

      %% RUN SIMULATION
      disp(' ');
      disp('                  ========> RUN SIMULATION. <========             ');
      disp(' ');
      result = sim(sysName, ...
        'StopTime', sprintf('%d', StopTime));
      
      generateXML(result);
      
      % collect results into the returnValues variable
      for i=1:length(responseNames)
        %fprintf('%s\n', responseNames{i});
        eval(['returnValues(' sprintf('%d',i) ') = ' ...
          'result.find(''' responseNames{i} ''').signals.values(end);']);
      end

      tmp_datestr = datestr(now);
      for k=1:length(responseNames)
          eval([responseNames{k} ' = ' sprintf('%f',returnValues(k)) ';']);

          tmp_name = responseNames{k};
          eval(['fprintf(''%33s = ' ...
            '%f\n'', tmp_name, ' responseNames{k} ');']);

          eval(['responseValues(' ...
            sprintf('%d',j) ',' ...
            sprintf('%d',k) ') = ' ...
            responseNames{k} ';']);

         %response plots
         pngFileName = ['Metrics_' responseNames{k}];
         eval(['tmp_plot = plot(' ...
           'result.find(''' responseNames{k} ''').time, ' ...
           'result.find(''' responseNames{k} ''').signals.values);']);

         xlabel('time [s]');
         if ~isempty(unitNames{k})       
           ylabel([responseNames{k} ' [' unitNames{k} ']']); % this must contain the unit as well
         else
           ylabel(responseNames{k});
         end
         title([sysName ' model (' tmp_datestr ')']);
         saveas(tmp_plot, pngFileName, 'png');
         close all;
      end
      disp(' ');
    end

    if isempty(responseNames)
      responseValues = [];
    end

    %% WRITE THE RESULTS
    disp('=== Write output variables to file. ==============================');
    writeOutfile(OutputFileName, ...
      responseNames, ...
      responseValues);

    %% SAVE AND CLOSE MODEL
    save_system(sysName);
    close_system(sysName);

    %% POST ACTIONS
    warning on;

    fprintf('=== Elapsed time: %fs. ===\n', toc); % print elapsed time
catch ex
    fileName = '_FAILED.txt';
    fid = fopen(fileName, 'w');
    if fid == -1
        error('Could not write sample data to %s\n', fileName);    
    end
    fprintf(fid, '%s', getReport(ex, 'extended'));
    fclose(fid);     
end
        