/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "Nastran.h"
#include <string>
#include <iostream>
#include <fstream>
#include <stdio.h>
#include <iomanip>
#include "MiscellaneousFunctions.h"
#

#pragma warning( disable : 4996 ) // Disable warining about not using the safe char* functions (e.g. sscanf_s, sprintf_s...
namespace isis_CADCommon
{
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void ReplaceCommasWithSpace( std::string &in_out_String )
	{
		size_t commaPosition;
		commaPosition = in_out_String.find(",");
		while ( commaPosition != std::string::npos )
		{
			in_out_String.replace(commaPosition,1, " " );
			commaPosition = in_out_String.find(",", commaPosition);
		}
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void ReplaceSpacesWithNull( std::string &in_out_String )
	{
		size_t spacePosition;
		spacePosition = in_out_String.find(" ");
		while ( spacePosition != std::string::npos )
		{
			in_out_String.replace(spacePosition,1, "" );
			spacePosition = in_out_String.find(" ", spacePosition);
		}
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void ReplaceDoubleCommasWithCommaSpaceComma( std::string &in_out_String )
	{
		size_t doubleComma;
		doubleComma = in_out_String.find(",,");
		while ( doubleComma != std::string::npos )
		{
			in_out_String.replace(doubleComma,2, ", ," );
			doubleComma = in_out_String.find(",,", doubleComma);
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////
	//	class NastranDeck
	/////////////////////////////////////////////////////////////////////////////////////////////////////

	NastranDeck::NastranDeck():maxCoordinateID(0) {};
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::ReadNastranDeck( const std::string &in_InputFileName ) throw (isis::application_exception)
	{
		enum DeckSection { e_EXECUTIVE , e_CASE, e_BULK };

		DeckSection deckSection = e_EXECUTIVE;

		std::ifstream  deckFile;
		deckFile.open (in_InputFileName.c_str(), std::ios::in );

		if (!deckFile.is_open())
		{
			std::string errorString;
			errorString = "Function - NastranDeck::NastranDeck, could not open file: " + in_InputFileName;
			throw isis::application_exception(errorString.c_str());	
		}

		std::string deckLine;
		while ( deckFile.good() )
		{
			getline(deckFile, deckLine );

			switch (  deckSection )
			{
				case e_EXECUTIVE:
					if ( deckLine.find("CEND") != std::string::npos ) 
					{
						deckSection = e_CASE;
						endExecutiveControl = deckLine;
					}
					else
					{
						executiveControl.push_back(deckLine);
					}

					break;
				case e_CASE:
					if ( deckLine.find("BEGIN BULK") != std::string::npos )
					{
						deckSection = e_BULK;
						beginBulkData = deckLine;
					}
					else
					{
						caseControl.push_back(deckLine);
					}
					break;
				case e_BULK:
					if ( deckLine.find("ENDDATA") != std::string::npos )
					{
						endBulkData = deckLine;
					}
					else
					{
						bulkData.push_back(deckLine);
						if ( (deckLine.find("CORD1") != std::string::npos ) ||
							 (deckLine.find("CORD2") != std::string::npos ))
						{
							ReplaceCommasWithSpace(deckLine);
							char  name[32];
							int	  coordinateID;
							//maxCoordinateID
							sscanf(deckLine.c_str(), "%s %d", name, &coordinateID );
							if ( coordinateID > maxCoordinateID ) maxCoordinateID = coordinateID;
						}
					}
					/*
					if ( deckLine.find("GRID") != std::string::npos )
					{
						GridPoint gridPoint;
						char  name[32];
						//std::cout << std::endl << deckLine;
						ReplaceCommasWithSpace(deckLine);
						//std::cout << std::endl << deckLine;
						sscanf(deckLine.c_str(), "%s %d %d %lf %lf %lf %d", 
														name, 
														&gridPoint.ID, 
														&gridPoint.locationCoordinateSystem_ID, 
														&gridPoint.point.x,
														&gridPoint.point.y,
														&gridPoint.point.z,
														&gridPoint.displacementCoordinateSystem_ID );
						//std::cout << std::endl << gridPoint;
						gridPoints_map[gridPoint.ID] = gridPoint;					
					}
					*/
					break;
			}
		} // End 	while ( deckFile.good() )
		deckFile.close();
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	int	 NastranDeck::getMaxCoordinateID() { return maxCoordinateID;}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::WriteNastranDeck(const std::string &in_OutputFileName ) throw (isis::application_exception)
	{
		std::ofstream  deckFile;
		deckFile.open (in_OutputFileName.c_str(),std::ios::out | std::ios::trunc  );

		deckFile << *this;

		deckFile.close(); 
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::GetGridPoints( std::map<int, GridPoint>  &out_GridPoints_map)
	{

		for(std::list<std::string>::const_iterator i(bulkData.begin()); i != bulkData.end(); ++i )
		{
			if ( i->find("GRID") != std::string::npos )
			{
				std::string buldDataLine = *i;
				GridPoint gridPoint;
				char  name[32];
				//std::cout << std::endl << deckLine;
				ReplaceCommasWithSpace(buldDataLine);
				//std::cout << std::endl << deckLine;
				sscanf(buldDataLine.c_str(), "%s %d %d %lf %lf %lf %d", 
												name, 
												&gridPoint.ID, 
												&gridPoint.locationCoordinateSystem_ID, 
												&gridPoint.point.x,
												&gridPoint.point.y,
												&gridPoint.point.z,
												&gridPoint.displacementCoordinateSystem_ID );
				//std::cout << std::endl << gridPoint;
				out_GridPoints_map[gridPoint.ID] = gridPoint;					
			}
		}
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	//void NastranDeck::SetConstraintID( int in_ConstraintID) { constraintID = in_ConstraintID; }
	//void NastranDeck::SetLoadID( int in_LoadID) { loadID = in_LoadID; }
	
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::AppendHeaderComment( const std::string &in_Comment )
	{
		headerComments.push_back(in_Comment);
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::AddSubCaseAndLoadStatement(	int in_SubCaseID, 
													int in_ConstraintSetID, 
													int in_LoadStatementID, 
													int in_LoadSetID )
	{
		std::string tempString;

		tempString = "SUBCASE = " + IntegerToString(in_SubCaseID);
		caseControl.push_back( tempString );

		tempString = "   SPC = " + IntegerToString(in_ConstraintSetID);
		caseControl.push_back( tempString );

		tempString = "   LOAD = " + IntegerToString(in_LoadStatementID);
		caseControl.push_back( tempString );

		tempString = "LOAD," + IntegerToString(in_LoadStatementID) + ",1.,1.," + 
							   IntegerToString(in_LoadSetID);
				
		bulkData.insert( bulkData.begin(), tempString );		//bulkData.push_front( tempString );

	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::AddConstraintToDeck(	int in_ConstraintID, int in_GridID, 
											double in_xDisp, double in_yDisp, double in_zDisp )
	{
		std::string prefixString;
		std::string tempString;
		char buffer_1[64];

		prefixString = "SPC," +  IntegerToString(in_ConstraintID) + 
						"," +    IntegerToString(in_GridID);

		sprintf(buffer_1, "%lf", in_xDisp ); 
		tempString =   prefixString + ",1," +  std::string(buffer_1);
		bulkData.push_back( tempString );

		sprintf(buffer_1, "%lf", in_yDisp ); 
		tempString =   prefixString + ",2," +  std::string(buffer_1);
		bulkData.push_back( tempString );

		sprintf(buffer_1, "%lf", in_zDisp ); 
		tempString =   prefixString + ",3," +  std::string(buffer_1);
		bulkData.push_back( tempString );
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::AddConstraintToDeck(	int in_ConstraintID, 
											int in_GridID, 
											int in_DegreeOfFreedomID, 
											double in_ConstraintValue )
	{
		std::string prefixString;
		std::string tempString;
		char buffer_1[64];
		char buffer_2[64];
		char buffer_3[64];

		prefixString = "SPC," +  std::string(itoa(in_ConstraintID,		 buffer_1,10)) + 
						"," +    std::string(itoa(in_GridID,			 buffer_2,10)) +
						"," +    std::string(itoa(in_DegreeOfFreedomID,  buffer_3,10));

		sprintf(buffer_1, "%lf", in_ConstraintValue ); 
		tempString =   prefixString + "," +  std::string(buffer_1);
		bulkData.push_back( tempString );
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	std::string VectorString( double in_Vector_x, double in_Vector_y, double in_Vector_z)
	{
		std::string tempString;
		char buffer_1[64];

		sprintf(buffer_1, "%lf", in_Vector_x ); 
		tempString =  "," + std::string(buffer_1);

		sprintf(buffer_1, "%lf", in_Vector_y ); 
		tempString +=  "," + std::string(buffer_1);

		sprintf(buffer_1, "%lf", in_Vector_z ); 
		tempString +=  "," + std::string(buffer_1);

		return tempString;
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	std::string ScaleFactorVectorString( double in_ScaleFactor, double in_Vector_x, double in_Vector_y, double in_Vector_z)
	{
		std::string tempString;
		char buffer_1[64];

		sprintf(buffer_1, "%lf", in_ScaleFactor ); 
		tempString =  "," + std::string(buffer_1);

		return tempString + VectorString(in_Vector_x, in_Vector_y, in_Vector_z );
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::AddForceToDeck( int in_LoadID, int in_GridID, int in_CoordinateSystemID, double in_ForceScaleFactor, 
									  double in_Vector_x, double in_Vector_y, double in_Vector_z )
	{
		std::string tempString;


		tempString = "FORCE,"	+ IntegerToString(in_LoadID) +
					 ","		+ IntegerToString(in_GridID) +
					 ","		+ IntegerToString(in_CoordinateSystemID);
		
		tempString +=  ScaleFactorVectorString( in_ForceScaleFactor, in_Vector_x, in_Vector_y, in_Vector_z );
		bulkData.push_back( tempString );


	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
			// Acceleration		GRAV,459,1,333.,0.,-1.,0.
			// GRAV SID CID G N1 N2 N3
			//		SlD Set identification number.
			//		CID Coordinate system identification number.
			//		G Acceleration vector scale factor.
			//		Ni Acceleration vector components measured in coordinate system CID.
	void NastranDeck::AddAccelerationToDeck( int in_LoadID,  int in_CoordinateSystemID, double in_AccelerationScaleFactor, 
											 double in_Vector_x, double in_Vector_y, double in_Vector_z ) 
	{
		std::string tempString;


		tempString = "GRAV,"	+ IntegerToString(in_LoadID) +
					 ","		+ IntegerToString(in_CoordinateSystemID);
		
		tempString +=  ScaleFactorVectorString( in_AccelerationScaleFactor, in_Vector_x, in_Vector_y, in_Vector_z );
		bulkData.push_back( tempString );
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::ReplaceMaterialTokens_ReturnMaterialToComponentID( 
					const std::string								&in_AssemblyComponentID,
					std::map<std::string, std::string>              &in_MaterialKey_ComponentID_map,
					std::map<std::string, double>					&in_ComponentID_PoissonsRatio_map,
					std::map<std::string, std::string>              &out_MaterialID_to_CompnentID_map )
	{
		
		bool aMaterialFound = false;
		for ( std::list<std::string>::iterator i(bulkData.begin()); i != bulkData.end(); ++i )
		{	
			if ( i->find("MAT1") != std::string::npos ) 
			{
				aMaterialFound = true;
				// See "Nastran Linear Static Analysis Guide.pdf"
				char	Mat1[32];	//Mat1[0] = '\0';
				char	MID[32];	//MID[32];
				char	E[32];		//E[32];
				char	G[32];		//G[32];
				char	NU_key[32];  // poisson's ratio key
				char    RHO[32];
				char	A[32];
				char	TREF[32];	TREF[0]		= '\0';  
				char	ST[32];		ST[0]		= '\0';  
				char	MCSID[32];  MCSID[0]	= '\0';  // This is needed becasue sscanf will not populate the last field if it is empty.

				// Read current MAT1 values 
				char buffer[256];
				std::string tempString = *i;
				// Need to replace double commans(i.e. ,,) with space comma space (i.e. , ,).  This is needed becasue scanf will not 
				// scan double commas.
				ReplaceDoubleCommasWithCommaSpaceComma(tempString);

				sscanf(tempString.c_str(), "%[^','],%[^','],%[^','],%[^','],%[^','],%[^','],%[^','],%[^','],%[^','],%[^',']", Mat1, MID, E, G, NU_key, RHO, A, TREF, ST, MCSID);


				std::string ComponentID = in_MaterialKey_ComponentID_map[NU_key];
				double NU_actual = in_ComponentID_PoissonsRatio_map[ComponentID];
				out_MaterialID_to_CompnentID_map[std::string(MID)] = ComponentID;

				//std::cout << std::endl << "-----------------> ComponentID: " << ComponentID;
				//std::cout << std::endl << "-----------------> NU_key, Nu_Actual: " <<  std::setprecision (10)<< NU_key << "  "  <<  std::setprecision (10) << NU_actual;

				sprintf(buffer, "%s,%s,%s,%s,%lf,%s,%s,%s,%s,%s", Mat1, MID, E, G, NU_actual, RHO, A, TREF, ST, MCSID);
				//sprintf(buffer, "%s,%s,%s,%s,%f,%s,%s,%s,%s", Mat1, MID, E, G, componentVistorMaterialTokens.materialKey_MaterialDefintion_map[NU].poissonsRatio, RHO, A, TREF, ST, MCSID);
				tempString = std::string(buffer);
				ReplaceSpacesWithNull(tempString);
				//i->empty();
				//i->assign("Material");
				i->assign(tempString);
			}
			else
			{
				// We are assuming that the materials are group togather in the deck.  Once we have read a material
				// and reach a record that does not beging with "MAT1", then there are no other materials.
				if ( aMaterialFound ) break;
			}

		} // End for ( std::list<std::string>::iterator i(bulkData.begin()); i != bulkData.end(); ++i )
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void NastranDeck::AddCylindricalOrSphericalCoordinateSystem( 
							const Point_3D			&in_PointOrgin,
							const Point_3D			&in_PointZAxis,
							const Point_3D			&in_PointClockingPlane,
							int						in_ReferenceCoordinateSystemID,
							e_CoordinateSystemType	in_CoordinateSystemType,
							int						&out_AddedCoordinateID )
	{
		// Insert coordinate system after the last existing coordinate system
		for (std::list<std::string>::iterator i(bulkData.begin()); i != bulkData.end(); ++i )
		{
			std::string deckLine = *i;
			if ( (deckLine.find("CORD1") != std::string::npos ) ||
			      (deckLine.find("CORD2") != std::string::npos ))
			{
					ReplaceCommasWithSpace(deckLine);
				char  name[32];
				int	  coordinateID;
				//maxCoordinateID
				sscanf(deckLine.c_str(), "%s %d", name, &coordinateID );
				if ( coordinateID >= maxCoordinateID )
				{
					// Need to move past the two line CORD2C.. line 
					std::list<std::string>::iterator j = i;
					++j;
					if ( j !=  bulkData.end()) ++j;  //This would be a malformed deck if at the end here 
					++maxCoordinateID;
					char line_1[82];
					char line_2[82];

					std::string coordinateSystemName = "ERROR";
					if (in_CoordinateSystemType == CYLINDRICAL_COORDINATE_SYSTEM)
						coordinateSystemName = "CORD2C";
					else
						if (in_CoordinateSystemType == SPHERICAL_COORDINATE_SYSTEM)
							coordinateSystemName = "CORD2S";

					sprintf(line_1,	"%s,%d,%d,%lf,%lf,%lf,%lf,%lf,%lf", 
						coordinateSystemName.c_str(),
						maxCoordinateID,
						in_ReferenceCoordinateSystemID,
						in_PointOrgin.x, 
						in_PointOrgin.y, 
						in_PointOrgin.z, 
						in_PointZAxis.x, 
						in_PointZAxis.y, 
						in_PointZAxis.z);

					sprintf(line_2,	",%lf,%lf,%lf", 
						in_PointClockingPlane.x, 
						in_PointClockingPlane.y, 
						in_PointClockingPlane.z);

					bulkData.insert(j,line_1);
					bulkData.insert(j,line_2);

					out_AddedCoordinateID = maxCoordinateID;
					break;

				}
			}	
		}
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	// GRID ID CP X1 X2 X3 CD PS SEID
	// lD Grid point identification number.
	// CP Identification number of coordinate system in which the location of the grid point is
	// defined.
	// X1, X2, X3 Location of the grid point in coordinate system CP.
	// CD Identification number of coordinate system in which the displacements, degrees of
	// freedom, constraints, and solution vectors are defined at the grid point.
	// PS Permanent single-point constraints associated with the grid point.
	// SEID Superelement identification number.
	void NastranDeck::ModifyGridPointsDisplacementCoordinateSystemID( const std::set<int> &in_GridPointIDs,
															int	in_CoordinateSystemID )
	{
		for (std::list<std::string>::iterator i(bulkData.begin()); i != bulkData.end(); ++i )
		{
			if ( i->find("GRID") != std::string::npos )
			{
				std::string buldDataLine = *i;
				GridPoint gridPoint;
				char  name[32];
				//std::cout << std::endl << deckLine;
				ReplaceCommasWithSpace(buldDataLine);
				//std::cout << std::endl << deckLine;
				sscanf(buldDataLine.c_str(), "%s %d %d %lf %lf %lf %d", 
												name, 
												&gridPoint.ID, 
												&gridPoint.locationCoordinateSystem_ID, 
												&gridPoint.point.x,
												&gridPoint.point.y,
												&gridPoint.point.z,
												&gridPoint.displacementCoordinateSystem_ID );

				if ( in_GridPointIDs.find(gridPoint.ID) != in_GridPointIDs.end() )
				{
					char newBuldDataLine[82];
					sprintf(newBuldDataLine, "%s,%d,%d,%lf,%lf,%lf,%d", 
												name, 
												gridPoint.ID, 
												gridPoint.locationCoordinateSystem_ID, 
												gridPoint.point.x,
												gridPoint.point.y,
												gridPoint.point.z,
												in_CoordinateSystemID );

					*i = std::string(newBuldDataLine);
					
				}
			}
		}

	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////

	std::ostream& operator<<(std::ostream &output, const NastranDeck &in_NastranDeck )
	{
		//output << std::endl << "Nastran Deck";

		bool firstLineWritten = false;
		// Header Comments
		for ( std::list<std::string>::const_iterator i(in_NastranDeck.headerComments.begin()); 
			  i != in_NastranDeck.headerComments.end();
			  ++i )
		{	
			if ( firstLineWritten )
			{
				output << std::endl << *i;
			}
			else
			{
				output << *i;
				firstLineWritten = true;
			}
		}

		// Executive Control
		for ( std::list<std::string>::const_iterator i(in_NastranDeck.executiveControl.begin()); 
			  i != in_NastranDeck.executiveControl.end();
			  ++i )
		{	
			output << std::endl << *i;
		}
		output << std::endl << in_NastranDeck.endExecutiveControl;

		// Case Control
		for ( std::list<std::string>::const_iterator i(in_NastranDeck.caseControl.begin()); 
			  i != in_NastranDeck.caseControl.end();
			  ++i )
		{	
			output << std::endl << *i;
		}

		// Bulk Data
		output << std::endl << in_NastranDeck.beginBulkData;
		for ( std::list<std::string>::const_iterator i(in_NastranDeck.bulkData.begin()); 
			  i != in_NastranDeck.bulkData.end();
			  ++i )
		{	
			output << std::endl << *i;
		}
		output << std::endl << in_NastranDeck.endBulkData;

		return output;
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////

	std::ostream& operator<<(std::ostream &output, const std::map<int, GridPoint> &in_gridPoints_map )
	{
		for ( std::map<int, GridPoint>::const_iterator i(in_gridPoints_map.begin()); i !=in_gridPoints_map.end(); ++i )
		{
			output << i->second;
		}
		return output;
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////

	void NastranDeck::GetExecutiveControl(CommonNastranDS &out_execCtrl) throw (isis::application_exception)
	{
		for ( std::list<std::string>::iterator i(executiveControl.begin()); i != executiveControl.end(); ++i )
		{	
			std::string data = *i;
			if (data != "")
			{
				if (data[0] != '$')
				{
					std::vector<std::string> tokens;
					isis_CADCommon::tokenize_strtok(tokens, data, " ");

					if (tokens.empty())
						throw isis::application_exception(std::string("Problem parsing ExecutiveControl statements - Line:" + data).c_str());	

					if (tokens.size() >= 2)
					{
						if (tokens[0] == "SOL")
							out_execCtrl.solverType = (tokens[1] == "SESTATIC") ? "101" : tokens[1];
						else if (tokens[0] == "TIME")
							out_execCtrl.time = atoi(tokens[1].c_str());

					}
					else
						std::cout << "WARNING: ExecutiveControl Parsing [" << data << "]" << std::endl;
				}
			}
		}
	}
	
	void NastranDeck::GetCaseControl(CaseControl &out_caseCtrl) throw (isis::application_exception)
	{
		bool subcase = 0;	
		std::vector<std::string> subcaseStr;
		//int size = caseControl.size();
		//for ( int i = 0; i < size;)
		std::list<std::string>::const_iterator zi = caseControl.end();
		for (std::list<std::string>::const_iterator yi = caseControl.begin(); yi != zi;)
		{
			std::string data = *yi, card, outputConfig;
			std::vector<std::string> tokens;
			isis_CADCommon::tokenize_strtok(tokens, data, " =");
			if (tokens.size() != 2)
				throw isis::application_exception(std::string("Problem parsing CaseControl statements - Line:" + data).c_str());

			card = tokens[0];
			outputConfig = tokens[1];

			if (card == "SUBCASE")		// start of subcase
			{		
				std::vector<std::pair<std::string, std::string>> sss;
				yi++;
				while (yi != zi)				// not end of caseControl vector
				{
					std::string line = *yi;
					if (line.find("SUBCASE") == std::string::npos)
					{
						std::vector<std::string> tokens_subcase;
						tokenize_strtok(tokens_subcase, line, " =");
						if (tokens_subcase.size() != 2)
							throw isis::application_exception(std::string("Subcase statements - Keyword has no value! -  Line:" + line).c_str());

						sss.push_back(make_pair(tokens_subcase[0], tokens_subcase[1]));
						tokens_subcase.clear();

						yi++;
					}
					else
						break;
				}
				out_caseCtrl.subcases[outputConfig] = sss;
			}
			else
			{	
				tokens.clear();
				tokenize_strtok(tokens, card, "()");
				out_caseCtrl.commands[tokens[0]] = make_pair((tokens.size() > 1 ? tokens[1] : ""), outputConfig);				
				yi++;
			}

		}

	}

	void NastranDeck::GetBulkData(CommonNastranDS &bulkDataStructure)
	{
		//int size = bulkData.size();
		//for (int i = 0; i < size; i++)
		std::list<std::string>::const_iterator zi = bulkData.end(), yi = bulkData.begin();
		for (; yi != zi ; yi++)
		{
			std::string data = *yi;
			if (data == "")
				continue;
			if (data[0] != '$' || data[0] != ',')	// not a comment or extention line
			{
				std::vector<std::string> tokens;
				std::string card = data.substr(0, data.find_first_of(','));
				ReplaceDoubleCommasWithCommaSpaceComma(data);
				if (card == "MAT1")
				{
					Material material;
					stringTokenize(data.c_str(), tokens, ",");		//tokenize(data)
					if (tokens.size() != 9)
					{
						std::cout << "MAT1: Not enough entries, need to be 9 [" << data << "]" << std::endl;
						continue;
					}
					material.name = tokens[0];
					material.MID = atoi(tokens[1].c_str());
					material.Young = tokens[2];
					material.Shear = tokens[3];
					material.Poisson = tokens[4];
					material.Density = tokens[5];
					material.Thermal = tokens[6];
					material.RefTemp = (tokens[7] == " " ? "0": tokens[7]);
					material.Damping = tokens[8];
					bulkDataStructure.materialData[material.MID] = material;
				}
				else if (card == "LOAD")
				{
					//Load load;
					//stringTokenize(data.c_str(), tokens, ",");		//tokenize(data)

				}
				else if (card == "CORD2R" || card == "CORD2C" || card == "CORD2S")							// 2 line
				{
					bool global = 0, grid = 0, disp = 0;

					if (yi != bulkData.begin())
					{
						std::list<std::string>::const_iterator xi = yi;
						xi--;
						std::string coordComment = *xi;

						if (coordComment.find("Global Coordinate") != std::string::npos)
							global = 1;
						else if (coordComment.find("grid coordinates") != std::string::npos)
							grid = 1;
						else if (coordComment.find("grid displacement") != std::string::npos)
							disp = 1;
					}

					yi++;
					data += *yi;
					stringTokenize(data.c_str(), tokens, ",");
					if (tokens.size() < 12)
					{
						std::cout<< "CORD2R: <12 entries [" << data << "]" << std::endl;
						continue;
					}

					CoordSystem coord;
					coord.CID = atoi(tokens[1].c_str());
					coord.RID = atoi(tokens[2].c_str());
					coord.A.x = atof(tokens[3].c_str());
					coord.A.y = atof(tokens[4].c_str());
					coord.A.z = atof(tokens[5].c_str());
					coord.B.x = atof(tokens[6].c_str());
					coord.B.y = atof(tokens[7].c_str());
					coord.B.z = atof(tokens[8].c_str());
					coord.C.x = atof(tokens[9].c_str());
					coord.C.y = atof(tokens[10].c_str());
					coord.C.z = atof(tokens[11].c_str());
					coord.system = card;

					bulkDataStructure.coordSystems[coord.CID] = coord;

					if (global)
						bulkDataStructure.globalCord_ID = coord.CID;
					else if (grid)
						bulkDataStructure.gridCord_ID = coord.CID;
					else if (disp)
						bulkDataStructure.gridDisplacementCord_ID = coord.CID;
				}
				else if (card == "PSOLID")		
				{
					stringTokenize(data.c_str(), tokens, ",");		//tokenize(data)

					if (tokens.size() < 3)
					{
						std::cout<< "PSOLID: <3 entries [" << data << "]" << std::endl;
						continue;
					}
					PSolid psolid;
					psolid.PID = atoi(tokens[1].c_str());
					psolid.MID = atoi(tokens[2].c_str());
					bulkDataStructure.psolidData[psolid.PID] = psolid;
				}
				else if (card == "GRID")
				{
					stringTokenize(data.c_str(), tokens, ",");		//tokenize(data)
					if (tokens.size() < 7)
					{
						std::cout<< "GRID: <7 entries [" << data << "]" << std::endl;
						continue;
					}
					GridPoint grid;
					grid.ID = atoi(tokens[1].c_str());
					grid.locationCoordinateSystem_ID = atoi(tokens[2].c_str());
					grid.point.x = atof(tokens[3].c_str());
					grid.point.y = atof(tokens[4].c_str());
					grid.point.z = atof(tokens[5].c_str());
					grid.displacementCoordinateSystem_ID = atoi(tokens[6].c_str());

					bulkDataStructure.gridPointData[grid.ID] = grid;
					bulkDataStructure.displaceCord_Elements_KEYS.insert(grid.displacementCoordinateSystem_ID);
					bulkDataStructure.displaceCord_Elements.insert(std::make_pair(grid.displacementCoordinateSystem_ID, grid.ID));
				}
				else if (card == "CTETRA")							// 2 line
				{
					yi++;
					if (yi != zi)
					{
						data += *yi;
						stringTokenize(data.c_str(), tokens, ",");			// tokenize(data)
					}

					if (tokens.size() < 13)
					{
						std::cout << "CTETRA: [ " << data << "]" << std::endl;
						continue;
					}

					SolidElement element;
					element.EID = atoi(tokens[1].c_str());
					element.PID = atoi(tokens[2].c_str());
					element.Type = CTETRA;
					element.GID.assign(tokens.begin() + 3, tokens.end());
					bulkDataStructure.elementData.insert(std::make_pair(element.PID, element));
					bulkDataStructure.TetraCount++;
				}
				else if (card == "CPENTA")							// 3 lines
				{
					yi++;
					for (int j = 1; j < 3 && yi != zi; j++, yi++)
						data += *yi;
			
					stringTokenize(data.c_str(), tokens, ",");			// tokenize(data)

					if (tokens.size() < 18)
					{											
						std::cout << "CPENTA: [ " << data << "]" << std::endl;
						continue;
					}
					SolidElement element;
					element.EID = atoi(tokens[1].c_str());
					element.PID = atoi(tokens[2].c_str());
					element.Type = CPENTA;
					element.GID.assign(tokens.begin() + 3, tokens.end());
					bulkDataStructure.elementData.insert(std::make_pair(element.PID, element));
					bulkDataStructure.PentaCount++;
				}
				else if (card == "CHEXA")							// 3 lines
				{
					yi++;
					for (int j = 1; j < 3 && yi != zi; j++, yi++)
						data += *yi;
		
					stringTokenize(data.c_str(), tokens, ",");			// tokenize(data)

					if (tokens.size() < 23)
					{						
						std::cout << "CHEXA: [ " << data << "]" << std::endl;
						continue;
					}
					
					SolidElement element;
					element.EID = atoi(tokens[1].c_str());
					element.PID = atoi(tokens[2].c_str());
					element.Type = CHEXA;
					element.GID.assign(tokens.begin() + 3, tokens.end());
					bulkDataStructure.elementData.insert(std::make_pair(element.PID, element));
					bulkDataStructure.HexaCount++;
				}
				else if (card == "SPC")		
				{
					stringTokenize(data.c_str(), tokens, ",");		// tokenize(data);
					if (tokens.size() < 5)
					{
						std::cout<< "SPC: Wrong number of entries [" << data << "]" << std::endl;
						continue;
					}

					SPC spc;
					spc.SID = atoi(tokens[1].c_str());
					spc.G1 = tokens[2];
					spc.C1 = tokens[3];
					spc.D1 = tokens[4];
					bulkDataStructure.spcData.insert(std::make_pair(spc.SID, spc));
				}
				else if (card == "FORCE")	
				{
					stringTokenize(data.c_str(), tokens, ",");		// tokenize(data);
					if (tokens.size() < 8)
					{
						std::cout<< "FORCE: Wrong number of entries [" << data << "]" << std::endl;
						continue;
					}

					Force force;
					force.SID = atoi(tokens[1].c_str());
					force.GID = atoi(tokens[2].c_str());
					force.CID = atoi(tokens[3].c_str());
					force.F = tokens[4];
					force.N1 = tokens[5];
					force.N2 = tokens[6];
					force.N3 = tokens[7];
					bulkDataStructure.forceLoadData.insert(std::make_pair(force.SID, force));
				}
				else if (card == "PARAM")
				{

				}
				else
					std::cout<< "GetBulkData(): Unsupported card found [" << data << "]" << std::endl;
			} // if
		} // for

		if (bulkDataStructure.HexaCount > 0)
			bulkDataStructure.ElementTypeCount++;
		if (bulkDataStructure.PentaCount > 0)
			bulkDataStructure.ElementTypeCount++;
		if (bulkDataStructure.TetraCount > 0)
			bulkDataStructure.ElementTypeCount++;
	}

	void NastranDeck::CreateCommonNastranDS(CommonNastranDS	&nastranDS)
	{
		GetExecutiveControl(nastranDS);
		GetCaseControl(nastranDS.caseControlData);
		GetBulkData(nastranDS);
	}

	/******************************************************************
					Helper sub-parsing functions
	/******************************************************************/
	void NastranDeck::ParseSubcase(std::vector<std::string> &subcaseStr, 
											SubcaseStatements &subcaseOptions) 
											throw (isis::application_exception)
	{
		size_t size = subcaseStr.size();
		for (int i = 0; i < size; i++)
		{
			std::vector<std::string> tokens;
			tokenize_strtok(tokens, subcaseStr[i], " =");
			if (tokens.size() != 2)
				throw isis::application_exception(std::string("Subcase statements - Keyword has no value! -  Line:" + subcaseStr[i]).c_str());
			subcaseOptions[tokens[0]] = tokens[1];
			tokens.clear();
		}
	}
} // End namespace isis