/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#ifndef NASTRAN_H
#define NASTRAN_H

#include <list>
#include <string>
#include <iostream>
#include <map>
#include <set>
#include <vector>
#include "isis_application_exception.h"
#include "GraphicsFunctions.h"

namespace isis_CADCommon
{

	struct CoordSystem
	{		
	
		//enum CoordType {
		//	CORD2R,
		//	CORD2C,
		//	CORD2S
		//}; 
		
		
		std::string system;
		int CID;
		int RID;
		Point_3D A;
		Point_3D B;
		Point_3D C;		
	};

	struct Material
	{
		int MID;
		std::string name;
		std::string Young;
		std::string Shear;
		std::string Poisson;
		std::string Density;
		std::string Thermal;
		std::string RefTemp;
		std::string Damping;
	};

	enum NastranCardType
	{
		DISPLACEMENT,
		SPCFORCES,
		FORCE,
		OLOAD,
		STRESS,
		STRAIN,
		ESE,
	};
	
	typedef std::map<std::string, std::string> SubcaseStatements;			// keyword, value
	struct CaseControl
	{
		// key, pair<option, value>
		std::map<std::string, std::pair<std::string, std::string>> commands;

		// subcaseID, map<keyword, value>
		//std::map<std::string, SubcaseStatements> subcases;
		std::map<std::string, std::vector<std::pair<std::string, std::string>>> subcases;
	};
	
	struct PSolid
	{
		int PID;
		int MID;
		int Cordm;
		std::string IN;
		std::string Stress;
		std::string Isop;
		std::string Fctn;
	};

	enum ElementType 
	{
		CHEXA,
		CPENTA,
		CTETRA
	};

	struct SolidElement
	{
		ElementType Type;
		int EID;
		int PID;
		std::vector<std::string> GID;
	};

	struct SPC
	{
		int SID;
		std::string G1;
		std::string C1;
		std::string D1;
	};

	struct Force
	{
		int SID;
		int GID;
		int CID;
		std::string F;
		std::string N1;
		std::string N2;
		std::string N3;
	};

	struct Load
	{
		int SID;
		std::string ScaleFactor;
		std::vector<std::map<std::string, std::string>> ScaleLoad_Pairs;		// map<Scale, LoadID>
	};

	struct CommonNastranDS
	{
		CommonNastranDS(std::string fileName):
			Name(fileName),
			TetraCount(0),
			PentaCount(0),
			HexaCount(0),
			ElementTypeCount(0){}
					
		std::string Name;


		//***** Executive Control *****
		std::string solverType;
		int time;
		//*****************************
		



		//******** Case Control *******
		CaseControl caseControlData;
		//*****************************


		//********* Bulk Data *********
		std::map<int, Material>						materialData;				// MAT1 materials, others not supported right now
		std::map<int, GridPoint>					gridPointData;				// grid points
		std::map<int, PSolid>						psolidData;					// PSolids: <PID, psolid>
		std::multimap<int, SolidElement>			elementData;				// elements: <PID, element>

		std::map<int, CoordSystem>					coordSystems;

		std::multimap<int, SPC>						spcData;					// SPC
		std::multimap<int, Force>					forceLoadData;				// Force

		// Book-keeping Stuff
		int globalCord_ID;
		int gridCord_ID;
		int gridDisplacementCord_ID;

		std::multimap<int, int>						displaceCord_Elements;		// <dispCord_ID, Grid_ID>
		std::set<int>								displaceCord_Elements_KEYS;


		int TetraCount;
		int PentaCount;
		int HexaCount;
		int ElementTypeCount;
		//*******************


	};
	//////////////////////////////////////////
	// class NastranDeck
	//////////////////////////////////////////

	class NastranDeck
	{
		private:
			std::list<std::string>		headerComments;	

			std::list<std::string>		executiveControl;
			std::string					endExecutiveControl;

			std::list<std::string>		caseControl;

			std::string					beginBulkData;
			std::list<std::string>		bulkData;
			std::string					endBulkData;

			int							maxCoordinateID;

		public:

			NastranDeck();

			//////////////////////////////
			// Initialize/Load Deck
			//////////////////////////////
			void ReadNastranDeck( const std::string &in_InputFileName ) throw (isis::application_exception);

			//////////////////////////////
			// Get Functions
			//////////////////////////////
			void GetGridPoints( std::map<int, GridPoint>  &out_GridPoints_map);
			// void GetMetaHeaderComments( 
			// void GetSolutionType   // Problem should use Nastran types
			// void GetSubCase ....
			//void GetMaterialProperties ( std::map<int, Material> &out_Material_map ) throw (isis::application_exception);
			// void GetCoordinateSystems
			// void GetSolidToMaterialMap
			// void GetTetras10Pt (std::map<int, ????>  &out_Tetras10Pt_map);
			void GetExecutiveControl(CommonNastranDS &out_execCtrl) throw (isis::application_exception);
			void GetCaseControl(CaseControl &out_classCtrl) throw (isis::application_exception);

			int	 getMaxCoordinateID();

			//////////////////////////////
			// Write/Stream Functions
			//////////////////////////////
			void WriteNastranDeck( const std::string &in_OutputFileName ) throw (isis::application_exception);
			friend std::ostream& operator<<(std::ostream &output, const NastranDeck &in_NastranDeck );

			//////////////////////////////
			// Edit Functions
			//////////////////////////////
			void AppendHeaderComment( const std::string &in_Comment );

			// Note this adds a Subcase and Load entires
			// SUBCASE = 0
			//		SPC = 259
			//		LOAD = 559
			// LOAD,559,1.,1.,59
			// LOAD SID S S1 L1 S2 L2 S3 L3
			//		SID Load set identification number. (Integer > 0)
			//		S Overall scale factor. (Real)
			//		Si Scale factor on Li. (Real)
			//		Li Load set identification numbers defined on entry types listed above. (Integer > 0)
			void AddSubCaseAndLoadStatement( int in_SubCaseID, int in_ConstraintSetID, int in_LoadStatementID, int in_LoadSetID );

			// Constraint		SPC,259,6825,1,0.
			// SPC SID G1 C1 D1 G2 C2 D2
			//		SID Identification number of the single-point constraint set.
			//		Gi Grid or scalar point identification number.
			//		Ci Component number
			//		Di Value of enforced displacement for all degrees of freedom designated by Gi and Ci.
			void AddConstraintToDeck( int in_ConstraintID, int in_GridID, double in_xDisp, double in_yDisp, double in_zDisp ); 

			// For Cylindrical Coordinate System
			// Degree of Freedon	Direction
			// -----------------	----------
			//		1				r - radial
			//	`	2				theta - rotational
			//		3				z - axis
			//
			// For Sperical Coordinate System
			// Degree of Freedon	Direction
			// -----------------	----------
			//		1				r - radial
			//	`	2				theta - rotational
			//		3				z - axis
			void AddConstraintToDeck( int in_ConstraintID, int in_GridID, int in_DegreeOfFreedomID, double in_ConstraintValue ); 

			// FORCE SID G CID F N1 N2 N3
			// FORCE,59,1706,1,1.0,-0.0583861,0.0,0.0
			//		SID Load set identification number.
			//		G Grid point identification number
			//		CID Coordinate system identification number.
			//		F Scale factor.	
			//		Ni Components of a vector measured in coordinate system defined by CID.
			// Note in_LoadID must be the same in_LoadID in the function call AddSubCaseAndLoadStatement(...)
			void AddForceToDeck( int in_LoadID, int in_GridID, int in_CoordinateSystemID, double in_ForceScaleFactor, 
								 double in_Vector_x, double in_in_Vector_y, double in_in_Vector_z ); 

			// Acceleration		GRAV,459,1,333.,0.,-1.,0.
			// GRAV SID CID G N1 N2 N3
			//		SlD Set identification number.
			//		CID Coordinate system identification number.
			//		G Acceleration vector scale factor.
			//		Ni Acceleration vector components measured in coordinate system CID.
			// GRAV,459,1,333.,0.,-1.,0.
			void AddAccelerationToDeck( int in_LoadID, int in_CoordinateSystemID, double in_AccelerationScaleFactor, 
										double in_Vector_x, double in_Vector_y, double in_Vector_z ); 

			// Pressure			PLOAD4,59,1875,14904.,14904.,14904.,,355,1221
			// PLOAD4 SID EID P1 P2 P3 P4 G1 G3 or G4 
			//		  CID N1 N2 N3 SORL LDIR
			//		SID Load set identification number
			//		P Pressure
			//		Gi Grid point identification numbers
			// PLOAD4,59,5877,14904.,14904.,14904.,,418,1416
			// ??? Presure loads require retriving elements,  should do this later 
			///void AddPressureToDeck( ...); 


			// MAT1 MID E G NU RHO A TREF GE ST SC SS MCSID
			// MID Material identification number
			// E Youngs modulus
			// G Shear modulus
			// NU Poissons ratio
			// RHO Mass density
			// A Thermal expansion coefficient
			// TREF Reference temperature
			// ST, SC, SS Stress limits for tension, compression, and shear
			// MCSID Material coordinate system identification number
			void ReplaceMaterialTokens_ReturnMaterialToComponentID( 
					const std::string								&in_AssemblyComponentID,
					std::map<std::string, std::string>              &in_MaterialKey_ComponentID_map,
					std::map<std::string, double>					&in_ComponentID_PoissonsRatio_map,
					std::map<std::string, std::string>              &out_MaterialID_to_CompnentID_map );

			enum e_CoordinateSystemType
			{
				// RECTANGULAR_COORDINATE_SYSTEM,  Don't allow this 
				CYLINDRICAL_COORDINATE_SYSTEM,
				SPHERICAL_COORDINATE_SYSTEM
			};
			// CORD2C,4,2,2239.38,1094.41,870.,2239.38,1093.41,870.,
			// ,2240.38,1094.41,870.
			// CORD2C CID RID A1 A2 A3 B1 B2 B3
			// C1 C2 C3
			// CID Coordinate system identification number.
			// RID Identification number of a coordinate system that is defined independently from
			// this coordinate system.
			// Ai, Bi, Ci Coordinates of three points in coordinate system defined in field 3.
			//
			// CORD2S CID RID A1 A2 A3 B1 B2 B3
			// C1 C2 C3
			// CID Coordinate system identification number.
			// RID Identification number of a coordinate system that is defined independently from this
			// coordinate system
			// Ai, Bi, Ci Coordinates of three points in coordinate system defined in field 3.
			void AddCylindricalOrSphericalCoordinateSystem( 
									const Point_3D &in_PointOrgin,
									const Point_3D &in_PointZAxis,
									const Point_3D &in_PointClockingPlane,
									int			in_ReferenceCoordinateSystemID,
									e_CoordinateSystemType	in_CoordinateSystemType,
									int			&out_AddedCoordinateID );

			// GRID ID CP X1 X2 X3 CD PS SEID
			// lD Grid point identification number.
			// CP Identification number of coordinate system in which the location of the grid point is
			// defined.
			// X1, X2, X3 Location of the grid point in coordinate system CP.
			// CD Identification number of coordinate system in which the displacements, degrees of
			// freedom, constraints, and solution vectors are defined at the grid point.
			// PS Permanent single-point constraints associated with the grid point.
			// SEID Superelement identification number.
			void ModifyGridPointsDisplacementCoordinateSystemID( const std::set<int> &in_GridPointIDs,
																 int	in_CoordinateSystemID );


			void GetBulkData(CommonNastranDS&);

			void CreateCommonNastranDS(CommonNastranDS&);		// main function to populate CommonNastranDS


			/******************************************************************
					Helper sub-parsing functions
			/******************************************************************/
			void ParseSubcase(
					std::vector<std::string>				&subcaseStr, 
					SubcaseStatements						&subcaseOptions)
					throw (isis::application_exception);
	};

	std::ostream& operator<<(std::ostream &output, const std::map<int, GridPoint> &in_gridPoints_map );

} // End namespace isis

#endif 