/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CADSoftwareEnvirUtils.h"
#include <CommonUtilities.h>
#include <windows.h>
#include "WindowsFunctions.h"
//#include <malloc.h>
//#include <iostream>
//#include <vector>
//#include <tchar.h>
#include <algorithm>

// #define BUFFER 8192

namespace isis
{		
	const int MAX_KEY_LENGTH =	256;
	//const int MAX_VALUE_NAME  = 16383;  // See http://msdn.microsoft.com/en-us/library/windows/desktop/ms724872(v=vs.85).aspx
	const int MAX_VALUE_NAME  = 2048;   // Note - Set to 2048 because the values we are searching for in
										// HKEY_LOCAL_MACHINE\SOFTWARE\PTC\Creo Parametric\1.0\2011190
										// are typically just MAX_PATH.  Allocated the larger size for safety.  

	// Return empty string if registry value was not found.
	// WARNING - Do not use this function for values larger that 2048.
	//////////////////////////////////////////////////////////////////////////////////////////////// 
	std::string RetrieveRegistryStringValue( const HKEY in_hkey, const std::string &in_Key )
	{
		std::string tempString = "";

		DWORD	pdwType;
		char	pvData[MAX_VALUE_NAME];
		DWORD	pcbData = MAX_VALUE_NAME;

		LSTATUS errorStatus = 
			RegGetValueA(	in_hkey,		// A handle to an open registry key
							NULL,			// subkey of the key specified by the hkey parameter.
							in_Key.c_str(), // The name of the registry value. 
							RRF_RT_REG_SZ,  // Flags that restrict the data type of value to be queried.  RRF_RT_REG_SZ only strings.
							&pdwType,		// A pointer to a variable that receives a code indicating the type of data stored
							pvData,			// A pointer to a buffer that receives the value's data. 
							&pcbData  );		// A pointer to a variable that specifies the size of the buffer pointed to by the pvData 
		
		if ( errorStatus == ERROR_SUCCESS ) tempString = (char *) pvData;

		return tempString;
	}

	//////////////////////////////////////////////////////////////////////////////////////////////// 
	// out_SubKeys will be empty of no sub-keys were found
	void RetrieveRegistryListOfSubkeys( const HKEY in_hkey, std::vector<std::string> &out_SubKeys )
	{
		// The following code is from http://msdn.microsoft.com/en-us/library/ms724256(VS.85).aspx

		CHAR    achKey[MAX_KEY_LENGTH];   // buffer for subkey name
		DWORD    cbName;                   // size of name string 
		CHAR     achClass[MAX_PATH] = "";  // buffer for class name 
		DWORD    cchClassName = MAX_PATH;  // size of class string 
		DWORD    cSubKeys=0;               // number of subkeys 
		DWORD    cbMaxSubKey;              // longest subkey size 
		DWORD    cchMaxClass;              // longest class string 
		DWORD    cValues;                  // number of values for key 
		DWORD    cchMaxValue;              // longest value name 
		DWORD    cbMaxValueData;           // longest value data 
		DWORD    cbSecurityDescriptor;     // size of security descriptor 
		FILETIME ftLastWriteTime;          // last write time 
 
		DWORD i, retCode; 
 
		//WCHAR  achValue[MAX_VALUE_NAME]; 
		//DWORD cchValue = MAX_VALUE_NAME; 
 
		// Get the class name and the value count. 
		retCode = RegQueryInfoKeyA(
			in_hkey,                    // key handle 
			achClass,                // buffer for class name 
			&cchClassName,           // size of class string 
			NULL,                    // reserved 
			&cSubKeys,               // number of subkeys 
			&cbMaxSubKey,            // longest subkey size 
			&cchMaxClass,            // longest class string 
			&cValues,                // number of values for this key 
			&cchMaxValue,            // longest value name 
			&cbMaxValueData,         // longest value data 
			&cbSecurityDescriptor,   // security descriptor 
			&ftLastWriteTime);       // last write time 
 
		// Enumerate the subkeys, until RegEnumKeyEx fails.
    
		if (cSubKeys)
		{
			//printf( "\nNumber of subkeys: %d\n", cSubKeys);

			for (i=0; i<cSubKeys; i++) 
			{ 
				cbName = MAX_KEY_LENGTH;
				retCode = RegEnumKeyExA(in_hkey, i,
						 achKey, 
						 &cbName, 
						 NULL, 
						 NULL, 
						 NULL, 
						 &ftLastWriteTime); 
				if (retCode == ERROR_SUCCESS) 
				{
					//wprintf(TEXT("(%d) %s\n"), i+1, achKey);
					out_SubKeys.push_back(achKey);
				}
			}
		} 

	} 



	//////////////////////////////////////////////////////////////////////////////////////////////// 
	//  Registry Hierarchy
	//	HKEY_LOCAL_MACHINE
	//		SOFTWARE
	//			PTC
	//				Creo Parametric
	//					1.0    // Note - this could be 1.1, 1.2 should pick the highest 1.x number, Note 2.x not supported yet
	//						2011190  // Year build number,  this could be many numbers, should pick the highest number
	//							InstallDir // e.g. C:\Program Files\PTC\Creo 1.0\Parametric
	//							CommonFilesLocation // e.g. C:\Program Files\PTC\Creo 1.0\Common Files\F000							
	//
	//	in_SupportedVersionPrefixes -			1.  2.  // This means 1.x, 2.x, but only enter 1., 2. 
	//  in_SupportedVersionString_ForErrorMsg - 1.x, 2.x, or 3.x  // currently only support 1.x
	//	out_CreoParametricInstallPath -			This should be short name.  e.g. C:\Progra~1\PTC\CREO1~1.0\PARAME~1\
	//	out_CreoParametricCommMsgExe -			This should be long name.   e.g. C:\Program Files\PTC\Creo 1.0\Common Files\F000\x86e_win64\obj\pro_comm_msg 

	void RetrieveFromRegistryCreoInstallLocations( 
								const std::vector<std::string> &in_SupportedVersionPrefixes,
								const std::string			   &in_SupportedVersionString_ForErrorMsg,
								std::string &out_CreoParametricInstallPath,
								std::string &out_CreoParametricCommMsgExe ) throw (isis::application_exception)
	{
		std::string key_Creo_Parametric = "SOFTWARE\\PTC\\Creo Parametric";

		std::string errorMsg_1 =  "Creo Parametric is not installed.  Please install Creo " + in_SupportedVersionString_ForErrorMsg + ". Note - Higher versions are currently not supported.  ";
		std::string errorMsg_2 =  "Creo Parametric is not installed or the registry information for the install is incorrect or the format of the registry information has changed.  If not installed, please install Creo " + in_SupportedVersionString_ForErrorMsg + ". Note - Higher versions are currently not supported.  ";

		std::string key_Temp;
		//////////////////////////////////////////////////////////////////////
		// Retrieve Key to HKEY_LOCAL_MACHINE\SOFTWARE\PTC\Creo Parametric
		/////////////////////////////////////////////////////////////////////
		HKEY hKey;
		if ( RegOpenKeyExA(HKEY_LOCAL_MACHINE,
			 key_Creo_Parametric.c_str(),
			 0, KEY_READ|KEY_WOW64_64KEY, &hKey) != ERROR_SUCCESS)
		{
			std::string TempError =   errorMsg_1 +
				std::string("Could not find registry key HKEY_LOCAL_MACHINE\\") + key_Creo_Parametric; 
			throw isis::application_exception(TempError.c_str());
		}
		///////////////////////////////////////////////////////////////////
		// Get Subkeys of HKEY_LOCAL_MACHINE\SOFTWARE\PTC\Creo Parametric
		//////////////////////////////////////////////////////////////////
		std::vector<std::string> subKeys_vec;
		RetrieveRegistryListOfSubkeys( hKey, subKeys_vec );

		std::vector<std::string> versionNumber_vec;
		for ( std::vector<std::string>::const_iterator i(subKeys_vec.begin()); i != subKeys_vec.end(); ++i )
		{
			//std::cout << std::endl << *i;
			for ( std::vector<std::string>::const_iterator j(in_SupportedVersionPrefixes.begin()); j != in_SupportedVersionPrefixes.end(); ++ j )
			{
				if ( i->find(*j) != i->front())
				{
					versionNumber_vec.push_back(*i);
					break;
				} 
			}
		}

		if ( versionNumber_vec.size() == 0 )
		{
			std::string TempError =   errorMsg_2 + 
				std::string("Could not find registry keys subordinate to HKEY_LOCAL_MACHINE\\") + key_Creo_Parametric; 
			throw isis::application_exception(TempError.c_str());
		}

		// Sort the keys, so that the highest key (highest Creo Version) could be selected.
		std::sort( versionNumber_vec.begin(), versionNumber_vec.end() );

		key_Temp = key_Creo_Parametric + "\\" + versionNumber_vec[versionNumber_vec.size() - 1 ];

		//std::cout << std::endl << key_Temp;

		////////////////////////////////////////////////////////////////////////////////
		// Retrieve next Key (e.g. HKEY_LOCAL_MACHINE\SOFTWAREPTC\Creo Parametric\1.0 )
		////////////////////////////////////////////////////////////////////////////////
		if ( RegOpenKeyExA(HKEY_LOCAL_MACHINE,
			 key_Temp.c_str(),
			 0, KEY_READ|KEY_WOW64_64KEY, &hKey) != ERROR_SUCCESS)
		{
			std::string TempError =   errorMsg_2 +
				std::string("Could not find key HKEY_LOCAL_MACHINE\\") + key_Temp; 
			throw isis::application_exception(TempError.c_str());
		}
		subKeys_vec.empty();

		/////////////////////////////////////////////////////////////////////////////////////////////
		// Get Subkeys of HKEY_LOCAL_MACHINE\SOFTWARE\PTC\Creo Parametric\1.0   could be 1.1, 2.0 ...
		/////////////////////////////////////////////////////////////////////////////////////////////
		RetrieveRegistryListOfSubkeys( hKey, subKeys_vec );

		if ( subKeys_vec.size() == 0 )
		{
			std::string TempError =  errorMsg_2 + 
				std::string("Could not find registry keys subordinate to HKEY_LOCAL_MACHINE\\") + key_Temp; 
			throw isis::application_exception(TempError.c_str());
		}


		//for ( std::vector<std::string>::const_iterator i(subKeys_vec.begin()); i != subKeys_vec.end(); ++i )
		//{
		//	std::cout << std::endl << *i;
		//}

		// Sort the keys, so that the highest key (highest Creo Version) could be selected.
		std::sort( subKeys_vec.begin(), subKeys_vec.end() );

		key_Temp = key_Temp + "\\" + subKeys_vec[subKeys_vec.size() - 1 ];

		//std::cout << std::endl << key_Temp;

		////////////////////////////////////////////////////////////////////////////////
		// Retrieve next Key (e.g. HKEY_LOCAL_MACHINE\SOFTWAREPTC\Creo Parametric\1.0\2011109 )
		////////////////////////////////////////////////////////////////////////////////
		if ( RegOpenKeyExA(HKEY_LOCAL_MACHINE,
			 key_Temp.c_str(),
			 0, KEY_READ|KEY_WOW64_64KEY, &hKey) != ERROR_SUCCESS)
		{
			std::string TempError =   errorMsg_2 +
				std::string("Could not find key HKEY_LOCAL_MACHINE\\") + key_Temp; 
			throw isis::application_exception(TempError.c_str());
		}


		////////////////////////////////
		// Get CommonFilesLocation
		////////////////////////////////
		std::string commonFilesLocation = RetrieveRegistryStringValue( hKey, "CommonFilesLocation" );

		if ( commonFilesLocation.size() == 0 )
		{
			std::string TempError =   errorMsg_2 + 
				std::string("Could not find registry value \"CommonFilesLocation\" subordinate to HKEY_LOCAL_MACHINE\\") + key_Temp; 
			throw isis::application_exception(TempError.c_str());
		}

		//std::cout << std::endl << "CommonFilesLocation: " <<  commonFilesLocation;

		////////////////////////////////
		// Get InstallDir
		////////////////////////////////
		std::string installDir = RetrieveRegistryStringValue( hKey, "InstallDir" );

		if ( commonFilesLocation.size() == 0 )
		{
			std::string TempError =   errorMsg_2 +
				std::string("Could not find registry value \"InstallDir\" subordinate to HKEY_LOCAL_MACHINE\\") + key_Temp; 
			throw isis::application_exception(TempError.c_str());
		}

		//std::cout << std::endl << "InstallDir: " <<  installDir;
		
		DWORD bufferSize = installDir.size() * 2;  // the shortname should be less, allocated double the space as a safety measure
		char *installPath_ShortName = new char[bufferSize ];  
		GetShortPathNameA(  installDir.c_str(), installPath_ShortName, bufferSize );						
		out_CreoParametricInstallPath =  installPath_ShortName + std::string("\\");
		delete installPath_ShortName;

		out_CreoParametricCommMsgExe =  commonFilesLocation + "\\x86e_win64\\obj\\pro_comm_msg";
		//std::cout << std::endl << "out_CreoParametricInstallPath: " <<  out_CreoParametricInstallPath;
		//std::cout << std::endl << "out_CreoParametricCommMsgExe: " <<  out_CreoParametricCommMsgExe;
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	void SetupCreoEnvironmentVariables(std::string &out_CreoStartCommand ) throw (isis::application_exception)
	{

		//std::cout << std::endl << "************* IN SetupCreoEnvironmentVariables";

		std::string creoStartCommandSuffix = "bin\\parametric.exe -g:no_graphics -i:rpc_input";

		bool readEnvVariablesFromRegistry = true;

		char *EnvVariable_USE_ENVIR_VARS;
		EnvVariable_USE_ENVIR_VARS = getenv ("CREO_PARAMETRIC_USE_ENVIR_VARS");

		if ( EnvVariable_USE_ENVIR_VARS != NULL )
		{
			std::string tempString_1 = EnvVariable_USE_ENVIR_VARS;
			std::string tempString_2 = isis::ConvertToUpperCase(tempString_1);
			std::remove( tempString_2.begin(), tempString_2.end(), ' '); 
			if ( tempString_2 == "TRUE" ) readEnvVariablesFromRegistry = false;
		}

		if ( readEnvVariablesFromRegistry )
		{
			std::vector<std::string>	supportedVersionPrefixes;
			std::string					supportedVersionString_ForErrorMsg = "1.x";  // e.g. 1.x, 2.x, or 3.x  
			supportedVersionPrefixes.push_back("1.");  // Currently on support 1.
			std::string creoParametricInstallPath;
			std::string creoParametricCommMsgExe;

			isis::RetrieveFromRegistryCreoInstallLocations( supportedVersionPrefixes,
															supportedVersionString_ForErrorMsg,
															creoParametricInstallPath, 
															creoParametricCommMsgExe );

			//_putenv_s( "PROE_INSTALL_PATH", creoParametricInstallPath.c_str() ); // PROE_INSTALL_PATH does not needed to run Creo
			_putenv_s( "PRO_COMM_MSG_EXE", creoParametricCommMsgExe.c_str() );

			out_CreoStartCommand = creoParametricInstallPath + creoStartCommandSuffix;

			char *EnvVariableCOMM_MSG_EXE  = getenv ("PRO_COMM_MSG_EXE");
			//std::cout << std::endl << "************* PRO_COMM_MSG_EXE: " << creoParametricCommMsgExe;
			//std::cout << std::endl << "************* getenv (\"PRO_COMM_MSG_EXE\"): " << EnvVariableCOMM_MSG_EXE;
		}
		else  // For this case, the system environment variables must be set.
		{
			//std::cout << std::endl << "************* IN Make sure environment variables";

			// Make sure environment variables are set 
			// (i.e. CREO_PARAMETRIC_INSTALL_PATH, CREO_PARAMETRIC_COMM_MSG_EXE	)
			char *EnvVariable_INSTALL_PATH, *EnvVariableCOMM_MSG_EXE;
			EnvVariable_INSTALL_PATH = getenv ("CREO_PARAMETRIC_INSTALL_PATH");
			EnvVariableCOMM_MSG_EXE  = getenv ("CREO_PARAMETRIC_COMM_MSG_EXE");
			if (EnvVariable_INSTALL_PATH == NULL || EnvVariableCOMM_MSG_EXE == NULL )
			{
				std::string TempError = "Environment variables CREO_PARAMETRIC_INSTALL_PATH and/or CREO_PARAMETRIC_COMM_MSG_EXE are not set properly.  Please see \"...\\Proe ISIS Extensions\\0Readme - CreateAssembly.txt\" for instructions on how to properly configure your system." +
					std::string("This file is typically located at \"C:\\Program Files\\META\\Proe ISIS Extensions\".  Alternately, if the system environment variable PROE_ISIS_EXTENSIONS is set, at a cmd prompt enter \"echo %PROE_ISIS_EXTENSIONS%\" to view the file location.");
				throw isis::application_exception(TempError.c_str());
			}
			//_putenv_s( "PROE_INSTALL_PATH", EnvVariable_INSTALL_PATH );  // PROE_INSTALL_PATH does not needed to run Creo
			_putenv_s( "PRO_COMM_MSG_EXE",  EnvVariableCOMM_MSG_EXE );

			out_CreoStartCommand = EnvVariable_INSTALL_PATH + creoStartCommandSuffix;
		}

		//std::cout << std::endl << "SetupCreoEnvironmentVariables: out_CreoStartCommand:    " << out_CreoStartCommand);
		//std::cout << std::endl << "SetupCreoEnvironmentVariables: PRO_COMM_MSG_EXE:     "    << getenv ("PRO_COMM_MSG_EXE");
		//std::cout << std::endl << "SetupCreoEnvironmentVariables: out_CreoStartCommand: "    << out_CreoStartCommand;

	}  // End SetupCreoEnvironmentVariables()


} // end namespace isis