/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "ParametricParameters.h"
#include <CommonUtilities.h>
#include <CADCommonConstants.h>
#include <iostream>
#include <string>
#include <sstream>


namespace isis
{

const int MAX_STRING_PARAMETER_LENGTH = 79;

void SetParametricParameter(  
				const std::string	&in_model_name,
				ProMdl				*in_p_model, 
				const std::string   in_ParameterName,
				//const std::string	in_ParameterType,
				e_CADParameterType	in_ParameterType,
				const std::string   in_ParameterValue)
									throw (isis::application_exception)
{

	
	/****
	std::cout << std::endl <<  "   CADParameter";
	char temp_string[100];
	std::cout << std::endl <<  "      Model Name     "  <<  in_model_name;	
	std::cout << std::endl <<  "      Type           "  <<  in_ParameterType;
	std::cout << std::endl <<  "      Name           "	<<  in_ParameterName;
	std::cout << std::endl <<  "      Value          "	<<  in_ParameterValue;

	*****/

	//typedef wchar_t	ProName[PRO_NAME_SIZE];
	if ( in_ParameterName.size() >= PRO_NAME_SIZE )
	{
		char temp_char_array[ISIS_CHAR_BUFFER_LENGTH];
		std::string err_str = "Error, Exceeded maximum number of characters. Parameter Name: "  + std::string(in_ParameterName) + ", Maximum allowed characters: " + itoa(PRO_NAME_SIZE - 1, temp_char_array, 10);
		std::clog << std::endl << err_str;			
		throw isis::application_exception(err_str.c_str());
	}

	try
	{
		ProName ParameterName;
		ProStringToWstring(ParameterName, (char *)in_ParameterName.c_str() );

		//std::cout << std::endl << "ParameterName: " << ProWstringToString(temp_string, ParameterName);	
		//std::cout << std::endl << "in_p_model:    " << in_p_model;
		//std::cout << std::endl << "*in_p_model:    " << *in_p_model;

		ProModelitem  ParameterModelItem_struct;
	
		isis::isis_ProMdlToModelitem ( *in_p_model, &ParameterModelItem_struct );
	
		ProParameter  ProParameter_struct;

		isis::isis_ProParameterInit ( &ParameterModelItem_struct, ParameterName, &ProParameter_struct);
		
		ProParamvalue  ProParamvalue_struct;

		//switch ( CADParameterType_enum(in_ParameterType) )
		switch ( in_ParameterType )
		{
			case CAD_FLOAT:
				ProParamvalue_struct.type = PRO_PARAM_DOUBLE;
				ProParamvalue_struct.value.d_val = atof(in_ParameterValue.c_str());

				break;

			case CAD_INTEGER:
				ProParamvalue_struct.type = PRO_PARAM_INTEGER;
				ProParamvalue_struct.value.i_val = atoi(in_ParameterValue.c_str());

				break;
			
			case CAD_BOOLEAN:
				ProParamvalue_struct.type = PRO_PARAM_BOOLEAN;
				ProParamvalue_struct.value.l_val = isis::String_To_ProBoolean(in_ParameterValue);
				break;

			case CAD_STRING:

				if ( in_ParameterValue.size() > MAX_STRING_PARAMETER_LENGTH )
				{
					std::stringstream errorString;
					errorString <<
					"Erroneous CADParameter Value, Parameter: " <<  CADParameterType_string(in_ParameterType) <<
					"  Value: " << in_ParameterValue << ", Value must be " << MAX_STRING_PARAMETER_LENGTH << 
					" characters or less.";
					throw isis::application_exception(errorString.str().c_str());
				}
				ProParamvalue_struct.type = PRO_PARAM_STRING;
				ProStringToWstring(ProParamvalue_struct.value.s_val,(char*)in_ParameterValue.c_str());
				break;

			default:
				//std::string err_str = "Erroneous CADParameter Type: " +  in_ParameterType + ", Should be FLOAT, INTEGER, or BOOLEAN.";
				std::string err_str = "Erroneous CADParameter Type: " +  CADParameterType_string(in_ParameterType) + ", Should be FLOAT, INTEGER, or BOOLEAN.";
				std::clog << std::endl << err_str;			
				throw isis::application_exception(err_str.c_str());
		}


		isis::isis_ProParameterValueSet( &ProParameter_struct, &ProParamvalue_struct );

		//std::cout << std::endl << "   Modified parameter: " <<  in_model_name << "::" <<  in_ParameterName << " --> " << in_ParameterValue;
		//std::clog << std::endl << "   Modified parameter: " <<  in_model_name << "::" <<  in_ParameterName << " --> " << in_ParameterValue;

		// Note - we are not using the units (i.e. k->CADValue().Units().present()).  If we were using units,
		//        then ProUnitConversionGet() would probably be needed to compute the converstion factor.  Also,
		//		  if the parameter units were set then they would be used for computing the scaling factor;
		//		  otherwise, the module units would be used.

	}
	catch ( isis::application_exception& ex )
	{
		std::string err_str =	"Part/Assembly Name: " + in_model_name +
								", Parameter Name: " + in_ParameterName + ", " + ex.what();
		throw isis::application_exception(err_str.c_str()); 

	}

}  // end SetParametricParameter

////////////////////////////////////////////////////////////////////////

std::string GetParametricParameter(  
				const std::string	&in_model_name,
				ProMdl				*in_p_model,
				const std::string   &in_ParameterOwner,
				const std::string   &in_ParameterName )
									throw (isis::application_exception)
{


	//typedef wchar_t	ProName[PRO_NAME_SIZE];
	if ( in_ParameterOwner.size() >= PRO_NAME_SIZE )
	{
		char temp_char_array[ISIS_CHAR_BUFFER_LENGTH];
		std::string err_str = "Error, Exceeded maximum number of characters. Owner Name: "  + std::string(in_ParameterOwner) + ", Maximum allowed characters: " + itoa(PRO_NAME_SIZE - 1, temp_char_array, 10);
		std::clog << std::endl << err_str;			
		throw isis::application_exception(err_str.c_str());
	}

	//typedef wchar_t	ProName[PRO_NAME_SIZE];
	if ( in_ParameterName.size() >= PRO_NAME_SIZE )
	{
		char temp_char_array[ISIS_CHAR_BUFFER_LENGTH];
		std::string err_str = "Error, Exceeded maximum number of characters. Parameter Name: "  + std::string(in_ParameterName) + ", Maximum allowed characters: " + itoa(PRO_NAME_SIZE - 1, temp_char_array, 10);
		std::clog << std::endl << err_str;			
		throw isis::application_exception(err_str.c_str());
	}


	char  ReadValue[ISIS_CHAR_BUFFER_LENGTH];
	
	try
	{
		std::string ParameterOwner_string_upper = ConvertToUpperCase(in_ParameterOwner);

		ProName ParameterOwner_wchar;
		ProStringToWstring(ParameterOwner_wchar, (char *)in_ParameterOwner.c_str() );

		ProName ParameterName_wchar;
		ProStringToWstring(ParameterName_wchar, (char *)in_ParameterName.c_str() );

		//std::cout << std::endl << "ParameterName: " << ProWstringToString(temp_string, ParameterName_wchar);	
		//std::cout << std::endl << "in_p_model:    " << in_p_model;
		//std::cout << std::endl << "*in_p_model:    " << *in_p_model;

		ProModelitem  ParameterModelItem_struct;

		if ( ParameterOwner_string_upper == "MODEL" )
			isis::isis_ProMdlToModelitem ( *in_p_model, &ParameterModelItem_struct );
		else
			isis::isis_ProModelitemByNameInit (*in_p_model, PRO_FEATURE, ParameterOwner_wchar, &ParameterModelItem_struct);
	
		ProParameter  ProParameter_struct;

		isis::isis_ProParameterInit ( &ParameterModelItem_struct, ParameterName_wchar, &ProParameter_struct);
		
		ProParamvalue  ProParamvalue_struct;

		isis::isis_ProParameterValueGet( &ProParameter_struct, &ProParamvalue_struct );
		

		switch ( ProParamvalue_struct.type ) 
		{
			case PRO_PARAM_DOUBLE:
				sprintf(ReadValue,"%lf", ProParamvalue_struct.value.d_val);
				break;

			case PRO_PARAM_STRING:
				char narrow_string[ISIS_CHAR_BUFFER_LENGTH];
				ProWstringToString(narrow_string,ProParamvalue_struct.value.s_val); 
				sprintf(ReadValue,"%s", narrow_string );
				break;

			case PRO_PARAM_INTEGER:
				sprintf(ReadValue,"%d", ProParamvalue_struct.value.i_val);
				break;
			
			case PRO_PARAM_BOOLEAN:
				if ( ProParamvalue_struct.value.l_val == PRO_B_TRUE )
					sprintf(ReadValue,"%s", "TRUE");
				else
					sprintf(ReadValue,"%s", "FALSE");
				break;

			default:
				std::string err_str = "Erroneous CADReadParameter Type, Supported types are DOUBLE, INTEGER, STRING, and BOOLEAN.";
				std::clog << std::endl << err_str;			
				throw isis::application_exception(err_str.c_str());
		}


	}
	catch ( isis::application_exception& ex )
	{
		std::string err_str =	"Part/Assembly Name: " + in_model_name +
								", Parameter Name: " + in_ParameterName + ", " + ex.what();
		throw isis::application_exception(err_str.c_str()); 

	}

	return std::string(ReadValue);

}  // end GetParametricParameter

} // end namespace isis