/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include <ApplyModelConstraints.h>
#include <BuildAssembly.h>
#include <AssembleComponents.h>
#include <DiagnosticUtilities.h>
#include <AssembleUtils.h>
#include <ToolKitPassThroughFunctions.h> 


/*
Notes:
1.	Reference [1] (search on [1] to find the place in the code that is 
	applicable to this note).

	When assembling Pro/E models, there is the concept of the added_model and base_model.  
	The base_model is the model used to constrain the added_model.   It is important to 
	identify the added_model and base_model.  A clear indication of the added model is 
	the case where the component is constrained to geometry within itself and to geometry 
	in another component.  Because it is constrained to geometry within itself, it is the 
	added_model.  It gets more complicated when the model is not constrained using geometry 
	within itself.  This is the case when assembling an assembly where the assembly datums 
	are not used for constraining.  For that case, the  added_model (i.e. added assembly) 
	is the assembly for which one of its subordinates (ComponentIDChildOf) contain the 
	constraining geometry. 
	
	The following statement assures compliance with the above rules because the 
	ComponentIDChildOf() function would return true only if the constraint feature was 
	in a model subordinate to the current assembly.  If the constraint feature was in 
	the assembly model, then ComponentIDChildOf() function would return false, which would 
	force that constraint to be applied to the base_model.

	if ( !added_model_defined  && 
		 ( l->componentID == in_ComponentID || ComponentIDChildOf( l->componentID, in_ComponentID, in_CADComponentData_map ) )
		 ) 

	Another way of accomplishing the same result would be with a statement such as

	if ( !added_model_defined  &&  ( l->ComponentID() == i->ComponentID() || 
		 l->ComponentID() != Assembly_ComponentID))   // Note would have to pass Assembly_ComponentID
													  // into ApplyModelConstraints()

	This approach would be computationally faster but would not catch the error condition due to 
	the XML containing incorrect data.  The current approach results in an exception if the 
	added_model is not constrained by a model in the assembly hierarchy.  For now, stick with 
	the current approach unless performance becomes an issue.

End Notes: */
namespace isis
{


void PopulateOneConstraintInConstraintStructure( 
	const std::string &in_ComponentID,
	const std::string &in_Model_Name,
	ProMdlType     mdltype,
	ProSolid	 *in_assembly_model,				// typedef struct sld_part* ProSolid;
	const list<int>    &in_base_model_path_list,    // list of component ids leading to the part/assembly
	const list<int>    &in_added_model_path_list,   // list of component ids leading to the part/assembly
	ProType		 in_pro_datum_type,					// enum PRO_SURFACE, PRO_AXIS
	ProName		 in_base_model_datum_name,			// ASM_RIGHT, A_1..
	ProDatumside in_base_model_datum_side,			// enum PRO_DATUM_SIDE_YELLOW (SIDE_A), PRO_DATUM_SIDE_RED (SIDE_B), PRO_DATUM_SIDE_NONE
	ProName		 in_added_model_datum_name,			// RIGHT, A23 ..
	ProDatumside in_added_model_datum_side,			// enum PRO_DATUM_SIDE_YELLOW (SIDE_A), PRO_DATUM_SIDE_RED (SIDE_B), PRO_DATUM_SIDE_NONE
	ProAsmcompConstrType	in_constraint_type,		// enum PRO_ASM_ALIGN, PRO_ASM_ALIGN_OFF...
	double		 in_offset_between_datums,			// This is only used if in_constraint_type == PRO_ASM_ALIGN_OFF or PRO_ASM_MATE_OFF
	ProAsmcompconstraint* &out_constraints )
												throw (isis::application_exception)
{
	clog <<	endl << "********** Begin PopulateOneConstraintInConstraintStructure  ***************";
	stream_PopulateOneConstraintInConstraintStructure(
					in_assembly_model,				// typedef struct sld_part* ProSolid;
					in_base_model_path_list,		
					in_added_model_path_list,
					in_pro_datum_type,				// enum PRO_SURFACE, PRO_AXIS
					in_base_model_datum_name,		// ASM_RIGHT, A_1..
					in_base_model_datum_side,		// enum PRO_DATUM_SIDE_YELLOW (SIDE_A), PRO_DATUM_SIDE_RED (SIDE_B), PRO_DATUM_SIDE_NONE
					in_added_model_datum_name,		// RIGHT, A23 ..
					in_added_model_datum_side,		// enum PRO_DATUM_SIDE_YELLOW (SIDE_A), PRO_DATUM_SIDE_RED (SIDE_B), PRO_DATUM_SIDE_NONE
					in_constraint_type,				// enum PRO_ASM_ALIGN, PRO_ASM_ALIGN_OFF...
					in_offset_between_datums,		// This is only used if in_constraint_type == PRO_ASM_ALIGN_OFF or PRO_ASM_MATE_OFF
					out_constraints,
					clog);
	clog <<	endl << "********** End PopulateOneConstraintInConstraintStructure  ***************" << endl;
	
	// *******************************************
	// Initialize the path to the base component
	// ********************************************
	ProIdTable c_id_table_base_model;

	int base_model_table_size;
	Populate_c_id_table( in_base_model_path_list, c_id_table_base_model, base_model_table_size );

	//ProAsmcomppath  comp_path;
	ProAsmcomppath	base_comp_path;
	isis::isis_ProAsmcomppathInit (	*in_assembly_model,	//ProSolid   p_solid_handle
									c_id_table_base_model,			// ProIdTable 
									base_model_table_size,			// table_size 
									&base_comp_path);	// ProAsmcomppath *p_handle

	// ***********************************************
	// Initialize the path to the assembled component
	// ***********************************************

	ProIdTable c_id_table_added_model;
	int added_model_table_size;
	Populate_c_id_table( in_added_model_path_list, c_id_table_added_model, added_model_table_size );

	//ProAsmcomppath  comp_path;
	ProAsmcomppath	added_comp_path;
	isis::isis_ProAsmcomppathInit (	*in_assembly_model,	//ProSolid   p_solid_handle
									c_id_table_added_model,			// ProIdTable 
									added_model_table_size,			// table_size 
									&added_comp_path);	// ProAsmcomppath *p_handle

	// **************************************************
	// Find the addresse to the base and added compoents
	// **************************************************
	ProMdl		 base_model;					// typedef void* ProMdl
	ProMdl		 added_model;
	isis::isis_ProAsmcomppathMdlGet( &base_comp_path, &base_model);
	isis::isis_ProAsmcomppathMdlGet( &added_comp_path, &added_model);


	//	Find the assembly datum 
	ProModelitem  base_model_datum;
	isis::isis_ProModelitemByNameInit_WithDescriptiveErrorMsg (
		in_ComponentID, in_Model_Name, mdltype,   // Added arguments
		base_model, in_pro_datum_type, in_base_model_datum_name, &base_model_datum);  // Original arguments

	//	Find the component datum
	ProModelitem  add_model_datum;
	isis::isis_ProModelitemByNameInit_WithDescriptiveErrorMsg (
		in_ComponentID, in_Model_Name, mdltype,   // Added arguments
		added_model,in_pro_datum_type, in_added_model_datum_name, &add_model_datum); // Original arguments

	//	For the assembly reference, initialize a component path.
	//	This is necessary even if the reference geometry is in the assembly.

	//	Allocate the references
	ProSelection base_model_select;
	isis::isis_ProSelectionAlloc(&base_comp_path, &base_model_datum, &base_model_select);

	ProSelection added_model_select;
	isis::isis_ProSelectionAlloc (&added_comp_path, &add_model_datum, &added_model_select);
	//isis::isis_ProSelectionAlloc (NULL, &add_model_datum, &added_model_select);

	//	Allocate and fill the constraint.
	ProAsmcompconstraint constraint;
	isis::isis_ProAsmcompconstraintAlloc (&constraint);

	isis::isis_ProAsmcompconstraintTypeSet (
					constraint, 
					in_constraint_type); // PRO_ASM_ALIGN, PRO_ASM_ALIGN_OFF..

	isis::isis_ProAsmcompconstraintAsmreferenceSet (constraint, base_model_select, in_base_model_datum_side);

	isis::isis_ProAsmcompconstraintCompreferenceSet (constraint, added_model_select, in_added_model_datum_side);
	
	if (  in_constraint_type == PRO_ASM_ALIGN_OFF || 
		  in_constraint_type == PRO_ASM_MATE_OFF )
		  //in_constraint_type == PRO_ASM_LINE_DIST || PRO_ASM_LINE_PARL)
	{
		isis::isis_ProAsmcompconstraintOffsetSet( constraint, in_offset_between_datums );
	}

	isis::isis_ProArrayObjectAdd (
					(ProArray*)&out_constraints,
					-1, // int index, if -1 append the constraint to the end 
					 1, // int number of object to append
					 &constraint);

	// The following line is commented out because it will cause an exception later in the code.  Hopefully, isis::isis_ProArrayFree will clear the added constraints. 
	//isis::isis_ProAsmcompconstraintFree(constraint);  
	isis::isis_ProSelectionFree (&added_model_select);
	isis::isis_ProSelectionFree (&base_model_select);


} // end PopulateOneConstraintInConstraintStructure


///////////////////////////////////////////////////////////////////////////////////////////////////////
bool ComponentIDChildOf(	const string &in_ChildID,
							const string &in_ParentID,
							std::map<string, isis::CADComponentData> &in_CADComponentData_map )
{
	bool found = false;

	for (	std::list<std::string>::const_iterator i( in_CADComponentData_map[in_ParentID].children.begin() ); 
			i != in_CADComponentData_map[in_ParentID].children.end(); ++i )
	{
		//clog << std::endl << "   ****** in_ComponentID, i->ComponentID(): " << in_ComponentID << "  " << i->ComponentID();
		if ( *i == in_ChildID )
		{
			//clog << std::endl << "   return true;";
			return true;

		}
		else
		{
			found = ComponentIDChildOf( in_ChildID,   *i , in_CADComponentData_map);
		}

		if (found)
		{
			//clog << std::endl << "   return true;";
			return true;
		}

	}
	//clog << std::endl << "   return found:  "  << found;
	return found;
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	Description: 
//		This function reads the following type of xml and returns the FeatureInterfaceType (i.e. CAD_DATUM, 
//		CAD_MODEL_INTERFACE, or CAD_MODEL_USER_DATA ).  For a <Constraint>, the FeatureInterfaceTypes
//		must be of the same type.  In other words, they must all be CAD_DATUM or they must all be 
//		CAD_MODEL_INTERFACE, or they must all be CAD_MODEL_USER_DATA.  If this rule is violated, an
//		exception would be thrown.
//				
//      <CADComponent ComponentID="100000701" Name="board" Type="PART">
//        <Constraint>
//          <Pair FeatureAlignmentType="ALIGN" FeatureGeometryType="SURFACE" FeatureInterfaceType="CAD_DATUM">
//            <ConstraintFeature ComponentID="100000701" FeatureName="FRONT" FeatureOrientationType="SIDE_A"/>
//            <ConstraintFeature ComponentID="100000036" FeatureName="ASM_FRONT" FeatureOrientationType="SIDE_A"/>
//          </Pair>
//          <Pair FeatureAlignmentType="ALIGN" FeatureGeometryType="SURFACE" FeatureInterfaceType="CAD_DATUM">
//            <ConstraintFeature ComponentID="100000701" FeatureName="TOP" FeatureOrientationType="SIDE_A"/>
//            <ConstraintFeature ComponentID="100000036" FeatureName="ASM_TOP" FeatureOrientationType="SIDE_A"/>
//          </Pair>
//          <Pair FeatureAlignmentType="ALIGN" FeatureGeometryType="SURFACE" FeatureInterfaceType="CAD_DATUM">
//            <ConstraintFeature ComponentID="100000701" FeatureName="RIGHT" FeatureOrientationType="SIDE_A"/>
//            <ConstraintFeature ComponentID="100000036" FeatureName="ASM_RIGHT" FeatureOrientationType="SIDE_A"/>
//          </Pair>
//        </Constraint>
//      </CADComponent>
//
//	Pre-Conditions:
//		in_CADComponent_itr must point to valid CADComponent construct (i.e. created from valid xml).
//
//	Post-Condtions:
//		if all FeatureInterfaceTypes are of one type (i.e. CAD_DATUM, CAD_MODEL_INTERFACE, or CAD_MODEL_USER_DATA) and
//		there is at least one <Pair ..
//		then
//			return FeatureInterfaceType
//		else
//			throw isis::application_exception
//
/*
isis::e_FeatureInterfaceType GetFeatureInterfaceType_enum( 
		AssemblyType::CADComponent_type::CADComponent_const_iterator in_CADComponent_itr )
			throw (isis::application_exception)
{

	isis::e_FeatureInterfaceType  FeatureInterfaceType;
	bool FeatureInterfaceTypeSet = false;
	std::string FeatureInterfaceType_string;

	for ( AssemblyType::CADComponent_type::Constraint_const_iterator j(in_CADComponent_itr->Constraint().begin());
		      j != in_CADComponent_itr->Constraint().end();
				  ++j )
	{
		for ( AssemblyType::CADComponent_type::Constraint_type::Pair_const_iterator k(j->Pair().begin());
					  k != j->Pair().end();
					  ++k )

	//if ( in_CADComponent_itr->Constraint().present() )
	//{
	//	for ( AssemblyType::CADComponent_type::Constraint_type::Pair_const_iterator k(in_CADComponent_itr->Constraint().get().Pair().begin());
	//				  k != in_CADComponent_itr->Constraint().get().Pair().end();
	//				  ++k )
		{
			if (FeatureInterfaceTypeSet )
			{
				if ( FeatureInterfaceType != FeatureInterfaceType_enum( k->FeatureInterfaceType()) )
				{
					string err_str = "Erroneous constraint FeatureInterfaceType (e.g. CAD_DATUM, CAD_MODEL_INTERFACE, CAD_MODEL_USER_DATA).  Within a <Constraint>, all FeatureInterfaceTypes must be of the same type.  Encountered types: " +
					FeatureInterfaceType_string + " and " + k->FeatureInterfaceType() + " in ComponentID: " +   in_CADComponent_itr->ComponentID();
					throw isis::application_exception(err_str.c_str());  
				}
			}
			else
			{
				FeatureInterfaceType = FeatureInterfaceType_enum( k->FeatureInterfaceType());
				FeatureInterfaceType_string = k->FeatureInterfaceType();
				FeatureInterfaceTypeSet = true;
			}

		}

		if ( !FeatureInterfaceTypeSet )
		{
			string err_str = "ComponentID: " +   in_CADComponent_itr->ComponentID() + "does not have constraints defined of the type CAD_DATUM, CAD_MODEL_INTERFACE, or CAD_MODEL_USER_DATA"; 
			throw isis::application_exception(err_str.c_str());  
		}
	}  // end if
	
	return FeatureInterfaceType;
}
*/

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Apply_CADDatum_ModelConstraints( 
				ProSolid				*in_assembly_model,
				const std::string		&in_ComponentID,		
				const ConstraintDefinition	&in_ConstraintDefinition,
				std::map<string, isis::CADComponentData> &in_CADComponentData_map )
										throw (isis::application_exception)
{

			ProAsmcomp		added_model_assembled_feature;
				
			isis::CADComponentData ComponentAssembledInfo_temp = in_CADComponentData_map[in_ComponentID];
			added_model_assembled_feature = ComponentAssembledInfo_temp.assembledFeature;


			ProAsmcompconstraint* constraints;
			isis::isis_ProArrayAlloc (0, sizeof (ProAsmcompconstraint), 1, (ProArray*)&constraints);

			//for ( AssemblyType::CADComponent_type::Constraint_const_iterator j(in_CADComponent_itr->Constraint().begin());
			//      j != in_CADComponent_itr->Constraint().end();
			//	  ++j )
			
			std::clog << std::endl;

			
			//int numberOfConstraints = 0;
			//for each (ConstraintData j_temp in in_ConstraintDefinition.constraints ) 
			//	numberOfConstraints += j_temp.constraintPairs.size();


			for ( std::vector<ConstraintData>::const_iterator j(in_ConstraintDefinition.constraints.begin());
				  j != in_ConstraintDefinition.constraints.end();
				  ++j )

			{		
				//ProAsmcomp	 base_model_assembled_feature;	
				list<int>  base_model_path_list;
				list<int>  added_model_path_list;
				
	   			for (	std::vector<ConstraintPair>::const_iterator k(j->constraintPairs.begin());
						k != j->constraintPairs.end();
						++k )
				{

					ProAsmcompConstrType constraint_type = k->featureAlignmentType;

					ProType	 pro_datum_type =  k->featureGeometryType;	

					//if ( constraint_type == PRO_ASM_ALIGN &&  pro_datum_type== PRO_AXIS && numberOfConstraints > 2 ) constraint_type = PRO_ASM_LINE_DIST;
					//if ( constraint_type == PRO_ASM_ALIGN &&  pro_datum_type== PRO_AXIS && numberOfConstraints > 2  ) constraint_type = PRO_ASM_LINE_PARL;
					//if ( constraint_type == PRO_ASM_ALIGN &&  pro_datum_type== PRO_AXIS && numberOfConstraints > 2  ) constraint_type = PRO_ASM_AUTO;
					

					ProName		 base_model_datum_name;	 // ASM_RIGHT, A_1..
					ProDatumside base_model_datum_side;	 // enum PRO_DATUM_SIDE_YELLOW (SIDE_A), PRO_DATUM_SIDE_RED (SIDE_B), PRO_DATUM_SIDE_NONE
						

					ProName		 added_model_datum_name; // RIGHT, A23 ..
					ProDatumside added_model_datum_side; // enum PRO_DATUM_SIDE_YELLOW (SIDE_A), PRO_DATUM_SIDE_RED (SIDE_B), PRO_DATUM_SIDE_NONE	

					bool added_model_defined = false;
					bool base_model_defined = false;
					string added_model_constraint_feature_component_ID;
					string base_model_constraint_feature_component_ID;
					std::cout << std::endl << "      ";
					clog << std::endl << "      ";

					//for ( AssemblyType::CADComponent_type::Constraint_type::Pair_type::ConstraintFeature_const_iterator l(k->ConstraintFeature().begin());
					//	  l != k->ConstraintFeature().end();
					//	  ++l )

					for (	std::vector<ConstraintFeature>::const_iterator l(k->constraintFeatures.begin());
							l != k->constraintFeatures.end();
							++l )
					{												
						// See Note 1 (i.e. Reference [1]) at the top of this file for an explanation of the following if statement.
						if ( !added_model_defined  && 
							 ( l->componentID == in_ComponentID || ComponentIDChildOf( l->componentID, in_ComponentID, in_CADComponentData_map ) )
							 ) 
						{   // treat as added model constraint							
							added_model_path_list = in_CADComponentData_map[l->componentID].componentPaths;
							// Note:l->featureName is enforced by XMLToProEStructures.cpp(SetConstraintAttributes)
							//	    to be no more than 31 characters.		
							ProStringToWstring(added_model_datum_name, (char *)l->featureName.c_str() );
							added_model_datum_side = l->featureOrientationType;
							added_model_defined = true;
							added_model_constraint_feature_component_ID = l->componentID;
							std::cout << in_CADComponentData_map[l->componentID].name << "::" << l->featureName;
							clog << in_CADComponentData_map[l->componentID].name << "::" << l->featureName;
							if (!base_model_defined)
							{ 
								std::cout << " --> ";
								clog << " --> ";
							}
						}
						else
						{	// constraint features on a component in the assembly.
							base_model_path_list = in_CADComponentData_map[l->componentID].componentPaths;
							// Note:l->featureName is enforced by XMLToProEStructures.cpp(SetConstraintAttributes)
							//	    to be no more than 31 characters.		
							ProStringToWstring(base_model_datum_name, (char *)l->featureName.c_str() );
							base_model_datum_side = l->featureOrientationType;
							base_model_defined = true;
							base_model_constraint_feature_component_ID = l->componentID;
							std::cout << in_CADComponentData_map[l->componentID].name << "::" << l->featureName;
							clog << in_CADComponentData_map[l->componentID].name << "::" << l->featureName;
							if (!added_model_defined) 
							{ 
								std::cout << " --> ";
								clog << " --> ";
							}
						}		
					
					}
					if ( !base_model_defined || !added_model_defined)
					{

						string err_str = "Erroneous constraint pair.  Assembled Component ID: " + in_ComponentID + 
										 " Constraint Feature Component IDs: " + added_model_constraint_feature_component_ID +
										 ", " + base_model_constraint_feature_component_ID;
						throw isis::application_exception(err_str.c_str());  
					}
				
					double offset_between_datums = 0;
					if ( k->constraintOffsetPresent )
					{	
						offset_between_datums = k->constraintOffset.value;
						clog << endl  << "      ConstraintOffset Value:               " <<  k->constraintOffset.value;
						clog << endl  << "      ConstraintOffset OffsetAlignmentType: " <<  ProAsmcompConstrType_string( k->constraintOffset.offsetAlignmentType );
						if ( k->constraintOffset.unitsPresent )
						{
							clog << endl << "      ConstraintOffset Units Type:          " << CADUnitsDistance_string( k->constraintOffset.units );
						}
					}
					
					
					PopulateOneConstraintInConstraintStructure( 
						in_ComponentID,
						in_CADComponentData_map[in_ComponentID].name,
						in_CADComponentData_map[in_ComponentID].modelType,
						in_assembly_model,			// ProSolid, typedef struct sld_part* ProSolid;
						base_model_path_list,       // Assembly path to the base part
						added_model_path_list,      // Assembly path to the added part
						pro_datum_type,				// ProType, enum PRO_SURFACE, PRO_AXIS
						base_model_datum_name,		// ProName, ASM_RIGHT, A_1..
						base_model_datum_side,		// ProDatumside, enum PRO_DATUM_SIDE_YELLOW (SIDE_A), PRO_DATUM_SIDE_RED (SIDE_B), PRO_DATUM_SIDE_NONE
						added_model_datum_name,		// ProName, RIGHT, A23 ..
						added_model_datum_side,		// ProDatumside, enum PRO_DATUM_SIDE_YELLOW (SIDE_A), PRO_DATUM_SIDE_RED (SIDE_B), PRO_DATUM_SIDE_NONE
						constraint_type, // ProAsmcompConstrType, enum PRO_ASM_ALIGN, PRO_ASM_ALIGN_OFF...
						offset_between_datums,		// double, This is only used if in_constraint_type == PRO_ASM_ALIGN_OFF or PRO_ASM_MATE_OFF
						constraints ); // ProAsmcompconstraint*	out_constraints

				}  // END for ( AssemblyType::CADComponent_type::Constraint_type::Pair_const_iterator k(j->Pair().begin());

			} // END for ( AssemblyType::CADComponent_type::Constraint_const_iterator j(in_CADComponent_itr->Constraint().begin());


			// Set the assembly component constraints
			clog << std::endl << "********* Begin Set Constraint Set - ProAsmcompConstraintsSet ";
			clog << std::endl << "added_model_assembled_feature";
			clog << std::endl <<	 "   added_model_assembled_feature.id:     " << added_model_assembled_feature.id <<  std::endl <<
	     	 "   added_model_assembled_feature.type:   " << added_model_assembled_feature.type <<  std::endl <<
	     	 "   added_model_assembled_feature.owner:  " << added_model_assembled_feature.owner;
			clog << std::endl << "Constraints address: " <<  constraints;
			clog << std::endl << "********* End Set Constraint Set - ProAsmcompConstraintsSet " << std::endl;

			ProAsmcomppath assem_path;
			ProIdTable c_id_table;
			c_id_table [0] = -1;
			isis::isis_ProAsmcomppathInit(*in_assembly_model, c_id_table, 0, &assem_path);	

			isis::isis_ProAsmcompConstraintsSet(&assem_path, &added_model_assembled_feature, constraints);
			
			// The following line was an attempt to force the regeneration of the constraint
			// so that it would properly allign kinemtic joints.  It did not help.
			//isis::isis_ProAsmcompRegenerate (	&added_model_assembled_feature, PRO_B_TRUE);

			isis::isis_ProArrayFree ((ProArray*)&constraints);

			 //Uncomment this line for testing purposes.  Regeneration after constraining each component
			 //unacceptably slows down this application.  
			 //std::cout << std::endl << "Regenerating....";
			 //isis::isis_ProSolidRegenerate (*in_assembly_model, PRO_REGEN_NO_RESOLVE_MODE);
				
}
///////////////////////////////////////////////////////////////////////////////////////////////////
/* This function is not used,  KEEP for example purposes
void Get_CADModelInterface_ModelAddresses( 
				ProSolid	 *in_assembly_model,
				AssemblyType::CADComponent_type::CADComponent_const_iterator in_CADComponent_itr,
				std::map<string, isis::CADComponentData> &in_CADComponentData_map )
										throw (isis::application_exception)
{

	// WARNING - This function has NOT been tested.

	
	// For CAD_MODEL_INTERFACE FeatureInterfaceType, there should only be only one <Constraint> <Pair>
	int ConstraintPairCount = 0;

	vector<list<int>>  model_path(2);

	for ( AssemblyType::CADComponent_type::Constraint_const_iterator j(in_CADComponent_itr->Constraint().begin());
		      j != in_CADComponent_itr->Constraint().end();
				  ++j )
	{
		++ConstraintPairCount;
		if ( ConstraintPairCount >  1 )
		{
			string err_str = "Erroneous constraint pairs (i.e. <CADComponent...> <Constraint> <Pair...>. For FeatureInterfaceType == CAD_MODEL_INTERFACE, there must be one and only one pair (i.e. <Pair...>, Component ID: " + in_CADComponent_itr->ComponentID(); 
			throw isis::application_exception(err_str.c_str());  	
		}
				

	//if ( in_CADComponent_itr->Constraint().present() )
	//{
	//	for ( AssemblyType::CADComponent_type::Constraint_type::Pair_const_iterator k(in_CADComponent_itr->Constraint().get().Pair().begin());
	//				  k != in_CADComponent_itr->Constraint().get().Pair().end();
	//				  ++k )
		for ( AssemblyType::CADComponent_type::Constraint_type::Pair_const_iterator k(j->Pair().begin());
				  k != j->Pair().end();
				  ++k )
		{
			
			++ConstraintPairCount;
			if ( ConstraintPairCount >  1 )
			{
				string err_str = "Erroneous constraint pairs (i.e. <CADComponent...> <Constraint> <Pair...>. For FeatureInterfaceType == CAD_MODEL_INTERFACE, there must be one and only one pair (i.e. <Pair...>, Component ID: " + in_CADComponent_itr->ComponentID(); 
				throw isis::application_exception(err_str.c_str());  	
			}

			int ConstraintFeatureIndex = 0;
			for ( AssemblyType::CADComponent_type::Constraint_type::Pair_type::ConstraintFeature_const_iterator l(k->ConstraintFeature().begin());
					  l != k->ConstraintFeature().end();
					  ++l )
			{
				model_path[ConstraintFeatureIndex] =  in_CADComponentData_map[l->ComponentID()].componentPaths;
				++ConstraintFeatureIndex;
				if ( ConstraintFeatureIndex >  1 )
				{
					string err_str = "Erroneous ConstraintFeature(i.e. <CADComponent...> <Constraint> <Pair...> <ConstraintFeature...>. For FeatureInterfaceType == CAD_MODEL_INTERFACE, there must be two and only two ConstraintFeatures (i.e. <ConstraintFeature...>, Component ID: " + in_CADComponent_itr->ComponentID(); 
					throw isis::application_exception(err_str.c_str());  	
				}

			}

		}
		
	 }  // end if

	std::cout <<  std::endl <<  "******** Path list for CAD_MODEL_INTERFACE*******************";
	std::cout <<  std::endl << "ComponentID: " << in_CADComponent_itr->ComponentID();
	std::cout <<  std::endl <<  " ConstraintFeature 0 ";

	for ( list<int>::const_iterator i = model_path[1].begin();
			  i != model_path[1].end(); ++i )
	{
		std::cout << std::endl << "  Feature ID: " << *i;
	}

	std::cout <<  std::endl <<  " ConstraintFeature 1 ";
	for ( list<int>::const_iterator i = model_path[1].begin();
			  i != model_path[1].end(); ++i )
	{
		std::cout << std::endl << "  Feature ID: " << *i;
	}

} // Get_CADModelInterface_ModelAddresses
*/

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// PRO_E_FEATURE_TREE
//     |
//     |--PRO_E_FEATURE_TYPE
//     |
//     |--PRO_E_COMP_PLACE_INTERFACE
//     |    |--PRO_E_COMP_PLACE_INTERFACE_TYPE
//     |    |--PRO_E_COMP_PLACE_INTERFACE_COMP
//     |    |--PRO_E_COMP_PLACE_INTERFACE_ASSEMS
//     |         |--PRO_E_COMP_PLACE_INTERFACE_ASSEM
//     |             |--PRO_E_COMP_PLACE_INTERFACE_ASSEM_REF

/*** KEEP this function as an example

void Apply_CADModelInterface_ModelConstraints( 
				ProSolid	 *in_assembly_model,
				AssemblyType::CADComponent_type::CADComponent_const_iterator in_CADComponent_itr,
				std::map<string, isis::CADComponentData> &in_CADComponentData_map )
										throw (isis::application_exception)
{
	//////////////////////////////////////////////////
	// Find the addresses of the CAD Model interfaces
	//////////////////////////////////////////////////

	Get_CADModelInterface_ModelAddresses(in_assembly_model, in_CADComponent_itr, in_CADComponentData_map);

	return;

	//////////////////////////////////////////////
	// Create the root of the element tree.
	//////////////////////////////////////////////
	ProElement elem_tree;
	isis::isis_ProElementAlloc (PRO_E_FEATURE_TREE, &elem_tree);

	///////////////////////////////////////////////////////////
	//  Allocate and set the feature type element. 
	///////////////////////////////////////////////////////////
	ProElement elem_ftype;
	isis::isis_ProElementAlloc (PRO_E_FEATURE_TYPE, &elem_ftype);
	isis::isis_ProElementIntegerSet(elem_ftype, PRO_FEAT_COMPONENT);
	isis::isis_ProElemtreeElementAdd(elem_tree, NULL, elem_ftype);

	////////////////////////////////////////////////////////////////////
	// Add a PRO_E_COMP_PLACE_INTERFACE element to the root of the tree.
	////////////////////////////////////////////////////////////////////
	ProElement elem_comp_place_interface;
	isis::isis_ProElementAlloc(PRO_E_COMP_PLACE_INTERFACE, &elem_comp_place_interface);
	isis::isis_ProElemtreeElementAdd (elem_tree, NULL, elem_comp_place_interface);

	////////////////////////////////////////////////////////////////////
	// Allocate and set the PRO_E_COMP_PLACE_INTERFACE_TYPE element.
	////////////////////////////////////////////////////////////////////
	ProElement elem_comp_place_interface_type;
	isis::isis_ProElementAlloc (PRO_E_COMP_PLACE_INTERFACE_TYPE, &elem_comp_place_interface_type);
	isis::isis_ProElementIntegerSet (elem_comp_place_interface_type, PRO_ASM_INTFC_TO_INTFC);
	ProElemtreeElementAdd (elem_comp_place_interface, NULL, elem_comp_place_interface_type);

	////////////////////////////////////////////////////////////////////
	// Allocate the PRO_E_COMP_PLACE_INTERFACE_COMP element.  This is the
	// interface on the assembled model.
	// User Guide, Section "Placement via Interface": PRO_E_COMP_PLACE_INTERFACE_COMP
	// Specifies the component model interface. This should contain the component
	// interface feature.
	////////////////////////////////////////////////////////////////////
	ProElement elem_comp_place_interface_comp;
	isis::isis_ProElementAlloc (PRO_VALUE_TYPE_SELECTION, &elem_comp_place_interface_comp);
	//isis::isis_ProElementReferencesSet (elem_comp_place_interface_comp, ??? reference geometry);
	ProElemtreeElementAdd (elem_comp_place_interface, NULL,  elem_comp_place_interface_comp);

}

***********/
////////////////////////////////////////////////////////////////////////////////////////////////
/*  KEEP for the example switch switch( FeatureInterfaceType_enum )
void ApplyModelConstraints( 
			ProSolid	 *in_assembly_model,
			AssemblyType::CADComponent_type::CADComponent_const_iterator in_Begin,
			AssemblyType::CADComponent_type::CADComponent_const_iterator in_End,
			std::map<string, isis::CADComponentData> &in_CADComponentData_map,
			const std::set<std::string> &in_ConstrainComponentsSet )
										throw (isis::application_exception)
{
		for ( AssemblyType::CADComponent_type::CADComponent_const_iterator i(in_Begin); i != in_End; ++i )
		{
			if ( in_ConstrainComponentsSet.find( i->ComponentID() ) != in_ConstrainComponentsSet.end() )
			{

				isis::e_FeatureInterfaceType FeatureInterfaceType_enum;

				FeatureInterfaceType_enum = GetFeatureInterfaceType_enum(i);

				switch( FeatureInterfaceType_enum )
				{
					case CAD_DATUM:
						Apply_CADDatum_ModelConstraints_old( in_assembly_model, i, in_CADComponentData_map);
						break;
					//case CAD_MODEL_INTERFACE:
					//  this is being worked
					//	Apply_CADModelInterface_ModelConstraints(in_assembly_model, i, in_CADComponentData_map);
					//	break;
					default:
						string err_str = "Erroneous constraint FeatureInterfaceType. FeatureInterfaceType must be of type CAD_DATUM or CAD_MODEL_INTERFACE. Component ID: " + i->ComponentID(); 
						throw isis::application_exception(err_str.c_str());  	
				}
			}

		}  // End for ( AssemblyType::CADComponent_type::CADComponent_const_iterator i(++in_Begin); i != in_End; ++i )

} // End ApplyModelConstraints
*/
////////////////////////////////////////////////////////////////////////////////////////////////////////
void ApplyModelConstraints( 
			ProSolid									*in_assembly_model,
			const std::list<std::string>				&in_ComponentIDsToBeConstrained,  
			std::map<string, isis::CADComponentData>	&in_CADComponentData_map,
														// Provide for the case where the first assembled part does not have
														// the datums front, top, and right defined. 
			bool										in_FirstComponentToBePositionedAsIntiiallyPlaced_IfDatumsCannotBeFound )  
														
										throw (isis::application_exception)
{
	int count = 0;
	for ( std::list<std::string>::const_iterator i(in_ComponentIDsToBeConstrained.begin()); 
		  i != in_ComponentIDsToBeConstrained.end(); ++i )
	{
		try
		{
			Apply_CADDatum_ModelConstraints(	in_assembly_model,
											*i,
											in_CADComponentData_map[*i].constraintDef, 
											in_CADComponentData_map);
		}
		catch (...)
		{
			if ( count == 0 && in_FirstComponentToBePositionedAsIntiiallyPlaced_IfDatumsCannotBeFound)
			{
				std::cout << std::endl << "      "  << 	in_CADComponentData_map[*i].name << "::Coordinate System --> Assembly Coordinate System";
				std::clog << std::endl << "      "  << 	in_CADComponentData_map[*i].name << ":: --> Assembly Coordinate System";
			}
			else
			{
				throw;
			}
		}
		++count;
	}

} // End ApplyModelConstraints_2


} // End namespace isis