/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include <AssembleUtils.h>
#include <CommonUtilities.h>
#include <CADCommonConstants.h>
#include <JsonHelper.h>
#include <time.h>

namespace isis
{



void Populate_c_id_table( const list<int> &in_path_list, ProIdTable out_c_id_table, int &out_c_id_table_size )
{
	int index = 0;
	for ( list<int>::const_iterator i = in_path_list.begin();
	  i != in_path_list.end(); ++i )
	{
		out_c_id_table[index] = *i;
		++index;
	}
	out_c_id_table [index] = -1;
	out_c_id_table_size = in_path_list.size();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////
void RetrieveTranformationMatrix_Assembly_to_Child (  
							const std::string  &in_AssemblyComponentID,
							const std::string  &in_ChildComponentID,
							const list<int>	   &in_ChildComponentPaths,
							std::map<std::string, isis::CADComponentData>		&in_CADComponentData_map,  
							ProBoolean   in_bottom_up,
							double out_TransformationMatrix[4][4] )  throw (isis::application_exception)
{
	// Must get the path from the assembly to the child
	ProIdTable	c_id_table;
	int			c_id_table_size;

	Populate_c_id_table( in_ChildComponentPaths, c_id_table, c_id_table_size );

	//std::cout << std::endl << std::endl << "in_AssemblyComponentID: " << in_AssemblyComponentID << " Name: " << in_CADComponentData_map[in_AssemblyComponentID].name ;
	//std::cout << std::endl << "in_CADComponentData_map[in_AssemblyComponentID].modelHandle: " << in_CADComponentData_map[in_AssemblyComponentID].modelHandle;
	//std::cout << std::endl << "in_ChildComponentID: " << in_ChildComponentID << " Name: " << in_CADComponentData_map[in_ChildComponentID].name;
	//std::cout << std::endl << "c_id_table[0]: " << c_id_table[0];

	ProAsmcomppath	comp_path;
	isis::isis_ProAsmcomppathInit (	in_CADComponentData_map[in_AssemblyComponentID].modelHandle,	//ProSolid   p_solid_handle
									c_id_table,			// ProIdTable 
									c_id_table_size,	// table_size 
									&comp_path);		// ProAsmcomppath *p_handle


	isis::isis_ProAsmcomppathTrfGet (	&comp_path, 				//	ProAsmcomppath *p_path,
										in_bottom_up,				// ProBoolean   bottom_up,
										out_TransformationMatrix ); //ProMatrix    transformation);

}
////////////////////////////////////////////////////////////////////////////////////////////////////////////
//	Pre-Conditions:
//		0 < in_UniqueNameIndex  < 9999
//		7 <= in_AllowedSize <= PRO_NAME_SIZE
//		in_OrigName must 
//			no greater than 31 characters,
//			must be characters with no special characters other than '_','<', and '>'
//			Examples:
//				Engine						// Non family table example
//				Chassis_8_Wheel<Chassis>	// Family table example, If '<' exists then '>' must exist. One and only
//											// one '<' and one and only one '>'
// 	Post-Conditions	
//		An exception will be thrown if the size of the resulting strings are greater than 31.
//
//		otherwise, return out_ModelName, and out_CompleteName.  Examples follow:
//		a) Non Family Table - out_ModelName="EngineZ1Z",	out_CompleteName="EngineZ1Z"
//		b) Family table		- out_ModelName="ChassisZ1Z",	out_CompleteName="Chassis_8_Wheel<ChassisZ1Z>"
void CreateModelNameWithUniqueSuffix(  
					int					in_UniqueNameIndex, 
					const std::string	&in_OrigName,
					std::string			&out_OrigNameWithoutFamilyEntry,
					std::string			&out_ModelName,
					std::string			&out_CompleteName, // For family tables, would be the complete name
														   // e.g. Chassis_8_Wheel<Chassis>
														   // otherwise, same as out_ModelName
					int in_AllowedSize = PRO_NAME_SIZE )   // Warning PRO_NAME_SIZE (i.e. 32) is the max allowed 
{

	const int numSuffixChars = 6;

	int maxPartNameLength_BeforeSuffix = in_AllowedSize - numSuffixChars - 1;  //  32 - 6 - 1  = 25, example Z1234Z  
	std::string tempPartName;

	size_t lessThanPos = in_OrigName.find('<');  // Family table would have the form Chassis_8_Wheel<Chassis>

	if ( lessThanPos != std::string::npos  )
	{  
		/////////////////
		// Family Table 
		/////////////////

		size_t greaterThanPos =  in_OrigName.find('>');

		if ( greaterThanPos == std::string::npos  )
		{
			std::cout << std::endl << "EXCEPTION";
		}

		std::string origModelName = in_OrigName.substr(lessThanPos + 1, greaterThanPos - lessThanPos - 1 );

		out_OrigNameWithoutFamilyEntry = origModelName;

		int newSizeAllowable = in_AllowedSize - lessThanPos - 2; // 2 for <>

		//std::cout << std::endl << "newSizeAllowable: " << newSizeAllowable;

		if (newSizeAllowable < 1 )
		{
			// This will cause an exception to be thrown at the end of this function.
			// Allow the computation to go forward becasue the error message will make more
			// sense if the strings are computed and displayed in the error message.
			newSizeAllowable = numSuffixChars;
		}
		std::string tempCompleteModelName;
		CreateModelNameWithUniqueSuffix(in_UniqueNameIndex, origModelName, out_OrigNameWithoutFamilyEntry, out_ModelName, tempCompleteModelName, newSizeAllowable );
		out_CompleteName = in_OrigName.substr(0, lessThanPos) + "<" + out_ModelName + ">";	
	}
	else
	{ 
		/////////////////////
		// Non Family Table
		/////////////////////

		char buffer[64];

		// Need to make sure that the new name is not too long.  A Creo part name can be 31 characters
		// Assume a name of 25 characters with 6 characters for the suffix (e.g. Z1050Z). 
		if (in_OrigName.length() > maxPartNameLength_BeforeSuffix )
			out_ModelName = in_OrigName.substr(0,maxPartNameLength_BeforeSuffix)+ "Z" + itoa(in_UniqueNameIndex,buffer,10) + "Z";
		else
			out_ModelName = in_OrigName + "Z" + itoa(in_UniqueNameIndex,buffer,10) + "Z";

		out_CompleteName = out_ModelName;
		out_OrigNameWithoutFamilyEntry = in_OrigName;
	}

	if ( out_CompleteName.size() > 31 )
	{
		std::string errorString;
					errorString = "Function - CreateModelNameWithUniqueSuffix, string lengths exceed 31 characters.  " +
						std::string("in_OrigName: ") + 	in_OrigName + "  out_ModelName: " + out_ModelName + "  out_CompleteName: " + out_CompleteName;  	   
					throw isis::application_exception(errorString.c_str());	
	}
}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////

	// STEP file name case:
	// AP203_E2_SEPERATE_PART_FILES
	//		Assembly:						As entered	e.g.  FourBar.asm  -->  FourBar_asm.stp
	//		Part Included in Assembly:		Lower Case  e.g.  BAR_02.prt   -->	bar_02_prt.stp
	//		Part standalone					As entered	e.g.  MaSs		   -->	MaSs_prt.stp 
	// AP203_E2_SINGLE_FILE
	//		Assembly:						As entered	e.g.  FourBar.asm  -->  FourBar_asm.stp
	//		Part standalone					As entered	e.g.  MaSs		   -->	MaSs_prt.stp 	
	// AP203_SINGLE_FILE
	//		Assembly:						As entered	e.g.  FourBar.asm  -->  FourBar_asm.stp
	//		Part standalone					As entered	e.g.  MaSs		   -->	MaSs_prt.stp 	
	// AP214_SEPERATE_PART_FILES
	//		Assembly:						As entered	e.g.  FourBar.asm  -->  FourBar_asm.stp
	//		Part Included in Assembly:		Lower Case  e.g.  BAR_02.prt   -->	bar_02_prt.stp
	//		Part standalone					As entered	e.g.  MaSs		   -->	MaSs_prt.stp 
	// AP214_SINGLE_FILE
	//		Assembly:						As entered	e.g.  FourBar.asm  -->  FourBar_asm.stp
	//		Part standalone					As entered	e.g.  MaSs		   -->	MaSs_prt.stp 	

	////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void UpdateManufacturingManifestWithSTEPFileInfo( 
									e_DataExchangeVersion in_DataExchangeVersion, // AP203_SINGLE_FILE, AP203_E2_SINGLE_FILE, AP203_E2_SEPERATE_PART_FILES...
									const std::string	&in_ComponentID,
									bool				in_OnlyUpdateManifestForParts,
									bool				in_ChangeCaseOfPartStepFileToLowerCase,
									std::map<std::string, isis::CADComponentData> &in_CADComponentData_map )
														throw (isis::application_exception)
	{
		try
		{
			std::string stepFileDirectory = isis::DataExchangeVersion_string(in_DataExchangeVersion) + "/";

			if ( isis::FileExists( manufacturingManifestJson_PathAndFileName.c_str() ))
			{
				isis::ComponentVistorBuildListOfComponentIDs  assembllyComponenteIDs;
				isis::VisitComponents(in_ComponentID, in_CADComponentData_map, assembllyComponenteIDs );

				std::map<std::string, std::string> componentInstanceId_to_StepFile_map;

				for each ( std::string cID in assembllyComponenteIDs.listOfComponentIDs )
				{
					//  When Creo outputs STEP files with separate part-files option, 
					//  only the part files are created separately and are named with _prt.
					//  The assembly hierarchy is represented in the top-assembly STEP file.

					std::string tempFullPathToStepFile;
					if (in_ChangeCaseOfPartStepFileToLowerCase)
						tempFullPathToStepFile = stepFileDirectory + isis::ConvertToLowerCase(in_CADComponentData_map[cID].name + "_prt.stp");
					else
						tempFullPathToStepFile = stepFileDirectory + in_CADComponentData_map[cID].name + "_prt.stp";
					
					if (in_OnlyUpdateManifestForParts )
					{
						if ( in_CADComponentData_map[cID].modelType == PRO_PART )				
							componentInstanceId_to_StepFile_map[cID] = tempFullPathToStepFile;
					}
					else
					{
						componentInstanceId_to_StepFile_map[cID] = tempFullPathToStepFile;
					}
				}

				std::set<std::string> componentInstanceIds_AddedToManifest;
				isis_CADCommon::AddStepFileMappingToManufacturingManifest(
														manufacturingManifestJson_PathAndFileName,
														componentInstanceId_to_StepFile_map,
														componentInstanceIds_AddedToManifest );
				/////////////////////////////////////////
				// Log StEP File Names added to manifest
				/////////////////////////////////////////
				std::clog << std::endl << std::endl << "Added STEP file names to " <<  
										  manufacturingManifestJson_PathAndFileName;
				bool flagAdded = false;
				for each ( std::string compID in componentInstanceIds_AddedToManifest )
				{
					std::clog << std::endl << "   Component Instance ID: " << compID <<
						            "  STEP File Name: " << componentInstanceId_to_StepFile_map[compID];
					flagAdded = true;
				}
				if ( !flagAdded ) std::clog << std::endl << "   None";

				/////////////////////////////////////////////
				// Log StEP File Names Not added to manifest
				/////////////////////////////////////////////
				std::clog << std::endl << std::endl << "Did Not add STEP file names to " <<  
										  manufacturingManifestJson_PathAndFileName;

				flagAdded = false;
				for each ( std::string compID in assembllyComponenteIDs.listOfComponentIDs )
				{
					if ( componentInstanceIds_AddedToManifest.find(compID) == componentInstanceIds_AddedToManifest.end() &&
						 in_CADComponentData_map[compID].modelType == PRO_PART )
					{
						std::clog << std::endl << "   Component Instance ID: " << compID <<
						            "  STEP File Name: " << componentInstanceId_to_StepFile_map[compID];
						flagAdded = true;
					}
				}
				if ( !flagAdded ) std::clog << std::endl << "   None";
				
			}  
			else
			{
				std::clog << std::endl << std::endl << CouldNotFindManufacturingManifestError;
				std::cout << std::endl << std::endl << CouldNotFindManufacturingManifestError;
			} // END if ( isis::FileExists(
		}
		catch (isis::application_exception &ex )
		{
			std::clog << std::endl << std::endl << "ERROR, Function: UpdateManufacturingManifestWithSTEPFileInfo, Error Message: " << ex.what(); 
			std::cout << std::endl << std::endl << "ERROR, Function: UpdateManufacturingManifestWithSTEPFileInfo, Error Message: " << ex.what(); 
		}
		catch (...)
		{
			std::clog << std::endl << std::endl << "ERROR, Function: UpdateManufacturingManifestWithSTEPFileInfo, Error Message: Unkown Error"; 
			std::clog << std::endl << std::endl << "ERROR, Function: UpdateManufacturingManifestWithSTEPFileInfo, Error Message: Unkown Error"; 
		}
	}
	////////////////////////////////////////////////////////////////////////////////////////////////////////////

	///////////////////////////////////////////////////////////////////////////////////////////////////////////
	// If a part name (not assembly name) appears more than once in in_out_CADComponentData_map
	//	then
	//		this function modifies in_out_CADComponentData_map to have unique name for the second
	//		and later occurrences of the particular part name.  The new and old part names are added
	//      to out_ToPartName_FromPartName. 
	void ModifyToHaveAUniqueNameForEachPart( 
							int &in_out_UniqueNameIndex,
							std::map<std::string, isis::CADComponentData> &in_out_CADComponentData_map, 
							std::map<std::string, std::string>			  &out_ToPartName_FromPartName )
																		throw (isis::application_exception)
	{
		std::set<std::string>  partNames;

		//int index = 1;
		//int maxPartNameLength_BeforeSuffix = PRO_NAME_SIZE - 8;  //  32 - 8 = 24

		char buffer[isis::ISIS_CHAR_BUFFER_LENGTH];

		for( std::map<std::string, isis::CADComponentData>::iterator i(in_out_CADComponentData_map.begin());
			 i != in_out_CADComponentData_map.end();
			 ++i )
		{
			//std::cout << std::endl << "Model Name +++++++++>           " << i->second.name;
			//std::cout << std::endl << " i->second.modelType +++++++++> " << i->second.modelType;

			// Only check for parts.  Not concerned about assemblies
			if ( i->second.modelType == PRO_PART  && 
				 partNames.find(isis::ConvertToUpperCase(i->second.name)) !=  partNames.end() )
			{
				std::string origNameWithoutFamilyEntry;
				std::string modelName;
				std::string completeName;
				CreateModelNameWithUniqueSuffix(  in_out_UniqueNameIndex, 
												i->second.name,
												origNameWithoutFamilyEntry,
												modelName, 
												completeName );
				//std::cout << std::endl << "origNameWithoutFamilyEntry +++++++++> "	<< origNameWithoutFamilyEntry;
				//std::cout << std::endl << "modelName +++++++++> "					<< modelName;
				//std::cout << std::endl << "completeName +++++++++> "				<< completeName;

				i->second.name = completeName;
				out_ToPartName_FromPartName[modelName] =  origNameWithoutFamilyEntry;
				++in_out_UniqueNameIndex;
			}
			partNames.insert(isis::ConvertToUpperCase(i->second.name));
		}
	}	// END ModifyToHaveAUniqueNameForEachPart

	////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// If part/assem appears more than once in in_out_CADComponentData_map and is a parametric part/assembly
	//	then
	//		this function modifies in_out_CADComponentData_map to have unique name for the second
	//		and later occurrences of the particular part/assembly that is a parametric part/assembly.  
	//		The new and old part names are added to out_ToPartAssemName_FromPartAssemName. 
	void ModifyToHaveAUniqueNameForEachParametricPartOrAssembly( 
							int &in_out_UniqueNameIndex,
							std::map<std::string, isis::CADComponentData> &in_out_CADComponentData_map, 
							std::map<std::string, std::string>			  &out_ToPartAssemName_FromPartAssemName )
																		throw (isis::application_exception)
	{
		std::set<std::string>  partAssemNames;

		//int index = 1;
		//int maxPartNameLength_BeforeSuffix = PRO_NAME_SIZE - 8;  //  32 - 8 = 24

		char buffer[isis::ISIS_CHAR_BUFFER_LENGTH];

		for( std::map<std::string, isis::CADComponentData>::iterator i(in_out_CADComponentData_map.begin());
			 i != in_out_CADComponentData_map.end();
			 ++i )
		{
			// Check for parametric parts
			if ( i->second.parametricParametersPresent && 
				 partAssemNames.find(isis::ConvertToUpperCase(i->second.name)) !=  partAssemNames.end())
			{
				std::string origNameWithoutFamilyEntry;
				std::string modelName;
				std::string completeName;
				CreateModelNameWithUniqueSuffix(	in_out_UniqueNameIndex, 
													i->second.name,
													origNameWithoutFamilyEntry,
													modelName, 
													completeName );
				i->second.name = completeName;
				out_ToPartAssemName_FromPartAssemName[modelName] =  origNameWithoutFamilyEntry;

				++in_out_UniqueNameIndex;
			}
			partAssemNames.insert(isis::ConvertToUpperCase(i->second.name));
		}
	}	// END ModifyToHaveAUniqueNameForEachParametricPartOrAssembly	 

	////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void VisitComponents(	const std::string								&in_ComponentID, 
							std::map<std::string, isis::CADComponentData>	&in_out_CADComponentData_map,
							ComponentVistor									&in_componentVistor) 
																		throw (isis::application_exception) 
	{

		in_componentVistor(	in_ComponentID, in_out_CADComponentData_map);

		if ( in_out_CADComponentData_map[in_ComponentID].modelType == PRO_MDL_ASSEMBLY )
		{
			for ( std::list<string>::const_iterator i(in_out_CADComponentData_map[in_ComponentID].children.begin());
				i != in_out_CADComponentData_map[in_ComponentID].children.end();
				++i )
			{
				VisitComponents( *i,in_out_CADComponentData_map, in_componentVistor);
			}
		}
	}
	////////////////////////////////////////////////////////////////////////////////////////////////////////////

	void ComponentVistorMaterialTokens::operator() ( const std::string  &in_ComponentID, 
									  std::map<std::string, isis::CADComponentData> &in_out_CADComponentData_map )
	{
		if ( in_out_CADComponentData_map[in_ComponentID].modelType == PRO_MDL_PART )
		{
			if ( in_out_CADComponentData_map[in_ComponentID].analysisTempMaterialDefined )
			{
				AnalysisTempMaterial analysisTempMaterialDefinition = in_out_CADComponentData_map[in_ComponentID].analysisTempMaterialDefinition;
				componentID_PoissonsRatio_map[in_ComponentID] = analysisTempMaterialDefinition.poissonsRatio;
				materialKey_ComponentID_map[analysisTempMaterialDefinition.tokenPoissonsRatio] = in_ComponentID;
			}
			else
			{
				std::string errorString;
				errorString = "Function - ComponentVistorMaterialTokens Functor, called from ReplaceMaterialTokens_ReturnMaterialToComponentID), material properties not defined for component ID: " + in_ComponentID;
				throw isis::application_exception(errorString.c_str());	
			}
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////
	ComponentVistorCountAssemblyComponents::ComponentVistorCountAssemblyComponents():numberOfComponents(0){};

	void ComponentVistorCountAssemblyComponents::operator() ( const std::string  &in_ComponentID, 
									  std::map<std::string, isis::CADComponentData> &in_out_CADComponentData_map )
	{
		if ( in_out_CADComponentData_map[in_ComponentID].children.size() == 0 ) ++numberOfComponents; // This is a part or a leaf assembly

	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void ComponentVistorBuildListOfComponentIDs::operator() ( const std::string  &in_ComponentID, 
									  std::map<std::string, isis::CADComponentData> &in_out_CADComponentData_map )
	{
		listOfComponentIDs.push_back(in_ComponentID);
	}
	////////////////////////////////////////////////////////////////////////////////////////////////////////////
	std::string GetDayMonthTimeYear()
	{
		time_t time_start;		// calendar time 
		time_start=time(NULL);	// get current cal time 

		std::string dayMonthTimeYear = asctime(localtime(&time_start));
		if (dayMonthTimeYear.size() > 0 )
		{
			// Remove the extra linefeed from the above string.
			dayMonthTimeYear.replace(dayMonthTimeYear.end() -1, dayMonthTimeYear.end(), "");

		}
		return dayMonthTimeYear;
	}


//	ComponentVistorBuildListOfBoundingBoxComputations::ComponentVistorBuildListOfBoundingBoxComputations(){};
//
//	void ComponentVistorBuildListOfBoundingBoxComputations::operator() ( const std::string  &in_ComponentID, 
//									  std::map<std::string, isis::CADComponentData> &in_out_CADComponentData_map )
//	{
//		if ( in_out_CADComponentData_map[in_ComponentID].cADComputations.boundingBoxMetricDefined ) 
//			boundingBoxComputationsComponentIDs.push_back( in_ComponentID );
//
//	}
	////////////////////////////////////////////////////////////////////////////////////////////////////////////

} // end namespace isis