/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include <BuildAssembly.h>
#include <isis_ptc_toolkit_functions.h>
#include <isis_application_exception.h>
#include <isis_include_ptc_headers.h>
#include <AssembleComponents.h>
#include <StringToEnumConversions.h>
#include <ApplyModelConstraints.h>
#include <CommonUtilities.h>
#include <DiagnosticUtilities.h>
#include <CADCommonConstants.h>
#include <ISISConstants.h>
#include <FiniteElementAnalysis.h>
#include <SetCADModelParamters.h>
#include <CommonFunctions.h>
#include <ToolKitPassThroughFunctions.h>
#include <stdio.h>
#include <stdlib.h>
#include <set>
#include <iterator>
#include <algorithm>
#include <sstream>

using namespace std;

namespace isis
{

/////////////////////////////////////////////////////////////////////////////////////////
void UpdateAllSubassemblyComponentsToIncludeParentAssemblyPath( 
						const std::string in_ComponentID,
						std::map<string, isis::CADComponentData> &in_out_CADComponentData_map,
						int assembled_feat_id )
{
	//for ( AssemblyType::CADComponent_type::CADComponent_const_iterator i(in_Begin); i != in_End; ++i )
	//{
	for ( std::list<string>::const_iterator i(in_out_CADComponentData_map[in_ComponentID].children.begin());
		i != in_out_CADComponentData_map[in_ComponentID].children.end();
		++i )
	{
		in_out_CADComponentData_map[*i].componentPaths.push_front(assembled_feat_id);
		UpdateAllSubassemblyComponentsToIncludeParentAssemblyPath( *i,
																	in_out_CADComponentData_map,
																	assembled_feat_id );
	}
}
/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
// If all the members in A are in B
//		return true;
// else
//		return false;
//
//  Special Cases:
//  -------------
//	If A is empty and B is not empty
//		return true;
//
//	If B is empty and A is not empty
//		return false;
//
//	If B is empty and A is empty
//		return true;
bool AinB(  const std::set<std::string> &in_A,  const std::set<std::string> &in_B )
{
	std::set<std::string>::const_iterator itr;

	std::set<std::string>  Intersection;

	std::insert_iterator<std::set<std::string> > ii(Intersection, Intersection.begin()); 

	std::set_intersection ( in_A.begin(), in_A.end(), in_B.begin(), in_B.end(), ii );

	//std::cout << std::endl << "Begin Intersection"; 
	//for ( std::set<std::string>::const_iterator itr = Intersection.begin(); itr !=  Intersection.end(); ++itr )
	//{
	//	std::cout << std::endl << *itr; 
	//}
	//std::cout << std::endl << "End Intersection"; 

	if ( in_A.size() == Intersection.size() )
		return true;
	else 
		return false;
}

/////////////////////////////////////////////////////////////////////////////////////
void	AddSubordinateComponentIDsToSet( const std::string &in_ComponentID, 
										 std::map<std::string, CADComponentData> &in_ComponentAssembledInfo,
										 std::set<std::string> &out_Existing )
{
	if ( in_ComponentAssembledInfo[in_ComponentID].children.size() > 0 )
	{
		for ( std::list<std::string>::const_iterator i = in_ComponentAssembledInfo[in_ComponentID].children.begin();
													i != in_ComponentAssembledInfo[in_ComponentID].children.end();
													++i )
		{
			//std::clog << std::endl << "   AddSubordinateComponentIDsToSet, ComponentID: " << *i; 
			out_Existing.insert(*i);
			AddSubordinateComponentIDsToSet( *i, in_ComponentAssembledInfo, out_Existing );

		}													
	}
}

/////////////////////////////////////////////////////////////////////////////////////
void  LogSortingSet(	const std::string	&in_Indent,
						const std::string	&in_Title,
						const std::string	&in_ParentName,
						const std::string	&in_ParentComponentID,
						std::map<std::string, CADComponentData> &in_ComponentAssembledInfo,
						const std::set<std::string> &in_Components )
{

	std::clog << std::endl << in_Indent << in_Title;

	std::clog << std::endl << in_Indent << "   Parent: " << in_ParentComponentID << "   " << in_ParentName;

		for ( std::set<std::string>::const_iterator i = in_Components.begin(); i != in_Components.end(); ++i )
		{
			std::clog << std::endl << in_Indent << "      " <<
				in_ComponentAssembledInfo[*i].componentID  <<   "   " <<
				in_ComponentAssembledInfo[*i].name;
		}
}
////////////////////////////////////////////////////////////////////////////////////
void  LogSortingList(	const std::string	&in_Indent,
						const std::string	&in_Title,
						const std::string	&in_ParentName,
						const std::string	&in_ParentComponentID,
						std::map<std::string, CADComponentData> &in_ComponentAssembledInfo,
						const std::list<std::string>  &in_Components )
{

	std::clog << std::endl << in_Indent << in_Title;

	std::clog << std::endl << in_Indent << "   Parent: " << in_ParentComponentID << "   " << in_ParentName;

		for ( std::list<std::string>::const_iterator i = in_Components.begin(); i != in_Components.end(); ++i )
		{
			std::clog << std::endl << std::endl << in_Indent << "      " << in_ComponentAssembledInfo[*i].name  << 
				"  " << in_ComponentAssembledInfo[*i].componentID;
			std::clog << std::endl << in_Indent << "                   Depends On: ";
			for ( std::set<std::string>::const_iterator j = in_ComponentAssembledInfo[*i].dependsOn.begin(); 
				   j != in_ComponentAssembledInfo[*i].dependsOn.end(); ++j )
			{
				std::clog << std::endl << in_Indent << "                       "  <<
				in_ComponentAssembledInfo[*j].name  << "    " <<
				in_ComponentAssembledInfo[*j].componentID;

			}

		}
}
/////////////////////////////////////////////////////////////////////////////////////
void LogSortingInputs(	const std::string	&in_ParentName,
								const std::string  &in_ParentComponentID,
								const std::list<std::string> &in_Components, 
								std::map<std::string, CADComponentData> &in_ComponentAssembledInfo )
{
	LogSortingList(	"   ", 
					"BEGIN ##########   SortCADComponents, List of components to be sorted   ###########",
					in_ParentName,
					in_ParentComponentID,
					in_ComponentAssembledInfo,
					in_Components );
	clog << std::endl << "END ############   SortCADComponents, List of components to be sorted  ###########";
}

/////////////////////////////////////////////////////////////////////////////////////
void LogStateWhenSortingFailed(	const std::string	&in_ParentName,
								const std::string  &in_ParentComponentID,
								const std::list<std::string> &in_Components, 
								const std::list<std::string> &out_Components, 
								const std::set<std::string>  &in_Existing,
								std::map<std::string, CADComponentData> &in_ComponentAssembledInfo )
{

	// This is a fatal error.  It means the list cannot be sorted.
	// Throw exception
	
	std::cout  << std::endl << std::endl << "ERROR: Could not sort the component list";
	//clog << std::endl << std::endl << "BEGIN ##########   SORTING ERROR   ###########";
	//LogSortingList(	"   ", 
	//				"BEGIN SortCADComponents, To-be-added Components",
	//				in_ParentName,
	//				in_ParentComponentID,
	//				in_ComponentAssembledInfo,
	//				in_Components );
	//clog << std::endl << "END SortCADComponents, To-be-added Components";
	clog << std::endl;
	LogSortingList(	"", 
					"BEGIN SortCADComponents, Components that were sorted when the sort failed",
					in_ParentName,
					in_ParentComponentID,
					in_ComponentAssembledInfo,
					out_Components );
	clog << std::endl << "END  SortCADComponents, Components that were sorted when the sort failed";

	/////////////////
	// Log Not used
	/////////////////
	std::list<std::string> Components_temp = in_Components;
	for ( std::list<std::string>::const_iterator itr_2 =  out_Components.begin(); itr_2 != out_Components.end(); ++itr_2 )
	{
		Components_temp.remove( *itr_2 );
	}
	
	clog << std::endl << std::endl << "BEGIN SortCADComponents, Not-sorted-yet list when the sort failed"; 
	for ( std::list<std::string>::const_iterator itr_2 =  Components_temp.begin(); itr_2 != Components_temp.end(); ++itr_2 )
	{
		std::clog << std::endl <<  "     "  <<
			in_ComponentAssembledInfo[*itr_2].componentID  << "    " <<
			in_ComponentAssembledInfo[*itr_2].name;     
	}
	clog <<  std::endl << "END SortCADComponents, Not-sorted-yet list when the sort failed"; 

	clog <<  std::endl;
	LogSortingSet(	"", 
					"BEGIN Components Higher-up in Sort Tree (i.e. existing components from a sort perspective)",
					in_ParentName,
					in_ParentComponentID,
					in_ComponentAssembledInfo,
					in_Existing );
	clog << std::endl << "END Higher-up in Sort Tree (i.e. existing components from a sort perspective)";

	clog << std::endl << "END ###########   SORTING ERROR   ###########" << std::endl;

}

/////////////////////////////////////////////////////////////////////////////////////
void SortCADComponents ( const std::string	&in_ParentName,
						 const std::string  &in_ParentComponentID,
						 const std::list<std::string> &in_Components,
						 std::map<std::string, CADComponentData> &in_ComponentAssembledInfo,
						 std::list<std::string> &out_Components )						 
																throw (isis::application_exception)
{
	
	std::list<std::string> Components = in_Components;
	std::set<std::string> Existing;

	Existing.insert(in_ParentComponentID);

	int NumComponents = Components.size();

	//LogSortingInputs(in_ParentName, in_ParentComponentID, in_Components, in_ComponentAssembledInfo ); 

	for ( int i = 0; i < NumComponents; ++i )
	{
		bool FoundComponent = false;
		std::string FoundComponentID;
		for ( std::list<std::string>::const_iterator itr = Components.begin(); itr != Components.end(); ++itr )
		{
			std::set<std::string> Existing_temp = Existing;

			AddSubordinateComponentIDsToSet( *itr, in_ComponentAssembledInfo, Existing_temp );

			if ( AinB( in_ComponentAssembledInfo[*itr].dependsOn, Existing_temp ) )
			{			
				// found suitbale candidate
				out_Components.push_back(*itr);
				Existing.insert(*itr);
			
				// If this is an assembly, must add all the assemblies subordinates to Existing set.
				// This is necessary because the children of the added assembly can be referenced for 
				// constraint purposes.
				//std::clog << std::endl << "BEGIN !!!!!!!!!!!!!! Adding AddSubordinateComponentIDsToSet, Component ID: " << *itr; 
				AddSubordinateComponentIDsToSet( *itr, in_ComponentAssembledInfo, Existing );
				//std::clog << std::endl << "END !!!!!!!!!!!!!! Adding AddSubordinateComponentIDsToSet, Component ID: " << *itr; 

				FoundComponentID = *itr;
				FoundComponent = true;
				break;
			}
		}

		if (!FoundComponent)
		{
			LogStateWhenSortingFailed( in_ParentName, in_ParentComponentID, in_Components, out_Components, Existing, in_ComponentAssembledInfo );
			throw isis::application_exception("Sorting error.  See log file for additional information.");  
		}

		Components.remove(FoundComponentID);
	}
}

/////////////////////////////////////////////////////////////////////////////////////////
enum e_InclusionExclusionInstructions
{
	INCLUDE_ALL_COMPONENTS,
	INCLUDE_ALL_COMPONENTS_EXCEPT_SIZE_TO_FIT,
	INCLUDE_ONLY_SIZE_TO_FIT
};


/////////////////////////////////////////////////////////////////////////////////////////

void BuildFiltereListOfComponents(	const std::string							&in_AssemblyComponentID, 
									std::map<string, isis::CADComponentData>	&in_CADComponentData_map, 
									e_InclusionExclusionInstructions			in_Instructions, 
									std::list<std::string>						&out_FilteredComponents )
{
	for ( std::list<string>::const_iterator i (in_CADComponentData_map[in_AssemblyComponentID].children.begin());
		  i != in_CADComponentData_map[in_AssemblyComponentID].children.end();
		  ++i )
	{
		if ( in_Instructions == INCLUDE_ALL_COMPONENTS_EXCEPT_SIZE_TO_FIT )
		{
			if ( in_CADComponentData_map[*i].specialInstruction == CAD_SPECIAL_INSTRUCTION_SIZE_TO_FIT ) continue;
		}
		else
		{
			if ( in_Instructions == INCLUDE_ONLY_SIZE_TO_FIT )
			{
				if ( in_CADComponentData_map[*i].specialInstruction != CAD_SPECIAL_INSTRUCTION_SIZE_TO_FIT ) continue;
			}
		}

		out_FilteredComponents.push_back(*i);
	}
}
/////////////////////////////////////////////////////////////////////////////////////////
void Add_Subassemblies_and_Parts( 
					ProMdl															in_p_asm,
					const std::string												&in_ParentName,
					const std::list<std::string>									&in_Components,
					std::map<string, isis::CADComponentData>						&in_CADComponentData_map )
{
	for ( std::list<std::string>::const_iterator itr = in_Components.begin(); itr != in_Components.end(); ++itr )
	{

			// Note - the following "new" is not deleted.  This is a memory leak by design. The contents of this 
			// new are needed up unitl the program exits.  Will let the OS clean this up.
			// Also, note that assembled_feat_handle must go on the heap; otherwise, it would get deleted when the stack unwinds.
			ProAsmcomp *assembled_feat_handle = new ProAsmcomp;
			ProMdl     *p_model = new ProMdl;

			std::string ModelNameWithSuffix = AmalgamateModelNameWithSuffix ( 
												in_CADComponentData_map[*itr].name, 
												in_CADComponentData_map[*itr].modelType );

			wchar_t  PartName[ISIS_CHAR_BUFFER_LENGTH];
			ProStringToWstring(PartName, (char *)in_CADComponentData_map[*itr].name.c_str() );

			isis::isis_ProMdlRetrieve_WithDescriptiveErrorMsg(
				*itr, in_CADComponentData_map[*itr].name,   // Added arguments
				PartName,in_CADComponentData_map[*itr].modelType, p_model);	// Original Arguments

			// fixfixfix add try catch here to support the case where the first part added into the assembly
			// does not have the default datums. fixfixfix qqqqqq
			isis::AddComponentToAssembly( (ProAssembly)in_p_asm, (ProSolid)*p_model,assembled_feat_handle);

			// in_CADComponentData_map[itr->ComponentID()].ComponentID =	itr->ComponentID();
			// in_CADComponentData_map[itr->ComponentID()].Name =			itr->Name;

			in_CADComponentData_map[*itr].p_model = p_model;
			in_CADComponentData_map[*itr].modelHandle =	(ProSolid)*p_model;
			in_CADComponentData_map[*itr].assembledFeature =  *assembled_feat_handle;
			in_CADComponentData_map[*itr].componentPaths.push_back((*assembled_feat_handle).id);	

			clog << endl << endl << "   Assembly: " << in_ParentName << "   Added: "  << ModelNameWithSuffix; 
			clog << endl << "       Added Model  componentID:           " <<  in_CADComponentData_map[*itr].componentID;
			clog << endl << "       Added Model  parentComponentID:     " <<  in_CADComponentData_map[*itr].parentComponentID;
			clog << endl << "       Added Model  p_model:               " <<  in_CADComponentData_map[*itr].modelHandle; 
			clog << endl << "       Added Model  modelHandle:           "  << in_CADComponentData_map[*itr].modelHandle;
			clog << endl << "       Added Model  assembledFeature.id:   " <<  in_CADComponentData_map[*itr].assembledFeature.id;
			clog << endl << "       Added Model  assembledFeature.type: " <<  in_CADComponentData_map[*itr].assembledFeature.type;
			clog << endl << "       Added Model  specialInstruction:    " <<  isis::SpecialInstruction_string( in_CADComponentData_map[*itr].specialInstruction );
			

			cout << endl << "   Assembly: " << in_ParentName << "   Added: "  << ModelNameWithSuffix; 

			//////////////////////////////////////////////
			// Set DDP parameters in the assembly feature
			/////////////////////////////////////////////

			if ( in_CADComponentData_map[*itr].componentID.size() > 79 )
			{
				std::stringstream errorString;
				errorString <<
				"The ComponentID string is too long to store as a paramter in the Creo model.  Component ID: " <<  in_CADComponentData_map[*itr].componentID;
				throw isis::application_exception(errorString.str().c_str());
			}		

			ProError      err;
			ProName       name;
			ProParameter  param;
			ProParamvalue value;

			// DDP_COMPONENT_INSTANCE_ID, for now only output this one.  Later, after schema change, add the following 
			// DDP_COMPONENT_ID
			// DDP_COMPONENT_VERSION
			// DDP_COMPONENT_REVISION
			// DDP_DESIGN_ID
			ProStringToWstring(name, "DDP_COMPONENT_INSTANCE_ID" );

			value.type = PRO_PARAM_STRING;
			ProStringToWstring(value.value.s_val, (char *) in_CADComponentData_map[*itr].componentID.c_str() );

			err = ProParameterInit(assembled_feat_handle, name, &param);

			if( err == PRO_TK_E_NOT_FOUND)
				 isis::isis_ProParameterCreate(assembled_feat_handle, name, &value, &param);
			else
				 isis::isis_ProParameterValueSet(&param, &value);

	} // END for ( std::list<std::string>::const_iterator itr = in_Components.begin(); itr != in_Components.end(); ++itr )
								 
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////

// Size-to-Fit components are constrained to Non-Size-to-Fit components (i.e. the constraint points outward), 
// but no other components are constrained to the Size-to-Fit components (no inward pointing constraints).   
// Also, a Size-to-Fit would NOT be constrained to another Size-to-Fit component.  Therefore, if a Size-to-Fit 
// component is constrained to Non-Size-to-Fit components that have already been added to the assembly structure, 
// then the Size-to-Fit components can be added to the current assembly.
void For_SizeToFit_ComponentsWithDependentsPresentAtThisLevel_AddAndConstrain(
									const std::string							&in_AssemblyComponentID,
									std::map<string, isis::CADComponentData>	&in_CADComponentData_map,
									const std::list<std::string>				&ComponentsAtCurrentLevel,
									std::list<std::string>						&in_out_SIZE_TO_FIT_Components )
																			throw ( isis::application_exception )
{	
	
	////////////////////////////////////////////////////
	// First get all existing components at the level
	//////////////////////////////////////////////////
	std::set<std::string> ExistingComponentsSet;
	
	// Add non Size-to-Fit components to existing component set
	for ( std::list<std::string>::const_iterator t = ComponentsAtCurrentLevel.begin(); t != ComponentsAtCurrentLevel.end(); ++t )
	{
		ExistingComponentsSet.insert( *t );
		AddSubordinateComponentIDsToSet( *t,in_CADComponentData_map, ExistingComponentsSet );

	}

	std::list<std::string> Added_in_out_SIZE_TO_FIT_Components;

	for ( std::list<std::string>::const_iterator t = in_out_SIZE_TO_FIT_Components.begin(); t != in_out_SIZE_TO_FIT_Components.end(); ++t )
	{
		if ( AinB( in_CADComponentData_map[*t].dependsOn, ExistingComponentsSet ) )
		{
			std::list<std::string> Single_SIZE_TO_FIT_Component;
			
			// The component path would have been incorrectly maintined up to this point.  
			// Since the component was just added to the assembly. must reset the path.
			in_CADComponentData_map[*t].componentPaths.erase(
										in_CADComponentData_map[*t].componentPaths.begin(),
										in_CADComponentData_map[*t].componentPaths.end());

			Single_SIZE_TO_FIT_Component.push_back(*t);

			
			Add_Subassemblies_and_Parts( in_CADComponentData_map[in_AssemblyComponentID].modelHandle, 
										 in_CADComponentData_map[in_AssemblyComponentID].name,  
										 Single_SIZE_TO_FIT_Component, 
										 in_CADComponentData_map );
			
			//	At this point, the xml assembly tree and in_CADComponentData_map probably do not match the model 
			//  Creo assembly tree.  This is because a SIZE_TO_FIT probably was probably moved higher in the 
			//	assembly tree.  This must be corrected because the tree is used to obtain metrics and if the tree 
			//	is incorrect then the routine that computes the translation between an assembly and its children will fail.

			std::string oldParentID = in_CADComponentData_map[*t].parentComponentID;

			if ( in_AssemblyComponentID != oldParentID )
			{
				// Remove Child from old assembly
				in_CADComponentData_map[oldParentID].children.remove(*t);
				// Add Child to new/correct assembly
				in_CADComponentData_map[in_AssemblyComponentID].children.push_back(*t);
				// Reset parent Id
				in_CADComponentData_map[*t].parentComponentID = in_AssemblyComponentID;
			}

			// No need to update the path to its sub-assemblies/parts.  A SIZE_TO_FIT component will always be a leaf assembly.

			// Can constrain this SIZE_TO_FIT component
			isis::ApplyModelConstraints( &in_CADComponentData_map[in_AssemblyComponentID].modelHandle, //ProSolid	 in_assembly_model,
											Single_SIZE_TO_FIT_Component,
											in_CADComponentData_map);
			//isis::ApplyModelConstraints( (ProSolid*)&in_p_asm, //ProSolid	 in_assembly_model,
			//								Single_SIZE_TO_FIT_Component,
			//								in_CADComponentData_map);

			Added_in_out_SIZE_TO_FIT_Components.push_back(*t);
			
		}
	} // END for ( std::list<std::string>::const_iterator t = in_out_in_out_SIZE_TO_FIT_Components.begin(); t != in_out_in_out_SIZE_TO_FIT_Components.end(); ++t )

	// Remove the added SIZE_TO_FIT components from Added_in_out_SIZE_TO_FIT_Components
	for ( std::list<std::string>::const_iterator t = Added_in_out_SIZE_TO_FIT_Components.begin(); t != Added_in_out_SIZE_TO_FIT_Components.end(); ++t )
	{
		in_out_SIZE_TO_FIT_Components.remove(*t);
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void	RegenerateModel( ProSolid in_p_asm,
						 const std::string in_ParentName,
						 const std::string in_ParentComponentID,
						 bool  &out_RegenerationSucceeded,
						 bool  in_PresentDetailedErrorMessage = false)
{
	bool ProSolidRegenerate_ThrewException = false;
	
	try 
	{
		std::cout << std::endl << std::endl << "   Regenerating:     " << in_ParentName << "  " << in_ParentComponentID;
		std::clog << std::endl << std::endl << "   Regenerating:     " << in_ParentName << "  " << in_ParentComponentID;

		//	"Pro/TOOLKIT Users Guide" Section "Regenerating a Solid":
		//	PRO_REGEN_NO_RESOLVE_MODESpecifies the
		//	No-Resolve mode introduced in Pro/ENGINEER Wildfire 5.0.
		//	This is the default mode in Pro/ENGINEER Wildfire 5.0. In this
		//	mode, if a model and feature regeneration fails, failed features
		//	and children of failed features are created and regeneration of
		//	other features continues.
		//

		isis::isis_ProSolidRegenerate (in_p_asm, PRO_REGEN_NO_RESOLVE_MODE);
	}
	catch (...)
	{
		ProSolidRegenerate_ThrewException = true;
		out_RegenerationSucceeded = false;
	}



	ProSolidRegenerationStatus RegenStatus;
	//PRO_SOLID_FAILED_REGENERATION

	if ( !ProSolidRegenerate_ThrewException )
	{
		isis::isis_ProSolidRegenerationstatusGet(in_p_asm, &RegenStatus);
	}

	if ( ProSolidRegenerate_ThrewException || RegenStatus == PRO_SOLID_FAILED_REGENERATION )
	{
		out_RegenerationSucceeded = false;
		// Treat this as a warning.  Continue building the rest of the assemblies.  The assembly with
		// the regeneration error should open; however, the components that failed to regenerate will show up
		// as red items in the assembly tree.  They will not be positioned properly.

		string err_str;

		if ( in_PresentDetailedErrorMessage )
		{
			err_str = "WARNING - Assembly failed to regenerate. Error caused by Model Name: " + 
				in_ParentName + " ComponentID: " + in_ParentComponentID + "." +
				"\nNote - Assembly/constraint process: " +
				"\n         a) add all the parts/sub-assemblies to the assembly" +
				"\n         b) apply constraints to a part/sub-assembly" +
				"\n         c) repeat step b until all constraints have been applied" +
				"\n         d) regnerate the assembly"  +
				"\nBecause of this process the regeneration could have failed because any one" +
				"\nof the added parts/sub-assemblies would not regenerate, or because the" +
				"\nconstraints used to position a part/sub-assembly were erroneous." +
				"\nThis warning message could appear multiple times.  This is because" +
				"\nonce a failure occurs, all subsequent regenerations will fail." +
				"\nOpen the parent assembly using Creo, and the anomalous" +
				"\nparts/sub-assemblies will appear in red in the assembly tree.\n";
		}
		else
		{
			err_str = "   Regeneration FAILED.";
		}

		std::cout << std::endl << err_str; 
		std::clog << std::endl << err_str; 
	}

	if ( out_RegenerationSucceeded )
	{
		std::cout << std::endl << "   Regeneration succeeded.";
		std::clog << std::endl << "   Regeneration succeeded.";
	}

}
////////////////////////////////////////////////////////////////////////////////////////////////////////////

void AssembleCADComponent( 	const std::string							&in_AssemblyComponentID,
							const std::string							&in_WORKING_DIR,
							//isis::ComponentData_struct					&in_ParentComponentData,
							std::map<string, isis::CADComponentData>	&in_CADComponentData_map,
							std::list<std::string>						&in_out_SIZE_TO_FIT_Components,
							bool										&out_RegenerationSucceeded )
																		throw (isis::application_exception)
{
		
	// *************************************
	// Build Sub-assemblies
	// **************************************
//	for ( AssemblyType::CADComponent_type::CADComponent_const_iterator i(in_Begin); i != in_End; ++i )
	for ( std::list<string>::const_iterator i (in_CADComponentData_map[in_AssemblyComponentID].children.begin());
		  i != in_CADComponentData_map[in_AssemblyComponentID].children.end();
		  ++i )
	{
			//ProMdlType pro_model_type = isis::ProMdlType_enum( i->Type());
			//if ( pro_model_type == PRO_MDL_ASSEMBLY  && i->CADComponent().size() > 0) // Note - an existing assmbly would have no children. 
			if ( in_CADComponentData_map[*i].modelType == PRO_MDL_ASSEMBLY  && in_CADComponentData_map[*i].children.size() > 0) // Note - an existing assmbly would have no children. 
			{
				//isis::C8omponentData_struct ParentComponentData_temp( i->ComponentID(), i->Name() );

				AssembleCADComponent( *i,
									  in_WORKING_DIR,
									  //ParentComponentData_temp,
									  in_CADComponentData_map,
									  in_out_SIZE_TO_FIT_Components, 
									  out_RegenerationSucceeded );			
			}
	}

	wchar_t TempString_wide[ISIS_CHAR_BUFFER_LENGTH];
	string temp_string;	

	// *********************
	// 1) Delete current assembly if it exists
	// 2) Copy template assembly to new assembly name
	// 3) Retrieve copied assembly.
	// *********************
	
	//wchar_t  AssemblyName[PRO_FAMILY_NAME_SIZE];
	wchar_t  AssemblyName[ISIS_CHAR_BUFFER_LENGTH];
	
	ProMdl     p_asm;
	//ProStringToWstring(AssemblyName, (char *)in_ParentComponentData.Name.c_str() );
	ProStringToWstring(AssemblyName, (char *)in_CADComponentData_map[in_AssemblyComponentID].name.c_str() );
	
	// Delete assembly file if it exists. This program will rebuild the assembly.
	//isis::ExecuteSystemCommand("IF EXIST " + in_ParentComponentData.Name + ".asm.*" +  " DEL " + in_ParentComponentData.Name + ".asm.*" );
	isis::IfFileExists_DeleteFile( in_WORKING_DIR + "\\" + in_CADComponentData_map[in_AssemblyComponentID].name + ".asm.*");

	// Note - ???? should change this to copy the correct template based on the units
	// in the xml file
	isis::isis_ProMdlfileCopy ( (ProMdlType)PRO_ASSEMBLY, 
								ProStringToWstring(TempString_wide,(char*)TEMPLATE_MODEL_NAME_METRIC.c_str()), AssemblyName);

	isis::isis_ProMdlRetrieve_WithDescriptiveErrorMsg(
		   in_AssemblyComponentID, in_CADComponentData_map[in_AssemblyComponentID].name,  // Added arguments
		   AssemblyName,(ProMdlType)PRO_ASSEMBLY,&p_asm);  // Original arguments


	clog << endl << endl << "######################### Begin Assembly " << in_CADComponentData_map[in_AssemblyComponentID].name << " ########################";
	clog << std::endl << std::endl << "Created Assembly: " << in_CADComponentData_map[in_AssemblyComponentID].name;
	std::cout << std::endl << std::endl << "Created Assembly: " << in_CADComponentData_map[in_AssemblyComponentID].name;


	in_CADComponentData_map[in_AssemblyComponentID].modelHandle =(ProSolid)p_asm;
	// *************************************
	// Add all detail parts / sub-assemblies 
	// **************************************

	// Sort the order of assembly.  This is necessary because a constraint cannot reference
	// a model lower down in the tree.

	std::list<std::string> UnsortedComponents;
	std::list<std::string> SIZE_TO_FIT_Components;

	
	BuildFiltereListOfComponents(	in_AssemblyComponentID, 
									in_CADComponentData_map, 
									INCLUDE_ALL_COMPONENTS_EXCEPT_SIZE_TO_FIT, 
									UnsortedComponents);

	BuildFiltereListOfComponents(	in_AssemblyComponentID, 
									in_CADComponentData_map, 
									INCLUDE_ONLY_SIZE_TO_FIT, 
									SIZE_TO_FIT_Components);

	for ( std::list<std::string>::const_iterator t = SIZE_TO_FIT_Components.begin(); 
		  t != SIZE_TO_FIT_Components.end(); ++t )
	{
		in_out_SIZE_TO_FIT_Components.push_back( *t );
	}

	std::list<std::string> SortedComponents;

	SortCADComponents(	in_CADComponentData_map[in_AssemblyComponentID].name, 
						in_CADComponentData_map[in_AssemblyComponentID].componentID,
						UnsortedComponents,
						in_CADComponentData_map,
						SortedComponents );

	clog  << std::endl << std::endl << "BEGIN ----------   Unsorted List   ------------";
	LogSortingList(	"   ", 
					"UnsortedComponents, Components",
					in_CADComponentData_map[in_AssemblyComponentID].name,
					in_CADComponentData_map[in_AssemblyComponentID].componentID,
					in_CADComponentData_map,
					UnsortedComponents );
	clog << std::endl << "END ------------   Unsorted List   ------------";

	clog  << std::endl << std::endl << "BEGIN ----------   Sorted List   --------------";
	LogSortingList(	"", 
					"SortedComponents, Components",
					in_CADComponentData_map[in_AssemblyComponentID].name,
					in_CADComponentData_map[in_AssemblyComponentID].componentID,
					in_CADComponentData_map,
					SortedComponents );
	clog << std::endl << "END ------------   Sorted List   --------------";

	////////////////////////////////////////
	// Add sub-assemblies and detail parts
	////////////////////////////////////////
	Add_Subassemblies_and_Parts( p_asm, 
								 in_CADComponentData_map[in_AssemblyComponentID].name, 
								 SortedComponents, 
								 in_CADComponentData_map);

	//stream_CADComponentData_map( in_CADComponentData_map, clog );

	/////////////////////////////////////////////////////////////////////
	// For each added assembly update the path to its sub-assemblies/parts 
	// to include feature id of the added assembly. 
	//////////////////////////////////////////////////////////////////////
	// Note - the following also updates the SIZE_TO_FIT components.  That
	// path information will be corrected when the SIZE_TO_FIT components 
	// are added.  We cannot skip the SIZE_TO_FIT components because once
	// they are added, the path information must be maintained.
	//for ( AssemblyType::CADComponent_type::CADComponent_const_iterator t(in_Begin); t != in_End; ++t )
	//{
	for ( std::list<string>::const_iterator t (in_CADComponentData_map[in_AssemblyComponentID].children.begin());
	  t != in_CADComponentData_map[in_AssemblyComponentID].children.end();
	  ++t )
	{
		//ProMdlType pro_model_type = isis::ProMdlType_enum( in_CADComponentData_map[t->ComponentID()].type_string);
		if (  in_CADComponentData_map[*t].modelType == PRO_MDL_ASSEMBLY  ) 
		{
			UpdateAllSubassemblyComponentsToIncludeParentAssemblyPath( 
											*t,
											in_CADComponentData_map,
											in_CADComponentData_map[*t].assembledFeature.id );
		}
	}

	///////////////////////////////////
	// Set Parametric Values
	///////////////////////////////////
	ApplyParametricParameters(SortedComponents, in_CADComponentData_map);

	///////////////////////////////////
	// Log tree contents at this level
	///////////////////////////////////
	//clog << std::endl << std::endl << "************ Begin Sub-Tree (Name ComponentID) ***************";
	//stream_IndentedAssemblyTree( in_Begin, in_End, clog );
	//clog << std::endl << "************ End Sub-Tree  (Name ComponentID) *****************";

	//clog << std::endl << "************ Begin Sub-Tree (Detailed Info) ***************";
	//stream_AssemblyTree( in_Begin, in_End, clog );
	//clog << std::endl << "************ End Sub-Tree (Detailed Info) ***************";

	//////////////////////////////////////////////////
	// Log in_CADComponentData_map contents
	//////////////////////////////////////////////////
	//clog << std::endl << std::endl << "************** Begin map CADComponent **************"; 
	//stream_CADComponentData_map( in_CADComponentData_map, clog );
	//clog << std::endl << "************** End map CADComponent **************";


	isis::ApplyModelConstraints( (ProSolid*)&p_asm, //ProSolid	 in_assembly_model,
									SortedComponents,
									in_CADComponentData_map,
									true);


	////////////////////////////////////////////////////////////////////////////
	// Determine if SIZE_TO_FIT Componenets can be added and set constrained
	///////////////////////////////////////////////////////////////////////////

	For_SizeToFit_ComponentsWithDependentsPresentAtThisLevel_AddAndConstrain(
									in_AssemblyComponentID,
									in_CADComponentData_map,
									SortedComponents,
									in_out_SIZE_TO_FIT_Components );

	/////////////////////
	// Regenerate Model
	/////////////////////
	// When using Creo interactively, the first regenerate will fail and the second, in some cases, will
	// succeed.  Therefore, if the first regenerate fails, try regnerate a second time. 

	RegenerateModel(	(ProSolid)p_asm,
						in_CADComponentData_map[in_AssemblyComponentID].name,
						in_CADComponentData_map[in_AssemblyComponentID].componentID,
						out_RegenerationSucceeded);

	if ( !out_RegenerationSucceeded )  // try to regenerate a second time
	{
		out_RegenerationSucceeded = true;
		RegenerateModel(	(ProSolid)p_asm,
							in_CADComponentData_map[in_AssemblyComponentID].name,
							in_CADComponentData_map[in_AssemblyComponentID].componentID,
							out_RegenerationSucceeded, true);
	}

	if ( !out_RegenerationSucceeded )
	{
		// Try regeneration each sub-component seperately to identify the non-regeneration part/sub-assembly

		std::clog << std::endl << std::endl << "Regenerating parts/sub-assemblies. This will help indicate which part/sub-assembly would not generate."; 
		std::cout << std::endl << std::endl << "Regenerating parts/sub-assemblies. This will help indicate which part/sub-assembly would not generate."; 
		bool local_RegenerationSucceeded;
		for ( std::list<string>::const_iterator i_regen (in_CADComponentData_map[in_AssemblyComponentID].children.begin());
		  i_regen != in_CADComponentData_map[in_AssemblyComponentID].children.end();
		  ++i_regen )
		{
			local_RegenerationSucceeded = true;
			RegenerateModel( in_CADComponentData_map[*i_regen].modelHandle,
							 in_CADComponentData_map[*i_regen].name,
							 in_CADComponentData_map[*i_regen].componentID,
							 local_RegenerationSucceeded, false);

		}
	}

	///////////////////////////////////
	// Output Step Files
	///////////////////////////////////
	//std::string createAnalysisDir = "if not exist STEP mkdir STEP";

	/*
	int Arg_2 = 1;

	std::cout << std::endl << "Creating STEP file";

	ProName		optionName;
	ProPath		optionValue;

	wcscpy( optionName, L"step_export_format");
	wcscpy( optionValue, L"ap203_e2");
	//wcscpy( optionValue, L"ap214_is");
	isis_ProConfigoptSet(optionName, optionValue  );
	
	wcscpy( optionName, L"intf3d_out_datums_by_default");
	wcscpy( optionValue, L"yes");
	isis_ProConfigoptSet(optionName, optionValue  );

	wcscpy( optionName, L"step_export_ap214_asm_def_mode");
	wcscpy( optionValue, L"separate_parts_only");
	isis_ProConfigoptSet(optionName, optionValue  );

	// Also should set step_out_suppl_geom, set_out_material, 

	isis_ProOutputFileWrite(p_asm,
							AssemblyName,
							PRO_STEP_FILE,
							NULL,
							&Arg_2,
							NULL,
							NULL);
    */
	/////////////////////////
	// Save the assembly
	/////////////////////////
	isis::isis_ProMdlSave(p_asm);

	std::cout << std::endl << std::endl << "Saved Assembly:   " << in_CADComponentData_map[in_AssemblyComponentID].name + "  " + in_CADComponentData_map[in_AssemblyComponentID].componentID;
	std::clog << std::endl << std::endl << "Saved Assembly:   " << in_CADComponentData_map[in_AssemblyComponentID].name + "  " + in_CADComponentData_map[in_AssemblyComponentID].componentID;

		clog << endl << endl << "########################## End Assembly " << in_CADComponentData_map[in_AssemblyComponentID].name << " #########################";

	// Delete the version 1 of the assembly file.  It was just a temporary file that was copied from the 
	// template file.
	isis::DeleteFile(in_WORKING_DIR + "\\" + in_CADComponentData_map[in_AssemblyComponentID].name + ".asm.1" );

}

//////////////////////////////////////////////////////////////////////////////////////////////////////
void BuildAssembly( const std::string									&in_AssemblyComponentID, 
					const std::string									&in_WORKING_DIR,
					std::map<string, isis::CADComponentData>			&in_CADComponentData_map,
					bool												&out_RegenerationSucceeded )
																	throw (isis::application_exception)
{

	//clog << endl;
	//clog << endl << "Component Name: "  << in_CADComponentData_map[in_AssemblyComponentID].name;
	//clog << endl<<	"Component ID:   "  << in_CADComponentData_map[in_AssemblyComponentID].componentID;
	//clog << endl << "Component Type: "  << ProMdlType_string( in_CADComponentData_map[in_AssemblyComponentID].modelType );
	//clog << endl;

   clog << endl << endl << "*************************** Begin Assembly Creation **************************";

	std::list<std::string>  SIZE_TO_FIT_Components; 

	out_RegenerationSucceeded = true;
	AssembleCADComponent(	in_AssemblyComponentID,
							in_WORKING_DIR,
							in_CADComponentData_map,
							SIZE_TO_FIT_Components,
							out_RegenerationSucceeded);



	// At this point all SIZE_TO_FIT Components should have been assembled.  If not, throw an exception
	if ( SIZE_TO_FIT_Components.size() != 0 )
	{
		string err_str = "All assemblies were built; however, some Size-to-Fit components were not added. " +
						std::string(" This condition would occur if the Size-to-Fit components were constrained to ComponentIDs") + 
						 " that were not present in the XML file. The unassembled Size-to-Fit components are: ";

		std::string comman_or_blank;
		for ( std::list<string>::const_iterator i ( SIZE_TO_FIT_Components.begin());
			  i !=  SIZE_TO_FIT_Components.end();
			  ++i )
		{
			err_str += comman_or_blank + in_CADComponentData_map[*i].componentID + " " + in_CADComponentData_map[*i].componentID;
			comman_or_blank = ", ";
		}
		err_str += ".";										
		throw isis::application_exception(err_str.c_str());  
	}

	clog << endl << endl << "************************** End Assembly Creation *************************";


}

//////////////////////////////////////////////////////////////////////////////////////////////////////
void CopyModels(const std::map<std::string, std::string>			&in_ToPartName_FromPartName )
																	throw (isis::application_exception)
{

	std::set<std::string> savedToWorkingDirSourceModels;

	for( std::map<std::string, std::string>::const_iterator i(in_ToPartName_FromPartName.begin());
		 i != in_ToPartName_FromPartName.end();
		 ++i )

	{
		//ProMdl  handle;
		ProFamilyName  fromModelName;
		ProFamilyName  toModelName;

		ProStringToWstring(fromModelName, (char *)i->second.c_str() );
		ProStringToWstring(toModelName, (char *)i->first.c_str() );
	
		// Assure that the source model is saved to the working directory only once.
		if ( savedToWorkingDirSourceModels.find( i->second ) == savedToWorkingDirSourceModels.end() )
		{
			// Since the source model (i.e. fromModelName) could be anywhere in the search path, we
			// must open the source model and save it to force a copy to exist in the working directory.
			ProMdl     p_model; 
			isis::isis_ProMdlRetrieve(fromModelName,PRO_MDL_PART, &p_model);
			isis::isis_ProMdlSave(p_model);
			savedToWorkingDirSourceModels.insert(i->second);
		}

		isis::isis_ProMdlfileCopy (PRO_MDL_PART, fromModelName, toModelName);
	}
}

} // end namespace isis