/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
/*! \file CADCreoParametricCreateAssembly.cpp
    \brief The main(...) function for CADCreoParametricCreateAssembly.exe
   
	CADCreoParametricCreateAssembly.exe reads an XML file that describes an assembly, and then builds the 
	assembly with "Creo Parametric 1.0" (i.e. PTC's ( http://www.ptc.com/ )  parametric solids modeler). 
 
 
	Pre-Conditions:

		1.	"Creo Parametric 1.0" must be installed and functioning properly on your machine.  The license for 
			"Creo Parametric 1.0" does not need the the Toolkit license.  The toolkit license is only 
			needed during development.

		2.	The environment variables must be set per 
			...\Meta SVN\sandbox\CAD\Installs\Proe ISIS\Extensions\0ReadMe.txt

		3.	The requirements for the arguments passed to the exe follow: 

			Arg #   Required    Description

-           0       Yes         exe 
		
-           1       Yes         workingDir	
			
-           2       Yes         ASSEMBLY_XML_FILE		e.g. RLC_Assembly_5_CAD.xml
			
-           3       Yes         LOG_FILE				e.g. RLC_Assembly_5_CAD.xml.log	
			
-           4       No          Prompt before exiting	e.g. YES  or NO,  NO is the default.

			Note - an example bat file is located at
			...META_SVN\trunk\deploy\CAD_Installs\Proe ISIS Extensions\docs\examples\CADCreoParametricCreateAssembly.bat

		4.	The input xml must be compliant with 
			...META_SVN\trunk\deploy\CAD_Installs\Proe ISIS Extensions\schemas\AssemblyInterface.xsd


	Post-Conditions:

	1.	If an exception occurs, the program will log the error message, display the error message to the console, 
		and exit.  Some reasons that exceptions could occur follow:

-              Could not locate a part/assembly.

-              Could not find a datum in a part/assembly.

-              The input XML file constrained a Non-Size-to-Fit part/assembly to a Size-to-Fit part/assembly.

	2.	If a warning occurs, the program will log the warning message, display the warning message to the console, 
		and continue processing.  An example of a warning would be if the assembly would not regenerate.  The assembly
		would have been built and saved, but when opened via Creo, the anomalous models would appear in red in the 
		history tree.

	3.	If no exceptions occur, the assemblies defined in the XML are built and saved in the working directory.


	Special Notes:

	1.	For a parametric parts, if a parameter is modified, the modification applies to all instances of the component.  
		Therefore, for this program, a parametric part should either appear only one time in the assembly or if it 
		appears multiple times, then it should have the same parameters for each instance.

	2.	This program uses UDM 64 bit version to parse the XML file.   

*/

#include <isis_application_exception.h>
//#include <AssemblyInterface.hxx>  //// this is needed for the time functions and schema exception.
#include <BuildAssembly.h>
#include <CommonUtilities.h>
#include <DiagnosticUtilities.h>
#include <ISISConstants.h>
#include <CommonStructures.h>
#include <XMLToProEStructures.h>
#include <ProEStructuresUtils.h>
#include <Metrics.h>
#include <FiniteElementAnalysis.h>
#include <ModelMaterials.h>
#include <AssembleUtils.h>
#include <MaterialProperties.h>
#include "CADEnvironmentSettings.h"
#include <Computations.h>
#include <DataExchange.h>
#include "WindowsHDependentCommonFunctions.h"
#include <string>
#include <sstream>
#include <iostream>
#include <fstream>
#include <map>
#include <time.h>

int main( int argc, char *argv[] )
{
    
	// argc != 6 && argc != 7  option should be deleted when CyPhy2CAD outputs then new xml format
	if ( argc != 4 && argc != 5 &&   // new case: %EXE%      %workingDir%      %ASSEMBLY_XML_FILE%     %LOG_FILE%     %EXIT_PROMPT%
		 argc != 6 && argc != 7 )    // old case: %EXE%     %PROE_START_CMD%    "%PROE_ISIS_EXTENSIONS%\."     %workingDir%      %ASSEMBLY_XML_FILE%     %LOG_FILE%     %EXIT_PROMPT%
    {
	  printf("Usage Option 1: \n%PROE_START_CMD%    \"%PROE_ISIS_EXTENSIONS%\"     %workingDir%      %ASSEMBLY_XML_FILE%     %LOG_FILE%   [Optional]%EXIT_PROMPT%\n");
	  printf("Usage Option 2: \n%workingDir%      %ASSEMBLY_XML_FILE%     %LOG_FILE%   [Optional]%EXIT_PROMPT%\n");
	  printf("\n See \"C:\\Program Files\\Proe ISIS Extensions\\docs\\examples\\CADCreoParametricCreateAssembly.bat\""); 
      printf("\n or  \"%PROE_ISIS_EXTENSIONS%\"\\docs\\examples\\CADCreoParametricCreateAssembly.bat\" for an example of how to invoke this executable."); 
	  printf("\nType Enter to exit.");
	  getc(stdin);
      exit(1);
	}
	

	int ExitCode = 0;

	std::string			creoStartCommand;
	std::string			proeIsisExtensionsDir;
	std::string			workingDir;
	std::string			cADPartsLibDir;
	std::string			xmlInputFile_PathAndFileName;
	std:string			logFile_PathAndFileName;
	std::string			templateFile_PathAndFileName;
	std::stringstream	exceptionErrorStringStream;
	bool				promptBeforeExiting = true;

	bool Pro_E_Running = false;
	bool Template_Copied = false;

	std::ofstream LogFile;


	try
	{
		bool regenerationSucceeded_ForAllAssemblies = true;

		//////////////////////////////////
		// Open Log file
		/////////////////////////////////////////////////
		// logfile will always be the fourth argument
		// e.g. %EXE%     %WORKING_DIR%	   %PARTS_DIR%      %ASSEMBLY_XML_FILE%     %LOG_FILE%     %EXIT_PROMPT%
		int logFileArg = 4;
		std::string logFileName = argv[logFileArg];
		// if argv[logFileArg] does not conatain a "\", then assume a full path was not passed 
		// if full path was not passed, then put the log file in .\log directory if possible.
		if (  logFileName.find("\\") ==  std::string::npos )
		{
			// Only the log file name was passed via argv[logFileArg].  No path information passed.
			logFileName = "CAD_" + logFileName;
			std::string logDirectory = "log";
			bool logDirectoryExists = false;
			if ( isis::DirectoryExists( logDirectory ) )
			{
				logDirectoryExists = true;
			}
			else
			{
				std::string createLogDir = "if not exist " +  logDirectory +  " mkdir  " + logDirectory;
				isis::ExecuteSystemCommand( createLogDir);
				// Must check the case where a file named "log" exists.  For that case, the above ExecuteSystemCommand
				// would not have made a "log" directory.
				if ( isis::DirectoryExists( logDirectory )) logDirectoryExists = true;
			}

			if ( logDirectoryExists ) 
				logFile_PathAndFileName = "log\\" + logFileName;
			else
				logFile_PathAndFileName = logFileName;	
		}
		else
		{
			logFile_PathAndFileName = logFileName;
		}

		LogFile.open (logFile_PathAndFileName.c_str(), std::ios::out | std::ios::trunc );
		if (LogFile.is_open())
			clog.rdbuf(LogFile.rdbuf());
		else
		{
			std::string TempError = "Could not open log file: " + logFile_PathAndFileName;
			throw isis::application_exception(TempError.c_str());
		} 

		std::string programName_Version_TimeStamp;
		programName_Version_TimeStamp = "CADCreoParametricCreateAssembly " + isis::ASSEMBLE_PTC_VERSION + "      ";

		///////////////////
		// Add Time Stamp
		///////////////////

		programName_Version_TimeStamp += isis::GetDayMonthTimeYear();
		clog << programName_Version_TimeStamp;

		clog << std::endl << std::endl<< "Notes: ";
		clog << std::endl << "   1. The ComponentIDs in this file are equivalent to Component-Instance-IDs in CyPhy.";
		clog << std::endl << "   2. To map ComponentIDs in this file to AVM-IDs, see .\\log\\CyPhy2CAD.log.";
		clog << std::endl;

		time_t time_start; /* calendar time */
		time_start=time(NULL); /* get current cal time */

		isis::SetCreoEnvirVariable_RetrieveSystemSettings(	argc, argv,
															creoStartCommand,
															proeIsisExtensionsDir,
															workingDir,
															cADPartsLibDir,
															xmlInputFile_PathAndFileName,
															logFile_PathAndFileName,
															templateFile_PathAndFileName,
															promptBeforeExiting );
		

		// Verify that the XML file exists.
		if (!isis::FileExists( xmlInputFile_PathAndFileName.c_str()))
		{
			std::string TempError = "Could not find the assembly definition xml file, file name: " + xmlInputFile_PathAndFileName;
			throw isis::application_exception(TempError.c_str());
		} 

		std::map<std::string, isis::CADComponentData> CADComponentData_map;
		isis::CADAssemblies CADComponentAssemblies;

		FromXMLFile_PopulateCADComponentAssemblyAndMap( xmlInputFile_PathAndFileName.c_str(),
												   CADComponentAssemblies,
												   CADComponentData_map);


		if ( CADComponentAssemblies.topLevelAssemblies.size() == 0 ) 
		{
			clog << endl << endl << "*************************** Begin Assembly Creation **************************";
			std::stringstream informationString;
			informationString << "No assemblies were created because the input xml file, " 
					  << std::endl << xmlInputFile_PathAndFileName 
					  << ", did not define any assemblies.";
			clog << endl << informationString.str() << endl;
			cout << endl << informationString.str() << endl;

			if ( CADComponentAssemblies.unassembledComponents.size() == 0 )
			{
				std::stringstream errorString;
				errorString <<
						"The input xml file, "
						 << std::endl << xmlInputFile_PathAndFileName 
						 << ", did not define any assemblies nor did it define unassembled parts/sub-assemblies.  "
						 << std::endl << "The input xml file must specify assemblies and/or unassembled parts/sub-assemblies."; 
						throw isis::application_exception(errorString.str().c_str());
			}
			clog << endl << "************************** End Assembly Creation *****************************" << endl;
		}

		// Rename parametric parts that have multiple instances of the same part.
		std::map<std::string, std::string>  ToPartName_FromPartName_map; 
		int UniqueNameIndex = 1;
		isis::ModifyToHaveAUniqueNameForEachParametricPartOrAssembly( UniqueNameIndex, CADComponentData_map, ToPartName_FromPartName_map );

		std::clog << std::endl << std::endl << "************** Begin Modified Part Names for Multiple Parametric Parts *****************";
		std::clog << std::endl << "From_Part_Name   To_Part_Name";
		for each( std::pair<std::string, std::string> i in ToPartName_FromPartName_map) 
		{
			std::clog << std::endl << i.second << "  " <<  i.second;
		}
		std::clog << std::endl << std::endl << "************** End Modified Part Names for Multiple Parametric Parts *****************";


		// If this is an analysis run, must modify so that each part has a unique name.  
		// e.g. if Part_A appears three times
		// in the assembly, then copies must be made such that there is a Part_A, Part_A_C001, and Part_A_C002.
		// This is needed to assure that the material properties in the generated mesh will have a unique name.
		// A subsequent part of this program will create dummy materials for each part and the dummy materials
		// will have the name of the form ZZ_FEA_MAT_1_ZZ, ZZ_FEA_MAT_2_ZZ... This will assure that all materials
		// have a unique name and thus have a separate MAT property in the Nastran deck.
		// Note - We are assuming if any assembly (CADComponentAssemblies) is an analysis run then all 
		// assemblies will be for analaysis purposes.  We will not mix building assembies for analsys purposes
		// and building assembies for general purposes.
		
		bool analysisRun = IsAnAnlysisRun(CADComponentAssemblies);

		//std::cout << std::endl << "*****analysisRun: " << analysisRun;

		if ( analysisRun)
		{
			/////////////////////////////////////////////////
			// Verify One and only One Assembly per FEA Run
			/////////////////////////////////////////////////
			if ( CADComponentAssemblies.topLevelAssemblies.size() != 1 )
			{
				std::stringstream errorString;
				errorString << "One and only one assembly allowed per FEA run.  The XML file defines " <<
				CADComponentAssemblies.topLevelAssemblies.size() << std::string(" assemblies.");
					throw isis::application_exception(errorString.str().c_str());
			}

			if ( CADComponentAssemblies.unassembledComponents.size() != 0 )
			{
				std::stringstream errorString;
				errorString << "Unassembled components were found.  Cannot perform FEA on partial assemblies.  Number of unassembled components: " <<
				CADComponentAssemblies.unassembledComponents.size();
					throw isis::application_exception(errorString.str().c_str());
			}

			////////////////////////////////////////////////////////////
			// Uniquely name parts for parts that occur multiple times
			////////////////////////////////////////////////////////////
 			isis::ModifyToHaveAUniqueNameForEachPart( UniqueNameIndex, CADComponentData_map, ToPartName_FromPartName_map );

			////////////////////////////
			// Log Part Copy/Renames
			///////////////////////////
			std::clog << std::endl << std::endl << "************** Begin Modified Part Names for Analysis Purposes *****************";
			std::clog << std::endl << "From_Part_Name   To_Part_Name";

			for each( std::pair<std::string, std::string> i in ToPartName_FromPartName_map) 
			{
				 std::clog << std::endl << i.second << "  " <<  i.first;
			}
			std::clog << std::endl << std::endl << "************** End Modified Part Names for Analysis Purposes *****************";
		}

		// Add the depends-on information to the CADComponentData
		isis::Add_dependsOn( CADComponentData_map );

		//stream_CADComponentData_map(CADComponentData_map, clog );
		//stream_CADComponentData_map(CADComponentData_map, cout );

		//for ( std::list<isis::TopLevelAssemblyData>::const_iterator i(CADComponentAssemblies.topLevelAssemblies.begin());
		//	  i != CADComponentAssemblies.topLevelAssemblies.end();
		//	  ++i )
		for each( isis::TopLevelAssemblyData i in CADComponentAssemblies.topLevelAssemblies)
		{

			std::clog << std::endl << std::endl << "************** Begin Entire Tree For a Single Assembly  *****************";

			std::clog << std::endl << std::endl << "************** Begin Analysis Data For a Single Assembly  *****************";
			stream_AnalysisInputData( i.analysesCAD, clog);
			std::clog << std::endl << std::endl << "************** End Analysis Data For a Single Assembly  *****************";

			std::clog << std::endl << std::endl << "************** Begin Entire Component Data Tree (CAD Internal Structures) *****************";
			stream_AssemblyCADComponentData( i.assemblyComponentID, CADComponentData_map, clog );
			std::clog << std::endl << std::endl << "************** Begin Entire Component Data Tree (CAD Internal Structures) *****************";

			std::clog << std::endl << std::endl << "************** End Entire Tree For a Single Assembly  *****************";
		}


		///////////////////////////////////////////////////////////////////////////////////
		// Check for Non-Size-To-Fit components being constrained to Size-To-Fit components
		////////////////////////////////////////////////////////////////////////////////////
		std::set<std::string> TempIntersection = NonSizeToFitComponents_ReferencedBy_SizeToFitConstraints( CADComponentData_map );
		if ( TempIntersection.size() > 0 )
		{
			string err_str = "Erroneous XML File: A NON_SIZE_TO_FIT component cannot be constrained to a SIZE_TO_FIT component. " +
				std::string("The SIZE_TO_FIT component(s) (i.e. ComponentIDs) that were erroneously referenced are:");
			for ( std::set<std::string>::const_iterator i(TempIntersection.begin()); i != TempIntersection.end(); ++i )
			{
				err_str += " " + *i;
			}
			err_str += ".  Search for the ComponentIDs in the log file to identify the places where the SIZE_TO_FIT components are erroneously referenced.";
			throw isis::application_exception(err_str.c_str());  

		}

		std::string configPro_PathAndFileName = workingDir + "\\config.pro";
		// * 1-10-2013 Cphy2CAD now creates the search_META.pro" std::string searchMetaPro_PathAndFileName = workingDir + "\\search_META.pro";
		isis::IfFileExists_DeleteFile( configPro_PathAndFileName);
		// * 1-10-2013 Cphy2CAD now creates the search_META.pro" isis::IfFileExists_DeleteFile( searchMetaPro_PathAndFileName);

		////////////////////////////////////////////////
		// Write config.pro 
		///////////////////////////////////////////////
		ofstream config_Pro;
		std::string searchMetaFileName = ".\\search_META.pro";
		config_Pro.open (configPro_PathAndFileName );
		config_Pro << "override_store_back yes\n";
		config_Pro << "search_path_file " <<  searchMetaFileName;
		config_Pro.close();

		////////////////////////////////////////////////
		// Write search_META.pro for X+0 Clients
		///////////////////////////////////////////////
		// If the user has META Tools x+0, CAD jobs sent to an x+4 server will fail. They fail because starting 
		// with X+2, the Creo search path file (search_META.pro) was created via CyPhy2CAD. With x+0, the 
		// search path file was created via CADCreoParametricCreateAssembly.exe. With the combination 
		// client x+0 and server x+4, the search path is never created. This results in 
		// CADCreoParametricCreateAssembly.exe not being able to find the CAD models. The correction is 
		// for the case where search_META.pro does not exists, then  create it.
		//  X+0 Meta 13.0, released 1/14/2013
		//  X+2 Meta 13.2  released 1/28/2013
		//  X+4 Meta 13.4  released 2/11/2013
		//  NOTE - the following code must be kept because if this program was invoked autonomously,
		//  then search_META.pro must be created.

		if ( !isis::FileExists( searchMetaFileName.c_str() ))
		{
			// Must create ".\search_META.pro"

			std::clog << std::endl << "File .\\search_META.pro does not exist, creating .\\search_META.pro...";

			// Determine if .\Cad_Directory should be added to the search path 
			bool add_CADDirectory_ToSearchPath = false;
			std::string cadDirectoryName =  ".\\Cad_Directory";
			if ( isis::DirectoryExists( cadDirectoryName )) add_CADDirectory_ToSearchPath = true;

			// Determine if cADPartsLibDir (derived from the bat file) should be added to the search path
			bool add_CADPartsLibDir_ToSearchPath = true;
			if ( add_CADDirectory_ToSearchPath )
			{
				// Check if cADPartsLibDir and cadDirectoryName are the3 same
				std::string tempCADPartsLibDir = isis::ConvertToUpperCase( cADPartsLibDir );
				if( tempCADPartsLibDir.find(isis::ConvertToUpperCase(cadDirectoryName)) !=  std::string::npos )
				{
					add_CADPartsLibDir_ToSearchPath = false;
				}
			}

			// Create .\search_META.pro
			ofstream search_Meta_Pro;
			search_Meta_Pro.open (searchMetaFileName );
			if ( search_Meta_Pro.is_open() )
			{
				bool addLineFeed = false;
				if ( add_CADPartsLibDir_ToSearchPath )
				{
					// Check if cADPartsLibDir is in double quotes.  If not add double quotes.
					// This is necessary because Creo Config options with spaces must be enclosed in double quotes.
					if ( cADPartsLibDir.find("\"")  == string::npos )	cADPartsLibDir = "\"" + cADPartsLibDir + "\"";
					search_Meta_Pro << cADPartsLibDir;
					addLineFeed = true;
				}

				if ( addLineFeed ) search_Meta_Pro << std::endl;

				if ( add_CADDirectory_ToSearchPath ) search_Meta_Pro  << cadDirectoryName;

				search_Meta_Pro.close();		
				std::clog << std::endl << "File .\\search_META.pro created.";
			}
			else
			{
				std::string TempError = "Could not open file: " + searchMetaFileName;
				throw isis::application_exception(TempError.c_str());
			}
		}
		else
		{
			std::clog << std::endl << "File .\\search_META.pro exists and was created prior to running CADCreoParametricCreateAssembly.exe.";
		}

		/////////////////////////////
		/////// Start Pro/E /////////
		/////////////////////////////
		cout << endl << "CADCreoParametricCreateAssembly "<< isis::ASSEMBLE_PTC_VERSION;
		cout << endl << endl << "Starting Creo-Parametric, this takes about 10 seconds...";

		char tempBuffer[1024];
		strcpy(tempBuffer, creoStartCommand.c_str() );
		isis::isis_ProEngineerStart(tempBuffer,"");

		Pro_E_Running = true;
		cout <<  endl << "Creo-Parametric successfully started." << endl;


		/////////////////////////////////////////////////
		// Set Parts Library Search Patch
		////////////////////////////////////////////////
		//std::string configPro_PathAndFileName = workingDir + "\\config.pro";
		// * 1-10-2013 Cphy2CAD now creates the search_META.pro" std::string searchMetaPro_PathAndFileName = workingDir + "\\search_META.pro";
		//isis::IfFileExists_DeleteFile( configPro_PathAndFileName);
		// * 1-10-2013 Cphy2CAD now creates the search_META.pro" isis::IfFileExists_DeleteFile( searchMetaPro_PathAndFileName);


		// * 1-10-2013 Cphy2CAD now creates the search_META.pro".  This program only needs to create the
		// the config.pro file
		//if ( cADPartsLibDir.size() > 0 )
		//{
			//ProName		optionName;
			//ProPath		optionValue;
			

			// Check if cADPartsLibDir is in double quotes.  If not add double quotes.
			// This is necessary because Creo Config options with spaces must be enclosed in double quotes.
			//if ( cADPartsLibDir.find("\"")  == string::npos )	cADPartsLibDir = "\"" + cADPartsLibDir + "\"";
			//ProStringToWstring(optionValue, (char *)cADPartsLibDir.c_str() );
			//isis::isis_ProConfigoptSet(optionName, optionValue  );


			////////////////////////////////////////////////
			// Write config.pro 
			///////////////////////////////////////////////
			//ofstream config_Pro;
			//config_Pro.open (configPro_PathAndFileName );
			//config_Pro << "override_store_back yes\n";
			//config_Pro << "search_path_file " <<  ".\\search_META.pro";
			//config_Pro.close();

			//wcscpy( optionName, L"override_store_back");
			//wcscpy( optionValue, L"yes");			
			//isis::isis_ProConfigoptSet(optionName, optionValue  );

			// force the loading of the search file 
			//wcscpy( optionName, L"search_path");	
			//wcscpy( optionValue, L".\\search_META.pro");			
			//isis::isis_ProConfigoptSet(optionName, optionValue  );


			//ofstream search_Meta_Pro;
			//search_Meta_Pro.open (searchMetaPro_PathAndFileName );
			//search_Meta_Pro << cADPartsLibDir;
			//search_Meta_Pro.close();
		//}

		ProPath  WorkingDirPath_wchar;

		ProStringToWstring(WorkingDirPath_wchar, (char *)workingDir.c_str() );

		isis::isis_ProDirectoryChange( WorkingDirPath_wchar );

		// Copy template model to the working directory
		isis::CopyFile( templateFile_PathAndFileName,  workingDir );
		Template_Copied = true;

		// *** If model renames occured
		// WARNING - isis::CopyModels must be called after 
		//			 1) Creo has started
		//			 2) isis_ProDirectoryChange changed to the working directory
		//           been created.
		//			 3) the search_META.pro has been set 
		if ( ToPartName_FromPartName_map.size() > 0 ) isis::CopyModels(ToPartName_FromPartName_map);


		////////////////////////////////////////////////////////////////////////////
		// Setup for writing STEP file informatoin to \\manufacturing.manifest.json
		////////////////////////////////////////////////////////////////////////////
		bool manufacturingManifestJsonExists = isis::FileExists( isis::manufacturingManifestJson_PathAndFileName.c_str());
		bool seperateStepPartFilesRequested = false;
		isis::e_DataExchangeVersion  stepFileVersion_ForManufacturing;

		if ( isis::DataExchange_Format_Version_InList(	isis::DATA_EXCHANGE_FORMAT_STEP,  
														isis::AP203_E2_SEPERATE_PART_FILES, 
														CADComponentAssemblies.DataExchangeSpecifications) )
		{
			stepFileVersion_ForManufacturing = isis::AP203_E2_SEPERATE_PART_FILES;
			seperateStepPartFilesRequested = true;
		}
		else
		{
			if ( isis::DataExchange_Format_Version_InList(	isis::DATA_EXCHANGE_FORMAT_STEP,  
															isis::AP214_SEPERATE_PART_FILES, 
															CADComponentAssemblies.DataExchangeSpecifications) )
			{
				stepFileVersion_ForManufacturing = isis::AP214_SEPERATE_PART_FILES;
				seperateStepPartFilesRequested = true;
			}
		}

		////////////////////////////////////////
		// Build the assemblies
		////////////////////////////////////////
		for ( std::list<isis::TopLevelAssemblyData>::const_iterator i(CADComponentAssemblies.topLevelAssemblies.begin());
			  i != CADComponentAssemblies.topLevelAssemblies.end();
			  ++i )
		{
			bool regenerationSucceeded;
			isis::BuildAssembly( i->assemblyComponentID, workingDir, CADComponentData_map, regenerationSucceeded );

			if ( !regenerationSucceeded ) regenerationSucceeded_ForAllAssemblies = false;

			//////////////////////////////////////
			// Display assembly created messages
			////////////////////////////////////
			time_t time_end;		/* calendar time */
			time_end=time(NULL);	/* get current cal time */

			time_t time_elapsed = time_end - time_start;

			std::cout << std::endl << std::endl << "Assembly creation completed successfully.";
			std::clog << std::endl << std::endl << "Assembly creation completed successfully.";

			// Get component count
			isis::ComponentVistorCountAssemblyComponents  componentVistorCountAssemblyComponents;
			isis::VisitComponents(i->assemblyComponentID, CADComponentData_map, componentVistorCountAssemblyComponents);

			// Log component count
			std::clog << std::endl << "   Number of assembled components: " << componentVistorCountAssemblyComponents.numberOfComponents;		
			std::clog << std::endl << "   Elapsed wall-clock time:        " << time_elapsed << " seconds";

			std::cout << std::endl << "   Number of assembled components: " << componentVistorCountAssemblyComponents.numberOfComponents;	
			std::cout << std::endl << "   Elapsed wall-clock time:        " << time_elapsed << " seconds";

			time_start=time(NULL); // reset start time for subsequent assemblies if any

			///////////////////////
			//	Create STEP Files
			///////////////////////	
			// For testing, un-comment out this code to Export all STEP files
			//if ( CADComponentAssemblies.DataExchangeSpecifications.size() == 0 )
			//{
			//	CADComponentAssemblies.DataExchangeSpecifications.push_back(
			//		isis::DataExchangeSpecification(isis::DATA_EXCHANGE_FORMAT_STEP, isis::AP203_SINGLE_FILE ));
			//	CADComponentAssemblies.DataExchangeSpecifications.push_back(
			//		isis::DataExchangeSpecification(isis::DATA_EXCHANGE_FORMAT_STEP, isis::AP203_E2_SINGLE_FILE ));
			//	CADComponentAssemblies.DataExchangeSpecifications.push_back(
			//		isis::DataExchangeSpecification(isis::DATA_EXCHANGE_FORMAT_STEP, isis::AP203_E2_SEPERATE_PART_FILES ));
			//	CADComponentAssemblies.DataExchangeSpecifications.push_back(
			//		isis::DataExchangeSpecification(isis::DATA_EXCHANGE_FORMAT_STEP, isis::AP214_SINGLE_FILE ));
			//	CADComponentAssemblies.DataExchangeSpecifications.push_back(
			//		isis::DataExchangeSpecification(isis::DATA_EXCHANGE_FORMAT_STEP, isis::AP214_SEPERATE_PART_FILES ));
			//}	

			bool manufacturingManifestUpdated = false; 

			if ( CADComponentAssemblies.DataExchangeSpecifications.size() > 0 )
			{
				if ( regenerationSucceeded )
				{
					std::cout << std::endl;
					std::cout << std::endl << "Exporting STEP files, this could take several minutes...";
					
					std::clog << std::endl << "Exporting STEP files";
					std::clog << std::endl << "STEP file formats: ";

					for each( isis::DataExchangeSpecification de in CADComponentAssemblies.DataExchangeSpecifications)
					{
					//	std::cout << std::endl << "   " << isis::DataExchangeVersion_string(de.dataExchangeVersion);
						std::clog << std::endl << "   " << isis::DataExchangeVersion_string(de.dataExchangeVersion);
					}
		
					isis::ExportDataExchangeFiles(	CADComponentData_map[i->assemblyComponentID].name,
													CADComponentData_map[i->assemblyComponentID].modelType,
													workingDir,
													CADComponentAssemblies.DataExchangeSpecifications,
													CADComponentData_map,
													true);

					/////////////////////////////////
					// manufacturing.manifest.json
					/////////////////////////////////
					if ( manufacturingManifestJsonExists && seperateStepPartFilesRequested )
					{
						std::clog << std::endl << std::endl << "Updating Manufacturing Manifest (manufacturing.manifest.json) for assembly: " << i->assemblyComponentID;
						std::cout << std::endl << std::endl << "Updating Manufacturing Manifest (manufacturing.manifest.json) for assembly: " << i->assemblyComponentID;
						isis::UpdateManufacturingManifestWithSTEPFileInfo(
														stepFileVersion_ForManufacturing, 
														i->assemblyComponentID, 
														true, // only update manifest for parts.  We are saving an assembly and the manifest contains the list of detail parts
														true, // change case of step file to lower case.  By default, when an assembly is saved, the detail part STEP files are all lower case.
														CADComponentData_map );
						manufacturingManifestUpdated = true;
					}
				}
				else
				{
					std::cout << std::endl << std::endl << "STEP Files - Did NOT export STEP file(s) because both attempts to regenerate the assembly failed.";	
					std::clog << std::endl << std::endl << "STEP Files - Did NOT export STEP file(s) because both attempts to regenerate the assembly failed.";		
				}
			}
			else
			{
				std::clog << std::endl << std::endl << "STEP Files - Export of STEP files was not requested.";		
			}

			/////////////////////////////////////////////
			// Log Status of Updating Manufacturing Json
			/////////////////////////////////////////////
			if ( !manufacturingManifestUpdated )
			{
				std::clog << std::endl << std::endl << "Manufacturing manifest (i.e. manufacturing.manifest.json) NOT updated for assembly: " << CADComponentData_map[i->assemblyComponentID].name;
				std::clog << std::endl << "   For the manufacturing manifest to be updated, all three of the following must be True"; 
				if (seperateStepPartFilesRequested )
					std::clog << std::endl << "      Separate STEP Part Files Requested = True";
				else
					std::clog << std::endl << "      Separate STEP Part Files Requested = False";

				if (regenerationSucceeded )
					std::clog << std::endl << "      Assembly Regeneration Succeeded    = True";
				else
					std::clog << std::endl << "      Assembly Regeneration Succeeded    = False";

				if (manufacturingManifestJsonExists )
					std::clog << std::endl << "      Manufacturing Manifest Exists      = True";
				else
				{
					std::clog << std::endl << "      Manufacturing Manifest Exists      = False";
					std::clog << std::endl <<  isis::CouldNotFindManufacturingManifestError;
				}
			}


			// isis::ExportRasterImage(	i->assemblyComponentID, CADComponentData_map  );

			///////////////////////
			//	Analysis Run
			///////////////////////
			if ( regenerationSucceeded && analysisRun) 
			{
				isis::Create_FEADecks_BatFiles( *i,   // TopLevelAssemblyData	
												CADComponentAssemblies.materials, 
												workingDir,
												programName_Version_TimeStamp,
												CADComponentData_map );
				// WARNING - Do not save the assembly/models after this point.  Doing so will save the temporarily created material.
			}
		}
															
		///////////////////////////////////
		//	Output Requested Computations
		//////////////////////////////////
		
		// Must delete ComputedValues.xml if it exists; otherwise, consumers of this file might grap
		// an outdated file.
		std::string computedValues_PathAndFileName = workingDir + "\\ComputedValues.xml";
		isis::IfFileExists_DeleteFile( computedValues_PathAndFileName);

		if ( regenerationSucceeded_ForAllAssemblies )
		{
			CreateXMLFile_ComputedValues( computedValues_PathAndFileName,
												 CADComponentAssemblies,
												 CADComponentData_map );
		}
		else
		{
			std::cout << std::endl << std::endl << "ComputedValues File - Did NOT create a ComputedValues file because both attempts to regenerate the assembly failed.";	
			std::clog << std::endl << std::endl << "ComputedValues File - Did NOT create a ComputedValues file because both attempts to regenerate the assembly failed.";	
		}
		
		////////////////////////////////////////////
		// Output Step File for Unassembled Models
		////////////////////////////////////////////
		bool manufacturingManifestUpdated = false;
		if ( CADComponentAssemblies.unassembledComponents.size() > 0 )
		{
			std::cout << std::endl << std::endl << "Exporting STEP files for unassembled models, this could take several minutes...";
			std::clog << std::endl << std::endl << "Exporting STEP files for unassembled models, this could take several minutes...";

			bool logCouldNotFindManifestError = true;

			for ( std::list<isis::UnassembledComponent>::const_iterator unusedComp( CADComponentAssemblies.unassembledComponents.begin());
				  unusedComp != CADComponentAssemblies.unassembledComponents.end(); ++unusedComp )
			{

				//std::cout << std::endl << "Exporting STEP file for " + unusedComp->name + ", this could take several seconds...";
				isis::ExportDataExchangeFiles(	unusedComp->name,
												unusedComp->modelType,
												workingDir,
												CADComponentAssemblies.DataExchangeSpecifications,
												CADComponentData_map, true );
				////////////////////////////////////////////////////////
				// Write STEP file names to manufacturing.manifest.json
				////////////////////////////////////////////////////////
				if ( manufacturingManifestJsonExists && seperateStepPartFilesRequested )
				{
					isis::UpdateManufacturingManifestWithSTEPFileInfo(
												stepFileVersion_ForManufacturing, 
												unusedComp->componentID,
												false, // Need to update the manifest for parts and assmeblies
												false, // The case should be the same as entered in CyPhy and the input xml
												CADComponentData_map );
					std::clog << std::endl << std::endl << "Updating Manufacturing Manifest (manufacturing.manifest.json) for unassembled model: " << unusedComp->name;
					manufacturingManifestUpdated = true; 
				}
			}
		}	

		/////////////////////////////////////////////
		// Log Status of Updating Manufacturing Json
		/////////////////////////////////////////////
		if ( !manufacturingManifestUpdated  && CADComponentAssemblies.unassembledComponents.size() > 0 )
		{
			std::clog << std::endl << std::endl << "Manufacturing manifest (i.e. manufacturing.manifest.json) NOT updated for unassembled parts/sub-assemblies";
			std::clog << std::endl << "   For the manufacturing manifest to be updated, both of the following must be True"; 
			if (seperateStepPartFilesRequested )
				std::clog << std::endl << "      Separate STEP Part Files Requested = True";
			else
				std::clog << std::endl << "      Separate STEP Part Files Requested = False";

			if (manufacturingManifestJsonExists )
				std::clog << std::endl << "      Manufacturing Manifest Exists      = True";
			else
			{
				std::clog << std::endl << "      Manufacturing Manifest Exists      = False";
				std::clog << std::endl <<  isis::CouldNotFindManufacturingManifestError;
			}
		}			

		////////////////////////
		// Compute Metrics
		////////////////////////
		
		std::string MeticsOutputXML_PathAndFileName = xmlInputFile_PathAndFileName;	 		
		MeticsOutputXML_PathAndFileName.replace(  MeticsOutputXML_PathAndFileName.end() - 4, 
													MeticsOutputXML_PathAndFileName.end(), "_metrics.xml");  
		// Should delete the metrics file between runs.  This would prevent an xml file from an old run being
		// used for the case where a new run failed to create an xml file.
		isis::IfFileExists_DeleteFile( MeticsOutputXML_PathAndFileName);

		if ( regenerationSucceeded_ForAllAssemblies )
		{
			std::cout << std::endl << std::endl << "Creating Metrics File";	
			std::clog << std::endl << std::endl << "Creating Metrics File";	
			
			bool metricsErrorOccurred;
			isis::OutputCADMetricsToXML(CADComponentAssemblies, CADComponentData_map,  MeticsOutputXML_PathAndFileName, metricsErrorOccurred);
			std::cout << std::endl << "   Created: " + MeticsOutputXML_PathAndFileName;

			if ( metricsErrorOccurred )
			{
				std::cout << std::endl << "   Error(s) occurred when creating the metrics file.  To view the errors, search on" << std::endl <<
										  "   " << isis::METRICS_FILE_ERROR_string <<" in "<< logFile_PathAndFileName << ".  The errors are listed at the end of this file." << std::endl <<
										  "   Also, the errors are listed in the \"Anomalies\" section of " << MeticsOutputXML_PathAndFileName << ".";
			}
			std::cout << std::endl;
		}
		else
		{		
			std::cout << std::endl << std::endl << "Metrics File - Did NOT create a metrics file because both attempts to regenerate the assembly failed.";	
			std::clog << std::endl << std::endl << "Metrics File - Did NOT create a metrics file because both attempts to regenerate the assembly failed.";	
		}
		
		if ( !regenerationSucceeded_ForAllAssemblies )
		{
			std::stringstream errorString;
				errorString <<
						"At least one assembly would not regenerate.  See the log file to locate the assembly/part that would not regenerate."
						<< std::endl << "Log File: " <<  logFile_PathAndFileName;
				throw isis::application_exception(errorString.str().c_str());
		}

	} // END Try

	//catch (const xml_schema::exception& e)
	//{
	//	cout << endl << "Schema exception: " << e.what() << endl << endl;
	//	if (LogFile.is_open()) clog << endl << "Schema exception: " << e.what();
	//}
	catch ( isis::application_exception& ex )
	{
		exceptionErrorStringStream  << "ERROR: " << ex.what();
		ExitCode = -1;
	}
	catch ( std::exception& ex )
	{
		exceptionErrorStringStream << "ERROR: " << ex.what();
		ExitCode = -2;
	}
	catch ( ... )
	{
		exceptionErrorStringStream << "ERROR: std::exception: Caught exception (...).  Please report the error to the help desk.";
		ExitCode = -3;
	}

	if ( ExitCode != 0 )
	{
		std::cerr << std::endl << std::endl << exceptionErrorStringStream.str() << std::endl << std::endl;
		if (LogFile.is_open()) std::clog << std::endl << std::endl << exceptionErrorStringStream.str();

		std::string failedTxtFileName = "_FAILED.txt";
		bool addLineFeed = false;
		if ( isis::FileExists( failedTxtFileName.c_str() )) addLineFeed = true;

		ofstream failedTxtFileStream;
		failedTxtFileStream.open (failedTxtFileName, ios::app );
		if ( failedTxtFileStream.is_open() )
		{
			if ( addLineFeed ) failedTxtFileStream << std::endl;
			failedTxtFileStream <<  isis::GetDayMonthTimeYear() << ", CADCreoParametricCreateAssembly.exe error code: " << ExitCode << ".  For additional information, scroll to the bottom of " << logFile_PathAndFileName;
			failedTxtFileStream.close();
		}
	}

	// Deleted the copied template assembly file if it exists.
	// Note - Added "if ( Template_Copied )" because the function call was returning a message to the 
	// console if the file did not exist.  
	if ( Template_Copied )
	{
		isis::IfFileExists_DeleteFile( workingDir + "\\" + isis::TEMPLATE_MODEL_NAME_METRIC + isis::TEMPLATE_MODEL_NAME_METRIC_SUFFIX);
	}
	
	/////// Stop Pro/E /////////
	if (Pro_E_Running) ProEngineerEnd();
	if (LogFile.is_open())LogFile.close();

	if ( promptBeforeExiting )
	{
		printf("\nType Enter to exit.");
		getc(stdin);
	}

	exit(ExitCode);
}