/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#ifndef  COMMON_STRUCTURES_H
#define  COMMON_STRUCTURES_H
#include <isis_ptc_toolkit_functions.h>
#include <isis_application_exception.h>
#include <isis_include_ptc_headers.h>
#include <StringToEnumConversions.h>
#include <iostream>
#include <ostream>
#include <list>
#include <set>
#include <vector>
#include <map>

namespace isis
{
	const int c_InifiteLifeNumberOfCycles = 500000000;
	/*! \struct ConstraintOffset
    \brief Structure that defines a constraint offset.

	A constraint offset is the distance from a datum to the desired placement.
	*/
	struct ConstraintOffset
	{
		ProAsmcompConstrType	offsetAlignmentType; /**< PRO_ASM_ALIGN_OFF, MATE_OFFSET */
		double					value;

		bool					unitsPresent;				
		e_CADUnitsDistance		units;				/**<CAD_UNITS_MM, CAD_UNITS_CM, CAD_UNITS_INCH */

		ConstraintOffset(): unitsPresent(false) {}

	};


	/*! \struct ConstraintFeature
    \brief Structure that defines a constraint feature.

	A constraint feature defines the geometry that is used to define the constraint.
	*/
	struct ConstraintFeature
	{
		std::string		componentID;
		std::string		featureName;
		ProDatumside	featureOrientationType;   /**< PRO_DATUM_SIDE_NONE, PRO_DATUM_SIDE_YELLOW, PRO_DATUM_SIDE_RED */
	};


	/*! \struct ConstraintPair
    \brief Structure that defines a constraint pair.

	A constraint pair defines the meta data about constraint.
	*/
	struct ConstraintPair
	{
		ProAsmcompConstrType		featureAlignmentType;  /**< PRO_ASM_ALIGN, PRO_ASM_MATE, PRO_ASM_MATE_OFF, always PRO_ASM_ALIGN for now */
		ProType						featureGeometryType;   /**< PRO_SURFACE, PRO_AXIS	*/
		e_FeatureInterfaceType		featureInterfaceType;  /**<  CAD_DATUM, CAD_MODEL_INTERFACE, CAD_MODEL_USER_DATA */
		
		std::vector<ConstraintFeature>	constraintFeatures;

		bool						constraintOffsetPresent;
		ConstraintOffset			constraintOffset;

		ConstraintPair() : constraintOffsetPresent(false) {}
	};



	/*! \struct ConstraintData
    \brief Structure that contains a constraint pair.

	A constraint pair defines the alignment of a datum on one part/assembly to a datum on another part/assembly.
	*/
	struct ConstraintData
	{
		std::vector<ConstraintPair>	constraintPairs;  /**< Container for constraint pairs. */

		//ConstraintData() : constraintPairs(3) {}
	};


	/*! \struct ConstraintDefinition
    \brief Structure defining constraints.

	Constraints are used to position parts/assemblies.  Constraints are defined via sets of 
	part/assembly datums, which are used to position (mate/align) parts/assemblies.
	*/
	struct ConstraintDefinition
	{
		bool applyAllConstraints;		/**<  Currently not used. Will be used in the future. */
		int  applySingleConstaintID;	/**<  Currently not used. Future use: 0 - first constraint, 1 - second constraint, 2 - third constraint */

		std::vector<ConstraintData> constraints;    /**< List of constraints */
		
		ConstraintDefinition() : constraints(3),  applyAllConstraints(true), applySingleConstaintID(0) {}

	};

	/*! \struct CADParameter
    \brief Structure defining parameters.

	Parameters are used to change features about a part/assembly.  For example, a parameter named LENGTH might be used to vary the length  of a part. 
	A boolean parameter might be used to keep or remove a part/assembly feature.
	*/
	struct CADParameter
	{
		std::string			name;   /**<  Parameter name. */
		e_CADParameterType	type;   /**<  CAD_FLOAT, CAD_INTEGER, CAD_BOOLEAN */ 
		std::string			value;  /**<   Parameter value (e.g 1.4, 34...) */

	};

	struct AnalysisTempMaterial
	{
		std::string			name;
		double				poissonsRatio;
		std::string				tokenPoissonsRatio;
		AnalysisTempMaterial() : name(""), poissonsRatio(0), tokenPoissonsRatio("") {}
	};


	struct ScalarsDistanceXyx
	{
		e_CADUnitsDistance units;
		double	x;
		double	y;
		double	z;

		ScalarsDistanceXyx(): units(CAD_UNITS_MM), x(0),y(0), z(0) {};
	};

	struct ScalarsDirectionXyx
	{
		double	x;
		double	y;
		double	z;

		ScalarsDirectionXyx():  x(0),y(0), z(0) {};
	};


	struct ScalarsRotationXyx
	{
		e_CADUnitsAngle units;
		e_AnalysisConstraintFixedFree	x;
		e_AnalysisConstraintFixedFree	y;
		e_AnalysisConstraintFixedFree	z;

		ScalarsRotationXyx(): units(CAD_UNITS_RADIAN), x(ANALYSIS_CONSTRAINT_FIXED),y(ANALYSIS_CONSTRAINT_FIXED), z(ANALYSIS_CONSTRAINT_FIXED) {};
	};


	struct ScalarsForceXyx
	{
		e_CADUnitsForce units;
		double	x;
		double	y;
		double	z;

		ScalarsForceXyx(): units(CAD_UNITS_N), x(0),y(0), z(0) {};
	};

	struct ScalarsMomentXyx
	{
		e_CADUnitsMoment units;
		double	x;
		double	y;
		double	z;

		ScalarsMomentXyx(): units(CAD_UNITS_N_MM), x(0),y(0), z(0) {};
	};

	struct ScalarPressureValue
	{
		e_CADUnitsPressure	units;
		double		value;

		ScalarPressureValue(): units(CAD_UNITS_PA), value(0) {};
	};

	struct ScalarAccelerationAndXyzDirection
	{
		e_CADUnitsAcceleration	units;
		//double		magnitude;
		ScalarsDirectionXyx	direction;

		ScalarAccelerationAndXyzDirection(): units(CAD_UNITS_MM_PER_SEC_SQ) {};
	};

	/*
	struct ScalarAngleValue
	{
		e_CADUnitsAngle	units;
		double		value;

		ScalarAngleValue(): units(CAD_UNITS_RADIAN), value(0) {};
	};
   */

/*
	struct BoolsXyx
	{
		bool	x;
		bool	y;
		bool	z;

		BoolsXyx(): x(false),y(false), z(false) {};
		BoolsXyx( bool in_x, bool in_y, bool in_z): x(in_x),y(in_y), z(in_z) {};
	};
*/

	struct AnalysisDisplacement
	{
		//BoolsXyx	translation_Fixed;
		ScalarsDistanceXyx	translation;	
		
		//BoolsXyx	rotation_Fixed;
		ScalarsRotationXyx	rotation;

		AnalysisDisplacement()//: translation_Fixed(true, true,true),
								/*rotation_Fixed(true, true, true) */  {};
	};

	struct AnalysisPin
	{
		e_AnalysisConstraintFixedFree axialRotationConstraint;  // true if fixed, false if free
		e_AnalysisConstraintFixedFree axialDisplacementConstraint;

		AnalysisPin(): axialRotationConstraint(ANALYSIS_CONSTRAINT_FIXED), axialDisplacementConstraint(ANALYSIS_CONSTRAINT_FIXED) {};
	};

	struct CADFeature
	{
		std::string					componentID;
		std::string					datumName;    
	};


	struct AnalysisGeometryFeature
	{	
		std::string					featureID;
		e_CADGeometryType			geometryType;			//  CAD_GEOMETRY_POLYGON, CAD_GEOMETRY_CIRCLE, CAD_GEOMETRY_CONCENTRIC_CIRCLES,
															//	CAD_GEOMETRY_CYLINDER_SURFACE, CAD_GEOMETRY_CYLINDER_SURFACE, CAD_GEOMETRY_SPHERE_SURFACE
															//  CAD_GEOMETRY_SPHERE_SURFACE, CAD_GEOMETRY_EXTRUSION, CAD_GEOMETRY_EXTRUSION_SURFACE
		std::list<CADFeature>		features;            
		e_FeatureInterfaceType		featureInterfaceType;	// CAD_DATUM 
		e_CADFeatureGeometryType	featureGeometryType;	// POINT, maybe other type later. for now points define polygons, lines, and spheres

		bool							primaryGeometryQualifierDefined;
		bool							secondaryGeometryQualifierDefined;
		
		e_CADPrimaryGeometryQualifier   primaryGeometryQualifier;
		e_CADSecondaryGeometryQualifier secondaryGeometryQualifier;
		
		//bool							geometrySetOperationDefined;
		//e_GeometrySetOperation			geometrySetOperation;

		AnalysisGeometryFeature(): primaryGeometryQualifierDefined(false), secondaryGeometryQualifierDefined(false) {};
	};


	struct AnalysisGeometrySetOperation
	{
		e_GeometrySetOperation		setOperation;
		std::vector<std::string>	featureID;	
		//std::string				feature_1_ID;
		//std::string				feature_2_ID;
	
		//e_CADPrimaryGeometryQualifier	 feature_1_PrimaryGeometryQualifier;
		//e_CADPrimaryGeometryQualifier	 feature_2_PrimaryGeometryQualifier;
		
		//e_CADSecondaryGeometryQualifier  feature_1_SecondaryGeometryQualifier;
		//e_CADSecondaryGeometryQualifier  feature_2_SecondaryGeometryQualifier;
	};

	struct AnalysisGeometry
	{	
		std::list<AnalysisGeometryFeature>		features;				// actual datum names in the Creo model

		//e_CADPrimaryGeometryQualifier   primaryGeometryQualifier;
		//e_CADSecondaryGeometryQualifier secondaryGeometryQualifier;

		bool setOperationDefined;
		std::list<AnalysisGeometrySetOperation> setOperations;

		AnalysisGeometry(): setOperationDefined(false){};
	};

//	struct AnalysisGeometry
//	{	
//		e_CADGeometryType			geometryType;			// CAD_GEOMETRY_POLYGON, CAD_GEOMETRY_CYLINDER, CAD_GEOMETRY_SPHERE
//		std::string					componentID;
//		e_FeatureInterfaceType		featureInterfaceType;	// CAD_DATUM 
//		e_CADFeatureGeometryType	featureGeometryType;	// POINT, maybe other type later. for now points define polygons, lines, and spheres
//		std::list<std::string>		features;				// actual datum names in the Creo model
//	};

	struct AnalysisConstraint
	{	
		AnalysisGeometry		geometry;

		bool					analysisDisplacementDefined;
		AnalysisDisplacement    displacement;

		bool					analysisPinDefined;
		AnalysisPin				pin;

		bool					analysisBallDefined;
		// for a ball constraint, rotation is always free (never fixed)
		AnalysisConstraint() : analysisDisplacementDefined(false), analysisPinDefined(false), analysisBallDefined(false) {};
	};


	struct Force
	{
		ScalarsForceXyx	force;
		ScalarsMomentXyx	moment;
	};

	struct AnalysisLoad
	{	
		//e_AnalysisLoadType					type; // ANALYSIS_LOAD_FORCE, ANALYSIS_LOAD_PRESSURE, ANALYSIS_LOAD_BEARING...
		AnalysisGeometry					geometry;

		bool								forceDefined;
		Force								force;	

		bool								pressureDefined;
		ScalarPressureValue					pressure;

		bool								accelerationDefined;
		ScalarAccelerationAndXyzDirection	acceleration;

		AnalysisLoad() : forceDefined(false), pressureDefined(false), accelerationDefined(false) {};

	};


	struct AnalysisPartInterface
	{
		e_AnalysisPartInterfaceType	type;
		std::string					componentID_1;
		std::string					componentID_2;
		AnalysisPartInterface() : type(ANALYSIS_PART_INTERFACE_BONDED) {}; // by default assume BONDED
	};


	struct AnalysisSolver
	{
		pro_fem_solver_type			type;
		pro_fem_mesh_type			meshType;
		pro_fem_shell_mesh_type		shellElementType;
		pro_fem_elem_shape_type		elementShapeType;
	};

	struct CADAnalysisComponentMetrics
	{
		std::string type;
		std::string ID;

		CADAnalysisComponentMetrics( const std::string &in_Type, const std::string &in_ID): type(in_Type), ID(in_ID) {}; 
	};

	struct CADAnalysisComponent
	{
		std::string		componentID;
		bool			infiniteCycle;

		std::list<CADAnalysisComponentMetrics> 	cadAnalysisComponentMetrics;

	};

	struct AnalysisFEA
	{
		std::string		ID;
		e_AnalysisType	type; // ANALYSIS_STRUCTURAL, ANALYSIS_BUCKLING, ANALYSIS_MODAL, ANALYSIS_THERMAL, ANALYSIS_FATIGUE

		std::list<AnalysisConstraint>		analysisConstraints;
		std::list<AnalysisLoad>				analysisLoads;
		std::list<AnalysisPartInterface>	analysisPartInterfaces;
		std::list<AnalysisSolver>			analysisSolvers;
		std::list<CADAnalysisComponent>		cADAnalysisComponents;  // these are the components for which anlaysis results will be returned.
	};


	struct CADAnalyses
	{
		list<AnalysisFEA> analysisFEA;
	};

	//struct BoundingBoxMetric_SingleComponent
	//{
	//	std::string boundingBoxMetricID;
	//	e_ComputationDimension boundingBoxDimension;

	//	BoundingBoxMetric_SingleComponent( const string				&in_BoundingBoxMetricID, 
	//									   e_ComputationDimension	in_BoundingBoxDimension ) :
	//									   boundingBoxMetricID(in_BoundingBoxMetricID),
	//										   boundingBoxDimension(in_BoundingBoxDimension) {};
	//};

	struct CADComputation
	{
		e_ComputationType		computationType;  // COMPUTATION_BOUNDING_BOX, COMPUTATION_CG, COMPUTATION_POINT, COMPUTATION_MASS
		e_ComputationDimension	computationDimension; // COMPUTATION_X_COORDINATE, COMPUTATION_Y_COORDINATE, COMPUTATION_Z_COORDINATE, COMPUTATION_VECTOR, COMPUTATION_SCALAR 
		std::string				componentID;
		std::string				metricID;
		std::string				datumName;
		CADComputation(const std::string &in_ComponentID ):componentID(in_ComponentID){};

		//std::string		componentID;
		//bool			vectorMetricDefined;  // if this is false, then individual metrics must be defined.
		//bool			individualMetricDefined;

		//std::string		vector_MetricID;
		//std::string		x_MetricID; 
		//std::string		y_MetricID; 
		//std::string		z_MetricID;  

		//CADComputation_BoundingBox(const std::string &in_ComponentID ):componentID(in_ComponentID),vectorMetricDefined(false), individualMetricDefined(false) {};	
	};

	struct CADComputations
	{
		std::list<CADComputation> computations;
	};
	/*! \struct CADComponentData
    \brief Structure containing the bulk of the data about a part/assembly.

    This structure contains meta data about a component, the list of children, the set of dependences, and 
	the address/handle of the Creo part/assembly.  
	*/
	struct UnassembledComponent
	{
		std::string					name;					/**<  Part/Assembly name without the suffix. */
		ProMdlType					modelType;		
		std::string					componentID;
		/**<  PRO_MDL_PART, PRO_MDL_ASSEMBLY */
		UnassembledComponent( const std::string &in_Name, ProMdlType in_ModelType, const std::string &in_ComponentID) : 
										 name ( in_Name),
										 modelType(in_ModelType),
									     componentID(in_ComponentID) {};
	};


	struct CADComponentData
	{
		std::string					componentID;
		int							metricID;
		std::string					parentComponentID;
		std::string					name;					/**<  Part/Assembly name without the suffix. */
		std::string					materialID;
		ProMdlType					modelType;				/**<  PRO_ASSEMBLY, PRO_PART */
		//std::string					type_string; 
		e_CADSpecialInstruction		specialInstruction;		/**< CAD_SPECIAL_INSTRUCTION_NONE, CAD_SPECIAL_INSTRUCTION_SIZE_TO_FIT */
		// either p_model or modelHandle is not necessary, should review thier uses and eliminate one.
		ProMdl						*p_model;				/**<  Pointer to the model. */
		ProSolid					modelHandle;			/**< typedef struct sld_part* ProSolid;  */
		ProAsmcomp					assembledFeature;		/**< typedef struct pro_model_item  ProAsmcomp; */
		list<int>					componentPaths;			/**< This is the path from the current assembly (i.e. active assembly) to the part/assembly defined by this structure. */
		std::list<std::string>		children;				/**<  Component IDs of the children. */
		std::set<std::string>		dependsOn;				/**< Depends on (i.e. constrained to) ComponentIDs. */

		ConstraintDefinition		constraintDef;			/**< Definition of the constraints that position the part/assembly. */

		bool						parametricParametersPresent; /**< True if parametric parameters apply to this part/assembly. */
		std::list<CADParameter>		parametricParameters;		/**< List of parametric parameters. */

		bool						analysisTempMaterialDefined;
		AnalysisTempMaterial		analysisTempMaterialDefinition;

		//CADComputations_SingleComponent		cADComputations;

		CADComponentData() : p_model(0), modelHandle(0), parametricParametersPresent(false), analysisTempMaterialDefined(false)
		{
			assembledFeature.id = 0;
			assembledFeature.type = PRO_TYPE_INVALID;
			assembledFeature.owner = 0;
		}

	};


	/*! \struct CADComponentData
    \brief Structure containing information about a top-level assembly.

	The structure contains the top-level assembly component ID and the configuration ID.  
	Both of these IDs originate in GME.  The component ID is the ID that uniquely identifies a 
	GME component. The configuration ID is used to link the metrics information created by this 
	program back to a configuration in GME.
	*/
	struct TopLevelAssemblyData
	{
		std::string					assemblyComponentID;
		std::string					configurationID;
		CADAnalyses					analysesCAD;
		CADComputations				cADComputations;

		TopLevelAssemblyData() 
		{
			assemblyComponentID = "";
			configurationID = "";
		}

		TopLevelAssemblyData( const std::string &in_AssemblyComponentID, const std::string &in_ConfigurationID ) 
		{
			assemblyComponentID = in_AssemblyComponentID;
			configurationID = in_ConfigurationID;
		}
	
	};


	// This structure contains the allowables.  The values are based on
	// the values in AnalysisMaterialProperties; howerver, they have been 
	// adjusted based on the number of cycles.  Units will always be in MPa.
	struct AnalysisMaterialProperties_Allowables
	{	
		e_MaterialType materialType;
		std::string		materialName;

		double		tensileStrength;	// MPa
		double		bearingStrength;	// MPa
		double		shearStrength;		// MPa

		long 		numberOfCycles;		// Integer
	};

	// Standardized on SI(mm)(Creo millimeter Newton Second) units as follows:
	//		Force	N
	//		Stress	MPa
	//		Length	mm
	//		Time	s
	struct AnalysisMaterialProperties
	{	
		e_MaterialType materialType;
		std::string		materialName;

		bool		modulusOfElasticityDefined;	
		bool		poissonsRatioDefined;			

		bool		tensileYieldStrengthDefined;
		bool		tensileUltimateStrengthDefined;
		bool		shearStrengthDefined;			
		bool		bearingYieldStrengthDefined;	
		bool		bearingUltimateStrengthDefined;	

		bool		fatigueStrengthDefined;	
		bool		fatigueNumberOfCyclesDefined;

		double		modulusOfElasticity;		// MPa
		double		poissonsRatio;				// Unitless

		double		tensileYieldStrength;		// MPa
		double		tensileUltimateStrength;	// MPa
		double		shearStrength;				// MPa
		double		bearingYieldStrength;		// MPa
		double		bearingUltimateStrength;	// MPa

		double		fatigueStrength;			// MPa
		long 		fatigueNumberOfCycles;		// Integer


		AnalysisMaterialProperties():	modulusOfElasticityDefined(false),
										poissonsRatioDefined(false),
										tensileYieldStrengthDefined(false),
										tensileUltimateStrengthDefined(false),
										shearStrengthDefined(false),  
										bearingYieldStrengthDefined(false), 
										bearingUltimateStrengthDefined(false), 
										fatigueStrengthDefined(false),
										fatigueNumberOfCyclesDefined(false){};
	};


	struct MaterialMetrics
	{
		double value;
		std::string units;

		MaterialMetrics( double in_Value,  const std::string &in_Units ): value(in_Value), units(in_Units) {};
	};

	struct Material
	{
		e_MaterialType	materialType;
		std::string		materialName;
		std::map<std::string, MaterialMetrics>		materialMetrics_map;  // Key property name (e.g. TensileYieldStrength)
		AnalysisMaterialProperties	analysisMaterialProperties;
		//AnalysisMaterialProperties_Allowables analysisMaterialProperties_Allowables;

		Material(){};
		Material( const std::string &in_MaterialName, e_MaterialType in_MaterialType) :materialName(in_MaterialName), materialType(in_MaterialType) {};
	};


	struct DataExchangeSpecification
	{
		e_DataExchangeFormat  dataExchangeFormat;  // Only currently support DATA_EXCHANGE_FORMAT_STEP
		e_DataExchangeVersion dataExchangeVersion; // AP203_SINGLE_FILE, AP203_E2_SINGLE_FILE, AP203_E2_SEPERATE_PART_FILES...

		DataExchangeSpecification(	e_DataExchangeFormat in_DataExchangeFormat, 
									e_DataExchangeVersion in_DataExchangeVersion ) : 
										dataExchangeFormat(in_DataExchangeFormat), 
										dataExchangeVersion( in_DataExchangeVersion ) {};
	};


	/*! \struct CADAssemblies
    \brief Structure containing the top-level assemblies.

    This structure is the top-level container for assemblies.  If the input XML contained multiple assemblies, 
	then cadComponentIDs list would contain the top level component IDs for each assembly.
	*/
	struct CADAssemblies
	{
		std::map<std::string, Material>		 materials;  // Key Material ID
		std::list<TopLevelAssemblyData>		 topLevelAssemblies; 
		std::list<DataExchangeSpecification> DataExchangeSpecifications;
		std::list<UnassembledComponent>		 unassembledComponents;
	};

	std::ostream& operator<<(std::ostream& output, const CADComponentData &in_AssembledInfo); 

	std::string AmalgamateModelNameWithSuffix ( const std::string &in_Name, ProMdlType in_pro_model_type );


	struct CADPoint
	{
		double x;
		double y;
		double z;
	};

}  // END namespace isis
#endif // COMMON_STRUCTURES_H
