/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include <Computations.h>
#include <CADPostProcessingParameters.h>
#include <FiniteElementAnalysis.h>
#include <Metrics.h>
#include <sstream>

namespace isis
{

///////////////////////////////////////////////////////////////////////////////////////////////////////////

//void BuildList_ComponentsRequiringBoundingBoxes( 
//			std::map<string, isis::CADComponentData>	&in_CADComponentData_map,
//			std::list<std::string>						&out_componentsRequiringBoundingBoxes )
//															throw (isis::application_exception)
//{
//	for( std::map<std::string, isis::CADComponentData>::iterator i(in_CADComponentData_map.begin());
//	i != in_CADComponentData_map.end();
//	++i )
//	{
//		if ( i->second.cADComputations.boundingBoxMetricDefined ) 
//			out_componentsRequiringBoundingBoxes.push_back(i->first);
//
//	}
//}

/*
///////////////////////////////////////////////////////////////////////////////////////////////////////////
// In general this function does not honor the ConfigurationID as an input field.  This is because
// the current schema (CADPostProcessingParameters.xsd) only supports one ConfigurationID.
// The schema should modified such that there  is a hierarchy  of components.
void CreateXMLFile_ComputedValues( 
						const std::string									&in_PathAndFileName,
						const isis::CADAssemblies							&in_CADAssemblies,
						std::map<std::string, isis::CADComponentData>		&in_CADComponentData_map )
																	throw (isis::application_exception)
{
	// Example XML file
	//
	// <?xml version="1.0" encoding="UTF-8" standalone="no" ?>
	// <Components ConfigurationID="" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="CADPostProcessingParameters.xsd">
	//
	//	<Component ComponentID="100000445" FEAElementID="" _id="id9960">
    //		<Metrics _id="id9961">
    //			<Metric MetricID="6a632aaf-da49-49c1-81e8-168ff0cd7bb1" Type="BoundingBoxLength" Units="mm" Value="7116.749755347874" _id="id9962"/>
    //		</Metrics>
	//	</Component>
	//
	//	<Component ComponentID="100000452" FEAElementID="" _id="id9963">
    //		<Metrics _id="id9964">
    //			<Metric MetricID="6a632aaf-da49-49c1-81e8-168f999" Type="BoundingBoxWidth" Units="mm" Value="649.2874999999999" _id="id9965"/>
    //		</Metrics>
	//	</Component>
	//	</Components>


	Udm::SmartDataNetwork dn_FEAPostProcessingParameters( CADPostProcessingParameters::diagram );
	dn_FEAPostProcessingParameters.CreateNew( in_PathAndFileName, "CADPostProcessingParameters", CADPostProcessingParameters::Components::meta);
	
	try
	{
		CADPostProcessingParameters::Components componentsRoot = 
			      CADPostProcessingParameters::Components::Cast(dn_FEAPostProcessingParameters.GetRootObject());
	
		componentsRoot.ConfigurationID() = "";

		for ( std::list<TopLevelAssemblyData>::const_iterator i(in_CADAssemblies.topLevelAssemblies.begin());
				i != in_CADAssemblies.topLevelAssemblies.end();
				++i )
		{
			for( std::list<CADComputation>::const_iterator j(i->cADComputations.computations.begin());
				 j != i->cADComputations.computations.end(); ++j )
			{
				/////////////////////////////////////////////////////////////////////
				// Add ComponentID and FEAElementID
				// e.g. <Component ComponentID="100000678" FEAElementID="PSOLID_1">
				/////////////////////////////////////////////////////////////////////
				CADPostProcessingParameters::Component  componentRoot = CADPostProcessingParameters::Component::Create( componentsRoot );
				componentRoot.ComponentID() = j->componentID;
				componentRoot.FEAElementID() = "";

				/////////////////////////////////////////////////////////////////////
				// Add Metrics
				//
				/////////////////////////////////////////////////////////////////////

				CADPostProcessingParameters::Metrics metricsRoot = CADPostProcessingParameters::Metrics::Create( componentRoot );

				Pro3dPnt  r_outline_points[2];

				double temp_x;
				double temp_y;
				double temp_z;

				switch ( j->computationType)
				{
				case COMPUTATION_BOUNDING_BOX:
					isis::isis_ProSolidOutlineGet( in_CADComponentData_map[j->componentID].modelHandle, r_outline_points);
					temp_x  =  abs(r_outline_points[1][0] - r_outline_points[0][0]); // Typically Width  x direction	
					temp_y =  abs(r_outline_points[1][1] - r_outline_points[0][1]);  // Typically Height y direction
					temp_z =  abs(r_outline_points[1][2] - r_outline_points[0][2]);  // Typically Length Z direction
					break;			
				case COMPUTATION_CG:
					ProMassProperty  mass_prop;
					isis::isis_ProSolidMassPropertyGet( in_CADComponentData_map[j->componentID].modelHandle, NULL, &mass_prop );
					double   MatrixBuffer[4][4];
					temp_x =  mass_prop.center_of_gravity[0];
					temp_y  = mass_prop.center_of_gravity[1];
					temp_z =  mass_prop.center_of_gravity[2];
					break;
				case COMPUTATION_POINT:
					CADPoint point;
					RetrieveDatumPointCoordinates( i->assemblyComponentID,
												   j->componentID,
												   in_CADComponentData_map,
												   j->datumName,
												   point); 
					temp_x =  point.x;
					temp_y  = point.y;
					temp_z =  point.z;
					break;
				default:
					std::stringstream errorString;
					errorString <<
					"Error: (Function - CreateXMLFile_ComputationsParameters), received ComputationType that is not currently supported." << std::endl <<
					"ComputationType: " << ComputationType_string( j->computationType);
					throw isis::application_exception(errorString.str().c_str());
				}

			
				CADPostProcessingParameters::Metric metricRoot = CADPostProcessingParameters::Metric::Create( metricsRoot );
				metricRoot.MetricID() = j->metricID;
				metricRoot.Type() = ComputationDimension_string(j->computationDimension);
				metricRoot.Units() = "mm";

				if ( j->computationDimension == COMPUTATION_VECTOR )
				{				
					std::vector<double> values;
					values.push_back(temp_x);
					values.push_back(temp_y);
					values.push_back(temp_z);
					metricRoot.ArrayValue() = values;
				}
				else
				{
					double temp_cord;
					switch ( j->computationDimension )
					{
						case COMPUTATION_X_COORDINATE:
							temp_cord = temp_x;
							break;
						case COMPUTATION_Y_COORDINATE:
							temp_cord = temp_y;
							break;
						case COMPUTATION_Z_COORDINATE:
							temp_cord = temp_z;
							break;
					}

					std::vector<double> values;
					values.push_back(temp_cord);
					metricRoot.ArrayValue() = values;
				}
	
			
			}  // END for j loop

		} // END for ( std::list<TopLevelAssemblyData>::const_...
		///////////////////////////
		// Write XML File
		//////////////////////////
		dn_FEAPostProcessingParameters.CloseWithUpdate();
	}
	catch ( ... )
	{
		// If any error occurs, do not save the metrics xml file.  It makes no sense to save a metrics file
		// with erroneous data.
		dn_FEAPostProcessingParameters.CloseNoUpdate();
		throw;
	}

}
*/

///////////////////////////////////////////////////////////////////////////////////////////////////////////
// In general this function does not honor the ConfigurationID as an input field.  This is because
// the current schema (CADPostProcessingParameters.xsd) only supports one ConfigurationID.
// The schema should modified such that there  is a hierarchy  of components.
//
//	If there are no computations requested, this function returns without taking any action (does not
//	create the xml file).
void CreateXMLFile_ComputedValues( 
						const std::string									&in_PathAndFileName,
						const isis::CADAssemblies							&in_CADAssemblies,
						std::map<std::string, isis::CADComponentData>		&in_CADComponentData_map )
																	throw (isis::application_exception)
{
	// Example XML file
	//
	// <?xml version="1.0" encoding="UTF-8" standalone="no" ?>
	// <Components ConfigurationID="" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="CADPostProcessingParameters.xsd">
	//
	//	<Component ComponentID="100000445" FEAElementID="" _id="id9960">
    //		<Metrics _id="id9961">
    //			<Metric MetricID="6a632aaf-da49-49c1-81e8-168ff0cd7bb1" Type="BoundingBoxLength" Units="mm" Value="7116.749755347874" _id="id9962"/>
    //		</Metrics>
	//	</Component>
	//
	//	<Component ComponentID="100000452" FEAElementID="" _id="id9963">
    //		<Metrics _id="id9964">
    //			<Metric MetricID="6a632aaf-da49-49c1-81e8-168f999" Type="BoundingBoxWidth" Units="mm" Value="649.2874999999999" _id="id9965"/>
    //		</Metrics>
	//	</Component>
	//	</Components>

	std::map<std::string, std::list<CADComputation>> componentID_to_ListofComputations_map;
	std::map<std::string, std::string> componentID_to_AssemblyComponentID_map;

	// Need to group all of the requested computations by component_ID.
	for each ( TopLevelAssemblyData i in in_CADAssemblies.topLevelAssemblies )
	{
		for each ( CADComputation j in i.cADComputations.computations )
		{
			componentID_to_AssemblyComponentID_map[j.componentID] = i.assemblyComponentID;
			componentID_to_ListofComputations_map[j.componentID].push_back(j);
		}
	}

	// Check for no computations
	if ( componentID_to_ListofComputations_map.size() == 0 ) 
	{
		std::cout << std::endl << std::endl << "No computations requested.  ComputedValues file NOT created.";	
		std::clog << std::endl << std::endl << "No computations requested.  ComputedValues file NOT created.";	
		return;
	}
	else
	{
		std::cout << std::endl << std::endl << "Creating ComputedValues File";	
		std::clog << std::endl << std::endl << "Creating ComputedValues File";	
	}

	Udm::SmartDataNetwork dn_FEAPostProcessingParameters( CADPostProcessingParameters::diagram );
	dn_FEAPostProcessingParameters.CreateNew( in_PathAndFileName, "CADPostProcessingParameters", CADPostProcessingParameters::Components::meta);
	
	try
	{
		CADPostProcessingParameters::Components componentsRoot = 
			      CADPostProcessingParameters::Components::Cast(dn_FEAPostProcessingParameters.GetRootObject());
	
		componentsRoot.ConfigurationID() = "";

		for each( std::pair<std::string, std::list<CADComputation>> i in componentID_to_ListofComputations_map)
		{	

			////////////////////////
			// Units
			////////////////////////

			std::string		distanceUnit_LongName;
			std::string		distanceUnit_ShortName;
		
			std::string		massUnit_LongName;
			std::string		massUnit_ShortName;

			std::string		forceUnit_LongName;
			std::string		forceUnit_ShortName;

			std::string		temperatureUnit_LongName;
			std::string		temperatureUnit_ShortName;
		
			std::string		timeUnit_LongName;
			std::string		timeUnit_ShortName;
			
			RetrieveUnits( in_CADComponentData_map[i.first].modelHandle, 
						   distanceUnit_ShortName,		distanceUnit_LongName, 
						   massUnit_ShortName,			massUnit_LongName, 
						   forceUnit_ShortName,			forceUnit_LongName, 
						   timeUnit_ShortName,			timeUnit_LongName, 
						   temperatureUnit_ShortName,	temperatureUnit_LongName );

			/////////////////////////////////////////////////////////////////////
			// Add ComponentID and FEAElementID
			// e.g. <Component ComponentID="100000678" FEAElementID="PSOLID_1">
			/////////////////////////////////////////////////////////////////////
			CADPostProcessingParameters::Component  componentRoot = CADPostProcessingParameters::Component::Create( componentsRoot );
			componentRoot.ComponentID() = i.first;
			componentRoot.FEAElementID() = "";
							CADPostProcessingParameters::Metrics metricsRoot = CADPostProcessingParameters::Metrics::Create( componentRoot );

			for each ( CADComputation j in i.second )
			{
				/////////////////////////////////////////////////////////////////////
				// Add Metrics
				//
				/////////////////////////////////////////////////////////////////////

				Pro3dPnt  r_outline_points[2];

				double temp_x;
				double temp_y;
				double temp_z;
				double temp_scalar;

				ProMassProperty  mass_prop;
				double   MatrixBuffer[4][4];
				std::string temp_units = distanceUnit_LongName;

				switch ( j.computationType)
				{
					case COMPUTATION_BOUNDING_BOX:
						isis::isis_ProSolidOutlineGet( in_CADComponentData_map[i.first].modelHandle, r_outline_points);
						temp_x  =  abs(r_outline_points[1][0] - r_outline_points[0][0]); // Typically Width  x direction	
						temp_y =  abs(r_outline_points[1][1] - r_outline_points[0][1]);  // Typically Height y direction
						temp_z =  abs(r_outline_points[1][2] - r_outline_points[0][2]);  // Typically Length Z direction
						break;			
					case COMPUTATION_CG:
						isis::isis_ProSolidMassPropertyGet( in_CADComponentData_map[i.first].modelHandle, NULL, &mass_prop );
						temp_x =  mass_prop.center_of_gravity[0];
						temp_y  = mass_prop.center_of_gravity[1];
						temp_z =  mass_prop.center_of_gravity[2];
						break;

					case COMPUTATION_MASS:
						isis::isis_ProSolidMassPropertyGet( in_CADComponentData_map[i.first].modelHandle, NULL, &mass_prop );
						//isis::isis_ProSolidMassPropertyGet( in_CADComponentData_map[j.componentID].modelHandle, NULL, &mass_prop );
						temp_scalar =  mass_prop.mass;
						temp_units = massUnit_LongName;
						break;

					case COMPUTATION_POINT:
						CADPoint point;
						RetrieveDatumPointCoordinates( componentID_to_AssemblyComponentID_map[i.first],
													   j.componentID,
													   in_CADComponentData_map,
													   j.datumName,
													   point); 
						temp_x =  point.x;
						temp_y  = point.y;
						temp_z =  point.z;
						break;
					default:
						std::stringstream errorString;
						errorString <<
						"Function - CreateXMLFile_ComputationsParameters, received ComputationType that is not currently supported." << std::endl <<
						"ComputationType: " << ComputationType_string( j.computationType);
						throw isis::application_exception(errorString.str().c_str());
				} // End switch ( j.computationType)
			
				CADPostProcessingParameters::Metric metricRoot = CADPostProcessingParameters::Metric::Create( metricsRoot );
				metricRoot.MetricID() = j.metricID;
				metricRoot.Type() = ComputationDimension_string(j.computationDimension);
				metricRoot.Units() = temp_units;

				std::vector<double> values;
				double temp_value;
				
				switch ( j.computationDimension )
				{
					case COMPUTATION_VECTOR:			
						values.push_back(temp_x);
						values.push_back(temp_y);
						values.push_back(temp_z);
						metricRoot.ArrayValue() = values;
						break;
				
					case COMPUTATION_X_COORDINATE:
					case COMPUTATION_Y_COORDINATE:
					case COMPUTATION_Z_COORDINATE:
					case COMPUTATION_DIMENSION_NONE:
						switch ( j.computationDimension )
						{
								case COMPUTATION_X_COORDINATE:
									temp_value = temp_x;
									break;
								case COMPUTATION_Y_COORDINATE:
									temp_value = temp_y;
									break;
								case COMPUTATION_Z_COORDINATE:
									temp_value = temp_z;
									break;
								case COMPUTATION_DIMENSION_NONE:
									temp_value = temp_scalar;
									break;
								default:
									std::stringstream errorString;
									errorString <<
									"Function - CreateXMLFile_ComputationsParameters, received computationDimension that is not currently supported." << std::endl <<
									"ComputationType: " << ComputationDimension_string( j.computationDimension);
									throw isis::application_exception(errorString.str().c_str());
						}
						values.push_back(temp_value);
						metricRoot.ArrayValue() = values;
						break;
		
					default:
						std::stringstream errorString;
						errorString <<
						"Function - CreateXMLFile_ComputationsParameters, received computationDimension that is not currently supported." << std::endl <<
						"ComputationType: " << ComputationDimension_string( j.computationDimension);
						throw isis::application_exception(errorString.str().c_str());
				}
			
			}  // END for j loop

		} // END for ( std::list<TopLevelAssemblyData>::const_...
		///////////////////////////
		// Write XML File
		//////////////////////////
		dn_FEAPostProcessingParameters.CloseWithUpdate();

	}
	catch ( ... )
	{
		// If any error occurs, do not save the metrics xml file.  It makes no sense to save a metrics file
		// with erroneous data.
		dn_FEAPostProcessingParameters.CloseNoUpdate();
		throw;
	}

}



} // end namespace isis