/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include <DataExchange.h>
#include <CommonUtilities.h>
#include <CADCommonConstants.h>

namespace isis
{

	typedef struct raster_settings {
    ProRasterDepth depth;
    ProRasterType type;
    ProDotsPerInch inch;
    double  size[2];
} RasterSettings;


	bool DataExchange_Format_Version_InList( 
			e_DataExchangeFormat  in_DataExchangeFormat,  // Only currently support DATA_EXCHANGE_FORMAT_STEP
			e_DataExchangeVersion in_DataExchangeVersion, // AP203_SINGLE_FILE, AP203_E2_SINGLE_FILE, AP203_E2_SEPERATE_PART_FILES...
			const  std::list<DataExchangeSpecification> &in_DataExchangeSpecifications )
	{
		for each ( isis::DataExchangeSpecification dexSpec in in_DataExchangeSpecifications )
		{
			if (  in_DataExchangeFormat   ==  dexSpec.dataExchangeFormat &&
				  in_DataExchangeVersion  ==  dexSpec.dataExchangeVersion ) return true;
		}
		return false;
	}
	
	/*
	bool SeperateSTEPPartFilesSpecified(const  std::list<DataExchangeSpecification> &in_DataExchangeSpecifications)
	{

		for each ( isis::DataExchangeSpecification dexSpec in in_DataExchangeSpecifications )
		{
			if ( dexSpec.dataExchangeVersion == isis::AP203_E2_SEPERATE_PART_FILES || 
					dexSpec.dataExchangeVersion == isis::AP214_SEPERATE_PART_FILES )
			{
				return true;
			}
		}
		return false;
	}
	
	*/
void ExportRasterImage(	const std::string									&in_AssemblyComponentID, 
						std::map<string, isis::CADComponentData>			&in_CADComponentData_map )
																	throw (isis::application_exception)
{

	ProMdl     p_asm;
	wchar_t  AssemblyName[ISIS_CHAR_BUFFER_LENGTH];
	ProStringToWstring(AssemblyName, (char *)in_CADComponentData_map[in_AssemblyComponentID].name.c_str() );
	isis::isis_ProMdlRetrieve(AssemblyName,(ProMdlType)PRO_ASSEMBLY,&p_asm);


	RasterSettings set;
    set.depth = PRORASTERDEPTH_8;
    set.type = PRORASTERTYPE_TIFF;
    set.inch = PRORASTERDPI_100;
    set.size[0] = 4;
    set.size[1] = 5;

    int            window_id;
	isis::isis_ProWindowCurrentGet(&window_id);

	std::cout << std::endl << "@@@@@@@@@@@@@@@ Writing tiff file";
	wchar_t  OutFileName_wide[ISIS_CHAR_BUFFER_LENGTH];

	string OutfileName = in_CADComponentData_map[in_AssemblyComponentID].name + ".tif";
	
	ProStringToWstring(OutFileName_wide, (char *)OutfileName.c_str() );

	isis::isis_ProRasterFileWrite(window_id, set.depth, set.size[0], set.size[1],
        set.inch, set.type, OutFileName_wide);
}

// Pre-Conditions
//		The Creo model identified by in_AssemblyComponentID must be open and active in the Creo session
// Post-Conditions
//		If no exceptions, exported files (e.g. STEP) specified in in_DataExchangeSpecifications
void ExportDataExchangeFiles( 
					const std::string									&in_ModelName, 
					ProMdlType											in_ModelType,
					const std::string									&in_WORKING_DIR,
					const std::list<DataExchangeSpecification>          &in_DataExchangeSpecifications,
					std::map<string, isis::CADComponentData>			&in_CADComponentData_map,
					bool												in_LogProgress )
																	throw (isis::application_exception)
{
	ProPath  WorkingDirPath_wchar;
	wchar_t  ModelName[ISIS_CHAR_BUFFER_LENGTH];
	ProMdl     p_Model;

	//////////////////////
	// Retrieve Model
	//////////////////////
	//ProStringToWstring(AssemblyName, (char *)in_CADComponentData_map[in_ComponentID].name.c_str() );
	ProStringToWstring(ModelName, (char *)in_ModelName.c_str() );

	//isis::isis_ProMdlRetrieve(AssemblyName,(ProMdlType)PRO_ASSEMBLY,&p_asm);
	//isis::isis_ProMdlRetrieve(AssemblyName,in_CADComponentData_map[in_ComponentID].modelType,&p_asm);
	isis::isis_ProMdlRetrieve(ModelName,in_ModelType,&p_Model);

	//////////////////////////////////
	// Set Macroscopic Config Setting
	/////////////////////////////////
	ProName		optionName;
	ProPath		optionValue;

	wcscpy( optionName, L"intf3d_out_datums_by_default");
	wcscpy( optionValue, L"yes");
	isis_ProConfigoptSet(optionName, optionValue  );

	wcscpy( optionName, L"intf3d_out_parameters");
	wcscpy( optionValue, L"all");
	isis_ProConfigoptSet(optionName, optionValue  );

	for ( std::list<DataExchangeSpecification>::const_iterator i(in_DataExchangeSpecifications.begin());
		  i != in_DataExchangeSpecifications.end();
		  ++i )
	{
		switch(i->dataExchangeVersion)
		{
			case AP203_SINGLE_FILE:
				wcscpy( optionName, L"step_export_format");
				wcscpy( optionValue, L"ap203_is");			
				isis_ProConfigoptSet(optionName, optionValue  );
				break;
			case AP203_E2_SINGLE_FILE:
				wcscpy( optionName, L"step_export_format");
				wcscpy( optionValue, L"ap203_e2");			
				isis_ProConfigoptSet(optionName, optionValue  );

				// Note - there is not a step_export_ap203_e2_asm_def_mode config function.
				//		  Creo honors the 214 setting for 203_e2
				wcscpy( optionName,  L"step_export_ap214_asm_def_mode");
				wcscpy( optionValue, L"single_file");
				isis_ProConfigoptSet(optionName, optionValue  );

				break;
			case AP203_E2_SEPERATE_PART_FILES:
				wcscpy( optionName, L"step_export_format");
				wcscpy( optionValue, L"ap203_e2");			
				isis_ProConfigoptSet(optionName, optionValue  );

				// Note - there is not a step_export_ap203_e2_asm_def_mode config function.
				//		  Creo honors the 214 setting for 203_e2
				wcscpy( optionName,  L"step_export_ap214_asm_def_mode");
				wcscpy( optionValue, L"separate_parts_only");
				isis_ProConfigoptSet(optionName, optionValue  );

				break;
			case AP214_SINGLE_FILE:
				wcscpy( optionName, L"step_export_format");
				wcscpy( optionValue, L"ap214_is");
				isis_ProConfigoptSet(optionName, optionValue  );

				wcscpy( optionName,  L"step_export_ap214_asm_def_mode");
				wcscpy( optionValue, L"single_file");
				isis_ProConfigoptSet(optionName, optionValue  );

				break;
			case AP214_SEPERATE_PART_FILES:
				wcscpy( optionName, L"step_export_format");
				wcscpy( optionValue, L"ap214_is");
				isis_ProConfigoptSet(optionName, optionValue  );

				wcscpy( optionName,  L"step_export_ap214_asm_def_mode");
				wcscpy( optionValue, L"separate_parts_only");
				isis_ProConfigoptSet(optionName, optionValue  );

				break;

			default:
				char temp_char_array[ISIS_CHAR_BUFFER_LENGTH];
				string temp_string = "Function ExportDataExchangeFiles was passed " + 
					std::string(itoa(i->dataExchangeVersion, temp_char_array, 10)) + 
					" which is an erroneous type.  Allowed enum values are AP203_SINGLE_FILE, AP203_E2_SINGLE_FILE, AP203_E2_SEPERATE_PART_FILES, AP214_SINGLE_FILE and AP214_SEPERATE_PART_FILES.";
				throw isis::application_exception(temp_string.c_str());
		}


		//////////////////////////
		// Create the directory
		//////////////////////////
		std::string outDirName = isis::DataExchangeVersion_string(i->dataExchangeVersion);
		std::string createOutDir = "if not exist " +  outDirName +  " mkdir  " + outDirName;
		isis::ExecuteSystemCommand( createOutDir);



		// Set the Creo directory to the created directory
		std::string outPathAndDir = in_WORKING_DIR + "\\" + outDirName;
		ProStringToWstring(WorkingDirPath_wchar, (char *)outPathAndDir.c_str() );
		isis::isis_ProDirectoryChange( WorkingDirPath_wchar );



		// Write Data Exchange file (e.g. STEP File)
		int Arg_2 = 1;

		//wchar_t  AssemblyName[PRO_FAMILY_NAME_SIZE];
		wchar_t  StepFileName_wchar[ISIS_CHAR_BUFFER_LENGTH];

		std::string StepFileName_string;

		switch (in_ModelType)
		{
			case PRO_PART:
				StepFileName_string = in_ModelName + "_prt.stp";
				break;
			case PRO_ASSEMBLY:
				StepFileName_string = in_ModelName + "_asm.stp";
				break;
			default:
				StepFileName_string = in_ModelName + ".stp";
		}

		//std::cout << std::endl << std::endl << "================> in_ModelName: " << in_ModelName;

		ProStringToWstring(StepFileName_wchar, (char *)StepFileName_string.c_str() );


		//isis_ProOutputFileWrite(	&in_CADComponentData_map[in_AssemblyComponentID].p_model,
		//							AssemblyName,
		//							PRO_STEP_FILE,
		//							NULL,
		//							&Arg_2,
		//							NULL,
		//							NULL );

		isis_ProOutputFileWrite(	p_Model,
									StepFileName_wchar,
									PRO_STEP_FILE,
									NULL,
									&Arg_2,
									NULL,
									NULL );
	

		if ( in_LogProgress )
		{
			std::cout << std::endl << "   Created " << outDirName << " STEP file(s) for "  << in_ModelName;
			std::clog << std::endl << "   Created " << outDirName << " STEP file(s) for "  << in_ModelName;
		}

	}

	// Set Creo back to the original working directory.  This would only be necessary
	// if multiple assemblies were begin created.
	ProStringToWstring(WorkingDirPath_wchar, (char *)in_WORKING_DIR.c_str() );
	isis::isis_ProDirectoryChange( WorkingDirPath_wchar );
}


	///////////////////////////////////
	// Output Step Files
	///////////////////////////////////
	//std::string createAnalysisDir = "if not exist STEP mkdir STEP";

	/*
	int Arg_2 = 1;

	std::cout << std::endl << "Creating STEP file";

	ProName		optionName;
	ProPath		optionValue;

	wcscpy( optionName, L"step_export_format");
	wcscpy( optionValue, L"ap203_e2");
	//wcscpy( optionValue, L"ap214_is");
	isis_ProConfigoptSet(optionName, optionValue  );
	
	wcscpy( optionName, L"intf3d_out_datums_by_default");
	wcscpy( optionValue, L"yes");
	isis_ProConfigoptSet(optionName, optionValue  );

	wcscpy( optionName, L"step_export_ap214_asm_def_mode");
	wcscpy( optionValue, L"separate_parts_only");
	isis_ProConfigoptSet(optionName, optionValue  );

	// Also should set step_out_suppl_geom, set_out_material, 

	isis_ProOutputFileWrite(p_asm,
							AssemblyName,
							PRO_STEP_FILE,
							NULL,
							&Arg_2,
							NULL,
							NULL);
    */


}  // End namespace isis