/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include <FiniteElementAnalysis.h>
#include <AssembleUtils.h>
#include <ModelMaterials.h>
#include <Nastran.h>
#include <GraphicsFunctions.h>
#include <CommonStructures.h>
#include <CommonUtilities.h>
#include <MaterialProperties.h>
#include "UdmBase.h"
#include <CADPostProcessingParameters.h>
#include <CADCommonDefinitions.h>
#include <ToolKitPassThroughFunctions.h>  
#include <iostream>
#include <fstream>
#include <sstream>

namespace isis
{
	const std::string NASTRAN_COMMENT_TAG = "$";
	const std::string NASTRAN_META_COMMENT_TAG = "$META  ";

////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	void ValidateAnalysisGeometry(  const std::string &in_ConfigurationIDSentence,
								const std::string &in_ConstraintOrLoad,
								const AnalysisGeometry &in_AnalysisGeometry ) throw (isis::application_exception)
{
	std::string TempError = in_ConfigurationIDSentence;

	if (  (in_AnalysisGeometry.geometryType == CAD_GEOMETRY_POLYGON) && (in_AnalysisGeometry.features.size()) < 3 )
	{
		TempError += "For a polygon " + in_ConstraintOrLoad + ", there must at least three features (i.e. Datum Points)";
		throw isis::application_exception(TempError.c_str());	
	}

	if (  (in_AnalysisGeometry.geometryType == CAD_GEOMETRY_CYLINDER) && (in_AnalysisGeometry.features.size()) != 2 )
	{
		TempError += "For a pin " + in_ConstraintOrLoad + ", there must two features (i.e. Datum Points)";
		throw isis::application_exception(TempError.c_str());	
	}

	if (  (in_AnalysisGeometry.geometryType == CAD_GEOMETRY_SPHERE) && (in_AnalysisGeometry.features.size()) != 2 )
	{
		TempError += "For a ball " + in_ConstraintOrLoad + ", there must two features (i.e. Datum Points)";
		throw isis::application_exception(TempError.c_str());	
	}


}
*/
void ValidateAnalysisGeometry(  const std::string &in_ConfigurationIDSentence,
								const std::string &in_ConstraintOrLoad,
								const AnalysisGeometry &in_AnalysisGeometry ) throw (isis::application_exception)
{
	std::string TempError = in_ConfigurationIDSentence;

	int numberOfFeatures = in_AnalysisGeometry.features.begin()->features.size();

	if (  (in_AnalysisGeometry.features.begin()->geometryType == CAD_GEOMETRY_POLYGON) && ( numberOfFeatures < 3 ))
	{
		TempError += "For a polygon " + in_ConstraintOrLoad + ", there must at least three features (i.e. Datum Points)";
		throw isis::application_exception(TempError.c_str());	
	}

	if (  (in_AnalysisGeometry.features.begin()->geometryType == CAD_GEOMETRY_CYLINDER) && (numberOfFeatures != 3 ))
	{
		TempError += "For a pin " + in_ConstraintOrLoad + ", there must two features (i.e. Datum Points)";
		throw isis::application_exception(TempError.c_str());	
	}

	if (  (in_AnalysisGeometry.features.begin()->geometryType == CAD_GEOMETRY_SPHERE) && (numberOfFeatures != 2 ))
	{
		TempError += "For a ball " + in_ConstraintOrLoad + ", there must two features (i.e. Datum Points)";
		throw isis::application_exception(TempError.c_str());	
	}


}

////////////////////////////////////////////////////////////////////////////////////////////////////////////
void ValidateAnalysisInputs (const std::string	&in_ConfigurationID, const CADAnalyses &in_AnalysesCAD ) 
																	throw (isis::application_exception)	
{
	//char tempBuffer[64];
	std::string TempError =  "Configuration ID: " + in_ConfigurationID + " ";

	//  Verify that there is one and only one FEA analysis.  Later, we will support multiple analyses.   
	if ( in_AnalysesCAD.analysisFEA.size() != 1 )
	{
		TempError += "Per assembly, there can be one and only one FEA analysis.  Therefore, there must be one and only <Analyses> <FEA...> xml entries per assembly.  In the future multiple analyses per assembly will be supported.  <FEA...> count: " + in_AnalysesCAD.analysisFEA.size();
		throw isis::application_exception(TempError.c_str());	
	}

	for ( list<AnalysisFEA>::const_iterator i(in_AnalysesCAD.analysisFEA.begin());
		  i != in_AnalysesCAD.analysisFEA.end();
		  ++ i )
	{


		//////////////////////////////
		// Constraints
		//////////////////////////////
		// Verify AnalysisComponents contains at least one components
		if (i->cADAnalysisComponents.size() == 0 )
		{
			TempError += "Per FEA analysis, there must be at least one Component in AnalysisComponents.";
			throw isis::application_exception(TempError.c_str());	
		}

		//////////////////////////////
		// Constraints
		//////////////////////////////
		// Verify at least one constraint
		if (i->analysisConstraints.size() == 0 )
		{
			TempError += "Per FEA analysis, there must be at least on AnalysisConstraint.";
			throw isis::application_exception(TempError.c_str());	
		}

		for ( std::list<AnalysisConstraint>::const_iterator j(i->analysisConstraints.begin());
			  j != i->analysisConstraints.end();
			  ++j )
		{
			int count = 0;
			if ( j->analysisBallDefined ) ++count;
			if ( j->analysisDisplacementDefined ) ++count;
			if ( j->analysisPinDefined ) ++count;
			if ( count != 1 )
			{
				TempError += "For each AnalysisConstraint, there must one and only one of Displacement, Ball, or Pin.";
				throw isis::application_exception(TempError.c_str());	
			}
			
			ValidateAnalysisGeometry(TempError, "AnalysisConstraint", j->geometry );

		}

		//////////////////////////////
		// Loads
		//////////////////////////////
		// Verify at least one load
		if (i->analysisLoads.size() == 0 )
		{
			TempError += "Per FEA analysis, there must be at least on Load.";
			throw isis::application_exception(TempError.c_str());	
		}

		for ( std::list<AnalysisLoad>::const_iterator j(i->analysisLoads.begin());
			  j != i->analysisLoads.end();
			  ++j )
		{
			int count = 0;
			if ( j->forceDefined ) ++count;
			if ( j->pressureDefined ) ++count;
			if ( j->accelerationDefined ) ++count;
			if ( count != 1 )
			{
				TempError += "For each Load, there must one and only one of ForceMoment, Pressure, or Acceleration.";
				throw isis::application_exception(TempError.c_str());	
			}
			
			ValidateAnalysisGeometry(TempError, "Load", j->geometry );

		}

		//////////////////////////////
		// Solvers
		//////////////////////////////
		// Verify one and only one solver
		if ( i->analysisSolvers.size() != 1 )
		{
			TempError += "Per FEA analysis, there must be on and only one solver specified.";
			throw isis::application_exception(TempError.c_str());	
		}

		for ( std::list<AnalysisSolver>::const_iterator j( i->analysisSolvers.begin());
			  j != i->analysisSolvers.end();
			  ++j )
		{
			if ( j->type != PRO_FEM_FEAS_ABAQUS || j->meshType != PRO_FEM_SOLID_MESH || j->elementShapeType != PRO_FEM_MIDPNT_PARABOLIC_FIXED )
			{
				TempError += " For FEA analysis, the only supported solver settings are Type=\"ABAQUS\", MeshType=\"SOLID\", ShellElementType=\"N/A\", and ElementShapeType=\"MIDPOINT_PARABOLIC_FIXED\"";
				throw isis::application_exception(TempError.c_str());		
			}
		}
	}

}
////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
void MeshModel (
	ProSolid				in_pro_solid, 
	pro_fem_analysis_type	in_AnalysisType,//  PRO_FEM_ANALYSIS_STRUCTURAL, PRO_FEM_ANALYSIS_MODAL.
											//	PRO_FEM_ANALYSIS_THERMAL, PRO_FEM_ANALYSIS_FLOW

	pro_fem_solver_type		in_SolverType,	//	PRO_FEM_FEAS_NONE, PRO_FEM_FEAS_NEUTRAL, PRO_FEM_FEAS_NASTRAN, 
											//	PRO_FEM_FEAS_ANSYS, PRO_FEM_FEAS_ABAQUS

	pro_fem_mesh_type		in_MeshType,	//  PRO_FEM_FEAS_ANSYS, PRO_FEM_FEAS_NASTRAN,
											//  PRO_FEM_FEAS_ABAQUS

	pro_fem_shell_mesh_type	in_ShellMeshType, //PRO_FEM_SHELL_MESH_TRIANGLE, PRO_FEM_SHELL_MESH_QUADRANGLE 

	pro_fem_elem_shape_type in_ElemShapeType, //PRO_FEM_MIDPNT_LINEAR, PRO_FEM_MIDPNT_PARABOLIC, PRO_FEM_MIDPNT_PARABOLIC_FIXED  

	const std::string	&in_PathAndFileName ) throw (isis::application_exception)
{

	//std::cout << std::endl << "MeshModel, in_pro_solid: " << in_pro_solid;

	ProFemmeshData		femMeshData;

	femMeshData.mesh_type	= in_MeshType;		// PRO_FEM_SOLID_MESH, PRO_FEM_SHELL_MESH, PRO_FEM_MIXED_MESH, 
												// PRO_FEM_QUILT_MESH, PRO_FEM_BOUNDARY_MESH, PRO_FEM_BAR_MESH 
	
	femMeshData.shell_type	= in_ShellMeshType; // This field is ignored for a solid mesh.  
												// PRO_FEM_SHELL_MESH_TRIANGLE   = 0, PRO_FEM_SHELL_MESH_QUADRANGLE

	femMeshData.num_quilts	= 0;
	//femMeshData.pro_quilt_ref_arr = NULL; 

	femMeshData.analysis = in_AnalysisType;		// PRO_FEM_ANALYSIS_STRUCTURAL, PRO_FEM_ANALYSIS_MODAL, 
												// PRO_FEM_ANALYSIS_THERMAL, PRO_FEM_ANALYSIS_FLOW  

	femMeshData.elem_shape = in_ElemShapeType;	// PRO_FEM_MIDPNT_LINEAR, PRO_FEM_MIDPNT_PARABOLIC, 
												// PRO_FEM_MIDPNT_PARABOLIC_FIXED


	femMeshData.solver = in_SolverType;			// PRO_FEM_FEAS_NONE, PRO_FEM_FEAS_NEUTRAL, PRO_FEM_FEAS_NASTRAN, 
												// PRO_FEM_FEAS_ANSYS, PRO_FEM_FEAS_ABAQUS

	ProFemcsysref  femCsyRef;
	femCsyRef.id = PRO_VALUE_UNUSED;
	femMeshData.csys_ref = femCsyRef;   		// if id is PRO_VALUE_UNUSED - not used 

	femMeshData.num_aux_csys = 0;

	femMeshData.aux_csys_ref_arr = NULL;

	wchar_t  femOutputFileName_wide[PRO_PATH_SIZE ];
	char	 femOutputFileName[PRO_PATH_SIZE ];
	strcpy(femOutputFileName, in_PathAndFileName.c_str() );
	ProStringToWstring(femOutputFileName_wide, femOutputFileName );

	// Must display a model before it can be meshed.
	isis::isis_ProMdlDisplay( (ProMdl)in_pro_solid );

	//std::cout << std::endl <<  " femOutputFileName: " << femOutputFileName;
	isis::isis_ProFemmeshExport( in_pro_solid, &femMeshData, femOutputFileName_wide );
}
//end using solid elements ****/

////////////////////////////////////////////////////////////////////////////////////////////////////////////
void RetrieveDatumPointCoordinates( const std::string							&in_AssemblyComponentID,
									const std::string							&in_PartComponentID,
									std::map<string, isis::CADComponentData>	&in_CADComponentData_map,
									const std::string							&in_DatumName,
									CADPoint									&out_CADPoint) 
																				throw (isis::application_exception)						
{


	wchar_t  datum_name[PRO_NAME_SIZE ];

	ProStringToWstring(datum_name, (char *)in_DatumName.c_str() );


	ProModelitem  datum_point;
	isis::isis_ProModelitemByNameInit_WithDescriptiveErrorMsg (
		in_PartComponentID, in_CADComponentData_map[in_PartComponentID].name, in_CADComponentData_map[in_PartComponentID].modelType,
		in_CADComponentData_map[in_PartComponentID].modelHandle, PRO_POINT, datum_name, &datum_point);

	ProPoint  point;
	isis::isis_ProPointInit (	(ProSolid) datum_point.owner,  // ProSolid   owner_handle,
								datum_point.id,
								&point);

	ProVector part_xyz_point;
	isis::isis_ProPointCoordGet (point, part_xyz_point);



	
	double transformationMatrix[4][4];
	RetrieveTranformationMatrix_Assembly_to_Child (	in_AssemblyComponentID,
														in_PartComponentID,
														in_CADComponentData_map[in_PartComponentID].componentPaths,
														in_CADComponentData_map,  
														PRO_B_TRUE,  // bottom up
														transformationMatrix);
			
	 ProVector from_assembly_xyz_point;
	 isis::isis_ProPntTrfEval( part_xyz_point, transformationMatrix, from_assembly_xyz_point);

	 out_CADPoint.x = from_assembly_xyz_point[0];
	 out_CADPoint.y = from_assembly_xyz_point[1];
	 out_CADPoint.z = from_assembly_xyz_point[2];
	 
}

// If at lease one of the assemblies in in_CADAssemblies contains analysisFEA, then return true.
	bool IsAnAnlysisRun( const CADAssemblies &in_CADAssemblies )
	{
		for ( std::list<isis::TopLevelAssemblyData>::const_iterator i( in_CADAssemblies.topLevelAssemblies.begin()); 
				i !=  in_CADAssemblies.topLevelAssemblies.end();
				++i)
		{
			if ( i->analysesCAD.analysisFEA.size() > 0 ) return true;
		}
		return false;
	}
////////////////////////////////////////////////////////////////////////////////////////////////////////////
void GetPolygonAnalysisGeometry( 
					const std::string								&in_AssemblyComponentID,
					std::map<std::string, isis::CADComponentData>	&in_CADComponentData_map,
					const AnalysisGeometry							&in_AnalysisGeometry, 
					std::vector<isis_CADCommon::Point_3D>			&out_Polygon ) 
														throw (isis::application_exception)
{
	int numberPolygonPoints = 0;
	
	//for (std::list<AnalysisGeometryFeature>::const_iterator	l(in_AnalysisGeometry.features.begin()); l != in_AnalysisGeometry.features.end(); ++l ) 
	for (std::list<CADFeature>::const_iterator	l(in_AnalysisGeometry.features.begin()->features.begin()); l != in_AnalysisGeometry.features.begin()->features.end(); ++l ) 
	{
		isis::CADPoint  point;
		//isis::RetrieveDatumPointCoordinates(	in_AssemblyComponentID,
		//										in_AnalysisGeometry.componentID,
		//										in_CADComponentData_map,
		//										*l,  // Datum Point Name
		//										point);

		isis::RetrieveDatumPointCoordinates(	in_AssemblyComponentID,
												l->componentID,
												in_CADComponentData_map,
												l->datumName,  // // Datum Point Name 
												point);

		out_Polygon.push_back( isis_CADCommon::Point_3D( point.x, point.y, point.z));
		++numberPolygonPoints;

	}
	if ( numberPolygonPoints < 3 )
	{
		//std::string TempError = "Polygon does not have at least three points. ComponentID: " + 
		//						l->componentID + "ComponentName: " +
		//						in_CADComponentData_map[in_AnalysisGeometry.componentID].name + 
		//						" Datum feature Names: ";
		//for ( std::list<std::string>::const_iterator l( in_AnalysisGeometry.features.begin()); l != in_AnalysisGeometry.features.end(); ++l )
		//		TempError += *l + " ";	

		//std::string TempError = "Polygon does not have at least three points. ComponentID: " + 
		//						l->componentID + "ComponentName: " +
		//						in_CADComponentData_map[in_AnalysisGeometry.componentID].name + 
		//						" Component ID, Component Name, Datum Name: ";
		//	for (std::list<AnalysisGeometryFeature>::const_iterator	l(in_AnalysisGeometry.features.begin()); l != in_AnalysisGeometry.features.end(); ++l ) 
		//		TempError += l->componentID + " ";	
		std::string TempError = "Polygon does not have at least three points.  "; 
							//	l->componentID + "ComponentName: " +
							//	in_CADComponentData_map[in_AnalysisGeometry.componentID].name + 
							//	" Component ID, Component Name, Datum Name: ";
		for (std::list<CADFeature>::const_iterator	l(in_AnalysisGeometry.features.begin()->features.begin()); l != in_AnalysisGeometry.features.begin()->features.end(); ++l ) 
			TempError += "Component ID: " + l->componentID + " Datum Name: " + l->datumName;	

		throw isis::application_exception(TempError.c_str());
	}

}
////////////////////////////////////////////////////////////////////////////////////////////////////////////
std::string AnalysisGeometry_error(	const AnalysisGeometry	&in_AnalysisGeometry,
									const std::string       &in_CallingFunction,
									int						in_ExpectedNumPoints,
									int						in_ReceivedNumPoints )
{
	std::stringstream errorString;
	errorString << "Function - " << in_CallingFunction << ", received incorrect number of points defining the analysis geometry." << std::endl <<
	"Expected Number of Points: " << in_ExpectedNumPoints << std::endl <<
	"Received Number of Points: " << in_ReceivedNumPoints << std::endl <<
	//"Geometry Type: " << CADGeometryType_string(in_AnalysisGeometry) << std::endl <<
	//"Component ID: "  << in_AnalysisGeometry.componentID << std::endl <<
	"Geometry Type: " << CADGeometryType_string(in_AnalysisGeometry.features.begin()->geometryType) << std::endl;
	//"Component ID: "  << in_AnalysisGeometry.componentID << std::endl <<
	//"Datum Names: " << std::endl;
	for (std::list<CADFeature>::const_iterator i(in_AnalysisGeometry.features.begin()->features.begin()); i !=in_AnalysisGeometry.features.begin()->features.end(); ++ i )
		errorString << std::endl << "Component ID: " << i->componentID << "  Datum Name: " << i->datumName;

	return errorString.str();
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
std::string NoGridPointsWithInMesh_ErrorSstring(	const std::string &in_ComponentID, 
												const std::string &in_ComponentName, 
												const std::list<std::string> &in_DatumPointNames ) 
{
	std::string TempError = "Component_ID: " + in_ComponentID + ", Component Name: " +  in_ComponentName + " Datum Points: ";

	for ( std::list<std::string>::const_iterator tmp_i( in_DatumPointNames.begin()); tmp_i != in_DatumPointNames.end(); ++tmp_i )
	{
		TempError += " " + *tmp_i;
	}
	TempError +=  " This problem could be due to one of three problems:  1) The above datum points are not ";
	TempError +=  "ordered in a manner that would define a polygon 2) The above datum points do not define a ";
	TempError +=  "convex polygon (concave polygons are not supported at this time) or 3) The polygon is well ";
	TempError +=  "formed, but there are no mesh grid points within the polygon.";

	return TempError;
}
*/
std::string NoGridPointsWithInMesh_ErrorSstring( const std::list<CADFeature> &in_ComponentID_DatumName ) 
{
	std::string TempError;

	for ( std::list<CADFeature>::const_iterator tmp_i( in_ComponentID_DatumName.begin()); tmp_i != in_ComponentID_DatumName.end(); ++tmp_i )
	{
		TempError += "\nComponent_ID: "  + tmp_i->componentID + "  Datum Point: " + tmp_i->datumName;
	}

	return TempError;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////
void GetGridPointsWithinAnalysisGeometry( 
	const std::string								&in_AssemblyComponentID,
	std::map<std::string, isis::CADComponentData>	&in_CADComponentData_map,
	const AnalysisGeometry							&in_AnalysisGeometry,
	std::map<int,isis_CADCommon::GridPoint>			&in_GridPoints_map,
	double											in_Precision,
	std::vector<int>								&out_GridPointIds_WithinGeometry ) 
										throw (isis::application_exception)
{

	if ( in_AnalysisGeometry.features.size() != 1  || in_AnalysisGeometry.setOperationDefined )  // Not supporting set operations yet
	{
		// qqqq Need a beter error message
		std::string TempError = "Set operations are not currently supported."; 
		throw isis::application_exception(TempError.c_str());	
	}

	///////////////////////////////////////////////////////////////////////////////////////
	// Retrieve CAD Point Coordinates that define the Geometry (e.g. cylinder, sphere,...)
	///////////////////////////////////////////////////////////////////////////////////////

	std::vector<int>			gridPointIds_WithinGeometry;

	for ( std::list<AnalysisGeometryFeature>::const_iterator i(in_AnalysisGeometry.features.begin());
		  i != in_AnalysisGeometry.features.end(); ++i )
	{
			
		std::vector< isis_CADCommon::Point_3D >  geometry3DPoints;

		// Retrieve point coordinates defining the geometry
		for ( std::list<CADFeature>::const_iterator k( i->features.begin());
				k != in_AnalysisGeometry.features.begin()->features.end();
				++k )
		{
			isis::CADPoint  point;
			isis::RetrieveDatumPointCoordinates(	in_AssemblyComponentID,
													k->componentID,
													in_CADComponentData_map,
													k->datumName,  // Datum Point Name
													point);
			geometry3DPoints.push_back( isis_CADCommon::Point_3D( point.x, point.y, point.z));
			//++numberPolygonPoints;
		}

		isis_CADCommon::Geometry_3D *Geometry_3D_ptr;

		switch (i->geometryType)
		{
			case CAD_GEOMETRY_POLYGON : 
				Geometry_3D_ptr = new isis_CADCommon::Polygon_3D(geometry3DPoints, in_Precision, "GetGridPointsWithinAnalysisGeometry" );
				break;
			case CAD_GEOMETRY_CIRCLE : 
				if ( geometry3DPoints.size() != 3) throw isis::application_exception(
					AnalysisGeometry_error(	in_AnalysisGeometry, "GetGridPointsWithinAnalysisGeometry", 3, geometry3DPoints.size()).c_str());
				Geometry_3D_ptr = new isis_CADCommon::Circle_3D( geometry3DPoints[0], geometry3DPoints[1], geometry3DPoints[2], in_Precision );
				break;
			case CAD_GEOMETRY_CONCENTRIC_CIRCLES:
				if ( geometry3DPoints.size() != 3) throw isis::application_exception(
					AnalysisGeometry_error(	in_AnalysisGeometry, "GetGridPointsWithinAnalysisGeometry", 3, geometry3DPoints.size()).c_str());
				Geometry_3D_ptr = new isis_CADCommon::ConcentricCircles_3D( geometry3DPoints[0], geometry3DPoints[1], geometry3DPoints[2], in_Precision );
				break;
			case CAD_GEOMETRY_CYLINDER : 
				if ( i->secondaryGeometryQualifier != CAD_INCLUDE_END_CAPS && i->secondaryGeometryQualifier != CAD_EXCLUDE_END_CAPS)
				{
					std::string TempError = "For Cylinders, the End Caps qualifier must be set to " + CADSecondaryGeometryQualifier_string(CAD_INCLUDE_END_CAPS) + 
						" or " + CADSecondaryGeometryQualifier_string(CAD_EXCLUDE_END_CAPS) + ".  Feature ID: " + i->featureID; 
					throw isis::application_exception(TempError.c_str());	
				}
				isis_CADCommon::e_CylinderGeometryInclusionSpecifier  cylinderGeometryInclusionSpecifier;
				if ( i->secondaryGeometryQualifier == CAD_INCLUDE_END_CAPS ) 
					cylinderGeometryInclusionSpecifier = isis_CADCommon::GEOMETRY_INCLUDE_END_CAP;
				else
					cylinderGeometryInclusionSpecifier = isis_CADCommon::GEOMETRY_EXCLUDE_END_CAP;

				if ( geometry3DPoints.size() != 3) throw isis::application_exception(
					AnalysisGeometry_error(	in_AnalysisGeometry, "GetGridPointsWithinAnalysisGeometry", 3, geometry3DPoints.size()).c_str());
				Geometry_3D_ptr = new isis_CADCommon::Cylinder_3D(geometry3DPoints[0], geometry3DPoints[1], geometry3DPoints[2], cylinderGeometryInclusionSpecifier, in_Precision );
				break;
			//case CAD_GEOMETRY_CYLINDER_SURFACE :   // This was replace by computing the cylinder-all-points and subtracting the cylinder-internal-points 
			//	 if ( geometry3DPoints.size() != 3) throw isis::application_exception(
			//			AnalysisGeometry_error(	in_AnalysisGeometry, "GetGridPointsWithinAnalysisGeometry", 3, geometry3DPoints.size()).c_str());
			//	 Geometry_3D_ptr = new isis_CADCommon::Cylinder_3D(geometry3DPoints[0], geometry3DPoints[1], geometry3DPoints[2], isis_CADCommon::GEOMETRY_EXCLUDE_END_CAP, in_Precision );
			//	 break;
			case CAD_GEOMETRY_SPHERE : 
				if ( geometry3DPoints.size() != 2) throw isis::application_exception(
					AnalysisGeometry_error(	in_AnalysisGeometry, "GetGridPointsWithinAnalysisGeometry", 2, geometry3DPoints.size()).c_str());
				Geometry_3D_ptr = new isis_CADCommon::Sphere_3D(geometry3DPoints[0], geometry3DPoints[1], in_Precision );
				break;
			//case CAD_GEOMETRY_SPHERE_SURFACE: 
			//	if ( geometry3DPoints.size() != 2) throw isis::application_exception(
			//		AnalysisGeometry_error(	in_AnalysisGeometry, "GetGridPointsWithinAnalysisGeometry", 2, geometry3DPoints.size()).c_str());
			//	Geometry_3D_ptr = new isis_CADCommon::SphereSurface_3D(geometry3DPoints[0], geometry3DPoints[1], in_Precision );
			//	break;
			default:
			std::stringstream errorString;
			errorString <<
				"Function - GetGridPointsWithinAnalysisGeometry, received geometryType that is not currently supported." << std::endl <<
				//"Geometry Type: " << CADGeometryType_string(in_AnalysisGeometry.geometryType);
				"Geometry Type: " << CADGeometryType_string(in_AnalysisGeometry.features.begin()->geometryType);
			throw isis::application_exception(errorString.str().c_str());
		}

		if ( i->primaryGeometryQualifier != CAD_INTERIOR_ONLY && 
			 i->primaryGeometryQualifier != CAD_BOUNDARY_ONLY && 
			 i->primaryGeometryQualifier != CAD_INTERIOR_AND_BOUNDARY )
		{
			std::string TempError = "Function: GetGridPointsWithinAnalysisGeometry, recieved an erroneous PrimaryGeometryQualifier (" + CADPrimaryGeometryQualifier_string(i->primaryGeometryQualifier) + 
				" ).  Feature ID: " + i->featureID; 
			throw isis::application_exception(TempError.c_str());	
		}

		isis_CADCommon::e_GeneralGeometryInclusionSpecifier generalGeometryInclusionSpecifier;
		switch ( i->primaryGeometryQualifier )
		{
			case CAD_INTERIOR_ONLY:			generalGeometryInclusionSpecifier = isis_CADCommon::GEOMETRY_INTERIOR_ONLY;	break;
			case CAD_BOUNDARY_ONLY:			generalGeometryInclusionSpecifier = isis_CADCommon::GEOMETRY_BOUNDARY_ONLY;	break;
			case CAD_INTERIOR_AND_BOUNDARY: generalGeometryInclusionSpecifier = isis_CADCommon::GEOMETRY_INTERIOR_AND_BOUNDARY; break;
		}

		//std::cout << std::endl << "in_GridPoints_map size: " << in_GridPoints_map.size();
		isis_CADCommon::GetGridPointsWithinGeometry( in_GridPoints_map, *Geometry_3D_ptr, generalGeometryInclusionSpecifier, in_Precision, gridPointIds_WithinGeometry);
		delete Geometry_3D_ptr;
	}

	if ( gridPointIds_WithinGeometry.size() == 0 )
	{
		std::stringstream errorString;
		errorString <<  "Function - GetGridPointsWithinAnalysisGeometry, Failed to find FEA grid points within the geometry defined by:" << std::endl <<
						NoGridPointsWithInMesh_ErrorSstring(  in_AnalysisGeometry.features.begin()->features ); 
		throw isis::application_exception(errorString.str().c_str());		
	}
	// Remove duplicates, if any
	set<int> tempSet( gridPointIds_WithinGeometry.begin(), gridPointIds_WithinGeometry.end() );
	out_GridPointIds_WithinGeometry.assign(tempSet.begin(), tempSet.end());

}

pro_fem_analysis_type ProFemAnalysisType( e_AnalysisType in_AnalysisType )
{
	switch ( in_AnalysisType )
	{
		case ANALYSIS_STRUCTURAL:
			return PRO_FEM_ANALYSIS_STRUCTURAL;
			break;
		case ANALYSIS_BUCKLING:
			return PRO_FEM_ANALYSIS_STRUCTURAL;
			break;

		case ANALYSIS_MODAL:
			return PRO_FEM_ANALYSIS_MODAL;
			break;

		case ANALYSIS_THERMAL:
			return PRO_FEM_ANALYSIS_THERMAL;
			break;

		case ANALYSIS_FATIGUE:
			return PRO_FEM_ANALYSIS_STRUCTURAL;
			break;

		default:
			return PRO_FEM_ANALYSIS_STRUCTURAL;
	}
}



////////////////////////////////////////////////////////////////////////////////////////////////////////////
// 1. Output mesh of existing assembly
// 2. Create uniquely defined materials
// 3. Modify the mesh as follows:
//		a) Add Comments, Date, .exe version number, assembly config ID, analysis ID, Material to ComponentIDs.
//		b) Add constraints
//		c) Add loads
//
void CreateFEADeck( const std::string								&in_ProgramName_Version_TimeStamp,
					const std::string								&in_WorkingDir,  
					const std::string								&in_OriginalMeshFileName,
					const std::string								&in_ModifiedMeshFileName,
					const std::string								&in_AssemblyConfigurationID,
					const std::string								&in_AssemblyComponentID,
					const CADAnalyses								&in_CADAnalyses,
					std::map<std::string, isis::CADComponentData>	&in_CADComponentData_map,
					std::map<std::string, std::string>				&out_NastranMaterialID_to_CompnentID_map) 
													throw (isis::application_exception)
{
	// Must modify the parts to have a default material that is unique for each Creo part.
	// This is necessary so that the mesh file will have a seperate material for each part; and thus,
	// each material can be mapped to a GME component ID.
	isis::CreateUniquelyNamedMaterials(in_AssemblyComponentID, in_CADComponentData_map );


	// The precision is based on the significant figures in the mesh file generated by Creo.
	// Should read the grid points in the mesh file and determine an appropriate precision.
	// This is important because the value will vary based on the dimensions of the model.
	// For now, since mm is always used, use .01
	double precision = .01;  							 

	int subCaseID = 0;
	int	constraintSetID = 259;
	int	loadStatementID = 559;
	int	loadSetID = 59;

	for ( std::list<AnalysisFEA>::const_iterator i(in_CADAnalyses.analysisFEA.begin()); i != in_CADAnalyses.analysisFEA.end(); ++i )
	{
		//std::string MeshUnmodified_PathAndFileName = in_WorkingDir + "\\" + in_CADComponentData_map[in_AssemblyComponentID].name +
		//											 "_Nastran_orig.nas";	

		//std::string MeshModified_PathAndFileName = in_WorkingDir + "\\" + in_CADComponentData_map[in_AssemblyComponentID].name +
		//											 "_Nastran_mod.nas";	

		std::string MeshUnmodified_PathAndFileName = in_WorkingDir + "\\" + in_OriginalMeshFileName;

		std::string MeshModified_PathAndFileName =   in_WorkingDir + "\\" + in_ModifiedMeshFileName;

		isis::MeshModel ( in_CADComponentData_map[in_AssemblyComponentID].modelHandle,
						  ProFemAnalysisType(i->type),
						  PRO_FEM_FEAS_NASTRAN,
						  (*i->analysisSolvers.begin()).meshType,
						  (*i->analysisSolvers.begin()).shellElementType,
						  (*i->analysisSolvers.begin()).elementShapeType,
						   MeshUnmodified_PathAndFileName );

		// Read the mesh
		isis_CADCommon::NastranDeck     nastranDeck;  // Only supporting Nastran deck for now, will have to change this later.
		nastranDeck.ReadNastranDeck(MeshUnmodified_PathAndFileName);


		nastranDeck.AppendHeaderComment( NASTRAN_META_COMMENT_TAG + in_ProgramName_Version_TimeStamp);
		nastranDeck.AppendHeaderComment(NASTRAN_COMMENT_TAG );  // Add space

		std::string tempString =  in_AssemblyConfigurationID;
		nastranDeck.AppendHeaderComment(NASTRAN_META_COMMENT_TAG + "Assembly ConfigurationID: " + tempString );
		nastranDeck.AppendHeaderComment(NASTRAN_META_COMMENT_TAG + "FEA AnalysisID:           " + i->ID );
		nastranDeck.AppendHeaderComment(NASTRAN_META_COMMENT_TAG + "FEA AnalysisType:         " + AnalysisType_string(i->type));
		nastranDeck.AppendHeaderComment(NASTRAN_COMMENT_TAG );  // Add space

		//std::map<std::string, std::string> NastranMaterialID_to_CompnentID_map;
		
		//////////////////////////////
		// Replace Material Tokens
		/////////////////////////////
		
		// in_AssemblyComponentID is the top assembly.  We need to modify the material properties
		// for all parts subordinate to the top assembly.
		
		ComponentVistorMaterialTokens componentVistorMaterialTokens;
		VisitComponents(in_AssemblyComponentID, in_CADComponentData_map, componentVistorMaterialTokens);

		nastranDeck.ReplaceMaterialTokens_ReturnMaterialToComponentID(  
														in_AssemblyComponentID,
														componentVistorMaterialTokens.materialKey_ComponentID_map,
														componentVistorMaterialTokens.componentID_PoissonsRatio_map,
														out_NastranMaterialID_to_CompnentID_map );

		for ( std::map<std::string, std::string>::const_iterator j(out_NastranMaterialID_to_CompnentID_map.begin());
			  j != out_NastranMaterialID_to_CompnentID_map.end();
			  ++j )
		{
			std::string tempString = "MaterialID: " + j->first + "  ComponentID: " + j->second + "  ComponentName: " + in_CADComponentData_map[j->second].name;
			nastranDeck.AppendHeaderComment(NASTRAN_META_COMMENT_TAG + tempString);
		}

		nastranDeck.AppendHeaderComment(NASTRAN_COMMENT_TAG );  // Add space

		nastranDeck.AddSubCaseAndLoadStatement(subCaseID, constraintSetID, loadStatementID, loadSetID );


		////////////////////////////////
		// Retrieve mesh grid points.
		////////////////////////////////
		std::map<int,isis_CADCommon::GridPoint> gridPoints_map;
		nastranDeck.GetGridPoints(gridPoints_map);

		//std::cout << std::endl << "Before  for ( list<AnalysisFEA>::const_iterator j , count: " << in_CADAnalyses.analysisFEA.size();
		for ( list<AnalysisFEA>::const_iterator j(in_CADAnalyses.analysisFEA.begin());
			  j != in_CADAnalyses.analysisFEA.end();
			  ++j )
		{
			//////////////////////////
			// Add the constraints
			//////////////////////////
			for ( std::list<AnalysisConstraint>::const_iterator k(j->analysisConstraints.begin());
				  k != j->analysisConstraints.end();
				  ++ k)
			{
	
				std::vector<isis_CADCommon::Point_3D>   polygon;
				int numberPolygonPoints = 0;

				std::vector<int>		gridPointIds_WithinGeometry;

				//////////////////////////////
				// Get points within geometry
				//////////////////////////////
				GetGridPointsWithinAnalysisGeometry( in_AssemblyComponentID, 
													 in_CADComponentData_map,
													 k->geometry, 
													 gridPoints_map,
													 precision,
													 gridPointIds_WithinGeometry);

				if ( gridPointIds_WithinGeometry.size() == 0 )
				{
					//std::string TempError = NoGridPointsWithInMesh_ErrorSstring(  k->geometry.componentID, 
					//								in_CADComponentData_map[k->geometry.componentID].name, 
					//								 k->geometry.features );
					// qqqqq
					std::string TempError = NoGridPointsWithInMesh_ErrorSstring(  k->geometry.features.begin()->features); 

					throw isis::application_exception(TempError.c_str());					
				}

				//////////////////////////////////////////
				// Add the Contraints to the Nastran Deck
				//////////////////////////////////////////
				///////////////////////
				// Displacement
				///////////////////////
				if( k->analysisDisplacementDefined )
				{
					for ( std::vector<int>::const_iterator m( gridPointIds_WithinGeometry.begin()); m != gridPointIds_WithinGeometry.end(); ++m)
					{
						nastranDeck.AddConstraintToDeck( constraintSetID, *m, k->displacement.translation.x, 
																 k->displacement.translation.y,
																 k->displacement.translation.z);
					}
				}

				///////////////////////
				// Pin
				///////////////////////
				if( k->analysisPinDefined )
				{
					//if ( k->geometry.geometryType != CAD_GEOMETRY_CYLINDER_SURFACE )
					if ( k->geometry.features.begin()->geometryType != CAD_GEOMETRY_CYLINDER )
					{
						std::stringstream errorString;
						errorString <<
							"Function - CreateFEADeck, when defining a Pin constrait, the geometry type must be " << CADGeometryType_string(CAD_GEOMETRY_CYLINDER)  << std::endl <<
						//	"Geometry Type: " << CADGeometryType_string(k->geometry.geometryType);
							"Geometry Type: " << CADGeometryType_string(k->geometry.features.begin()->geometryType);
						throw isis::application_exception(errorString.str().c_str());	
					}

						std::vector< isis_CADCommon::Point_3D >  geometry3DPoints;

						//for ( std::list<std::string>::const_iterator l( k->geometry.features.begin());
						//		l != k->geometry.features.end();
						//		++l )
						for ( std::list<CADFeature>::const_iterator l( k->geometry.features.begin()->features.begin());
								l != k->geometry.features.begin()->features.end();
								++l )
						{
							isis::CADPoint  point;
							//isis::RetrieveDatumPointCoordinates(	in_AssemblyComponentID,
							//										k->geometry.componentID,
							//										in_CADComponentData_map,
							//										*l,  // Datum Point Name
							//										point);
							isis::RetrieveDatumPointCoordinates(	in_AssemblyComponentID,
																	l->componentID,
																	in_CADComponentData_map,
																	l->datumName,  // Datum Point Name
																	point);

							geometry3DPoints.push_back( isis_CADCommon::Point_3D( point.x, point.y, point.z));
							++numberPolygonPoints;
						}

						if ( geometry3DPoints.size() != 3 )
						{
							std::stringstream errorString;
							errorString <<
							"Function - CreateFEADeck, when defining a Pin constrait, the defining geometry must have 3 Datum Points " << std::endl <<
							//"Geometry Type: " << CADGeometryType_string(k->geometry.geometryType) << std::endl <<
							//"Component ID: " << k->geometry.componentID;
							// qqqqq
							NoGridPointsWithInMesh_ErrorSstring(k->geometry.features.begin()->features);
							throw isis::application_exception(errorString.str().c_str());	

						}

						int addedCoordinateSystemID;
						nastranDeck.AddCylindricalOrSphericalCoordinateSystem(   geometry3DPoints[0], // orgin
															 geometry3DPoints[1],  // Z axis
															 geometry3DPoints[2],  // point in z x plane
															 1,
															 isis_CADCommon::NastranDeck::CYLINDRICAL_COORDINATE_SYSTEM,
															 addedCoordinateSystemID);

						std::set<int> gridPointIds_WithinGeometry_set(gridPointIds_WithinGeometry.begin(), gridPointIds_WithinGeometry.end());
						nastranDeck.ModifyGridPointsDisplacementCoordinateSystemID( gridPointIds_WithinGeometry_set, addedCoordinateSystemID );

						
					for ( std::vector<int>::const_iterator m( gridPointIds_WithinGeometry.begin()); m != gridPointIds_WithinGeometry.end(); ++m)
					{
						nastranDeck.AddConstraintToDeck( constraintSetID, *m, 1, 0 ); 
						if ( k->pin.axialDisplacementConstraint == ANALYSIS_CONSTRAINT_FIXED ) nastranDeck.AddConstraintToDeck( constraintSetID, *m, 3, 0 ); 
						if ( k->pin.axialRotationConstraint == ANALYSIS_CONSTRAINT_FIXED ) nastranDeck.AddConstraintToDeck( constraintSetID, *m, 2, 0 ); 
					}
				}

				///////////////////////
				// Ball
				///////////////////////
				if( k->analysisBallDefined )
				{
					// if ( k->geometry.geometryType != CAD_GEOMETRY_SPHERE_SURFACE )
					if ( k->geometry.features.begin()->geometryType != CAD_GEOMETRY_SPHERE )
					
					{
						std::stringstream errorString;
						errorString <<
							"Function - CreateFEADeck, when defining a Ball constrait, the geometry type must be " << CADGeometryType_string(CAD_GEOMETRY_SPHERE)  << std::endl <<
							// "Geometry Type: " << CADGeometryType_string(k->geometry.geometryType);
						    "Geometry Type: " << CADGeometryType_string(k->geometry.features.begin()->geometryType);
						throw isis::application_exception(errorString.str().c_str());	
					}

						std::vector< isis_CADCommon::Point_3D >  geometry3DPoints;

						//for ( std::list<std::string>::const_iterator l( k->geometry.features.begin());
						//		l != k->geometry.features.end();
						//		++l )
						for ( std::list<CADFeature>::const_iterator l( k->geometry.features.begin()->features.begin());
								l != k->geometry.features.begin()->features.end();
								++l )
						{
							isis::CADPoint  point;
							//isis::RetrieveDatumPointCoordinates(	in_AssemblyComponentID,
							//										k->geometry.componentID,
							//										in_CADComponentData_map,
							//										*l,  // Datum Point Name
							//										point);
							isis::RetrieveDatumPointCoordinates(	in_AssemblyComponentID,
																	l->componentID,
																	in_CADComponentData_map,
																	l->datumName,  // Datum Point Name
																	point);
							geometry3DPoints.push_back( isis_CADCommon::Point_3D( point.x, point.y, point.z));
							++numberPolygonPoints;
						}

						if ( geometry3DPoints.size() != 2 )
						{
							std::stringstream errorString;
							errorString <<
							"Function - CreateFEADeck, when defining a Ball constrait, the defining geometry must have 2 Datum Points " << std::endl <<
							//"Geometry Type: " << CADGeometryType_string(k->geometry.geometryType) << std::endl <<
							//"Component ID: " << k->geometry.componentID;
							NoGridPointsWithInMesh_ErrorSstring(k->geometry.features.begin()->features);
							throw isis::application_exception(errorString.str().c_str());	
						}

						int addedCoordinateSystemID;
						
						// Need a point other than geometry3DPoints[1]
						std::vector<double>  vector_Z;
						vector_Z.push_back(geometry3DPoints[1].x - geometry3DPoints[0].x );
						vector_Z.push_back(geometry3DPoints[1].y - geometry3DPoints[0].y );
						vector_Z.push_back(geometry3DPoints[1].z - geometry3DPoints[0].z );

						std::vector<double>  vector_Temp;
						vector_Temp.push_back( vector_Z[0] + 10 );
						vector_Temp.push_back( vector_Z[1]);
						vector_Temp.push_back( vector_Z[2] );

						std::vector<double>  vector_YZ = isis_CADCommon::CrossProduct_3D(vector_Z, vector_Temp);

						isis_CADCommon::Point_3D point_YZ( 
											vector_YZ[0] +  geometry3DPoints[0].x,
											vector_YZ[1] +  geometry3DPoints[0].y,
											vector_YZ[2] +  geometry3DPoints[0].z);


						nastranDeck.AddCylindricalOrSphericalCoordinateSystem(   geometry3DPoints[0], // orgin
															 geometry3DPoints[1],  // Z axis
															 point_YZ,  // point in z x plane
															 1,
															 isis_CADCommon::NastranDeck::SPHERICAL_COORDINATE_SYSTEM,
															 addedCoordinateSystemID);

						std::set<int> gridPointIds_WithinGeometry_set(gridPointIds_WithinGeometry.begin(), gridPointIds_WithinGeometry.end());
						nastranDeck.ModifyGridPointsDisplacementCoordinateSystemID( gridPointIds_WithinGeometry_set, addedCoordinateSystemID );

						
					for ( std::vector<int>::const_iterator m( gridPointIds_WithinGeometry.begin()); m != gridPointIds_WithinGeometry.end(); ++m)
					{
						nastranDeck.AddConstraintToDeck( constraintSetID, *m, 1, 0 ); 
					}
				}

			} // END for ( std::list<AnalysisConstraint>::const_iterator k(j->analysisConstraints.begin());

			//////////////////////////
			// Add Loads
			//////////////////////////

			//std::cout << std::endl << "Before  for ( std::list<AnalysisLoad>::const_iterator , count: " << j->analysisLoads.size();
			for ( std::list<AnalysisLoad>::const_iterator k(j->analysisLoads.begin());
				  k != j->analysisLoads.end();
				  ++ k)
			{
	
				std::vector<isis_CADCommon::Point_3D>   polygon;
				int numberPolygonPoints = 0;

				//////////////////////////////
				// Get points within geometry
				//////////////////////////////
				std::vector<int>		gridPointIds_WithinGeometry;

				GetGridPointsWithinAnalysisGeometry( in_AssemblyComponentID, 
													 in_CADComponentData_map,
													 k->geometry,
													 gridPoints_map,
													 precision,
													 gridPointIds_WithinGeometry);


				if ( gridPointIds_WithinGeometry.size() == 0 )
				{
					//std::string TempError = NoGridPointsWithInMesh_ErrorSstring(  k->geometry.componentID, 
					//								in_CADComponentData_map[k->geometry.componentID].name, 
					//								 k->geometry.features );

					std::stringstream errorString;
					errorString <<  "Function - CreateFEADeck, Failed to find FEA grid points within the geometry defined by:" << std::endl <<
									NoGridPointsWithInMesh_ErrorSstring(  k->geometry.features.begin()->features ); 
					throw isis::application_exception(errorString.str().c_str());					
				}

				//std::cout << std::endl << "Points within polygon, count: " << gridPointIds_WithinPolygon.size();

				//////////////////////////////////////////
				// Add Forces to Nastran Deck
				//////////////////////////////////////////
				if( k->forceDefined )
				{
					double numberPoints = gridPointIds_WithinGeometry.size();
					double force_x = k->force.force.x / numberPoints;
					double force_y = k->force.force.y / numberPoints;
					double force_z = k->force.force.z / numberPoints;

					for ( std::vector<int>::const_iterator m( gridPointIds_WithinGeometry.begin()); m != gridPointIds_WithinGeometry.end(); ++m)
					{
						nastranDeck.AddForceToDeck( loadSetID, *m, 1, 1, force_x,
																  force_y,
																  force_z);
					}
				}

				if( k->accelerationDefined )
				{
					nastranDeck.AddAccelerationToDeck(loadSetID, 1, 1,	k->acceleration.direction.x, 
																		k->acceleration.direction.y,
																		k->acceleration.direction.z);
				}

			} // END for ( std::list<AnalysisConstraint>::const_iterator k(j->analysisConstraints.begin());


		} // END for ( list<AnalysisFEA>::const_iterator j(in_CADAnalyses.analysisFEA.begin());
		nastranDeck.WriteNastranDeck( MeshModified_PathAndFileName);

	} // END for ( std::list<AnalysisFEA>::const_iterator i(in_CADAnalyses.analysisFEA.begin()); i != in_CADAnalyses.analysisFEA.end(); ++i )
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Pre-Conditions
//	in_TopLevelAssemblyData.CADAnalyses.list<AnalysisFEA> can contain only one item.  Multiple Analysis
//  per assembly are not currently supported.
//
void CreateXMLFile_FEAPostProcessingParameters( 
						const std::string									&in_PathAndFileName,
						const TopLevelAssemblyData							&in_TopLevelAssemblyData,
						std::map<std::string, Material>						&in_Materials,
						std::map<std::string, std::string>                  &in_NastranMaterialID_to_CompnentID_map,
						std::map<std::string, isis::CADComponentData>		&in_CADComponentData_map )
																	throw (isis::application_exception)
{
	// Example XML file
	//
	//<?xml version="1.0" encoding="UTF-8" standalone="no" ?>
	//<Components ConfigurationID="100000452" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="CADPostProcessingParameters.xsd">
	//	<Component ComponentID="0" FEAElementID="PSOLID_1" _id="id78">
	//		<Material Bearing="165" Mises="165" Shear="82.64999999999999" Units="MPa" _id="id79"/>
	//		<Metrics _id="id7a">
	//			<Metric MetricID="id-0066-00005242" Type="Mises" _id="id7b"/>
	//			<Metric MetricID="id-0066-00005243" Type="Shear" _id="id7c"/>
	//			<Metric MetricID="id-0066-00005244" Type="Bearing" _id="id7d"/>
	//			<Metric MetricID="id-0066-00005245" Type="FactorOfSafety" _id="id7e"/>
	//		</Metrics>
	//  </Component>
	//</Components>

	std::map<std::string, std::string> ComponentID_to_NastranMaterialID_map;

	for ( std::map<std::string, std::string>::const_iterator ii( in_NastranMaterialID_to_CompnentID_map.begin());
		  ii != in_NastranMaterialID_to_CompnentID_map.end(); ++ii)
	{
		ComponentID_to_NastranMaterialID_map[ii->second] = ii->first;
	}


	//std::cout << std::endl << "in_FileName: " << in_FileName;

	Udm::SmartDataNetwork dn_FEAPostProcessingParameters( CADPostProcessingParameters::diagram );
	dn_FEAPostProcessingParameters.CreateNew( in_PathAndFileName, "CADPostProcessingParameters", CADPostProcessingParameters::Components::meta);
	
	try
	{

		CADPostProcessingParameters::Components componentsRoot = 
			      CADPostProcessingParameters::Components::Cast(dn_FEAPostProcessingParameters.GetRootObject());

		componentsRoot.ConfigurationID() = in_TopLevelAssemblyData.configurationID;
	
		// Outer loop - CADAnalysisComponents  ( std::list<CADAnalysisComponent>
		// We can loop through via the following function because list<AnalysisFEA> analysisFEA can only contain 
		// one item (i.e. multiple analyses per assembly are currently not supported.

		for (std::list<isis::CADAnalysisComponent>::const_iterator i(in_TopLevelAssemblyData.analysesCAD.analysisFEA.begin()->cADAnalysisComponents.begin());
		i != in_TopLevelAssemblyData.analysesCAD.analysisFEA.begin()->cADAnalysisComponents.end();
		++i )
		{

			/////////////////////////////////////////////////////////////////////
			// Add ComponentID and FEAElementID
			// e.g. <Component ComponentID="100000678" FEAElementID="PSOLID_1">
			/////////////////////////////////////////////////////////////////////
			CADPostProcessingParameters::Component  componentRoot = CADPostProcessingParameters::Component::Create( componentsRoot );
			componentRoot.ComponentID() = i->componentID;
			componentRoot.FEAElementID() = "PSOLID_" + ComponentID_to_NastranMaterialID_map[i->componentID];
		
			/////////////////////////////////////////////////////////////////////
			// Add Material Properties
			// e.g <Material Mises="165" Bearing="165" Shear="82.65" Units="MPa"/>
			/////////////////////////////////////////////////////////////////////

			int numberOfCycles;
			if ( i->infiniteCycle ) 
					numberOfCycles = isis::c_InifiteLifeNumberOfCycles;
			else
				numberOfCycles = 1;

			isis::AnalysisMaterialProperties_Allowables  materialPropertiesAllowables; 
			std::string tempMatierialID = in_CADComponentData_map[i->componentID].materialID;

			isis::ComputeAllowableStressLevels(	numberOfCycles,in_Materials[tempMatierialID].analysisMaterialProperties, 
										materialPropertiesAllowables );

			CADPostProcessingParameters::Material materialRoot = CADPostProcessingParameters::Material::Create( componentRoot );

			materialRoot.Mises()   = materialPropertiesAllowables.tensileStrength;
			materialRoot.Bearing() = materialPropertiesAllowables.bearingStrength;
			materialRoot.Shear()   = materialPropertiesAllowables.shearStrength;
			materialRoot.Units()   = "MPa";

			/////////////////////////////////////////////////////////////////////
			// Add Metrics
			// e.g   <Metrics>
			//			<Metric MetricID="m1" Type="Shear"/>
			//			<Metric MetricID="m2" Type="Bearing"/>
			//			<Metric MetricID="m3" Type="Mises"/>
			//			<Metric MetricID="m4" Type="FactorOfSafety"/>
			//		</Metrics>
			/////////////////////////////////////////////////////////////////////
			if ( i->cadAnalysisComponentMetrics.size() > 0 )
			{
				CADPostProcessingParameters::Metrics metricsRoot = CADPostProcessingParameters::Metrics::Create( componentRoot );
				for (std::list<isis::CADAnalysisComponentMetrics>::const_iterator j(i->cadAnalysisComponentMetrics.begin());
					j != i->cadAnalysisComponentMetrics.end(); ++j )
				{
					CADPostProcessingParameters::Metric metricRoot = CADPostProcessingParameters::Metric::Create( metricsRoot );
					metricRoot.MetricID() = j->ID;
					metricRoot.Type() = j->type;
				}
			}
		}  // end for i loop


		///////////////////////////
		// Write XML File
		//////////////////////////
		dn_FEAPostProcessingParameters.CloseWithUpdate();
	}
	catch ( ... )
	{
		// If any error occurs, do not save the metrics xml file.  It makes no sense to save a metrics file
		// with erroneous data.
		dn_FEAPostProcessingParameters.CloseNoUpdate();
		throw;
	}

}

////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Create_FEADecks_BatFiles( 
					const TopLevelAssemblyData							&in_TopLevelAssemblyData,
					std::map<std::string, Material>						&in_Materials,
					const std::string									&in_WORKING_DIR,
					const std::string									in_ProgramName_Version_TimeStamp,
					std::map<std::string, isis::CADComponentData>		&in_CADComponentData_map )
																	throw (isis::application_exception)
{

	//////////////////////////////////////
	// Create Analysis Directories
	//////////////////////////////////////
	std::string createAnalysisDir = "if not exist Analysis mkdir Analysis";
	isis::ExecuteSystemCommand( createAnalysisDir);
	isis::ExecuteSystemCommand( "if not exist Analysis\\Abaqus mkdir Analysis\\Abaqus");
	isis::ExecuteSystemCommand( "if not exist Analysis\\CalculiX mkdir Analysis\\CalculiX");

	//////////////////////////////////////
	// Set File Name Strings
	//////////////////////////////////////
	std::string analysisDirName = "Analysis";
	std::string abaqusDirName   = "Abaqus";
	std::string caculixDirName  = "CalculiX";

	std::string analysisWorkingDir = in_WORKING_DIR +     "\\" + analysisDirName;
	std::string abaqusWorkingDir   = analysisWorkingDir + "\\" + abaqusDirName;
	std::string caculixWorkingDir  = analysisWorkingDir + "\\" + caculixDirName;

	// Since Creo will create the Nastran deck, the name of the created file cannot exceed 31 characters.
	// Therefore, allot 23 characters plus the suffix of 8 charaters (i.e. _Nas_org or _Nas_mod)
	std::string tempMeshFilePreFix = in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name;
	if (tempMeshFilePreFix.size() > 23 ) tempMeshFilePreFix = tempMeshFilePreFix.substr(0, 23 );			

	std::string originalMeshFileName = tempMeshFilePreFix + "_Nas_org.nas";	
	std::string modifiedMeshFileName = tempMeshFilePreFix + "_Nas_mod.nas";	

	// No length restriction on the length of the Calculix deck name
	std::string meshFileName_Calculix = in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name + "_CalculiX";	
    std::string abaqusPostProcessingParametersXMLFileName = in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name + "_Abaqus.xml";

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// The ValidateAnalysisInputs line will throw an execption if the input xml for the FEA analysis contains errors.
	// Note - This fucntion inforces that only one FEA analysis per assembly is currently supported.
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	ValidateAnalysisInputs (in_TopLevelAssemblyData.configurationID, in_TopLevelAssemblyData.analysesCAD );

	////////////////////////////
	// Write Abaqus bat file
	////////////////////////////

	std::ofstream  abaqusAnalysisBatFile; 
	std::string abaqusAnalysisBatFileName =  in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name + "_Abaqus.bat" ;
	std::string analysisDirectoryAndBatFileName = abaqusWorkingDir + "\\" + abaqusAnalysisBatFileName ;
	abaqusAnalysisBatFile.open (analysisDirectoryAndBatFileName.c_str(),std::ios::out | std::ios::trunc  );


	abaqusAnalysisBatFile << "REM " + in_ProgramName_Version_TimeStamp << std::endl;
	abaqusAnalysisBatFile << "REM The following system environment variable must be defined:" << std::endl;
	abaqusAnalysisBatFile << "REM    PROE_ISIS_EXTENSIONS   // typically set to C:\\Program Files\\Proe ISIS Extensions" << std::endl;
	abaqusAnalysisBatFile	<< "echo off" << std::endl;
	abaqusAnalysisBatFile << "REM Invoke Abaqus" << std::endl;
	abaqusAnalysisBatFile	<< "echo." << std::endl;
	abaqusAnalysisBatFile << "echo Invoke Abaqus" << std::endl;
	abaqusAnalysisBatFile	<< std::endl;
	abaqusAnalysisBatFile << "call abaqus fromnastran job=" + in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name + " input=..\\" + modifiedMeshFileName << std::endl;
	abaqusAnalysisBatFile << "call abaqus analysis interactive job=" + in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name << std::endl;
	abaqusAnalysisBatFile	<< "call abaqus odbreport job=" + in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name + " results" << std::endl;
	abaqusAnalysisBatFile	<< std::endl;
	abaqusAnalysisBatFile	<< "echo." << std::endl;
	abaqusAnalysisBatFile << "echo Analysis completed. See the " << in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name << ".rep for the results." << std::endl;
	abaqusAnalysisBatFile	<< "echo." << std::endl;
	abaqusAnalysisBatFile	<< std::endl;
	abaqusAnalysisBatFile << "REM Invoke Post Processing" << std::endl;
	abaqusAnalysisBatFile	<< "echo Invoking Post Processing" << std::endl;
	abaqusAnalysisBatFile	<< "REM The following line (set PYTHONPATH...) is needed so that other python scripts (e.g. ComputedMetricsSummary.py) can be located when invoking Abaqus post processing." << std::endl;
	abaqusAnalysisBatFile	<< "set PYTHONPATH=%PROE_ISIS_EXTENSIONS%bin" << std::endl;
	abaqusAnalysisBatFile	<< "call abaqus cae noGUI=\"%PROE_ISIS_EXTENSIONS%\"\\bin\\ABQ_CompletePostProcess.py";
	abaqusAnalysisBatFile	<< " -- -o " <<  in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name << ".odb" << 
		  "  -p " <<  abaqusPostProcessingParametersXMLFileName << std::endl;	
	abaqusAnalysisBatFile	<< std::endl;
	abaqusAnalysisBatFile	<< "echo." << std::endl;
	abaqusAnalysisBatFile	<< "echo Post processing completed" << std::endl;
	abaqusAnalysisBatFile	<< "echo." << std::endl;
	abaqusAnalysisBatFile	<< "set /p ExitPrompt= Type Enter to exit" << std::endl;
	abaqusAnalysisBatFile.close();

	////////////////////////////////////////////////////
	// Write "convert Nastran to CalculiX" bat file
	////////////////////////////////////////////////////
	std::ofstream  calculixBatFile_ConvertDeck; 
	std::string calculixDeckBatFileName =  in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name + "_CalculiX_Deck.bat" ;
	analysisDirectoryAndBatFileName = caculixWorkingDir + "\\" + calculixDeckBatFileName ;

	calculixBatFile_ConvertDeck.open (analysisDirectoryAndBatFileName.c_str(),std::ios::out | std::ios::trunc  );

	calculixBatFile_ConvertDeck << "REM " + in_ProgramName_Version_TimeStamp << std::endl;
	calculixBatFile_ConvertDeck << "REM	The following system environment variable must be defined:" << std::endl;
	calculixBatFile_ConvertDeck << "REM	   PROE_ISIS_EXTENSIONS	// typically set to	C:\\Program Files\\Proe ISIS Extensions" << std::endl;
	calculixBatFile_ConvertDeck << "REM Invoke DeckConverter ( Convert Nastran deck to CalculiX deck)" << std::endl;
	calculixBatFile_ConvertDeck	<< "echo off" << std::endl;
	calculixBatFile_ConvertDeck	<< std::endl;
	calculixBatFile_ConvertDeck	<< "echo." << std::endl;
	calculixBatFile_ConvertDeck	<< "echo Invoke deck conversion from Nastran to CalculiX" << std::endl;
	calculixBatFile_ConvertDeck	<< "echo." << std::endl;
	calculixBatFile_ConvertDeck << "\"%PROE_ISIS_EXTENSIONS%\"\\bin\\DeckConverter.exe -i ..\\" + modifiedMeshFileName + " -o " + meshFileName_Calculix << std::endl;
	calculixBatFile_ConvertDeck	<< "echo." << std::endl;
	calculixBatFile_ConvertDeck	<< "echo Conversion completed" << std::endl;
	calculixBatFile_ConvertDeck	<< "echo." << std::endl;
	calculixBatFile_ConvertDeck	<< "set /p ExitPrompt= Type Enter to exit" << std::endl;
	calculixBatFile_ConvertDeck.close();

	////////////////////////////////////////////////
	// Write "invoke CalculiX" bat file
	///////////////////////////////////////////////
	std::ofstream  calculixBatFile_RunDeck; 
	std::string calculixLinuxBatFileName =  in_CADComponentData_map[in_TopLevelAssemblyData.assemblyComponentID].name + "_CalculiX_LinuxRun.bat" ;
	analysisDirectoryAndBatFileName = caculixWorkingDir + "\\" + calculixLinuxBatFileName ;
				
	calculixBatFile_RunDeck.open (analysisDirectoryAndBatFileName.c_str(),std::ios::out | std::ios::trunc  );
	calculixBatFile_RunDeck << "REM " + in_ProgramName_Version_TimeStamp << std::endl;
	calculixBatFile_RunDeck << "REM	This bat file should be invoked on a Linux machine with CalculiX 2.3 installed:";
	calculixBatFile_RunDeck	<< std::endl;
	calculixBatFile_RunDeck << "ccx_2.3 -i " + meshFileName_Calculix;
	calculixBatFile_RunDeck.close();

	/////////////////////////////
	// Create FEA Deck
	/////////////////////////////
	std::map<std::string, std::string> NastranMaterialID_to_CompnentID_map;

	std::cout << std::endl << std::endl << "Creating finite element mesh, this could take a few minutes ...";
	// WARNING - Do not save the assembly/models after this point.  Doing so will save the temporarily created material.
	isis::CreateFEADeck(	in_ProgramName_Version_TimeStamp, 
							analysisWorkingDir, 
							originalMeshFileName,
							modifiedMeshFileName,
							in_TopLevelAssemblyData.configurationID,
							in_TopLevelAssemblyData.assemblyComponentID,  
							in_TopLevelAssemblyData.analysesCAD,
							in_CADComponentData_map,
							NastranMaterialID_to_CompnentID_map);

	///////////////////////////////////////////////////////////////////////////////////
	// Create XML File containing Component IDs, Material Allowables, and Metric IDs
	///////////////////////////////////////////////////////////////////////////////////
	std::string fEAPostProcessingParametersFile = abaqusWorkingDir + "\\" + abaqusPostProcessingParametersXMLFileName;
	CreateXMLFile_FEAPostProcessingParameters(	fEAPostProcessingParametersFile,
												in_TopLevelAssemblyData,
												in_Materials,
												NastranMaterialID_to_CompnentID_map,
												in_CADComponentData_map );

	// Modify fEAPostProcessingParametersFile to have Version and TimeStamp
	std::ofstream  postProcessingParametersFile; 
	postProcessingParametersFile.open( fEAPostProcessingParametersFile, ios::app );
	postProcessingParametersFile << std::endl;
	postProcessingParametersFile << "<!--  " + in_ProgramName_Version_TimeStamp << " -->";
	postProcessingParametersFile.close();

	std::cout << std::endl << "   Created: .\\" + analysisDirName + "\\" << originalMeshFileName;
	std::cout << std::endl << "   Created: .\\" + analysisDirName + "\\" << modifiedMeshFileName ;
	std::cout << std::endl << "   Created: .\\" + analysisDirName + "\\" + abaqusDirName + "\\" <<  abaqusAnalysisBatFileName ;
	std::cout << std::endl << "   Created: .\\" + analysisDirName + "\\" + abaqusDirName + "\\" <<  abaqusPostProcessingParametersXMLFileName;
	std::cout << std::endl << "   Created: .\\" + analysisDirName + "\\" + caculixDirName + "\\" <<  calculixDeckBatFileName;
	std::cout << std::endl << "   Created: .\\" + analysisDirName + "\\" + caculixDirName + "\\" <<  calculixLinuxBatFileName;

}
////////////////////////////////////////////////////////////////////////////////////////////////////////////
} // end  namespace isis 
