/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "MaterialProperties.h"
#include <CommonUtilities.h>
#include <string>
#include <stdlib.h>
#include <locale>
#include <iostream>
#include <ostream>
#include <sstream>


namespace isis
{
/*
	std::string ConvertToUpperCase(const std::string &in_String)
	{
		std::string temp_string(in_String);
		std::locale loc;
		for (std::string::iterator p = temp_string.begin(); temp_string.end() != p; ++p)    *p = toupper(*p, loc);
		return temp_string;
	}

*/


void stream_AnalysisMaterialProperties( const AnalysisMaterialProperties &in_CADAnalyses, std::ostream &out_Stream )																	
{
	out_Stream << std::endl << "Material Name: " << in_CADAnalyses.materialName;
	out_Stream << std::endl << "Material Type: " << isis::MaterialType_string(in_CADAnalyses.materialType);
	
	out_Stream << std::endl << "   ModulusOfElasticity";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.modulusOfElasticityDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.modulusOfElasticity;

	out_Stream << std::endl << "   PoissonsRatio";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.poissonsRatioDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.poissonsRatio;

	out_Stream << std::endl << "   TensileYieldStrength";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.tensileYieldStrengthDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.tensileYieldStrength;

	out_Stream << std::endl << "   TensileUltimateStrength";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.tensileUltimateStrengthDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.tensileUltimateStrength;

	out_Stream << std::endl << "   ShearStrength";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.shearStrengthDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.shearStrength;

	out_Stream << std::endl << "   BearingYieldStrength";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.bearingYieldStrengthDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.bearingYieldStrength;

	out_Stream << std::endl << "   BearingUltimateStrength";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.bearingUltimateStrengthDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.bearingUltimateStrength;

	out_Stream << std::endl << "   FatigueStrength";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.fatigueStrengthDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.fatigueStrength;

	out_Stream << std::endl << "   fatigueNumberOfCycles";
	out_Stream << std::endl << "      Defined: " << in_CADAnalyses.fatigueNumberOfCyclesDefined;
	out_Stream << std::endl << "      Value: "   << in_CADAnalyses.fatigueNumberOfCycles;

};

void stream_AnalysisMaterialProperties_Allowables( const AnalysisMaterialProperties_Allowables	&in_AnalysisMaterialProperties, std::ostream &out_Stream )																	
{
	out_Stream << std::endl << "Material Name: " << in_AnalysisMaterialProperties.materialName;
	out_Stream << std::endl << "Material Type: " << isis::MaterialType_string(in_AnalysisMaterialProperties.materialType);

	out_Stream << std::endl << "   TensileStrength";
	out_Stream << std::endl << "      Value: "   << in_AnalysisMaterialProperties.tensileStrength;

	out_Stream << std::endl << "   ShearStrength";
	out_Stream << std::endl << "      Value: "   << in_AnalysisMaterialProperties.shearStrength;

	out_Stream << std::endl << "   BearingStrength";
	out_Stream << std::endl << "      Value: "   << in_AnalysisMaterialProperties.bearingStrength;

	out_Stream << std::endl << "   NumberOfCycles";
	out_Stream << std::endl << "      Value: "   << in_AnalysisMaterialProperties.numberOfCycles;

};

///////////////////////////////////////////////////////////////////////////////////////////////
bool	StressValueComputed(	bool	in_Value_1_Defined,
								double	in_Value_1,	
								bool	in_Value_2_Defined,
								double	in_Value_2,
								double	in_ReferenceValue,
								double  &out_ComputedValue )
{

		if ( in_Value_1_Defined && in_Value_2_Defined  )
		{
			out_ComputedValue =  (in_ReferenceValue / in_Value_1 ) * in_Value_2;
			//std::cout << std::endl << "BEGIN StressValueComputed";
			//std::cout << std::endl << "  in_Value_1_Defined: "	<< in_Value_1_Defined;
			//std::cout << std::endl << "  in_Value_1: "			<< in_Value_1;
			//std::cout << std::endl << "  in_Value_2_Defined: "	<< in_Value_1_Defined;
			//std::cout << std::endl << "  in_Value_2: "			<< in_Value_2;
			//std::cout << std::endl << "  in_ReferenceValue:  "	<< in_ReferenceValue;
			//std::cout << std::endl << "  out_ComputedValue:  "	<< out_ComputedValue;
			//std::cout << std::endl << "END StressValueComputed";
			return true;
		}
		else
		{
			return false;
		} 
}
/////////////////////////////////////////////////////////////////////////////////////////////////

// General Rules:
//	1)	If in_NumberOfCycles == 0 assume infinite number of cycles
//	2)	Consider 500,000,000 (5M) cycles to be infinite cycles 
//	3)  Consider less than 1,000 cycles to be equivalent to 1 cycle for determining the allowable stress levels. 
//	3)	Aluminum must have in_AnalysisMaterialProperties.fatigueStrength and fatigueNumberOfCycles set.
//	4)	For Steel, if in_AnalysisMaterialProperties.fatigueStrength is not set, assume that it is 
//		50% of in_AnalysisMaterialProperties.tensileUltimateStrength
//	5)	TBD For cycle values between 1,000 and 500,000,000, we need a graph of X-Number Cycles, and Y-Stress.  These
//		are usually log base 10 graphs. 
//		
//	Material  		Fatigue		Number 		Tensile							Bearing							Shear
//					Given		Cycles		Strength						Strength						Strength
//	-----------		--------	-------		--------------					-----------						---------------
//	Steel/Aluminum	Yes			N < 1000	tensileYieldStrength			bearingYieldStrength			shearStrength
//	Stee/Aluminum	Yes			1000 < N 	fatigueStrength					SF * bearingUltimateStrength	SF * shearStrength
//	Steel			No			1000 < N 	.5 * tensileUltimateStrength	SF  * bearingYieldStrength		SF  * shearStrength
//	Aluminum		No			(Not Allowed, throw exception)
//
// SF - Scale Factor = fatigueStrength / tensileUltimateStrength; 

void ComputeAllowableStressLevels(	int										in_NumberOfCycles,
									const AnalysisMaterialProperties		&in_AnalysisMaterialProperties,
								    AnalysisMaterialProperties_Allowables	&out_AnalysisMaterialProperties )
																				throw (isis::application_exception)
{
	///////////////////////////////////////////////
	// zero out values
	///////////////////////////////////////////////
	out_AnalysisMaterialProperties.materialName = "";
	out_AnalysisMaterialProperties.tensileStrength	= 0;
	out_AnalysisMaterialProperties.bearingStrength	= 0;
	out_AnalysisMaterialProperties.shearStrength	= 0;
	out_AnalysisMaterialProperties.numberOfCycles	= 0;

	///////////////////////////////////////////////
	// Only Aluminum and Steel Currently Supported
	///////////////////////////////////////////////
	if ( ( in_AnalysisMaterialProperties.materialType != MATERIAL_ALUMINUM ) &&
		 ( in_AnalysisMaterialProperties.materialType != MATERIAL_STEEL ) )
																				
	{
		// zzz fix this to use the common functions
		std::string tempMaterialType = "Unknown";
		if (in_AnalysisMaterialProperties.materialType == MATERIAL_CERAMIC )tempMaterialType = "Ceramic";
		if (in_AnalysisMaterialProperties.materialType == MATERIAL_PLASTIC )tempMaterialType = "Plastic";

		std::string TempError = "Function ComputeAllowableStressLevels(...) for material " + in_AnalysisMaterialProperties.materialName + 
		" was passed: " + tempMaterialType + 
		".  Currently only Steel and Aluminum are supported. Plastic and Ceramic will be supported in future releases.";   
		throw isis::application_exception(TempError.c_str());
	}

	///////////////////////////////////////////////////////
	// Fatigue strength must be given defined for Aluminum
	///////////////////////////////////////////////////////
	if ( ( in_AnalysisMaterialProperties.materialType == MATERIAL_ALUMINUM ) &&
		 ( !in_AnalysisMaterialProperties.fatigueStrengthDefined ) )
	{
		std::string TempError = "Function ComputeAllowableStressLevels(...) for material " + in_AnalysisMaterialProperties.materialName + 
		" does not have a fatigue strength defined.  Aluminum must have fatigue strength defined";   
		throw isis::application_exception(TempError.c_str());		
	}

	////////////////////////////////////////
	// Tensile yield strength must be given
	////////////////////////////////////////
	if ( !in_AnalysisMaterialProperties.tensileYieldStrengthDefined )
	{
		std::string TempError = "Function ComputeAllowableStressLevels(...) for material " + in_AnalysisMaterialProperties.materialName + 
		" does not have a tensile yield strength defined.  All material must have tensile yield strength defined.";   
		throw isis::application_exception(TempError.c_str());		
	}

	////////////////////////////////////////
	// Tensile Ultimate strength must be given
	////////////////////////////////////////
	if ( !in_AnalysisMaterialProperties.tensileUltimateStrengthDefined )
	{
		std::string TempError = "Function ComputeAllowableStressLevels(...) for material " + in_AnalysisMaterialProperties.materialName + 
		" does not have a tensile ultimate strength defined.  All material must have tensile ultimate strength defined.";   
		throw isis::application_exception(TempError.c_str());		
	}


	//////////////////////////////////////////////
	// Shear Strength must be given for Aluminum
	/////////////////////////////////////////////
	if ( !in_AnalysisMaterialProperties.shearStrengthDefined &&  in_AnalysisMaterialProperties.materialType == MATERIAL_ALUMINUM  )
	{
		std::string TempError = "Function ComputeAllowableStressLevels(...) for material " + in_AnalysisMaterialProperties.materialName + 
		" does not have a shear strength defined.  Aluminum must have shear strength defined.";   
		throw isis::application_exception(TempError.c_str());		
	}

	////////////////////////////////////////
	// Set Material Name and Type
	////////////////////////////////////////
	out_AnalysisMaterialProperties.materialName = in_AnalysisMaterialProperties.materialName;
	out_AnalysisMaterialProperties.materialType = in_AnalysisMaterialProperties.materialType;

	////////////////////////////////////////
	// Set Number of cycles
	////////////////////////////////////////
	out_AnalysisMaterialProperties.numberOfCycles = in_NumberOfCycles;

	////////////////////////////////////////
	// Compute tempShearStrength
	////////////////////////////////////////
	double tempShearStrength;
	if ( in_AnalysisMaterialProperties.shearStrengthDefined )
	{
		tempShearStrength = in_AnalysisMaterialProperties.shearStrength;
	}
	else 
	{
		// Case for steel when shear strength is not defined, use rule of thumb
		tempShearStrength = .58 * in_AnalysisMaterialProperties.tensileYieldStrength;
	}

	////////////////////////////////////////
	// No Fatigue Case
	////////////////////////////////////////
	if ( ( in_NumberOfCycles != 0 ) && ( in_NumberOfCycles < 1000 ) )
	{
		out_AnalysisMaterialProperties.tensileStrength = in_AnalysisMaterialProperties.tensileYieldStrength;
		out_AnalysisMaterialProperties.shearStrength   = tempShearStrength;
		
		if (in_AnalysisMaterialProperties.bearingYieldStrengthDefined )
			out_AnalysisMaterialProperties.bearingStrength = in_AnalysisMaterialProperties.bearingYieldStrength;
		else
			out_AnalysisMaterialProperties.bearingStrength = in_AnalysisMaterialProperties.tensileYieldStrength;
		
	}
	else
	{
		/////////////////////////////////////////
		// Fatigue Case - Steel and Aluminum
		////////////////////////////////////////

		// Note fatigue strength would be defined for aluminum as checked above, maybe not for steel.
		double tempFatigueStrength;

		if ( in_AnalysisMaterialProperties.fatigueStrengthDefined )
			tempFatigueStrength = in_AnalysisMaterialProperties.fatigueStrength;
		else
			tempFatigueStrength = .5 * in_AnalysisMaterialProperties.tensileUltimateStrength;

		// Tensile
		out_AnalysisMaterialProperties.tensileStrength = tempFatigueStrength;		

		// Bearing
		bool stressComputed = false;
		stressComputed = StressValueComputed(	
							in_AnalysisMaterialProperties.tensileUltimateStrengthDefined,
							in_AnalysisMaterialProperties.tensileUltimateStrength,	
							in_AnalysisMaterialProperties.bearingUltimateStrengthDefined,
							in_AnalysisMaterialProperties.bearingUltimateStrength,
							tempFatigueStrength,
							out_AnalysisMaterialProperties.bearingStrength );

		
		if ( !stressComputed )
		   stressComputed = StressValueComputed(	
							in_AnalysisMaterialProperties.tensileYieldStrengthDefined,
							in_AnalysisMaterialProperties.tensileYieldStrength,	
							in_AnalysisMaterialProperties.bearingYieldStrengthDefined,
							in_AnalysisMaterialProperties.bearingYieldStrength,
							tempFatigueStrength,
							out_AnalysisMaterialProperties.bearingStrength );

		if ( !stressComputed )
			out_AnalysisMaterialProperties.bearingStrength = tempFatigueStrength;

		// Shear
		stressComputed = StressValueComputed(	
							in_AnalysisMaterialProperties.tensileUltimateStrengthDefined,
							in_AnalysisMaterialProperties.tensileUltimateStrength,	
							true,
							tempShearStrength, 
							tempFatigueStrength,
							out_AnalysisMaterialProperties.shearStrength );

		if ( !stressComputed )
			stressComputed = StressValueComputed(	
							in_AnalysisMaterialProperties.tensileYieldStrengthDefined,
							in_AnalysisMaterialProperties.tensileYieldStrength,	
							true,
							tempShearStrength, 
							tempFatigueStrength,
							out_AnalysisMaterialProperties.shearStrength );
		if ( !stressComputed )
			out_AnalysisMaterialProperties.shearStrength =  .5 * in_AnalysisMaterialProperties.shearStrength;

	}

	///////////////////////////////////////////////
	//	At this point, the strength properties 
	//	should be set.  It would be a programming
	//	error if this was not the case. Throw
	//	and exception if the strength properties 
	//	are not set.
	///////////////////////////////////////////////

	if ( out_AnalysisMaterialProperties.tensileStrength	== 0 ||
		 out_AnalysisMaterialProperties.bearingStrength	== 0 ||
	     out_AnalysisMaterialProperties.shearStrength	== 0 )
	{
		std::ostringstream tensileStrength_str;
		std::ostringstream bearingStrength_str;
		std::ostringstream shearStrength_str;

		tensileStrength_str << out_AnalysisMaterialProperties.tensileStrength;
		bearingStrength_str << out_AnalysisMaterialProperties.bearingStrength;
		shearStrength_str   << out_AnalysisMaterialProperties.shearStrength;

		std::string TempError = "Function ComputeAllowableStressLevels(...) for material " + in_AnalysisMaterialProperties.materialName + 
		" failed to compute the all of the strength material properties.  Computed properties are: Tensile Strength: " +
		tensileStrength_str.str() + ", Bearing Strength: " + bearingStrength_str.str() + ", Shear Strength: " + shearStrength_str.str() + ".";   
		throw isis::application_exception(TempError.c_str());		
	}
}

enum e_UnitType
{
	UNIT_MPA,
	UNIT_GPA,
	UNIT_NA
};


e_UnitType RetrieveUnitType(	const std::string &in_MaterialName,
								const std::string &in_Units )
																	throw (isis::application_exception)
{															
	std::string units = isis::ConvertToUpperCase(in_Units);	

	if ( strcmp(units.c_str(), "MPA") == 0  )		
		return UNIT_MPA;
	else 
		if ( strcmp(units.c_str(), "GPA")  == 0  ) 
			return UNIT_GPA;
		else 
			if ( strcmp(units.c_str(), "N/A")  == 0  ) 
				return UNIT_NA;

	// No known type found.
	std::string TempError = "Function RetrieveUnitType(...) for material " + in_MaterialName + 
		" was passed: " + in_Units + 
		".  Acceptable types (case insensitive) are: MPA, GPA, and N/A.";   
	throw isis::application_exception(TempError.c_str());
}
////////////////////////////////////////////////////////////////////////////////////////////////////
// Description:
//		This function converts string representations of material properties
//		(i.e. in_MaterialMetrics_map[in_MaterialPropertyName] ) to doubles and stores 
//		the results in out_Property.
//		
//		Note -	All units stored in out_Property will be in MPa except for unit-less values
//				such as PoissonsRatio.
//		
// Pre-Conditions:
//		if n_MaterialPropertyName[in_MaterialPropertyName] exists
//			in_MaterialPropertyName[in_MaterialPropertyName].units must be MPa, GPa, or N/A
//
// Post-Conditions:
//		if n_MaterialPropertyName[in_MaterialPropertyName] not exists
//			return (no action taken)
//		else
//			if in_MaterialPropertyName[in_MaterialPropertyName].units != MPa, GPa, or N/A
//				throw isis::application_exception
//			else
//				out_Property =  in_MaterialPropertyName[in_MaterialPropertyName].value
//				out_PropertyDefnied = true
//				if in_MaterialPropertyName[in_MaterialPropertyName].units == GPa
//					scale out_Property to be in MPA
//				
void SetSingleAnalysisMaterialProperty(  
					const std::string						&in_MaterialPropertyName,
					const std::map<std::string, MaterialMetrics> &in_MaterialMetrics_map,
					double									&out_Property,
					bool									&out_PropertyDefnied) 
																throw (isis::application_exception)
{
	std::map<std::string, MaterialMetrics>::const_iterator itr = in_MaterialMetrics_map.find(in_MaterialPropertyName);

	if (itr != in_MaterialMetrics_map.end() )
	{
		// Note Poissoin's ration could be zero.  Since we will analyze metals, this should not be a problem.
		if ( itr->second.value == 0 )  
		{
			out_PropertyDefnied = false;
		}
		else
		{
			out_Property = itr->second.value;
			out_PropertyDefnied = true;
			if( RetrieveUnitType(in_MaterialPropertyName,itr->second.units ) == UNIT_GPA ) out_Property *= 1000;
		}
	}
}
////////////////////////////////////////////////////////////////////////////////////////////////////
// This funcdtion will convert integer values.  The only integer value is FatigueNumberOfCycles
// which has no units.
void SetSingleAnalysisMaterialProperty(  
					const std::string						&in_MaterialPropertyName,
					const std::map<std::string, MaterialMetrics> &in_MaterialMetrics_map,
					long									&out_Property,
					bool									&out_PropertyDefnied)
{
	std::map<std::string, MaterialMetrics>::const_iterator itr = in_MaterialMetrics_map.find(in_MaterialPropertyName);

	if (itr != in_MaterialMetrics_map.end() )
	{
		//out_Property = strtol(itr->second.value.c_str(), NULL,10 );
		out_Property = itr->second.value;

		out_PropertyDefnied = true;
	}
}
////////////////////////////////////////////////////////////////////////////////////////////////////
void PopulateAnalysisMaterialStruct( const Material				 &in_Material, 
									 AnalysisMaterialProperties  &out_AnalysisMaterialProperties_struct )
																		throw (isis::application_exception)
{
	out_AnalysisMaterialProperties_struct.materialType = in_Material.materialType;
	out_AnalysisMaterialProperties_struct.materialName = in_Material.materialName;

	std::map<std::string, MaterialMetrics> i;

	SetSingleAnalysisMaterialProperty(	"ModulusOfElasticity", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.modulusOfElasticity,
										out_AnalysisMaterialProperties_struct.modulusOfElasticityDefined);

	SetSingleAnalysisMaterialProperty(	"PoissonsRatio", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.poissonsRatio,
										out_AnalysisMaterialProperties_struct.poissonsRatioDefined);

	SetSingleAnalysisMaterialProperty(	"TensileYieldStrength", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.tensileYieldStrength,
										out_AnalysisMaterialProperties_struct.tensileYieldStrengthDefined);

	SetSingleAnalysisMaterialProperty(	"TensileUltimateStrength", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.tensileUltimateStrength,
										out_AnalysisMaterialProperties_struct.tensileUltimateStrengthDefined);

	SetSingleAnalysisMaterialProperty(	"ShearStrength", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.shearStrength,
										out_AnalysisMaterialProperties_struct.shearStrengthDefined);

	SetSingleAnalysisMaterialProperty(	"BearingYieldStrength", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.bearingYieldStrength,
										out_AnalysisMaterialProperties_struct.bearingYieldStrengthDefined);

	SetSingleAnalysisMaterialProperty(	"BearingUltimateStrength", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.bearingUltimateStrength,
										out_AnalysisMaterialProperties_struct.bearingUltimateStrengthDefined);

	SetSingleAnalysisMaterialProperty(	"FatigueStrength", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.fatigueStrength,
										out_AnalysisMaterialProperties_struct.fatigueStrengthDefined);

	SetSingleAnalysisMaterialProperty(	"FatigueNumberOfCycles", 
										in_Material.materialMetrics_map, 
										out_AnalysisMaterialProperties_struct.fatigueNumberOfCycles,
										out_AnalysisMaterialProperties_struct.fatigueNumberOfCyclesDefined);

} 


} // namespace isis