/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include <Metrics.h>
#include <CommonStructures.h>
#include <iostream>
#include <fstream>
#include <map>
#include <CADMetrics.h>
#include <AssembleUtils.h>
#include <iomanip>

#ifndef ISIS_VERSION_NUMBER_H
#define ISIS_VERSION_NUMBER_H
#include <ISISVersionNumber.h>
#endif

namespace isis
{


///////////////////////////////////////////////////////////////////////////////////////

void  ConvertCreoUnitToGMEUnit_Distance ( ProName in_DistanceUnit, std::string &out_ShortName, std::string &out_LongName  )
{
	char stringBuffer[PRO_NAME_SIZE];  // PRO_NAME_SIZE = 32

	std::string unit = ProWstringToString( stringBuffer, in_DistanceUnit );

	out_ShortName	= unit;
	out_LongName	= unit;

	if ( unit == "in" ) { out_ShortName = "inch";	out_LongName = "inch"; }
	if ( unit == "ft" ) { out_ShortName = "foot";	out_LongName = "foot"; }
	if ( unit == "mm" ) { out_ShortName = "mm";		out_LongName = "millimeter"; }
	if ( unit == "cm" ) { out_ShortName = "cm";		out_LongName = "centimeter"; }
	if ( unit == "m" )	{ out_ShortName = "m";		out_LongName = "meter"; }
	if ( unit == "km" )	{ out_ShortName = "km";		out_LongName = "kilometer"; }

}

///////////////////////////////////////////////////////////////////////////////////////
void  ConvertCreoUnitToGMEUnit_Mass ( ProName in_MassUnit,  std::string &out_ShortName, std::string &out_LongName  )
{
	char stringBuffer[PRO_NAME_SIZE];  // PRO_NAME_SIZE = 32

	std::string unit = ProWstringToString( stringBuffer, in_MassUnit );

	out_ShortName	= unit;
	out_LongName	= unit;

	if ( unit == "lbm" ){ out_ShortName = "lbm";		out_LongName = "poundmass"; }
	if ( unit == "g" )	{ out_ShortName = "g";			out_LongName = "gram"; }
	if ( unit == "kg" )	{ out_ShortName = "kg";			out_LongName = "kilogram"; }
	if ( unit == "tonne" )	{ out_ShortName = "tonne";  out_LongName = "tonne"; }

}
///////////////////////////////////////////////////////////////////////////////////////
void  ConvertCreoUnitToGMEUnit_Force ( ProName in_ForceUnit, std::string &out_ShortName, std::string &out_LongName  )
{
	char stringBuffer[PRO_NAME_SIZE];    // PRO_NAME_SIZE = 32

	std::string unit = ProWstringToString( stringBuffer, in_ForceUnit );

	out_ShortName	= unit;
	out_LongName	= unit;

	if ( unit == "lbf" ) { out_ShortName = "lbf";	out_LongName = "poundforce"; }
	if ( unit == "N" )	 { out_ShortName = "N";		out_LongName = "newton"; }

}
///////////////////////////////////////////////////////////////////////////////////////
void  ConvertCreoUnitToGMEUnit_Temperature ( ProName in_Temperature, std::string &out_ShortName, std::string &out_LongName  )
{
	char stringBuffer[PRO_NAME_SIZE];  // PRO_NAME_SIZE = 32

	std::string unit = ProWstringToString( stringBuffer, in_Temperature );

	out_ShortName	= unit;
	out_LongName	= unit;

	if ( unit == "C" )	{ out_ShortName = "C";	out_LongName = "centigrade"; }
	if ( unit == "F" )	{ out_ShortName = "F";	out_LongName = "fahrenheit"; }
	if ( unit == "K" )	{ out_ShortName = "K";	out_LongName = "kelvin"; }

}
///////////////////////////////////////////////////////////////////////////////////////
void ConvertCreoUnitToGMEUnit_Time ( ProName in_TimeUnit, std::string &out_ShortName, std::string &out_LongName  )
{
	char stringBuffer[PRO_NAME_SIZE];  // PRO_NAME_SIZE = 32

	std::string unit = ProWstringToString( stringBuffer, in_TimeUnit );

	out_ShortName	= unit;
	out_LongName	= unit;

	if ( unit == "sec" ) { out_ShortName = "sec";	out_LongName = "second"; }

}

///////////////////////////////////////////////////////////////////////////////////////////////////
void RetrieveUnits( ProMdl			in_Model,
					std::string		&out_DistanceUnit_ShortName,
					std::string		&out_DistanceUnit_LongName,
					
					std::string		&out_MassUnit_ShortName,
					std::string		&out_MassUnit_LongName,

					std::string		&out_ForceUnit_ShortName,
					std::string		&out_ForceUnit_LongName,

					std::string		&out_TimeUnit_ShortName,
					std::string		&out_TimeUnit_LongName,

					std::string		&out_TemperatureUnit_ShortName, 
					std::string		&out_TemperatureUnit_LongName )
											throw(isis::application_exception)
{
	std::string unitsString;

	ProUnitsystem unitSystem;
	//ProUnititem unit, forceUnit, timeUnit, lengthUnit;
	ProUnititem massUnit, forceUnit, timeUnit, lengthUnit, temperatureUint;
	ProLine massUnitsLabel;
	ProUnitsystemType type;

	isis::isis_ProMdlPrincipalunitsystemGet (in_Model, &unitSystem);

	//  PRO_UNITTYPE_LENGTH          L
	//  PRO_UNITTYPE_MASS            M
	//  PRO_UNITTYPE_FORCE           F
	//  PRO_UNITTYPE_TIME            T
	//	PRO_UNITTYPE_TEMPERATURE     D 

	
	isis::isis_ProUnitsystemUnitGet (&unitSystem, PRO_UNITTYPE_LENGTH, &lengthUnit); 
	ConvertCreoUnitToGMEUnit_Distance( lengthUnit.name,out_DistanceUnit_ShortName, out_DistanceUnit_LongName  );

	try 
	{
		isis::isis_ProUnitsystemUnitGet (&unitSystem, PRO_UNITTYPE_MASS, &massUnit); 
		ConvertCreoUnitToGMEUnit_Mass(  massUnit.name, out_MassUnit_ShortName, out_MassUnit_LongName );
	}
	catch(...)
	{
		out_MassUnit_LongName = "Derived";
	}
	
	try 
	{
	isis::isis_ProUnitsystemUnitGet (&unitSystem, PRO_UNITTYPE_FORCE, &forceUnit); 
	ConvertCreoUnitToGMEUnit_Force( forceUnit.name, out_ForceUnit_ShortName, out_ForceUnit_LongName );
	}
	catch(...)
	{
		out_ForceUnit_ShortName = "Derived";
		out_ForceUnit_LongName = "Derived";
	}
	
	isis::isis_ProUnitsystemUnitGet (&unitSystem, PRO_UNITTYPE_TEMPERATURE , &timeUnit); 
	ConvertCreoUnitToGMEUnit_Temperature( timeUnit.name, out_TemperatureUnit_ShortName, out_TemperatureUnit_LongName );
	
	isis::isis_ProUnitsystemUnitGet (&unitSystem, PRO_UNITTYPE_TIME, &temperatureUint);  
	ConvertCreoUnitToGMEUnit_Time( temperatureUint.name, out_TimeUnit_ShortName, out_TimeUnit_LongName );

}
///////////////////////////////////////////////////////////////////////////////////////
void RetrieveMaterial(	const std::string &in_ModelName, 
						ProSolid part, 
						std::string &out_MaterialName ) throw(isis::application_exception)
{

	ProMaterial  material;

	try
	{
		isis::isis_ProMaterialCurrentGet( part, &material );

		char stringBuffer[PRO_NAME_SIZE];  // PRO_NAME_SIZE = 32
		out_MaterialName = ProWstringToString( stringBuffer, material.matl_name );	
	}
	catch (...)
	{
		// out_MaterialName = "NOT_DEFINED";
		std::string TempError = 
					"Material not defined for part: " + in_ModelName + 
					".  To compute mass properties, all parts must have a material assignment. "  +
					" Please open the model with Creo and select File Prepare \"Model Properties\"" +
					" to assign a material to the model.";
					throw isis::application_exception(TempError.c_str());		
	}
}
///////////////////////////////////////////////////////////////////////////////////////////////////
void PopulateMatrix(	int					in_NumRows, 
						int					in_NumColumns, 
						const double		in_MatrixBuffer[4][4],  
						CADMetrics::Rows	&out_RowsRoot)
{
	for (int i = 0; i < in_NumRows; ++i )
	{
		CADMetrics::Row		   rowRoot = CADMetrics::Row::Create(out_RowsRoot);

		for (int j = 0; j < in_NumColumns; ++j )
		{
			CADMetrics::Column	columnRoot = CADMetrics::Column::Create(rowRoot);
			columnRoot.Value() = in_MatrixBuffer[i][j];
		}
	}
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////
void PopulateScalar( const std::string		in_Name,
					 double					in_Value,
					 const std::string		in_Unit,
					 CADMetrics::Scalars	&out_ScalarsRoot )
{
	CADMetrics::Scalar   scalarRoot = CADMetrics::Scalar::Create( out_ScalarsRoot );
	scalarRoot.Name()  = in_Name;
	scalarRoot.Value() = in_Value;
	scalarRoot.Unit()  = in_Unit;
}

struct MetricsDefined
{
	bool massProp_Defined;
	bool interiaAtDefaultCSys_Defined;
	bool interiaAtCOG_Defined;
	bool surfaceArea_Defined;

	MetricsDefined(): massProp_Defined(false), 
						interiaAtDefaultCSys_Defined(false),
						interiaAtCOG_Defined(false),
						surfaceArea_Defined(false) {}
	bool anomalyOccurred() 
	{ 
		return (!massProp_Defined || !interiaAtDefaultCSys_Defined || !interiaAtCOG_Defined || !surfaceArea_Defined);
	}
	
							 
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////
void Populate_Single_MetricComponent( 
			const std::string													&in_ComponentID,
			std::map<std::string, isis::CADComponentData>						&in_CADComponentData_map,  
			std::set<int>														&in_out_DataCreated_for_MetricIDs,
			std::map<int, MetricsDefined>										&out_MetricID_to_Anomalies_map,
			CADMetrics::MetricComponents										&out_metricComponentsRoot )
																						throw (isis::application_exception)
{
	// zzzz should throw an exception if MetricId has not been created.  This should never happen
	// Check if data for the MetricID has NOT been created.
	if ( in_out_DataCreated_for_MetricIDs.find( in_CADComponentData_map[in_ComponentID].metricID ) == in_out_DataCreated_for_MetricIDs.end() )
	{
		////////////////////////////
		// Create Metric Component
		////////////////////////////
		CADMetrics::MetricComponent  metricComponentRoot = 	CADMetrics::MetricComponent::Create( out_metricComponentsRoot );

		////////////////////////
		// Add Attributes
		////////////////////////
		metricComponentRoot.CoordinateSystem() = std::string("DEFAULT");
		metricComponentRoot.MetricID() = in_CADComponentData_map[in_ComponentID].metricID;
		metricComponentRoot.Name() = std::string(in_CADComponentData_map[in_ComponentID].name);
		metricComponentRoot.SpecialInstruction() = isis::SpecialInstruction_string(in_CADComponentData_map[in_ComponentID].specialInstruction);

		// Model Type
		if ( in_CADComponentData_map[in_ComponentID].modelType == PRO_MDL_PART )
			metricComponentRoot.Type() = "PART";
		else
			metricComponentRoot.Type() = "ASSEMBLY";

		///////////////////////////////
		// Get the Creo Mass Properties
		///////////////////////////////
		ProMassProperty  mass_prop;
		isis::isis_ProSolidMassPropertyGet( in_CADComponentData_map[in_ComponentID].modelHandle, NULL, &mass_prop );
		double   MatrixBuffer[4][4];

		/////////////////////////////////////////
		// Check if Mass Properties are Defined
		////////////////////////////////////////
		MetricsDefined MetricsDefined;

		// if mass_prop.density == 1.0, then mass properties were never set in Creo.  The never-set mode
		// means that the geometry and density of 1.0 would be used to compute the mass properties; however,
		// those computed values would be erroneous.
		// ERROR - ERROR Leave off the mass_prop.density != 1.0 for now.  This will allow erroneous mass props through, must
		// provide a better check later.
		//if ( mass_prop.volume != 0.0 && mass_prop.density != 0.0 && mass_prop.density != 1.0 && mass_prop.mass != 0.0 ) 
		if ( mass_prop.volume != 0.0 && mass_prop.density != 0.0 && mass_prop.mass != 0.0 ) 
			MetricsDefined.massProp_Defined = true;


		for ( int i = 0; i < 3; ++i )
			for ( int j = 0; j < 3; ++j )
				if (mass_prop.coor_sys_inertia_tensor[i][j] != 0) 
				{ 
					MetricsDefined.interiaAtDefaultCSys_Defined = true;
					break;
				}

		for ( int i = 0; i < 3; ++i )
			for ( int j = 0; j < 3; ++j )
				if (mass_prop.cg_inertia_tensor[i][j] != 0) 
				{ 
					MetricsDefined.interiaAtCOG_Defined = true;
					break;
				}

		if ( mass_prop.surface_area != 0 ) MetricsDefined.surfaceArea_Defined = true;

		if ( MetricsDefined.anomalyOccurred() )
			out_MetricID_to_Anomalies_map[in_CADComponentData_map[in_ComponentID].metricID] = MetricsDefined;

		////////////////////////
		// CG
		////////////////////////
		
		CADMetrics::CG  cGRoot = CADMetrics::CG::Create( metricComponentRoot );

		cGRoot.X() = mass_prop.center_of_gravity[0];
		cGRoot.Y() = mass_prop.center_of_gravity[1];
		cGRoot.Z() = mass_prop.center_of_gravity[2];

		//std::cout << std::endl <<  "mass_prop.center_of_gravity[0]"  <<  mass_prop.center_of_gravity[0];
		//std::cout << std::endl <<  "mass_prop.center_of_gravity[1]"  <<  mass_prop.center_of_gravity[1];
		//std::cout << std::endl <<  "mass_prop.center_of_gravity[2]"  <<  mass_prop.center_of_gravity[2];

		////////////////////////
		// Bounding Box
		////////////////////////
		Pro3dPnt  r_outline_points[2];
		isis::isis_ProSolidOutlineGet( in_CADComponentData_map[in_ComponentID].modelHandle, r_outline_points);

		CADMetrics::BoundingBox boundingBoxRoot = CADMetrics::BoundingBox::Create(metricComponentRoot);

		//if (in_CADComponentData_map[in_ComponentID].cADComputations.boundingBoxMetricDefined);
		//{
		//	boundingBoxRoot.MetricID_Height() = in_CADComponentData_map[in_ComponentID].cADComputations.heightMetricID;
		//	boundingBoxRoot.MetricID_Length() = in_CADComponentData_map[in_ComponentID].cADComputations.lengthMetricID;
		//	boundingBoxRoot.MetricID_Width()  = in_CADComponentData_map[in_ComponentID].cADComputations.widthMetricID;
		//}

		double tempLength =  abs(r_outline_points[1][2] - r_outline_points[0][2]);  // Length Z direction
		double tempHeight =  abs(r_outline_points[1][1] - r_outline_points[0][1]);  // Height y direction
		double tempWidth  =  abs(r_outline_points[1][0] - r_outline_points[0][0]);  // Width  x direction

		boundingBoxRoot.X() = tempWidth;
		boundingBoxRoot.Y() = tempHeight;
		boundingBoxRoot.Z()  = tempLength;

		CADMetrics::OutlinePoints rootOutlinePoints = CADMetrics::OutlinePoints::Create(boundingBoxRoot);

		CADMetrics::Point  rootPoint_1 = CADMetrics::Point::Create(rootOutlinePoints);
		CADMetrics::Point  rootPoint_2 = CADMetrics::Point::Create(rootOutlinePoints);
		rootPoint_1.X() = r_outline_points[0][0];
		rootPoint_1.Y() = r_outline_points[0][1];
		rootPoint_1.Z() = r_outline_points[0][2];

		rootPoint_2.X() = r_outline_points[1][0];
		rootPoint_2.Y() = r_outline_points[1][1];
		rootPoint_2.Z() = r_outline_points[1][2];


		///////////////////////////////////////////////
		// Interia Tensor at Default Coordinate System
		//////////////////////////////////////////////
		if ( MetricsDefined.massProp_Defined && MetricsDefined.interiaAtDefaultCSys_Defined )
		{
			for ( int i = 0; i < 3; ++i )
				for ( int j = 0; j < 3; ++j )
					MatrixBuffer[i][j] = mass_prop.coor_sys_inertia_tensor[i][j];
		

			CADMetrics::InertiaTensor  inertiaDefaultCSysTensorRoot = CADMetrics::InertiaTensor::Create( metricComponentRoot );
			inertiaDefaultCSysTensorRoot.At() = isis::CADInertiaAt_string(CAD_DEFAULT_CSYS);
			CADMetrics::Rows		   rowsInertiaDefaultCSysTensorRoot = CADMetrics::Rows::Create(inertiaDefaultCSysTensorRoot);

			PopulateMatrix(	3, 3, MatrixBuffer, rowsInertiaDefaultCSysTensorRoot  );
		}
		///////////////////////////////////////////////
		// Interia Tensor at Center of Gravity
		//////////////////////////////////////////////
		if (  MetricsDefined.massProp_Defined && MetricsDefined.interiaAtCOG_Defined )
		{
			for ( int i = 0; i < 3; ++i )
				for ( int j = 0; j < 3; ++j )
					MatrixBuffer[i][j] = mass_prop.cg_inertia_tensor[i][j];
		
			CADMetrics::InertiaTensor  inertiaCOGTensorRoot = CADMetrics::InertiaTensor::Create( metricComponentRoot );
			inertiaCOGTensorRoot.At() = isis::CADInertiaAt_string(CAD_CENTER_OF_GRAVITY);
			CADMetrics::Rows		   rowsInertiaCOGTensorRoot = CADMetrics::Rows::Create(inertiaCOGTensorRoot);

			PopulateMatrix(	3, 3, MatrixBuffer, rowsInertiaCOGTensorRoot  );
		}
		//////////////////////////////////
		// Principal Moments of Intertia
		//////////////////////////////////

		if (  MetricsDefined.massProp_Defined )
		{

			// Rotation Matrix
			CADMetrics::PrincipleMomentsOfInertia   principleMomentsOfInertiaRoot = CADMetrics::PrincipleMomentsOfInertia::Create( metricComponentRoot );
			CADMetrics::RotationMatrix			    rotationMatrixRoot = CADMetrics::RotationMatrix::Create( principleMomentsOfInertiaRoot );
			CADMetrics::Rows						rowsRotationMatrixRoot = CADMetrics::Rows::Create(rotationMatrixRoot);

			for ( int i = 0; i < 3; ++i )
				for ( int j = 0; j < 3; ++j )
					MatrixBuffer[i][j] =  mass_prop.principal_axes[i][j];
			PopulateMatrix(	3, 3, MatrixBuffer, rowsRotationMatrixRoot  );
	
			// Principal Moments
			CADMetrics::Rows	rowsPrincipleMomentsOfInertiaValues = CADMetrics::Rows::Create(principleMomentsOfInertiaRoot);

			for ( int i = 0; i < 3; ++i ) MatrixBuffer[i][0] =  mass_prop.principal_moments[i];

			PopulateMatrix(	3, 1,MatrixBuffer, rowsPrincipleMomentsOfInertiaValues  );
		}	

		////////////////////////
		// Units
		////////////////////////

		std::string		distanceUnit_LongName;
		std::string		distanceUnit_ShortName;
		
		std::string		massUnit_LongName;
		std::string		massUnit_ShortName;

		std::string		forceUnit_LongName;
		std::string		forceUnit_ShortName;

		std::string		temperatureUnit_LongName;
		std::string		temperatureUnit_ShortName;
		
		std::string		timeUnit_LongName;
		std::string		timeUnit_ShortName;
		
		//std::cout << std::endl << "=====> Model Units, Model: " << in_CADComponentData_map[in_ComponentID].name;
		
		RetrieveUnits( in_CADComponentData_map[in_ComponentID].modelHandle, 
					   distanceUnit_ShortName,		distanceUnit_LongName, 
					   massUnit_ShortName,			massUnit_LongName, 
					   forceUnit_ShortName,			forceUnit_LongName, 
					   timeUnit_ShortName,			timeUnit_LongName, 
					   temperatureUnit_ShortName,	temperatureUnit_LongName );
		
		std::string forceUnit_temp;
		if ( forceUnit_LongName == "Derived" )
			forceUnit_temp = massUnit_ShortName + " " + distanceUnit_ShortName + "/" + timeUnit_ShortName + "2";
		else
			forceUnit_temp = forceUnit_ShortName;


		CADMetrics::Units  unitsRoot = CADMetrics::Units::Create( metricComponentRoot );

		unitsRoot.Distance()	= distanceUnit_LongName;
		unitsRoot.Force()		= forceUnit_temp;
		unitsRoot.Mass()		= massUnit_LongName;
		unitsRoot.Temperature() = temperatureUnit_LongName;
		unitsRoot.Time()		= timeUnit_LongName;

		////////////////////////
		// Scalars
		////////////////////////
		if ( MetricsDefined.massProp_Defined || MetricsDefined.surfaceArea_Defined )
		{
			CADMetrics::Scalars  scalarsRoot = CADMetrics::Scalars::Create( metricComponentRoot );
			//   ScalarType (const Value_type&,
			//        const Name_type&,
			//        const Unit_type&);

			// Surface Area
			if ( MetricsDefined.surfaceArea_Defined )
				PopulateScalar( "SurfaceArea", mass_prop.surface_area, distanceUnit_ShortName + "2", scalarsRoot );

			if (MetricsDefined.massProp_Defined)
			{
				// Volume
				PopulateScalar( "Volume", mass_prop.volume, distanceUnit_ShortName + "3", scalarsRoot );

				// Density
				// Note - Only parts have a density.  Assemblies do not have densities.
				if ( in_CADComponentData_map[in_ComponentID].modelType == PRO_MDL_PART )
				{
					PopulateScalar( "Density", mass_prop.density, massUnit_ShortName + "/" + distanceUnit_ShortName + "3", scalarsRoot );
				}

				// Mass
				PopulateScalar( "Mass", mass_prop.mass, massUnit_ShortName, scalarsRoot );
			}
		}

		//////////////////////////////////
		// Material
		//////////////////////////////////
		
		if ( in_CADComponentData_map[in_ComponentID].modelType == PRO_MDL_PART )
		{
			std::string material_temp = "Undefined";
			try
			{ // zzzz Material need to check for 
				RetrieveMaterial(	in_CADComponentData_map[in_ComponentID].name,
									in_CADComponentData_map[in_ComponentID].modelHandle, material_temp );
			}
			catch (...)
			{
			}

			CADMetrics::Material  materialRoot = CADMetrics::Material::Create( metricComponentRoot );
			materialRoot.Type() = material_temp;
		}

		////////////////////////
		// Children
		////////////////////////
		if ( in_CADComponentData_map[in_ComponentID].children.size() > 0 )
		{

			CADMetrics::Children childrenRoot = CADMetrics::Children::Create( metricComponentRoot );

			for ( std::list<std::string>::const_iterator i = in_CADComponentData_map[in_ComponentID].children.begin();
				  i != in_CADComponentData_map[in_ComponentID].children.end();
				  ++i )
			{

				// The last item in componentPaths is the path from the parent assembly to the child.  The other
				// entries in componentPaths are paths from higher up assemblies.  
				std::list<int>  assemblyToChild_compentPath;
				assemblyToChild_compentPath.push_back( in_CADComponentData_map[*i].componentPaths.back());

				RetrieveTranformationMatrix_Assembly_to_Child ( in_ComponentID,  
																*i,
																assemblyToChild_compentPath,
																in_CADComponentData_map,  
																PRO_B_FALSE,  // bottom up = False
																MatrixBuffer );
				// At this point MatrixBuffer would contain
				// Rotation		3 X 3,  rows 0 thru 2
				// Translation	1 X 3,  row 3

				// Rotation Matrix
				CADMetrics::ChildMetric		 childMetricRoot = CADMetrics::ChildMetric::Create( childrenRoot );
				childMetricRoot.MetricID() = in_CADComponentData_map[*i].metricID;
				CADMetrics::RotationMatrix	 childRotationMatrixRoot = CADMetrics::RotationMatrix::Create( childMetricRoot );
				CADMetrics::Rows			 rowsChildRotationMatrixRoot = CADMetrics::Rows::Create( childRotationMatrixRoot );

				//for ( int i = 0; i < 3; ++i )
				//	for ( int j = 0; j < 3; ++j )
				//		MatrixBuffer[i][j] =  888;
				PopulateMatrix(	3, 3, MatrixBuffer, rowsChildRotationMatrixRoot  );

				CADMetrics::Translation	 childTranslationRoot = CADMetrics::Translation::Create( childMetricRoot );
				childTranslationRoot.X() = MatrixBuffer[3][0];
				childTranslationRoot.Y() = MatrixBuffer[3][1];
				childTranslationRoot.Z() = MatrixBuffer[3][2];

			}

		}
	
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
void Populate_MetricComponents( 
			const std::string																&in_ComponentID,
			std::map<std::string, isis::CADComponentData>									&in_CADComponentData_map,  
			std::set<int>																	&in_out_DataCreated_for_MetricIDs,
			std::map<int, MetricsDefined>													&out_MetricID_to_Anomalies_map,
			CADMetrics::MetricComponents													&out_metricComponentsRoot )
																						throw (isis::application_exception)
{

	Populate_Single_MetricComponent(	in_ComponentID, 
										in_CADComponentData_map,  
										in_out_DataCreated_for_MetricIDs,
										out_MetricID_to_Anomalies_map,
										out_metricComponentsRoot );
	

	if ( in_CADComponentData_map[in_ComponentID].modelType == PRO_MDL_ASSEMBLY )
	{
		for ( std::list<string>::const_iterator i(in_CADComponentData_map[in_ComponentID].children.begin());
			i != in_CADComponentData_map[in_ComponentID].children.end();
			++i )
		{

			Populate_MetricComponents( 	*i,
										in_CADComponentData_map,  
										in_out_DataCreated_for_MetricIDs,
										out_MetricID_to_Anomalies_map,
										out_metricComponentsRoot );
		}
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
void Populate_Assemblies( 
			const std::string													&in_ComponentID,
			std::map<std::string, isis::CADComponentData>						&in_CADComponentData_map,  
			CADMetrics::CADComponent											&out_CADComponent )
																					throw (isis::application_exception)
{
	if ( in_CADComponentData_map[in_ComponentID].modelType == PRO_MDL_ASSEMBLY )
	{
		for ( std::list<string>::const_iterator i(in_CADComponentData_map[in_ComponentID].children.begin());
			i != in_CADComponentData_map[in_ComponentID].children.end();
			++i )
		{

			CADMetrics::CADComponent	cADComponentRoot = CADMetrics::CADComponent::Create(out_CADComponent);

			cADComponentRoot.ComponentID() = *i;
			cADComponentRoot.MetricID() = in_CADComponentData_map[*i].metricID;

			if ( in_CADComponentData_map[*i].modelType == PRO_MDL_ASSEMBLY )
			{
				Populate_Assemblies(	*i,
										in_CADComponentData_map,  
										out_CADComponent );
			}
		}
	}
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////
void Assign_MetricIDs_To_Components( 
			const std::string									&in_ComponentID,
			std::map<std::string, isis::CADComponentData>		&in_out_CADComponentData_map,  
			int													&in_out_NextMetricID,
			std::map< std::string, int>							&in_out_ModelName_MetricID_map,
			std::map<int, std::string>							&in_out_MetricID_ModelName_map )
																						throw (isis::application_exception)
{

	std::string ModelNameWithSuffix = AmalgamateModelNameWithSuffix( in_out_CADComponentData_map[in_ComponentID].name, 
																	 in_out_CADComponentData_map[in_ComponentID].modelType );

	if ( ( in_out_CADComponentData_map[in_ComponentID].parametricParametersPresent )  || 
		 ( in_out_CADComponentData_map[in_ComponentID].specialInstruction == CAD_SPECIAL_INSTRUCTION_SIZE_TO_FIT ) ||
		 ( in_out_ModelName_MetricID_map.find( ModelNameWithSuffix ) == in_out_ModelName_MetricID_map.end() ) )
	{
		in_out_CADComponentData_map[in_ComponentID].metricID = in_out_NextMetricID;

		// Don't add parametric and SIZE_TO_FIT models to in_out_ModelName_MetricID.  This will force
		// parametric and SIZE_TO_FIT models to always have a unique MetricID.
		if ( ( !in_out_CADComponentData_map[in_ComponentID].parametricParametersPresent )  && 
		 ( in_out_CADComponentData_map[in_ComponentID].specialInstruction != CAD_SPECIAL_INSTRUCTION_SIZE_TO_FIT ))
		{
			in_out_ModelName_MetricID_map[ModelNameWithSuffix] = in_out_NextMetricID;
		}
		in_out_MetricID_ModelName_map[in_out_NextMetricID] = ModelNameWithSuffix;
		++in_out_NextMetricID;
	}
	else
	{
		in_out_CADComponentData_map[in_ComponentID].metricID = in_out_ModelName_MetricID_map[ ModelNameWithSuffix ];
	}
	
	if ( in_out_CADComponentData_map[in_ComponentID].modelType == PRO_MDL_ASSEMBLY )
	{
		for ( std::list<string>::const_iterator i(in_out_CADComponentData_map[in_ComponentID].children.begin());
			i != in_out_CADComponentData_map[in_ComponentID].children.end();
			++i )
		{
			Assign_MetricIDs_To_Components( *i,
											in_out_CADComponentData_map,  
											in_out_NextMetricID,
											in_out_ModelName_MetricID_map,
											in_out_MetricID_ModelName_map);
		}
	}
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void MetricsNotDefinedErrors( const MetricsDefined &in_MetricsDefined,
							  std::list<std::string> &out_ErrorStrings )
{
	if ( !in_MetricsDefined.massProp_Defined ) 
	{
		out_ErrorStrings.push_back( CADMetricsError_string(CAD_MASS_PROPERTIES_NOT_DEFINED));
	}
	else
	{
		if ( !in_MetricsDefined.interiaAtDefaultCSys_Defined ) 
			out_ErrorStrings.push_back( CADMetricsError_string(CAD_INTERIA_AT_DEFAULT_CSYS_NOT_DEFINED));
		if ( !in_MetricsDefined.interiaAtCOG_Defined ) 
			out_ErrorStrings.push_back( CADMetricsError_string(CAD_INTERIA_AT_CENTER_OF_GRAVITY_NOT_DEFINED));

	}

	if ( !in_MetricsDefined.surfaceArea_Defined )
		out_ErrorStrings.push_back( CADMetricsError_string(CAD_SURFACE_AREA_NOT_DEFINED));


	if (out_ErrorStrings.size() == 0 )
		out_ErrorStrings.push_back("NONE");

}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Populate_Anomalies(	std::map<int, MetricsDefined>	&in_MetricID_to_Anomalies_map, 
							CADMetrics::Anomalies			&out_AanomaliesRoot)
{
	for each ( std::pair<int, MetricsDefined>  i in in_MetricID_to_Anomalies_map )
	{
		char metricID_buffer[64];
		itoa(i.first, metricID_buffer,10);

		std::list<std::string> errorStrings;

		MetricsNotDefinedErrors(i.second, errorStrings);

		for each (std::string j in errorStrings )
		{
			CADMetrics::Anomaly	anomalyRoot = CADMetrics::Anomaly::Create(out_AanomaliesRoot);
			anomalyRoot.MetricID() = metricID_buffer;
			anomalyRoot.Error() = j;
		}
	}
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Log_Anomalies(	const std::map<int, MetricsDefined>	&in_MetricID_to_Anomalies_map, 
					const std::string					&in_MeticsOutputXML_PathAndFileName,
						  std::map<int, std::string>	&in_MetricID_ModelName_map)
{
	
	if (in_MetricID_to_Anomalies_map.size() > 0 ) std::clog << std::endl;

	for each ( std::pair<int, MetricsDefined>  i in in_MetricID_to_Anomalies_map )
	{
		std::list<std::string> errorStrings;

		MetricsNotDefinedErrors(i.second, errorStrings);

		for each (std::string j in errorStrings ) 
			std::clog << std::endl << METRICS_FILE_ERROR_string << ", Metric ID: " << i.first << "  Model Name: " <<  setiosflags(ios::left) << std::setw(35) <<in_MetricID_ModelName_map[i.first]  << " Error: " << j;
	}
	if (in_MetricID_to_Anomalies_map.size() > 0 ) 
		std::clog << std::endl << "For a mapping of Metric ID to Component ID, see " << in_MeticsOutputXML_PathAndFileName;

}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void OutputCADMetricsToXML( const isis::CADAssemblies						&in_CADAssemblies,
							std::map<std::string, isis::CADComponentData>	&in_CADComponentData_map,  
							const std::string								&in_MeticsOutputXML_PathAndFileName, 
							bool											&out_ErrorOccurred )
							throw (isis::application_exception, std::exception)
{
	////////////////////////
	// Setup UDM out file
	////////////////////////
	Udm::SmartDataNetwork dn_CADMetricRoot(CADMetrics::diagram);
	dn_CADMetricRoot.CreateNew(in_MeticsOutputXML_PathAndFileName, "CADMetrics", CADMetrics::CADMetricRoot::meta);

	try
	{
		int nextMetricID = 1;
		std::map< std::string, int > modelName_MetricID_map;
		std::map<int, std::string>	 metricID_ModelName_map; 
		std::set<int>				 dataCreated_for_MetricIDs;
		//std::set<int>				 metricIDs_ChildrenDefined;
	
		//////////////////////////////////////////////////////////////////////////////////
		// Assign MetricIDs to CADComponents (i.e. contained in in_CADComponentData_map )
		///////////////////////////////////////////////////////////////////////////////////
		for ( std::list<TopLevelAssemblyData>::const_iterator i(in_CADAssemblies.topLevelAssemblies.begin());
				i != in_CADAssemblies.topLevelAssemblies.end();
				++i )
		{	
			Assign_MetricIDs_To_Components( i->assemblyComponentID, 
											in_CADComponentData_map,  
											nextMetricID, 
											modelName_MetricID_map,
											metricID_ModelName_map);
		}

		//for ( std::map< std::string, int >::const_iterator temp_i(ModelName_MetricID.begin());
		//	  temp_i != ModelName_MetricID.end(); ++temp_i )
		//{
		//	std::cout << std::endl << "ModelName_MetricID, Model Name: " << temp_i->first << " MetricID: " << temp_i->second;
		//}

		//////////////////////////////
		// Cast to root of UDM object
		//////////////////////////////
		CADMetrics::CADMetricRoot cADMetricRoot = CADMetrics::CADMetricRoot::Cast(dn_CADMetricRoot.GetRootObject());
		cADMetricRoot.VersionInfo() = ISIS_METRIC_FILE_VERSION;

		//  Anomolies
		CADMetrics::Anomalies  anomaliesRoot = CADMetrics::Anomalies::Create( cADMetricRoot );

		//  Assemblies
		CADMetrics::Assemblies  assembliesRoot = CADMetrics::Assemblies::Create( cADMetricRoot );

		// MetricComponents
		CADMetrics::MetricComponents metricComponentsRoot = CADMetrics::MetricComponents::Create( cADMetricRoot );


		std::map<int, MetricsDefined> metricID_to_Anomalies_map;

		////////////////////////
		// Populate XML Tree
		////////////////////////
		for ( std::list<TopLevelAssemblyData>::const_iterator i(in_CADAssemblies.topLevelAssemblies.begin());
				i != in_CADAssemblies.topLevelAssemblies.end();
				++i )
		{	
			//CADMetrics::MetricComponent  metricComponentRoot = 	CADMetrics::MetricComponent::Create( metricComponentsRoot );
			Populate_MetricComponents(	i->assemblyComponentID, 
										in_CADComponentData_map, 								
										dataCreated_for_MetricIDs,
										metricID_to_Anomalies_map,
										metricComponentsRoot );	

			CADMetrics::Assembly    assemblyRoot = 	CADMetrics::Assembly::Create( assembliesRoot );
			assemblyRoot.ConfigurationID() = i->configurationID;
			CADMetrics::CADComponent    cADComponentRoot = 	CADMetrics::CADComponent::Create( assemblyRoot );
			cADComponentRoot.ComponentID() = i->assemblyComponentID;
			cADComponentRoot.MetricID() = in_CADComponentData_map[i->assemblyComponentID].metricID;

			Populate_Assemblies(	i->assemblyComponentID,
									in_CADComponentData_map,  
									cADComponentRoot );
		}

		Populate_Anomalies( metricID_to_Anomalies_map, anomaliesRoot );
		Log_Anomalies( metricID_to_Anomalies_map, in_MeticsOutputXML_PathAndFileName, metricID_ModelName_map );

		if ( metricID_to_Anomalies_map.size() == 0 )
			out_ErrorOccurred = false;
		else
			out_ErrorOccurred = true;

		///////////////////////////
		// Write XML File
		//////////////////////////
		dn_CADMetricRoot.CloseWithUpdate();
	}
	catch ( ... )
	{
		// If any error occurs, do not save the metrics xml file.  It makes no sense to save a metrics file
		// with erroneous data.
		dn_CADMetricRoot.CloseNoUpdate();
		throw;
	}

}
	

} // end namespace isis