/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include <ModelMaterials.h>
#include <isis_application_exception.h>
#include <isis_include_ptc_headers.h>
#include <isis_ptc_toolkit_functions.h>
#include <CommonUtilities.h>
#include <stdlib.h>
#include <iostream>

namespace isis
{

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// If a current material is not set in the Creo model, then the functions in this file will not work; and 
	// therefore, throw an exception if the current material is not set.
	void TestRetrievingCurrentMaterial (	const std::string							&in_AssemblyComponentID, 
											std::map<string, isis::CADComponentData>	&in_CADComponentData_map) 
													throw (isis::application_exception)
	{
		try
		{
			// Get current material
			ProMaterial currentMaterial;
			isis::isis_ProMaterialCurrentGet( in_CADComponentData_map[in_AssemblyComponentID].modelHandle, &currentMaterial );
		}
		catch (isis::application_exception &e )
		{
			std::string TempError = "Function TestRetrievingCurrentMaterial(...) failed to retrieve a current(i.e. default) material for model: " + 
				in_CADComponentData_map[in_AssemblyComponentID].name +
				", Component ID: " + in_AssemblyComponentID + ", Creo Exception: " + e.what() ;
			throw isis::application_exception(TempError.c_str());
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// If the current material does not have the properties needed by FEA, then some of functions in this file 
	// will not work; and therefore, throw an exception.
	void TestIfCurrentMaterialHasNeededFEAProperties (	const std::string							&in_AssemblyComponentID, 
														std::map<string, isis::CADComponentData>	&in_CADComponentData_map)
														throw (isis::application_exception)
	{
		std::string materialPropertyType;
		try
		{
			// Get current material
			ProMaterial currentMaterial;
			isis::isis_ProMaterialCurrentGet( in_CADComponentData_map[in_AssemblyComponentID].modelHandle, &currentMaterial );	
			ProModelitem  currentMaterialModelItem;
		
			isis_ProModelitemByNameInit ( in_CADComponentData_map[in_AssemblyComponentID].modelHandle, PRO_RP_MATERIAL, currentMaterial.matl_name, &currentMaterialModelItem );

			ProParamvalue materialValue;
			ProUnititem   materialUnit;

			// Density
			materialPropertyType = "Density";
			isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem,		  PRO_MATPROP_MASS_DENSITY, &materialValue, &materialUnit );

			// Poisson' Ratio
			materialPropertyType = "Poisson' Ratio";
			isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem,		  PRO_MATPROP_POISSON_RATIO, &materialValue, &materialUnit );

			// Young's Modulus
			materialPropertyType = "Young's Modulus";
			isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem,		  PRO_MATPROP_YOUNG_MODULUS, &materialValue, &materialUnit );
		
			// Coeef. of Thermal Expansion
			materialPropertyType = "Coefficient of Thermal Expansion";
			isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem,		  PRO_MATPROP_THERMAL_EXPANSION_COEFFICIENT, &materialValue, &materialUnit );

			// Stress-Strain Response e.g. Linear, Hyperelastic, Elastroplastic
			// Couldn't find the enum entry
		
			// Specific Heat Capicity
			materialPropertyType = "Specific Heat Capicity";
			isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem, PRO_MATPROP_SPECIFIC_HEAT, &materialValue, &materialUnit );

			// Thermal Conductivity
			materialPropertyType = "Thermal Conductivity";
			isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem, PRO_MATPROP_THERMAL_CONDUCTIVITY, &materialValue, &materialUnit );

			// Symmetry e.g. isotropic
			// Could not find the enum entry


		}
		catch (isis::application_exception &e )
		{
			std::string TempError = "Function: TestIfCurrentMaterialHasNeededFEAProperties(...). For the material property assigned to the Creo model, could not retrieve " + materialPropertyType +
				" material property. Model name: " + in_CADComponentData_map[in_AssemblyComponentID].name +
				", Component ID: " + in_AssemblyComponentID + ", Creo Exception: " + e.what() ;
			throw isis::application_exception(TempError.c_str());
		}

	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Return true if in_MaterialName material exists in in_Part
	// in_MaterialName length must be < 32 chars
	bool  MaterialExists( ProSolid in_Part, const std::string &in_MaterialName ) 
													throw (isis::application_exception)
	{
		std::string MaterialName = isis::ConvertToUpperCase(in_MaterialName);

		ProName * p_name;
		isis::isis_ProArrayAlloc(0, sizeof(ProName), 1, (ProArray*) &p_name);
		isis::isis_ProPartMaterialsGet(in_Part, &p_name);

		int numLines;

		isis::isis_ProArraySizeGet((ProArray)p_name, &numLines);

		char stringBuffer[PRO_NAME_SIZE];  // PRO_NAME_SIZE = 32
		for (int i=0; i < numLines; ++i)
		{
			ProWstringToString(stringBuffer, p_name[i]);
			//std::cout << std::endl << stringBuffer;
			if ( strcmp(MaterialName.c_str(), _strupr(stringBuffer ))  == 0  ) 
			{
				isis::isis_ProArrayFree((ProArray*)&p_name);
				return true; 
			}
		}

		return false;
	}

	//	Description:
	//		Create in_MaterialName 
	//
	//	Pre-Conditions:
	//		in_Part must be a model retrieved into memory.
	//		in_MaterialName length < 32 chars
	//
	//	Post-Conditions
	//		if in_MaterialName already exist
	//			return; //no action required
	//		else
	//			create in_MaterialName material
	void CreateMaterial( ProSolid in_part, const std::string &in_MaterialName ) 
													throw (isis::application_exception)
	{
		// Check if material exist
		if ( MaterialExists(in_part, in_MaterialName) )
		{
			return;
		}

		// Create new material
		ProMaterial newMaterial;
		ProName  materialName_wchar;
		ProStringToWstring(materialName_wchar, (char *)in_MaterialName.c_str() );
		isis::isis_ProMaterialCreate ( in_part, materialName_wchar, NULL, &newMaterial  );

	}

	//	Description:
	//		Set in_MaterialName material properties to be the same as the current material (i.e. material that
	//		is set to current in the Creo Model)
	//
	//	Pre-Conditions:
	//		in_Part must be a model retrieved into memory.
	//		in_MaterialName length < 32 chars
	//
	//	Post-Conditions
	//		if in_MaterialName is already the current material
	//			return; // no action required.
	//		else
	//			if in_MaterialName material does not exist
	//				throw isis::application_exception
	//			else
	//				set in_MaterialName properties to be equal to the current material properties
	////////////////////////////////////////////////////////////////////////////////////////////////////////
	void SetMaterialPropertiesToBeTheSameAsCurrentMaterial(ProSolid in_part, const std::string &in_MaterialName ) 
													throw (isis::application_exception)
	{

		// Check if material exist
		if ( !MaterialExists(in_part, in_MaterialName) )
		{
			std::string TempError = "Function SetMaterialPropertiesToBeTheSameAsCurrentMaterial(...) was passed a material name that does not exist in the model (i.e. argument in_part).  in_MaterialName: " +
				in_MaterialName;
			throw isis::application_exception(TempError.c_str());
		}


		// Get current material
		ProMaterial currentMaterial;
		isis::isis_ProMaterialCurrentGet( in_part, &currentMaterial );
		ProModelitem  currentMaterialModelItem;
		isis_ProModelitemByNameInit ( in_part, PRO_RP_MATERIAL, currentMaterial.matl_name, &currentMaterialModelItem );

		// Check if in_MaterialName is the current material, if so, no action required.
		char stringBuffer[PRO_NAME_SIZE];  // PRO_NAME_SIZE = 32
		ProWstringToString(stringBuffer, currentMaterial.matl_name);
		std::string tempString = isis::ConvertToUpperCase(in_MaterialName);
		if ( strcmp(tempString.c_str(), _strupr(stringBuffer ))  == 0  )
		{
			// The current material is in_MaterialName. No action needed.
			return;
		}

		// Destination material
		ProMaterial destinationMaterial;
		ProName  materialName_wchar;
		ProStringToWstring(materialName_wchar, (char *)in_MaterialName.c_str() );
		ProModelitem  destinationMaterialModelItem;
		isis_ProModelitemByNameInit ( in_part, PRO_RP_MATERIAL, materialName_wchar, &destinationMaterialModelItem );

		// Copy the material properties from the current to destination material property
		ProParamvalue materialValue;
		ProUnititem   materialUnit;

		// Density
		isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem,		  PRO_MATPROP_MASS_DENSITY, &materialValue, &materialUnit );
		isis::isis_ProMaterialPropertySet( &destinationMaterialModelItem,	  PRO_MATPROP_MASS_DENSITY, &materialValue, &materialUnit );

		// Poisson' Ratio
		isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem,		  PRO_MATPROP_POISSON_RATIO, &materialValue, &materialUnit );
		isis::isis_ProMaterialPropertySet( &destinationMaterialModelItem,	  PRO_MATPROP_POISSON_RATIO, &materialValue, &materialUnit );

		// Young's Modulus
		isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem,		  PRO_MATPROP_YOUNG_MODULUS, &materialValue, &materialUnit );
		isis::isis_ProMaterialPropertySet( &destinationMaterialModelItem,	  PRO_MATPROP_YOUNG_MODULUS, &materialValue, &materialUnit );
		
		// Coeef. of Thermal Expansion
		isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem,		  PRO_MATPROP_THERMAL_EXPANSION_COEFFICIENT, &materialValue, &materialUnit );
		isis::isis_ProMaterialPropertySet( &destinationMaterialModelItem,	  PRO_MATPROP_THERMAL_EXPANSION_COEFFICIENT, &materialValue, &materialUnit );

		// Stress-Strain Response e.g. Linear, Hyperelastic, Elastroplastic
		// Couldn't find the enum entry
		
		// Specific Heat Capicity
		isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem, PRO_MATPROP_SPECIFIC_HEAT, &materialValue, &materialUnit );
		isis::isis_ProMaterialPropertySet( &destinationMaterialModelItem,	  PRO_MATPROP_SPECIFIC_HEAT, &materialValue, &materialUnit );

		// Thermal Conductivity
		isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem, PRO_MATPROP_THERMAL_CONDUCTIVITY, &materialValue, &materialUnit );
		isis::isis_ProMaterialPropertySet( &destinationMaterialModelItem,	  PRO_MATPROP_THERMAL_CONDUCTIVITY, &materialValue, &materialUnit );

		// Symmetry e.g. isotropic
		// Could not find the enum entry

		try 
		{  // Don't fail if this doesnt't exist.
		// Initial Bend Y-Factor
			isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem, PRO_MATPROP_INITIAL_BEND_Y_FACTOR, &materialValue, &materialUnit );
			isis::isis_ProMaterialPropertySet( &destinationMaterialModelItem,	  PRO_MATPROP_INITIAL_BEND_Y_FACTOR, &materialValue, &materialUnit );
		}
		catch (...)
		{
		}

		// Bend Table, skip this 
		//isis::isis_ProMaterialPropertyGet( &currentMaterialModelItem, PRO_MATPROP_BEND_TABLE, &materialValue, &materialUnit );
		//isis::isis_ProMaterialPropertySet( &destinationMaterialModelItem,	  PRO_MATPROP_BEND_TABLE, &materialValue, &materialUnit );

		// Cross Hatching
		// Could not find the enum entry

	}
	////////////////////////////////////////////////////////////////////////////////////////////////////////
	void SetMaterial_to_CurrentMaterial( ProSolid in_part, const std::string &in_MaterialName ) 
													throw (isis::application_exception)
	{
		// Check if material exists
		if ( !MaterialExists(in_part, in_MaterialName) )
		{
			std::string TempError = "Function SetMaterial_to_CurrentMaterial(...) was passed a material name that does not exist in the model (i.e. argument in_part).  in_MaterialName: " +
				in_MaterialName;
			throw isis::application_exception(TempError.c_str());
		}

		ProMaterial material;
		ProName  materialName_wchar;
		ProStringToWstring(material.matl_name, (char *)in_MaterialName.c_str() );
		material.part = in_part;

		// Set material as the current material
		ProMaterialCurrentSet(&material);

	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	void AddTokenMaterialIdentifier(const std::string							&in_AssemblyComponentID, 
									std::map<string, isis::CADComponentData>	&in_out_CADComponentData_map, 
									const std::string							&in_MaterialName,
									double										in_MaterialTokenNumber ) 
	{
		ProSolid part = in_out_CADComponentData_map[in_AssemblyComponentID].modelHandle;

		// Get current material
		ProMaterial material;
		isis::isis_ProMaterialCurrentGet( part, &material );
		ProModelitem  materialModelItem;
		isis_ProModelitemByNameInit ( part, PRO_RP_MATERIAL, material.matl_name, &materialModelItem );

		ProParamvalue materialValue;
		ProUnititem   materialUnit;

		// Get Poisson's ration
		isis::isis_ProMaterialPropertyGet( &materialModelItem, PRO_MATPROP_POISSON_RATIO, &materialValue, &materialUnit );

		// Set the map contents
		in_out_CADComponentData_map[in_AssemblyComponentID].analysisTempMaterialDefinition.poissonsRatio = materialValue.value.d_val;

		char buffer[64];
		sprintf(buffer, "%6.5lf", in_MaterialTokenNumber ); 

		in_out_CADComponentData_map[in_AssemblyComponentID].analysisTempMaterialDefinition.tokenPoissonsRatio = std::string(buffer);
		in_out_CADComponentData_map[in_AssemblyComponentID].analysisTempMaterialDefinition.name = in_MaterialName;

		in_out_CADComponentData_map[in_AssemblyComponentID].analysisTempMaterialDefined = true;


		//std::cout << std::endl << "in_MaterialTokenNumber: " << in_MaterialTokenNumber;
		// Set the token value in the model's material
		materialValue.value.d_val = in_MaterialTokenNumber;
		isis::isis_ProMaterialPropertySet( &materialModelItem, PRO_MATPROP_POISSON_RATIO, &materialValue, &materialUnit );
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void CreateUniquelyNamedMaterials_impl( 
					const std::string							&in_AssemblyComponentID, 
					std::map<string, isis::CADComponentData>	&in_out_CADComponentData_map, // in_out_CADComponentData_map modified by AddTokenMaterialIdentifier
					double										&in_out_MaterialTokenNumber )
																throw (isis::application_exception)
	{
		int materialIndex = 0;	

	    // Warining - in_out_MaterialTokenNumber should be of the form -0.11111, -0.11112, -0.11113...
		// Changin the precision of in_out_MaterialTokenNumber will cause problems because this double
		// is converted to a string in the function AddTokenMaterialIdentifier(..) with the following code:
		//   sprintf(buffer, "%6.5lf", in_MaterialTokenNumber );  
		//	The precission (i.e. %6.5lf) is critical to using the sting version as a key into a map.
		double decrement = -.00001;
		
		char buffer[64];
		for ( std::list<string>::const_iterator i (in_out_CADComponentData_map[in_AssemblyComponentID].children.begin());
		  i != in_out_CADComponentData_map[in_AssemblyComponentID].children.end();
		  ++i )
		{
			++materialIndex;
			std::string materialName = "ZZ_FEA_MAT_" + std::string(itoa(materialIndex,buffer,10)) + "_ZZ";

			if ( materialName.size() >= PRO_NAME_SIZE )
			{
				std::string TempError = "Function CreateUniquelyNamedMaterials_impl(...) created a material name that exceeds 31 characters.  in_MaterialName: " + materialName;
				throw isis::application_exception(TempError.c_str());
			}

			//std::cout << std::endl << "materialName: " << materialName;

			if ( in_out_CADComponentData_map[*i].modelType ==  PRO_MDL_PART) 
			{
				TestRetrievingCurrentMaterial (	*i,	in_out_CADComponentData_map ); 
				TestIfCurrentMaterialHasNeededFEAProperties(*i,	in_out_CADComponentData_map );
				CreateMaterial(in_out_CADComponentData_map[*i].modelHandle, materialName );
				SetMaterialPropertiesToBeTheSameAsCurrentMaterial(in_out_CADComponentData_map[*i].modelHandle, materialName );
				SetMaterial_to_CurrentMaterial( in_out_CADComponentData_map[*i].modelHandle, materialName );
				AddTokenMaterialIdentifier( *i, in_out_CADComponentData_map, materialName, in_out_MaterialTokenNumber );
				
				in_out_MaterialTokenNumber += decrement;

				// add the following temporarily to save the material changes for debuggin purposes
				//isis::isis_ProMdlSave(in_out_CADComponentData_map[*i].modelHandle );
			}
			if ( in_out_CADComponentData_map[*i].modelType == PRO_MDL_ASSEMBLY  && in_out_CADComponentData_map[*i].children.size() > 0) // Note - an existing assmbly would have no children. 
			{
				CreateUniquelyNamedMaterials_impl( *i,in_out_CADComponentData_map, in_out_MaterialTokenNumber);
			}
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	void CreateUniquelyNamedMaterials( 
					const std::string							&in_AssemblyComponentID, 
					std::map<string, isis::CADComponentData>	&in_out_CADComponentData_map) // in_out_CADComponentData_map modified by AddTokenMaterialIdentifier
																	throw (isis::application_exception)

	{
		// Since Poisson's ratio can be in the range of -1 to .5, starting with -.11111 and decrementing by .00001 will
		// result in about 80,000 values before -.1 is reached.  Since our assemblies will be at most 5,000 parts, and the
		// analysis assemblies will be only a few parts (probably less than 30), this approach should meet our needs.
		// Also, negative Poisson's ratios are rare; and therefore, collisions are unlikely.
		double materialTokenNumber = - .11111;
		CreateUniquelyNamedMaterials_impl(in_AssemblyComponentID, in_out_CADComponentData_map, materialTokenNumber );
	}


} // END namespace isis
