/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using GME.MGA;
using System.Diagnostics.Contracts;
using System.IO;

namespace ClmLib
{
    public class Clm
    {

        public IMgaProject Project { get; set; }

        public Clm(IMgaProject project)
        {
            Contract.Requires(project != null);

            Project = project;
        }

        public List<IMgaFCO> GetRelatedComponents(IMgaFCO component)
        {
            Contract.Requires(component != null);

            var componentReal = component is IMgaReference ?
                (component as IMgaReference).Referred :
                component;

            if (componentReal == null)
            {
                return new List<IMgaFCO>();
            }

            string ontologyNodeToMatch = componentReal.StrAttrByName["Classifications"];

            return GetRelatedComponents(ontologyNodeToMatch);
        }

        public List<IMgaFCO> GetRelatedComponents(string ontologyNodeToMatch)
        {
            Contract.Requires(string.IsNullOrWhiteSpace(ontologyNodeToMatch) == false);

            List<IMgaFCO> result = new List<IMgaFCO>();

            MgaFilter filter = Project.CreateFilter();
            filter.Kind = "Component";

            var components = Project.AllFCOs(filter);

            foreach (IMgaFCO component in components)
            {
                if (CompareClassifications(component.StrAttrByName["Classifications"], ontologyNodeToMatch))
                {
                    result.Add(component);
                }
            }

            return result;
        }

        public List<IMgaFCO> InsertComponents(
            IMgaFCO designContainer,
            IMgaFCO componentRef,
            List<IMgaFCO> components)
        {
            Contract.Requires(designContainer as IMgaModel != null);
            Contract.Requires(componentRef as IMgaReference != null);
            Contract.Requires((componentRef as IMgaReference).Referred != null);
            Contract.Requires(components != null);

            List<IMgaFCO> result = new List<IMgaFCO>();

            IMgaModel container = designContainer as IMgaModel;
            IMgaReference compRef = componentRef as IMgaReference;

            var childComps = container.
                ChildFCOs.
                Cast<IMgaFCO>().
                Where(x => x.MetaBase.Name == "ComponentRef").
                Cast<IMgaReference>().
                Select(x => x.Referred).
                ToList();

            // get all connections which has the componentRef as an endpoint
            var childConnections = container.
                ChildFCOs.
                Cast<IMgaFCO>().
                Where(x => x is IMgaSimpleConnection).
                Cast<IMgaSimpleConnection>().
                ToList();


            // ith new component
            int iNewComponent = 0;

            foreach (var compToCreate in components)
            {
                if (childComps.Contains(compToCreate))
                {
                    // If the component already exists this function will not create it again
                    // skip
                    continue;
                }

                // create reference
                var newRef = container.CreateReference(componentRef.MetaRole, compToCreate as MgaFCO);
                newRef.Name = compToCreate.Name;

                result.Add(newRef);

                bool compatibale = true;

                // create connections
                foreach (var connectionToCreate in childConnections)
                {
                    if (SafeMgaObjectCompare(connectionToCreate.SrcReferences.Cast<IMgaFCO>().FirstOrDefault(), compRef) ||
                        SafeMgaObjectCompare(connectionToCreate.DstReferences.Cast<IMgaFCO>().FirstOrDefault(), compRef))
                    {
                        try
                        {
                            var connRole = connectionToCreate.MetaRole;
                            var connSrc = connectionToCreate.Src;
                            var connDst = connectionToCreate.Dst;
                            var connSrcReference = connectionToCreate.SrcReferences.Cast<MgaFCO>().FirstOrDefault();
                            var connDstReference = connectionToCreate.DstReferences.Cast<MgaFCO>().FirstOrDefault();

                            // overwrite the new endpoints
                            if (SafeMgaObjectCompare(connectionToCreate.SrcReferences.Cast<IMgaFCO>().FirstOrDefault(), compRef))
                            {
                                connSrcReference = newRef;

                                var srcCandidate = compToCreate.ObjectByPath[connSrc.Name] as MgaFCO;
                                if (srcCandidate.Meta.MetaRef != connSrc.Meta.MetaRef)
                                {
                                    continue;
                                }
                                connSrc = srcCandidate;
                            }

                            if (SafeMgaObjectCompare(connectionToCreate.DstReferences.Cast<IMgaFCO>().FirstOrDefault(), compRef))
                            {
                                connDstReference = newRef;

                                var dstCandidate = compToCreate.ObjectByPath[connDst.Name] as MgaFCO;
                                if (dstCandidate.Meta.MetaRef != connDst.Meta.MetaRef)
                                {
                                    continue;
                                }
                                connDst = dstCandidate;
                            }

                            // check end points
                            if (connSrc != null &&
                                connDst != null)
                            {
                                var newConnection = container.CreateSimpleConnDisp(
                                    connRole,
                                    connSrc,
                                    connDst,
                                    connSrcReference,
                                    connDstReference);
                            }
                        }
                        catch (Exception ex)
                        {
                            System.Diagnostics.Trace.TraceError(ex.ToString());
                            System.Diagnostics.Trace.TraceError("Probably some ports do not match.");
                            compatibale = false;
                        }
                    }
                }

                if (compatibale)
                {
                    iNewComponent = iNewComponent + 1;

                    foreach (IMgaPart part in newRef.Parts)
                    {
                        int x = 0;
                        int y = 0;
                        string icon;

                        componentRef.PartByMetaPart[part.Meta].GetGmeAttrs(out icon, out x, out y);

                        part.SetGmeAttrs(icon, x, y + 80 * iNewComponent);
                    }

                }
                else
                {
                    result.Remove(newRef);
                    newRef.DestroyObject();
                }

            }

            return result;
        }

        private bool SafeMgaObjectCompare(IMgaObject obj1, IMgaObject obj2)
        {
            if (obj1 == obj2)
            {
                return true;
            }
            else if (obj1 != null && obj2 != null)
            {
                return obj1.ID == obj2.ID;
            }
            else
            {
                return false;
            }
        }

        public bool CompareClassifications(
            string classification1,
            string classification2)
        {
            if (string.IsNullOrWhiteSpace(classification1) ||
                string.IsNullOrWhiteSpace(classification2))
            {
                return false;
            }

            return classification1.Trim() == classification2.Trim();
        }
    }
}
