/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Diagnostics.Contracts;
using GME.MGA;

using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using Newtonsoft.Json;
using Newtonsoft.Json.Converters;
using System.Diagnostics;

namespace AVM.DDP
{
    public class MetaTBReport
    {
        public abstract class ValueWithUnit
        {
            public string Name { get; set; }
            public string ID { get; set; }
            public string DisplayedName { get; set; }
            public string Unit { get; set; }
            public string Value { get; set; }
            public string GMEID { get; set; }
        }

        public class Parameter : ValueWithUnit
        {
        }

        public class Metric : ValueWithUnit
        {
        }
		
		public class Artifact
		{
			public string Name { get; set; }
			public string Path { get; set; }
		}

        public enum AnalysisStatusEnum
        {
            UNEXECUTED,
            OK,
            FAILED
        }

        [JsonConverter(typeof(StringEnumConverter))]
        public AnalysisStatusEnum AnalysisStatus { get; set; } 

        public string Design { get; set; }
        public string DesignID { get; set; }
        //public string ConfigurationUniqueID { get; set; }
        public string DesignName { get; set; }
        public string TestBench { get; set; }
        public string Time { get; set; }
        public string Details { get; set; }
        public List<Metric> Metrics { get; set; }
        public List<Parameter> Parameters { get; set; }
		public List<Artifact> Artifacts { get; set; }

        public MetaTBReport()
        {
            
            Metrics = new List<Metric>();
            Parameters = new List<Parameter>();
			Artifacts = new List<Artifact>();
            Details = "";
        }


        public void GenerateSummary(
            CyPhy.TestBenchType testBench,
            string fileName = "summary.testresults.json",
            string outputDir = "")
        {
            Contract.Requires(testBench != null);

            string fullpathFileName = Path.GetFullPath(fileName);
            if (fullpathFileName != fileName)
            {
                fullpathFileName = Path.Combine(
                    outputDir,
                    Path.GetFileName(fileName));
            }

            this.TestBench = string.IsNullOrEmpty((testBench.Impl as MgaFCO).RegistryValue["TestBenchUniqueName"]) ?
                testBench.Name :
                (testBench.Impl as MgaFCO).RegistryValue["TestBenchUniqueName"];

            this.Time = DateTime.Now.ToString("yyyy-MM-dd HH-mm-ss");

            foreach (var item in testBench.Children.MetricCollection)
            {
                var oldData = this.Metrics.FirstOrDefault(x => x.ID == item.Guid.ToString());
                if (oldData != null)
                {
                    this.Metrics.Remove(oldData);
                }

                AVM.DDP.MetaTBReport.Metric metric = new AVM.DDP.MetaTBReport.Metric();
                metric.Name = item.Name;
                metric.Unit = (item.Impl as MgaReference).Referred == null ?
                    "" :
                    (item.Impl as MgaReference).Referred.Name;

                metric.Value = item.Attributes.Value;
                metric.ID = item.Guid.ToString();
                metric.GMEID = item.ID;
                this.Metrics.Add(metric);
            }

            if (testBench is CyPhy.CADTestBench)
            {

                foreach (var item in (testBench as CyPhy.CADTestBench).Children.StructuralAnalysisMetricsCollection)
                {
                    var oldData = this.Metrics.FirstOrDefault(x => x.ID == item.Guid.ToString());
                    if (oldData != null)
                    {
                        this.Metrics.Remove(oldData);
                    }

                    AVM.DDP.MetaTBReport.Metric metric = new AVM.DDP.MetaTBReport.Metric();
                    metric.Name = item.Name;
                    metric.Unit = "";
                    metric.Value = "";
                    metric.ID = item.Guid.ToString();

                    string disp_name = "";

                    var tip2smetric = item.SrcConnections.TIP2StructuralMetricCollection.FirstOrDefault();
                    if (tip2smetric != null)
                    {
                       disp_name = String.Format("{0}.{1}",
                           tip2smetric.SrcEnds.TestInjectionPoint.Name,
                           item.Name);
                    }

                    metric.DisplayedName = disp_name;
                    metric.GMEID = item.ID;
                    this.Metrics.Add(metric);
                }

            }

            foreach (var item in testBench.Children.ParameterCollection)
            {
                var oldData = this.Parameters.FirstOrDefault(x => x.ID == item.Guid.ToString());
                if (oldData != null)
                {
                    this.Parameters.Remove(oldData);
                }

                AVM.DDP.MetaTBReport.Parameter parameter = new AVM.DDP.MetaTBReport.Parameter();
                parameter.Name = item.Name;
                parameter.Unit = (item.Impl as MgaReference).Referred == null ?
                    "" :
                    (item.Impl as MgaReference).Referred.Name;

                parameter.Value = item.Attributes.Value;
                parameter.ID = item.Guid.ToString();
                parameter.GMEID = item.ID;
                this.Parameters.Add(parameter);
            }

            this.Design = (testBench.Impl as IMgaFCO).RegistryValue["DB_Config_ID"] + ".metadesign.json";
            this.Design = testBench.Impl.Name + ".metadesign.json";
            // FIX ME: put design name here...
            this.DesignName = testBench.Impl.Name;
            
            // get designID
            string designID = null;
            var tlsut = testBench.Children.TopLevelSystemUnderTestCollection.FirstOrDefault();
            var catlsut = testBench.Children.ComponentAssemblyCollection.FirstOrDefault();
            if (tlsut != null)
            {
                // if it is a reference
                if (tlsut.Referred.DesignEntity != null)
                {
                    designID = tlsut.Referred.DesignEntity.Properties.Guid.ToString("B");
                    if (tlsut.Referred.DesignEntity is CyPhy.ComponentAssembly)
                    {
                        catlsut = tlsut.Referred.ComponentAssembly;
                        var cid = catlsut.Attributes.ConfigurationUniqueID;
                        //this.ConfigurationUniqueID = cid;

                        if (string.IsNullOrWhiteSpace(cid))
                        {
                            cid = Guid.NewGuid().ToString("B");
                            catlsut.Attributes.ConfigurationUniqueID = cid;
                        }

                        if (!string.IsNullOrEmpty(cid))
                        {
                            try
                            {
                                Guid guid = new Guid(cid);
                                designID = guid.ToString("B");
                            }
                            catch (System.FormatException ex)
                            {
                                Trace.TraceError("{0} is not a vaild GUID.", cid);
                                Trace.TraceError(ex.ToString());
                            }
                        }
                    }
                }
            }
            else if (catlsut != null)
            {
                // if it is an instance
                var cid = catlsut.Attributes.ConfigurationUniqueID;
                //this.ConfigurationUniqueID = cid;

                if ( ! string.IsNullOrEmpty(cid) )
                {
                    Guid guid = new Guid(cid);
                    designID = guid.ToString("B");
                }
            }

            this.DesignID = designID;

            string reportJson = Newtonsoft.Json.JsonConvert.SerializeObject(
                this,
                Newtonsoft.Json.Formatting.Indented);

            using (StreamWriter writer = new StreamWriter(fullpathFileName))
            {
                writer.WriteLine(reportJson);
            }
        }

    }
}
