/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Linq.Expressions;
using System.Text;
using AVM.META;
using AVM.iFAB;
using Common.Command;
using Common.Notification;
using AVM;
using AVM.VehicleForge;
using System.IO;
using ComponentAuthoringAssistant.Framework;
using File = AVM.File;
using System.Windows;

namespace ComponentAuthoringAssistant.Models
{
    public class ComponentUI : NotifyBase
    {
        private Component _avmComponent;
        public Component AVMComponent
        {
            get { return _avmComponent; }
            private set
            {
                _avmComponent = value;
                _originalName = AVMComponent.Name;
                _originalDescription = AVMComponent.Description;

                var itarFeature = AVMComponent.Features.FirstOrDefault(x => x is ITAR_Restriction) as ITAR_Restriction;
                if (itarFeature == null)
                    _originalITARRestriction = null;
                else
                    _originalITARRestriction = itarFeature.Level;

                OnPropertyChanged(() => AVMComponent);
                OnPropertyChanged(() => ComponentName);
                OnPropertyChanged(() => ComponentDescription);
                OnPropertyChanged(() => ITARResctriction);
                OnPropertyChanged(() => Artifacts);
                
            }
        }

        private HashSet<File> _newFiles = new HashSet<File>();
        public void AddFile(string fileName, string hash)
        {
            var newFile = new File {Location = fileName, Hash = "md5:"+hash};

            _newFiles.Add(newFile);
            AVMComponent.Files.Add(newFile);
            OnPropertyChanged(()=>Artifacts);
            HasArtifactAdded = true;
        }

        private HashSet<Model> _newModels = new HashSet<Model>();
        public void PromoteGenericFileToCADModel(ArtifactDescriptor genericFile)
        {
            var newModel = new CADModel
            {
                Location = genericFile.Name,
            };
            _newModels.Add(newModel);
            AVMComponent.Features.Add(newModel);
        }

        public void PromoteGenericFileToManufactoringModel(ArtifactDescriptor genericFile)
        {
            var newModel = new ManufacturingModel
            {
                Location = genericFile.Name,
            };
            _newModels.Add(newModel);
            AVMComponent.Features.Add(newModel);
        }


        private string _originalName;
        public string ComponentName
        {
            get
            {
                return AVMComponent.Name;
            }
            set
            {
                AVMComponent.Name = value;
                OnPropertyChanged(() => ComponentName);
                OnPropertyChanged(()=>HasComponentNameChanged);
                OnPropertyChanged(() => HasChanged);
            }
        }
        public bool HasComponentNameChanged
        {
            get {return _originalName != ComponentName; }
        }

        private string _originalDescription;
        public string ComponentDescription
        {
            get { return AVMComponent.Description; }
            set
            {
                AVMComponent.Description = value;
                OnPropertyChanged(() => ComponentDescription);
                OnPropertyChanged(() => HasDescriptionChanged);
                OnPropertyChanged(() => HasChanged);
            }
        }
        public bool HasDescriptionChanged
        {
            get { return _originalDescription != ComponentDescription; }
        }

        private bool _hasArtifactAdded;

        public bool HasArtifactAdded
        {
            get { return _hasArtifactAdded; }
            set
            {
                _hasArtifactAdded = value;
                OnPropertyChanged(() => HasArtifactAdded);
                OnPropertyChanged(() => HasChanged);
            }
        }

        private bool _hasArtifactDeleted;

        public bool HasArtifactDeleted
        {
            get { return _hasArtifactDeleted; }
            set
            {
                _hasArtifactDeleted = value;
                OnPropertyChanged(() => HasArtifactDeleted);
                OnPropertyChanged(() => HasChanged);
            }
        }

        private ITARRestrictionLevelEnum? _originalITARRestriction;
        public ITARRestrictionLevelEnum ITARResctriction
        {
            get
            {
                var itarFeature = AVMComponent.Features.FirstOrDefault(x => x is ITAR_Restriction) as ITAR_Restriction;

                if (itarFeature == null)
                {
                    itarFeature = new ITAR_Restriction {Level = ITARRestrictionLevelEnum.NotITAR};
                    AVMComponent.Features.Add(itarFeature);
                }

                return itarFeature.Level;
            }
            set
            {
                var itarFeature = (ITAR_Restriction)AVMComponent.Features.First(x => x is ITAR_Restriction);
                itarFeature.Level = value;
                OnPropertyChanged(() => ITARResctriction);
                OnPropertyChanged(() => HasItarRestrictionChanged);
                OnPropertyChanged(() => HasChanged);
            }
        }

        public bool HasChanged
        {
            get { return HasComponentNameChanged || HasDescriptionChanged || HasItarRestrictionChanged || Artifacts.Any(x=>x.HasArtifactChanged) || HasArtifactDeleted || HasArtifactAdded; }
        }

        public bool HasErrors
        {
            get
            {
                var result = _artifacts.Any(x => x.ArtifactStatus != ArtifactStatus.Ok);
                return result;
            }
        }

        public bool HasItarRestrictionChanged
        {
            get { return _originalITARRestriction != ITARResctriction; }
        }

        public void RefreshArtifacts()
        {
            OnPropertyChanged(() => Artifacts);
            OnPropertyChanged(() => HasErrors);
            OnPropertyChanged(() => HasChanged);
        }

        public File RecheckLocation(ArtifactDescriptor afd)
        {
            if (afd.OriginalModel is CADModel)
            {
                var model = (CADModel) afd.OriginalModel;
                var avmFile = AVMComponent.Files.FirstOrDefault(x => x.Location.Replace('/', '\\') == model.Location.Replace('/', '\\'));
                if (avmFile == null)
                {
                    afd.ArtifactStatus = ArtifactStatus.FileNotFound;
                    afd.ArtifactstatusError = String.Format("There is no related AVMFile with name: {0}", model.Location);
                }
                else
                {
                    afd.ArtifactStatus = ArtifactStatus.Ok;
                    _hiddenAVMFiles.Add(avmFile);
                    return avmFile;
                }
            }
            else if (afd.OriginalModel is ManufacturingModel)
            {
                var model = (ManufacturingModel)afd.OriginalModel;
                var avmFile = AVMComponent.Files.FirstOrDefault(x => x.Location.Replace('/', '\\') == model.Location.Replace('/', '\\'));
                if (avmFile == null)
                {
                    afd.ArtifactStatus = ArtifactStatus.FileNotFound;
                    afd.ArtifactstatusError = String.Format("There is no related AVMFile with name: {0}",
                                                            model.Location);
                }
                else
                {
                    afd.ArtifactStatus = ArtifactStatus.Ok;
                    _hiddenAVMFiles.Add(avmFile);
                    return avmFile;
                }
            }
            else if (afd.OriginalModel is AVM.File)
            {
                var file = (File) afd.OriginalModel;
                var baseUri = new Uri(ComponentPath);
                var relativeUri = new Uri(file.Location, UriKind.Relative);
                var fullUri = new Uri(baseUri, relativeUri);

                var fi = new FileInfo(Uri.UnescapeDataString(fullUri.AbsolutePath));

                if (fi.Exists) afd.ArtifactStatus = ArtifactStatus.Ok;
                else afd.ArtifactStatus = ArtifactStatus.FileNotFound;

            }
            return null;
        }

        private Dictionary<File, ArtifactDescriptor> _fileCache = new Dictionary<File, ArtifactDescriptor>();
        private Dictionary<Model, ArtifactDescriptor> _modelCache = new Dictionary<Model, ArtifactDescriptor>();
        private ObservableCollection<ArtifactDescriptor> _artifacts;
        private HashSet<File>  _hiddenAVMFiles = new HashSet<File>();
        public ObservableCollection<ArtifactDescriptor> Artifacts
        {
            get
            {
                _artifacts.Clear();

                foreach (var model in AVMComponent.Features.OfType<Model>())
                {
                    if (_modelCache.ContainsKey(model)) { _artifacts.Add(_modelCache[model]); continue; }

                    ArtifactDescriptor afd = null;

                    if (model is BehaviorModel)
                    {
                        afd = new ArtifactDescriptor(ArtifactTypes.BehaviorModel, model)
                                  {
                                      Name = model.Location,
                                      EditedName = model.Location,
                                      ArtifactStatus = ArtifactStatus.Ok
                                  };
                        _modelCache[model] = afd;
                    }
                    else if (model is CADModel)
                    {
                        afd = new ArtifactDescriptor(ArtifactTypes.CADModel, model)
                        {
                            Name = model.Location,
                            EditedName = model.Location,
                        };

                        RecheckLocation(afd);
                        
                        if (_newModels.Contains(model)) afd.HasArtifactTypeChanged = true;
                        _modelCache[model] = afd;
                    }
                    else if (model is ManufacturingModel)
                    {
                        afd = new ArtifactDescriptor(ArtifactTypes.ManufacturingModel, model)
                        {
                            Name = model.Location,
                            EditedName = model.Location,
                        };

                        RecheckLocation(afd);

                        if (_newModels.Contains(model)) afd.HasArtifactTypeChanged = true;
                        _modelCache[model] = afd;
                    }

                    if (afd!=null)
                        _artifacts.Add(afd);
                }

                foreach (var avmFile in AVMComponent.Files.Where(x => !_hiddenAVMFiles.Contains(x)))
                {
                    if (_fileCache.ContainsKey(avmFile)) {_artifacts.Add(_fileCache[avmFile]); continue;}
                    

                    var baseUri = new Uri(ComponentPath);
                    var relativeUri = new Uri(avmFile.Location, UriKind.Relative);
                    var fullUri = new Uri(baseUri, relativeUri);

                    var fi = new FileInfo(Uri.UnescapeDataString(fullUri.AbsolutePath));

                    var afd = new ArtifactDescriptor(ArtifactTypes.GenericFile, avmFile)
                                  {
                                      Name = avmFile.Location,
                                      ArtifactStatus = (fi.Exists) ? ArtifactStatus.Ok : ArtifactStatus.FileNotFound,
                                      ArtifactstatusError = (fi.Exists) ? string.Empty : String.Format("AVMFile is missing: {0}", avmFile.Location),
                                  };



                    if (_newFiles.Contains(avmFile)) afd.HasArtifactTypeChanged = true;

                    _fileCache[avmFile] = afd;
                    _artifacts.Add(afd);

                }

                foreach (var library in AVMComponent.Features.OfType<UsesLibrary>())
                {
                    var afd = new ArtifactDescriptor(ArtifactTypes.SharedLibrary, library)
                                  {
                                      Name = string.Format("{0} version {1}", library.Name, library.Version),
                                      ArtifactStatus = ArtifactStatus.Ok
                                  };

                    _artifacts.Add(afd);
                }


                return _artifacts;
            }
            set
            {
                _artifacts = value;
                OnPropertyChanged(() => Artifacts);
            }
        }

        public void DeleteModelOrFile(object entityToBeDeleted)
        {
            if (entityToBeDeleted is Model)
            {
                if (MessageBox.Show(String.Format("Do you really want to delete the following model: {0} ?",((Model) entityToBeDeleted).Location), "Confirmation", MessageBoxButton.YesNo,MessageBoxImage.Question) == MessageBoxResult.Yes)
                {
                    var model = (Model) entityToBeDeleted;
                    var file = AVMComponent.Files.FirstOrDefault(x => x.Location.Replace('/', '\\') == model.Location.Replace('/', '\\'));

                    if (file != null && _hiddenAVMFiles.Contains(file)) _hiddenAVMFiles.Remove(file);
                    
                    if (_modelCache.ContainsKey(model))
                        _modelCache.Remove(model);
                    AVMComponent.Features.Remove(model);

                    // Delete the file as well I-830
                    if (file != null)
                    {
                        var relativePath = file.Location;
                        var filePath = Utilities.GetAbsolutePath(ComponentPath, relativePath);
                        var fi = new FileInfo(filePath);

                        if (_fileCache.ContainsKey(file))
                            _fileCache.Remove(file);
                        AVMComponent.Files.Remove(file);

                        if (fi.Exists)
                        {
                            try
                            {
                                fi.Delete();
                            }
                            catch
                            {

                            }
                        }
                    }

                    HasArtifactDeleted = true;
                    RefreshArtifacts();
                }
            }
            else if (entityToBeDeleted is File)
            {
                if (MessageBox.Show(String.Format("Do you really want to remove the following file reference from the component: {0} ?",((File)entityToBeDeleted).Location), "Confirmation", MessageBoxButton.YesNo,MessageBoxImage.Question) == MessageBoxResult.Yes)
                {
                    var relativePath = ((File) entityToBeDeleted).Location;
                    var filePath = Utilities.GetAbsolutePath(ComponentPath, relativePath);

                    var fi = new FileInfo(filePath);

                    if (_fileCache.ContainsKey((File) entityToBeDeleted))
                        _fileCache.Remove((File) entityToBeDeleted);
                    AVMComponent.Files.Remove((File) (entityToBeDeleted));
                    _hiddenAVMFiles.Remove((File) entityToBeDeleted);
                    HasArtifactDeleted = true;

                    if (fi.Exists && MessageBox.Show(String.Format("Do you want to delete the file from the filesystem: {0} ?",((File) entityToBeDeleted).Location), "Confirmation", MessageBoxButton.YesNo,MessageBoxImage.Question) == MessageBoxResult.Yes)
                    {
                        try
                        {
                            fi.Delete();
                        }
                        catch
                        {

                        }
                    }
                }
            }
            else if (entityToBeDeleted is UsesLibrary)
            {
                if (
                    MessageBox.Show(
                        String.Format("Do you really want to delete the following library: {0} ?",
                                      ((UsesLibrary)entityToBeDeleted).Name), "Confirmation", MessageBoxButton.YesNo,
                        MessageBoxImage.Question) == MessageBoxResult.Yes)
                {
                    AVMComponent.Features.Remove((UsesLibrary) entityToBeDeleted);
                    HasArtifactDeleted = true;
                }

            }

            OnPropertyChanged(()=>Artifacts);
        }

        public List<File> AVMFiles
        {
            get { return AVMComponent.Files; }
        }

        private string _componentPath;

        public string ComponentPath
        {
            get { return _componentPath; }
            set
            {
                _componentPath = value;
                OnPropertyChanged(() => ComponentPath);
            }
        }

        public ComponentUI(Component component, string componentPath)
        {
            AVMComponent = component;
            ComponentPath = componentPath;
            Artifacts = new ObservableCollection<ArtifactDescriptor>();
        }

        internal void SaveComponent()
        {
            AVMComponent.SerializeToFile(ComponentPath);
            _newFiles = new HashSet<File>();
        }

        internal void SaveComponentAs(string path)
        {
            AVMComponent.SerializeToFile(path);
        }
    }
}
