/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Windows;
using Common.Notification;
using Common.Command;
using ComponentAuthoringAssistant.Models;
using ComponentAuthoringAssistant.Views;
using Ionic.Zip;
using Microsoft.Win32;
using AVM;
using System.IO;
using Component = AVM.Component;
using System.Threading.Tasks;
using System.Threading;
using AVM.META;

namespace ComponentAuthoringAssistant.ViewModels
{
    public class ComponentAuthoringViewModel : NotifyBase
    {

        private readonly Window _owner;

        #region Commands

        #region Properties

        public DelegateCommand DiscardCommand { get; set; }
        public DelegateCommand LoadCommand { get; set; }
        public DelegateCommand SaveCommand { get; set; }
        public DelegateCommand AddFileCommand { get; set; }
        public DelegateCommand ExportZipCommand { get; set; }
        public DelegateCommand ExportGmeCommand { get; set; }
        public DelegateCommand ChangeTypeCommand { get; set; }
        public DelegateCommand ViewArtifactCommand { get; set; }
        public DelegateCommand EditLocationCommand { get; set; }
        public DelegateCommand DiscardArtifactChangeCommand { get; set; }
        public DelegateCommand AcceptArtifactChangeCommand { get; set; }
        public DelegateCommand ArtifactDblClick { get; set; }
        public DelegateCommand ConfirmExitCommand { get; set; }
        public DelegateCommand DeleteTempCommand { get; set; }
        public DelegateCommand DeleteArtifactCommand { get; set; }

        #endregion

        #region Implementation

        private void ConfirmNoSave(string message = "You have unsaved changes. Would you like to save your component now?")
        {
            if (AVMComponent == null ||
                !AVMComponent.HasComponentNameChanged && !AVMComponent.HasDescriptionChanged &&
                !AVMComponent.HasItarRestrictionChanged && !AVMComponent.Artifacts.Any(x => x.HasArtifactChanged))
                return;

            if (MessageBox.Show(message,
                                "Unsaved changes", MessageBoxButton.YesNo, MessageBoxImage.Question)==MessageBoxResult.Yes)
            {
                SaveCommand.Execute(null);
            }
        }

        private bool ConfirmDiscardChanges(string message = "Do you want to discard your changes?")
        {
            if (AVMComponent == null || !AVMComponent.HasComponentNameChanged && !AVMComponent.HasDescriptionChanged &&
                !AVMComponent.HasItarRestrictionChanged && !AVMComponent.Artifacts.Any(x => x.HasArtifactChanged))
                return true;

            if (MessageBox.Show(message,"Discard changes", MessageBoxButton.YesNo, MessageBoxImage.Question) == MessageBoxResult.No)
            {
                return false;
            }

            return true;
        }

        public void ExecuteDeleteTemp(object state)
        {
            var ww = new WaitView(WaitTypes.DeleteTempfiles){WindowStartupLocation = WindowStartupLocation.CenterScreen};
            var task = new Task(Framework.Temp.DeleteTempDirectories);
            task.ContinueWith(o =>
                                  {
                                      ww.Dispatcher.Invoke((Action)ww.Close);
                                  });
            task.Start();

            ww.ShowDialog();
        }

        public void ExecuteConfirmExit(object state)
        {
            if (!(state is CancelEventArgs)) return;

            var result = !ConfirmDiscardChanges("Do you want to exit without save?");
            ((CancelEventArgs) state).Cancel = result;

            if (!result)
            {
                DeleteTempCommand.Execute(null);
            }
        }

        public void ExecuteChangeType(object state)
        {
            var artifact = state as ArtifactDescriptor;
            if (artifact == null) return;

            var cv = new ChangeVisibilityView(artifact) { Owner = _owner };
            if (cv.ShowDialog() == true)
            {
                // What is the new artifact?
                switch (artifact.ArtifactType)
                {
                    // Add cadmodel
                    case ArtifactTypes.CADModel:
                        AVMComponent.PromoteGenericFileToCADModel(artifact);
                        break;
                    case ArtifactTypes.ManufacturingModel:
                        AVMComponent.PromoteGenericFileToManufactoringModel(artifact);
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }

                SaveCommand.CanExecute(null);
                DiscardCommand.CanExecute(null);
            }
        }

        private bool _isGmeRunning;
        public bool IsGmeRunning
        {
            get { return _isGmeRunning; }
            set
            {
                _isGmeRunning = value;
                OnPropertyChanged(() => IsGmeRunning);
            }
        }

        private bool _isComponentReloaded;
        public bool IsComponentReloaded
        {
            get { return _isComponentReloaded; }
            set
            {
                _isComponentReloaded = value;
                OnPropertyChanged(() => IsComponentReloaded);
            }
        }

        private Task _pingGmeTask;
        private CancellationTokenSource _cts;
        private CancellationToken _token;

        private void PingGME(int processId)
        {
            if (_pingGmeTask != null)
            {
                try
                {
                    _cts.Cancel();
                }
                catch
                {
                }
            }

            _cts = new CancellationTokenSource();
            _token = _cts.Token;

            _pingGmeTask = Task.Factory.StartNew(state =>
                                      {
                                              while (true)
                                              {
                                                  var proc = Process.GetProcesses().FirstOrDefault(x=>x.Id==processId);

                                                  if (proc != null)
                                                  {
                                                      IsGmeRunning = true;
                                                      Thread.Sleep(500);
                                                  }
                                                  if (((CancellationToken) state).IsCancellationRequested || proc==null)
                                                  {
                                                      IsGmeRunning = false;
                                                      return;
                                                  }
                                                  
                                              }
                                      }, _token);
        }

        public void ExecuteExportGme(object state)
        {
            ConfirmNoSave();
            Window ww = new WaitView(WaitTypes.ImportToGME) {Owner = _owner};
            string mgaPath = string.Empty;
            Task.Factory.StartNew(() =>
                                      {
                                          #region SaveAs JSON file to a temporary directory

                                          // Get temp dir
                                          var tempDir = new DirectoryInfo(Path.Combine(Path.GetTempPath(),"META_" + Guid.NewGuid()));
                                          tempDir.Create();

                                          // Add tempdir to framework
                                          Framework.Temp.AddTempDir(tempDir);

                                          // Copy JSON file to tempdir
                                          var newPath = Path.Combine(tempDir.FullName, Path.GetFileName(JSONFilePath));
                                          AVMComponent.SaveComponentAs(newPath);

                                          // Copy all directories
                                          foreach (var fullPath in new DirectoryInfo(JSONDirectory).GetDirectories("*",SearchOption.AllDirectories).Select(x => x.FullName))
                                          {
                                              var pathEnd = fullPath.Replace(JSONDirectory, "").Trim('\\', '/');
                                              tempDir.CreateSubdirectory(pathEnd);
                                          }

                                          // Copy all files (except *.JSON)
                                          foreach (var sourceFile in new DirectoryInfo(JSONDirectory).GetFiles("*", SearchOption.AllDirectories).Where(x=>x.Extension!=".json").Select(x => x.FullName))
                                          {
                                              var pathEnd = Path.GetDirectoryName(sourceFile).Replace(JSONDirectory, "").Trim('\\', '/');
                                              var dstFile = Path.Combine(Path.Combine(tempDir.FullName, pathEnd), Path.GetFileName(sourceFile));
                                              System.IO.File.Copy(sourceFile, dstFile, true);
                                          }

                                          #endregion

                                          #region Import

                                          var mgaName = Path.GetFileNameWithoutExtension(JSONFilePath) + ".mga";
                                          mgaPath = Path.Combine(Path.GetDirectoryName(newPath), mgaName);
                                          var proc = new Process
                                                         {
                                                             EnableRaisingEvents = false,
                                                             StartInfo =
                                                                 {
                                                                     FileName = "CyPhyComponentImporterCL.exe",
                                                                     Arguments = newPath + " " + mgaPath,
                                                                     CreateNoWindow = true,
                                                                     UseShellExecute = false
                                                                 }
                                                         };

                                          proc.Start();
                                          proc.WaitForExit();

                                          #endregion

                                          return mgaPath;
                                      }).ContinueWith(o =>
                                                          {
                                                              var gmeOldTaskIds = Process.GetProcesses().Where(x => x.ProcessName == "GME").Select(x=>x.Id);
                                                              var getNewGmeTask = new Task<int>(() =>
                                                                                               {
                                                                                                   while (true)
                                                                                                   {
                                                                                                       var tmpIds = Process.GetProcesses().Where(x =>x.ProcessName =="GME").Select(x =>x.Id);
                                                                                                       var newId =tmpIds.FirstOrDefault(x =>!gmeOldTaskIds.Contains(x));
                                                                                                       if (newId == 0)
                                                                                                       {
                                                                                                           Thread.Sleep(100);
                                                                                                           continue;
                                                                                                       }
                                                                                                       return newId;
                                                                                                       
                                                                                                   }
                                                                                               });
                                                              getNewGmeTask.Start();
                                                              Process.Start(o.Result);
                                                              getNewGmeTask.Wait();
                                                              PingGME(getNewGmeTask.Result);
                                                              ww.Dispatcher.Invoke((Action) (ww.Close));
                                                          });
            ww.ShowDialog();
            ww = new ConfirmEditView(this) {Owner = _owner};
            if (ww.ShowDialog() == true)
            {
                ww = new WaitView(WaitTypes.ImportToAuthoringTool) { Owner = _owner };
                Task.Factory.StartNew(() =>
                                          {
                                              var proc = new Process
                                                             {
                                                                 EnableRaisingEvents = false,
                                                                 StartInfo =
                                                                     {
                                                                         FileName = "CyPhyComponentExporterCL.exe",
                                                                         Arguments = mgaPath + " " + Path.GetDirectoryName(mgaPath),
                                                                         CreateNoWindow = true,
                                                                         UseShellExecute = false
                                                                     }
                                                             };

                                              proc.Start();
                                              proc.WaitForExit();
                                          }).ContinueWith(o =>
                                          {
                                              ww.Dispatcher.Invoke((Action)(ww.Close));

                                              // Load component

                                              // Get the new json directory
                                              // The name of the component can be changed in OpenMETA, AVMComponentName cannot be used here
                                              var exportedDirectory = Path.Combine(Path.GetDirectoryName(mgaPath), "Imported Components");

                                              // Hopefully only one component has been generated.
                                              var jsonFile = new DirectoryInfo(exportedDirectory).GetFiles("*", SearchOption.AllDirectories).FirstOrDefault(x => x.Extension == ".json");

                                              if (jsonFile != null)
                                              {
                                                  // Put the JSON file back to the original directory

                                                  var tempFileName = String.Format("{0}.json", Guid.NewGuid());
                                                  var tempFilePath = Path.Combine(Path.GetDirectoryName(AVMComponent.ComponentPath), tempFileName);


                                                  System.IO.File.Copy(jsonFile.FullName, tempFilePath);

                                                  Framework.Temp.AddTempFile(new FileInfo(tempFilePath));

                                                  //_owner.Dispatcher.Invoke((Action<string>)LoadComponent, new object[] { tempFilePath });
                                                  _owner.Dispatcher.Invoke((Action<string, string>) MergeComponent, new object[] {tempFilePath, JSONFilePath});
                                              }
                                          });
                ww.ShowDialog();
            }
            if (_cts!=null) _cts.Cancel();
            _owner.Focus();
        }

        public void ExecuteExportZip(object state)
        {
            ConfirmNoSave();
            using (var zf = new ZipFile())
            {
                zf.AddFile(JSONFilePath,"");
                var addedEntries = new HashSet<string>();
                foreach (var file in AVMComponent.AVMFiles)
                {
                    var baseUri = new Uri(JSONFilePath);
                    var relativeUri = new Uri(file.Location, UriKind.Relative);
                    var fullUri = new Uri(baseUri, relativeUri);

                    var absoluteDir = Path.GetDirectoryName(fullUri.AbsolutePath);
                    var absDirUri = new Uri(absoluteDir);

                    // Its in the root
                    if (absDirUri.LocalPath == Path.GetDirectoryName(baseUri.AbsolutePath))
                    {
                        zf.AddFile(Uri.UnescapeDataString(fullUri.AbsolutePath), "");
                        continue;
                    }


                    // It is in a subdirectory
                    relativeUri = baseUri.MakeRelativeUri(new Uri(Uri.UnescapeDataString(absDirUri.ToString())));
                    var newRelative = Uri.UnescapeDataString(relativeUri.ToString());

                    if (!addedEntries.Contains(newRelative))
                    {
                        zf.AddDirectoryByName(newRelative);
                        addedEntries.Add(newRelative);
                    }
                    zf.AddFile(Uri.UnescapeDataString(fullUri.AbsolutePath),newRelative);
                }


                var tmpFile = String.Format("{0}.zip",
                                            Path.Combine(Path.GetTempPath(),
                                                         Path.GetFileNameWithoutExtension(JSONFilePath)));

                zf.Save(tmpFile);

                var sfd = new SaveFileDialog {Title = "Save Component Package", Filter = "Zip file|*.zip"};
                
                if (sfd.ShowDialog() == true)
                {
                    System.IO.File.Copy(tmpFile, sfd.FileName);
                }
            }
        }

        public void ExecuteDiscard(object state)
        {
            if (ConfirmDiscardChanges())
                LoadComponent(JSONFilePath);

            IsComponentReloaded = false;
            _owner.Focus();
        }

        public void ExecuteSave(object state)
        {
            AVMComponent.SaveComponent();
            LoadComponent(JSONFilePath);
            IsComponentReloaded = false;
        }

        public void ExecuteLoad(object state)
        {
            ConfirmNoSave();

            var openDialog = new OpenFileDialog {Filter = "JSON files|*.json", Multiselect = false};
            if (openDialog.ShowDialog().Value)
            {
                JSONFilePath = openDialog.FileName;
                LoadComponent(JSONFilePath);

                // Collect junk files
                foreach (var junkfile in new DirectoryInfo(JSONDirectory).GetFiles("*.json",SearchOption.TopDirectoryOnly))
                {
                    Guid guid;
                    if (Guid.TryParse(Path.GetFileNameWithoutExtension(junkfile.FullName), out guid))
                    {
                        Framework.Temp.AddTempFile(junkfile);
                    }
                }
            }
        }

        public void ExecuteEditLocation(object state)
        {
            var o = state as ArtifactDescriptor;
            if (o == null) return;
            var artifact = AVMComponent.Artifacts.FirstOrDefault(x=>x==o);
            if (artifact == null) return;

            artifact.EditedName = artifact.Name;
            artifact.EditLocation = true;
        }

        private ComponentUI DeserializeComponent(string file)
        {
            return new ComponentUI(Component.DeserializeFromFile(file), file);
        }

        private void LoadComponent(string file)
        {
            var ww = new WaitView(WaitTypes.LoadComponent){Owner = _owner};

            Task.Factory.StartNew(() =>
                                      {
                                          AVMComponent = DeserializeComponent(file);
                                          AVMComponent.PropertyChanged += (o, e) =>
                                                                              {
                                                                                  if (e.PropertyName ==
                                                                                      "HasComponentNameChanged" ||
                                                                                      e.PropertyName ==
                                                                                      "HasDescriptionChanged" ||
                                                                                      e.PropertyName ==
                                                                                      "HasItarRestrictionChanged")
                                                                                  {
                                                                                      _owner.Dispatcher.Invoke((Action) (() => {
                                                                                                            SaveCommand.CanExecute(null);
                                                                                                            DiscardCommand.CanExecute(null);
                                                                                                        }));
                                                                                  }
                                                                              };

                                          _owner.Dispatcher.Invoke((Action) (() => {
                                                                AddFileCommand.CanExecute(null);
                                                                SaveCommand.CanExecute(null);
                                                                DiscardCommand.CanExecute(null);
                                                                ExportZipCommand.CanExecute(null);
                                                                ExportGmeCommand.CanExecute(null);
                                                            }));
                                      }).ContinueWith(o=> {
                                                              ww.Dispatcher.Invoke((Action) ww.Close);
                                                          });

            ww.ShowDialog();
        }

        private void MergeComponent(string newFile, string oldFile)
        {
            LoadComponent(newFile);
            var ww = new WaitView(WaitTypes.LoadComponent) { Owner = _owner };

            Task.Factory.StartNew(() =>
                                      {
                                          var oldComponent = DeserializeComponent(oldFile);
                                          AVMComponent.ComponentDescription = oldComponent.ComponentDescription;
                                          AVMComponent.ITARResctriction = oldComponent.ITARResctriction;
                                          AVMComponent.ComponentPath = oldComponent.ComponentPath;
                                          IsComponentReloaded = true;

                                      }).ContinueWith(o =>
                                                          {
                                                              ww.Dispatcher.Invoke((Action) ww.Close);
                                                              _owner.Dispatcher.Invoke((Func<bool>)(_owner.Focus));
                                                          });

            ww.ShowDialog();
            SaveCommand.CanExecute(null);
            DiscardCommand.CanExecute(null);
        }

        public void ExecuteViewArtifact(object state)
        {
            if (!(state is ArtifactDescriptor)) return;

            var ad = (ArtifactDescriptor) state;
            if (!(ad.OriginalModel is Model)) return;

            var mdv = new ModelDetailView((Model) (ad).OriginalModel) {Owner = _owner};
            mdv.ShowDialog();
        }

        public void ExecuteArtifactDiscardChange(object state)
        {
            var o = state as ArtifactDescriptor;
            if (o == null) return;
            o.EditedName = o.Name;
            o.EditLocation = false;
        }

        public void ExecuteArtifactAcceptChange(object state)
        {
            var o = state as ArtifactDescriptor;
            if (o == null) return;
            o.Name = o.EditedName;
            o.EditLocation = false;
            o.HasArtifactLocationChanged = true;
            if (o.OriginalModel is Model)
                ((Model) o.OriginalModel).Location = o.Name;
            else if (o.OriginalModel is AVM.File)
                ((AVM.File)o.OriginalModel).Location = o.Name;
            AVMComponent.RecheckLocation(o);
            AVMComponent.RefreshArtifacts();
            SaveCommand.CanExecute(null);
            DiscardCommand.CanExecute(null);
            ExportZipCommand.CanExecute(null);
        }

        public void ExecuteArtifactDblClick(object state)
        {
            //var artifact = state as ArtifactDescriptor;
            //if (artifact == null) return;

            //artifact.EditLocation = true;
            EditLocationCommand.Execute(state);
        }

        public void ExecuteDeleteArtifact(object state)
        {
            var afd = state as ArtifactDescriptor;
            if (afd != null)
            {
                AVMComponent.DeleteModelOrFile(afd.OriginalModel);
                SaveCommand.CanExecute(null);
                DiscardCommand.CanExecute(null);
                ExportZipCommand.CanExecute(null);
            }
        }

        public void ExecuteAddFile(object state)
        {
            var openDialog = new OpenFileDialog { Multiselect = true, DereferenceLinks = false};
            if (openDialog.ShowDialog().Value)
            {
                foreach (var fileName in openDialog.FileNames)
                {
                    var newFile = Path.Combine(JSONDirectory, Path.GetFileName(fileName));

                    // Add file to the JSON's directory
                    if ((System.IO.File.Exists(newFile) &&
                         MessageBox.Show(
                             String.Format("File already exists: {0}. Overwrite?", Path.GetFileName(fileName)),
                             "File already exist!", MessageBoxButton.YesNo, MessageBoxImage.Question) ==
                         MessageBoxResult.Yes) ||
                        !System.IO.File.Exists(newFile))
                    {
                        try
                        {
                            System.IO.File.Copy(fileName, newFile, true);
                            var hash = string.Empty;
                            using (var fs = new FileStream(newFile, FileMode.Open, FileAccess.Read))
                            {
                                var cryptoService = new MD5CryptoServiceProvider();
                                var byteHash = cryptoService.ComputeHash(fs);
                                var sb = new StringBuilder();
                                for (var i = 0; i < byteHash.Length; i++)
                                {
                                    sb.Append(byteHash[i].ToString("x2"));
                                }
                                hash = sb.ToString();
                            }
                            AVMComponent.AddFile(Path.GetFileName(fileName), hash);
                            SaveCommand.CanExecute(null);
                            DiscardCommand.CanExecute(null);
                        }
                        catch (IOException e)
                        {
                            MessageBox.Show(e.Message, "Error!", MessageBoxButton.OK, MessageBoxImage.Error);
                        }
                    }

                }
            }
        }

        #endregion

        #region Rights

        public bool CanSaveExecute(object s)
        {
            if (AVMComponent == null) return false;
            return AVMComponent.HasChanged || AVMComponent.Artifacts.Any(x => x.HasArtifactChanged) || IsComponentReloaded;
        }

        public bool CanAddExecute(object s)
        {
            return AVMComponent != null;
        }

        public bool CanExecuteExportZip(object s)
        {
            return AVMComponent != null && AVMComponent.Artifacts.Where(x => x.ArtifactType == ArtifactTypes.GenericFile).All(x => x.ArtifactStatus == ArtifactStatus.Ok);
        }

        public bool CanExecuteChange(object s)
        {
            return SelectedArtifact != null && SelectedArtifact.ArtifactType == ArtifactTypes.GenericFile;
        }

        public bool CanExecuteDelete(object state)
        {
            return SelectedArtifact != null;
        }

        #endregion

        #endregion

        private string JSONDirectory { get; set; }

        private string _jsonFilePath;
        public string JSONFilePath
        {
            get { return _jsonFilePath; }
            set
            {
                _jsonFilePath = value;
                JSONDirectory = Path.GetDirectoryName(JSONFilePath);
                OnPropertyChanged(() => JSONFilePath);
            }
        }

        private ComponentUI _avmComponent;
        public ComponentUI AVMComponent
        {
            get { return _avmComponent; }
            set
            {
                _avmComponent = value;
                OnPropertyChanged(() => AVMComponent);
            }
        }

        

        private ArtifactDescriptor _selectedArtifact;
        public ArtifactDescriptor SelectedArtifact
        {
            get { return _selectedArtifact; }
            set
            {
                _selectedArtifact = value;
                OnPropertyChanged(() => SelectedArtifact);
                ChangeTypeCommand.CanExecute(null);
                DeleteArtifactCommand.CanExecute(null);
            }
        }

        public ComponentAuthoringViewModel(Window owner)
        {
            _owner = owner;

            LoadCommand                  = new DelegateCommand(ExecuteLoad, o => true);
            AddFileCommand               = new DelegateCommand(ExecuteAddFile, CanAddExecute);
            SaveCommand                  = new DelegateCommand(ExecuteSave, CanSaveExecute);
            DiscardCommand               = new DelegateCommand(ExecuteDiscard, CanSaveExecute);
            ExportZipCommand             = new DelegateCommand(ExecuteExportZip, CanExecuteExportZip);
            ExportGmeCommand             = new DelegateCommand(ExecuteExportGme, CanExecuteExportZip);
            ChangeTypeCommand            = new DelegateCommand(ExecuteChangeType, CanExecuteChange);
            ViewArtifactCommand          = new DelegateCommand(ExecuteViewArtifact, o=>true);
            EditLocationCommand          = new DelegateCommand(ExecuteEditLocation, o=>true);
            DiscardArtifactChangeCommand = new DelegateCommand(ExecuteArtifactDiscardChange, o => true);
            AcceptArtifactChangeCommand  = new DelegateCommand(ExecuteArtifactAcceptChange, o => true);
            ArtifactDblClick             = new DelegateCommand(ExecuteArtifactDblClick, o => true);
            ConfirmExitCommand           = new DelegateCommand(ExecuteConfirmExit, o => true);
            DeleteTempCommand            = new DelegateCommand(ExecuteDeleteTemp, o=>true);
            DeleteArtifactCommand = new DelegateCommand(ExecuteDeleteArtifact, CanExecuteDelete);

            IsGmeRunning = false;
        }

        
    }
}
