/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "stdafx.h"
#include "ConnectorAssist.h"
#include <Console.h>

using namespace std;
using namespace BON;

string getKind(const BON::Object& model) {
	string infoString = model->getInfoString(Util::IO_Meta);
	int kindStart = infoString.find("[name: ") + 7;
	int kindEnd = infoString.find_first_of("]",kindStart);
	return infoString.substr(kindStart,kindEnd-kindStart);
};
string getGUID(const BON::Object& obj) {
	CComBSTR bstrGUID;
	COMTHROW( obj->getObjectI()->GetGuidDisp( &bstrGUID ) );
	return Util::Copy( bstrGUID );
};
struct ConnInfo {
	Connection c;
	string cKind;
	ConnectionEnd ce;
	string role;
};
set<ConnInfo*> getConnected(const BON::FCO& fco) {
	set<ConnInfo*> rtn;
	set<Connection> conns = fco->getConnLinks();
	
	for (set<Connection>::const_iterator i = conns.begin(); i != conns.end(); i++) {
		ConnInfo* ci = new ConnInfo;
		ci->c = *i;
		ci->cKind = getKind(*i);
		ci->ce = (*i)->getSrc();
		ci->role = "src";
		if (ci->ce == fco) {
			ci->ce = (*i)->getDst();
			ci->role = "dst";
		}
		rtn.insert(ci);
	}
	return rtn;
};
struct connSpec {
	FCO srcCE;
	FCO dstCE;
	string kind;
};
set<connSpec*> getDerivedConnSpecs(const BON::Connection& conn ) {
	set<connSpec*> rtn;
	
	// Now, let's get both destinations.
	ConnectionEnd srcCE(conn->getSrc()), dstCE(conn->getDst());
	Model src(srcCE), dst(dstCE);
	
	Model parent = FCO(conn)->getParentModel();
	
	// Okay, let's take SRC and see what the component ports are.
	set<FCO> srcKids = src->getChildFCOs();
	set<FCO> dstKids = dst->getChildFCOs();
	for (set<FCO>::const_iterator i = srcKids.begin(); i != srcKids.end(); i++) {
		FCO sPort(*i);
		
		// Alright, let's do the same for the dst and try to find a matching port.
		FCO dPort;
		for (set<FCO>::const_iterator j = dstKids.begin(); !dPort && j != dstKids.end(); j++) {
			FCO dPort_(*j);
			if (dPort_->getName() == sPort->getName())
				dPort = dPort_;
		}
		
		// Okay, now we have ports on both sides. Let's see what's connected to them.
		set<ConnInfo*> srcConns = getConnected(sPort);
		set<ConnInfo*> dstConns = getConnected(dPort);
		for (set<ConnInfo*>::const_iterator j = srcConns.begin(); j != srcConns.end(); j++) {
			ConnInfo* cij(*j);
			for (set<ConnInfo*>::const_iterator k = dstConns.begin(); k != dstConns.end(); k++) {
				ConnInfo* cik(*k);
				// Okay, now create an entry
				
				if (cij->role == "src" && cik->role == "dst" && (cij->cKind == cik->cKind)) {
					connSpec* cs = new connSpec;
					cs->srcCE = cij->ce;
					cs->dstCE = cik->ce;
					cs->kind = cij->cKind;
					rtn.insert(cs);
				}
				if (cik->role == "src" && cij->role == "dst" && (cij->cKind == cik->cKind)) {
					connSpec* cs = new connSpec;
					cs->srcCE = cik->ce;
					cs->dstCE = cij->ce;
					cs->kind = cij->cKind;
					rtn.insert(cs);
				}
			}
		}
	}	
	
	return rtn;
};
Connection findConnFromSpec(const connSpec* cs) {
	Model parent = FCO(cs->srcCE)->getParentModel()->getParentModel();
	if (parent != FCO(cs->dstCE)->getParentModel()->getParentModel())
		return NULL;
	
	set<Connection> cKids = parent->getChildConnections();
	for (set<Connection>::const_iterator i = cKids.begin(); i != cKids.end(); i++) {
		ConnectionEnd src = (*i)->getSrc();
		ConnectionEnd dst = (*i)->getDst();
		string kind = getKind(*i);
		if (   kind == cs->kind
			&& ( (src == cs->srcCE && dst == cs->dstCE) 
			    || (dst == cs->srcCE && src == cs->dstCE)))
		{
			return *i;
		}
	}
	
	return NULL;
};
void processComponentAssembly(const Model& ca) {
	// Current scope: Just the current view.
	// Get all child connections.
	map<Connection,bool> cIsSupported;
	set<Connection> connSet = ca->getChildConnections();
	for (set<Connection>::const_iterator i = connSet.begin(); i != connSet.end(); i++) {
		string connKind = getKind(*i);
		
		// For those that are "ConnectorComposition," check for the necessary derived connections.
		if (connKind == "ConnectorComposition") {
			Connection ci(*i);
			set<connSpec*> csSet = getDerivedConnSpecs(ci);
			Model parent(ci->getParent());
			for (set<connSpec*>::const_iterator i = csSet.begin(); i != csSet.end(); i++) {
				connSpec* csi(*i);
				Connection c = findConnFromSpec(csi);
				if (c) {
					// This connection already exists, no need to create it.
					// But let's claim it anyway.
					c->getRegistry()->setValueByPath("/lineType","dash");
					c->getRegistry()->setValueByPath("/META/ConnectorAssist/derivedFrom",getGUID(ci));
				} else {
					c = Connection::create( parent, csi->srcCE , csi->dstCE , csi->kind );
					if (c) {
						c->getRegistry()->setValueByPath("/lineType","dash");
						c->getRegistry()->setValueByPath("/META/ConnectorAssist/derivedFrom",getGUID(ci));
					}
				}
				if (c)
					cIsSupported[c] = true;
				delete *i;
			}
		}
	}

	// For connections that are not "supported," check to see if they are marked as derived. If so, delete them.
	for (set<Connection>::const_iterator i = connSet.begin(); i != connSet.end(); i++) {
		Connection ci(*i);
		if ( cIsSupported[ci] == false && ci->getRegistry()->getValueByPath("/META/ConnectorAssist/derivedFrom") != "")
			ci->destroy();
	}
};