/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "AssemblyOutputWriter.h"
#include "string_utils.h"
#include "CommonHelper.h"
#include "UdmGme.h"
#include "UdmFormatter.h"
#include "VersionNo.h"
#include "CommonTraverse.h"
#include "CommonData.h"
#include <boost/filesystem/path.hpp>
#include <boost/algorithm/string.hpp>

#include "Json_Helper.h"
#include "ScriptWriter.h"

#include <time.h>
#include <fstream>


const string AssemblyOutputWriter::kreg_config_id = "DB_Config_ID";
const string AssemblyOutputWriter::kmat_al = "Aluminum";
const string AssemblyOutputWriter::kmat_steel = "Steel";
const string AssemblyOutputWriter::kmat_ceramic = "Ceramic";
const string AssemblyOutputWriter::kmat_poly = "Polymer";
const string AssemblyOutputWriter::kprop_thermal_exp =  "CoefficientOfThermalExpansion";
const string AssemblyOutputWriter::kprop_density =	"Density";
const string AssemblyOutputWriter::kprop_mod_elasticity = "ModulusOfElasticity";
const string AssemblyOutputWriter::kprop_tensile_ultm = "TensileUltimateStrength";
const string AssemblyOutputWriter::kprop_tensile_yield = "TensileYieldStrength";
const string AssemblyOutputWriter::kprop_shear_mod = "ShearModulus";
const string AssemblyOutputWriter::kprop_poisson = "PoissonRatio";
const string AssemblyOutputWriter::kprop_shear_strength = "ShearStrength";
const string AssemblyOutputWriter::kprop_bearing_ultm = "BearingUltimateStrength";
const string AssemblyOutputWriter::kprop_bearing_yield = "BearingYieldStrength";
const string AssemblyOutputWriter::kprop_fatigue_strength = "FatigueStrength";
const string AssemblyOutputWriter::kprop_fatigue_cycles = "FatigueCycles";
const string AssemblyOutputWriter::kprop_min_melt_pt = "MinMeltingPoint";
const string AssemblyOutputWriter::kprop_heat_capacity = "HeatCapacity";
const string AssemblyOutputWriter::kprop_thermal_conductivity = "ThermalConductivity";
const string AssemblyOutputWriter::kprop_fracture_toughness = "FractureToughness";
const string AssemblyOutputWriter::kprop_max_safe_op_temp = "MaxSafeOperatingTemperature";
const string AssemblyOutputWriter::kprop_flexural_strength = "FlexuralStrength";
const string AssemblyOutputWriter::kprop_vicatb_soft_pt = "VicatBSofteningPoint";
const string AssemblyOutputWriter::kprop_flame_rating = "FlammabilityRating";
const string AssemblyOutputWriter::kprop_heat_deflect_temp = "HeatDeflectionTemp";
const string AssemblyOutputWriter::kprop_heat_deflect_load = "HeatDeflectionLoad";
const map<string,string> AssemblyOutputWriter::m_materialUnitLookup =  AssemblyOutputWriter::CreateMaterialUnitLookupTable(); 

 
void AssemblyOutputWriter::WriteGraph(CAssembly* const topAssembly_in, 
								      bool directed_in)
{
	string name = CleanString((string)topAssembly_in->CyPhyRef.name()) + "_" + topAssembly_in->GetID_Str();
	if (directed_in)
		name += "_Directed";	
	string fullName = Globals::Instance()->DebugLogDirectory + "\\CyPhy2CAD_" + name + ".gv";		//string fullName = this->outputDirectoryName + "\\debug_log\\" + name + ".gv";

	FileChannel file(Globals::Instance()->DebugLogDirectory, "CyPhy2CAD_" + name, ".gv");			//FileChannel file(this->outputDirectoryName + "\\debug_log\\", name, ".gv");
	file.Stream()<<"digraph G {\n";
	file.Stream()<<"node[style=filled];\n";
	topAssembly_in->PrintGraphviz(file.Stream(), directed_in);
	file.Stream()<<"}\n";
	std::string entry = "\"dot -Tgif -O \""+ fullName + "\"\"";	
	system(entry.c_str());
}

void AssemblyOutputWriter::WriteGraph(list<CAssembly*> &topAssemblies_in, 
									  vector<CComponent*> &orphans_in, 
									  bool directed_in)
{
	string filename = "CyPhy2CAD_Assembly_Graphviz";
	if (directed_in)
		filename += "_Directed.gv";
	else
		filename += ".gv";

	string fullpath = Globals::Instance()->DebugLogDirectory + "\\" + filename;		

	{	
		FileChannel file(Globals::Instance()->DebugLogDirectory, filename);			
		file.Stream()<<"digraph G {\n";
		file.Stream()<<"node[style=filled];\n";		

		for (list<CAssembly*>::iterator i = topAssemblies_in.begin(); i != topAssemblies_in.end(); i++)
		{
			(*i)->PrintGraphviz(file.Stream(), directed_in);
		}

		for (unsigned int i = 0; i < orphans_in.size(); i++)
		{
			orphans_in[i]->PrintGraphviz(file.Stream());
		}

		file.Stream()<<"}\n";
	}

	std::string entry = "\"dot -Tgif -O \""+ fullpath + "\"\"";	
	system(entry.c_str());
}

void AssemblyOutputWriter::WriteAssemblyInterfaceFile(list<CAssembly*> &roots, vector<CComponent*> &orphans_in)
{
	Udm::SmartDataNetwork dn_assemblyInterface(AssemblyInterface::diagram);
	std::string outFile = "\\" + CleanString(this->assemblyName, 300) + "_Cad.xml"; 

	dn_assemblyInterface.CreateNew( this->outputDirectoryName+outFile, "AssemblyInterface", AssemblyInterface::Assemblies::meta, Udm::CHANGES_LOST_DEFAULT);
	AssemblyInterface::Assemblies TargetRootParent = AssemblyInterface::Assemblies::Cast(dn_assemblyInterface.GetRootObject()); 

	for (list<CAssembly*>::iterator i = roots.begin(); i != roots.end(); i++)
	{
		AssemblyInterface::Assembly TargetRoot = AssemblyInterface::Assembly::Create(TargetRootParent);
		TargetRoot.ConfigurationID() = (*i)->GetAssemblyInterface_ID();
		AssemblyInterface::Properties properties = AssemblyInterface::Properties::Create(TargetRoot);
		AssemblyInterface::Units unit = AssemblyInterface::Units::Create(properties);
		unit.Value() = "value1";
		WriteAssemblyInterfaceFile(*i, TargetRoot);
	}

	CreateMaterials(this->materialsUsed, TargetRootParent);
	if (!orphans_in.empty())
		CreateOrphanComponents(orphans_in, TargetRootParent);


	if (!Globals::Instance()->m_selectedStepFormats.empty())
	{
		AssemblyInterface::DataExchange dataExchange = AssemblyInterface::DataExchange::Create(TargetRootParent);
		for (set<string>::const_iterator ci = Globals::Instance()->m_selectedStepFormats.begin(); ci != Globals::Instance()->m_selectedStepFormats.end(); ci++)
		{
			AssemblyInterface::STEPFormat step = AssemblyInterface::STEPFormat::Create(dataExchange);
			step.Name() = *ci;
		}
	}

	dn_assemblyInterface.CloseWithUpdate();			
			
	std::string cleanName = CleanString(this->assemblyName, 300);
	this->WriteAssemblyBatFile(cleanName);

	LOGMSG("Generated CAD Xml File in: " + this->outputDirectoryName, MSG_INFO);
	LOGMSG("Debug log files at: " + Globals::Instance()->DebugLogDirectory, MSG_INFO);

	GenerateExecutionScripts();
}


// Note: Used by WriteAssemblyInterfaceFile(list<CAssembly*> roots) and WriteAssemblyInterfaceFile(CAssembly* root, CyPhyML::TestBench &testBench)
void AssemblyOutputWriter::WriteAssemblyInterfaceFile(CAssembly* cassemblyParent, Udm::Object& AIParent)
{
	Udm::Object udmAssemblyParent = cassemblyParent->GetMyUDMObj();
	Uml::Class type = udmAssemblyParent.type();
	AssemblyInterface::CADComponent cadComponentTop = AssemblyInterface::CADComponent::Create(AIParent);
	m_AllAICADComponents[udmAssemblyParent.uniqueId()] = cadComponentTop;

	if (type == CyPhyML::CADTestBench::meta)
		this->FillInAttributes(CyPhyML::CADTestBench::Cast(udmAssemblyParent), cadComponentTop);		
	else
		this->FillInAttributes(cassemblyParent, cadComponentTop);	


	for (unsigned int i = 0; i < cassemblyParent->child_Component.size(); i++)
	{
		CComponent* ccomponent = cassemblyParent->child_Component[i];
		AssemblyInterface::CADComponent cadComponent = AssemblyInterface::CADComponent::Create(cadComponentTop);
		Udm::Object udmObj = ccomponent->GetMyUDMObj();		
		m_AllAICADComponents[udmObj.uniqueId()] = cadComponent;
		this->FillInAttributes(ccomponent, cadComponent);	

		////////////////////////////////////////
		// TODO: 7-26-2012 ADD ROOT CONSTRAINT
		////////////////////////////////////////
		if (ccomponent->GetID() == cassemblyParent->GetRootComponentID())
		{
			AddRootConstraint2CADComponent(cadComponent, cadComponentTop.ComponentID());
		}

	}

	// size-2-fit
	for (unsigned int i = 0; i < cassemblyParent->child_Size2Fits.size(); i++)
	{
		CComponentSize2Fit* ccomponent = cassemblyParent->child_Size2Fits[i];
		AssemblyInterface::CADComponent cadComponent = AssemblyInterface::CADComponent::Create(cadComponentTop);
		Udm::Object udmObj = ccomponent->GetMyUDMObj();		
		m_AllAICADComponents[udmObj.uniqueId()] = cadComponent;
		this->FillInAttributes(ccomponent, cadComponent);		
	}


	for (unsigned int i = 0; i < cassemblyParent->child_Assembly.size(); i++)
	{	
		CAssembly* cassembly = cassemblyParent->child_Assembly[i];
		if (cassembly->enabled)
			this->WriteAssemblyInterfaceFile(cassembly, cadComponentTop);          // further create for subassemblies
	}

	// 11-28-2012: Replaced the below commented out code
	AddCADComponentConstraints(cassemblyParent->child_Edges);
	AddCADComponentConstraints(cassemblyParent->child_Size2FitEdges);

#if 0
	for (unsigned int i = 0; i < cassemblyParent->child_Edges.size(); i++)
	{
		CEdge* curEdge = cassemblyParent->child_Edges[i];
		CADData* destination = curEdge->destination;
		CADData* source = curEdge->source;
		CyPhyML::JoinStructures edge_cyphyRef = curEdge->CyPhyRef;

		Udm::Object src_udmObj;
		Udm::Object dst_udmObj;	

		if (source->GetType() == "Forwarder")
			src_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(source->GetMyUDMObj()).parent();
		else
			src_udmObj = source->GetMyUDMObj();

		if (destination->GetType() == "Forwarder")
			dst_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(destination->GetMyUDMObj()).parent();
		else
			dst_udmObj = destination->GetMyUDMObj();

		long cnt_Src_ID = src_udmObj.uniqueId(), cnt_Dst_ID = dst_udmObj.uniqueId();	

		AssemblyInterface::CADComponent constraintCadComponentContainer;
		map<long, AssemblyInterface::CADComponent>::iterator k = m_AllAICADComponents.find(cnt_Dst_ID);
		if (k == m_AllAICADComponents.end())
		{
			string identifier;
			to_string(identifier, cnt_Dst_ID);
			LOGMSG("Can't find AssemblyInterface::CADComponent with ID [" + identifier + "] in m_AllAICADComponents!", MSG_ERROR);
				continue;
		}
		else
			constraintCadComponentContainer = k->second;

		std::map<CyPhyML::JoinStructures, pair<CyPhyML::StructuralInterface, CyPhyML::StructuralInterface>>::iterator si = CommonData::JS_SI_Lookup_Table.find(edge_cyphyRef);
		if (si == CommonData::JS_SI_Lookup_Table.end())
			continue;

		AddConstraint2CADComponent(constraintCadComponentContainer, si->second.first, si->second.second);
	}


	// size-2-fit
	for (unsigned int i = 0; i < cassemblyParent->child_Size2FitEdges.size(); i++)
	{
		CEdge* curEdge = cassemblyParent->child_Size2FitEdges[i];
		CADData* destination = curEdge->destination;
		CADData* source = curEdge->source;
		CyPhyML::JoinStructures edge_cyphyRef = curEdge->CyPhyRef;

		Udm::Object src_udmObj;
		Udm::Object dst_udmObj;	

		if (source->GetType() == "Forwarder")
			src_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(source->GetMyUDMObj()).parent();
		else
			src_udmObj = source->GetMyUDMObj();

		if (destination->GetType() == "Forwarder")
			dst_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(destination->GetMyUDMObj()).parent();
		else
			dst_udmObj = destination->GetMyUDMObj();

		long cnt_Src_ID = src_udmObj.uniqueId(), cnt_Dst_ID = dst_udmObj.uniqueId();	

		AssemblyInterface::CADComponent constraintCadComponentContainer;
		map<long, AssemblyInterface::CADComponent>::iterator k = m_AllAICADComponents.find(cnt_Dst_ID);
		if (k == m_AllAICADComponents.end())
		{

			string identifier;
			to_string(identifier, cnt_Dst_ID);
			LOGMSG("Can't find AssemblyInterface::CADComponent with ID [" + identifier + "] in m_AllAICADComponents!", MSG_ERROR);
				continue;
		}
		else
			constraintCadComponentContainer = k->second;

		std::map<CyPhyML::JoinStructures, pair<CyPhyML::StructuralInterface, CyPhyML::StructuralInterface>>::iterator si = CommonData::JS_SI_Lookup_Table.find(edge_cyphyRef);
		if (si == CommonData::JS_SI_Lookup_Table.end())
			continue;
		AddConstraint2CADComponent(constraintCadComponentContainer, si->second.first, si->second.second);
	}
#endif
}

void AssemblyOutputWriter::AddCADComponentConstraints(const vector<CEdge*> &edges_in)
{
	for (unsigned int i = 0; i < edges_in.size(); i++)
	{
		CEdge* curEdge = edges_in[i];
		CADData* destination = curEdge->destination;
		CADData* source = curEdge->source;
		CyPhyML::JoinStructures edge_cyphyRef = curEdge->CyPhyRef;

		Udm::Object src_udmObj, dst_udmObj;	
		string source_type = source->GetType(), dest_type = destination->GetType();

		if (source_type == "Forwarder")
			src_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(source->GetMyUDMObj()).parent();
		else
			src_udmObj = source->GetMyUDMObj();

		if (dest_type == "Forwarder")
			dst_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(destination->GetMyUDMObj()).parent();
		else
			dst_udmObj = destination->GetMyUDMObj();

		long cnt_Src_ID = src_udmObj.uniqueId(), cnt_Dst_ID = dst_udmObj.uniqueId();	

		AssemblyInterface::CADComponent constraintCadComponentContainer;
		map<long, AssemblyInterface::CADComponent>::iterator k = m_AllAICADComponents.find(cnt_Dst_ID);
		if (k == m_AllAICADComponents.end())
		{
			string identifier;
			to_string(identifier, cnt_Dst_ID);
			LOGMSG("Can't find AssemblyInterface::CADComponent with ID [" + identifier + "] in m_AllAICADComponents!", MSG_ERROR);
				continue;
		}
		else
			constraintCadComponentContainer = k->second;

		std::map<CyPhyML::JoinStructures, pair<CyPhyML::StructuralInterface, CyPhyML::StructuralInterface>>::iterator si = CommonData::JS_SI_Lookup_Table.find(edge_cyphyRef);
		if (si == CommonData::JS_SI_Lookup_Table.end())
			continue;

		if (source_type == "Component")

		AddConstraint2CADComponent(constraintCadComponentContainer, si->second.first, si->second.second);
	}
}

void AssemblyOutputWriter::WriteAssemblyInterfaceFile(list<CAssembly*> &roots_in, 
													vector<CComponent*> & orphans_in, 
													CyPhyML::TestBench &testbench_in)
{
	string tbName = CleanString(std::string(testbench_in.name()), 300);
	Udm::SmartDataNetwork dn_assemblyInterface(AssemblyInterface::diagram);
	std::string outFile = "\\" + tbName + "_Cad.xml"; 

	dn_assemblyInterface.CreateNew( this->outputDirectoryName+outFile, "AssemblyInterface", AssemblyInterface::Assemblies::meta, Udm::CHANGES_LOST_DEFAULT);
	AssemblyInterface::Assemblies TargetRootParent = AssemblyInterface::Assemblies::Cast(dn_assemblyInterface.GetRootObject()); 

	AssemblyInterface::Assembly metricParent;
	for (list<CAssembly*>::iterator i = roots_in.begin(); i != roots_in.end(); i++)
	{
		AssemblyInterface::Assembly TargetRoot = AssemblyInterface::Assembly::Create(TargetRootParent);
		TargetRoot.ConfigurationID() = (*i)->GetAssemblyInterface_ID();
		AssemblyInterface::Properties properties = AssemblyInterface::Properties::Create(TargetRoot);
		AssemblyInterface::Units unit = AssemblyInterface::Units::Create(properties);
		unit.Value() = "value1";
		WriteAssemblyInterfaceFile(*i, TargetRoot);
		metricParent = TargetRoot;
	}

	if (!orphans_in.empty())
		CreateOrphanComponents(orphans_in, TargetRootParent);
	
	// Test Bench Specific Generation here.
	if (roots_in.size() == 1 && orphans_in.empty())
		CreateTestBenchMetrics(metricParent, testbench_in, metricParent.ConfigurationID());

	CreateTestBenchPostProcessScripts(testbench_in);


	if (!Globals::Instance()->m_selectedStepFormats.empty())
	{
		AssemblyInterface::DataExchange dataExchange = AssemblyInterface::DataExchange::Create(TargetRootParent);
		for (set<string>::const_iterator ci = Globals::Instance()->m_selectedStepFormats.begin(); ci != Globals::Instance()->m_selectedStepFormats.end(); ci++)
		{
			AssemblyInterface::STEPFormat step = AssemblyInterface::STEPFormat::Create(dataExchange);
			step.Name() = *ci;
		}
	}

	dn_assemblyInterface.CloseWithUpdate();			
			
	this->WriteAssemblyBatFile(tbName);

	LOGMSG("Generated CAD Xml File in: " + this->outputDirectoryName, MSG_INFO);
	LOGMSG("Debug log files at: " + Globals::Instance()->DebugLogDirectory, MSG_INFO);
	GenerateExecutionScripts();
}

void AssemblyOutputWriter::WriteAssemblyInterfaceFile(CAssembly *root, 
													CyPhyML::CADTestBenchType &testBench)
{
	string id = GetRegistryValue(testBench, AssemblyOutputWriter::kreg_config_id);
	if (id == "")
		to_string(id, testBench.uniqueId());

	string tbName = CleanString(std::string(testBench.name()), 300);
	Udm::SmartDataNetwork dn_assemblyInterface(AssemblyInterface::diagram);
	std::string outFile = "\\" + tbName + "_Cad.xml"; 

	dn_assemblyInterface.CreateNew( this->outputDirectoryName+outFile, "AssemblyInterface", AssemblyInterface::Assemblies::meta, Udm::CHANGES_LOST_DEFAULT);
	AssemblyInterface::Assemblies TargetRootParent = AssemblyInterface::Assemblies::Cast(dn_assemblyInterface.GetRootObject());  //AssemblyInterface::Assembly TargetRoot = AssemblyInterface::Assembly::Cast(dn_assemblyInterface.GetRootObject());
	AssemblyInterface::Assembly TargetRoot = AssemblyInterface::Assembly::Create(TargetRootParent);  
	TargetRoot.ConfigurationID() = id;																							// temporary
	AssemblyInterface::Properties properties = AssemblyInterface::Properties::Create(TargetRoot);
	AssemblyInterface::Units unit = AssemblyInterface::Units::Create(properties);
	unit.Value() = "value1";

	WriteAssemblyInterfaceFile(root, TargetRoot);	

	// Test Bench Specific Generation here.
	Uml::Class tbType = testBench.type();
	CreateMaterials(this->materialsUsed, TargetRootParent);
	CreateFEAAnalysisInput(CyPhyML::CADTestBench::Cast(testBench), TargetRoot);


	if (!Globals::Instance()->m_selectedStepFormats.empty())
	{
		AssemblyInterface::DataExchange dataExchange = AssemblyInterface::DataExchange::Create(TargetRootParent);
		for (set<string>::const_iterator ci = Globals::Instance()->m_selectedStepFormats.begin(); ci != Globals::Instance()->m_selectedStepFormats.end(); ci++)
		{
			AssemblyInterface::STEPFormat step = AssemblyInterface::STEPFormat::Create(dataExchange);
			step.Name() = *ci;
		}
	}

	dn_assemblyInterface.CloseWithUpdate();			
			
	this->WriteAssemblyBatFile(tbName);

	LOGMSG("Generated CAD Xml File in: " + this->outputDirectoryName, MSG_INFO);
	LOGMSG("Debug log files at: " + Globals::Instance()->DebugLogDirectory, MSG_INFO);
	GenerateExecutionScripts();
}



void AssemblyOutputWriter::CreateTestBenchMetrics(Udm::Object &parent_in, 
												CyPhyML::TestBench &testBench_in, 
												string topCompID_in)
{	
	AssemblyInterface::Computations aiComputations;
	set<CyPhyML::Metric> metric_set = testBench_in.Metric_kind_children();
	for (set<CyPhyML::Metric>::const_iterator ci = metric_set.begin(); ci != metric_set.end(); ci++)
	{
		CyPhyML::Metric metric(*ci);
		string mid = GetGMEGUID(metric);

		set<CyPhyML::CADComputation2Metric> comp_set = ci->srcCADComputation2Metric();
		set<CyPhyML::PointCoordinates2Metric> pt_set = ci->srcPointCoordinates2Metric();
		int size = comp_set.size() + pt_set.size();
		if (size > 1)
		{
			LOGMSG("Metric can only be connected to 1 CAD Computation port or 1 Analysis Point [" + (string)ci->name() + "]", MSG_ERROR);
			continue;
		}

		if (aiComputations == Udm::null)
			aiComputations = AssemblyInterface::Computations::Create(parent_in);

		for (set<CyPhyML::CADComputation2Metric>::const_iterator di = comp_set.begin(); di != comp_set.end(); di++)
		{
			CyPhyML::CADComputationType computation = di->srcCADComputation2Metric_end();
			Uml::Class type = computation.type();
			if (type == CyPhyML::BoundingBox::meta)
			{
				AssemblyInterface::BoundingBox bb = AssemblyInterface::BoundingBox::Create(aiComputations);					
				bb.ComponentID() = topCompID_in;
				bb.MetricID() = mid;
				bb.RequestedValueType() = CyPhyML::BoundingBox::Cast(computation).CADComputationRequestedValue();

			}
			else if (type == CyPhyML::CenterOfGravity::meta)
			{
				AssemblyInterface::CenterOfGravity cg = AssemblyInterface::CenterOfGravity::Create(aiComputations);
				cg.ComponentID() = topCompID_in;
				cg.MetricID() = mid;
				cg.RequestedValueType() = CyPhyML::CenterOfGravity::Cast(computation).CADComputationRequestedValue();
			}
			else if (type == CyPhyML::Mass::meta)
			{
				AssemblyInterface::Mass cg = AssemblyInterface::Mass::Create(aiComputations);
				cg.ComponentID() = topCompID_in;
				cg.MetricID() = mid;
				cg.RequestedValueType() = "";
			}
		}

		for (set<CyPhyML::PointCoordinates2Metric>::const_iterator di = pt_set.begin(); di != pt_set.end(); di++)
		{
			CyPhyML::AnalysisPoint ap = di->srcPointCoordinates2Metric_end();
			AnalysisPoint_FeaturePoint_Map::const_iterator ei = CommonData::geometryConn_AnalysisPoint_table.find(ap);
			if (ei != CommonData::geometryConn_AnalysisPoint_table.end())
			{
				AssemblyInterface::PointCoordinates pt = AssemblyInterface::PointCoordinates::Create(aiComputations);
				pt.FeatureDatumName() = ei->second.Datum();
				pt.ComponentID() = GetFeaturePointComponentParentID(ei->second);
				pt.MetricID() = mid;
				pt.RequestedValueType() = "Vector";
			}
		}
	}
}

void AssemblyOutputWriter::CreateOrphanComponents(vector<CComponent*> &orphans_in, Udm::Object &parent_in)
{
	AssemblyInterface::UnassembledComponents unassembledTag = AssemblyInterface::UnassembledComponents::Create(parent_in);
	for (unsigned int i = 0; i < orphans_in.size(); i++)
	{
		AssemblyInterface::CADComponent cadComponent = AssemblyInterface::CADComponent::Create(unassembledTag);
		FillInAttributes(orphans_in[i], cadComponent);
	}
}


void AssemblyOutputWriter::WriteAssemblyBatFile(std::string fileName)
{
	std::string outFile = this->outputDirectoryName + "\\" + fileName + "_Cad.bat";
	Globals::Instance()->RunCommand = fileName + "_Cad.bat";
	std::ofstream file (outFile.c_str());

	if (!file.is_open())
	{
		LOGMSG("Could not start a new bat file." + outFile, MSG_ERROR);
		return;
	}

	bool Automation = Globals::Instance()->Automation;

	file<< "REM	This file was created by CyPhy2CAD.dll "<<PRODUCT_VERSION_WITH_v_AND_DOTS << "\n";
	file<< "REM 	Date:"<< GetTimeStamp() << "\n";
	file<< "REM\n";
	file<< "REM	The following system environment variable must be set:\n";
	file<< "REM	    PROE_ISIS_EXTENSIONS	// typically set to C:\\Program Files\\META\\Proe ISIS Extensions\n";
	file<< "REM\n";
	file<< "REM	See \"C:\\Program Files\\META\\Proe ISIS Extensions\\0Readme - CreateAssembly.txt\" for the complete setup instructions." << "\n\n";

	file<< "set PARTS_DIR=\".\\Cad_Components_Directory\"" << "\n";				//file<< "set PARTS_DIR=\"" << this->cadFileDirectoryName << "\"" << "\n";			// 11/5/12: Prob going to be deprecated later
	file<< "set WORKING_DIR=\".\""<<"\n";	// file<< "set WORKING_DIR=\""<<this->outputDirectoryName<<"\""<<"\n";				// 11/6/12

	file<< "set ERROR_CODE=0" << "\n\n";
	file<< "set ERROR_MSG=\"\"" << "\n\n";


	file<< "\n\nRem ****************************\n";
	file<< "REM Create Creo Assembly\n";
	file<< "Rem ****************************\n\n";
	file<< "set EXE_FILE_NAME=CADCreoParametricCreateAssembly.exe" << "\n";
	file<< "set EXE=\"%PROE_ISIS_EXTENSIONS%\\bin\\%EXE_FILE_NAME%\"" << "\n";	
	file<< "\nset ASSEMBLY_XML_FILE=\"" << fileName <<"_Cad.xml\""<<"\n";	
	file<< "set LOG_FILE=%ASSEMBLY_XML_FILE%.log" << "\n";

	if (Automation)
		file<< "set EXIT_PROMPT=\"NO\"" << "\n\n";
	else
		file<< "set EXIT_PROMPT=\"YES\"" << "\n\n";

	file<<"if exist %EXE% goto  :EXE_FOUND"<<"\n";
	file<<"@echo off"<<"\n";
	file<<"echo		Error: Could not find %EXE_FILE_NAME%."<<"\n";  
	file<<"echo		Your system is not properly configured to run %EXE_FILE_NAME%."<<"\n";
	file<<"echo		Please see For instructions on how to configure your system, please see \"0Readme - CreateAssembly.txt\""<<"\n"; 
	file<<"echo		which is typically located at \"C:\\Program Files\\META\\Proe ISIS Extensions\""<<"\n";  
	file<<"set ERROR_CODE=2"<<"\n";  
	file<<"set ERROR_MSG=\"Error: Could not find CADCreoParametricCreateAssembly.exe.\""<<"\n";
	file<<"goto :ERROR_SECTION" << "\n";
	file<<":EXE_FOUND"<<"\n\n"; 

	file<<"%EXE%     %WORKING_DIR%	   %PARTS_DIR%      %ASSEMBLY_XML_FILE%     %LOG_FILE%     %EXIT_PROMPT%"<<"\n\n";

	file<< "if not ERRORLEVEL 0 (" << "\n";
	file<< "set ERROR_CODE=%ERRORLEVEL%" << "\n";
	file<< "goto :ERROR_SECTION" << "\n";
	file<< ")" << "\n\n";

	if (Automation)
	{
		file << "@echo off" << "\n";

		file<< "\n\nRem ****************************\n";
		file<< "REM Python Metric Update Script\n";
		file<< "Rem ****************************\n\n";

		file<< "set RESULT_XML_FILE=ComputedValues.xml" << "\n";
		file<< "set PY_SCRIPT_NAME=UpdateReportJson_CAD.py"  << "\n";
		file<< "set PY_SCRIPT=\"%PROE_ISIS_EXTENSIONS%\\bin\\%PY_SCRIPT_NAME%\"" << "\n";
		file<< "set PYTHONPATH=%PYTHONPATH%;%PROE_ISIS_EXTENSIONS%" << "\n\n";
	
		file<<"if exist %PY_SCRIPT% goto  :PY_FOUND"<<"\n";
		file<<"@echo off"<<"\n";
		file<<"echo		Error: Could not find %PY_SCRIPT_NAME%."<<"\n";  
		file<<"echo		Your system is not properly configured to run %PY_SCRIPT_NAME%."<<"\n";
		file<<"echo		Please see For instructions on how to configure your system, please see \"0Readme - CreateAssembly.txt\""<<"\n"; 
		file<<"echo		which is typically located at \"C:\\Program Files\\META\\Proe ISIS Extensions\""<<"\n";  
		file<<"set ERROR_CODE=2"<<"\n";  
		file<<"set ERROR_MSG=\"Error: Could not find UpdateReportJson_CAD.py.\""<<"\n";
		file<<"goto :ERROR_SECTION" << "\n\n";

		file<<":PY_FOUND"<<"\n"; 
		file<< "python %PY_SCRIPT%    %RESULT_XML_FILE%" << "\n";

		file<< "if not %ERROR_CODE% == 0 (" << "\n";
		file<< "goto :ERROR_SECTION" << "\n";
		file<< ")\n\n";
		file<< "if exist TestBench_PostProcess.cmd ("<<"\n";
		file<< "call TestBench_PostProcess.cmd" << "\n";
		file<< ")\n\n";
	}

		file<< "goto :EOF" << "\n\n";

		file<< ":ERROR_SECTION" << "\n";
		file<< "echo %ERROR_MSG% >>" << "_FAILED.txt" << "\n";
		file<< "echo \"\"" << "\n";
		file<< "echo \"See Error Log: " << "_FAILED.txt\"" << "\n";
		file<< "timeout /T 8" << "\n";
		file<< "exit /b %ERROR_CODE%" << "\n";



	file.close();

	LOGMSG("Generated Bat File: " + outFile, MSG_INFO);

}

void AssemblyOutputWriter::WriteCADData(CAssembly* const topAssembly)
{
//	LOGMSG("PrintCADData() for [" + (string)topAssembly->CyPhyRef.name() + "]", MSG_INFO);
	string name = Globals::Instance()->DebugLogDirectory + "\\CyPhy2CAD_" + (string)topAssembly->CyPhyRef.name() + "_" + topAssembly->GetID_Str() + ".log";
	std::ofstream file(name.c_str());
	if (file.is_open())
	{
		topAssembly->Print(file);
		file.close();
	}
	else
		LOGMSG("Can't open file:" + name, MSG_ERROR);
}


void AssemblyOutputWriter::WriteCADData(list<CAssembly*> &topAssembly)
{
	for (list<CAssembly*>::iterator i = topAssembly.begin(); i != topAssembly.end(); i++)
	{
		WriteCADData(*i);
	}
}

void AssemblyOutputWriter::FillInAttributes(CADData* cadData, AssemblyInterface::CADComponent& cadComp)
{
	std::string type, cad_file_name;
	std::string deType = cadData->GetType();
	CyPhyML::DesignElement de = CyPhyML::DesignElement::Cast(cadData->GetMyUDMObj());

	if (deType == "Component")
	{
		string cadRelative;
		CyPhyML::CADModel link = ((CComponent*)cadData)->CadModelRef;
		cad_file_name = CleanString(CyPhyML::Component::Cast(cadData->GetMyUDMObj()).name());
		if (link != Udm::null)
		{
			type = link.FileType();
			toupper(type);
			if (link.URI() == "")
			{
				LOGMSG("CADModel object's Model URI attribute is empty, using the name of object's parent Component. There might not be a corresponding .prt file with the same name. [" +  GMEConsole::Formatter::MakeObjectHyperlink(UdmGme::UdmId2GmeId(link.uniqueId()), link) + "]", MSG_WARNING);
			}
			else
			{
				string urlStr = link.URI();
				urlStr = boost::trim_left_copy_if(urlStr, boost::is_any_of("/\\"));
				boost::filesystem::path p(urlStr);
				urlStr = p.make_preferred().filename().string();
				unsigned int dotpos = urlStr.find_first_of('.');
				if (dotpos != string::npos)
					urlStr = urlStr.substr(0, dotpos);

				cad_file_name = CleanString(urlStr);
				cadRelative = p.make_preferred().parent_path().string();
			}

			cadComp.Type() = type;

			std::set<CyPhyML::CADParameter> CADParam_Set = link.CADParameter_kind_children();
			if (CADParam_Set.size() > 0)
			{
				AssemblyInterface::ParametricParameters parametric = AssemblyInterface::ParametricParameters::Create(cadComp);
				std::set<CyPhyML::CADParameter>::iterator i = CADParam_Set.begin();
				for (; i != CADParam_Set.end(); i++)
				{
					AssemblyInterface::CADParameter cad_param = AssemblyInterface::CADParameter::Create(parametric);
					cad_param.Name() = (i->ParameterName() == "") ? i->name() : i->ParameterName();
					cad_param.Type() = i->CADParameterType();
					//cad_param.Value() = (i->Value() == "") ? "0" : (std::string)i->Value();		// or from valueFlow
					cad_param.Value() = (i->Value() == "") ? (i->DefaultValue() == "") ? "0" : (std::string)i->DefaultValue() : (std::string)i->Value();
				}
			}
		}
		else
		{
			cadComp.Type() = "PART";
		}

		CyPhyML::SizedToFit size2Fit_child = de.SizedToFit_child();

		if (size2Fit_child != Udm::null)
			cadComp.SpecialInstruction() = "SIZE_TO_FIT";

		CyPhyML::Component comp = CyPhyML::Component::Cast(de);

		// 2/17/12 Material Stuff
		set<CyPhyML::Material> materialRefSet = comp.Material_kind_children();
		if (!materialRefSet.empty())
		{

			CyPhyML::MaterialsBase mb = materialRefSet.begin()->ref();
			materialsUsed.insert(mb);
			cadComp.MaterialID() = UdmGme::UdmId2GmeId(mb.uniqueId());
		}

		cadComp.Name() = cad_file_name;
		//m_cad_part_files.insert(cad_file_name);

		Json_Helper::ComponentManifest data;
		data.AvmID = comp.AVMID();
		data.Revision = comp.Revision();
		data.Version = comp.Version();
		data.ModelURI = cadRelative;
		data.InstanceGUID = comp.InstanceGUID();
		string key = (string)comp.AVMID() + (string)comp.Revision() + (string)comp.Version();
		if (m_componentManifestAdded.find(key) == m_componentManifestAdded.end())
			m_componentManifestData.push_back(data);
	}
	else if (deType == "Assembly")
	{
		cadComp.Name() = CleanString(CyPhyML::ComponentAssembly::Cast(cadData->GetMyUDMObj()).name());
		cadComp.Type() = "ASSEMBLY";		
	}

	cadComp.ComponentID() = cadData->GetAssemblyInterface_ID();
}


void AssemblyOutputWriter::FillInAttributes(const CyPhyML::CADTestBench& tb, 
											AssemblyInterface::CADComponent& cadComp)
{
	string id;
	cadComp.Name() = CleanString(tb.name());

	cadComp.Type() = "ASSEMBLY"; 
	to_string(id, tb.uniqueId());
	cadComp.ComponentID() = GetGMEGUID(tb); // DY: 12/13/12 cadComp.ComponentID() = id;
}

void AssemblyOutputWriter::CreateFEAAnalysisInput(CyPhyML::CADTestBench& testBench, 
												AssemblyInterface::Assembly& targetRoot)
{
	AssemblyInterface::Analyses analyses = AssemblyInterface::Analyses::Create(targetRoot);
	AssemblyInterface::FEA fea = AssemblyInterface::FEA::Create(analyses);
	fea.AnalysisID() = UdmGme::UdmId2GmeId(testBench.uniqueId());
	fea.Type() = "STRUCTURAL";	

	AssemblyInterface::AnalysisComponents analysisComponents;
	set<CyPhyML::TestInjectionPoint> tips = testBench.TestInjectionPoint_kind_children();
	for (set<CyPhyML::TestInjectionPoint>::const_iterator ci = tips.begin(); ci != tips.end(); ci++)
	{
		set<CyPhyML::TIP2StructuralMetric> tipConns = ci->dstTIP2StructuralMetric();
		if (tipConns.size() > 0)
		{
			string tmp;
			//CyPhyML::DesignEntity de = ci->ref();
			CyPhyML::TIPRefBase de = ci->ref();

			if (de != Udm::null)
			{
				if (de.type() == CyPhyML::Component::meta)
				{			
					to_string(tmp, CyPhyML::Component::Cast(de).ID());

					if (analysisComponents == Udm::null)
					{
						analysisComponents = AssemblyInterface::AnalysisComponents::Create(fea);
					}

					AssemblyInterface::Component component = AssemblyInterface::Component::Create(analysisComponents);
					component.ComponentID() = GetGMEGUID(de); //DY: 12/13/12 component.ComponentID() = tmp;
					component.InfiniteCycle() = testBench.InfiniteCycle();				
				 

					AssemblyInterface::Metrics metrics = AssemblyInterface::Metrics::Create(component);
					for (set<CyPhyML::TIP2StructuralMetric>::const_iterator cj = tipConns.begin(); cj != tipConns.end(); cj++)
					{
						CyPhyML::StructuralAnalysisMetrics metricCY = cj->dstTIP2StructuralMetric_end();
						AssemblyInterface::Metric metricAI = AssemblyInterface::Metric::Create(metrics);
					
						Uml::Class type = metricCY.type();
						if (type == CyPhyML::StressMetric::meta)
						{
							metricAI.Type() = CyPhyML::StressMetric::Cast(metricCY).Type();
						}
						else if (type == CyPhyML::FactorOfSafety::meta)
						{
							metricAI.Type() = "FactorOfSafety";
						}
						else
						{
							metricAI.Type() = "Displacement";
						}				
					
						string id = GetGMEGUID(metricCY);			//string id = GetRegistryValue(metricCY, REG_METRIC_ID);
						metricAI.ID() = id;
					}
				}
				else
				{
					LOGMSG("TIP [" + UdmGme::UdmId2GmeId(de.uniqueId()) + "] does not point to a Component!", MSG_ERROR);
				}
			}			
		}		
	}

	// AssemblyInterface::PartInterface

	AssemblyInterface::Solvers solvers = AssemblyInterface::Solvers::Create(fea);
	AssemblyInterface::Solver solver = AssemblyInterface::Solver::Create(solvers);
	solver.Type() = testBench.SolverType();
	solver.MeshType() = testBench.MeshType();
	solver.ShellElementType() = testBench.ShellElementType();
	solver.ElementShapeType() =testBench.ElementShapeType();

	AssemblyInterface::AnalysisConstraints constraintsAI;	
	set<CyPhyML::AnalysisConstraint> constraintSet = testBench.AnalysisConstraint_kind_children();	
	if (!constraintSet.empty())
		constraintsAI = AssemblyInterface::AnalysisConstraints::Create(fea);
	for (set<CyPhyML::AnalysisConstraint>::const_iterator ci = constraintSet.begin(); ci != constraintSet.end(); ci++)
	{
		AddAnalysisConstraint(*ci, constraintsAI);
	}

	AssemblyInterface::Loads loadsAI;
	set<CyPhyML::AnalysisLoad> loadSet = testBench.AnalysisLoad_kind_children();
	if (!loadSet.empty())
		loadsAI = AssemblyInterface::Loads::Create(fea);
	for (set<CyPhyML::AnalysisLoad>::const_iterator ci = loadSet.begin(); ci != loadSet.end(); ci++)
	{
		AddAnalysisLoad(*ci, loadsAI);
	}

	AssemblyInterface::SurfaceTreatments treatmentAI;
	set<CyPhyML::AdjoiningSurfacesTreatment> surface_treatment_set= testBench.AdjoiningSurfacesTreatment_kind_children();
	if (!surface_treatment_set.empty())
		treatmentAI = AssemblyInterface::SurfaceTreatments::Create(fea);
	for (set<CyPhyML::AdjoiningSurfacesTreatment>::const_iterator ci = surface_treatment_set.begin(); ci != surface_treatment_set.end(); ci++)
	{
		AddSurfaceTreatment(*ci, treatmentAI);
	}
}

bool AssemblyOutputWriter::AddSurface(const CyPhyML::GeometryTypes& surface, 
										Udm::Object& parent)
{
	AssemblyInterface::Geometry geometry_container = (parent.type() == AssemblyInterface::Geometry::meta) ? AssemblyInterface::Geometry::Cast(parent) : AssemblyInterface::Geometry::Create(parent);

	Uml::Class sType = surface.type();
	if (sType == CyPhyML::Polygon::meta)
	{
		AddPolygon(CyPhyML::Polygon::Cast(surface), geometry_container);
	}
	else if (sType == CyPhyML::Sphere::meta)
	{
		AddSphere(CyPhyML::Sphere::Cast(surface), geometry_container);

	}
	else if (sType == CyPhyML::Cylinder::meta)					// Cylindrical Surface
	{
		AddCylinder(CyPhyML::Cylinder::Cast(surface), geometry_container);
	}
	else if (sType == CyPhyML::Circle::meta)					// Cylindrical Surface
	{
		AddCircle(CyPhyML::Circle::Cast(surface), geometry_container);

	}
	else if (sType == CyPhyML::ConcentricCircles::meta)					// ConcentricCircle Surface
	{
		AddConcentricCircles(CyPhyML::ConcentricCircles::Cast(surface), geometry_container);

	}
	else if (sType == CyPhyML::Extrusion::meta)							// Extrusion
	{
		AddExtrusion(CyPhyML::Extrusion::Cast(surface), geometry_container);
	}
	else if (sType == CyPhyML::CustomGeometry::meta)					// CustomGeometry
	{
		AddCustom(CyPhyML::CustomGeometry::Cast(surface), geometry_container);
		//LOGMSG("CustomGeometry not supported right now!", MSG_INFO);
	}
	return 1;
}

void AssemblyOutputWriter::AddCircle(CyPhyML::Circle& circle_in, 
									 Udm::Object& parent_in)
{
	std::string featuresID, id;
	to_string(featuresID, circle_in.uniqueId());
	set<CyPhyML::CircleCenter2Component> centerSet = circle_in.dstCircleCenter2Component();
	set<CyPhyML::CircleSurface2Component> surfaceSet = circle_in.dstCircleSurface2Component();

	if (centerSet.size()== 1 && surfaceSet.size() == 2)
	{
		AssemblyInterface::Features features = AssemblyInterface::Features::Create(parent_in);
		features.GeometryType() = "CIRCLE";
		features.FeatureInterfaceType() = "CAD_DATUM";
		features.FeatureGeometryType() = "POINT";
		features.FeatureID() = featuresID;
		features.PrimaryGeometryQualifier() = circle_in.PrimaryGeometryQualifier_IntBnd();

		CyPhyML::AnalysisPoint center_pt = centerSet.begin()->dstCircleCenter2Component_end();
		CreateFeature(center_pt, features);

		set<CyPhyML::CircleSurface2Component>::const_iterator ei = surfaceSet.begin();
		CyPhyML::AnalysisPoint surf1_pt = ei->dstCircleSurface2Component_end();
		CreateFeature(surf1_pt, features);

		ei++;
		CyPhyML::AnalysisPoint surf2_pt = ei->dstCircleSurface2Component_end();
		CreateFeature(surf2_pt, features);
	}
}

void AssemblyOutputWriter::AddConcentricCircles(CyPhyML::ConcentricCircles& circles_in, 
												Udm::Object& parent_in)
{
	std::string featuresID, id;
	to_string(featuresID, circles_in.uniqueId());
	set<CyPhyML::CircleCenter2Component> centerSet = circles_in.dstCircleCenter2Component();
	set<CyPhyML::InnerCircle2Component> innerSet = circles_in.dstInnerCircle2Component();
	set<CyPhyML::OuterCircle2Component> outerSet = circles_in.dstOuterCircle2Component();

	if (centerSet.size() == 1 && innerSet.size() == 1 && outerSet.size() == 1)
	{
		AssemblyInterface::Features features = AssemblyInterface::Features::Create(parent_in);
		features.GeometryType() = "CONCENTRIC_CIRCLES";
		features.FeatureInterfaceType() = "CAD_DATUM";
		features.FeatureGeometryType() = "POINT";
		features.FeatureID() = featuresID;
		features.PrimaryGeometryQualifier() = circles_in.PrimaryGeometryQualifier_IntBnd();
			
		CyPhyML::AnalysisPoint center_pt = centerSet.begin()->dstCircleCenter2Component_end();
		CreateFeature(center_pt, features);

		CyPhyML::AnalysisPoint out_pt = outerSet.begin()->dstOuterCircle2Component_end();
		CreateFeature(out_pt, features);

		CyPhyML::AnalysisPoint in_pt = innerSet.begin()->dstInnerCircle2Component_end();
		CreateFeature(in_pt, features);
	}
}

void AssemblyOutputWriter::AddCylinder(CyPhyML::Cylinder& cylinder_in,
									   Udm::Object& parent_in)
{
	std::string featuresID, id;
	to_string(featuresID, cylinder_in.uniqueId());

	set<CyPhyML::CylinderSurface2Component> centerSet = cylinder_in. dstCylinderSurface2Component();
	set<CyPhyML::CylinderEnds2Component> endsSet = cylinder_in.dstCylinderEnds2Component();
		
	if (endsSet.size() == 2 && centerSet.size() == 1)
	{
		AssemblyInterface::Features features = AssemblyInterface::Features::Create(parent_in);
		features.GeometryType() = "CYLINDER";
		features.FeatureInterfaceType() = "CAD_DATUM";
		features.FeatureGeometryType() = "POINT";
		features.FeatureID() = featuresID;
		features.PrimaryGeometryQualifier() = cylinder_in.PrimaryGeometryQualifier_Bnd();
		features.SecondaryGeometryQualifier() = cylinder_in.CylinderEndCap();

		set<CyPhyML::CylinderEnds2Component>::const_iterator gi = endsSet.begin();
		CyPhyML::CylinderEnds2Component start_conn = *gi;
		gi++;
		CyPhyML::CylinderEnds2Component end_conn = *gi;
		if (start_conn.CylinderPointLocation() == "END")
		{
			CyPhyML::CylinderEnds2Component tmp = end_conn;
			end_conn = start_conn;
			start_conn = tmp;
		}

		CreateFeature((CyPhyML::AnalysisPoint)start_conn.dstCylinderEnds2Component_end(), features);
		CreateFeature((CyPhyML::AnalysisPoint)end_conn.dstCylinderEnds2Component_end(), features);

		CyPhyML::AnalysisPoint center_pt = centerSet.begin()->dstCylinderSurface2Component_end();
		CreateFeature(center_pt, features);
	}
}

void AssemblyOutputWriter::AddSphere(CyPhyML::Sphere& sphere_in, 
									 Udm::Object& parent_in)
{
	std::string featuresID, id;
	to_string(featuresID, sphere_in.uniqueId());

	set<CyPhyML::SphereCenter2Component> CenterSet = sphere_in.dstSphereCenter2Component();
	set<CyPhyML::SphereSurface2Component> SurfaceSet = sphere_in.dstSphereSurface2Component();

	if (CenterSet.size() == 1  && SurfaceSet.size() == 1)
	{

		AssemblyInterface::Features features = AssemblyInterface::Features::Create(parent_in);
		features.GeometryType() = "SPHERE";
		features.FeatureInterfaceType() = "CAD_DATUM";
		features.FeatureGeometryType() = "POINT";
		features.FeatureID() = featuresID;
		features.PrimaryGeometryQualifier() = sphere_in.PrimaryGeometryQualifier_Bnd();

		// center first then surface			
		CyPhyML::AnalysisPoint center_pt = CenterSet.begin()->dstSphereCenter2Component_end();
		CreateFeature(center_pt, features);
			
		CyPhyML::AnalysisPoint surf_pt = SurfaceSet.begin()->dstSphereSurface2Component_end();
		CreateFeature(surf_pt, features);
	}
}

void AssemblyOutputWriter::AddPolygon(CyPhyML::Polygon& polygon_in, 
									  Udm::Object& parent_in)
{
	std::string featuresID, id;
	to_string(featuresID, polygon_in.uniqueId());	

	AssemblyInterface::Features features = AssemblyInterface::Features::Create(parent_in);
	features.GeometryType() ="POLYGON";
	features.FeatureInterfaceType() ="CAD_DATUM";
	features.FeatureID() = featuresID;
	features.FeatureGeometryType() = "POINT";
	features.PrimaryGeometryQualifier() = polygon_in.PrimaryGeometryQualifier_IntBnd();

	set<CyPhyML::Polygon2Component,  OrdinalPositionCompareFunctor<CyPhyML::Polygon2Component> > P2CSet = polygon_in.dstPolygon2Component_sorted(OrdinalPositionCompareFunctor<CyPhyML::Polygon2Component>());
	for (set<CyPhyML::Polygon2Component,  OrdinalPositionCompareFunctor<CyPhyML::Polygon2Component> >::const_iterator ci = P2CSet.begin(); ci != P2CSet.end(); ci++)
	{			
		CyPhyML::AnalysisPoint ap = ci->dstPolygon2Component_end();
		CreateFeature(ap, features);
	}
}

void AssemblyOutputWriter::AddExtrusion(CyPhyML::Extrusion& extrusion_in, 
										Udm::Object& parent_in)
{
	std::string featuresID, id;
	to_string(featuresID, extrusion_in.uniqueId());

	set<CyPhyML::ExtrusionOffset2Component> offset_Set = extrusion_in.dstExtrusionOffset2Component();
	set<CyPhyML::ExtrusionSurface2Component, OrdinalPositionCompareFunctor<CyPhyML::ExtrusionSurface2Component> > surf_Set = extrusion_in.dstExtrusionSurface2Component_sorted( OrdinalPositionCompareFunctor<CyPhyML::ExtrusionSurface2Component>());

	if (offset_Set.size() == 1 && surf_Set.size() > 2)
	{
		AssemblyInterface::Features features = AssemblyInterface::Features::Create(parent_in);
		features.GeometryType() = "EXTRUSION";
		features.FeatureInterfaceType() = "CAD_DATUM";
		features.FeatureGeometryType() = "POINT";
		features.FeatureID() = featuresID;
		features.PrimaryGeometryQualifier() = extrusion_in.PrimaryGeometryQualifier_IntBnd();

		for (set<CyPhyML::ExtrusionSurface2Component, OrdinalPositionCompareFunctor<CyPhyML::ExtrusionSurface2Component> >::const_iterator ci = surf_Set.begin(); ci != surf_Set.end(); ci++)
		{
			CyPhyML::AnalysisPoint surf_pt = ci->dstExtrusionSurface2Component_end();
			CreateFeature(surf_pt, features);
		}
			 
		CyPhyML::AnalysisPoint offset_pt = offset_Set.begin()->dstExtrusionOffset2Component_end();
		CreateFeature(offset_pt, features);
	}
}

void AssemblyOutputWriter::AddCustom(CyPhyML::CustomGeometry& custom_in, 
									 Udm::Object& parent_in)
{
	std::string featuresID, opt_type;
	to_string(featuresID, custom_in.uniqueId());

	set<CyPhyML::GeometryTypes> geometry;
	set<CyPhyML::CustomGeometryOperator> operator_set = custom_in.CustomGeometryOperator_kind_children();
	for (set<CyPhyML::CustomGeometryOperator>::const_iterator ci = operator_set.begin(); ci != operator_set.end(); ci++)
	{
		CyPhyML::CustomGeometryOperator opt(*ci);
		map<CyPhyML::CustomGeometryOperator, set<CyPhyML::GeometryTypes>>::iterator di = CommonData::GeometryOperator_table.find(opt);
		if (di != CommonData::GeometryOperator_table.end())
		{
			vector<string> geometry_ids;
			AssemblyInterface::SetOperation ai_operation = AssemblyInterface::SetOperation::Create(parent_in);
			for (set<CyPhyML::GeometryTypes>::const_iterator ei = di->second.begin(); ei != di->second.end(); ei++)
			{
				to_string(featuresID, ei->uniqueId());
				geometry_ids.push_back(featuresID);
				AddSurface(*ei, parent_in);
			}

			Uml::Class type = ci->type();
			if (type == CyPhyML::PlusOperator::meta)
				ai_operation.Type() = "+";
			else if (type == CyPhyML::MinusOperator::meta)
				ai_operation.Type() = "-";
			else if (type == CyPhyML::IntersectionOperator::meta)
				ai_operation.Type() = "Intersection";
			ai_operation.FeatureID() = geometry_ids;
		}
	}
}


void AssemblyOutputWriter::CreateFeature(CyPhyML::AnalysisPoint& ap_in, 
										 Udm::Object& parent_in)
{
	string id = "";
	AnalysisPoint_FeaturePoint_Map::const_iterator di = CommonData::geometryConn_AnalysisPoint_table.find(ap_in);
	if (di != CommonData::geometryConn_AnalysisPoint_table.end())
	{
		AssemblyInterface::Feature feature = AssemblyInterface::Feature::Create(parent_in);
		feature.Name() = di->second.Datum();	

		feature.ComponentID() = GetFeaturePointComponentParentID(di->second);
	}
}

string AssemblyOutputWriter::GetFeaturePointComponentParentID(const CyPhyML::PointGeometry &pt_in)
{		
	string id = "";
	CyPhyML::MgaObject cad_model_parent = pt_in.parent();
	if (cad_model_parent.type() == CyPhyML::CADModel::meta)
	{
		CyPhyML::ComponentType comp_type = CyPhyML::CADModel::Cast(cad_model_parent).parent();
		if (comp_type.type() == CyPhyML::Component::meta)
			id = CyPhyML::Component::Cast(comp_type).InstanceGUID();
		else
			id = GetGMEGUID(comp_type);			
	}
	return id;
}
 
bool AssemblyOutputWriter::AddSurfaceTreatment(const CyPhyML::AdjoiningSurfacesTreatment& treatment_cyphy_in, 
												Udm::Object& parent_in)
{
	AssemblyInterface::SurfaceTreatment treatment_ai = AssemblyInterface::SurfaceTreatment::Create(parent_in);
	set<CyPhyML::SurfaceTreatment2Geometry> conn_set = treatment_cyphy_in.dstSurfaceTreatment2Geometry();
	for (set<CyPhyML::SurfaceTreatment2Geometry>::const_iterator ci = conn_set.begin(); ci != conn_set.end(); ci++)
	{
		CyPhyML::GeometryTypes geometry = ci->dstSurfaceTreatment2Geometry_end();
		AddSurface(geometry, treatment_ai);
	}

	vector<string> part_ids;
	set<CyPhyML::AdjoiningPartConnection> part_conn_set = treatment_cyphy_in.dstAdjoiningPartConnection();
	for (set<CyPhyML::AdjoiningPartConnection>::const_iterator ci = part_conn_set.begin(); ci != part_conn_set.end(); ci++)
	{
		CyPhyML::TestInjectionPoint tip = ci->dstAdjoiningPartConnection_end();
		CyPhyML::TIPRefBase ref = tip.ref();
		if (ref != Udm::null)
			if (ref.type() == CyPhyML::Component::meta)
			{
				string tmp;
				to_string(tmp, CyPhyML::Component::Cast(ref).ID());
				part_ids.push_back(tmp);
			}
	}

	treatment_ai.AdjoiningParts() = part_ids;
	return true;
}

bool AssemblyOutputWriter::AddAnalysisConstraint(const CyPhyML::AnalysisConstraint& constraint, 
												Udm::Object& parent)
{
	bool status = 1;
	Uml::Class type = constraint.type();
	if (type == CyPhyML::DisplacementConstraint::meta)
	{
		CyPhyML::DisplacementConstraint displacementCY = CyPhyML::DisplacementConstraint::Cast(constraint);
		CyPhyML::Rotation rotationCY = displacementCY.Rotation_child();
		CyPhyML::Translation translationCY = displacementCY.Translation_child();

		set<CyPhyML::GeometryTypes> geometry_Set;		// <Geometry, GeometryQualifier>
		set<CyPhyML::Displacement2Geometry> twodGeom_Set = displacementCY.dstDisplacement2Geometry();
		for (set<CyPhyML::Displacement2Geometry>::const_iterator ci = twodGeom_Set.begin(); ci != twodGeom_Set.end(); ci++)
			geometry_Set.insert(ci->dstDisplacement2Geometry_end());
		
		// Not supported yet!
		// set<CyPhyML::DisplacementCustomGeometry> custGeom_Set = displacementCY.dstDisplacementCustomGeometry();	

		string tx = "FREE", ty = "FREE", tz = "FREE", tunit = "N/A", rx = "FREE", ry = "FREE", rz = "FREE", runit = "N/A";
		if (translationCY != Udm::null)
		{
			string x, y, z;
			if (translationCY.XDirection() == "SCALAR")
			{
				to_string(x, translationCY.XDirectionValue());
				tx = x;
			}
			else
				tx = translationCY.XDirection();
				
			if (translationCY.YDirection() == "SCALAR")
			{
				to_string(y, translationCY.YDirectionValue());
				ty = y;
			}
			else
				ty = translationCY.YDirection();

			if (translationCY.ZDirection() == "SCALAR")
			{
				to_string(z, translationCY.ZDirectionValue());
				tz = z;
			}
			else
				tz = translationCY.ZDirection();

			CyPhyML::ParamPropTarget unitRef = translationCY.ref();	
			if  (unitRef != Udm::null)
				tunit = unitRef.name();
		}

		
		if (rotationCY != Udm::null)
		{
			string x, y, z;
			if (rotationCY.XDirection() == "SCALAR")
			{
				to_string(x, rotationCY.XDirectionValue());
				rx = x;
			}
			else
				rx = rotationCY.XDirection();

			if (rotationCY.YDirection() == "SCALAR")
			{
				to_string(y, rotationCY.YDirectionValue());
				ry = y;
			}
			else
				ry = rotationCY.YDirection();

			if (rotationCY.ZDirection() == "SCALAR")
			{
				to_string(z, rotationCY.ZDirectionValue());
				rz = z;
			}
			else
				rz = rotationCY.YDirection();

			CyPhyML::ParamPropTarget unitRef = rotationCY.ref();
			if (unitRef != Udm::null) 
				runit = unitRef.name();
		}

		for (set<CyPhyML::GeometryTypes>::const_iterator ci = geometry_Set.begin(); ci != geometry_Set.end(); ci++)
		{
			AssemblyInterface::AnalysisConstraint constraintAI = AssemblyInterface::AnalysisConstraint::Create(parent);
			AssemblyInterface::Displacement displacement = AssemblyInterface::Displacement::Create(constraintAI);

			AssemblyInterface::Translation translation = AssemblyInterface::Translation::Create(displacement);
			translation.x() = tx;
			translation.y() = ty;
			translation.z() = tz;
			translation.Units() = tunit;
			AssemblyInterface::Rotation rotation = AssemblyInterface::Rotation::Create(displacement);
			rotation.x() = rx;
			rotation.y() = ry;
			rotation.z() = rz;
			rotation.Units() = runit;
			this->AddSurface(*ci, constraintAI);
		}
	}
	else if (type == CyPhyML::PinConstraint::meta)
	{
		CyPhyML::PinConstraint pinCY = CyPhyML::PinConstraint::Cast(constraint);		

		set<CyPhyML::Pin2Cylinder> cylinderSet = pinCY.dstPin2Cylinder();
		for (set<CyPhyML::Pin2Cylinder>::const_iterator ci = cylinderSet.begin(); ci != cylinderSet.end(); ci++)
		{
			AssemblyInterface::AnalysisConstraint constraintAI = AssemblyInterface::AnalysisConstraint::Create(parent);
			AssemblyInterface::Pin pin = AssemblyInterface::Pin::Create(constraintAI);

			AssemblyInterface::AxialDisplacement displacement = AssemblyInterface::AxialDisplacement::Create(pin);
			displacement.Property() = pinCY.AxialDisplacement();
			AssemblyInterface::AxialRotation rotation= AssemblyInterface::AxialRotation::Create(pin);
			rotation.Property() = pinCY.AxialRotation();

			CyPhyML::Cylinder cylinder = ci->dstPin2Cylinder_end();
			this->AddSurface(cylinder, constraintAI);
		}
	}
	else			// ball constraint
	{
		CyPhyML::BallConstraint ballCY = CyPhyML::BallConstraint::Cast(constraint);		

		set<CyPhyML::Ball2Sphere> sphereSet = ballCY.dstBall2Sphere();
		for (set<CyPhyML::Ball2Sphere>::const_iterator ci = sphereSet.begin(); ci != sphereSet.end(); ci++)
		{			
			AssemblyInterface::AnalysisConstraint constraintAI = AssemblyInterface::AnalysisConstraint::Create(parent);
			AssemblyInterface::Ball ball = AssemblyInterface::Ball::Create(constraintAI);

			CyPhyML::Sphere sphere = ci->dstBall2Sphere_end();
			this->AddSurface(sphere, constraintAI);
		}
	}
	
	return status;
}

bool AssemblyOutputWriter::AddAnalysisLoad(const CyPhyML::AnalysisLoad& load, 
											Udm::Object& parent)
{
	bool status = 1;

	Uml::Class type = load.type();
	if (type == CyPhyML::ForceLoad::meta)
	{
		CyPhyML::ForceLoad forceLoadCY = CyPhyML::ForceLoad::Cast(load);
		set<CyPhyML::Force> forceSet = forceLoadCY.Force_kind_children();		
		set<CyPhyML::Moment> momentSet = forceLoadCY.Moment_kind_children();

		double fx = 0.0, fy = 0.0, fz = 0.0, mx = 0.0, my = 0.0, mz = 0.0;
		string funit = "newton", munit = "newton-mm";
		if (forceSet.size() > 0)
		{
			CyPhyML::Force force(*forceSet.begin());
			fx = force.XDirectionValue();
			fy = force.YDirectionValue();
			fz = force.ZDirectionValue();
			CyPhyML::ParamPropTarget unitRef = force.ref();
			if (unitRef != Udm::null)
				funit = unitRef.name();
		}

		if (momentSet.size() > 0)
		{
			CyPhyML::Moment moment(*momentSet.begin());
			mx = moment.XDirectionValue();
			my = moment.YDirectionValue();
			mz = moment.ZDirectionValue();
			CyPhyML::ParamPropTarget unitRef = moment.ref();
			if (unitRef != Udm::null)
				munit = unitRef.name();
		}

		set<CyPhyML::GeometryTypes> geometry_Set;
		set<CyPhyML::Force2Geometry> twod_Set = forceLoadCY.dstForce2Geometry();
		for (set<CyPhyML::Force2Geometry>::const_iterator ci = twod_Set.begin(); ci != twod_Set.end(); ci++)
			geometry_Set.insert(ci->dstForce2Geometry_end());

		// Not supported
		// set<CyPhyML::CustomGeometry> custom_Set = forceLoadCY.dstForceCustomGeometry();		

		for (set<CyPhyML::GeometryTypes>::const_iterator ci = geometry_Set.begin(); ci != geometry_Set.end(); ci++)
		{
			AssemblyInterface::Load loadAI = AssemblyInterface::Load::Create(parent);
			AssemblyInterface::ForceMoment forceMoment = AssemblyInterface::ForceMoment::Create(loadAI);
			AssemblyInterface::Force forceAI = AssemblyInterface::Force::Create(forceMoment);
			forceAI.x() = fx;
			forceAI.y() = fy;
			forceAI.z() = fz;
			forceAI.Units() = funit;
			AssemblyInterface::Moment momentAI = AssemblyInterface::Moment::Create(forceMoment);
			momentAI.x() = mx;
			momentAI.y() = my;
			momentAI.z() = mz;
			momentAI.Units() = munit;

			this->AddSurface(*ci, loadAI);
		}
	}
	else if (type == CyPhyML::PressureLoad::meta)
	{
		CyPhyML::PressureLoad pressureCY = CyPhyML::PressureLoad::Cast(load);

		double value = pressureCY.Value();
		string unitName;
		CyPhyML::ParamPropTarget unitRef = pressureCY.ref();
		if (unitRef != Udm::null)
			unitName = unitRef.name();

		set<CyPhyML::GeometryTypes> geometry_Set;
		set<CyPhyML::Pressure2Geometry> twod_Set = pressureCY.dstPressure2Geometry();
		for (set<CyPhyML::Pressure2Geometry>::const_iterator ci = twod_Set.begin(); ci != twod_Set.end(); ci++)
			geometry_Set.insert(ci->dstPressure2Geometry_end());


		// Not supported
		//set<CyPhyML::CustomGeometry> custom_Set = pressureCY.dstPressureCustomGeometry();

		for (set<CyPhyML::GeometryTypes>::const_iterator ci = geometry_Set.begin(); ci !=  geometry_Set.end(); ci++)
		{
			AssemblyInterface::Load loadAI = AssemblyInterface::Load::Create(parent);
			AssemblyInterface::Pressure pressureAI = AssemblyInterface::Pressure::Create(loadAI);
			pressureAI.Value() = value;
			pressureAI.Units() = unitName;			
			
			this->AddSurface(*ci, loadAI);
		}
	}
	else				// Acceleration
	{
		CyPhyML::AccelerationLoad accelerationCY =	CyPhyML::AccelerationLoad::Cast(load);
		AssemblyInterface::Load loadAI = AssemblyInterface::Load::Create(parent);
		AssemblyInterface::Acceleration accelerationAI = AssemblyInterface::Acceleration::Create(loadAI);		
		accelerationAI.x() = accelerationCY.XDirectionValue();
		accelerationAI.y() = accelerationCY.YDirectionValue();
		accelerationAI.z() = accelerationCY.ZDirectionValue();		
		
		CyPhyML::ParamPropTarget unitRef = accelerationCY.ref();
		if (unitRef != Udm::null)
			accelerationAI.Units() = unitRef.name();
	}

	return status;
}


void AssemblyOutputWriter::CreateMaterials(set<CyPhyML::MaterialsBase>& materialsSet, 
											Udm::Object& parent)
{
	if (materialsSet.size() > 0)
	{
		AssemblyInterface::Materials materials = AssemblyInterface::Materials::Create(parent);

		set<CyPhyML::MaterialsBase>::const_iterator ci = materialsSet.begin();
		for (; ci != materialsSet.end(); ci++)
		{
			CyPhyML::MaterialsBase materialBase = *ci;
			if (materialBase != Udm::null)
			{
				AssemblyInterface::Material materialsAI = AssemblyInterface::Material::Create(materials);

				bool isAluminum = (materialBase.type() == CyPhyML::Aluminum::meta);
				bool isSteel = (materialBase.type() == CyPhyML::Steel::meta);
				if (isAluminum || isSteel)
				{
					CyPhyML::Metal material = CyPhyML::Metal::Cast(materialBase);
	
					materialsAI.MaterialName() = material.Name();
					materialsAI.MaterialID() = UdmGme::UdmId2GmeId(material.uniqueId()); //materialsAI.MaterialID() = material.MaterialID();

					if (isAluminum)
						materialsAI.MaterialType() = AssemblyOutputWriter::kmat_al;
					else
						materialsAI.MaterialType() = AssemblyOutputWriter::kmat_steel;

					AssemblyInterface::MaterialProperty mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_thermal_exp;
					mProperty.Value() = material.CoefficientOfThermalExpansion();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_thermal_exp)->second;			// TODO: m_materialUnitLookup.find[PROP_THERMAL_EXP]

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_density;
					mProperty.Value()  = material.Density();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_density)->second;				// TODO: m_materialUnitLookup.find[PROP_DENSITY]

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_mod_elasticity;
					mProperty.Value() = material.ModulusOfElasticity();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_mod_elasticity)->second;			// TODO: m_materialUnitLookup.find[PROP_MOD_ELASTICITY]

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_tensile_ultm;
					mProperty.Value() = material.TensileUltimateStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_tensile_ultm)->second;			// TODO: m_materialUnitLookup.find[PROP_TENSILE_ULTM]

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_bearing_ultm;
					mProperty.Value() = material.BearingUltimateStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_bearing_ultm)->second;			// TODO: m_materialUnitLookup.find[PROP_BEARING_ULTM]

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_poisson;
					mProperty.Value() = material.PoissonsRatio();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_poisson)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_min_melt_pt;
					mProperty.Value() = material.MinimumMeltingPoint();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_min_melt_pt)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_heat_capacity;
					mProperty.Value() = material.HeatCapacity();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_heat_capacity)->second;
										
					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_tensile_yield;
					mProperty.Value() = material.TensileYieldStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_tensile_yield)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_shear_mod;
					mProperty.Value() = material.ShearModulus();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_shear_mod)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_shear_strength;
					mProperty.Value() = material.ShearStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_shear_strength)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_bearing_yield;
					mProperty.Value() = material.BearingYieldStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_bearing_yield)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_fatigue_strength;
					mProperty.Value() = material.FatigueStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_fatigue_strength)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_fatigue_cycles;
					mProperty.Value() = material.FatigueNumberOfCycles();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_fatigue_cycles)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_thermal_conductivity;
					mProperty.Value() = material.ThermalConductivity();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_thermal_conductivity)->second;
				}
				else if (materialBase.type() == CyPhyML::Polymer::meta)
				{
					CyPhyML::Polymer material = CyPhyML::Polymer::Cast(materialBase);

					materialsAI.MaterialName() = material.Name();
					materialsAI.MaterialID() = UdmGme::UdmId2GmeId(material.uniqueId()); //materialsAI.MaterialID() = material.MaterialID();
					materialsAI.MaterialType() = AssemblyOutputWriter::kmat_poly;

					AssemblyInterface::MaterialProperty mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_thermal_exp;
					mProperty.Value() = material.CoefficientOfThermalExpansion();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_thermal_exp)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_density; 
					mProperty.Value() = material.Density();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_density)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_mod_elasticity; 
					mProperty.Value() = material.ModulusOfElasticity();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_mod_elasticity)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_tensile_ultm;
					mProperty.Value() = material.TensileUltimateStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_tensile_ultm)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_tensile_yield;
					mProperty.Value() = material.TensileYieldStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_tensile_yield)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_vicatb_soft_pt;
					mProperty.Value() = material.VicatBSofteningPoint();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_vicatb_soft_pt)->second;

#if 0				// need to look at how to support this
					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = PROP_FLAME_RATING;
					mProperty.Value() = material.FlammabilityRating();
					mProperty.Units() = m_materialUnitLookup.find(PROP_FLAME_RATING)->second;
#endif

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_heat_deflect_temp;
					mProperty.Value() = material.HeatDeflectionTemp();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_heat_deflect_temp)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_heat_deflect_load;
					mProperty.Value() = material.HeatDeflectionLoad();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_heat_deflect_load)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_bearing_yield;
					mProperty.Value() = material.BearingYieldStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_bearing_yield)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_shear_mod;
					mProperty.Value() = material.ShearModulus();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_shear_mod)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_flexural_strength;
					mProperty.Value() = material.FlexuralStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_flexural_strength)->second;												
				}
				else
				{
					CyPhyML::Ceramic material = CyPhyML::Ceramic::Cast(materialBase);

					materialsAI.MaterialName() = material.Name();
					materialsAI.MaterialID() = UdmGme::UdmId2GmeId(material.uniqueId()); //materialsAI.MaterialID() = material.MaterialID();
					materialsAI.MaterialType() = AssemblyOutputWriter::kmat_ceramic;

					AssemblyInterface::MaterialProperty mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_thermal_exp;
					mProperty.Value() = material.CoefficientOfThermalExpansion();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_thermal_exp)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_density;
					mProperty.Value() = material.Density();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_density)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_mod_elasticity;
					mProperty.Value() = material.ModulusOfElasticity();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_mod_elasticity)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_tensile_ultm;
					mProperty.Value() = material.TensileUltimateStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_tensile_ultm)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_bearing_ultm;
					mProperty.Value() = material.BearingUltimateStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_bearing_ultm)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_fracture_toughness;
					mProperty.Value() = material.FractureToughness();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_fracture_toughness)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_max_safe_op_temp;
					mProperty.Value() = material.MaxSafeOperatingTemperature();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_max_safe_op_temp)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_shear_mod;
					mProperty.Value() = material.ShearModulus();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_shear_mod)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_flexural_strength;
					mProperty.Value() = material.FlexuralStrength();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_flexural_strength)->second;

					mProperty = AssemblyInterface::MaterialProperty::Create(materialsAI);
					mProperty.PropertyName() = AssemblyOutputWriter::kprop_thermal_conductivity;
					mProperty.Value() = material.ThermalConductivity();
					mProperty.Units() = m_materialUnitLookup.find(AssemblyOutputWriter::kprop_thermal_conductivity)->second;
				}	
			}				
		}// end for
	}// end if
}



map<string, string> AssemblyOutputWriter::CreateMaterialUnitLookupTable()
{
	map<string, string> m; 
	m[AssemblyOutputWriter::kprop_thermal_exp] = "1/C";
	m[AssemblyOutputWriter::kprop_density] = "g/cm^3";
	m[AssemblyOutputWriter::kprop_mod_elasticity] = "GPa";
	m[AssemblyOutputWriter::kprop_tensile_ultm] = "MPa";
	m[AssemblyOutputWriter::kprop_tensile_yield] = "MPa";
	m[AssemblyOutputWriter::kprop_shear_mod] = "GPa";
	m[AssemblyOutputWriter::kprop_poisson] = "N/A";
	m[AssemblyOutputWriter::kprop_shear_strength] = "MPa";
	m[AssemblyOutputWriter::kprop_bearing_ultm] = "MPa";
	m[AssemblyOutputWriter::kprop_bearing_yield] = "MPa";
	m[AssemblyOutputWriter::kprop_fatigue_strength] = "MPa";
	m[AssemblyOutputWriter::kprop_fatigue_cycles] = "N/A";
	m[AssemblyOutputWriter::kprop_heat_capacity] = "J/(K-kg)";
	m[AssemblyOutputWriter::kprop_min_melt_pt] = "C";
	m[AssemblyOutputWriter::kprop_thermal_conductivity] = "W/(m-K)";
	m[AssemblyOutputWriter::kprop_fracture_toughness] = "MPa/m^2";
	m[AssemblyOutputWriter::kprop_max_safe_op_temp] = "C";
	m[AssemblyOutputWriter::kprop_flexural_strength] = "MPa";
	m[AssemblyOutputWriter::kprop_vicatb_soft_pt] = "C";
	m[AssemblyOutputWriter::kprop_flame_rating] = "UL94";
	m[AssemblyOutputWriter::kprop_heat_deflect_temp] = "C";
	m[AssemblyOutputWriter::kprop_heat_deflect_load] = "MPa";
    return m; 
}

string AssemblyOutputWriter::GetTimeStamp()
{
	time_t time_start;         // calendar time 
    time_start=time(NULL);     // get current cal time 

    std::string timeStamp = asctime(localtime(&time_start));  // e.g. Fri Jul 20 17:37:32 2012
	return timeStamp;
}




 void AssemblyOutputWriter::CreateTestBenchPostProcessScripts(const CyPhyML::TestBench &test_bench_in)
 {
	set<CyPhyML::PostProcessing> post_process_set_in = test_bench_in.PostProcessing_children();
	if (!post_process_set_in.empty())
	{
		vector<string> parameters;
		for (set<CyPhyML::PostProcessing>::const_iterator ci = post_process_set_in.begin(); ci != post_process_set_in.end(); ci++)
		{
			parameters.push_back((string)ci->ScriptPath());
		}
		ScriptWriter::GeneratePostProcessScript(this->outputDirectoryName, Globals::Instance()->ProjectRootDir, parameters);
	}
 }

 void AssemblyOutputWriter::GenerateExecutionScripts()
 {
	 string sharedLibPath = "";
	 vector<string> errMsg;
	 if (this->useProjectManifest)
		Json_Helper::ParseJsonFile(Globals::Instance()->ProjectRootDir, this->outputDirectoryName, this->m_componentManifestData, sharedLibPath, errMsg);
	 
	 for (unsigned int i = 0; i < errMsg.size(); i++)
		 LOGMSG("manifest.project.json Processing Error: " + errMsg[i], MSG_WARNING);

	 for (unsigned int i = 0; i < this->m_componentManifestData.size(); i++)
		 if (this->m_componentManifestData[i].ComponentCadFolder == "")
			 LOGMSG("AVMID [" + this->m_componentManifestData[i].AvmID + "] does not have modelpath attribute or attribute is empty in manifest.project.json file. It will be skipped in the generated search_META.pro and cad.manifest.json.", MSG_WARNING);

	 for (unsigned int i = 0; i < this->m_componentManifestData.size(); i++)
		 LOGMSG("InstanceGUID [" + this->m_componentManifestData[i].InstanceGUID + " ], AVMID [" + this->m_componentManifestData[i].AvmID + "]", MSG_INFO);

	 Json_Helper::SerializeManifestData(this->outputDirectoryName, this->m_componentManifestData);
	 ScriptWriter::GenerateCopyScript(this->outputDirectoryName, this->cadFileDirectoryName, sharedLibPath, this->m_componentManifestData, this->useProjectManifest);
	 ScriptWriter::GenerateCreoSearchPath(this->outputDirectoryName, this->cadFileDirectoryName, sharedLibPath, this->m_componentManifestData, this->useProjectManifest);
	 ScriptWriter::GenerateZipScript(this->outputDirectoryName, this->m_componentManifestData);
 }