/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CADData.h"
#include <fstream>
#include <stdio.h>
#include "UdmGme.h"
#include "string_utils.h"

string CADData::GetID_Str()
{
	string tmp;
	to_string(tmp, m_id);
	return tmp;
}

/** \brief Add a CComponent to CAssembly's child_Component collection if it hasn't already been added
    \param [in] comp Pointer to a CComponent
    \return void
*/
void CAssembly::AddComponent(CComponent* comp)
{
	bool old = 0;
	for (vector<CComponent*>::const_iterator ci = child_Component.begin(); ci != child_Component.end(); ci++)
	{
		if ((*ci)->CyPhyRef.uniqueId() == comp->CyPhyRef.uniqueId())
		{
			old = 1;
			break;
		}
	}

	if (!old)
		child_Component.push_back(comp);
}

/** \brief Add a CForwarder to CAssembly's child_Forwarders collection if it hasn't already been added
    \param [in] fwd Pointer to a CForwarder
    \return void
*/
void CAssembly::AddForwarder(CForwarder* fwd)
{
	bool old = 0;
	for (vector<CForwarder*>::const_iterator ci = child_Forwarders.begin(); ci != child_Forwarders.end(); ci++)
	{
		if ((*ci)->CyPhyRef.uniqueId() == fwd->CyPhyRef.uniqueId())
		{
			old = 1;
			break;
		}
	}

	if (!old)
		child_Forwarders.push_back(fwd);
}

/** \brief Prints CAssembly's content to a file
    \param [in] file Reference to a ofstream object
    \return void
*/
void CAssembly::Print(ofstream& file)
{
	file<<"\n<<<<<<<<<>>>>>>>>>>>\n";
	file<<"Assembly ["<<(string)CyPhyRef.name()<<","<<this->m_assemblyInterface_ID<<"]\n";

	file<<"<< Forwarders >>\n";
	for (vector<CForwarder*>::const_iterator ci = child_Forwarders.begin(); ci != child_Forwarders.end(); ci++)
		(*ci)->Print(file);

	file<<"\n<< Child Edges >>\n";
	for (vector<CEdge*>::const_iterator ci = child_Edges.begin(); ci != child_Edges.end(); ci++)
		(*ci)->Print(file);
	// size-2-fit edges
	file<<"\n++++++++++++++ Size-2-Fit-Edges\n";
	for (vector<CEdge*>::const_iterator ci = child_Size2FitEdges.begin(); ci != child_Size2FitEdges.end(); ci++)
		(*ci)->Print(file);

	file<<"\n<< Child Components >>\n";
	for (vector<CComponent*>::const_iterator ci = child_Component.begin(); ci != child_Component.end(); ci++)
		(*ci)->Print(file);
	// size-2-fit components
	for (vector<CComponentSize2Fit*>::const_iterator ci = child_Size2Fits.begin(); ci != child_Size2Fits.end(); ci++)
		(*ci)->Print(file);

	file<<"\n<< Child Assemblies >>\n";
	for (vector<CAssembly*>::const_iterator ci = child_Assembly.begin(); ci != child_Assembly.end(); ci++)
	{
		if ((*ci)->enabled)
			file<<"  "<<CyPhyML::ComponentAssembly::Cast((*ci)->GetMyUDMObj()).ID()<<"\n";
	}

	// Print Child Assemblies
	for (vector<CAssembly*>::const_iterator ci = child_Assembly.begin(); ci != child_Assembly.end(); ci++)
	{
		if ((*ci)->enabled)
			(*ci)->Print(file);
	}

	file<<"\n\n";
}

/** \brief Prints CComponent's content to a file
    \param [in] file Reference to a ofstream object
    \return void
*/
void CComponent::Print(ofstream& file)
{
	//file<< "Component [" << (string)CyPhyRef.name() << ", " << CyPhyRef.ID() << ", " << CyPhyRef.uniqueId() << "]\n";
	file<< "Component [" << (string)CyPhyRef.name() << ", " << this->GetAssemblyInterface_ID() << ", " << this->GetID_Str() << "]\n";
	for (map<long, CyPhyML::StructuralInterface>::const_iterator ci = child_StructuralInterfaceRoles.begin();  ci != child_StructuralInterfaceRoles.end(); ci++)
		file<<"  StructuralInterfaceRole Ports: "<<UdmGme::UdmId2GmeId(ci->second.uniqueId())<<"\n";
}

/** \brief Prints CEdge's content to a file
    \param [in] file Reference to a ofstream object
    \return void
*/
void CEdge::Print(ofstream& file)
{
	file<<"Edge [CyPhyJoinStructures Ref:"<<UdmGme::UdmId2GmeId(CyPhyRef.uniqueId())<<"]\n";

	Udm::Object src = source->GetMyUDMObj();
	Udm::Object dst = destination->GetMyUDMObj();

	string srcType = source->GetType(), dstType = destination->GetType();

	if (srcType == "Component")
		file<<"  Source[Type:"<<source->GetType()<<" ID:"<<CyPhyML::Component::Cast(src).ID()<<", Port:"<<UdmGme::UdmId2GmeId(sourcePort)<<"]"<<"\n";
	else
		file<<"  Source[Type:"<<source->GetType()<<" ID:"<<UdmGme::UdmId2GmeId(source->GetMyUDMObj().uniqueId())<<", Port:"<<UdmGme::UdmId2GmeId(sourcePort)<<"]"<<"\n";

	if (dstType == "Component")
		file<<"  Destination[Type:"<<destination->GetType()<<" ID:"<<CyPhyML::Component::Cast(dst).ID()<<", Port:"<<UdmGme::UdmId2GmeId(destinationPort)<<"]"<<"\n";
	else
		file<<"  Destination[Type:"<<destination->GetType()<<" ID:"<<UdmGme::UdmId2GmeId(destination->GetMyUDMObj().uniqueId())<<", Port:"<<UdmGme::UdmId2GmeId(destinationPort)<<"]"<<"\n";
	
}

/** \brief Prints CForwarder's content to a file
    \param [in] file Reference to a ofstream object
    \return void
*/
void CForwarder::Print(ofstream& file)
{
	file<<" Forwarder ["<<UdmGme::UdmId2GmeId(CyPhyRef.uniqueId())<<"]\n";
}

/** \brief Prints CComponentSize2Fit's content to a file
    \param [in] file Reference to a ofstream object
    \return void
*/
void CComponentSize2Fit::Print(ofstream& file)
{
	//file << "Component-Size2Fit [" << (string)CyPhyRef.name() << "," << CyPhyRef.ID() << ", " << CyPhyRef.uniqueId() << "]\n";
	file << "Component-Size2Fit [" << (string)CyPhyRef.name() << "," << this->GetAssemblyInterface_ID() << ", " << this->GetID_Str() << "]\n";
	for (map<long, CyPhyML::StructuralInterface>::const_iterator ci = child_StructuralInterfaceRoles.begin();  ci != child_StructuralInterfaceRoles.end(); ci++)
	{
		file<<"  StructuralInterfaceRole Ports: "<<UdmGme::UdmId2GmeId(ci->second.uniqueId())<<":";
		map<long, CyPhyML::StructuralInterface>::const_iterator di = child_StructuralInterfaceRolesENDs.find(ci->first);
		if (di != child_StructuralInterfaceRolesENDs.end())
			file<<UdmGme::UdmId2GmeId(di->second.uniqueId())<<"\n";
	}
}


void CComponent::PrintGraphviz(ofstream& file)
{
	//int ID = this->CyPhyRef.ID();	
	string ID = this->GetAssemblyInterface_ID();
	file<<this->GetID_Str()<<" [label=\""<<ID<<"\\n"<<this->CyPhyRef.name()<<"\"";	//file<<ID<<" [label=\""<<ID<<"\\n"<<this->CyPhyRef.name()<<"\"";
	
	file<<"];\n";
}

void CComponentSize2Fit::PrintGraphviz(ofstream& file)
{
	//int ID = this->CyPhyRef.ID();	
	string ID = this->GetAssemblyInterface_ID();
	file<<this->GetID_Str() <<" [label=\""<<ID<<"\\n"<<this->CyPhyRef.name()<<"\"";	//file<<ID<<" [label=\""<<ID<<"\\n"<<this->CyPhyRef.name()<<"\"";
	file<<", color=deepskyblue";
		
	file<<"];\n";
}

void CAssembly::PrintGraphviz(ofstream& file, bool directed)
{
	if (this->enabled)
	{
		file<<"subgraph cluster_"<<CleanString(this->m_assemblyInterface_ID, 31, "-")<<"{\n";
		file<<"label = \""<<this->m_assemblyInterface_ID<<"\\n"<<this->CyPhyRef.name()<<"\";\n";

		for (unsigned int i = 0; i < child_Component.size(); i++)
		{
			CComponent* comp = child_Component[i];
			comp->PrintGraphviz(file);
		}
		for (unsigned int i = 0; i < child_Size2Fits.size(); i++)
		{
			CComponentSize2Fit* comp = child_Size2Fits[i];
			comp->PrintGraphviz(file);
		}

		for (unsigned int i = 0; i < child_Assembly.size(); i++)
		{
			CAssembly* assembly = this->child_Assembly[i];
			//asm->PrintGraphviz(file);
		}

		PrintGraphviz(file, child_Edges, directed);
		PrintGraphviz(file, child_Size2FitEdges, directed);

		file<<"}\n";
	}
}


void CAssembly::PrintGraphviz(ofstream& file, vector<CEdge*> &edges, bool directed)
{
	for (unsigned int i = 0; i < edges.size(); i++)
	{
		CEdge* edge = edges[i];
		// if valid edge for this level: print
		// else: skip

		Udm::Object src_udmObj;
		Udm::Object dst_udmObj;	
		if (edge->source->GetType() == "Forwarder")
			src_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(edge->source->GetMyUDMObj()).parent();
		else
			src_udmObj = edge->source->GetMyUDMObj();

		if (edge->destination->GetType() == "Forwarder")
			dst_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(edge->destination->GetMyUDMObj()).parent();
		else
			dst_udmObj = edge->destination->GetMyUDMObj();

		if (this->m_ID == src_udmObj.uniqueId() || this->m_ID == dst_udmObj.uniqueId())	
			continue;

		string src_name = CyPhyML::DesignEntity::Cast(src_udmObj).name(), dst_name = CyPhyML::DesignEntity::Cast(dst_udmObj).name();
		if (!directed)
			file<<edge->source->GetID_Str()<<" -- "<<edge->destination->GetID_Str();			//file<<CyPhyML::DesignEntity::Cast(src_udmObj).ID()<<" -- "<<CyPhyML::DesignEntity::Cast(dst_udmObj).ID();
		else
			file<<edge->source->GetID_Str()<<" -> "<<edge->destination->GetID_Str();			//file<<CyPhyML::DesignEntity::Cast(src_udmObj).ID()<<" -> "<<CyPhyML::DesignEntity::Cast(dst_udmObj).ID();

		file<<" [label = \"DOF (" << edge->GetDegreeOfFreedom() << ")\"" << "];\n";
	}
}


bool CAssembly::IsMyForwarder(CyPhyML::StructuralInterfaceForwarder &forwarder)
{
	for (unsigned int i = 0; i < child_Forwarders.size(); i++)
	{
		CForwarder* fwder = child_Forwarders[i];
		long id = fwder->GetMyUDMObj().uniqueId();
		if (forwarder.uniqueId() == id)
			return 1;
	}
	return 0;
}


bool CAssembly::IsMyForwarder(long queryID)
{
	for (unsigned int i = 0; i < child_Forwarders.size(); i++)
	{
		CForwarder* fwder = child_Forwarders[i];
		long id = fwder->GetMyUDMObj().uniqueId();
		if (queryID == id)
			return 1;
	}
	return 0;
}