/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#ifndef CADDATA_H
#define CADDATA_H

#include "CyPhyML.h"
#include "string_utils.h"
#include "CommonHelper.h"

/** \file
    \brief Definition of intermediate data structures used during flattening assembly hierarchy and constraint generation		    
*/

class CEdge;
class CForwarder;

/** \class CADData
    \brief Abstract base class for the intermediate data structures
		   Contains some common functions and members
*/
class CADData
{
public:
	CADData(string type, long id):
			m_Type(type), 
			m_id(id),
			m_assemblyInterface_ID(""){}
	virtual ~CADData (){}
	virtual Udm::Object GetMyUDMObj () = 0;	///< Returns the UDM object
	virtual void Print (ofstream&) = 0;		///< Print functions

	string GetType(){return m_Type;}		///< Returns the type
	long GetID(){return m_id;}				///< returns the ID
	string GetID_Str();
	string GetAssemblyInterface_ID() { return m_assemblyInterface_ID;}
	void SetAssemblyInterface_ID(string id) { m_assemblyInterface_ID = id;}
protected:
	string m_Type;							///< The type of CADData structures, determined by derived classes
	long m_id;								///< Corresponding GME ID
	string m_assemblyInterface_ID;			///< ID used in AssemblyInterface output file
};


/** \class CComponent
    \brief Corresponds to a CyPhyML::Components from the model
*/
class CComponent : public CADData
{
public:
	CComponent(long id, CyPhyML::Component comp, CyPhyML::CADModel cad_model)
		: CADData("Component", id), 
		CyPhyRef(comp),
		CadModelRef(cad_model)
	{
		//to_string(m_assemblyInterface_ID, comp.ID());
		//SetAssemblyInterface_ID(GetGMEGUID(comp));		// 1-3-2013: Using InstanceGUID now
		SetAssemblyInterface_ID(comp.InstanceGUID());
	}
	virtual ~CComponent(){}
	virtual Udm::Object GetMyUDMObj() {return CyPhyRef;}
	virtual void Print(ofstream&);
	virtual void PrintGraphviz(ofstream&);

	CyPhyML::Component CyPhyRef;													///< Reference to a CyPhyML::Component from the model
	CyPhyML::CADModel CadModelRef;
	map<long, CyPhyML::StructuralInterface> child_StructuralInterfaceRoles;     ///< A map of StructuralInterfaceRole ports contained by CyPhyML::Component from the model

};

/** \class CComponentSize2Fit
    \brief Represents a size2fit cad component
*/
class CComponentSize2Fit : public CComponent
{
public:
	CComponentSize2Fit(long id, CyPhyML::Component ref, CyPhyML::CADModel cad_model)
		: CComponent(id, ref, cad_model) {}
	virtual ~CComponentSize2Fit(){}		
	virtual void Print(ofstream&);

	map<long, CyPhyML::StructuralInterface> child_StructuralInterfaceRolesENDs;	///< A map of StructuralInterfaceRole ports that is connected to a port from child_StructuralInterfaceRoles. The key is ID of a port from child_StructuralInterfaceRoles.
	virtual void PrintGraphviz(ofstream& file);
};

/** \class CAssembly
    \brief Corresponds to a CyPhyML::ComponentAssembly from the model
*/
class CAssembly : public CADData
{
public:
	CAssembly(long id)
		: CADData("Assembly", id), 
		needflattening(0), 
		enabled(1),
		m_ID(id),
		m_configID(LongToString(id)),
		m_rootComponentID(-1){}

	CAssembly(long id, CyPhyML::ComponentAssembly assembly)
		: CADData("Assembly", id), 
		CyPhyRef(assembly), 
		needflattening(0), 
		enabled(1),
		m_ID(assembly.ID()),
		m_configID(assembly.ConfigurationUniqueID()),
		m_rootComponentID(-1){}

	CAssembly(long id, CyPhyML::CADTestBench testBench)
		: CADData("Assembly", id), 
		CyPhyRef(testBench), 
		needflattening(0), 
		enabled(1),
		m_ID(id),
		m_configID(LongToString(id)),
		m_rootComponentID(-1){}

	virtual ~CAssembly(){}
	virtual Udm::Object GetMyUDMObj() {return CyPhyRef;}
	virtual void Print(ofstream&);
	void PrintGraphviz(ofstream&, bool directed = false);

	string LongToString(long input)
	{		
		std::stringstream strstream;
		strstream << input;
		return strstream.str();
	}

	CyPhyML::MgaObject CyPhyRef;				///< Reference to a CyPhyML::ComponentAssembly from the model

	bool needflattening;								///< 1 if this assembly needs to be flattened, 0 otherwise
	bool enabled;	
	long m_ID;
	string m_configID;	


	///< If this flag is 1 then it will appear in the outout CAD assembly xml file, otherwise it will be ignred in the generation process
	vector<CComponent*> child_Component;				///< A collection of child components
	vector<CAssembly*> child_Assembly;					///< A collection of child assemblies
	vector<CEdge*> child_Edges;							///< A collection of child edges
	vector<CForwarder*> child_Forwarders;				///< A collection of child forwarders
	vector<CComponentSize2Fit*> child_Size2Fits;
	vector<CEdge*> child_Size2FitEdges;

	void AddComponent(CComponent*);
	void AddForwarder(CForwarder*);
	bool IsMyForwarder(CyPhyML::StructuralInterfaceForwarder &);
	bool IsMyForwarder(long);
	long GetRootComponentID () { return m_rootComponentID; }
	void SetRootComponentID (long id) { m_rootComponentID = id; }

protected:		// data member
	long m_rootComponentID;

protected:		// function
	void PrintGraphviz(ofstream& file, vector<CEdge*> &, bool directed);
};


/** \class CEdge
    \brief Represents connection between a CComponent-CForwarder, CComponent-CComponent, or CForwarder-CForwarder
*/
class CEdge : public CADData
{
public:
	CEdge(long id)
		: CADData("Edge", id), 
		processed(0),
		degree_of_freedom(0){}

	CEdge(long id, CyPhyML::JoinStructures js)
		: CADData("Edge", id), 
		CyPhyRef(js),
		processed(0),
		degree_of_freedom(0){}
	virtual ~CEdge(){}
	virtual Udm::Object GetMyUDMObj() {return CyPhyRef;}
	virtual void Print(ofstream&);

	void SetDegreeOfFreedom(int n) { degree_of_freedom = n; }
	int GetDegreeOfFreedom() { return degree_of_freedom; }

	CADData* source;									///< Source of the edge
	CADData* destination;								///< Destination of the edge
	long sourcePort;									///< ID of port in the Source
	long destinationPort;								///< ID of port in the Destination
	CyPhyML::JoinStructures CyPhyRef;					///< Reference to a CyPhyML::JoinStructure from the model
	bool processed;										///< 1 indicates if the edge has been assigned, 0 otherwise


protected:
	int degree_of_freedom;
};


/** \class CForwarder
    \brief Represents a StructuralInterfaceForwarder from the model
*/
class CForwarder : public CADData
{
public:
	CForwarder(CyPhyML::StructuralInterfaceForwarder fwd, long id = 0)
		: CADData("Forwarder", id), 
		CyPhyRef(fwd){}
	virtual ~CForwarder(){}
	virtual Udm::Object GetMyUDMObj() {return CyPhyRef;}
	virtual void Print(ofstream&);

	CyPhyML::StructuralInterfaceForwarder CyPhyRef;			///< Reference to a CyPhyML::StructuralInterfaceRoleForwarder from the model
};



#endif