/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CEdgeAnalyzer.h"
#include "UdmConsole.h"
#include "UdmFormatter.h"
#include "UdmGme.h"
#include "UmlExt.h"
#include "CommonData.h"
#include "string_utils.h"

void CEdgeAnalyzer::AnalyzeEdges(CAssembly *assemblyParent, 
								list<CAssembly*> &islands)
{
	static long newAssemblyID_Seed = 200000000;

	map<long, CAssembly*> tmpAssemblies_Table;
	vector<CEdge*> closedEdges;
	vector<CEdge*> openEdges = assemblyParent->child_Edges;

	vector<CComponent*> openComponents = assemblyParent->child_Component, closedComponents;
	deque<CComponent*> openQ;

	//CEdge* lastEdge = openEdges.back();
	//CADData* src_End = lastEdge->source;	
	//FindCComponent(src_End->GetID(), openComponents, openQ);


	
	if (openComponents.size() == 1 && openEdges.empty())
	{		
		assemblyParent->SetAssemblyInterface_ID(assemblyParent->GetID_Str());
		islands.push_back(assemblyParent);
	}
	else
	{
		// find island assemblies
		int z = 1;
		while (!openEdges.empty())
		{
			//CEdge* lastEdge = openEdges.back();
			//CADData* src_End = lastEdge->source;	
			//FindCComponent(src_End->GetID(), openComponents, openQ);

			CEdge* lastEdge = openEdges.back();
			CADData* src_End = lastEdge->source;
			FindCComponent(src_End->GetID(), openComponents, openQ);
			WalkIncidentEdges(openQ, openComponents, closedComponents, openEdges, closedEdges);

			// create a new island CAssembly;
			long ID = newAssemblyID_Seed + assemblyParent->GetID();		//long asm_ID = (i) * newAssemblyID_Seed + assemblyParent->GetID();
			string tmp, assemblyInterface_ID = "";
			to_string(tmp, assemblyParent->GetAssemblyInterface_ID());
			assemblyInterface_ID = tmp + "|";
			tmp.clear();
			to_string(tmp, z);
			assemblyInterface_ID += tmp;

			//CAssembly* islandAssembly = CommonData::CreateCAssembly(CyPhyML::ComponentAssembly::Cast(assemblyParent->CyPhyRef), ID, assemblyInterface_ID);
			CAssembly* islandAssembly = CommonData::CreateCAssembly(assemblyParent->CyPhyRef, ID, assemblyInterface_ID);
			islandAssembly->child_Component = closedComponents;
			islandAssembly->child_Edges = closedEdges;
			islands.push_back(islandAssembly);
			tmpAssemblies_Table[ID] = islandAssembly;

			for (unsigned int j = 0; j < islandAssembly->child_Component.size(); j++)
				components2Asm_Table[islandAssembly->child_Component[j]->GetID()] = ID;

			closedComponents.clear();
			closedEdges.clear();

			newAssemblyID_Seed++;
			z++;
		}	// while

		// Add openComponents to OrphanList;
		if (!openComponents.empty())
		{
			this->OrphanComponents = openComponents;
		}

		// iterate through Size2Fit (are they orphans or belong to a CAssembly?)
		// for each size2fit, find its size2fit edges
		// find which assembly contains the edge's source
		// if they are in the same 
		for (unsigned int k = 0; k < assemblyParent->child_Size2Fits.size(); k++)
		{
			CComponentSize2Fit* s2f = assemblyParent->child_Size2Fits[k];
			vector<CEdge*> s2f_closeEdges;				// <source, CEdge*>
			long assemblyParent_ID = FindSize2FitEdges(s2f->GetID(), assemblyParent->child_Size2FitEdges, s2f_closeEdges);

			// orphan if no edges or parentID is -1
			if (s2f_closeEdges.empty() || assemblyParent_ID == -1)
			{
				OrphanComponents.push_back(s2f);
			}
			else
			{
				// add size2fit + size2fit edges to island assembly
				map<long, CAssembly*>::iterator m = tmpAssemblies_Table.find(assemblyParent_ID);
				if (m != tmpAssemblies_Table.end())
				{
					m->second->child_Size2Fits.push_back(s2f);
					m->second->child_Size2FitEdges.insert(m->second->child_Size2FitEdges.begin(), s2f_closeEdges.begin(), s2f_closeEdges.end());
				}
			}

		}

		tmpAssemblies_Table.clear();
	}
}

void CEdgeAnalyzer::WalkIncidentEdges(deque<CComponent*> &openQ, 
									vector<CComponent*> &openComponents_in, 
									vector<CComponent*> &closeComponents_out, 
									vector<CEdge*> &openEdges_in, 
									vector<CEdge*> &closeEdges_out)
{
	while (!openQ.empty())
	{
		bool found = 0;
		vector<CEdge*> newEdges;

		CComponent* targetNode = openQ.front();
		closeComponents_out.push_back(targetNode);
		openQ.pop_front();

		long cnt_Src_ID = targetNode->GetID(), cnt_Dst_ID;
		for (unsigned int i = 0; i < openEdges_in.size(); i++)
		{
			CEdge* curEdge =  openEdges_in[i];
			Udm::Object src_udmObj;
			Udm::Object dst_udmObj;	
		
			if (curEdge->source->GetType() == "Forwarder")
				src_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(curEdge->source->GetMyUDMObj()).parent();
			else
				src_udmObj = curEdge->source->GetMyUDMObj();

			if (curEdge->destination->GetType() == "Forwarder")
				dst_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(curEdge->destination->GetMyUDMObj()).parent();
			else
				dst_udmObj = curEdge->destination->GetMyUDMObj();

			bool myEdge = 0;
			if (cnt_Src_ID == src_udmObj.uniqueId())
			{
				cnt_Dst_ID = dst_udmObj.uniqueId();
				FindCComponent(cnt_Dst_ID, openComponents_in, openQ);			// FindCComponent
				closeEdges_out.push_back(curEdge);
			}
			else if (cnt_Src_ID == dst_udmObj.uniqueId())
			{
				cnt_Dst_ID = src_udmObj.uniqueId();
				FindCComponent(cnt_Dst_ID, openComponents_in, openQ);			// FindCComponent
				closeEdges_out.push_back(curEdge);
			}
			else
			{
				newEdges.push_back(curEdge);				// for next go around
			}
		}

		openEdges_in = newEdges;
		WalkIncidentEdges(openQ, openComponents_in, closeComponents_out, openEdges_in, closeEdges_out);
	}
}

void CEdgeAnalyzer::FindCComponent(long ID_in, 
					vector<CComponent*> &openComponents_inout, 
					deque<CComponent*> &destionationSet_inout)
{
	// [1] find CComponent of ID_in in destinationSet_inout. if found, do nothing
	// [2] find CComponent of ID_in in openComponents_inout
	//			if found, add to destinationSet_inout and remove from openComponents_inout
	
	for (unsigned int i = 0; i < destionationSet_inout.size(); i++)
	{
		if (destionationSet_inout[i]->GetID() == ID_in)
			return;
	}

	// if not in destinationSet_inout
	vector<CComponent*>::iterator i;
	for (i = openComponents_inout.begin(); i != openComponents_inout.end(); i++)
	{
		CComponent* ccomponent = *i;
		if (ccomponent->GetID() == ID_in)
		{
			destionationSet_inout.push_back(ccomponent);
			break;
		}
	}

	if (i != openComponents_inout.end())
	{
		openComponents_inout.erase(i);
	}
}


long CEdgeAnalyzer::FindSize2FitEdges(long ID_in, vector<CEdge*> &openEdges_in, vector<CEdge*> &closedEdges_out)
{
	// open edge's destination == ID_in, add to closedEdges_out
	long assemblyParentID = -1;
	for (unsigned int i = 0; i < openEdges_in.size(); i++)
	{
		if (openEdges_in[i]->destination->GetID() == ID_in)
		{
			closedEdges_out.push_back(openEdges_in[i]);
			long tmp = openEdges_in[i]->source->GetID();

			// find source component's parent assembly ID
			map<long, long>::const_iterator ci = components2Asm_Table.find(tmp);

			if (ci != components2Asm_Table.end())
			{
				tmp = ci->second;
				if (assemblyParentID < 0)
					assemblyParentID = tmp;
				else
				{
					if (assemblyParentID != tmp)
						assemblyParentID = -1;
				}
			}
		}
	}

	return assemblyParentID;
}