/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CheckConstraint.h"
#include "UdmFormatter.h"
#include "UdmGme.h"
#include "CommonTraverse.h"

#include <iterator>
#include "string_utils.h"
#include "Logger.h"
#include "CommonData.h"
#include "CommonHelper.h"


Checker::Checker():
	m_errorCount(0),
	m_warningCount(0),
	m_format("Creo")
{
}


bool Checker::CheckConstraintStart(const CyPhyML::ComponentAssembly& assembly_in, 
								set<CyPhyML::Component> &AllCyPhyComponents_inout, 
								set<CyPhyML::Component> &AllCyPhySize2Fit_inout,
								set<long> &RefCoordSystemComponentIDs_inout,
								int &errors_out,
								int &warnings_out)
{
	string message, tmp;

	CheckAssembly(assembly_in, AllCyPhyComponents_inout, AllCyPhySize2Fit_inout);
	CheckJoinStructureConnections();

	FindReferenceCoordSystemComponents(assembly_in, RefCoordSystemComponentIDs_inout);

	for (set<CyPhyML::Component>::const_iterator ci = RefCoordSystemComponents.begin(); ci != RefCoordSystemComponents.end(); ci++)
	{
		int six_degree_constraints = 0;
		set<CyPhyML::StructuralInterface> si_set = ci->StructuralInterface_kind_children();
		for (set<CyPhyML::StructuralInterface>::const_iterator di = si_set.begin(); di != si_set.end(); di++)
		{
			CyPhyML::StructuralInterface s_intf = *di;
			if (GetDegreesOfFreedom(s_intf) == 6)
				six_degree_constraints++;
		}

		if (six_degree_constraints < 1)
		{
			message = "Component choosen as a ReferenceCoordinateSystem only has partial constraints (StructuralInterface has <3 Features), assembly might not get built correctly by Creo [" + (string)ci->name() + "]";
			CreateLogMessage(message, WARNING);
		}
	}

	if (AllCyPhyComponents_inout.empty())
	{
		message = "Component Assembly does not contain any Components: [" + (string)assembly_in.name() + "]";
		CreateLogMessage(message, TIP_REFERENCE_ERROR);
	}


	errors_out = m_errorCount;
	warnings_out = m_warningCount;
	
	if (m_errorCount > 0)
		return 0;

	return 1;	
}

void Checker::CheckAssembly(const CyPhyML::ComponentAssembly& assembly_in, 
							set<CyPhyML::Component>& AllCyPhyComponents_inout, 
							set<CyPhyML::Component>& AllCyPhySize2Fit_inout)
{
	// Assembly has no ComponentRef 
	// Assembly has no DesignContainer
	// Components have no Forwarder
	// Assemblies have no SI
	// Checks connected StructuralInterfaces' Type attribute
	// Checks connected StructuralInterfaces' datum ports' name


	set<CyPhyML::ComponentRef> componentRef_Set = assembly_in.ComponentRef_kind_children();
	for (set<CyPhyML::ComponentRef>::const_iterator ci = componentRef_Set.begin(); ci != componentRef_Set.end(); ci++)
	{
		std::string message = "Found ComponentRef: [" + (string)ci->name() + "]"; //. Make sure this is an expanded design configuration.";
		CreateLogMessage(message, COMPONENT_REF);
	}

	set<CyPhyML::StructuralInterface> si_Set = assembly_in.StructuralInterface_kind_children();
	if (si_Set.size() > 0)
	{
		std::string message = "StructuralInterface in Assembly: [" + (string)assembly_in.name() + " {";
		for (set<CyPhyML::StructuralInterface>::const_iterator ci = si_Set.begin(); ci != si_Set.end(); ci++)
			message += (string)ci->name() + ",";
		message += " }]";			
		CreateLogMessage(message, ASSEMBLY_SI);
	}

	set<CyPhyML::Component> TIPs;
	set<CyPhyML::Component> comp_Set = assembly_in.Component_kind_children();
	for (set<CyPhyML::Component>::const_iterator ci = comp_Set.begin(); ci != comp_Set.end(); ci++)
	{
		CyPhyML::Component component(*ci);
		if (CheckComponent(component, TIPs))
		{
			set<CyPhyML::SizedToFit> size2fit_Set = ci->SizedToFit_kind_children();
			if (!size2fit_Set.empty())
				AllCyPhySize2Fit_inout.insert(*ci);
			else
				AllCyPhyComponents_inout.insert(*ci);
		}

	} // for Component
	
	set<CyPhyML::ComponentAssembly> asm_Set = assembly_in.ComponentAssembly_kind_children();
	for (set<CyPhyML::ComponentAssembly>::const_iterator zi = asm_Set.begin(); zi != asm_Set.end(); zi++)
	{
		CyPhyML::ComponentAssembly assembly = *zi;
		CheckAssembly(assembly, AllCyPhyComponents_inout, AllCyPhySize2Fit_inout);
	}
}



// new
void Checker::FindConnectedStructuralInterfacePort(const CyPhyML::StructuralInterface &begin_port_in)
{
	std::string message, link = begin_port_in.name();

	map<CyPhyML::StructuralInterface, CyPhyML::JoinStructures> JS_Set;
	set<CyPhyML::JoinStructures> srcJS = begin_port_in.srcJoinStructures();
	set<CyPhyML::JoinStructures> dstJS = begin_port_in.dstJoinStructures();

	int src_size_tmp = srcJS.size(), dst_size_tmp = dstJS.size();
	if (srcJS.size() + dstJS.size() > 1)
	{
		CreateLogMessage("StructuralInterface has multiple connections: [" + link + "]", SI_CONNECTION_ERROR);
		return;
	}

	for (set<CyPhyML::JoinStructures>::const_iterator zi = srcJS.begin(); zi != srcJS.end(); zi++)
	{
		if (m_visitedJoin.find(*zi) == m_visitedJoin.end())
			JS_Set[begin_port_in] = *zi;
	}

	for (set<CyPhyML::JoinStructures>::const_iterator xi = dstJS.begin(); xi != dstJS.end(); xi++)
	{
		if (m_visitedJoin.find(*xi) == m_visitedJoin.end())
			JS_Set[begin_port_in] = *xi;
	}


	for (map<CyPhyML::StructuralInterface, CyPhyML::JoinStructures>::const_iterator yi = JS_Set.begin(); yi != JS_Set.end(); yi++)
	{
		StructuralInterfacePortVisitor visitor;
		visitor.Visit_JoinStructures(yi->second);

		int size = visitor.m_StructuralInterfaceRolePairs.size();
		if (size > 2)
		{
			message = "Multiple StructuralInterface ports are connected: [" + link;
			for (int i = 0; i < size; i++)
			{
				CyPhyML::StructuralInterface endPort = visitor.m_StructuralInterfaceRolePairs[i];
				if (endPort != begin_port_in)
					message += ", " + (string)endPort.name();
			}

			CreateLogMessage(message, SI_CONNECTION_ERROR);
		}
		else if (size < 2)
		{
			message = "StructuralInterface connected to a Forwarder, check connection: [" + link + "]";
			CreateLogMessage(message, WARNING);												//CreateLogMessage(message, SI_CONNECTION_ERROR);
		}
		else
		{
			CyPhyML::StructuralInterface endPort;
			if (visitor.m_StructuralInterfaceRolePairs[0] == begin_port_in)
				endPort = visitor.m_StructuralInterfaceRolePairs[1];
			else
				endPort = visitor.m_StructuralInterfaceRolePairs[0];
			
			if (m_validStructuralInterface.find(endPort) != m_validStructuralInterface.end())
			{
				m_connectedStructuralInterface[begin_port_in] = endPort;
				m_connectedStructuralInterface[endPort] = begin_port_in;
				CheckDatumPortNames(begin_port_in, endPort);

				// add to JS_SI_Lookup_Table
				for (set<CyPhyML::JoinStructures>::const_iterator xi = visitor.m_JoinStructuresVisited.begin(); xi != visitor.m_JoinStructuresVisited.end(); xi++)
					CommonData::JS_SI_Lookup_Table[*xi] = make_pair(begin_port_in, endPort);
			}
			else
			{
				CreateLogMessage("StructuralInterface is connected to an invalid StructuralInterface [" + (string)CyPhyML::MgaObject::Cast(begin_port_in.parent()).name() + "/" + link + ", " + (string)CyPhyML::MgaObject::Cast(endPort.parent()).name() + "/" + (string)endPort.name() + "]", WARNING);
			}
		}
		m_visitedJoin.insert(visitor.m_JoinStructuresVisited.begin(), visitor.m_JoinStructuresVisited.end());
	}
}

bool Checker::HasConnectedGeometryFeature(const CyPhyML::StructuralInterfaceFeature &si_feature_in, 
										  const CyPhyML::CADModel &cad_model_in)
{
	// get connected geometry feature of si_feature_in
	// if parent of geometry feature == cad_model_in, then true
	int count = 0;
	std::string si_feature_type = si_feature_in.type().name(), 
		geometry_feature_type = (string)(si_feature_in.type().name()) + "Geometry",
		path = si_feature_in.getPath2("/",false);
	std::set<CyPhyML::FeatureMap> src_set = si_feature_in.srcFeatureMap();
	std::set<CyPhyML::FeatureMap> dst_set = si_feature_in.dstFeatureMap();
	CyPhyML::GeometryFeature geometry_feature;

	for (std::set<CyPhyML::FeatureMap>::const_iterator ci = src_set.begin(); ci != src_set.end(); ci++)
	{
		CyPhyML::SolidModelingFeature solid_feature = ci->srcFeatureMap_end();
		if (solid_feature.GetParent().uniqueId() == cad_model_in.uniqueId())
		{
			if ((string)solid_feature.type().name() != geometry_feature_type)
				CreateLogMessage("The type of StructuralInterface datum port Does Not match the type of connected datum port in CADModel object [" + path + "]", WARNING);
			else
			{
				count++;
				geometry_feature = CyPhyML::GeometryFeature::Cast(solid_feature);
			}
		}
	}

	for (std::set<CyPhyML::FeatureMap>::const_iterator ci = dst_set.begin(); ci != dst_set.end(); ci++)
	{
		CyPhyML::SolidModelingFeature solid_feature = ci->dstFeatureMap_end();
		if (solid_feature.GetParent().uniqueId() == cad_model_in.uniqueId())
		{
			if ((string)solid_feature.type().name() != geometry_feature_type)
				CreateLogMessage("The type of StructuralInterface datum port Does Not match the type of connected datum port in CADModel object [" + path + "]", WARNING);
			else
			{
				count++;
				geometry_feature = CyPhyML::GeometryFeature::Cast(solid_feature);
			}
		}
	}

	if (count > 1)			// n
	{	
		CreateLogMessage("StructuralInterface datum port is connected to multiple datum ports in CADModel object [" + path + "]", WARNING);
		CommonData::SIFeature_To_GeometryFeature_Lookup_table[si_feature_in] = geometry_feature;
		return 0;
	}
	else if (count < 1)		// 0
	{
		CreateLogMessage("StructuralInterface datum port is NOT connected to any datum ports in CADModel object [" + path + "]", WARNING);
		return 0;
	}
	else					// 1
	{
		CommonData::SIFeature_To_GeometryFeature_Lookup_table[si_feature_in] = geometry_feature;
		return 1;
	}
}

bool Checker::CheckComponent(CyPhyML::Component &component_in,
							set<CyPhyML::Component> &TIP_components_in)
{
	bool status = 1;
	int count = 0;
	std::string component_name = component_in.name();

	CyPhyML::CADModel cad_model_of_format;
	set<CyPhyML::CADModel> cadModel_Set = component_in.CADModel_kind_children();
	for (set<CyPhyML::CADModel>::const_iterator ci = cadModel_Set.begin(); ci != cadModel_Set.end(); ci++)
	{
		if ((std::string)ci->FileFormat() == m_format)
		{
			cad_model_of_format = *ci;
			count++;
			CommonData::CADModel_Lookup_table[component_in] = cad_model_of_format;
		}
	}

	if (count >= 1)
	{
		if (count > 1)
			CreateLogMessage("Component has >1 CADModel of file format -+" + m_format + "-+ [" + component_name + "]", WARNING);
	}
	else
	{
		CreateLogMessage("Component is missing CADModel object, it will be excluded from assembly xml: [" + component_name + "]", WARNING);
		return 0;			// no need to execute the rest of the function
	}


	set<CyPhyML::StructuralInterfaceForwarder> fwder_Set = component_in.StructuralInterfaceForwarder_kind_children();
	if (fwder_Set.size() > 0)
	{
		CreateLogMessage("Forwarder in Component: [" + (string)component_in.name() + "]", WARNING);
	}

	set<CyPhyML::StructuralInterface> si_Set = component_in.StructuralInterface_kind_children();
	if (si_Set.empty())
	{
		CreateLogMessage("Component does not have StructuralInterface ports: [" + component_name + "]", WARNING);
	}

	for (set<CyPhyML::StructuralInterface>::const_iterator di = si_Set.begin(); di != si_Set.end(); di++)
	{				
		CyPhyML::StructuralInterface beginPort = *di;
		std::string link = beginPort.name();

		if (!beginPort.IgnoreInterface())
		{
			int unconnected = 0;
			std::set<CyPhyML::StructuralInterfaceFeature> si_feature_port_set = beginPort.StructuralInterfaceFeature_kind_children();
			for (std::set<CyPhyML::StructuralInterfaceFeature>::const_iterator ei = si_feature_port_set.begin(); ei != si_feature_port_set.end(); ei++)
			{
				if (!HasConnectedGeometryFeature(*ei, cad_model_of_format))
					unconnected++;
			}

			if (unconnected == 0)
				m_validStructuralInterface.insert(*di);
			else
				CreateLogMessage("Can not match all of StructuralInterface's datum ports to CADModel's datum ports [" + component_name + "/" + link + "]", GENERAL_ERROR);
		}
	}

	return status;
}


void Checker::CheckDatumPortNames(const CyPhyML::StructuralInterface& a, 
								  const CyPhyML::StructuralInterface& b)
{
	std::string message;
	std::string a_hyperlink = (string)CyPhyML::MgaObject(a.parent()).name() + "/" + (string)a.name(), b_hyperlink = (string)CyPhyML::MgaObject(b.parent()).name() + "/" +  (string)b.name();

	set<CyPhyML::SolidModelingFeature> a_Features = a.SolidModelingFeature_kind_children();
	set<CyPhyML::SolidModelingFeature> b_Features = b.SolidModelingFeature_kind_children();
	size_t a_size = a_Features.size(), 
		   b_size = b_Features.size();

	std::string ta = a.Type(), tb = b.Type();
	if (MatchSIType(ta, tb) == false)
	{
		std::string message = "Mismatched Type attribute on connected StructuralInterfaces: [" +  a_hyperlink + "/" + ta + "," +  b_hyperlink + "/" + tb +"]";
		CreateLogMessage(message, MISMATCH_SI_TYPE);
	}


	if (a_Features.size() != b_Features.size())
	{
		message = "Connected StructuralInterfaces have different number of datum ports: [" + a_hyperlink + ", " + b_hyperlink + "]";
		CreateLogMessage(message, DATUM_PORT_ERROR);
		return;
	}

	set<CyPhyML::CoordinateSystem> a_coord_set = a.CoordinateSystem_kind_children();
	if (!a_coord_set.empty())
	{
		if (a_Features.size() > 1)
		{
			message = "StructuralInterface with CoordinateSystem datum port should not have other types of datum: [" + a_hyperlink + "]";
			CreateLogMessage(message, DATUM_PORT_ERROR);
		}
	}

	set<CyPhyML::CoordinateSystem> b_coord_set = b.CoordinateSystem_kind_children();
	if (!b_coord_set.empty())
	{
		if (b_Features.size() > 1)
		{
			message = "StructuralInterface with CoordinateSystem datum port should not have other types of datum: [" + b_hyperlink + "]";
			CreateLogMessage(message, DATUM_PORT_ERROR);
		}
	}


	set<string> a_port_names;
	set<string> b_port_names;
	for (set<CyPhyML::SolidModelingFeature>::const_iterator ci = a_Features.begin(); ci != a_Features.end(); ci++)
			a_port_names.insert(ci->name());
	for (set<CyPhyML::SolidModelingFeature>::const_iterator ci = b_Features.begin(); ci != b_Features.end(); ci++)
			b_port_names.insert(ci->name());
	if (a_port_names.size() != a_size)
	{
		message = "StructuralInterface datum ports do not have unique names: [" + a_hyperlink + "]";
		CreateLogMessage(message, DATUM_PORT_ERROR);
	}
	if (b_port_names.size() != b_size)
	{
		message = "StructuralInterface datum ports do not have unique names: [" + b_hyperlink + "]";
		CreateLogMessage(message, DATUM_PORT_ERROR);
	}

	if (a_size > 1 && b_size > 1)				// 1-8-2013: Hack for Ryan so Ricardo model w/ 1 datum port would work.
	{
		for (set<CyPhyML::SolidModelingFeature>::const_iterator ci = a_Features.begin(); ci != a_Features.end(); ci++)
		{
			string src_name = ci->name();
			a_port_names.insert(src_name);

			for (set<CyPhyML::SolidModelingFeature>::const_iterator di = b_Features.begin(); di != b_Features.end(); di++)
			{
					if (di->name() == src_name)
					{
						if (di->type() != ci->type())
						{
							message = "Matching StructuralInterface datum ports do not have the same type: [" + b_hyperlink + "/" + src_name + + ", " + a_hyperlink + "/" + src_name + "]";
							CreateLogMessage(message, DATUM_PORT_ERROR);
						}
						else
						{
							if (di->type() == CyPhyML::Surface::meta)
								if ((string)CyPhyML::Surface::Cast(*di).Alignment() != (string)CyPhyML::Surface::Cast(*ci).Alignment())
								{
									message = "Matching StructuralInterface datum ports do not have the same Alignment attribute value: [" + b_hyperlink + "/" + src_name + ", " + a_hyperlink + "/" + src_name + "]";
									CreateLogMessage(message, DATUM_PORT_ERROR);
								}
						}

						b_port_names.erase(di->name());
						a_port_names.erase(ci->name());

						break;
					}
			}
		
		}

		if (!a_port_names.empty() || !b_port_names.empty())
		{
			message = "Connected StructuralInterfaces contains unmatched named feature ports: [" + a_hyperlink;
			if (!a_port_names.empty())
			{
				string unmatched = "|";
				for (set<string>::const_iterator ei = a_port_names.begin(); ei != a_port_names.end(); ei++)
					unmatched += *ei + "|";
				message += "/" + unmatched; 
			}

			message += ", " + b_hyperlink;
			if (!b_port_names.empty())
			{
				string unmatched = "|";
				for (set<string>::const_iterator ei = b_port_names.begin(); ei != b_port_names.end(); ei++)
					unmatched += *ei + "|";
				message += "/" + unmatched; 
			}

			message += "]";
			CreateLogMessage(message, DATUM_PORT_ERROR);
		}
	}
}

void Checker::CreateLogMessage(std::string& message, ConstraintType type)
{
	int i = type;
	if (i > 0)
	{
		m_errorCount++;
		LOGMSG(message, MSG_ERROR);
	}
	else
	{
		m_warningCount++;
		LOGMSG(message, MSG_WARNING);
	}

}

bool Checker::CheckConstraintStart(const CyPhyML::CADTestBench& testBench_in, 
								   set<CyPhyML::Component>& AllCyPhyComponents_inout, 
								   set<CyPhyML::Component>& AllCyPhySize2Fit_inout,
								   int &errors_out,
								   int &warnings_out)
{
	string message, tmp;
	set<CyPhyML::Component> component_TIPs;

	set<CyPhyML::ComponentAssembly> ca_set = testBench_in.ComponentAssembly_kind_children();
	if (ca_set.size() != 1)
	{
		message = "CAD Test Bench must contain 1 Component Assembly to serve as SUT: [" + (string)testBench_in.name() + "]";
		CreateLogMessage(message, CAD_TB_CA_ERROR);
	}
	else
	{
		CyPhyML::ComponentAssembly ca = *ca_set.begin();
		string path = ca.getPath2("/", false);
	}

	set<CyPhyML::TestInjectionPoint> TIPSet = testBench_in.TestInjectionPoint_kind_children();
	for (set<CyPhyML::TestInjectionPoint>::const_iterator ci = TIPSet.begin(); ci != TIPSet.end(); ci++)
	{
		CyPhyML::TIPRefBase tip = ci->ref();
		if (tip != Udm::null)
		{
			if (tip.type() == CyPhyML::Component::meta)
			{
				CyPhyML::Component tmp = CyPhyML::Component::Cast(tip);
				component_TIPs.insert(tmp);
			}
			else
			{
				message = "Test Injection Point does not reference a Component: [" + (string)ci->name() + "]";
				CreateLogMessage(message, TIP_REFERENCE_ERROR);
			}
		}
	}

	for (set<CyPhyML::Component>::const_iterator ci = component_TIPs.begin(); ci != component_TIPs.end(); ci++)
	{
		CyPhyML::Component tmp(*ci);
		if (CheckComponent(tmp, component_TIPs))
		{
			set<CyPhyML::SizedToFit> s2f_Set = tmp.SizedToFit_kind_children();
			if (s2f_Set.empty())
				AllCyPhyComponents_inout.insert(tmp);
			else
				AllCyPhySize2Fit_inout.insert(tmp);
		}
	}

	CheckJoinStructureConnections();

	// CAD Test Bench : If there's only 1 
	if (component_TIPs.size() == 1)
	{
		AllCyPhyComponents_inout.insert(*component_TIPs.begin());
	}

	if (AllCyPhyComponents_inout.empty())
	{
		message = "Structural FEA Test Bench does not contain any Test Injection Points that references Components: [" + (string)testBench_in.name() + "]";
		CreateLogMessage(message, TIP_REFERENCE_ERROR);		
	}	
	
	// TODO: Add check for Geometry + AnalysisPoint
	CheckGeometry(testBench_in);

	errors_out = m_errorCount;
	warnings_out = m_warningCount;

	if (m_errorCount > 0)
		return 0;

	return 1;
}

bool Checker::IsTIPInSUT(string &sutPath, CyPhyML::Component &tip)
{
	bool status = true;
	return status;
}

void Checker::CheckGeometry(const CyPhyML::CADTestBench &testBench)
{
	string message;
	set<CyPhyML::GeometryTypes> geometry_set = testBench.GeometryTypes_kind_children();
	for (set<CyPhyML::GeometryTypes>::const_iterator ci = geometry_set.begin(); ci != geometry_set.end(); ci++)
	{
		CheckGeometry(*ci);
	}
}

void Checker::CheckGeometry(const CyPhyML::GeometryTypes &geometry_in)
{
	Uml::Class geometry_type = geometry_in.type();
	if (geometry_type == CyPhyML::Circle::meta)
	{
		CheckGeometry(CyPhyML::Circle::Cast(geometry_in));
	}
	else if (geometry_type == CyPhyML::ConcentricCircles::meta)
	{
		CheckGeometry(CyPhyML::ConcentricCircles::Cast(geometry_in));
	}
	else if (geometry_type == CyPhyML::Sphere::meta)
	{
		CheckGeometry(CyPhyML::Sphere::Cast(geometry_in));
	}
	else if (geometry_type == CyPhyML::Cylinder::meta)
	{
		CheckGeometry(CyPhyML::Cylinder::Cast(geometry_in));
	}
	else if (geometry_type == CyPhyML::Extrusion::meta)
	{
		CheckGeometry(CyPhyML::Extrusion::Cast(geometry_in));
	}
	else if (geometry_type == CyPhyML::Polygon::meta)
	{
		CheckGeometry(CyPhyML::Polygon::Cast(geometry_in));
	}
	else if (geometry_type == CyPhyML::CustomGeometry::meta)
	{
		CheckGeometry(CyPhyML::CustomGeometry::Cast(geometry_in));
	}
}

void Checker::CheckGeometry(const CyPhyML::Circle &geometry_in)
{	
	set<CyPhyML::CircleCenter2Component> center_Set = geometry_in.dstCircleCenter2Component();
	set<CyPhyML::CircleSurface2Component> surf_Set = geometry_in.dstCircleSurface2Component();
	if (center_Set.size() == 1 && surf_Set.size() == 2)
	{
		CyPhyML::CircleCenter2Component center_conn = *center_Set.begin();
		CyPhyML::AnalysisPoint start = center_conn.dstCircleCenter2Component_end();
		GetFeaturePoint(start);

		set<CyPhyML::CircleSurface2Component>::const_iterator surf_conn = surf_Set.begin();
		CyPhyML::CircleSurface2Component surf_conn1 = *surf_conn;
		start = surf_conn1.dstCircleSurface2Component_end();
		GetFeaturePoint(start);
		surf_conn++;
		CyPhyML::CircleSurface2Component surf_conn2 = *surf_conn;
		start = surf_conn2.dstCircleSurface2Component_end();
		GetFeaturePoint(start);
	}
	else
	{
		string message = "Circle geometry must have 1 center connection to define center of the circle and 2 connections to define 2 points on the circle using AnalysisPoints in Test Injection Point [" +  (string)geometry_in.name() + "]";
		CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
	}
}

void Checker::CheckGeometry(const CyPhyML::Polygon &geometry_in)
{
	set<CyPhyML::Polygon2Component> comp_Set = geometry_in.dstPolygon2Component();
	if (comp_Set.size() < 3)
		CreateLogMessage("Polygon geometry [" + (string)geometry_in.name() + "] needs to be connected to at least 3 analysis points!" , CAD_TB_GEOMETRY_ERROR);

	for (set<CyPhyML::Polygon2Component>::const_iterator di = comp_Set.begin(); di != comp_Set.end(); di++)
	{
		CyPhyML::Polygon2Component conn(*di);
		CyPhyML::AnalysisPoint start = conn.dstPolygon2Component_end();			
		GetFeaturePoint(start);
	}
}

void Checker::CheckGeometry(const CyPhyML::ConcentricCircles &geometry_in)
{
	set<CyPhyML::CircleCenter2Component> center_Set = geometry_in.dstCircleCenter2Component();
	set<CyPhyML::InnerCircle2Component> inner_Set = geometry_in.dstInnerCircle2Component();
	set<CyPhyML::OuterCircle2Component> outer_Set = geometry_in.dstOuterCircle2Component();

	if (center_Set.size() == 1 && inner_Set.size() == 1 && outer_Set.size() == 1)
	{
		CyPhyML::CircleCenter2Component cconn = *center_Set.begin();
		CyPhyML::AnalysisPoint start = cconn.dstCircleCenter2Component_end();
		GetFeaturePoint(start);
		CyPhyML::InnerCircle2Component iconn = *inner_Set.begin();
		start = iconn.dstInnerCircle2Component_end();
		GetFeaturePoint(start);
		CyPhyML::OuterCircle2Component oconn = *outer_Set.begin();
		start = oconn.dstOuterCircle2Component_end();
		GetFeaturePoint(start);
	}
	else
	{
		string message = "ConcentricCircles geometry must have 1 center connection, 1 inner surface connection, and 1 outer surface connection to define two concentric circles using AnalysisPoint in Test Injection Point [" + (string)geometry_in.name() + "]";
		CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
	}
}

void Checker::CheckGeometry(const CyPhyML::Sphere &geometry_in)
{
	set<CyPhyML::SphereCenter2Component> center_Set = geometry_in.dstSphereCenter2Component();
	set<CyPhyML::SphereSurface2Component> surf_Set = geometry_in.dstSphereSurface2Component();

	if (center_Set.size() == 1 && surf_Set.size() == 1)
	{
		string message = "A sphere geometry must have exactly 1 center connection and 1 surface connection using AnalysisPoints of a Test Injection Point [" + (string)geometry_in.name() +"]";
		CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
	}
	else
	{
		CyPhyML::SphereCenter2Component center_conn = *center_Set.begin();
		CyPhyML::AnalysisPoint start = center_conn.dstSphereCenter2Component_end();
		GetFeaturePoint(start);

		CyPhyML::SphereSurface2Component surf_conn = *surf_Set.begin();
		start = surf_conn.dstSphereSurface2Component_end();
		GetFeaturePoint(start);
	}
}

void Checker::CheckGeometry(const CyPhyML::Cylinder &geometry_in)
{
	set<CyPhyML::CylinderEnds2Component> ends_Set = geometry_in.dstCylinderEnds2Component();
	set<CyPhyML::CylinderSurface2Component> surf_Set = geometry_in.dstCylinderSurface2Component();

	if (ends_Set.size() == 2 && surf_Set.size() == 1)
	{
		CyPhyML::CylinderSurface2Component surf_conn = *surf_Set.begin();
		CyPhyML::AnalysisPoint start = surf_conn.dstCylinderSurface2Component_end();
		GetFeaturePoint(start);

		set<CyPhyML::CylinderEnds2Component>::const_iterator end_conn = ends_Set.begin();
		CyPhyML::CylinderEnds2Component end1 = *end_conn;
		end_conn++;
		CyPhyML::CylinderEnds2Component end2 = *end_conn;

		if ((end1.CylinderPointLocation() == "START" && end2.CylinderPointLocation() == "END") || (end1.CylinderPointLocation() == "END" && end2.CylinderPointLocation() == "START"))
		{
			CyPhyML::AnalysisPoint start = surf_conn.dstCylinderSurface2Component_end();
			GetFeaturePoint(start);
			start = end1.dstCylinderEnds2Component_end();
			GetFeaturePoint(start);
			start = end2.dstCylinderEnds2Component_end();
			GetFeaturePoint(start);
		}
		else
		{
			string message = "Cylinder geometry's 2 ends connections must have different Point Location attribute values [" + (string)geometry_in.name() + "]";
			CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
		}
	}
	else
	{
		string message = "Cylinder geometry must have 1 surface connection and 2 ends connections to AnalysisPoints of a Test Injection Point [" + (string)geometry_in.name() + "]";
		CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
	}
		
}

void Checker::CheckGeometry(const CyPhyML::Extrusion &geometry_in)
{	
	set<CyPhyML::ExtrusionOffset2Component> offset_Set = geometry_in.dstExtrusionOffset2Component();
	set<CyPhyML::ExtrusionSurface2Component> surf_Set = geometry_in.dstExtrusionSurface2Component();

	if (offset_Set.size() == 1 && surf_Set.size() > 2)
	{
		CyPhyML::ExtrusionOffset2Component offset_conn = *offset_Set.begin();
		CyPhyML::AnalysisPoint start = offset_conn.dstExtrusionOffset2Component_end();
		GetFeaturePoint(start);

		for (set<CyPhyML::ExtrusionSurface2Component>::const_iterator di = surf_Set.begin(); di != surf_Set.end(); di++)
		{
			CyPhyML::ExtrusionSurface2Component surf_conn = *di;
			CyPhyML::AnalysisPoint start = surf_conn.dstExtrusionSurface2Component_end();
			GetFeaturePoint(start);
		}
	}
	else
	{
		string message = "Extrusion geometry must have 1 offset connection to define offset distance and at least 3 connections to define a polygon using AnalysisPoints in Test Injection Point [" + (string)geometry_in.name() +  "]";
		CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
	}
}

void Checker::CheckGeometry(const CyPhyML::CustomGeometry &geometry_in)
{	
	string tmp, ids;

	set<CyPhyML::CustomGeometryOperator> operator_set = geometry_in.CustomGeometryOperator_kind_children();
	if (operator_set.empty())
		CreateLogMessage("Custom Geometry [" + (string)geometry_in.name() + "] must have at least 1 operator!", GENERAL_ERROR);

	int count = 0;
	for (set<CyPhyML::CustomGeometryOperator>::const_iterator ci = operator_set.begin(); ci != operator_set.end(); ci++)
	{
		set<CyPhyML::GeometryTypes> my_geometry;
		set<CyPhyML::ApplyGeometryOperator> dst_set = ci->dstApplyGeometryOperator();
		for (set<CyPhyML::ApplyGeometryOperator>::const_iterator di = dst_set.begin(); di != dst_set.end(); di++)
		{
			CyPhyML::CustomGeomtrySetConstruct construct = di->dstApplyGeometryOperator_end();
			if (Udm::IsDerivedFrom(construct.type(), CyPhyML::CustomGeometryOperator::meta))
				CreateLogMessage("Operator in custom geometry can not be connected to another operator [" + (string)ci->name() + "]", GENERAL_ERROR);
			else
			{
				my_geometry.insert(CyPhyML::GeometryTypes::Cast(construct));
				CheckGeometry(CyPhyML::GeometryTypes::Cast(construct));
				count++;
			}
		}

		set<CyPhyML::ApplyGeometryOperator> src_set = ci->srcApplyGeometryOperator();
		for (set<CyPhyML::ApplyGeometryOperator>::const_iterator ei = src_set.begin(); ei != src_set.end(); ei++)
		{
			CyPhyML::CustomGeomtrySetConstruct construct = ei->srcApplyGeometryOperator_end();
			if (Udm::IsDerivedFrom(construct.type(), CyPhyML::CustomGeometryOperator::meta))
				CreateLogMessage("Operator  in custom geometry can not be connected to another operator [" + (string)ci->name() + "]", GENERAL_ERROR);
			else
			{
				my_geometry.insert(CyPhyML::GeometryTypes::Cast(construct));
				CheckGeometry(CyPhyML::GeometryTypes::Cast(construct));
				count++;
			}
		}

		if (my_geometry.size() > 1)
		{
			CommonData::GeometryOperator_table[*ci] = my_geometry;
		}
		else
			CreateLogMessage("Operator  in custom geometry needs to be connected to at least 2 geometry [" + (string)ci->name() + "]", GENERAL_ERROR);
	}

	set<CyPhyML::GeometryTypes> total_geometry_set = geometry_in.GeometryTypes_kind_children();
	if (total_geometry_set.size() != count)
		CreateLogMessage("Custom Geometry [" + (string)geometry_in.name() + "] has geometry not connected to operator!", GENERAL_ERROR);
}

void Checker::GetFeaturePoint(CyPhyML::AnalysisPoint &start_in)
{	
	string message, nameDisplay = (string)CyPhyML::MgaObject(start_in.parent()).name();;
	AnalysisPointVisitor apVisitor;
	apVisitor.FindAnalysisPointEnds(start_in);
	
	if (apVisitor.GetPointsFoundSize() == 0)
	{
		message = "An Analysis Point does not end in a Feature Point [" +  nameDisplay + "]";
		CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
		return;
	}
	else if (apVisitor.GetPointsFoundSize() > 1)
	{
		message = "An Analysis Point ends in multiple Feature Points [" +  nameDisplay + "]";
		CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
		return;
	}
	else
	{
		CyPhyML::PointGeometry featurePoint = apVisitor.GetFoundPoint();
		CyPhyML::MgaObject mgaObj = featurePoint.parent();
		if (mgaObj.type() == CyPhyML::CADModel::meta)
			CommonData::geometryConn_AnalysisPoint_table[start_in] = featurePoint;
		else
		{
			message = "An Analysis Point must end in a Feature Point whose parent is a CADModel [" + nameDisplay + "]";
			CreateLogMessage(message, CAD_TB_GEOMETRY_ERROR);
		}
	}
}

void Checker::FindReferenceCoordSystemComponents(const CyPhyML::ComponentAssembly &assembly_in,
												set<long> &RefCoordSystemComponentIDs_inout)
{
	set<CyPhyML::ReferenceCoordinateSystem> refCoordSys_Set = assembly_in.ReferenceCoordinateSystem_kind_children();
	for (set<CyPhyML::ReferenceCoordinateSystem>::const_iterator ci = refCoordSys_Set.begin(); ci != refCoordSys_Set.end(); ci++)
	{
		CyPhyML::ReferenceCoordinateSystem coord(*ci);
		RefCoordinateSystemVisitor visitor;
		coord.Accept(visitor);

		RefCoordSystemComponents.insert(visitor.FoundRefCoordComponents.begin(), visitor.FoundRefCoordComponents.end());
		RefCoordSystemComponentIDs_inout.insert(visitor.FoundRefCoordIDs.begin(), visitor.FoundRefCoordIDs.end());
	}

	//int size = RefCoordSystemComponentIDs.size();
}


bool Checker::CheckConstraintStart(const CyPhyML::TestBench &testBench_in)
{
	bool status = true;
	set<CyPhyML::Metric> metric_set = testBench_in.Metric_kind_children();

	for (set<CyPhyML::Metric>::const_iterator ci = metric_set.begin(); ci != metric_set.end(); ci++)
	{
		CyPhyML::Metric metric(*ci);
		set<CyPhyML::CADComputation2Metric> comp2metric_set = metric.srcCADComputation2Metric();
		set<CyPhyML::PointCoordinates2Metric> pt2metric_set = metric.srcPointCoordinates2Metric();

		if ((comp2metric_set.size() + pt2metric_set.size()) > 1)
		{
			string msg = "Metric object connected to multiple computation objects and/or analysis points [" + (string)metric.name() + "]";
			CreateLogMessage(msg, CAD_TB_COMPUTATION_BLOCK_ERROR);
			status = false;
		}

		if (!pt2metric_set.empty())
		{
			CyPhyML::AnalysisPoint src_ap = pt2metric_set.begin()->srcPointCoordinates2Metric_end();
			GetFeaturePoint(src_ap);
		}
	}

	return status;
}

void Checker::CheckJoinStructureConnections()
{
	for (std::set<CyPhyML::StructuralInterface>::const_iterator ci = m_validStructuralInterface.begin(); ci != m_validStructuralInterface.end(); ci++)
	{
		FindConnectedStructuralInterfacePort(*ci);
	}
}


bool Checker::MatchSIType(string& a_original, 
						  string& b_original)
{
	string a = a_original;
	string b = b_original;
	tolower(a);
	tolower(b);

	unsigned int a_size = a.size();
	unsigned int b_size = b.size();

	if (a == b)
		return true;
	else
	{
		if (a_size != b_size)
			return false;

		for (unsigned int i = 0; i < a_size; i++)
		{
			if (a[i] != '*' && b[i] != '*')
				if (a[i] != b[i])
					return false;
		}

		return true;
	}
}