/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CommonData.h"
#include "UdmGme.h"
#include "string_utils.h"


map<long, CAssembly*> CommonData::AllAssemblies;						
map<long, CComponent*> CommonData::AllComponents;						
vector<CEdge*> CommonData::AllEdges;									
map<long, CForwarder*> CommonData::AllForwarders;						
map<long, CComponentSize2Fit*> CommonData::AllSize2Fit;				
Join_StructuralInterfacePair CommonData::JS_SI_Lookup_Table;
AnalysisPoint_FeaturePoint_Map CommonData::geometryConn_AnalysisPoint_table;
map<CyPhyML::CustomGeometryOperator, set<CyPhyML::GeometryTypes>> CommonData::GeometryOperator_table;
map<CyPhyML::TestInjectionPoint, CyPhyML::Component> CommonData::TIP_Component_table;
map<CyPhyML::Component, CyPhyML::CADModel> CommonData::CADModel_Lookup_table;
map<CyPhyML::StructuralInterfaceFeature, CyPhyML::GeometryFeature> CommonData::SIFeature_To_GeometryFeature_Lookup_table;

/** \brief Creates a new CComponent from CyPhyML::Component
	\param [in] cyphyComponent CyPhyML::Component reference
	\return void
*/
CComponent* CommonData::CreateCComponent(CyPhyML::Component& cyphyComponent_in)
{
	long id = cyphyComponent_in.uniqueId();
	map<long, CComponent*>::iterator i = AllComponents.find(id);
	if (i == AllComponents.end())
	{
		CyPhyML::CADModel cad_model;
		map<CyPhyML::Component, CyPhyML::CADModel>::const_iterator ci = CADModel_Lookup_table.find(cyphyComponent_in);
		if (ci != CADModel_Lookup_table.end())
			cad_model = ci->second;

		CComponent* ccomp = new CComponent(id, cyphyComponent_in, cad_model);
		AllComponents[id] = ccomp;
		return ccomp;
	}
	else
		return i->second;
}

/** \brief Creates a new CAssembly from CyPhyML::ComponentAssembly
	\param [in] cyphyAssembly CyPhyML::ComponentAssembly reference
	\return void
*/
CAssembly* CommonData::CreateCAssembly(CyPhyML::ComponentAssembly& cyphyAssembly_in)
{
	long id = cyphyAssembly_in.uniqueId();
	map<long, CAssembly*>::iterator i = AllAssemblies.find(id);
	if (i == AllAssemblies.end())
	{
		CAssembly* cassembly = new CAssembly(id, cyphyAssembly_in);
		//cassembly->needflattening = RequireFlattening(cyphyAssembly);
		//string tmp;
		//to_string(tmp, cyphyAssembly_in.ID());
		//cassembly->SetAssemblyInterface_ID(tmp);
		cassembly->SetAssemblyInterface_ID(cyphyAssembly_in.ConfigurationUniqueID());

		cassembly->needflattening = true;
		AllAssemblies[id] = cassembly;
		return cassembly;
	}
	else
		return i->second;
}

// Typically used when making islands
CAssembly* CommonData::CreateCAssembly(CyPhyML::ComponentAssembly& cyphyAssembly_in, 
									long ID_in)
{
	map<long, CAssembly*>::iterator i = AllAssemblies.find(ID_in);
	if (i == AllAssemblies.end())
	{
		CAssembly* cassembly = new CAssembly(ID_in, cyphyAssembly_in);
		//cassembly->needflattening = RequireFlattening(cyphyAssembly);
		//string tmp;
		//to_string(tmp, cyphyAssembly_in.ID());
		//cassembly->SetAssemblyInterface_ID(tmp);
		cassembly->SetAssemblyInterface_ID(cyphyAssembly_in.ConfigurationUniqueID());

		cassembly->needflattening = true;
		AllAssemblies[ID_in] = cassembly;
		return cassembly;
	}
	else
		return i->second;
}

// Typically used when making islands
CAssembly* CommonData::CreateCAssembly(CyPhyML::ComponentAssembly& cyphyAssembly_in, 
										long ID_in, 
										string assemblyInterfaceID_in)
{
	map<long, CAssembly*>::iterator i = AllAssemblies.find(ID_in);
	if (i == AllAssemblies.end())
	{
		CAssembly* cassembly = new CAssembly(ID_in, cyphyAssembly_in);
		//cassembly->needflattening = RequireFlattening(cyphyAssembly);
		cassembly->SetAssemblyInterface_ID(assemblyInterfaceID_in);

		cassembly->needflattening = true;
		AllAssemblies[ID_in] = cassembly;
		return cassembly;
	}
	else
		return i->second;
}

CAssembly* CommonData::CreateCAssembly(CyPhyML::CADTestBench& testBench)
{
	long id = testBench.uniqueId();
	map<long, CAssembly*>::iterator i = AllAssemblies.find(id);
	if (i == AllAssemblies.end())
	{
		string tmp;
		CAssembly* cassembly = new CAssembly(id, testBench);
		to_string(tmp, id);
		cassembly->SetAssemblyInterface_ID(tmp);
		//cassembly->needflattening = RequireFlattening(cyphyAssembly);
		cassembly->needflattening = true;
		AllAssemblies[id] = cassembly;
		return cassembly;
	}
	else
		return i->second;
}

CAssembly* CommonData::CreateCAssembly(CyPhyML::MgaObject& cyphyref_in, 
										long ID_in, 
										string assemblyInterfaceID_in)
{
	Uml::Class meta_type = cyphyref_in.type();
	if (meta_type == CyPhyML::ComponentAssembly::meta)
	{
		return CommonData::CreateCAssembly(CyPhyML::ComponentAssembly::Cast(cyphyref_in), ID_in, assemblyInterfaceID_in);
	}
	else if (meta_type == CyPhyML::CADTestBench::meta)
	{
		return CommonData::CreateCAssembly(CyPhyML::CADTestBench::Cast(cyphyref_in));
	}
}


/** \brief Creates a new CForwarder from CyPhyML::StructuralInterfaceForwarder
	\param [in] port CyPhyML::StructuralInterfaceForwarder reference
	\return void
*/
CForwarder* CommonData::CreateCForwarder(CyPhyML::StructuralInterfaceForwarder& port)
{
	string ntmp = port.name(), itmp = UdmGme::UdmId2GmeId(port.uniqueId());
	long id = port.uniqueId();
	map<long, CForwarder*>::iterator i = AllForwarders.find(id);
	if (i == AllForwarders.end())
	{
		CForwarder* cfwd = new CForwarder(port, id);
		AllForwarders[id] = cfwd;
		return cfwd;
	}
	else
		return i->second;
}

CComponentSize2Fit* CommonData::CreateCSize2Fit(CyPhyML::Component &cyphyComponent_in)
{
	long id = cyphyComponent_in.uniqueId();
	map<long, CComponentSize2Fit*>::iterator i = AllSize2Fit.find(id);
	if (i == AllSize2Fit.end())
	{
		CyPhyML::CADModel cad_model;
		map<CyPhyML::Component, CyPhyML::CADModel>::const_iterator ci = CADModel_Lookup_table.find(cyphyComponent_in);
		if (ci != CADModel_Lookup_table.end())
			cad_model = ci->second;

		CComponentSize2Fit* ccomp = new CComponentSize2Fit(id, cyphyComponent_in, cad_model);
		AllSize2Fit[id] = ccomp;
		return ccomp;
	}
	else
		return i->second;
}

CComponent* CommonData::FindCComponent(long id)
{
	CComponent* found = 0;
	map<long, CComponent*>::const_iterator ci =  AllComponents.find(id);
	if (ci != AllComponents.end())
		return ci->second;

	return found;
}

void CommonData::AddCEdge(CEdge *newEdge)
{
	AllEdges.push_back(newEdge);
}

void CommonData::CleanData()
{
	for (vector<CEdge*>::iterator i = AllEdges.begin(); i != AllEdges.end(); i++)
		delete *i;
	AllEdges.clear();

	for (map<long, CForwarder*>::iterator i = AllForwarders.begin(); i != AllForwarders.end(); i++)
		delete i->second;
	AllForwarders.clear();

	for (map<long, CComponent*>::iterator i = AllComponents.begin(); i != AllComponents.end(); i++)
		delete i->second;
	AllComponents.clear();

	for (map<long, CAssembly*>::iterator i = AllAssemblies.begin(); i != AllAssemblies.end(); i++)
		delete i->second;
	AllAssemblies.clear();

	for (map<long, CComponentSize2Fit*>::iterator i = AllSize2Fit.begin(); i != AllSize2Fit.end(); i++)
		delete i->second;
	AllSize2Fit.clear();
}