/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CommonHelper.h"
#include "UdmGme.h"
#include "string_utils.h"
#include "UdmFormatter.h"
#include "Logger.h"
#include "CommonData.h"



long FindFeatureArchetypeID(CyPhyML::SolidModelingFeature& feature)
{
	CyPhyML::SolidModelingFeature archetype = feature.Archetype();
	while (archetype.isInstance() || archetype.isSubtype())
	{
		archetype = archetype.Archetype();
	}

	return archetype.uniqueId();
}

void AddConstraint2CADComponent(AssemblyInterface::CADComponent& cadComponent, 
								const CyPhyML::StructuralInterface& a, 
								const CyPhyML::StructuralInterface& b)
{
	// GMEConsole::Console::writeLine("AddConstraint2CADComponent() cadComponentID:" + std::string(cadComponent.ComponentID()) + " aPortID:" + UdmGme::UdmId2GmeId(a.uniqueId()) + " bPortID:" + UdmGme::UdmId2GmeId(b.uniqueId()), MSG_INFO);
	CyPhyML::MgaObject aMga = a.parent();
	CyPhyML::MgaObject bMga = b.parent();

	if (aMga.type() != CyPhyML::Component::meta && bMga.type() != CyPhyML::Component::meta)
	{
		LOGMSG("StructuralInterface ports' parents must be Components. [" + (string)a.name() + ", " + (string)b.name() + "]", MSG_ERROR);				
		return;
	}

#ifdef USE_GMEID
	std::string aId = UdmGme::UdmId2GmeId(aMga.uniqueId());
	std::string bId = UdmGme::UdmId2GmeId(bMga.uniqueId());
#else
	//__int64 iAId = CyPhyML::DesignEntity::Cast(aMga).ID();
	//__int64 iBId = CyPhyML::DesignEntity::Cast(bMga).ID();
	std::string aId, bId;
	// DY: 12/13/12
	//to_string(aId, CyPhyML::DesignEntity::Cast(aMga).ID()); to_string(bId, CyPhyML::DesignEntity::Cast(bMga).ID());
	aId = CyPhyML::Component::Cast(aMga).InstanceGUID();
	bId = CyPhyML::Component::Cast(bMga).InstanceGUID();
#endif
	
	AssemblyInterface::Constraint constraint = AssemblyInterface::Constraint::Create (cadComponent);
	set<CyPhyML::StructuralInterfaceFeature> b_set = b.StructuralInterfaceFeature_kind_children();
	set<CyPhyML::StructuralInterfaceFeature> a_set = a.StructuralInterfaceFeature_kind_children();


	int count = 0;
	for (set<CyPhyML::StructuralInterfaceFeature>::iterator it = a_set.begin(); it != a_set.end(); it++)
	{
		CyPhyML::StructuralInterfaceFeature a_feature(*it);
		std::string a_TypeID = a_feature.name();

		// iterate through Structural Interface Role Dest_End's feature ports and see if any of them is connected to source feature port
		for (set<CyPhyML::StructuralInterfaceFeature>::iterator dit = b_set.begin(); dit != b_set.end(); dit++)
		{
			CyPhyML::StructuralInterfaceFeature b_feature(*dit);   				

			std::string b_TypeID = b_feature.name();
			if (a_TypeID == b_TypeID || (a_set.size() == 1 && b_set.size() == 1))		// 1-8-2013: Hack for Ryan so Ricardo model w/ 1 datum port would work.
			{

//				GMEConsole::Console::writeLine("======= Found matching features [" + ta + "," + tb + "]", MSG_INFO);

				// Create a pair
				AssemblyInterface::Pair pair = AssemblyInterface::Pair::Create(constraint);
				pair.FeatureInterfaceType() = "CAD_DATUM";
				pair.FeatureAlignmentType() = "ALIGN";
				if (a_feature.type() == CyPhyML::Axis::meta)
					pair.FeatureGeometryType() = "AXIS";
				else if (a_feature.type() == CyPhyML::Surface::meta)
				{
					pair.FeatureGeometryType() = "SURFACE";
					pair.FeatureAlignmentType() = CyPhyML::Surface::Cast(a_feature).Alignment();
				}
				else if (a_feature.type() == CyPhyML::Point::meta)
					pair.FeatureGeometryType() = "POINT";
				else if (a_feature.type() == CyPhyML::CoordinateSystem::meta)
				{
					pair.FeatureGeometryType() = "CSYS";
					pair.FeatureAlignmentType() = "CSYS";
				}

				AssemblyInterface::ConstraintFeature a_constraint_feature = AssemblyInterface::ConstraintFeature::Create(pair);
				AddCADModelFeatureDetails(a_feature, a_constraint_feature);
				a_constraint_feature.ComponentID() = aId;

				AssemblyInterface::ConstraintFeature b_constraint_feature = AssemblyInterface::ConstraintFeature::Create(pair);
				AddCADModelFeatureDetails(b_feature, b_constraint_feature);
				b_constraint_feature.ComponentID() = bId;

				break;
			}
			else
				count ++;	
		}
	}	
}


void AddCADModelFeatureDetails(CyPhyML::StructuralInterfaceFeature& feature_in,
							   AssemblyInterface::ConstraintFeature& ai_feature_inout)
{
	std::string orientation = "NONE";

	CyPhyML::GeometryFeature geometry_feature = CommonData::SIFeature_To_GeometryFeature_Lookup_table[feature_in];

	if (geometry_feature != Udm::null)
	{
		ai_feature_inout.FeatureName() = geometry_feature.Datum();	
		if (geometry_feature.type().name() == "SurfaceGeometry")
			orientation = CyPhyML::SurfaceGeometry::Cast(geometry_feature).Orientation();
	}
	ai_feature_inout.FeatureOrientationType() = orientation;
}

#if 0
void AddCADModelFeatureDetails(CyPhyML::StructuralInterfaceFeature& feature_in,
							   AssemblyInterface::ConstraintFeature& ai_feature_inout)
{
	std::string orientation = "NONE";

	
	set<CyPhyML::FeatureMap> a_src_Set = feature_in.srcFeatureMap();
	if (!a_src_Set.empty())
	{
		CyPhyML::SolidModelingFeature tmp = a_src_Set.begin()->srcFeatureMap_end();
		if (Uml::IsDerivedFrom(tmp.type(), CyPhyML::GeometryFeature::meta))
		{
			ai_feature_inout.FeatureName() = CleanString(CyPhyML::GeometryFeature::Cast(tmp).Datum());	
			if (tmp.type() == CyPhyML::SurfaceGeometry::meta)
				orientation = CyPhyML::SurfaceGeometry::Cast(tmp).Orientation();
		}
	}
	else
	{
		set<CyPhyML::FeatureMap> a_dst_Set = feature_in.dstFeatureMap();
		if (!a_dst_Set.empty())
		{
			CyPhyML::SolidModelingFeature tmp = a_dst_Set.begin()->dstFeatureMap_end();
			if (Uml::IsDerivedFrom(tmp.type(), CyPhyML::GeometryFeature::meta))
			{
				ai_feature_inout.FeatureName() = CleanString(CyPhyML::GeometryFeature::Cast(tmp).Datum());
				if (tmp.type() == CyPhyML::SurfaceGeometry::meta)
					orientation = CyPhyML::SurfaceGeometry::Cast(tmp).Orientation();
			}
		}
	}
	ai_feature_inout.FeatureOrientationType() = orientation;
}
#endif

void AddRootConstraint2CADComponent(AssemblyInterface::CADComponent& component, 
									std::string assemblyParentID)
{
	std::string name = component.Name();

	std::string id = component.ComponentID();
	bool rootIsComponentAssembly = component.Type() == "ASSEMBLY";

	AssemblyInterface::Constraint constraint = AssemblyInterface::Constraint::Create(component);

	AssemblyInterface::Pair front_pair = AssemblyInterface::Pair::Create(constraint);
	front_pair.FeatureGeometryType() = "SURFACE";
	front_pair.FeatureAlignmentType() = "ALIGN";
	front_pair.FeatureInterfaceType() = "CAD_DATUM";
	
	if (rootIsComponentAssembly)
		CreateConstraintFeature(front_pair, id,  "ASM_FRONT", "SIDE_A");
	else
		CreateConstraintFeature(front_pair, id,  "FRONT", "SIDE_A");
	CreateConstraintFeature(front_pair, assemblyParentID, "ASM_FRONT","SIDE_A");
	
	AssemblyInterface::Pair top_pair = AssemblyInterface::Pair::Create(constraint);
	top_pair.FeatureGeometryType() = "SURFACE";
	top_pair.FeatureAlignmentType() = "ALIGN";
	top_pair.FeatureInterfaceType() = "CAD_DATUM";
	
	if (rootIsComponentAssembly)
		CreateConstraintFeature(top_pair, id,  "ASM_TOP", "SIDE_A");
	else
		CreateConstraintFeature(top_pair, id,  "TOP", "SIDE_A");
	CreateConstraintFeature(top_pair, assemblyParentID, "ASM_TOP","SIDE_A");

	AssemblyInterface::Pair right_pair = AssemblyInterface::Pair::Create(constraint);
	right_pair.FeatureGeometryType() = "SURFACE";
	right_pair.FeatureAlignmentType() = "ALIGN";
	right_pair.FeatureInterfaceType() = "CAD_DATUM";
	
	if (rootIsComponentAssembly)
		CreateConstraintFeature(right_pair, id,  "ASM_RIGHT", "SIDE_A");
	else
		CreateConstraintFeature(right_pair, id,  "RIGHT", "SIDE_A");
	CreateConstraintFeature(right_pair, assemblyParentID, "ASM_RIGHT","SIDE_A");
}

void CreateConstraintFeature(AssemblyInterface::Pair& pair, 
							std::string id, 
							std::string name, 
							std::string orientation)
{
	AssemblyInterface::ConstraintFeature constraintFeature = AssemblyInterface::ConstraintFeature::Create(pair);
	constraintFeature.ComponentID() = id;
	constraintFeature.FeatureName() = name;
	constraintFeature.FeatureOrientationType() = orientation;
}

void Check4ZeroConstraints(const AssemblyInterface::CADComponent& start)
{
	//GMEConsole::Console::writeLine("Check4ZeroConstraits() for CADComponent [" + (std::string)start.ComponentID() + "," + (std::string)start.Name() + "]", MSG_INFO);
	std::string id = start.ComponentID(), name = start.Name();

	set<AssemblyInterface::Constraint> constraint_Set = start.Constraint_kind_children();
	if (AssemblyInterface::Assembly::Cast(start.Assembly_parent()) == Udm::null)
	{
		if (constraint_Set.size() < 1)
			GMEConsole::Console::writeLine("CADComponent [" + (std::string)start.Name() + "," + (string)start.ComponentID() + "] has 0 constraints!", MSG_ERROR);
		else if (constraint_Set.size() > 1)
		{
			if ((string)start.SpecialInstruction() != "SIZE_TO_FIT")
				GMEConsole::Console::writeLine("CADComponent [" + (std::string)start.Name() + "," + (string)start.ComponentID() + "] has >1 constraints!", MSG_WARNING);
		}
	}

	set<AssemblyInterface::CADComponent> comp_Set = start.CADComponent_kind_children();
	for (set<AssemblyInterface::CADComponent>::iterator i = comp_Set.begin(); i != comp_Set.end(); i++)
	{
		AssemblyInterface::CADComponent comp(*i);
		Check4ZeroConstraints(comp);
	}
}

void Check4ZeroConstraints(const AssemblyInterface::Assembly& start)
{
	set<AssemblyInterface::CADComponent> comp_Set = start.CADComponent_kind_children();
	for (set<AssemblyInterface::CADComponent>::iterator i = comp_Set.begin(); i != comp_Set.end(); i++)
	{
		AssemblyInterface::CADComponent comp(*i);
		Check4ZeroConstraints(comp);
	}
}



// pair(x, y)
pair<int, int> GetXYPosition(const CyPhyML::MgaObject& obj)
{
	string objID = UdmGme::UdmId2GmeId(obj.uniqueId());
	string position = obj.position();
	string x, y;

	size_t solidModeling = position.find("SolidModeling");
	if (solidModeling == std::string::npos)
	{
		x = position.substr(position.find_last_of("(")+1, position.find_last_of(",")-position.find_last_of("(")-1);
		y = position.substr(position.find_last_of(",")+1, position.find_last_of(")")-position.find_last_of(",")-1);
	}
	else
	{
		position = position.substr(solidModeling);
		x = position.substr(position.find_first_of("(")+1, position.find_first_of(",")-position.find_first_of("(")-1);
		y = position.substr(position.find_first_of(",")+1, position.find_first_of(")")-position.find_first_of(",")-1);
	}

	return make_pair(atoi(x.c_str()), atoi(y.c_str()));
}

bool CompareYPos::operator()(const CyPhyML::MgaObject& a, 
							const CyPhyML::MgaObject& b) 
{

	pair<int, int> a_XY = GetXYPosition(a);		// pair(x, y)
	pair<int, int> b_XY = GetXYPosition(b);     // pair(x, y)	
	
	// Compare Y-pos first before comparing X-pos
	if (a_XY.second < b_XY.second)
		return true;
	else if (a_XY.second > b_XY.second)
        return false;
	else if (a_XY.second == b_XY.second)
		return a_XY.first < b_XY.first;

    return false;
  }


string GetGMEGUID(const Udm::Object &object)
{
	string guid = "";
	CComQIPtr<IMgaObject> cObj;
	CComPtr<IUnknown> punk = UdmGme::Udm2Gme(object);
	punk.QueryInterface(&cObj);

	if (cObj)
	{
		CComBSTR cGuid;
		HRESULT res = cObj->GetGuidDisp(&cGuid);
		if (res == S_OK)
			guid = _bstr_t(cGuid);
	}

	size_t begin = guid.find('{');
	size_t end = guid.find('}');

	if (begin != string::npos)
		guid = guid.substr(begin+1);

	if (end != string::npos)
		guid = guid.substr(0, end-1);

	return guid;
}


int GetDegreesOfFreedom(const CyPhyML::StructuralInterface &si_in)
{
	set<CyPhyML::StructuralInterfaceFeature> feature_set = si_in.StructuralInterfaceFeature_kind_children();
	set<CyPhyML::CoordinateSystem> csys_set = si_in.CoordinateSystem_kind_children();
	int count = feature_set.size(), dof = 0;

	if (csys_set.empty())
	{
		switch(count)
		{
		case 0:
			dof = 0;
			break;
		case 1:
			{
				CyPhyML::StructuralInterfaceFeature feature = *feature_set.begin();
				if (feature.type() == CyPhyML::Axis::meta)
					dof = 4;
				else
					dof = 3;			
			}
			break;
		case 2:
			dof = 5;
			break;
		case 3:
			dof = 6;
			break;
		default:
			dof = 6;
			break;
		}
	}
	else
	{
		dof = 6;
	}

	return dof;
}